#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
Backend
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
HTML constructors
=============================================
|#

(deff b-headline (insert name &optional after)
 (list
  "Logiweb "
  (html-link0 (help-link "browser/index") "crossbrowser")
  " "
  insert
  " for the "
  (html-string name) " page" after))

#|
=============================================
TeX constructors
=============================================
|#

(deff b-tex-headline (insert name &optional after)
 (list
  (line "Logiweb crossbrowser")
  insert
  (line " for the ``" name "'' page " after)))

#|
=============================================
Convert reference to timestamp
=============================================
(ref2timestamp ref) converts the given cardinal to a sequence of bytes, parses it, and returns the timestamp of the reference
|#

(deff ref2timestamp (ref)
 (:let suffix (card2ref ref))
 (:let (nil . suffix) (m-parse-card suffix))
 (:let (nil . suffix) (m-parse-bytes 20 suffix nil))
 (:let (mantissa . suffix) (m-parse-card suffix))
 (:let (exponent) (m-parse-card suffix))
 (list mantissa exponent))

#|
=============================================
Execute program in given directory
=============================================
(exec-program dir program &rest args) executes the given program with the given command line arguments, with the default directory temporarily set to dir.
|#

(deff exec-program (dir program &rest args)
 (:let dir1 (cd))
 (unwind-protect
  (progn
   (cd (ct2string dir))
   (run-program program :arguments args :input nil))
  (cd dir1)))

(deff exec-silent (dir program args)
 (:let dir1 (cd))
 (unwind-protect
  (progn
   (cd (ct2string dir))
   (exec-silent1 program args))
  (cd dir1)))

(deff exec-silent1 (program args)
 (:let verbose (verbose '> 1))
 (:let out (if verbose :terminal nil))
 (:let result (run-program program :arguments args :input nil :output out))
 (:let result (default 0 result))
 (when (unequal result 0) (exec-complain program result)))

(deff exec-complain (program result)
 (format t "~a: result=~s. " program result)
 (format t "Rerun with verbose=2 to see messages")
 (:when (equal program "tex")    (format t " or browse .log file.~%"))
 (:when (equal program "latex")  (format t " or browse .log file.~%"))
 (:when (equal program "bibtex") (format t " or browse .blg file.~%"))
 (:when (equal program "mizf")   (format t " or browse .miz file.~%"))
 (format t ".~%"))

(deff exec-tex (dir args)
 (exec-silent dir "tex" args))

(deff exec-latex (dir args)
 (exec-silent dir "latex" args))

(deff exec-bibtex (dir args)
 (exec-silent dir "bibtex" args))

(deff exec-mizf (dir args)
 (exec-program dir "rm" "-f" "dict")
 (exec-program dir "ln" "-s" "." "dict")
 (exec-silent dir "mizf" args))

; exec-shell included because run-program cannot redirect stderr

(deff exec-shell (dir program args)
 (:let dir1 (cd))
 (unwind-protect
  (progn
   (cd (ct2string dir))
   (exec-shell1 program args))
  (cd dir1)))

(deff exec-add-space (args)
 (:let (arg . args) args)
 (:when (atom args) arg)
 (list* arg " " (exec-add-space args)))

(ntst (exec-add-space nil))
(etst (exec-add-space '("ab")) "ab")
(etst (exec-add-space '("ab" "cd")) '("ab" " " . "cd"))

(deff exec-shell1 (program args)
 (:let verbose (verbose '> 1))
 (:let out (if verbose "" " > /dev/null 2> /dev/null "))
 (:let ct (list program " " (exec-add-space args) " < /dev/null " out))
 (:let result (shell (ct2string ct)))
 (:let result (default 0 result))
 (when (unequal result 0) (format t "~a: result=~s~%" program result)))

(deff exec-makeindex (dir args)
 (exec-shell dir "makeindex" args))

(deff exec-dvipdfm (dir args)
 (exec-shell dir "dvipdfm" args))

#|
=============================================
Notify local server about submission
=============================================
(submit-notify ref url) notifies the local server that the given url associates to the given reference.
|#

(deff submit-notify (ref url)
 (:let prefix* nil)
 (:let address (m-ref2vector ref))
 (:let url (m-ct2vector url))
 (:let request (m-add-url prefix* address url))
 (:let dest (cons (option-udphost) (option "udpport")))
 (socket-protect fd (udp-open) (udp-sendto fd request dest)))

#|
=============================================
Conversion of term to Logiweb source
=============================================
|#

(deff b-vector2lgs (vector)
 (:let vector (card2vector vector))
 (:let (vector . vector*) (c-name-split vector 0))
 (list "\"" vector (b-vector*2lgs vector*) "\""))

(deff b-vector*2lgs (vector*)
 (:when (null vector*) nil)
 (:let (vector . vector*) vector*)
 (list* "\"!" vector (b-vector*2lgs vector*)))

(etst
 (ct2vector (b-vector2lgs (ct2vector "AB\"CD\"EF")))
 (ct2vector "\"AB\"!CD\"!EF\""))

(deff b-tree2lgs (tree state)
 (:when (null tree) " \"\" ")
 (:when (equalp tree :quote) "\"")
 (:let ((ref id) . tree*) tree)
 (:when (equalp ref nil) (list " { " (b-tree2lgs (head tree*) state) " } "))
 (:when (equalp ref 0) (b-vector2lgs id))
 (:let (name . name*) (c-symbol2nameaspect* ref id state))
 (cons name (b-tree*2lgs tree* name* state)))

(deff b-tree*2lgs (tree* name* state)
 (:when (and (atom tree*) (atom name*)) nil)
 (:let (tree . tree*) tree*)
 (:let tree (b-tree2lgs tree state))
 (:let (name . name*) name*)
 (:let name (default "@@" name))
 (list* tree name (b-tree*2lgs tree* name* state)))

#|
=============================================
Conversion of lgs to html
=============================================
|#

#|
(deff b-html-special (element)
 (:when (equalp element f-newline) (cons "<br>" f-newline))
 (:when (equalp element (char-code #\<)) "&lt;")
 (:when (equalp element (char-code #\>)) "&gt;")
 (:when (equalp element (char-code #\&)) "&amp;")
 (:when (equalp element (char-code #\")) "&quot;")
 nil)

(deff b-html-special* (vector index)
 (:let position (position-if 'b-html-special vector :start index))
 (:when (null position) (list (subvector vector index)))
 (:let vector1 (subvector vector index position))
 (:let element (b-html-special (aref vector position)))
 (:let vector* (b-html-special* vector (+ position 1)))
 (list* vector1 element vector*))

(deff b-array2html (vector)
 (b-html-special* vector 0))

(deff b-ct2html (ct)
 (b-array2html (ct2vector ct)))

(deff b-card*2html (ct)
 (b-array2html (card*2vector ct)))

(etst (ct2string (b-ct2html "a\"<&>
b")) "a&quot;&lt;&amp;&gt;<br>
b")

(deff b-tree2html (tree state)
 (b-ct2html (b-tree2lgs tree state)))
|#

(deff b-tree2html (tree state)
 (utf2html (b-tree2lgs tree state)))

#|
=============================================
Face making
=============================================
The 'face' of an expression is the appearance of that expression as seen by a human reader. Hence, a 'face' is what the rest of the world calls a 'rendering' ('face' has four characters, 'rendering' has nine, so I'm five characters ahead of the rest of the world:-).

In the present context, however, we shall use the word 'face' to denote the data that the Logiweb system communicates to rendering engines like the TeX|dvipdfm|acroread pipeline. The 'TeX|dvipdfm|acroread' pipeline takes TeX source as input, then TeX converts it to dvi, then dvipdfm converts the dvi to pdf, and then acroread renders the pdf.

From now on we shall refer to the TeX|dvipdfm|acroread pipeline simply as 'TeX'.

When the Logiweb system renders an expression using TeX, the Logiweb system sends TeX source for the expression to the pipeline. Hence, in the present perspective, the transmitted TeX source is the 'face' of the given expression. We shall refer to that face as the 'tex face' of the expression.

Once upon a time, Logiweb also supported MathML, but that gave little benefit for a lot of work.

Logiweb builds up the face of expressions from faces of the individual symbols that make up the expression. The 'face-making algorithm' for building faces of expressions from faces of symbols is a dumb text substitution algorithm. Anything more advanced than that should use the Logiweb computational engine. But it is convenient to have at least a name and a TeX face available at the initial stages of the development of Logiweb before the computational engine has been optimized and fine tuned to deliver breathtaking performance.

Now suppose a symbol L has the following name, tex use, and tex show aspects:
  name:     x less than y -> "* less than *"
  tex use:  x less than y -> ""[x]"<\\"[y]""
  tex show: x less than y -> ""[x]"<"[y]""

The name aspect indicates how to produce the 'name face' of an expression E with root L: replace the n'th asterisk by the name face of the n'th subexpression.

The tex use and tex show aspects indicate how to produce those other faces. For those faces, the places for insertion of sub-faces are indicated by parameters rather than asterisks.

The tex show aspect is used when talking about a construct (e.g. when the construct occurs on the left hand side of a definitions). The tex use aspect is used in other cases. Many constructs have identical tex use and tex show aspects. For that reason the convention is made that the tex show aspect defaults to be equal to the tex use aspect.

The tex use aspect of a symbol is defined thus: If the aspect is defined explicitly, then that definition is used. Else, if the name aspect is defined then a tex use aspect is constructed from the name aspect. Else, the tex use aspect is constructed from the reference of the symbol expressed in mixed endian kana and the index expressed in decimal.

The tex show aspect of a symbol is defined thus: If the aspect is defined explicitly, then that definitions is used. Else, if the tex show aspect is equal to the tex use aspect.

The tex show aspect of a symbol indicates what the symbol should look like when talking about the symbol rather than just using it. As an example, a unary symbol B could have the following tex use and tex show aspects:
  tex use      bold x -> "{\bf "[x]"}"
  tex show     bold x -> "\mathrm{bold}("[x]")"
The tex use face of the symbol B applied to a subexpression E will result in E being displayed in bold. In contrast, the tex show face of the symbol B applied to a subexpression E will result in E being enclosed in parentheses and prefixed by the letters 'b', 'o', 'l', and 'd'.

Now consider a symbol with the following aspect:
  tex use define ( x == y ) end define -> "( "[ name x ]" == "[ y ]" )"
When e.g. ( bold x == x + 2 ) is typeset in the tex use aspect, the 'name' in the tex use definition forces 'bold x' to be typeset using the tex show aspect.

In general, any unary operator applied to a parameter forces the corresponding subtree to be rendered using the tex show aspect. Without a unary parameter, the subtree is rendered in the same face as the supertree.

The face making algorithm is not neccessarily fast. A more reasonable approach would be to extract the information about tex use and tex show aspects once and store it in a separate branch of the state.
|#

#|
=============================================
Conversion of tree to tex
=============================================
Several tex conversion functions have a parameter named 'namep'. When true, that parameter forces the tree in question to be rendered using the tex show aspect.

Some functions uses 'tex macros'. A tex macro is a list (a_1 ... a_n) where each element is either a vector or a cons. A cons has form (namep . argnum) where argnum indexes the argument list and namep, when true, forces the given argument to be rendered using the tex show aspect.

As an example, ("f(" (nil . 0) ")") represents \newcommand{...}{f(#1)}.
|#

(defc b-open "
\\ \\linebreak[0]\\textcolor{blue}{[}\\linebreak[0]\\ ")

(defc b-close "
\\ \\linebreak[0]\\textcolor{blue}{]}\\linebreak[0]\\ ")

(deff b-paren (face)
 (list b-open face b-close))

(defc b-char2tex
 #(
  "(00)"
  "(01)"
  "(02)"
  "(03)"
  "(04)"
  "(05)"
  "(06)"
  "(07)"
  "(08)"
  "(09)"
  "
\\newline "
  "(0B)"
  "(0C)"
  "(0D)"
  "(0E)"
  "(0F)"
  "(10)"
  "(11)"
  "(12)"
  "(13)"
  "(14)"
  "(15)"
  "(16)"
  "(17)"
  "(18)"
  "(19)"
  "(1A)"
  "(1B)"
  "(1C)"
  "(1D)"
  "(1E)"
  "(1F)"
  "\\linebreak [0]\\ "
  "!"
  "\\mbox{\\tt\\char34}"
  "\\#"
  "\\$"
  "\\%"
  "\\&"
  "\\mbox{'}"
  "("
  ")"
  "{*}"
  "{+}"
  ","
  "\\mbox{-}"
  "."
  "/"
  "0"
  "1"
  "2"
  "3"
  "4"
  "5"
  "6"
  "7"
  "8"
  "9"
  "{:}"
  ";"
  "("
  "{=}"
  ")"
  "?"
  "@"
  "A"
  "B"
  "C"
  "D"
  "E"
  "F"
  "G"
  "H"
  "I"
  "J"
  "K"
  "L"
  "M"
  "N"
  "O"
  "P"
  "Q"
  "R"
  "S"
  "T"
  "U"
  "V"
  "W"
  "X"
  "Y"
  "Z"
  "["
  "\\mbox{$\\backslash$}"
  "]"
  "{\\char94}"
  "\\_"
  "\\mbox{`}"
  "a"
  "b"
  "c"
  "d"
  "e"
  "f"
  "g"
  "h"
  "i"
  "j"
  "k"
  "l"
  "m"
  "n"
  "o"
  "p"
  "q"
  "r"
  "s"
  "t"
  "u"
  "v"
  "w"
  "x"
  "y"
  "z"
  "\\{"
  "|"
  "\\}"
  "{\\char126}"
  "(7F)"
  "(80)"
  "(81)"
  "(82)"
  "(83)"
  "(84)"
  "(85)"
  "(86)"
  "(87)"
  "(88)"
  "(89)"
  "(8A)"
  "(8B)"
  "(8C)"
  "(8D)"
  "(8E)"
  "(8F)"
  "(90)"
  "(91)"
  "(92)"
  "(93)"
  "(94)"
  "(95)"
  "(96)"
  "(97)"
  "(98)"
  "(99)"
  "(9A)"
  "(9B)"
  "(9C)"
  "(9D)"
  "(9E)"
  "(9F)"
  "(A0)"
  "(A1)"
  "(A2)"
  "(A3)"
  "(A4)"
  "(A5)"
  "(A6)"
  "(A7)"
  "(A8)"
  "(A9)"
  "(AA)"
  "(AB)"
  "(AC)"
  "(AD)"
  "(AE)"
  "(AF)"
  "(B0)"
  "(B1)"
  "(B2)"
  "(B3)"
  "(B4)"
  "(B5)"
  "(B6)"
  "(B7)"
  "(B8)"
  "(B9)"
  "(BA)"
  "(BB)"
  "(BC)"
  "(BD)"
  "(BE)"
  "(BF)"
  "(C0)"
  "(C1)"
  "(C2)"
  "(C3)"
  "(C4)"
  "(C5)"
  "(C6)"
  "(C7)"
  "(C8)"
  "(C9)"
  "(CA)"
  "(CB)"
  "(CC)"
  "(CD)"
  "(CE)"
  "(CF)"
  "(D0)"
  "(D1)"
  "(D2)"
  "(D3)"
  "(D4)"
  "(D5)"
  "(D6)"
  "(D7)"
  "(D8)"
  "(D9)"
  "(DA)"
  "(DB)"
  "(DC)"
  "(DD)"
  "(DE)"
  "(DF)"
  "(E0)"
  "(E1)"
  "(E2)"
  "(E3)"
  "(E4)"
  "(E5)"
  "(E6)"
  "(E7)"
  "(E8)"
  "(E9)"
  "(EA)"
  "(EB)"
  "(EC)"
  "(ED)"
  "(EE)"
  "(EF)"
  "(F0)"
  "(F1)"
  "(F2)"
  "(F3)"
  "(F4)"
  "(F5)"
  "(F6)"
  "(F7)"
  "(F8)"
  "(F9)"
  "(FA)"
  "(FB)"
  "(FC)"
  "(FD)"
  "(FE)"
  "(FF)"
))

(deff b-string2quote (card*)
 (list "\\mbox{`}" (b-string2tex card*) "\\mbox{'}"))

(deff b-string2tex (card*)
 (b-string2tex1 card* nil))

(deff b-string2tex1 (card* result)
 (:when (null card*) result)
 (:let (card . card*) card*)
 (b-string2tex1 card* (cons result (aref b-char2tex card))))

(deff b-get-renderer (ref state)
 (:let renderer (codex-get state ref 0 0 card-render))
 (:when renderer renderer)
 (:let (:ref bed) (rack-get state ref card-bibliography))
 (:when (null bed) nil)
 (codex-get state bed 0 0 card-render))

(deff b-tree2tex (ref tree state &optional namep)
 (b-render-type 2 ref state)
 (:let renderer (b-get-renderer ref state))
 (:when (null renderer) (b-tree2tex1 tree state namep))
 (:let cache (aget state (id-cache) ref))
 (:let (:define :render :lhs renderer) renderer)
 (:let arglist nil)
 (:let term (tree2term cache arglist renderer))
 (:when (null (term-eval term)) (b-tree2tex1 tree state namep))
 (:let term (term-apply term (term-tv tree)))
 (:let term (term-apply term (term-tv (bool2val namep))))
 (:let term (term-apply term (term-tv cache)))
 (:catch () (c-error "The renderer raised an exception"))
 (tm2tv (term-eval term nil))) ; why not just (term-eval term) ?

(deff b-tree2tex1 (tree state &optional namep)
 (:when (null tree) "\\mbox {\\tt \\char34}")
 (:when (equalp tree :quote) (error "Check failed: :quote tree"))
 (:let ((ref id) . tree*) tree)
 (:when (null ref) (b-paren (b-tree2tex1 (car tree*) state namep)))
 (:when (equalp ref 0)
  (:let vector (card2vector id))
  (:when (null namep) vector)
  (b-string2quote (vector2card* vector)))
 (:let macro (b-symbol2macro ref id state namep))
 (b-macro-exec macro tree* state namep))

(deff b-macro-exec (macro tree* state namep)
 (:when (atom macro) macro)
 (:let (m1 . m2) macro)
 (:when (and (integerp m2) (null m1)) (b-tree2tex1 (nth m2 tree*) state namep))
 (:when (and (integerp m2) (equalp m1 t)) (b-tree2tex1 (nth m2 tree*) state t))
 (cons (b-macro-exec m1 tree* state namep) (b-macro-exec m2 tree* state namep)))

#| Old version where tex show aspect defaults to tex use aspect
; Try tex show aspect
(deff b-symbol2macro (ref id state namep)
 (:when (null namep) (b-symbol2macro1 ref id state))
 (:let def (codex-get state ref id 0 card-texname))
 (:let (:def :aspect (:lhs . arg*) rhs) def)
 (:when rhs (b-tree2macro arg* rhs state))
 (b-symbol2macro1 ref id state))

; Try tex use aspect
(deff b-symbol2macro1 (ref id state)
 (:let def (codex-get state ref id 0 card-tex))
 (:let (:def :aspect (:lhs . arg*) rhs) def)
 (:when rhs (b-tree2macro arg* rhs state))
 (b-name2macro ref id state))
|#

; New version where tex use aspect defaults to tex show aspect
; Try tex use aspect
(deff b-symbol2macro (ref id state namep)
 (:when namep (b-symbol2macro1 ref id state))
 (:let def (codex-get state ref id 0 card-tex))
 (:let (:def :aspect (:lhs . arg*) rhs) def)
 (:when rhs (b-tree2macro arg* rhs state))
 (b-symbol2macro1 ref id state))

; Try tex show aspect
(deff b-symbol2macro1 (ref id state)
 (:let def (codex-get state ref id 0 card-texname))
 (:let (:def :aspect (:lhs . arg*) rhs) def)
 (:when rhs (b-tree2macro arg* rhs state))
 (b-name2macro ref id state))

; Try name aspect
(deff b-name2macro (ref id state)
 (:let (name . name*) (c-symbol2nameaspect* ref id state))
 (cons (b-name2macro1 name) (b-name2macro* name* 0)))

(deff b-name2macro1 (name)
 (ct2vector (b-string2tex (ct2card* name))))

(deff b-name2macro* (name* index)
 (:when (null name*) nil)
 (:let (name . name*) name*)
 (:let macro (b-name2macro1 name))
 (:let macro* (b-name2macro* name* (+ index 1)))
 (list* (cons nil index) macro macro*))

#|
; Convert arg*/tree to macro
(deff b-tree2macro (arg* tree state)
 (reverse (b-tree2macro1 arg* tree nil state nil)))

(deff b-tree2macro1 (arg* tree namep state result)
 (:let ((ref id) . tree*) tree)
 (:when (equalp ref 0) (cons id result))
 (:let pos (position tree arg* :test 'c-tree-equal))
 (:when pos (acons namep pos result))
 (:when (atom tree*) result)
 (:let (tree . rest) tree*)
 (:when (null rest) (b-tree2macro1 arg* tree t state result))
 (b-tree2macro* arg* tree* state result))

(deff b-tree2macro* (arg* tree* state result)
 (:when (null tree*) result)
 (:let (tree . tree*) tree*)
 (:let result (b-tree2macro1 arg* tree nil state result))
 (:let result (b-tree2macro* arg* tree* state result))
 result)
|#

(deff b-tree2macro (arg* tree state)
 (b-tree2macro1 arg* tree nil state))

(defc *has-warned-about-value* nil)

(deff b-tree2macro1 (arg* tree namep state)
 (:let ((ref id) . tree*) tree)
 (when (aget state (id-cache) ref ref card-codex ref id 0 card-value)
  (unless *has-warned-about-value*
   (format t "Warning: Construct in rhs of TeX definition has a value:~%")
   (format t "~a~%" (c-symbol2name ref id state))
   (format t "That is reserved for future extensions~%")
   (setq *has-warned-about-value* t)))
 (:when (equalp ref 0) id)
 (:let pos (position tree arg* :test 'c-tree-equal))
 (:when pos (cons namep pos))
 (:when (atom tree*) nil)
 (:let (tree . rest) tree*)
 (:when (null rest) (b-tree2macro1 arg* tree t state))
 (b-tree2macro* arg* tree* state))

(deff b-tree2macro* (arg* tree* state)
 (:when (null tree*) nil)
 (:let (tree . tree*) tree*)
 (:let macro (b-tree2macro arg* tree state))
 (:let macro* (b-tree2macro* arg* tree* state))
 (cons macro macro*))

#|
=============================================
Format adaption functions
=============================================
|#

(deff b-symbol2tex (ref id state)
 (:when (equalp ref 0) (list "$\\tt " (card2vector id) "$"))
 (:let dict (rack-get state ref card-dictionary))
 (:let arity (aget dict id))
 (:let tree (cons (list ref id) (repeat arity nil)))
 (:let face (b-tree2tex ref tree state :tex-name))
 (list "$\\rm " face "$"))

(deff b-ref2tex (ref state)
 (b-symbol2tex ref 0 state))

#|
=============================================
Add parentheses to tree
=============================================
Before rendering in lgs/tex
|#

(deff b-add-parentheses (tree state)
 (b-add-par nil nil tree state))

(deff b-add-par (pre post tree state)
 (:let (root . tree*) tree)
 (:when (null tree*) tree)
 (:let (ref id) root)
 (:let (pre1 post1) (b-openness ref id state))
 (:let tree* (b-add-par* pre1 post1 tree* state))
 (:let tree (cons root tree*))
 (:when (or (and pre post1) (and pre1 post)) (list nil tree))
 tree)

(deff b-add-par* (pre post tree* state)
 (:let (tree . tree*) tree*)
 (:when (null tree*) (list (b-add-par pre post tree state)))
 (:let tree (b-add-par pre nil tree state))
 (:let tree* (b-add-par* nil post tree* state))
 (cons tree tree*))

(deff b-openness (ref id state)
 (:let name (fourth (codex-get state ref id 0 card-name)))
 (:when (null name) nil)
 (:let name (c-tree2string name))
 (:let name (string-trim '(#\Space #\Newline) name))
 (:let preopen (equalp (aref name 0) #\") nil)
 (:let postopen (equalp (aref name (- (length name) 1)) #\") nil)
 (list preopen postopen))

#|
=============================================
Delete file
=============================================
|#

(deff remove-pattern (dir pattern)
 (:let pattern (ct2string (cons dir pattern)))
 (when (verbose '= 1) (format t "Deleting ~s~%" pattern))
 (:let path* (directory pattern))
 (dolist (path path*)
  (when (verbose '> 1) (format t "Deleting ~s~%" path))
  (delete-file path)))

(deff clean-dir (dir)
 (ensure-directories-exist (ct2string (cons dir "/page.tex")))
 (remove-pattern dir "/page.*")
 (remove-pattern dir "/lgw*"))

#|
=============================================
Generate <dir>
=============================================
Generate <dir>/index.html plus subdirectories of <dir>
|#

(deff b-render-type (verbosity ref state)
 (:when (verbose '< verbosity) nil)
 (:let renderer (b-get-renderer ref state))
 (:when (null renderer) (format t "Backend: Default rendering~%"))
 (:let (:define :render :lhs renderer) renderer)
 (:let arglist nil)
 (:let cache (aget state (id-cache) ref))
 (:let term (tree2term cache arglist renderer))
 (:when (null (term-eval term)) (format t "Backend: Default rendering~%"))
 (format t "Backend: Custom rendering~%"))

(deff b-dir (dir lgt name ref state source mirror)
 (:let name (utf2html name))
 (b-render-type 1 ref state)
 (progress "Backend: writing index")
 (b-index dir lgt name (untag (rack-get state ref card-diagnose)) mirror)
 (progress "Backend: writing diagnose")
 (b-diagnose   (cons dir "/diagnose"	) lgt name ref state)
 (progress "Backend: writing reference")
 (b-ref        (cons dir "/reference"	) lgt name ref)
 (progress "Backend: writing source")
 (b-source     (cons dir "/vector")       lgt name ref state source)
 (:when (and (equalp mirror :nomirror) (level '< "body")) nil)
 (progress "Backend: writing body")
 (b-body  (cons dir "/body"	) lgt name ref state)
 (progress "Backend: writing bibliography")
 (b-bib        (cons dir "/bibliography") lgt name ref state)
 (progress "Backend: writing dictionary")
 (b-dictionary (cons dir "/dictionary"  ) lgt name ref state)
 (progress "Backend: writing external formats")
 (b-external (cons dir "/vector"        ) lgt name ref state)
 (:when (and (equalp mirror :nomirror) (level '< "codex")) nil)
 (progress "Backend: writing codex")
 (b-codex      (cons dir "/codex"	) lgt name ref state)
 (progress "Backend: writing expansion")
 (b-expansion  (cons dir "/expansion"	) lgt name ref state)
 (progress "Backend: writing header")
 (b-header     (cons dir "/header"	) lgt name ref state)
 (:when (and (equalp mirror :nomirror) (level '< "all")) nil)
 (progress "Backend: writing vector")
 (b-vector     (cons dir "/vector"	) lgt name ref state mirror))

#|
=============================================
Generate <dir>/index.html
=============================================
|#

(deff b-link (mirror level ref &rest ct)
 (line
  (html-ref ref ct)
  (when (and (equalp mirror :nomirror) (level '< level)) "*")))

(deff b-index (dir lgt name tree mirror)
 (:let diagnose
  (if (not (option "test"))
   "Testing has been suppressed"
   (if tree
    (html-link0 "diagnose/index.html" "Diagnose")
    "The page is correct")))
 (ct2file (cons dir "/index.html")
  (html-page lgt
   (list "Logiweb main menu of " name)
   (html-h2 "Logiweb main menu of " name)
   (list
    (html-table "0"
     (html-row
      "General:"
      (list
       (html-link0 "../index.html"                      "Up"          )
       (html-link0 (help-link "browser/main")           "Help"        )))
     (html-row
      "Starting point:"
      (html-link0            " reference/index.html"     "Reference"   ))
     (html-row
      "After fetching:"
      (b-link mirror  "all"   "vector/index.html"        "Vector"      ))
     (html-row
      "After unpacking:"
      (list
       (b-link mirror "body"  "body/index.html"          "Body"        )
       (b-link mirror "body"  "bibliography/index.html"  "Bibliography")
       (b-link mirror "body"  "dictionary/index.html"    "Dictionary"  )
       (b-link mirror "body"  "vector/external.html"     "External formats")))
     (html-row
      "After codifying:"
      (list
       (b-link mirror "codex" "codex/index.html"         "Codex"       )
       (b-link mirror "codex" "expansion/index.html"     "Expansion"   )
       (b-link mirror "codex" "header/header.lgs"        "Header"      )))
     (html-row
      "After verifying:"
      diagnose)
     (html-row
      "Where it all began:"
      (html-link0           "vector/source.html"        "Source"      )))
    (when (and (equalp mirror :nomirror) (level '< "all"))
     (html-p "* Not up to date"))
    (when (or (equalp mirror :mirror) (level '>= "body"))
     (html-p "Click 'Body' to see the page"))))))

#|
=============================================
Generate <dir>/reference
=============================================
Generate files that present the reference in <dir>/reference.
|#

(deff b-ref-2-hexbytes (ref)
 (b-ref2html 'b-byte2hex ref 0))

(deff b-ref-2-bytes (ref)
 (b-ref2html 'b-byte2dec ref 0))

(deff b-ref-2-kana (ref)
 (b-ref2html 'card2kana ref 0))

(deff b-ref-2-base16 (ref)
 (ct2card* (b-byte*2hex* (card2ref ref))))

(deff b-ref-2-base32 (ref)
 (card-2-base32 ref))

(deff b-ref-2-base64 (ref)
 (card-2-base64 ref))

(deff b-ref-2-decimal (ref)
 (html-dec ref))

(deff b-ref-2-hex (ref)
 (html-hex ref))

(deff b-ref (dir lgt name ref)
 (b-ref-index (cons dir "/index.html") lgt name)
 (b-ref-html  dir "hexbytes"           lgt name (b-ref-2-hexbytes ref))
 (b-ref-html  dir "bytes"              lgt name (b-ref-2-bytes    ref))
 (b-ref-html  dir "kana"               lgt name (b-ref-2-kana     ref))
 (b-ref-html  dir "base16"             lgt name (b-ref-2-base16   ref))
 (b-ref-html  dir "base32"             lgt name (b-ref-2-base32   ref))
 (b-ref-html  dir "base64"             lgt name (b-ref-2-base64   ref))
 (b-ref-html  dir "decimal"            lgt name (b-ref-2-decimal  ref))
 (b-ref-html  dir "hex"                lgt name (b-ref-2-hex      ref))
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/reference/index.html
=============================================
|#

(deff b-ref-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb reference of " name)
   (html-h2 "Logiweb reference of " name)
   (html-p
    (html-link0 "../index.html" "Up"            )
    (html-link0 "hexbytes.html" "Hex bytes"     )
    (html-link0 "bytes.html"    "Decimal bytes" )
    (html-link0 "kana.html"     "Kana bytes"    )
    (html-link0 "base16.html"   "Base16 ref"    )
    (html-link0 "base32.html"   "Base32 ref"    )
    (html-link0 "base64.html"   "Base64 ref"    )
    (html-link0 "decimal.html"  "Decimal number")
    (html-link0 "hex.html"      "Hex number"    )
    (html-link0 (help-link "browser/reference/index") "Help")))))

#|
=============================================
Generate <dir>/reference/<type>.html
=============================================
|#

(deff b-refbyte2html (fct byte index)
 (cons (b-index2space index 5 10) (funcall fct byte)))

(deff b-ref2html (fct ref index)
 (b-ref2html0 fct (card2ref ref) index))

(deff b-ref2html0 (fct ref index)
 (:when (atom ref) nil)
 (:let (byte . ref) ref)
 (:let element (b-refbyte2html fct byte index))
 (:when (< byte 128) (list element html-br (b-ref2html1 fct ref 0)))
 (cons element (b-ref2html fct ref (+ index 1))))

(deff b-ref2html1 (fct ref index)
 (:when (atom ref) nil)
 (:let (byte . ref) ref)
 (:let element (b-refbyte2html fct byte index))
 (:when (= index 19) (list element html-br (b-ref2html2 fct ref 0)))
 (cons element (b-ref2html1 fct ref (+ index 1))))

(deff b-ref2html2 (fct ref index)
 (:when (atom ref) nil)
 (:let (byte . ref) ref)
 (:let element (b-refbyte2html fct byte index))
 (cons element (b-ref2html2 fct ref (+ index 1))))

(deff b-ref-html (dir type lgt name ref)
 (:let file (list dir "/" type ".html"))
 (ct2file file
  (html-page lgt
   (list "Logiweb reference of " name)
   (html-h2 "Logiweb reference of " name " in " type)
   (html-p
    (html-link0 "./index.html" "Up")
    (html-link0 (help-link "browser/reference/index") "Help"))
   (html-p (html-tt f-newline ref)))))

#|
=============================================
Generate <dir>/vector
=============================================
Generate files in <dir>/vector.
|#

(deff b-vector (dir lgt name ref state mirror)
 (:let vector (rack-get state ref card-vector))
 (:let vector (coerce vector 'list))
 (b-vector-index (cons dir "/index.html") lgt name)
 (b-vector-html  dir "hex"                lgt name 'b-byte2hex vector)
 (b-vector-html  dir "decimal"            lgt name 'b-byte2dec vector)
 (b-vector-html  dir "kana"               lgt name 'card2kana  vector)
 (when (or (equalp mirror :mirror) (level '= "submit"))
  (b-vector-lgw   (list dir "/page.lgw")   vector))
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/vector/index.html
=============================================
|#

(deff b-vector-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb vector of " name)
   (html-h2 "Logiweb vector of " name)
   (html-p
    (html-link0 "../index.html" "Up"     )
    (html-link0 "hex.html"      "Hex"    )
    (html-link0 "decimal.html"  "Decimal")
    (html-link0 "kana.html"     "Kana"   )
    (html-link0 "page.lgw"      "Bytes"  )
    (html-link0 (help-link "browser/vector/index") "Help")))))

#|
=============================================
Generate <dir>/vector/<type>.html
=============================================
|#

(deff b-index2space (index block line)
 (:when (= index 0) nil)
 (:when (= (mod index line) 0) html-br)
 (:when (= (mod index block) 0) (cons html-nbsp f-newline))
 #\Space)

(deff b-chop (vector length result)
 (:let rest (nthcdr length vector))
 (:when (null rest) (cons vector result))
 (b-chop rest length (cons (subseq vector 0 length) result)))

(deff b-vector2html2 (fct vector index)
 (:when (null vector) html-br)
 (:let (byte . vector) vector)
 (list
  (b-index2space index 4 16)
  (funcall fct byte)
  (b-vector2html2 fct vector (+ index 1))))

(deff b-vector2html1 (fct vector result)
 (:when (null vector) result)
 (:let (line . vector) vector)
 (:let line (b-vector2html2 fct line 0))
 (b-vector2html1 fct vector (cons line result)))

(deff b-vector2html (fct vector)
 (:let vector (b-chop vector 16 nil))
 (b-vector2html1 fct vector nil))

(deff b-vector-html (dir type lgt name fct vector)
 (:let file (list dir "/" type ".html"))
 (ct2file file
  (html-page lgt
   (list "Logiweb vector of " name)
   (html-h2 "Logiweb vector of " name " in " type)
   (html-p
    (html-link0 "./index.html" "Up")
    (html-link0 (help-link "browser/vector/index") "Help"))
   (html-p
    (html-tt f-newline (b-vector2html fct vector))))))

#|
=============================================
Generate <dir>/vector/page.lgw
=============================================
|#

(deff b-vector-lgw (file vector)
 (ct2file file vector))

#|
=============================================
Generate <dir>/body
=============================================
Generate files that present the body in <dir>/body.
|#

(deff b-body (dir lgt name ref state)
 (:let tree (rack-get state ref card-body))
 (:let tree (if (option "parenthesize") (b-add-parentheses tree state) tree))
 (b-body-name      (cons dir "/lgs.html")    lgt name tree state)
 (b-body-tex       (cons dir "/tex")         lgt tree ref state)
 (b-body-index dir (cons dir "/index.html")  lgt name)
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/body/index.html
=============================================
|#

(deff b-probe-file (&rest file)
 (probe-file (ct2string file)))

(deff b-cond-link (dir link label)
 (:when (null (b-probe-file dir "/" link)) (line label))
 (html-link0 link label))

(deff b-body-index (dir file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb body of " name)
   (html-h2 "Logiweb body of " name)
   (html-p
    (html-link0      "../index.html"   "Up" )
    (b-cond-link dir "tex/page.pdf"    "Pdf")
    (b-cond-link dir "tex/page.html"   "Toc")
    (html-link0      "lgs.html"        "Lgs")
    (html-link0      "tex/lgwdir.html" "TeX")
    (html-link0      (help-link "browser/body/index") "Help"))
   (html-p "Click 'Pdf' to see the page"))))

#|
=============================================
Generate <dir>/body/lgs.html
=============================================
|#

(deff b-body-name (file lgt name tree state)
 (ct2file file
  (html-page lgt
   (list "Logiweb body of " name)
   (html-h2 "Logiweb body of " name " in lgs")
   (html-p
    (html-link0 "./index.html" "Up")
    (html-link0 (help-link "browser/body/index") "Help"))
   (html-p (b-tree2html tree state)))))

#|
=============================================
Generate <dir>/body/tex/lgwinclude.tex
=============================================
(b-lgwinclude dir ref state) generates a lgwinclude.tex file like the following:

\gdef\today{GRD-2005-04-07.UTC:09:05:22.929387}
\gdef\lgwlgt{LGT-4619581554929387e-6}
\gdef\lgwmjdtai{MJD-53467.TAI:09:05:54.929387}
\gdef\lgwgrdutc{GRD-2005-04-07.UTC:09:05:22.929387}
\gdef\lgwmantissa{4619581554929387}
\gdef\lgwexponent{6}
\gdef\lgwfraction{929387}
\gdef\lgwsecond{22}
\gdef\lgwminute{05}
\gdef\lgwhour{09}
\gdef\lgwday{07}
\gdef\lgwmonth{04}
\gdef\lgwyear{2005}
\gdef\lgwCheckKana{nani...}
\gdef\lgwCheckHex{01...}
\gdef\lgwCheckDec{001...}
\gdef\lgwCheckUrl{http://logiweb.eu/logiweb/server/relay/go/06089AAFC2ACDEDDEBF3B64C907231736A27AB7149945E4122F9C4DD7B01/2/}
\gdef\lgwBaseKana{nani...}
\gdef\lgwBaseHex{01...}
\gdef\lgwBaseDec{001...}
\gdef\lgwBaseUrl{http://logiweb.eu/logiweb/server/relay/go/06089AAE8FB6E3C1AB48D4A2C36753A9602A081317AE41132EC76516EE01/2/}
|#

(deff b-tex-def (cmd def)
 (list "\\gdef\\" cmd "{" def "}" #\Newline))

(deff b-tex-def-number (cmd number length)
 (b-tex-def cmd (html-number number length 10)))

(deff b-lgwinclude (dir ref state)
 (:let bib (b-include-bib ref state))
 (:let date (b-include-date ref))
 (ct2file (cons dir "/lgwinclude.tex") (list date bib)))

(deff b-include-bib (ref state)
 (list
  (b-tex-def "lgwrelay" (url-relay))
  (b-tex-def "lgwBreakRelay" (b-break (ct2card* (url-relay))))
  (b-tex-def "lgwbreak" "\\linebreak[0]\\hskip0em plus0.5em{}")
  (b-tex-def "lgwhyphen" "-")
  (b-tex-def "lgwunderscore" "\\_")
  (b-include-ref ref "this")
  (b-include-bib1 (rack-get state ref card-bibliography) state)))

(deff b-include-bib1 (bib state)
 (:when (atom bib) nil)
 (:let (ref . bib) bib)
 (:let name (c-symbol2name ref 0 state))
 (cons (b-include-ref ref name) (b-include-bib1 bib state)))

(deff char-upcase1 (card)
 (char-code (char-upcase (code-char card))))

(deff b-capitalize (card*)
 (:when (atom card*) nil)
 (:let (card . card*) card*)
 (:unless (<= (char-code #\a) card (char-code #\z)) (b-capitalize card*))
 (cons (char-upcase1 card) (b-capitalize1 card*)))

(deff b-capitalize1 (card*)
 (:when (atom card*) nil)
 (:let (card . card*) card*)
 (:unless (<= (char-code #\a) card (char-code #\z)) (b-capitalize card*))
 (cons card (b-capitalize1 card*)))

(etst (ct2ct "AbcDefGhi") (b-capitalize (ct2ct "  abc  def
ghi  ")))

(defc b-sep1 "\\\\
")

(deff b-break (ref)
 (:let (char . ref) ref)
 (:when (null ref) char)
 (list* (b-break1 char) "\\lgwbreak" f-newline (b-break ref)))

(deff b-break1 (char)
 (:when (equalp char (char-code #\_)) "\\lgwunderscore")
 (:when (equalp char (char-code #\-)) "\\lgwhyphen")
 char)

(deff b-tex-base (page name ref)
 (list
  (b-tex-def (list "lgw" page "Block" name) ref)
  (b-tex-def (list "lgw" page "Break" name) (b-break ref))))

(deff b-include-ref (ref name)
 (:let name (b-capitalize (ct2card* name)))
 (:let hex  (ref2kana0 ref b-sep1 "\\quad " #\Space #'b-byte2hex))
 (:let dec  (ref2kana0 ref b-sep1 "\\quad " #\Space #'b-byte2dec))
 (:let kana (ref2kana0 ref b-sep1 "\\quad " #\Space #'card2kana))
 (:let url (url-abs-redirect ref "/"))
 (list
  (b-tex-base name "BaseSixteen"   (b-ref-2-base16   ref))
  (b-tex-base name "BaseThirtyTwo" (b-ref-2-base32   ref))
  (b-tex-base name "BaseSixtyFour" (b-ref-2-base64   ref))
  (b-tex-base name "Dec"           (b-ref-2-decimal  ref))
  (b-tex-base name "Hex"           (b-ref-2-hex      ref))
  (b-tex-def (list "lgw" name "Kana") kana)
  (b-tex-def (list "lgw" name "Hex") hex)
  (b-tex-def (list "lgw" name "Dec") dec)
  (b-tex-def (list "lgw" name "Url") url)))

(deff b-include-date (ref)
 (:let lgt (ref2timestamp ref))
 (:let (m e) lgt)
 (:let (frac exp second minute hour day month year)(lgt-2-grd-utc lgt))
 (list
  (b-tex-def        "today"         (lgt-2-grd-utc-string lgt))
  (b-tex-def        "lgwlgt"        (lgt-2-string lgt))
  (b-tex-def        "lgwmjdtai"     (lgt-2-mjd-tai-string lgt))
  (b-tex-def        "lgwgrdutc"     (lgt-2-grd-utc-string lgt))
  (b-tex-def        "lgwmantissa"   (html-dec m))
  (b-tex-def        "lgwexponent"   (html-dec e))
  (b-tex-def-number "lgwfraction"   frac exp)
  (b-tex-def-number "lgwsecond"     second 2)
  (b-tex-def-number "lgwminute"     minute 2)
  (b-tex-def-number "lgwhour"       hour   2)
  (b-tex-def-number "lgwday"        day    2)
  (b-tex-def-number "lgwmonth"      month  2)
; (b-tex-def-number "lgwyear"       year   2)
  (b-tex-def        "lgwyear"       (html-dec year))))

#|
=============================================
Generate <dir>/body/tex/page.tex
=============================================
|#

(deff b-split-line (card*)
 (:let pos (position f-space card* :test '/=))
 (:when (null pos) nil)
 (:let card* (subseq card* pos))
 (:let pos (position f-space card*))
 (:when (null pos) (list (card*2string card*)))
 (:let item (card*2string (subseq card* 0 pos)))
 (:let item* (b-split-line (subseq card* (+ pos 1))))
 (cons item item*))

(etst (b-split-line (ct2ct "ab cd ef")) '("ab" "cd" "ef"))
(etst (b-split-line (ct2ct "ab   cd ef")) '("ab" "cd" "ef"))
(etst (b-split-line (ct2ct "  ab cd ef")) '("ab" "cd" "ef"))
(etst (b-split-line (ct2ct "ab cd ef  ")) '("ab" "cd" "ef"))
(ntst (b-split-line (ct2ct "")))
(ntst (b-split-line (ct2ct "   ")))

(deff b-prefix (x y)
 (:when (atom x) t)
 (:when (atom y) nil)
 (:when (unequal (car x) (car y)) nil)
 (b-prefix (cdr x) (cdr y)))

(ttst (b-prefix '(1 2) '(1 2 3)))
(ttst (b-prefix '(1 2) '(1 2)))
(ntst (b-prefix '(1 2) '(1)))
(ntst (b-prefix '(1 2) '(1 3)))

; Association list of functions which need special treatment

(defc b-exec-assoc '(
  ("text"      . exec-text     )
  ("binary"    . exec-binary   )
; ("tex"       . exec-tex      )
; ("latex"     . exec-latex    )
; ("bibtex"    . exec-bibtex   )
  ("makeindex" . exec-makeindex)
  ("dvipdfm"   . exec-dvipdfm  )
  ("mizf"      . exec-mizf     )
))

(deff legal-file-char-p (x)
 (or
  (char<= #\a x #\z)
  (char<= #\A x #\Z)
  (char<= #\0 x #\9)
  (char= x #\.)
  (char= x #\-)
  (char= x #\_)))

(deff b-parse-arg (arg*)
 (:when (atom arg*) (format t "Missing file name~%"))
 (:let (file . rest) arg*)
 (:let file (ct2string file))
 (:let char (find-if-not 'legal-file-char-p file))
 (:when (null char) (cons file rest))
 (format t "Illegal character ~s in file name ~s~%" char file))

(deff b-body-exec (dir ct)
 (:when (atom ct) nil)
 (:let (cmd . arg*) ct)
 (:unless (intp cmd) (b-body-exec-subtree dir ct))
 (:let cmd (card2vector cmd))
 (:let cmd (vector2string cmd))
 (:unless (member cmd (option "renderers") :test 'equal)
  (b-body-exec-subtree dir ct))
 (:let arg* (b-parse-arg arg*))
 (:when (null arg*) nil)
 (:let (:key . fct) (assoc cmd b-exec-assoc :test 'equal))
 (:when fct (funcall fct dir arg*))
 (exec-silent dir cmd arg*))

(deff b-body-exec-subtree (dir ct)
 (b-body-exec dir (car ct))
 (b-body-exec dir (cdr ct)))

(deff exec-text (dir arg*)
 (:let (file . contents) arg*)
 (format t "Extracting ~a~%" file)
 (ct2file (list* dir "/" file) contents))

(deff exec-binary (dir arg*)
 (:let (file . contents) arg*)
 (format t "Extracting ~a~%" file)
 (ct2file (list* dir "/" file) contents))

(deff b-body-tex1 (ref dir tree state)
 (:let ((sref sid)) tree)
 (:when (unequal sref 0) (b-body-exec dir (b-tree2tex ref tree state)))
 (ct2file (list* dir "/page.tex") (card2vector sid))
 (exec-latex dir '("page"))
 (exec-latex dir '("page"))
 (exec-latex dir '("page"))
 (exec-dvipdfm dir '("page")))

(deff b-body-tex (dir lgt tree ref state)
 (clean-dir dir)
 (b-lgwinclude dir ref state)
 (format t "Generating TeX face of body~%")
 (b-body-tex1 ref dir tree state)
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/bibliography
=============================================
Generate files that present the bibliography in <dir>/bibliography.
|#

(deff b-bib (dir lgt name ref state)
 (:let bib (rack-get state ref card-bibliography))
 (b-bib-index  (cons dir "/index.html") lgt name)
 (b-bib-html  dir "hexbytes"            lgt name 'b-ref-2-hexbytes bib)
 (b-bib-html  dir "bytes"               lgt name 'b-ref-2-bytes    bib)
 (b-bib-html  dir "kana"                lgt name 'b-ref-2-kana     bib)
 (b-bib-html  dir "base16"              lgt name 'b-ref-2-base16   bib)
 (b-bib-html  dir "base32"              lgt name 'b-ref-2-base32   bib)
 (b-bib-html  dir "base64"              lgt name 'b-ref-2-base64   bib)
 (b-bib-html  dir "decimal"             lgt name 'b-ref-2-decimal  bib)
 (b-bib-html  dir "hex"                 lgt name 'b-ref-2-hex      bib)
 (b-bib-lgs    (cons dir "/lgs.html")   lgt name bib state)
 (b-bib-tex    (cons dir "/tex")        lgt ref  bib state)
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/bibliography/index.html
=============================================
|#

(deff b-bib-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb bibliography of " name)
   (html-h2 "Logiweb bibliography of " name)
   (html-p
    (html-link0 "../index.html"   "Up"            )
    (html-link0 "tex/page.pdf"    "Pdf"           )
    (html-link0 "lgs.html"        "Lgs"           )
    (html-link0 "tex/lgwdir.html" "TeX"           )
    (html-link0 "hexbytes.html"   "Hex bytes"     )
    (html-link0 "bytes.html"      "Decimal bytes" )
    (html-link0 "kana.html"       "Kana bytes"    )
    (html-link0 "base16.html"     "Base16 ref"    )
    (html-link0 "base32.html"     "Base32 ref"    )
    (html-link0 "base64.html"     "Base64 ref"    )
    (html-link0 "decimal.html"    "Decimal number")
    (html-link0 "hex.html"        "Hex number"    )
    (html-link0 (help-link "browser/bibliography/index") "Help")))))

#|
=============================================
Generate <dir>/bibliography/<type>.html
=============================================
|#

(deff b-bibref (ref path index &rest caption)
 (:when (= index 0) (html-ref "../index.html" caption))
 (html-ref (url-redirect ref path) caption))

(deff b-bib2html (fct bib index)
 (:when (atom bib) nil)
 (:let (ref . bib) bib)
 (:let element (html-tt f-newline (funcall fct ref)))
 (:let rest (when bib (list html-br html-br (b-bib2html fct bib (+ index 1)))))
 (:let refnum
  (b-bibref ref "/index.html" index "Reference " (html-dec index)))
 (list refnum html-br element rest))

(deff b-bib-html (dir type lgt name fct bib)
 (:let file (list dir "/" type ".html"))
 (ct2file file
  (html-page lgt
   (list "Logiweb bibliography of " name)
   (html-h2 "Logiweb bibliography of " name " in " type)
   (html-p
    (html-link0 "./index.html" "Up")
    (html-link0 (help-link "browser/bibliography/index") "Help"))
   (html-p
    (html-tt (b-bib2html fct bib 0))))))

#|
=============================================
Generate <dir>/bibliography/lgs.html
=============================================
|#

(deff b-bib-lgs (file lgt name bib state)
 (ct2file file
  (html-page lgt
   (list "Logiweb bibliography of " name)
   (html-h2 "Logiweb bibliography of " name " in lgs")
   (html-p
    (html-link0 "./index.html" "Up")
    (html-link0 (help-link "browser/bibliography/index") "Help"))
   (html-p
    (b-bib-lgs1 bib 0 state)))))

(deff b-bib-lgs1 (bib index state)
 (:when (atom bib) nil)
 (:let (ref . bib) bib)
 (:let number (list "[" (html-dec index) "]"))
 (:let name (utf2html (c-symbol2name ref 0 state)))
 (:let link (b-bibref ref "/index.html" index number))
 (:let line (list link " " name))
 (list line html-br (b-bib-lgs1 bib (+ index 1) state)))

#|
=============================================
Generate <dir>/bibliography/tex/page.tex
=============================================
|#

(deff b-bib-body (bib name state)
 (:let base (slash (ct2string *hyperbaseurl*)))
 (list
  (tex-title "Logiweb bibliography of " name)
  (tex-paragraph
   (tex-link (list base "bibliography/index.html") "Up")
;  (tex-link "../index.html" "Up")
   (tex-link (help-abs-link "browser/bibliography/index") "Help"))
  (b-bib-body1 0 bib state)))

(deff b-bib-body1 (index bib state)
 (:when (atom bib) nil)
 (:let (ref . bib) bib)
 (cons
  (b-bib-body2 index ref state)
  (b-bib-body1 (+ index 1) bib state)))

(deff b-bib-body2 (index ref state)
 (:let tex (b-ref2tex ref state))
 (tex-paragraph
  (line (b-tex-bibref ref "/index.html" index "[" (html-dec index) "]"))
  tex))

(deff b-tex-bibref (ref path index &rest caption)
 (:let base (slash (ct2string *hyperbaseurl*)))
 (:when (= index 0) (tex-ref (list base "index.html") caption))
;(:when (= index 0) (tex-ref "../../index.html" caption))
 (tex-ref (url-abs-redirect ref path) caption))

(deff b-bib-tex (dir lgt ref bib state)
 (when (verbose '> 0) (format t "TeX face of bibliography~%"))
 (:let tex (b-ref2tex ref state))
 (:let body (b-bib-body bib tex state))
 (:let name (c-symbol2name ref 0 state))
 (:let title (list "Logiweb bibliography of " name))
 (:let ct (tex-page "/bibliography/tex/page.tex" lgt title body))
 (clean-dir dir)
 (ct2file (cons dir "/page.tex") ct)
 (when (verbose '> 0) (format t "TeX face of bibliography: Executing TeX~%"))
 (exec-latex dir '("page"))
 (when (verbose '> 0)
  (format t "TeX face of bibliography: Executing dvipdfm~%"))
 (exec-dvipdfm dir '("page"))
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/dictionary
=============================================
Generate files that present the dictionary in <dir>/dictionary.
|#

(deff b-dictionary (dir lgt name ref state)
 (b-dictionary-index (cons dir "/index.html") lgt name)
 (b-dictionary-lgs   (cons dir "/lgs.html")   lgt name ref state)
 (b-dictionary-tex   (cons dir "/tex")        lgt ref state)
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/dictionary/index.html
=============================================
|#

(deff b-dictionary-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb dictionary of " name)
   (html-h2 "Logiweb dictionary of " name)
   (html-p
    (html-link0 "../index.html"   "Up" )
    (html-link0 "tex/page.pdf"    "Pdf")
    (html-link0 "lgs.html"        "Lgs")
    (html-link0 "tex/lgwdir.html" "TeX")
    (html-link0 (help-link "browser/dictionary/index") "Help")))))

#|
=============================================
Generate <dir>/dictionary/lgs.html
=============================================
|#

(deff b-dictionary-lgs (file lgt name ref state)
 (:let dictionary (rack-get state ref card-dictionary))
 (:let dictionary (array2assoc-ascending dictionary))
 (ct2file file
  (html-page lgt
   (list "Logiweb dictionary of " name)
   (html-h2 "Logiweb dictionary of " name)
   (html-p
    (html-link0 "./index.html" "Up")
    (html-link0 (help-link "browser/dictionary/index") "Help")
   (html-p
    (html-table "0"
;    (html-row-header "Id" "Arity" "Lgs")
     (b-dictionary-lgs1 ref dictionary state)))))))

(deff b-dictionary-lgs1 (ref dictionary state)
 (:when (null dictionary) nil)
 (:let ((id . arity) . dictionary) dictionary)
 (cons
  (b-dictionary-lgs2 ref id arity state)
  (b-dictionary-lgs1 ref dictionary state)))

(deff b-dictionary-lgs2 (ref id arity state)
 (html-simple-row
  (html-right (html-dec id))
  (html-right (html-dec arity))
  (html-left (utf2html (b-tree2lgs (cons (list ref id) :quote) state)))))

#|
=============================================
Generate <dir>/dictionary/tex/page.tex
=============================================
|#

(deff b-dictionary-entry (id arity name)
 (tex-paragraph
  (line "\\makebox[\\idlength][r]{" id "}%")
  (line "\\makebox[0.5em]{}%")
  (line "\\makebox[\\aritylength][r]{" arity "}%")
  (line "\\makebox[0.5em]{}%")
  name))

(deff b-dictionary-body (ref tex dictionary state)
 (:let max-id (max 99 (caar dictionary)))
 (:let max-id (html-dec max-id))
 (:let dictionary (reverse dictionary))
 (list
  (tex-title "Logiweb dictionary of " tex)
  (tex-paragraph
   (tex-link "../index.html" "Up")
   (tex-link (help-link "browser/dictionary/index") "Help"))
  (line "\\newlength{\\idlength}")
  (line "\\settowidth{\\idlength}{" max-id "}")
  (line "\\newlength{\\aritylength}")
  (line "\\settowidth{\\aritylength}{99}")
  (line "\\vspace{2ex}")
; (tex-paragraph
;  (b-dictionary-entry "{\\bf Id}" "{\\bf Arity}" "{\\bf Name}"))
  (b-dictionary-body1 ref dictionary state)))

(deff b-dictionary-body1 (ref dictionary state)
 (:when (atom dictionary) nil)
 (:let ((id . arity) . dictionary) dictionary)
 (cons
  (b-dictionary-body2 ref id arity state)
  (b-dictionary-body1 ref dictionary state)))

(deff b-dictionary-body2 (ref id arity state)
 (:let tex (b-symbol2tex ref id state))
 (b-dictionary-entry
  (html-dec id)
  (html-dec arity)
  tex))

(deff b-dictionary-tex (dir lgt ref state)
 (:let dictionary (array2assoc (rack-get state ref card-dictionary)))
 (when (verbose '> 0) (format t "TeX face of dictionary~%"))
 (:let tex (b-ref2tex ref state))
 (:let name (c-symbol2name ref 0 state))
 (:let title (list "Logiweb dictionary of " name))
 (:let body (b-dictionary-body ref tex dictionary state))
 (:let ct (tex-page "/dictionary/tex/page.tex" lgt title body))
 (clean-dir dir)
 (ct2file (cons dir "/page.tex") ct)
 (when (verbose '> 0) (format t "TeX face of dictionary: executing TeX~%"))
 (exec-latex dir '("page"))
 (when (verbose '> 0) (format t "TeX face of dictionary: Executing dvipdfm~%"))
 (exec-dvipdfm dir '("page"))
 (html-dir lgt dir))

#|
=============================================
Generate external formast in <dir>/vector
=============================================
|#

(deff b-external (dir lgt name ref state)
 (b-external-index (cons dir "/external.html") lgt name)
 (b-lisp (cons dir "/page.lisp") ref state)
 (b-xml (cons dir "/page.xml") ref state))

#|
=============================================
Generate <dir>/codex/index.html
=============================================
|#

(deff b-external-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb external formats of " name)
   (html-h2 "Logiweb external formats of " name)
   (html-p
    (html-link0 "../index.html"    "Up" )
    (html-link0 "page.lisp"        "Common Lisp S-expression")
    (html-link0 "page.xml"         "XML")
    (html-link0 (help-link "browser/external/index") "Help")))))

#|
=============================================
Generate <dir>/vector/page.lisp
=============================================
Generate a lisp representation of bibliography, dictionary, and body in <dir>/vector/page.lisp.
|#

(deff b-lisp (file ref state)
 (:let file (ct2string file))
 (ensure-directories-exist file)
 (with-open-file
  (stream file
   :direction :output
   :if-exists :supersede)
  (b-lisp1 stream ref state)))

(defc b-lpar #\() (defc b-rpar #\))

(deff b-lisp1 (s ref state)
 (:let bib (rack-get state ref card-bibliography))
 (:let dict (rack-get state ref card-dictionary))
 (:let body (rack-get state ref card-body))
 (format s "~c~c" b-lpar b-lpar)
 (b-lisp-bib s 0 bib)
 (format s "~c~%" b-rpar)
 (format s " ~c" b-lpar)
 (b-lisp-dict s (array2assoc-ascending dict))
 (format s "~c~%" b-rpar)
 (b-lisp-tree s bib body)
 (format s "~c~%" b-rpar))

(deff b-lisp-bib (s index bib)
 (:let (ref . bib) bib)
 (format s "#~d=~d" index ref)
 (:when (atom bib) nil)
 (format s "~%  ")
 (b-lisp-bib s (+ index 1) bib))

(deff b-lisp-dict (s dict)
 (:let (item . dict) dict)
 (format s "~s" item)
 (:when (atom dict) nil)
 (format s "~%  ")
 (b-lisp-dict s dict))

(deff b-lisp-tree (s bib tree)
 (:let ((ref id) . tree*) tree)
 (:when (equalp ref 0) (format s " ((0 ~s))" id))
 (format s " ~c(#~d# ~d)" b-lpar (position ref bib) id)
 (b-lisp-tree* s bib tree*)
 (format s "~c" b-rpar))

(deff b-lisp-tree* (s bib tree*)
 (:when (atom tree*) nil)
 (:let (tree . tree*) tree*)
 (format s "~%")
 (b-lisp-tree s bib tree)
 (b-lisp-tree* s bib tree*))

#|
=============================================
Generate <dir>/vector/page.xml
=============================================
Generate an xml representation of bibliography, dictionary, and body in <dir>/vector/page.xml.
|#

(defc b-xml-part1
"<?xml version='1.0'?>
<!DOCTYPE page [

  <!ELEMENT page         (bibliography,dictionary,tree)>
  <!-- One unpacked Logiweb page -->

  <!ELEMENT bibliography (reference*)>
  <!--
    The reference of the page followed by the list of references
    to other Logiweb pages.
  -->

  <!ELEMENT reference    (#PCDATA)>
  <!--
    One reference to a Logiweb page.
    PCDATA shall be a natural number expressed in decimal.
  -->

  <!ELEMENT dictionary   (declaration*)>
  <!-- The list of symbols declared on the page -->

  <!ELEMENT declaration  EMPTY>
  <!ATTLIST declaration  idx       CDATA #REQUIRED>
  <!ATTLIST declaration  arity     CDATA #REQUIRED>
  <!--
    The declaration of one symbol.
    The idx is the index of the symbol being defined.
    The arity is the arity of the symbol being defined.
    The idx and arity shall be natural numbers expressed in decimal.
  -->

  <!ELEMENT tree         (tree*)>
  <!ATTLIST tree         ref       CDATA #REQUIRED>
  <!ATTLIST tree         idx       CDATA #REQUIRED>
  <!--
    One tree.
    The ref and idx represents one symbol or one string.

    If ref='s' then the idx must be a list of bytes.
    The bytes shall be expressed in decimal and separated by spaces.

    Otherwise, ref and idx shall be natural numbers expressed in decimal.
    The value of ref shall be less then the number of references
    in the bibliography.

    If ref='0' then the idx must be mentioned in the dictionary.
    In this case, the number of subtrees of the tree shall be equal to
    the arity associated to the given idx.

    If ref = n > 0 then the idx must be mentioned in the dictionary
    of the n'th reference (where the first reference in the bibliography
    is reference number zero) and the number of subtrees shall be equal
    to the arity associated to the given idx on the referenced page.
  -->
]>

<page>
 <bibliography>")

(defc b-xml-part2 "
 </bibliography>
 <dictionary>")

(defc b-xml-part3 "
 </dictionary>
")

(defc b-xml-part4 "
</page>
")

(deff b-xml (file ref state)
 (:let file (ct2string file))
 (ensure-directories-exist file)
 (with-open-file
  (stream file
   :direction :output
   :if-exists :supersede)
  (b-xml1 stream ref state)))

(deff b-xml1 (s ref state)
 (:let bib (rack-get state ref card-bibliography))
 (:let dict (rack-get state ref card-dictionary))
 (:let body (rack-get state ref card-body))
 (format s "~a" b-xml-part1)
 (b-xml-bib s bib)
 (format s "~a" b-xml-part2)
 (b-xml-dict s (array2assoc-ascending dict))
 (format s "~a" b-xml-part3)
 (b-xml-tree s bib body)
 (format s "~a" b-xml-part4))

(deff b-xml-bib (s bib)
 (:when (atom bib) nil)
 (:let (ref . bib) bib)
 (format s "~%  <reference>~%   ~a~%  </reference>" ref)
 (b-xml-bib s bib))

(deff b-xml-dict (s dict)
 (:when (atom dict) nil)
 (:let ((idx . arity) . dict) dict)
 (format s "~%  <declaration idx='~a' arity='~a'/>" idx arity)
 (b-xml-dict s dict))

(deff b-xml-tree (s bib tree)
 (:let ((ref id) . tree*) tree)
 (:when (equalp ref 0) (b-xml-string s id))
 (format s "~% <tree ref='~a' idx='~a'>" (position ref bib) id)
 (b-xml-tree* s bib tree*)
 (format s "~% </tree>"))

(deff b-xml-tree* (s bib tree*)
 (:when (atom tree*) nil)
 (:let (tree . tree*) tree*)
 (b-xml-tree s bib tree)
 (b-xml-tree* s bib tree*))

(deff b-xml-string (s id)
 (format s "~% <tree ref='bytes' idx='")
 (b-xml-string1 s 0 id)
 (format s "'/>"))

(deff b-xml-string1 (s n id)
 (:when (>= n (length id)) nil)
 (when (= (mod n 4) 0) (format s (if (= (mod n 16) 0) "~% " " ")))
 (format s " ~3d" (aref id n))
 (b-xml-string1 s (+ n 1) id))

#|
=============================================
Generate <dir>/codex
=============================================
Generate files that present the codex in <dir>/codex.
|#

(deff b-codex (dir lgt name ref state)
 (b-codex-index  (cons dir "/index.html") lgt name)
 (b-codex-lgs    (cons dir "/lgs")        lgt name ref state)
 (b-codex-tex    (cons dir "/tex")        lgt ref state)
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/codex/index.html
=============================================
|#

(deff b-codex-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb codex of " name)
   (html-h2 "Logiweb codex of " name)
   (html-p
    (html-link0 "../index.html"    "Up" )
    (html-link0 "tex/page.pdf"     "Pdf")
    (html-link0 "lgs/index.html"   "Lgs")
    (html-link0 "tex/lgwdir.html"  "TeX")
    (html-link0 (help-link "browser/codex/index") "Help")))))

#|
=============================================
Generate <dir>/codex/lgs
=============================================
|#

#|
Generate page header and pass control to b-codex-lgs1 for generation of page contents and subpages.
|#
(deff b-codex-lgs (dir lgt name ref state)
 (:let trie (array2assoc-ascending (rack-get state ref card-codex)))
 (:let bib (rack-get state ref card-bibliography))
 (ct2file (cons dir "/index.html")
  (html-page lgt
   (list "Logiweb codex of " name)
   (html-h2 "Logiweb codex of " name " in lgs")
   (html-p
    (html-link0 "../index.html" "Up")
    (html-link0 (help-link "browser/codex/index") "Help"))
   (html-p (html-table "0" (b-codex-lgs1 dir lgt bib trie state))))))

#|
Traverse symbol references. Pass control to b-codex-lgs2 for handling each symbol reference.
|#
(deff b-codex-lgs1 (dir lgt bib trie state)
 (:when (atom trie) nil)
 (:let ((sref . subtrie) . trie) trie)
 (cons
  (b-codex-lgs2 dir lgt bib sref (array2assoc-ascending subtrie) state)
  (b-codex-lgs1 dir lgt bib trie state)))

#|
Traverse symbol ids. Pass control to b-codex-lgs3 for handling each symbol.
|#
(deff b-codex-lgs2 (dir lgt bib sref trie state)
 (:when (null trie) nil)
 (:let ((sid . subtrie) . trie) trie)
 (cons
  (b-codex-lgs3 dir lgt bib sref sid subtrie state)
  (b-codex-lgs2 dir lgt bib sref trie state)))

#|
Generate reference to the subpage concerning the symbol that has the given symbol reference and symbol id. Pass control to b-codex-lgs4 in order to generate a subpage for the symbol.
|#
(deff b-codex-symbol (sref sid bib)
 (:when (equalp sref 0) (list "string-" (card2vector sid)))
 (:let relref (position sref bib))
 (:when (null relref) (error "Unknown reference~%"))
 (:let relref (html-dec relref))
 (:let id (html-dec sid))
 (list "ref-" relref "-id-" id))

(deff b-codex-lgs3 (dir lgt bib sref sid subtrie state)
 (:let symbol (b-codex-symbol sref sid bib))
 (:let file (list symbol ".html"))
 (:let name (utf2html (c-symbol2name sref sid state)))
 (b-codex-lgs4 (list dir "/" file) lgt name subtrie state)
 (html-row symbol (html-ref file name)))

#|
Generate page header for the subpage for the given symbol and pass control to b-codex-lgs5 for generation of subpage contents.
|#
(deff b-codex-lgs4 (file lgt name trie state)
 (ct2file file
  (html-page lgt
   (list "Logiweb aspects of " name)
   (html-h2 "Logiweb aspects of " name " in lgs")
   (html-p
    (html-link0 "./index.html" "Up" )
    (html-link0 (help-link "browser/codex/index") "Help"))
   (b-codex-lgs5 (array2assoc-ascending trie) state))))

#|
Traverse aspect references. Pass control to b-codex-lgs6 for handling each aspect reference.
|#
(deff b-codex-lgs5 (trie state)
 (:when (atom trie) nil)
 (:let ((aref . subtrie) . trie) trie)
 (cons
  (b-codex-lgs6 aref (array2assoc-ascending subtrie) state)
  (b-codex-lgs5 trie state)))

#|
Traverse aspect ids. Pass control to b-codex-lgs7 for handling each aspect.
|#
(deff b-codex-lgs6 (aref trie state)
 (:when (atom trie) nil)
 (:let ((aid . tree) . trie) trie)
 (cons
  (b-codex-lgs7 aref aid tree state)
  (b-codex-lgs6 aref trie state)))

#|
Generate header and body for the given aspect.
|#
(deff b-codex-lgs7 (aref aid tree state)
 (:let name (utf2html (c-symbol2name aref aid state)))
 (:let header (html-h3 "The " (html-string name) " aspect"))
 (:let tree (b-add-parentheses tree state))
 (:let ((sref sid)) tree)
 (:when (unequal sref 0)
  (list header (html-p (utf2html (b-tree2lgs tree state)))))
 (:let vector (card2vector sid))
 (:let string (vector2string vector))
 (list header (html-p "Proclaimed as " (html-string string))))

#|
=============================================
Generate <dir>/codex/tex/page.tex
=============================================
|#

(deff b-codex-tex (dir lgt ref state)
 (:let trie (array2assoc (rack-get state ref card-codex)))
 (when (verbose '> 0) (format t "TeX face of codex~%"))
 (:let tex (b-ref2tex ref state))
 (:let name (c-symbol2name ref 0 state))
 (:let title (list "Logiweb codex of " name))
 (:let body (b-codex-tex0 trie tex state))
 (:let ct (tex-page "/codex/tex/page.tex" lgt title body))
 (clean-dir dir)
 (ct2file (cons dir "/page.tex") ct)
 (when (verbose '> 0) (format t "TeX face of codex: Executing TeX~%"))
 (exec-latex dir '("page"))
 (when (verbose '> 0) (format t "TeX face of codex: Executing dvipdfm~%"))
 (exec-dvipdfm dir '("page"))
 (html-dir lgt dir))

#|
Collect page from header, table of contents, and body
|#
(deff b-codex-tex0 (trie tex state)
 (list
  (tex-title "Logiweb codex of " tex)
  (tex-paragraph
   (tex-link "../index.html" "Up")
   (tex-link (help-link "browser/codex/index") "Help"))
  (tex-paragraph (tex-left (b-codex-toc1 trie state nil)))
  (b-codex-tex1 trie state nil)))

#|
Traverse symbol references. Pass control to b-codex-tex2 for handling each symbol reference. Return table of contents.
|#
(deff b-codex-toc1 (trie state result)
 (:when (null trie) result)
 (:let ((ref . subtrie) . trie) trie)
 (:let result (b-codex-toc2 ref (array2assoc subtrie) state result))
 (:let result (b-codex-toc1 trie state result))
 result)

#|
Traverse symbol ids. Pass control to b-codex-toc3 for handling each symbol. Return table of contents.
|#
(deff b-codex-toc2 (ref trie state result)
 (:when (null trie) result)
 (:let ((id) . trie) trie)
 (:let result (cons (b-codex-toc3 ref id state) result))
 (:let result (b-codex-toc2 ref trie state result))
 result)

#|
Generate entries of table of contents
|#
(deff b-codex-toc3 (ref id state)
 (:let tex (b-symbol2tex ref id state))
 (line (tex-localref (symbol-anchor ref id) tex) ","))

#|
Traverse symbol references. Pass control to b-codex-tex2 for handling each symbol reference.
|#
(deff b-codex-tex1 (trie state result)
 (:when (null trie) result)
 (:let ((ref . subtrie) . trie) trie)
 (:let result (b-codex-tex2 ref (array2assoc subtrie) state result))
 (:let result (b-codex-tex1 trie state result))
 result)

#|
Traverse symbol ids. Pass control to b-codex-tex3 for handling each symbol.
|#
(deff b-codex-tex2 (ref trie state result)
 (:when (null trie) result)
 (:let ((id . subtrie) . trie) trie)
 (:let result (cons (b-codex-tex3 ref id (array2assoc subtrie) state) result))
 (:let result (b-codex-tex2 ref trie state result))
 result)

#|
Generate section header for the given symbol. Pass control to b-codex-tex4 in order to generate the body of the section.
|#
(deff b-codex-tex3 (ref id trie state)
 (:let tex (b-symbol2tex ref id state))
 (list
  (tex-anchor (symbol-anchor ref id))
  (tex-section* tex)
  (b-codex-tex4 ref trie state nil)))

#|
Traverse aspect references. Pass control to b-codex-tex5 for handling each aspect reference.
|#
(deff b-codex-tex4 (ref trie state result)
 (:when (null trie) (reverse result))
 (:let ((nil . subtrie) . trie) trie)
 (:let result (b-codex-tex5 ref (array2assoc subtrie) state result))
 (:let result (b-codex-tex4 ref trie state result))
 result)

#|
Traverse aspect ids. Pass control to b-codex-tex6 for handling each aspect.
|#
(deff b-codex-tex5 (ref trie state result)
 (:when (null trie) result)
 (:let ((nil . subtrie) . trie) trie)
 (:let result (cons (b-codex-tex6 ref subtrie state) result))
 (:let result (b-codex-tex5 ref trie state result))
 result)

#|
Generate the given aspect.
|#

(deff b-codex-tex6 (ref tree state)
 (:let text (b-tree2tex ref (b-add-parentheses tree state) state))
 (:when (unequal (caar tree) 0) (tex-paragraph "$\\rm" text "$"))
 (tex-paragraph "Proclaimed as ``" text "''"))

#|
=============================================
Generate <dir>/expansion
=============================================
Generate files that present the expansion in <dir>/expansion.
|#

(deff b-expansion (dir lgt name ref state)
 (:let tree (rack-get state ref card-expansion))
 (b-expansion-lgs       (cons dir "/lgs.html")    lgt name tree state)
 (b-expansion-tex       (cons dir "/tex")         lgt tree ref state)
 (b-expansion-index dir (cons dir "/index.html")  lgt name)
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/expansion/index.html
=============================================
|#

(deff b-expansion-index (dir file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb expansion of " name)
   (html-h2 "Logiweb expansion of " name)
   (html-p
    (html-link0      "../index.html"   "Up" )
    (b-cond-link dir "tex/page.pdf"    "Pdf")
    (b-cond-link dir "tex/page.html"   "Toc")
    (html-link0      "lgs.html"        "Lgs")
    (html-link0      "tex/lgwdir.html" "TeX")
    (html-link0      (help-link "browser/expansion/index") "Help")))))

#|
=============================================
Generate <dir>/expansion/lgs.html
=============================================
|#

(deff b-expansion-lgs (file lgt name tree state)
 (ct2file file
  (html-page lgt
   (list "Logiweb expansion of " name)
   (html-h2 "Logiweb expansion of " name " in lgs")
   (html-p
    (html-link0 "./index.html" "Up")
    (html-link0 (help-link "browser/expansion/index") "Help"))
   (html-p (b-tree2html tree state)))))

#|
=============================================
Generate <dir>/expansion/tex/page.tex
=============================================
|#

(deff b-expansion-tex (dir lgt tree ref state)
 (clean-dir dir)
 (b-lgwinclude dir ref state)
 (format t "Generating TeX face of expansion~%")
 (b-body-tex1 ref dir tree state)
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/header
=============================================
Generate files that present the header in <dir>/header.
|#

(deff b-header (dir :lgt :name ref state)
 (:let name (c-symbol2name ref 0 state))
 (:let page (list f-newline #\" name #\" #\Space))
 (:let prio (codex-get state ref 0 0 card-priority))
 (:let rhs (fourth prio))
 (:let header (b-header1 page ref rhs state nil))
 (ct2file (cons dir "/header.lgs") header))

(defc vector-pre       (card2vector card-pre      ))
(defc vector-post      (card2vector card-post     ))

(defc b-header-assoc
 (list
  (cons vector-pre  "PREASSOCIATIVE")
  (cons vector-post "POSTASSOCIATIVE")))

(deff b-header1 (page home tree state result)
 (:let ((ref id) section tree) tree)
 (:when (atom tree) (reverse result))
 (:let (:def :prio :lhs ((sref sid))) (codex-get state ref id 0 card-priority))
 (:when (unequal sref 0) (reverse result))
 (:let prio (assoc-get (card2vector sid) b-header-assoc))
 (:when (null prio) (reverse result))
 (b-header2 page home prio section tree state result))

(deff b-header2 (page home prio section tree state result)
 (:when (atom section) (b-header1 page home tree state result))
 (:let (:symbol construct section) section)
 (:when (atom construct) (b-header1 page home tree state result))
 (:let ((ref id) . :args) construct)
 (:when (unequal ref home) (b-header2 page home prio section tree state result))
 (:let construct (c-symbol2name ref id state))
 (:let result (list* f-newline construct page prio result))
 (b-header1 page home tree state result))

#|
=============================================
Generate <dir>/diagnose
=============================================
Generate files that present the diagnose (if any) in <dir>/diagnose.
|#

(deff b-diagnose (dir lgt name ref state)
 (:let tree (untag (rack-get state ref card-diagnose)))
 (:let tree (b-add-parentheses tree state))
 (b-diagnose-lgs dir lgt name tree state)
 (b-diagnose-tex (cons dir "/tex") lgt ref tree state)
 (:when (null tree) (b-noerror-index (cons dir "/index.html") lgt name))
 (b-diagnose-index (cons dir "/index.html") lgt name)
 (html-dir lgt dir))

(deff diagnose (&key (depth nil) (length nil))
 (spy1 (default -1 depth) length *diagnose* *spy-state*)
 (values))

#|
=============================================
Generate <dir>/diagnose/index.html
when no errors are found
=============================================
|#

(deff b-noerror-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb diagnose of " name)
   (html-h2 "Logiweb diagnose of " name)
   (html-p
    (html-link0 "../index.html" "Up" )
    (html-link0 (help-link "browser/diagnose/index") "Help"))
   (html-h3 "No errors found"))))

#|
=============================================
Generate <dir>/diagnose/index.html
=============================================
|#

(deff b-diagnose-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb diagnose of " name)
   (html-h2 "Logiweb diagnose of " name)
   (html-p
    (html-link0 "../index.html"   "Up" )
    (html-link0 "tex/page.pdf"    "Pdf")
    (html-link0 "lgs.html"        "Lgs as html")
    (html-link0 "diag.lgs"        "Lgs as utf8")
    (html-link0 "tex/lgwdir.html" "TeX")
    (html-link0 (help-link "browser/diagnose/index") "Help")))))

#|
=============================================
Generate <dir>/diagnose/lgs.html
=============================================
|#

(deff b-diagnose-lgs (dir lgt name tree state)
 (:let lgs (when tree (b-tree2lgs tree state)))
 (:let body (default "No errors found" lgs))
 (ct2file (cons dir "/diag.lgs") (when lgs (cons lgs f-newline)))
 (ct2file (cons dir "/lgs.html")
  (html-page lgt
   (list "Logiweb diagnose of " name)
   (html-h2 "Logiweb diagnose of " name " in lgs")
   (html-p
    (html-link0 "./index.html" "Up")
    (html-link0 (help-link "browser/diagnose/index") "Help"))
   (html-p (utf2html body)))))

#|
=============================================
Generate <dir>/diagnose/tex/page.tex
=============================================
|#

(deff diagnose2tex (ref diagnose state)
 (:when (null diagnose) "No errors found")
 (b-tree2tex ref diagnose state))

(deff b-diagnose-body (ref tex diagnose state)
 (:let diagnose (diagnose2tex ref diagnose state))
 (list
  (tex-title "Logiweb diagnose of " tex)
  (tex-paragraph
   (tex-link "../index.html" "Up")
   (tex-link (help-link "browser/diagnose/index") "Help"))
  (tex-paragraph diagnose)))

(deff b-diagnose-tex (dir lgt ref diagnose state)
 (when (verbose '> 0) (format t "TeX face of diagnose~%"))
 (:let tex (b-ref2tex ref state))
 (:let name (c-symbol2name ref 0 state))
 (:let title (list "Logiweb diagnose of " name))
 (:let body (b-diagnose-body ref tex diagnose state))
 (:let ct (tex-page "/diagnose/tex/page.tex" lgt title body))
 (clean-dir dir)
 (ct2file (cons dir "/page.tex") ct)
 (when (verbose '> 0) (format t "TeX face of diagnose: Executing TeX~%"))
 (exec-latex dir '("page"))
 (when (verbose '> 0) (format t "TeX face of diagnose: Executing dvipdfm~%"))
 (exec-dvipdfm dir '("page"))
 (html-dir lgt dir))

#|
=============================================
Generate <dir>/source
=============================================
Generate files that present the source in <dir>/vector.
|#

(deff b-source (dir lgt name ref state source)
 (ignoring ref state)
 (b-source-index (cons dir "/source.html") lgt name)
 (ct2file (cons dir "/page.lgs") source)
 (b-source-html (cons dir "/page.html") lgt name source))

#|
=============================================
Generate <dir>/source/index.html
=============================================
|#

(deff b-source-index (file lgt name)
 (ct2file file
  (html-page lgt
   (list "Logiweb source of " name)
   (html-h2 "Logiweb source of " name)
   (html-p
    (html-link0 "../index.html" "Up" )
    (html-link0 "page.lgs"    "Actual source as plain text")
    (html-link0 "page.html"   "Actual source as html")
    (html-link0 "../body/lgs.html" "Reverse engineered body.")
    (html-link0 (help-link "browser/source/index") "Help")))))

#|
=============================================
Generate <dir>/source/page.html
=============================================
|#

(deff b-source-html (file lgt name source)
;(:let source (b-array2html source))
 (:let source (utf2html source))
 (ct2file file
  (html-page lgt
   (list "Logiweb source of " name)
   (html-h2 "Logiweb source of " name)
   (html-p
    (html-link0 "./source.html" "Up" )
    (html-link0 (help-link "browser/source/index") "Help"))
   (html-p source))))





























