#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb System
=============================================
Trie functions
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Assoc function
=============================================
|#

(deff assoc-get (index assoc)
 (:when (null assoc) nil)
 (:let ((key . val) . assoc) assoc)
 (:when (equalp index key) val)
 (assoc-get index assoc))

(etst (assoc-get 3 '((4 . 40) (3 . 30) (2 . 20))) 30)
(ntst (assoc-get 3 '((4 . 40) (2 . 20))))

#|
=============================================
Array functions
=============================================
An "array" is an associative structure that maps cardinals (i.e. natural numbers) to values. Arrays are represented thus:

Nil represents the empty array.

(cons index value) where the given index is a cardinal represents the array that maps the given index to the given value and maps all other values to nil.

(cons array1 array2) represents the disjoint union of the two given arrays. See the definition of array-get to see which keys go in which of the two arrays array1 and array2.
|#

(deff array-get (index array)
 (array-get1 index array 0))

(deff array-get1 (index array level)
 (:when (null array) nil)
 (when (atom array) (error "array-get1: ~s~%" array))
 (:let (array1 . array2) array)
 (:when (numberp array1) (when (equalp index array1) array2))
 ;2008-09-01: Next line is for the sake of |compile ( " )|
 (:when (arrayp array1) (when (equalp index (vector2card array1)) array2))
 (array-get1 index (if (logbitp level index) array2 array1) (+ level 1)))

(deff array-put (index value array)
 (:when (null value) (array-remove index array 0))
 (array-put1 index value array 0))

(deff array-add (array1 array2)
 (:when (null array1)
  (:when (null array2) nil)
  (:when (numberp (car array2)) array2)
  (cons array1 array2))
 (:when (and (null array2) (numberp (car array1))) array1)
 (cons array1 array2))

(deff array-remove (index array level)
 (:when (null array) nil)
 (:let (array1 . array2) array)
 (:when (numberp array1) (when (unequal index array1) array))
 (if (logbitp level index)
  (array-add array1 (array-remove index array2 (+ level 1)))
  (array-add (array-remove index array1 (+ level 1)) array2)))

(deff array-put1 (index value array level)
 (:when (null array) (cons index value))
 (:let (array1 . array2) array)
 (:when (numberp array1)
  (if (equalp array1 index)
   (cons index value)
   (array-put2 index value array1 array2 level)))
 ;2008-09-01: Next branch is for the sake of |compile ( " )|
 (:when (arrayp array1)
  (if (equalp (vector2card array1) index)
   (cons index value)
   (array-put2 index value (vector2card array1) array2 level)))
 (if (logbitp level index)
  (cons array1 (array-put1 index value array2 (+ level 1)))
  (cons (array-put1 index value array1 (+ level 1)) array2)))

(deff array-put2 (index1 value1 index2 value2 level)
 (if (logbitp level index1)
  (if (logbitp level index2)
   (cons nil (array-put2 index1 value1 index2 value2 (+ level 1)))
   (cons (cons index2 value2) (cons index1 value1)))
  (if (logbitp level index2)
   (cons (cons index1 value1) (cons index2 value2))
   (cons (array-put2 index1 value1 index2 value2 (+ level 1)) nil))))

(etst (array-put 2 12 nil) (cons 2 12))
(etst (array-put 2 12 (array-put 3 13 nil)) (cons (cons 2 12) (cons 3 13)))
(etst (array-put 2 12 (array-put 2 22 nil)) (cons 2 12))
(etst (array-put 2 12 (array-put 4 14 nil))
 (cons (cons (cons 4 14) (cons 2 12)) nil))
(defc test-array (array-put 2 12 (array-put 3 13 (array-put 4 14 nil))))
(etst (array-get 0 test-array) nil)
(etst (array-get 1 test-array) nil)
(etst (array-get 2 test-array) 12)
(etst (array-get 3 test-array) 13)
(etst (array-get 4 test-array) 14)
(etst (array-get 5 test-array) nil)

#|
=============================================
Trie structures
=============================================
A 0-dimensional array is an arbitrary value.

An (n+1)-dimensional array is an array of n-dimensional arrays.

A trie is an arbitrary value or an arrays of trie structures (i.e. a multidimensional array where the dimensionality is unclear and may vary across the structure).

(check-trie n trie) checks that the given trie is a well-formed array of dimensionality (at least) n.

trie-put and trie-get are update and access functions.

(array-domain array) returns the domain of the given array as a descending list of cardinals.

(array2assoc array) converts the given array to an association list with descending keys.

(array2assoc-ascending array) converts the given array to an association list with ascending keys.

(array-merge array1 array2) merges the given arrays, giving priority to array1.
|#

(deff check-trie (level trie)
 (:when (<= level 0) t)
 (:when (null trie) t)
 (:when (atom trie) nil)
 (:let (trie1 . trie2) trie)
 (:when (numberp trie1) (check-trie (- level 1) trie2))
 (and
  (check-trie level trie1)
  (check-trie level trie2)))

(deff trie-put (adr val trie)
 (:when (atom adr) val)
 (:let (index . adr) adr)
 (:when (stringp index) (error "Old style trie-put: ~s" index))
 (array-put index (trie-put adr val (array-get index trie)) trie))

(deff trie-get (adr trie)
 (:when (null adr) trie)
 (when (atom adr) (error "trie-get: ~s~%" adr))
 (:let (index . adr) adr)
 (:when (stringp index) (error "Old style trie-get: ~s" index))
 (trie-get adr (array-get index trie)))

(etst (trie-get '(1 2 3 4) (trie-put '(1 2 3) test-array nil)) 14)
(etst (trie-get '(0 2 3 4) (trie-put '(1 2 3) test-array nil)) nil)
(etst (trie-get '(1 2 3 1) (trie-put '(1 2 3) test-array nil)) nil)
(defc test-trie (trie-put '(1 2 3) test-array test-array))
(etst (trie-get '(1 2 3 4) test-trie) 14)
(etst (trie-get '(0 2 3 4) test-trie) nil)
(etst (trie-get '(1 2 3 1) test-trie) nil)
(etst (trie-get '(4) test-trie) 14)
(etst (trie-put '(1 2) nil test-trie) test-array)

(defc *memo-trie* nil)

(defc *memo* (make-hash-table :test 'equalp))

(deff memo-trie-clear (trie)
 (setq *memo-trie* trie)
 (clrhash *memo*))

(deff memo-trie-get (adr trie)
 (:unless (eq trie *memo-trie*) (trie-get adr trie))
 (:let value (gethash adr *memo* :not-found))
 (:unless (eq value :not-found) value)
 (:let value (trie-get adr trie))
 (setf (gethash adr *memo*) value)
 value)

(memo-trie-clear test-trie)

(etst (memo-trie-get '(1 2 3 4) test-trie) 14)
(etst (memo-trie-get '(0 2 3 4) test-trie) nil)
(etst (memo-trie-get '(1 2 3 1) test-trie) nil)
(etst (memo-trie-get '(4) test-trie) 14)

(deff array-domain (array)
 (sort (array-domain1 array nil) #'>))

(deff array-domain1 (array result)
 (:when (null array) result)
 (when (atom array) (error "array-domain1: ~s~%" array))
 (:let (array1 . array2) array)
 (:when (numberp array1) (cons array1 result))
 (array-domain1 array1 (array-domain1 array2 result)))

(etst (array-domain test-array) '(4 3 2))

(deff array2assoc (array)
 (sort (array2assoc1 array nil) #'> :key 'car))

(deff array2assoc-ascending (array)
 (sort (array2assoc1 array nil) #'< :key 'car))

(deff array2assoc1 (array result)
 (:when (null array) result)
 (:let (array1 . array2) array)
 (:when (numberp array1) (acons array1 array2 result))
 (array2assoc1 array1 (array2assoc1 array2 result)))

(etst (array2assoc test-array) '((4 . 14) (3 . 13) (2 . 12)))

(deff array-merge (array1 array2)
 (:when (null array1) array2)
 (:let (array1a . array1b) array1)
 (:when (numberp array1a) (array-put array1a array1b array2))
 (array-merge array1a (array-merge array1b array2)))

(etst
 (array-merge
  (array-put 2 12 (array-put 3 13 nil))
  (array-put 2 22 (array-put 4 44 nil)))
 (array-put 2 12 (array-put 3 13 (array-put 4 44 nil))))

(deff aget (trie &rest adr)
 (trie-get adr trie))

(etst (aget test-trie 1 2 3 4) 14)

(deff memo-aget (trie &rest adr)
 (memo-trie-get adr trie))

(etst (aget test-trie 1 2 3 4) 14)

(deff aput (trie value &rest adr)
 (trie-put adr value trie))

(deff adom (trie &rest adr)
 (array-domain (trie-get adr trie)))






















