/*  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
*/

/*******************************************\
 * The Logiweb Server, Lisp/Unix interface *
\*******************************************/

/*
Tabel of contents:

General interface between Lisp and C
Test function
Time function
Internet interface functions
Demonizer
*/

#include <stdlib.h>
	/* void exit(int status); */
#include <errno.h>
        /* extern int errno; */
#include <sys/time.h>
	/* int gettimeofday(struct timeval *tv, struct timezone *tz); */
#include <stdio.h>
	/* void perror(const char *s); */
#include <string.h>
	/* char *strerror(int errnum); */
#include <string.h>
	/* int memcmp(const void *s1, const void *s2, size_t n); */
#include <stdio.h>
	/* int printf(const char *format, ...); */
#include <stdio.h>
	/* int fprintf(FILE *stream, const char *format, ...); */
#include <stdio.h>
	/* extern FILE *stderr */
	/* extern FILE *stdout */
#include <unistd.h>
	/* int close(int fd); */
#include <sys/types.h>
#include <sys/socket.h>
	/* int socket(int domain, int type, int protocol); */
#include <netdb.h>
       /* extern int h_errno; */
       /* struct hostent *gethostbyname(const char *name); */
#include <sys/types.h>
#include <sys/socket.h>
	/* int bind(int sockfd, struct sockaddr *my_addr, socklen_t addrlen); */
#include <sys/types.h>
#include <sys/socket.h>
        /* ssize_t recv(int s, void *buf, size_t len, int flags); */
	/* ssize_t  recvfrom(int s, void *buf, size_t len, int flags,
	   struct sock-addr *from, socklen_t *fromlen); */
#include <sys/types.h>
#include <sys/socket.h>
	/* ssize_t send(int s, const void *buf, size_t len, int flags); */
#include <sys/types.h>
#include <sys/socket.h>
       /* ssize_t send(int s, const void *buf, size_t len, int flags); */
	/* ssize_t  sendto(int  s,  const  void *buf, size_t len, int flags,
	   const struct sockaddr *to, socklen_t tolen); */
#include <sys/types.h>
#include <sys/socket.h>
	/* ssize_t  sendto(int  s,  const  void *buf, size_t len, int flags, 
	   const struct sockaddr *to, socklen_t tolen); */ 
#include <sys/socket.h>
	/* int listen(int s, int backlog); */
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
	/* int select(int n, fd_set *readfds, fd_set *writefds,
	   fd_set *exceptfds, struct timeval *timeout); 
	   FD_ZERO(fd_set *set);
	   FD_SET(int fd, fd_set *set); */
#include <sys/types.h>
#include <sys/socket.h>
	/* ssize_t recv(int s, void *buf, size_t len, int flags); */
#include <sys/socket.h>
#include <netinet/in.h>
	/* See man 7 ip */
#include <sys/types.h>
#include <unistd.h>
	/* pid_t fork(void) */
#include <unistd.h>
	/* pid_t setsid(void); */
#include <unistd.h>
	/* int chdir(const char *path); */
#include <sys/stat.h>
#include <sys/types.h>
	/* mode_t umask(mode_t mask); */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
	/* int open(const char *pathname, int flags) */
#include <sys/types.h>
#include <unistd.h>
	/* pid_t getpid(void); */
#include <sys/types.h>
#include <pwd.h>
	/* struct passwd *getpwnam(const char *name); */
#include <sys/types.h>
#include <unistd.h>
	/* int setuid(uid_t uid); */
#include <sys/types.h>
#include <unistd.h>
	/* int setgid(gid_t gid); */
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
       /* char *inet_ntoa(struct in_addr in); */
#include <string.h>
       /* size_t strlen(const char *s); */

/****************************************\
 * General interface between Lisp and C *
\****************************************/

/* Buffers for communication between Lisp and C */
#define BUFSIZE 1000000
int intBuffer[BUFSIZE];
char charBuffer[BUFSIZE];
int intPointer=0;
int charPointer=0;

/* Rewind buffer pointers */
void buffer_rewind (void) {
  charBuffer[BUFSIZE-1]=0; /* avoid runaway strings */
  intPointer=0;
  charPointer=0;
  return; }

void die(char* msg) {
  printf("%s\n",msg);
  exit(0); }

/* Read/write int/char to/from buffers */
/* These functions are both for internal use and for calling from Lisp */
int int_get (void) {
  if(intPointer>=BUFSIZE) die("int_get: buffer overflow");
  return intBuffer[intPointer++]; }
char char_get (void) {
  if(charPointer>=BUFSIZE) die("char_get: buffer overflow");
  return charBuffer[charPointer++]; }
void int_put (int value) {
  if(intPointer>=BUFSIZE) die("int_put: buffer overflow");
  intBuffer[intPointer++]=value; }
void char_put (char value) {
  if(charPointer>=BUFSIZE) die("char_put: buffer overflow");
  charBuffer[charPointer++]=value; }

/* Read/write char* to/from buffers */
/* These functions are for internal use only */
void chars_put (char* value, size_t length) {
  if(charPointer+length>BUFSIZE) die("chars_put: buffer overflow");
  int_put(length);
  memcpy((void *)(charBuffer+charPointer),value,length);
  charPointer+=length; }
void chars_advance(int length) {
  if(charPointer+length>BUFSIZE) die("chars_advance: buffer overflow");
  int_put(length);
  charPointer+=length; }
int chars_skip() {
  int length=int_get();
  if(charPointer+length>BUFSIZE) die("chars_skip: buffer overflow");
  charPointer+=length;
  return length; }

/* Macros for defining a Lisp function
DEFUN(f){...} defines the function f.
PARMB    Begin parameter reading block (B for Begin)
PARMI(x) Move integer parameter to x (I for Integer)
PARMC(x) Move byte parameter to x (C for Character)
PARMP(x) Move current buffer pointer to x. Must precede PARML (P for Pointer)
PARML(x) Move length of current field to x. Must follow PARMP (L for Length)
PARMS(x) Move byte* parameter to x (S for String)
PARME    End parameter reading block (E for End)
EXITI(x) Return integer x (I for Integer)
EXITC(x) Return byte x (C for Character)
EXITS(x) Return byte* x (S for String)
EXITP    Current buffer pointer. Must accompany EXITR and precede EXITA
EXITR    Current buffer room. Must accompany EXITP and precede EXITA
EXITA(x) Advance buffer pointer by x. Must follow EXITP and EXITR
EXITE    End exit block (E for End)
*/
#define DEFUN(x)   void x(void)
#define PARMB      { buffer_rewind(); }
#define PARMI(x)   { x=int_get(); }
#define PARMC(x)   { x=char_get(); }
#define PARMP(x)   { x=charBuffer+charPointer; }
#define PARML(x)   { x=chars_skip(); }
#define PARMS(x)   { PARMP(x) chars_skip(); }
#define PARME      { buffer_rewind(); }
#define EXITI(x)   { int_put(x); }
#define EXITC(x)   { char_put(x); }
#define EXITS(x)   { chars_put(x,strlen(x)); }
#define EXITP      ( charBuffer+charPointer )
#define EXITR      ( BUFSIZE-charPointer )
#define EXITA(x)   { int y=x; if(y<0) RETURN(-1) chars_advance(y); }
#define EXITE      { buffer_rewind(); return; }

/* Macro for returning a single integer */
#define RETURN(x) { buffer_rewind(); int_put(x); buffer_rewind(); return; }

/* Macro for reporting a unix error and returning -1 */
#define ERROR(msg) {unix_error(stderr,msg); RETURN(-1)}

/* Print error message to file (e.g. stderr or stdout) */
void unix_error (FILE *file,char *msg) {
  fprintf(file,"Unix call: %s\n",msg);
  fprintf(file,"Unix error message: %s\n",strerror(errno));}



/*****************\
 * Test function *
\*****************/

/* Function for test of dynamical linking, taken from
http://howtos.linux.com/howtos/Program-Library-HOWTO/more-examples.shtml */
void hello(void) {
  printf("Hello, library world.\n");
}



/*****************\
 * Time function *
\*****************/

/* Time function. Returns number of seconds since the Unix epoch.
The incorrect handling of leap seconds done by Unix is compensated
in time.lisp. The number of seconds is expressed as m*10^-e and
the function returns e and m. e is returned as an integer. m is
returned as a radix followed by m in little endian, terminated
by -1 */
DEFUN(unix_gettimeofday) {
  struct timeval tv;
  int exponent=6;
  int radix=1000000;
  int endOfNumber=-1;
  int result;
PARMB
PARME
  result=gettimeofday(&tv, NULL);
  if(result) ERROR("gettimeofday");
EXITI(0);
EXITI(exponent)
EXITI(radix)
EXITI(tv.tv_usec)
EXITI(tv.tv_sec%radix)
EXITI(tv.tv_sec/radix)
EXITI(endOfNumber)
EXITE }



/********************************\
 * Internet interface functions *
\********************************/

/* Close file descriptor.
   fd_close (int fd) -> (int 0) 
   Always succeeds. */
DEFUN(fd_close){
  int fd;
PARMB
PARMI(fd)
PARME
  close(fd);
EXITI(0)
EXITE}

/* Translate host name to ip */
DEFUN(host_ip){
  char *domain;
  struct hostent *host;
  struct sockaddr_in addr;
PARMB
PARMS(domain)
PARME
  host=gethostbyname(domain);
  if(!host) RETURN(-1)
  memcpy(&(addr.sin_addr),host->h_addr_list[0],host->h_length);
EXITS(inet_ntoa(addr.sin_addr))
EXITE}

/* Open udp socket
   udp_open -> (int fd)
   Returns -1 on error. */
DEFUN(udp_open){
  int fd;
PARMB
PARME
  fd=socket(AF_INET, SOCK_DGRAM, 0);
  if(fd<0) ERROR("socket")
EXITI(fd)
EXITE}

/* Bind udp socket
   udp_bind (int fd) (char* ip) (int port) -> (int 0)
   Returns -1 on error. */

void print_herror(void){
  switch(h_errno){
    case HOST_NOT_FOUND:
    fprintf(stderr,"The specified host is unknown.\n");
    break;
    case NO_ADDRESS:
    fprintf(stderr,"The requested name is valid but does not have\n");
    fprintf(stderr,"an IP address.\n");
    break;
    case NO_RECOVERY:
    fprintf(stderr,"A non-recoverable name server error occurred.\n");
    break;
    case TRY_AGAIN:
    fprintf(stderr,"A temporary error occurred on an authoritative\n");
    fprintf(stderr,"name server. Try again later.\n");
    break;
    default:
    fprintf(stderr,"Unknown error.\n");}}

int set_addr(struct sockaddr_in *addr,char* domain,int port){
  struct hostent *host=gethostbyname(domain);
  if(!host){
    fprintf(stderr,"Unix call: gethostbyname(%s)\n",domain);
    print_herror();
    return -1;}
  addr->sin_family=AF_INET;
  addr->sin_port=htons(port);
  memcpy(&(addr->sin_addr),host->h_addr_list[0],host->h_length);
  return 0;}

DEFUN(udp_bind){
  int fd;
  char* ip;
  int port;
  struct sockaddr_in sock;
PARMB
PARMI(fd)
PARMS(ip)
PARMI(port)
PARME
  if(set_addr(&sock,ip,port)) RETURN(-1)
  if(bind(fd,(void *)&sock,sizeof(sock))) ERROR("bind")
EXITI(0)
EXITE}

/* Receive message from udp socket
   udp_recvfrom (int fd) -> (char* msg) (char* ip) (int port)
   Returns -1 on error.
   Returns -2 when no message available. */
DEFUN(udp_recvfrom){
  int fd;
  ssize_t msglen;
  struct sockaddr_in from;
  socklen_t fromlen=sizeof(from);
PARMB
PARMI(fd)
PARME
  msglen=recvfrom(fd,EXITP,EXITR,MSG_DONTWAIT,(void *)&from,&fromlen);
  if(msglen<0) {if(errno==EAGAIN) RETURN(-2) else ERROR("recvfrom")}
EXITA(msglen)
EXITS(inet_ntoa(from.sin_addr))
EXITI(ntohs(from.sin_port))
EXITE}

/* Send message to udp socket
   udp_sendto (int fd) (char* msg) (char* ip) (int port) -> (int 0)
   Returns -1 on error. */

DEFUN(udp_sendto){
  int fd;
  char* buffer;
  int length;
  char* domain;
  int port;
  ssize_t msglen;
  struct sockaddr_in to;
PARMB
PARMI(fd)
PARMP(buffer)
PARML(length)
PARMS(domain)
PARMI(port)
PARME
  if(set_addr(&to,domain,port)) RETURN(-1)
  msglen=sendto(fd,buffer,length,0,(void *)&to,sizeof(to));
  if(msglen<0) ERROR("sendto")
  if(length!=msglen) ERROR("sendto: partial transmittion")
EXITI(0)
EXITE}

/* Open listening tcp socket
   tcp_open (char* ip) (int port) (int backlog) -> (int fd)
   Returns -1 on error. */
DEFUN(tcp_open){
  int fd;
  char *ip;
  int port;
  int backlog;
  struct sockaddr_in sock;
PARMB
PARMS(ip)
PARMI(port)
PARMI(backlog)
PARME
  fd=socket(AF_INET, SOCK_STREAM, 0);
  if(fd<0) ERROR("socket")
  if(set_addr(&sock,ip,port)) RETURN(-1)
  if(bind(fd,(void *)&sock,sizeof(sock))) ERROR("bind")
  if(listen(fd,backlog)) ERROR("listen")
EXITI(fd)
EXITE}

/* Accept incomming connection
   tcp_accept (int fd0) -> (int fd1) (char* ip)
   Returns -1 on error.
   Returns 0 if there are no incomming connections. */
DEFUN(tcp_accept){
  int fd0,fd1;
  fd_set set;
  struct timeval timeout;
  int result;
  struct sockaddr_in sock;
  socklen_t length;
PARMB
PARMI(fd0)
PARME
  timeout.tv_sec=0;
  timeout.tv_usec=0;
  FD_ZERO(&set);
  FD_SET(fd0,&set);
  result=select(fd0+1,&set,0,0,&timeout);
  if(result<0) ERROR("select")
  if(result==0) RETURN(0)
  length=sizeof(sock);
  fd1=accept(fd0, (void *)&sock, &length);
  if(fd1<0) ERROR("accept")
EXITI(fd1)
EXITS(inet_ntoa(sock.sin_addr))
EXITE}

/* Impatient read from connected socket
   tcp_recv (int fd) -> (char* msg)
   Returns -1 on error.
   Returns 0 is nothing is received within 0.1 second */
DEFUN(tcp_recv){
  int fd;
  fd_set set;
  struct timeval timeout;
  int result;
PARMB
PARMI(fd)
PARME
  timeout.tv_sec=0;
  timeout.tv_usec=100000;
  FD_ZERO(&set);
  FD_SET(fd,&set);
  result=select(fd+1,&set,0,0,&timeout);
  if(result<0) ERROR("select");
  if(result==0){RETURN(0);}
  result=recv(fd,EXITP,EXITR,0);
  if(result<0) ERROR("recv")
EXITA(result)
EXITE}

/* Impatient send to connected socket
   tcp_send (int fd) (char* msg) -> (int 0)
   Returns -1 on error.
   Missing outgoing bandwidth is not treated as an error. */
DEFUN(tcp_send){
  int fd;
  char* buffer;
  int length;
  int msglen;
PARMB
PARMI(fd)
PARMP(buffer)
PARML(length)
PARME
  msglen=send(fd,buffer,length,MSG_DONTWAIT);
  if(msglen>=0) RETURN(0)
  if(errno==EAGAIN) RETURN(0)
  ERROR("send")}

/* Impatiently send msg1 to domain/port, then patiently read response
   tcp_query
   (int port) (int usec) (int sec) (int Msec) (char* msg1) (char* domain)
   -> (card* msg2)
   Returns -1 on error. */
DEFUN(tcp_query){
  int port;
  int usec,sec,Msec;
  int length;
  char* msg1;
  char* domain;
  int fd;
  fd_set set;
  struct timeval timeout;
  int result;
  struct sockaddr_in to;
  ssize_t length0;
  ssize_t length1;
PARMB
PARMI(port);
PARMI(usec);
PARMI(sec);
PARMI(Msec);
PARMP(msg1);
PARML(length);
PARMP(domain);
PARME
  /* Set timeout */
  timeout.tv_sec=sec+1000000*Msec;
  timeout.tv_usec=usec;
  /* Open local socket */
  fd=socket(AF_INET, SOCK_STREAM, 0);
  if(fd<0) ERROR("socket")
  FD_ZERO(&set);
  FD_SET(fd,&set);
  /* Send request to server */
  if(set_addr(&to,domain,port)) {close(fd);RETURN(-1)}
  if(connect(fd,(void *)&to,sizeof(to))) {close(fd);ERROR("connect")}
  length1=send(fd,msg1,length,0);
  if(length1<0) {close(fd);ERROR("sendto")}
  if(length!=length1) {
    fprintf(stderr,"Unix call: write\n");
    fprintf(stderr,"Partial transmission\n");
    close(fd);
    RETURN(-1)}
  /* Receive response from server */
  for(length0=0;length0<BUFSIZE-1;length0+=length1){
    result=select(fd+1,&set,0,0,&timeout);
    if(result<0) ERROR("select");
    if(result==0){RETURN(0);}
    length1=recv(fd,charBuffer+length0,BUFSIZE-1-length0,0);
    if(length1<0) {close(fd);ERROR("recv")}
    if(length1==0) {close(fd);RETURN(length0)}}}

/* Wait for incomming messages or timeout
   inet_select (int usec) (int sec) (int Msec)
   (int cnt) (int fd_1) ... (int fd_cnt) -> (int 0)
   The timeout is given in microseconds (usec), seconds (sec), and
   megaseconds (Msec).
   Returns -1 on error. */
DEFUN(inet_select){
  int usec,sec,Msec,cnt,fd;
  int i;
  int max_fd;
  int result;
  struct timeval timeout;
  fd_set set;
PARMB
PARMI(usec)
PARMI(sec)
PARMI(Msec)
PARMI(cnt)
  FD_ZERO(&set);
  max_fd=0;
  for(i=0;i<cnt;i++){
    PARMI(fd)
    if(max_fd<fd) max_fd=fd;
    FD_SET(fd,&set);}
PARME
  timeout.tv_sec=sec+1000000*Msec;
  timeout.tv_usec=usec;
  result=select(max_fd+1,&set,0,&set,&timeout);
  if(result<0) ERROR("select")
EXITI(0)
EXITE}

/*********\
 * Files *
\*********/

/* Open non-blocking file */

DEFUN(file_open){
  char* filename;
  int fd;
PARMB
PARMS(filename)
PARME
  fd=open(filename,O_RDONLY|O_NONBLOCK);
  if(fd<0) ERROR("socket")
EXITI(fd)
EXITE}

/* Read from file descriptor
Returns -2 if no input ready
Returns -1 on error
Returns the empty string at EOF */

DEFUN(file_read){
  int fd;
  int result;
PARMB
PARMI(fd)
PARME
  result=read(fd,EXITP,EXITR);
  if(result==0) RETURN(0) else
  if(result<0){if(errno==EAGAIN) RETURN(-2) else ERROR("read");}
EXITA(result)
EXITE}

/*************\
 * Demonizer *
\*************/

/* Exit with error message to given file */

void demon_dief(FILE* file,char* msg)
{
  fprintf(file,"Demonizer\n");
  unix_error(file,msg);
  exit(0);
}

/* Exit with error message to stderr */

void demon_die(char* msg)
{
  demon_dief(stderr,msg);
}

/* Open log file:
Create it if it does not exist.
Overwrite it if it does exist.
Set permissions to user=read/write, group,other=read.
*/

int open_log_file(char* path)
{
  return open(
    path,
    O_WRONLY|O_CREAT|O_TRUNC/*|O_SYNC*/,
    S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
}

DEFUN(unix_demonize){
  char* log_file;
  char* pid_file;
  char* usr_name;
  pid_t pid;
  FILE *pid_stream;
  struct passwd *user;
  int fd;
PARMB
PARMS(log_file)
PARMS(pid_file)
PARMS(usr_name)
PARME

/* The demonizer exits on errors rather than returning control to Common Lisp */

/* From http://www.erlenstar.demon.co.uk/unix/faq_2.html#SEC16:

1.7 How do I get my program to act like a daemon?

A daemon process is usually defined as a background process that does not belong to a terminal session. Many system services are performed by daemons; network services, printing etc.

Simply invoking a program in the background isn't really adequate for these long-running programs; that does not correctly detach the process from the terminal session that started it. Also, the conventional way of starting daemons is simply to issue the command manually or from an rc script; the daemon is expected to put itself into the background.

Here are the steps to become a daemon:

1. fork() so the parent can exit, this returns control to the command line or shell invoking your program. This step is required so that the new process is guaranteed not to be a process group leader. The next step, setsid(), fails if you're a process group leader. */

  pid=fork();
  if(pid==-1) demon_die("first fork()");
  if(pid) exit(0);

/* 2. setsid() to become a process group and session group leader. Since a controlling terminal is associated with a session, and this new session has not yet acquired a controlling terminal our process now has no controlling terminal, which is a Good Thing for daemons. */

  if(setsid()==-1) demon_die("setsid()");

/* 3. fork() again so the parent, (the session group leader), can exit. This means that we, as a non-session group leader, can never regain a controlling terminal. */

  pid=fork();
  if(pid==-1) demon_die("second fork()");
  if(pid) exit(0);

/* If a pid_file is given, tell which pid to kill in order to kill the Logiweb server */

  if(pid_file[0]){
    fd=open(pid_file,O_WRONLY|O_CREAT|O_EXCL,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
    if(fd<0) demon_die("open(pid_file)");
    if(!(pid_stream=fdopen(fd,"w"))) demon_die("fdopen()");
    if(fprintf(pid_stream,"%d\n",getpid())<0) demon_die("fprintf(pid_stream)");
    if(fclose(pid_stream)) demon_die("fclose(pid_stream)");}

/* Drop priviledges (if the given user name is the empty string, do nothing) */

  if(usr_name[0])
  {
    user=getpwnam(usr_name);
    if(!user) demon_die("getpwnam()");
    if(setgid(user->pw_uid)) demon_die("setgid()");
    if(setuid(user->pw_uid)) demon_die("setuid()");
  }

/* 4. chdir("/") to ensure that our process doesn't keep any directory in use. Failure to do this could make it so that an administrator couldn't unmount a filesystem, because it was our current directory. [Equivalently, we could change to any directory containing files important to the daemon's operation.] */

  if(chdir("/")==-1) demon_die("chdir()");

/* 5. umask(0) so that we have complete control over the permissions of anything we write. We don't know what umask we may have inherited. [This step is optional] */

  umask(0);

/* 6. close() fds 0, 1, and 2. This releases the standard in, out, and error we inherited from our parent process. We have no way of knowing where these fds might have been redirected to. Note that many daemons use sysconf() to determine the limit _SC_OPEN_MAX. _SC_OPEN_MAX tells you the maximun open files/process. Then in a loop, the daemon can close all possible file descriptors. You have to decide if you need to do this or not. If you think that there might be file-descriptors open you should close them, since there's a limit on number of concurrent file descriptors. 

7. Establish new open descriptors for stdin, stdout and stderr. Even if you don't plan to use them, it is still a good idea to have them open. The precise handling of these is a matter of taste; if you have a logfile, for example, you might wish to open it as stdout or stderr, and open `/dev/null' as stdin; alternatively, you could open `/dev/console' as stderr and/or stdout, and `/dev/null' as stdin, or any other combination that makes sense for your particular daemon. */

  if(close(0)==-1) demon_die("close(0)");
  if(open("/dev/null",O_RDONLY)!=0) demon_die("open(0)");
  if(close(1)==-1) demon_die("close(1)");
  if(open_log_file(log_file)!=1) demon_die("open_log_file(1)");
  if(close(2)==-1) demon_die("close(2)");
  if(open_log_file(log_file)!=2) demon_dief(stdout,"open_log_file(2)");

/*
  Alternative reopen:
  open(log_file,
    O_WRONLY|O_CREAT|O_TRUNC|O_SYNC,
    S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
  freopen("/dev/null","r",stdin);
  freopen(log_file,"a",stdout);
  freopen(log_file,"a",stderr);
*/

/* Thanks to http://www.erlenstar.demon.co.uk/unix/faq_2.html#SEC16. */

/*buffer_rewind();*/
  EXITE

}




  
