/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* librvngabw
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2003 William Lachance (wrlach@gmail.com)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

/* "This product is not manufactured, approved, or supported by
 * Corel Corporation or Corel Corporation Limited."
 */
#ifndef _LISTSTYLE_HXX_
#define _LISTSTYLE_HXX_

#include <map>
#include <set>
#include <stack>
#include <vector>
#include <librevenge/librevenge.h>

#include "Style.hxx"
namespace librvngabw
{
class ABWGenerator;
class ListManager;
class ParagraphStyleManager;

class ListLevelStyle
{
public:
	//! constructor
	ListLevelStyle(const librevenge::RVNGPropertyList &xPropList, int level, bool ordered, int id) :
		m_propList(xPropList), m_level(level), m_ordered(ordered), m_id(id), m_parentId(0)
	{
	}
	//! destructor
	virtual ~ListLevelStyle() {}

	//! returns the current id
	int getId() const
	{
		return m_id;
	}
	//! sets the parent id
	void setParentId(int parentId)
	{
		m_parentId=parentId;
	}
	//! add level definition to propList and propPropList, what=0 means a paragraph style, what=1 a style definition
	void addTo(librevenge::RVNGPropertyList &propList, librevenge::RVNGPropertyList &propPropList, int what,
	           int parentId=-1) const;
	//! write the level definition
	void writeDefinition(ABWDocumentHandler *pHandler) const;
	//! comparison operator
	bool isEqualTo(const librevenge::RVNGPropertyList &xPropList, int level, bool ordered) const;
private:
	librevenge::RVNGPropertyList m_propList;
	//! the list level
	int m_level;
	//! a flag to know if this is an ordered list or not
	bool m_ordered;
	//! the main id
	int m_id;
	//! the parent id
	int m_parentId;
};

class List
{
public:
	//! constructor
	List(const int iListID, ListManager &manager);
	//! destructor
	virtual ~List();
	void updateListLevel(const int level, const librevenge::RVNGPropertyList &xPropList, bool ordered);
	//! add level definition to propList and propPropList
	void addTo(librevenge::RVNGPropertyList &propList, librevenge::RVNGPropertyList &propPropList, int level, int parentId=-1) const;
	//! returns the list id
	int getListId() const
	{
		return m_id;
	}
	//! returns the level id corresponding to a level ( or 0)
	int getLevelId(int level) const
	{
		if (level<0 || level>=int(m_levelIdList.size())) return 0;
		return m_levelIdList[size_t(level)];
	}
	//! check if a level is defined
	bool isListLevelDefined(int level) const
	{
		return level>=0 && level<int(m_levelIdList.size()) && m_levelIdList[size_t(level)]>0;
	}

protected:
	//! store the iListLevelStyle in the list manager and update the levelId structures
	void setListLevel(int level, shared_ptr<ListLevelStyle> iListLevelStyle);

private:
	List(const List &);
	List &operator=(const List &);
	//! the list manager
	ListManager &m_manager;
	//! the list id
	const int m_id;
	//! the list of level id
	std::vector<int> m_levelIdList;
};

/** a list manager */
class ListManager
{
	friend class ABWGenerator;
public:
	//! constructor
	explicit ListManager(ParagraphStyleManager &paragraphManager);
	//! destructor
	virtual ~ListManager();

	/// open a list level
	bool openLevel(const librevenge::RVNGPropertyList &propList, bool ordered);
	/// close a list level
	void closeLevel();
	/// returns a new list level id
	int getNewLevelId()
	{
		return ++m_levelId;
	}
	/// store a level
	void storeLevel(shared_ptr<ListLevelStyle> newLevel)
	{
		if (!newLevel) return;
		m_idToLevelMap[newLevel->getId()]=newLevel;
	}
	/// return a level corresponding to a level id
	shared_ptr<ListLevelStyle> getLevel(int levelId) const
	{
		if (m_idToLevelMap.find(levelId)==m_idToLevelMap.end())
			return shared_ptr<ListLevelStyle>();
		return m_idToLevelMap.find(levelId)->second;
	}

	/// open a list element and add the corresponding data in output
	bool openListElement(const librevenge::RVNGPropertyList &propList, DocumentElementVector &output, int &xId);
	/// close a list element and add the corresponding data in output
	void closeListElement(DocumentElementVector &output);

	/// write all styles
	virtual void write(ABWDocumentHandler *pHandler) const;

	/// pop the list state (if possible)
	void popState();
	/// push the list state by adding an empty value
	void pushState();

protected:
	// list state
	struct State
	{
		//! constructor
		State();
		//! copy constructor
		State(const State &state);
		//! copy operator
		State &operator=(const State &state);

		//! the current list
		List *m_currentList;
		//! the current level
		int m_currentLevel;
		//! the current send level
		int m_createdLevel;
		//! the list of actual level id: 0 means a level not open
		std::vector<int> m_levelIdList;
		std::stack<bool> m_listElementOpenedStack;
	private:
	};

	/// call to define a list level, return the level id
	void defineLevel(const librevenge::RVNGPropertyList &propList, bool ordered);
	/// access to the current list state
	State &getState();
	/// access to the state stack (for ABWGenerator)
	std::stack<State> &getStack()
	{
		return m_stateStack;
	}
	//! the paragraph manager
	ParagraphStyleManager &m_paragraphManager;

	//! the actual level id
	int m_levelId;
	//! a map of levelId to level
	std::map<int, shared_ptr<ListLevelStyle> > m_idToLevelMap;

	//! list of list
	std::vector<shared_ptr<List> > m_styleList;
	//! a map id -> last list style defined with id
	std::map<int, shared_ptr<List> > m_idToStyleMap;

	//! list states
	std::stack<State> m_stateStack;
};
}

#endif

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
