/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../includes.h"

extern int DEBUGLEVEL;


/*******************************************************************
makes a SAMR_Q_CLOSE_HND structure.
********************************************************************/
void make_samr_q_close_hnd(SAMR_Q_CLOSE_HND *q_c, POLICY_HND *hnd)
{
	if (q_c == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_close_hnd\n"));

	memcpy(&(q_c->pol), hnd, sizeof(q_c->pol));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_close_hnd(char *desc,  SAMR_Q_CLOSE_HND *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_close_hnd");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_u->pol), ps); 
	prs_align(ps);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_close_hnd(char *desc,  SAMR_R_CLOSE_HND *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_close_hnd");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(r_u->pol), ps); 
	prs_align(ps);

	prs_uint32("status", ps, &(r_u->status));
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_samr_q_open_domain(SAMR_Q_OPEN_DOMAIN *q_u,
				POLICY_HND *connect_pol, uint32 rid, char *sid)
{
	if (q_u == NULL) return;

	DEBUG(5,("samr_make_q_open_domain\n"));

	memcpy(&q_u->connect_pol, connect_pol, sizeof(q_u->connect_pol));
	q_u->rid = rid;
	make_dom_sid2(&(q_u->dom_sid), sid);
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_open_domain(char *desc,  SAMR_Q_OPEN_DOMAIN *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_open_domain");
	

	prs_align(ps);

	smb_io_pol_hnd("connect_pol", &(q_u->connect_pol), ps); 
	prs_align(ps);

	prs_uint32("rid", ps, &(q_u->rid));

	smb_io_dom_sid2("sid", &(q_u->dom_sid), ps); 
	prs_align(ps);
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_open_domain(char *desc,  SAMR_R_OPEN_DOMAIN *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_open_domain");
	

	prs_align(ps);

	smb_io_pol_hnd("domain_pol", &(r_u->domain_pol), ps); 
	prs_align(ps);

	prs_uint32("status", ps, &(r_u->status));
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_samr_q_unknown_3(SAMR_Q_UNKNOWN_3 *q_u,
				POLICY_HND *pol, uint32 switch_value)
{
	if (q_u == NULL) return;

	DEBUG(5,("samr_make_q_unknown_3\n"));

	memcpy(&q_u->pol, pol, sizeof(q_u->pol));
	q_u->switch_value = switch_value;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_3(char *desc,  SAMR_Q_UNKNOWN_3 *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_unknown_3");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_u->pol), ps); 
	prs_align(ps);

	prs_uint16("switch_value", ps, &(q_u->switch_value));
	prs_align(ps);
}

/*******************************************************************
 makes a SAMR_R_UNKNOWN3 structure.
 ********************************************************************/
void make_dom_sid3(DOM_SID3 *sid3, uint16 unk_0, uint16 unk_1, char *sid)
{
	if (sid3 == NULL) return;

	/* sid->len is _calculated_ in sam_io_dom_sid3 */
	sid3->unknown_0 = unk_0;
	sid3->unknown_1 = unk_1;
	make_dom_sid(&(sid3->sid), sid);
}

/*******************************************************************
reads or writes a SAM_SID3 structure.

this one's odd, because the length (in bytes) is specified at the beginning.
the length _includes_ the length of the length, too :-)

********************************************************************/
void sam_io_dom_sid3(char *desc,  DOM_SID3 *sid3, prs_struct *ps)
{
	int ptr_len = 0;
	int old_align = 0;

	if (sid3 == NULL) return;

	old_align = ps->align;
	ps->align = 2;

	prs_debug(ps, desc, "sam_io_dom_sid3");
	

	if (ps->io) 
	{
		/* reading.  do the length later */
		prs_uint16("len", ps, &(sid3->len));
	}
	else
	{
		/* storing */
		ps->offset += 2;
	}

	ptr_len = ps->offset;

	prs_uint16("unknown_0", ps, &(sid3->unknown_0));
	prs_uint16("unknown_1", ps, &(sid3->unknown_1));
	smb_io_dom_sid("", &(sid3->sid), ps); 

	if (!(ps->io)) /* storing not reading.  do the length, now. */
	{
		uint16 len = 2 + ps->offset - ptr_len;
		ps->offset = ptr_len;
		prs_uint16("len", ps, &len);
	}

	ps->offset += sid3->len;

	ps->align = old_align;
}

/*******************************************************************
makes a SAMR_R_UNKNOWN3 structure.

unknown_2   : 0x0001
unknown_3   : 0x8004

unknown_4,5 : 0x0000 0014

unknown_6   : 0x0002
unknown_7   : 0x5800 or 0x0070

********************************************************************/
void make_sam_sid_stuff(SAM_SID_STUFF *stf,
				uint16 unknown_2, uint16 unknown_3,
				uint32 unknown_4, uint16 unknown_6, uint16 unknown_7,
				int num_sid3s, DOM_SID3 sid3[MAX_SAM_SIDS])
{
	stf->unknown_2 = unknown_2;
	stf->unknown_3 = unknown_3;

	bzero(stf->padding1, sizeof(stf->padding1));

	stf->unknown_4 = unknown_4;
	stf->unknown_5 = unknown_4;

	stf->unknown_6 = unknown_6;
	stf->unknown_7 = unknown_7;

	stf->num_sids  = num_sid3s;

	stf->padding2  = 0x0000;

	memcpy(stf->sid, sid3, sizeof(DOM_SID3) * num_sid3s);
}

/*******************************************************************
reads or writes a SAM_SID_STUFF structure.
********************************************************************/
void sam_io_sid_stuff(char *desc,  SAM_SID_STUFF *stf, prs_struct *ps)
{
	int i;

	if (stf == NULL) return;

	DEBUG(5,("make_sam_sid_stuff\n"));

	prs_uint16("unknown_2", ps, &(stf->unknown_2));
	prs_uint16("unknown_3", ps, &(stf->unknown_3));

	prs_uint8s(False, "padding1", ps, stf->padding1, sizeof(stf->padding1)); 
	
	prs_uint32("unknown_4", ps, &(stf->unknown_4));
	prs_uint32("unknown_5", ps, &(stf->unknown_5));
	prs_uint16("unknown_6", ps, &(stf->unknown_6));
	prs_uint16("unknown_7", ps, &(stf->unknown_7));

	prs_uint32("num_sids ", ps, &(stf->num_sids ));
	prs_uint16("padding2 ", ps, &(stf->padding2 ));

	for (i = 0; i < stf->num_sids; i++)
	{
		sam_io_dom_sid3("", &(stf->sid[i]), ps); 
	}
}

/*******************************************************************
reads or writes a SAMR_R_UNKNOWN3 structure.
********************************************************************/
void make_samr_r_unknown_3(SAMR_R_UNKNOWN_3 *r_u,
				uint16 unknown_2, uint16 unknown_3,
				uint32 unknown_4, uint16 unknown_6, uint16 unknown_7,
				int num_sid3s, DOM_SID3 sid3[MAX_SAM_SIDS],
				uint32 status)
{
	if (r_u == NULL) return;

	DEBUG(5,("samr_make_r_unknown_3\n"));

	r_u->ptr_0 = 0;
	r_u->ptr_1 = 0;

	if (status == 0x0)
	{
		r_u->ptr_0 = 1;
		r_u->ptr_1 = 1;
		make_sam_sid_stuff(&(r_u->sid_stuff), unknown_2, unknown_3,
	               unknown_4, unknown_6, unknown_7,
	               num_sid3s, sid3);
	}

	r_u->status = status;
}


/*******************************************************************
reads or writes a SAMR_R_UNKNOWN_3 structure.

this one's odd, because the daft buggers use a different mechanism
for writing out the array of sids. they put the number of sids in
only one place: they've calculated the length of each sid and jumped
by that amount.  then, retrospectively, the length of the whole buffer
is put at the beginning of the data stream.

wierd.  

********************************************************************/
void samr_io_r_unknown_3(char *desc,  SAMR_R_UNKNOWN_3 *r_u, prs_struct *ps)
{
	int ptr_len0;
	int ptr_len1;
	int ptr_sid_stuff = 0;

	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_unknown_3");
	

	prs_align(ps);

	prs_uint32("ptr_0         ", ps, &(r_u->ptr_0         ));

	if (ps->io) 
	{
		/* reading.  do the length later */
		prs_uint32("sid_stuff_len0", ps, &(r_u->sid_stuff_len0));
	}
	else
	{
		/* storing */
		ptr_len0 = ps->offset; ps->offset += 4;
	}

	if (r_u->ptr_0 != 0)
	{
		prs_uint32("ptr_1         ", ps, &(r_u->ptr_1         ));
		if (ps->io)
		{
			/* reading.  do the length later */
			prs_uint32("sid_stuff_len1", ps, &(r_u->sid_stuff_len1));
		}
		else
		{
			/* storing */
			ptr_len1 = ps->offset; ps->offset += 4;
		}

		if (r_u->ptr_1 != 0)
		{
			ptr_sid_stuff = ps->offset;
			sam_io_sid_stuff("", &(r_u->sid_stuff), ps); 
		}
	}

	if (!(ps->io)) /* storing not reading.  do the length, now. */
	{
		if (ptr_sid_stuff != 0)
		{
			uint32 sid_stuff_len = ps->offset - ptr_sid_stuff;
			int old_len = ps->offset;

			ps->offset = ptr_len0;
			prs_uint32("sid_stuff_len0", ps, &sid_stuff_len); 

			ps->offset = ptr_len1;
			prs_uint32("sid_stuff_len1", ps, &sid_stuff_len);

			ps->offset = old_len;
		}
	}

	prs_uint32("status", ps, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_ENUM_DOM_USERS structure.
********************************************************************/
void make_samr_q_enum_dom_users(SAMR_Q_ENUM_DOM_USERS *q_e, POLICY_HND *pol,
				uint16 req_num_entries, uint16 unk_0,
				uint16 acb_mask, uint16 unk_1, uint32 size)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_enum_dom_users\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->req_num_entries = req_num_entries; /* zero indicates lots */
	q_e->unknown_0 = unk_0; /* this gets returned in the response */
	q_e->acb_mask  = acb_mask;
	q_e->unknown_1 = unk_1;
	q_e->max_size = size;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_enum_dom_users(char *desc,  SAMR_Q_ENUM_DOM_USERS *q_e, prs_struct *ps)
{
	if (q_e == NULL) return;

	prs_debug(ps, desc, "samr_io_q_enum_dom_users");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_e->pol), ps); 
	prs_align(ps);

	prs_uint16("req_num_entries", ps, &(q_e->req_num_entries));
	prs_uint16("unknown_0      ", ps, &(q_e->unknown_0      ));

	prs_uint16("acb_mask       ", ps, &(q_e->acb_mask       ));
	prs_uint16("unknown_1      ", ps, &(q_e->unknown_1      ));

	prs_uint32("max_size       ", ps, &(q_e->max_size       ));

	prs_align(ps);
}


/*******************************************************************
makes a SAMR_R_ENUM_DOM_USERS structure.
********************************************************************/
void make_samr_r_enum_dom_users(SAMR_R_ENUM_DOM_USERS *r_u,
		uint16 total_num_entries, uint16 unk_0,
		uint32 num_sam_entries, struct smb_passwd pass[MAX_SAM_ENTRIES], uint32 status)
{
	int i;

	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_enum_dom_users\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	r_u->total_num_entries = total_num_entries;
	r_u->unknown_0         = unk_0;

	if (total_num_entries > 0)
	{
		r_u->ptr_entries1 = 1;
		r_u->ptr_entries2 = 1;
		r_u->num_entries2 = num_sam_entries;
		r_u->num_entries3 = num_sam_entries;

		for (i = 0; i < num_sam_entries; i++)
		{
			char *acct_name = pass[i].smb_name;
			int len_acct_name = strlen(acct_name);

			make_sam_entry(&(r_u->sam[i]), acct_name, pass[i].smb_userid);
			make_unistr2(&(r_u->uni_acct_name[i]), acct_name, len_acct_name);
		}

		r_u->num_entries4 = num_sam_entries;
	}
	else
	{
		r_u->ptr_entries1 = 0;
		r_u->num_entries2 = num_sam_entries;
		r_u->ptr_entries2 = 1;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_enum_dom_users(char *desc,  SAMR_R_ENUM_DOM_USERS *r_u, prs_struct *ps)
{
	int i;

	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_enum_dom_users");
	

	prs_align(ps);

	prs_uint16("total_num_entries", ps, &(r_u->total_num_entries));
	prs_uint16("unknown_0        ", ps, &(r_u->unknown_0        ));
	prs_uint32("ptr_entries1", ps, &(r_u->ptr_entries1));

	if (r_u->total_num_entries != 0 && r_u->ptr_entries1 != 0)
	{
		prs_uint32("num_entries2", ps, &(r_u->num_entries2));
		prs_uint32("ptr_entries2", ps, &(r_u->ptr_entries2));
		prs_uint32("num_entries3", ps, &(r_u->num_entries3));

		for (i = 0; i < r_u->num_entries2; i++)
		{
			prs_grow(ps);
			smb_io_sam_entry("", &(r_u->sam[i]), ps);
		}

		for (i = 0; i < r_u->num_entries2; i++)
		{
			prs_grow(ps);
			smb_io_unistr2("", &(r_u->uni_acct_name[i]), r_u->sam[i].hdr_name.buffer, ps);
		}

		prs_align(ps);

		prs_uint32("num_entries4", ps, &(r_u->num_entries4));
	}

	prs_uint32("status", ps, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_ENUM_DOM_ALIASES structure.
********************************************************************/
void make_samr_q_enum_dom_aliases(SAMR_Q_ENUM_DOM_ALIASES *q_e, POLICY_HND *pol, uint32 size)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_enum_dom_aliases\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->unknown_0 = 0;
	q_e->max_size = size;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_enum_dom_aliases(char *desc,  SAMR_Q_ENUM_DOM_ALIASES *q_e, prs_struct *ps)
{
	if (q_e == NULL) return;

	prs_debug(ps, desc, "samr_io_q_enum_dom_aliases");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_e->pol), ps); 
	prs_align(ps);

	prs_uint32("unknown_0", ps, &(q_e->unknown_0));
	prs_uint32("max_size ", ps, &(q_e->max_size ));

	prs_align(ps);
}


/*******************************************************************
makes a SAMR_R_ENUM_DOM_ALIASES structure.
********************************************************************/
void make_samr_r_enum_dom_aliases(SAMR_R_ENUM_DOM_ALIASES *r_u,
		uint32 num_sam_entries, struct smb_passwd grps[MAX_SAM_ENTRIES],
		uint32 status)
{
	int i;

	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_enum_dom_aliases\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	r_u->num_entries  = num_sam_entries;

	if (num_sam_entries > 0)
	{
		r_u->ptr_entries  = 1;
		r_u->num_entries2 = num_sam_entries;
		r_u->ptr_entries2 = 1;
		r_u->num_entries3 = num_sam_entries;

		for (i = 0; i < num_sam_entries; i++)
		{
			char *grp_name = grps[i].smb_name;
			int len_grp_name = strlen(grp_name);

			make_sam_entry(&(r_u->sam[i]), grp_name, grps[i].smb_userid);
			make_unistr2(&(r_u->uni_grp_name[i]), grp_name, len_grp_name);
		}

		r_u->num_entries4 = num_sam_entries;
	}
	else
	{
		r_u->ptr_entries = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_enum_dom_aliases(char *desc,  SAMR_R_ENUM_DOM_ALIASES *r_u, prs_struct *ps)
{
	int i;

	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_enum_dom_aliases");
	

	prs_align(ps);

	prs_uint32("num_entries", ps, &(r_u->num_entries));
	prs_uint32("ptr_entries", ps, &(r_u->ptr_entries));

	if (r_u->num_entries != 0 && r_u->ptr_entries != 0)
	{
		prs_uint32("num_entries2", ps, &(r_u->num_entries2));
		prs_uint32("ptr_entries2", ps, &(r_u->ptr_entries2));
		prs_uint32("num_entries3", ps, &(r_u->num_entries3));

		for (i = 0; i < r_u->num_entries; i++)
		{
			smb_io_sam_entry("", &(r_u->sam[i]), ps);
		}

		for (i = 0; i < r_u->num_entries; i++)
		{
			smb_io_unistr2("", &(r_u->uni_grp_name[i]), r_u->sam[i].hdr_name.buffer, ps);
		}

		prs_align(ps);

		prs_uint32("num_entries4", ps, &(r_u->num_entries4));
	}

	prs_uint32("status", ps, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_QUERY_DISPINFO structure.
********************************************************************/
void make_samr_q_query_dispinfo(SAMR_Q_QUERY_DISPINFO *q_e, POLICY_HND *pol,
				uint16 switch_level, uint32 start_idx, uint32 size)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_query_dispinfo\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->switch_level = switch_level;

	q_e->unknown_0 = 0;
	q_e->start_idx = start_idx;
	q_e->unknown_1 = 0x000007d0;
	q_e->max_size  = size;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_query_dispinfo(char *desc,  SAMR_Q_QUERY_DISPINFO *q_e, prs_struct *ps)
{
	if (q_e == NULL) return;

	prs_debug(ps, desc, "samr_io_q_query_dispinfo");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_e->pol), ps); 
	prs_align(ps);

	prs_uint16("switch_level", ps, &(q_e->switch_level));
	prs_uint16("unknown_0   ", ps, &(q_e->unknown_0   ));
	prs_uint32("start_idx   ", ps, &(q_e->start_idx   ));
	prs_uint32("unknown_1   ", ps, &(q_e->unknown_1   ));
	prs_uint32("max_size    ", ps, &(q_e->max_size    ));

	prs_align(ps);
}


/*******************************************************************
makes a SAM_INFO_2 structure.
********************************************************************/
void make_sam_info_2(SAM_INFO_2 *sam, uint32 acb_mask,
		uint32 start_idx, uint32 num_sam_entries,
		struct smb_passwd pass[MAX_SAM_ENTRIES])
{
	int i;
	int entries_added;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_info_2\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	for (i = start_idx, entries_added = 0; i < num_sam_entries; i++)
	{
		if (IS_BITS_SET_ALL(pass[i].acct_ctrl, acb_mask))
		{
			char *acct_name = pass[i].smb_name;
			char *acct_desc = NULL;

			make_sam_entry2(&(sam->sam[entries_added]), start_idx + entries_added + 1,
			                &pass[i], acct_desc);
			make_sam_str2  (&(sam->str[entries_added]), acct_name, acct_desc);

			entries_added++;
		}
	}

	sam->num_entries   = entries_added;
	sam->ptr_entries   = 1;
	sam->num_entries2  = entries_added;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void sam_io_sam_info_2(char *desc,  SAM_INFO_2 *sam, prs_struct *ps)
{
	int i;

	if (sam == NULL) return;

	prs_debug(ps, desc, "sam_io_sam_info_2");
	

	prs_align(ps);

	prs_uint32("num_entries  ", ps, &(sam->num_entries  ));
	prs_uint32("ptr_entries  ", ps, &(sam->ptr_entries  ));

	prs_uint32("num_entries2 ", ps, &(sam->num_entries2 ));

	for (i = 0; i < sam->num_entries; i++)
	{
		smb_io_sam_entry2("", &(sam->sam[i]), ps);
	}

	for (i = 0; i < sam->num_entries; i++)
	{
		smb_io_sam_str2 ("", &(sam->str[i]),
							 sam->sam[i].hdr_srv_name.buffer,
							 sam->sam[i].hdr_srv_desc.buffer,
							 ps);
	}
}


/*******************************************************************
makes a SAM_INFO_1 structure.
********************************************************************/
void make_sam_info_1(SAM_INFO_1 *sam, uint32 acb_mask,
		uint32 start_idx, uint32 num_sam_entries,
		struct smb_passwd pass[MAX_SAM_ENTRIES])
{
	int i;
	int entries_added;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_info_1\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	for (i = start_idx, entries_added = 0; i < num_sam_entries; i++)
	{
		if (IS_BITS_SET_ALL(pass[i].acct_ctrl, acb_mask))
		{
			char *acct_name = pass[i].smb_name;
			char *full_name = "<full name>";
			char *acct_desc = "<account description>";

			make_sam_entry1(&(sam->sam[entries_added]), start_idx + entries_added + 1,
			                &pass[i], full_name, acct_desc);
			make_sam_str1  (&(sam->str[entries_added]), acct_name, full_name, acct_desc);

			entries_added++;
		}
	}

	sam->num_entries   = entries_added;
	sam->ptr_entries   = 1;
	sam->num_entries2  = entries_added;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void sam_io_sam_info_1(char *desc,  SAM_INFO_1 *sam, prs_struct *ps)
{
	int i;

	if (sam == NULL) return;

	prs_debug(ps, desc, "sam_io_sam_info_1");
	

	prs_align(ps);

	prs_uint32("num_entries  ", ps, &(sam->num_entries  ));
	prs_uint32("ptr_entries  ", ps, &(sam->ptr_entries  ));

	prs_uint32("num_entries2 ", ps, &(sam->num_entries2 ));

	for (i = 0; i < sam->num_entries; i++)
	{
		smb_io_sam_entry1("", &(sam->sam[i]), ps);
	}

	for (i = 0; i < sam->num_entries; i++)
	{
		smb_io_sam_str1 ("", &(sam->str[i]),
							 sam->sam[i].hdr_acct_name.buffer,
							 sam->sam[i].hdr_user_name.buffer,
							 sam->sam[i].hdr_user_desc.buffer,
							 ps);
	}
}


/*******************************************************************
makes a SAMR_R_QUERY_DISPINFO structure.
********************************************************************/
void make_samr_r_query_dispinfo(SAMR_R_QUERY_DISPINFO *r_u,
		uint16 switch_level, SAM_INFO_CTR *ctr, uint32 status)
{
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_query_dispinfo\n"));

	if (status == 0x0)
	{
		r_u->unknown_0 = 0x0000001;
		r_u->unknown_1 = 0x0000001;
	}
	else
	{
		r_u->unknown_0 = 0x0;
		r_u->unknown_1 = 0x0;
	}

	r_u->switch_level = switch_level;
	r_u->ctr = ctr;
	r_u->status = status;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_query_dispinfo(char *desc,  SAMR_R_QUERY_DISPINFO *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_query_dispinfo");
	

	prs_align(ps);

	prs_uint32("unknown_0    ", ps, &(r_u->unknown_0    ));
	prs_uint32("unknown_1    ", ps, &(r_u->unknown_1    ));
	prs_uint16("switch_level ", ps, &(r_u->switch_level ));

	prs_align(ps);

	switch (r_u->switch_level)
	{
		case 0x1:
		{
			sam_io_sam_info_1("users", r_u->ctr->sam.info1, ps);
			break;
		}
		case 0x2:
		{
			sam_io_sam_info_2("servers", r_u->ctr->sam.info2, ps);
			break;
		}
		default:
		{
			DEBUG(5,("samr_io_r_query_dispinfo: unknown switch value\n"));
			break;
		}
	}

	prs_uint32("status", ps, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_ENUM_DOM_GROUPS structure.
********************************************************************/
void make_samr_q_enum_dom_groups(SAMR_Q_ENUM_DOM_GROUPS *q_e, POLICY_HND *pol,
				uint16 switch_level, uint32 start_idx, uint32 size)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_enum_dom_groups\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->switch_level = switch_level;

	q_e->unknown_0 = 0;
	q_e->start_idx = start_idx;
	q_e->unknown_1 = 0x000007d0;
	q_e->max_size  = size;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_enum_dom_groups(char *desc,  SAMR_Q_ENUM_DOM_GROUPS *q_e, prs_struct *ps)
{
	if (q_e == NULL) return;

	prs_debug(ps, desc, "samr_io_q_enum_dom_groups");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_e->pol), ps); 
	prs_align(ps);

	prs_uint16("switch_level", ps, &(q_e->switch_level));
	prs_uint16("unknown_0   ", ps, &(q_e->unknown_0   ));
	prs_uint32("start_idx   ", ps, &(q_e->start_idx   ));
	prs_uint32("unknown_1   ", ps, &(q_e->unknown_1   ));
	prs_uint32("max_size    ", ps, &(q_e->max_size    ));

	prs_align(ps);
}


/*******************************************************************
makes a SAMR_R_ENUM_DOM_GROUPS structure.
********************************************************************/
void make_samr_r_enum_dom_groups(SAMR_R_ENUM_DOM_GROUPS *r_u,
		uint32 start_idx, uint32 num_sam_entries,
		struct smb_passwd pass[MAX_SAM_ENTRIES],
		uint32 status)
{
	int i;
	int entries_added;

	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_enum_dom_groups\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	if (status == 0x0)
	{
		for (i = start_idx, entries_added = 0; i < num_sam_entries; i++)
		{
			char *grp_name = pass[i].smb_name;
			char *grp_desc = "<account description>";

			entries_added++;

			make_sam_entry3(&(r_u->sam[i]), start_idx + entries_added,
			                &pass[i], grp_desc);
			make_sam_str3  (&(r_u->str[i]), grp_name, grp_desc);
		}

		if (entries_added > 0)
		{
			r_u->unknown_0 = 0x0000492;
			r_u->unknown_1 = 0x000049a;
		}
		else
		{
			r_u->unknown_0 = 0x0;
			r_u->unknown_1 = 0x0;
		}
		r_u->switch_level  = 3;
		r_u->num_entries   = entries_added;
		r_u->ptr_entries   = 1;
		r_u->num_entries2  = entries_added;
	}
	else
	{
		r_u->switch_level = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_enum_dom_groups(char *desc,  SAMR_R_ENUM_DOM_GROUPS *r_u, prs_struct *ps)
{
	int i;

	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_enum_dom_groups");
	

	prs_align(ps);

	prs_uint32("unknown_0    ", ps, &(r_u->unknown_0    ));
	prs_uint32("unknown_1    ", ps, &(r_u->unknown_1    ));
	prs_uint32("switch_level ", ps, &(r_u->switch_level ));

	if (r_u->switch_level != 0)
	{
		prs_uint32("num_entries  ", ps, &(r_u->num_entries  ));
		prs_uint32("ptr_entries  ", ps, &(r_u->ptr_entries  ));

		prs_uint32("num_entries2 ", ps, &(r_u->num_entries2 ));

		for (i = 0; i < r_u->num_entries; i++)
		{
			smb_io_sam_entry3("", &(r_u->sam[i]), ps);
		}

		for (i = 0; i < r_u->num_entries; i++)
		{
			smb_io_sam_str3 ("", &(r_u->str[i]),
			                     r_u->sam[i].hdr_grp_name.buffer,
			                     r_u->sam[i].hdr_grp_desc.buffer,
			                     ps);
		}
	}

	prs_uint32("status", ps, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_QUERY_ALIASINFO structure.
********************************************************************/
void make_samr_q_query_aliasinfo(SAMR_Q_QUERY_ALIASINFO *q_e,
				POLICY_HND *pol,
				uint16 switch_level)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_query_aliasinfo\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->switch_level = switch_level;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_query_aliasinfo(char *desc,  SAMR_Q_QUERY_ALIASINFO *q_e, prs_struct *ps)
{
	if (q_e == NULL) return;

	prs_debug(ps, desc, "samr_io_q_query_aliasinfo");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_e->pol), ps); 
	prs_align(ps);

	prs_uint16("switch_level", ps, &(q_e->switch_level));
}


/*******************************************************************
makes a SAMR_R_QUERY_ALIASINFO structure.
********************************************************************/
void make_samr_r_query_aliasinfo(SAMR_R_QUERY_ALIASINFO *r_u,
		uint16 switch_value, char *acct_desc,
		uint32 status)
{
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_query_aliasinfo\n"));

	r_u->ptr = 0;

	if (status == 0)
	{
		r_u->switch_value = switch_value;

		switch (switch_value)
		{
			case 3:
			{
				int acct_len = acct_desc ? strlen(acct_desc) : 0;

				r_u->ptr = 1;

				make_uni_hdr(&(r_u->alias.info3.hdr_acct_desc), acct_len , acct_len, acct_desc ? 1 : 0);
				make_unistr2(&(r_u->alias.info3.uni_acct_desc), acct_desc, acct_len);

				break;
			}

			default:
			{
				DEBUG(4,("make_samr_r_query_aliasinfo: unsupported switch level\n"));
				break;
			}
		}
	}

	r_u->status = status;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_query_aliasinfo(char *desc,  SAMR_R_QUERY_ALIASINFO *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_query_aliasinfo");
	

	prs_align(ps);

	prs_uint32("ptr         ", ps, &(r_u->ptr         ));
	
	if (r_u->ptr != 0)
	{
		prs_uint16("switch_value", ps, &(r_u->switch_value));
		prs_align(ps);

		if (r_u->switch_value != 0)
		{
			switch (r_u->switch_value)
			{
				case 3:
				{
					smb_io_unihdr ("", &(r_u->alias.info3.hdr_acct_desc), ps);
					smb_io_unistr2("", &(r_u->alias.info3.uni_acct_desc), r_u->alias.info3.hdr_acct_desc.buffer, ps);
					break;
				}
				default:
				{
					DEBUG(4,("samr_io_r_query_aliasinfo: unsupported switch level\n"));
					break;
				}
			}
		}
	}

	prs_align(ps);

	prs_uint32("status", ps, &(r_u->status));
}

/*******************************************************************
reads or writes a SAMR_Q_LOOKUP_IDS structure.
********************************************************************/
void samr_io_q_lookup_ids(char *desc,  SAMR_Q_LOOKUP_IDS *q_u, prs_struct *ps)
{
	fstring tmp;
	int i;

	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_lookup_ids");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_u->pol), ps); 
	prs_align(ps);

	prs_uint32("num_sids1", ps, &(q_u->num_sids1));
	prs_uint32("ptr      ", ps, &(q_u->ptr      ));
	prs_uint32("num_sids2", ps, &(q_u->num_sids2));

	for (i = 0; i < q_u->num_sids2; i++)
	{
		sprintf(tmp, "ptr[%02d]", i);
		prs_uint32(tmp, ps, &(q_u->ptr_sid[i]));
	}

	for (i = 0; i < q_u->num_sids2; i++)
	{
		if (q_u->ptr_sid[i] != 0)
		{
			sprintf(tmp, "sid[%02d]", i);
			smb_io_dom_sid2(tmp, &(q_u->sid[i]), ps); 
		}
	}

	prs_align(ps);
}


/*******************************************************************
makes a SAMR_R_LOOKUP_IDS structure.
********************************************************************/
void make_samr_r_lookup_ids(SAMR_R_LOOKUP_IDS *r_u,
		uint32 num_rids, uint32 *rid, uint32 status)
{
	int i;
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_lookup_ids\n"));

	if (status == 0x0)
	{
		r_u->num_entries  = num_rids;
		r_u->ptr = 1;
		r_u->num_entries2 = num_rids;

		for (i = 0; i < num_rids; i++)
		{
			r_u->rid[i] = rid[i];
		}
	}
	else
	{
		r_u->num_entries  = 0;
		r_u->ptr = 0;
		r_u->num_entries2 = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_lookup_ids(char *desc,  SAMR_R_LOOKUP_IDS *r_u, prs_struct *ps)
{
	fstring tmp;
	int i;
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_lookup_ids");
	

	prs_align(ps);

	prs_uint32("num_entries", ps, &(r_u->num_entries));
	prs_uint32("ptr        ", ps, &(r_u->ptr        ));
	prs_uint32("num_entries2", ps, &(r_u->num_entries2));

	if (r_u->num_entries != 0)
	{

		for (i = 0; i < r_u->num_entries2; i++)
		{
			sprintf(tmp, "rid[%02d]", i);
			prs_uint32(tmp, ps, &(r_u->rid[i]));
		}
	}

	prs_uint32("status", ps, &(r_u->status));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_lookup_names(char *desc,  SAMR_Q_LOOKUP_NAMES *q_u, prs_struct *ps)
{
	int i;

	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_lookup_names");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_u->pol), ps); 
	prs_align(ps);

	prs_uint32("num_rids1", ps, &(q_u->num_rids1));
	prs_uint32("rid      ", ps, &(q_u->rid      ));
	prs_uint32("ptr      ", ps, &(q_u->ptr      ));
	prs_uint32("num_rids2", ps, &(q_u->num_rids2));

	for (i = 0; i < q_u->num_rids2; i++)
	{
		smb_io_unihdr ("", &(q_u->hdr_user_name[i]), ps); 
	}
	for (i = 0; i < q_u->num_rids2; i++)
	{
		smb_io_unistr2("", &(q_u->uni_user_name[i]), q_u->hdr_user_name[i].buffer, ps); 
	}

	prs_align(ps);
}


/*******************************************************************
makes a SAMR_R_LOOKUP_NAMES structure.
********************************************************************/
void make_samr_r_lookup_names(SAMR_R_LOOKUP_NAMES *r_u,
		uint32 num_rids, uint32 *rid, uint32 status)
{
	int i;
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_lookup_names\n"));

	if (status == 0x0)
	{
		r_u->num_entries  = num_rids;
		r_u->undoc_buffer = 1;
		r_u->num_entries2 = num_rids;

		for (i = 0; i < num_rids; i++)
		{
			make_dom_rid3(&(r_u->dom_rid[i]), rid[i]);
		}

		r_u->num_entries3 = num_rids;
	}
	else
	{
		r_u->num_entries  = 0;
		r_u->undoc_buffer = 0;
		r_u->num_entries2 = 0;
		r_u->num_entries3 = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_lookup_names(char *desc,  SAMR_R_LOOKUP_NAMES *r_u, prs_struct *ps)
{
	int i;
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_lookup_names");
	

	prs_align(ps);

	prs_uint32("num_entries ", ps, &(r_u->num_entries ));
	prs_uint32("undoc_buffer", ps, &(r_u->undoc_buffer));
	prs_uint32("num_entries2", ps, &(r_u->num_entries2));

	if (r_u->num_entries != 0)
	{
		for (i = 0; i < r_u->num_entries2; i++)
		{
			smb_io_dom_rid3("", &(r_u->dom_rid[i]), ps);
		}

	}

	prs_uint32("num_entries3", ps, &(r_u->num_entries3));

	prs_uint32("status", ps, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_UNKNOWN_12 structure.
********************************************************************/
void make_samr_q_unknown_12(SAMR_Q_UNKNOWN_12 *q_u,
		POLICY_HND *pol, uint32 rid,
		uint32 num_gids, uint32 *gid)
{
	int i;
	if (q_u == NULL) return;

	DEBUG(5,("make_samr_r_unknwon_12\n"));

	memcpy(&(q_u->pol), pol, sizeof(*pol));

	q_u->num_gids1 = num_gids;
	q_u->rid       = rid;
	q_u->ptr       = 0;
	q_u->num_gids2 = num_gids;

	for (i = 0; i < num_gids; i++)
	{
		q_u->gid[i] = gid[i];
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_12(char *desc,  SAMR_Q_UNKNOWN_12 *q_u, prs_struct *ps)
{
	int i;
	fstring tmp;

	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_unknown_12");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_u->pol), ps); 
	prs_align(ps);

	prs_uint32("num_gids1", ps, &(q_u->num_gids1));
	prs_uint32("rid      ", ps, &(q_u->rid      ));
	prs_uint32("ptr      ", ps, &(q_u->ptr      ));
	prs_uint32("num_gids2", ps, &(q_u->num_gids2));

	for (i = 0; i < q_u->num_gids2; i++)
	{
		sprintf(tmp, "gid[%02d]  ", i);
		prs_uint32(tmp, ps, &(q_u->gid[i]));
	}

	prs_align(ps);
}


/*******************************************************************
makes a SAMR_R_UNKNOWN_12 structure.
********************************************************************/
void make_samr_r_unknown_12(SAMR_R_UNKNOWN_12 *r_u,
		uint32 num_aliases, fstring *als_name, uint32 *num_als_usrs,
		uint32 status)
{
	int i;
	if (r_u == NULL || als_name == NULL || num_als_usrs == NULL) return;

	DEBUG(5,("make_samr_r_unknown_12\n"));

	if (status == 0x0)
	{
		r_u->num_aliases1 = num_aliases;
		r_u->ptr_aliases  = 1;
		r_u->num_aliases2 = num_aliases;

		r_u->num_als_usrs1 = num_aliases;
		r_u->ptr_als_usrs  = 1;
		r_u->num_als_usrs2 = num_aliases;

		for (i = 0; i < num_aliases; i++)
		{
			int als_len = als_name[i] != NULL ? strlen(als_name[i]) : 0;
			make_uni_hdr(&(r_u->hdr_als_name[i]), als_len    , als_len, als_name[i] ? 1 : 0);
			make_unistr2(&(r_u->uni_als_name[i]), als_name[i], als_len);
			r_u->num_als_usrs[i] = num_als_usrs[i];
		}
	}
	else
	{
		r_u->num_aliases1 = num_aliases;
		r_u->ptr_aliases  = 0;
		r_u->num_aliases2 = num_aliases;

		r_u->num_als_usrs1 = num_aliases;
		r_u->ptr_als_usrs  = 0;
		r_u->num_als_usrs2 = num_aliases;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_unknown_12(char *desc,  SAMR_R_UNKNOWN_12 *r_u, prs_struct *ps)
{
	int i;
	fstring tmp;
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_unknown_12");
	

	prs_align(ps);

	prs_uint32("num_aliases1", ps, &(r_u->num_aliases1));
	prs_uint32("ptr_aliases ", ps, &(r_u->ptr_aliases ));
	prs_uint32("num_aliases2", ps, &(r_u->num_aliases2));

	if (r_u->ptr_aliases != 0 && r_u->num_aliases1 != 0)
	{
		for (i = 0; i < r_u->num_aliases2; i++)
		{
			sprintf(tmp, "als_hdr[%02d]  ", i);
			smb_io_unihdr ("", &(r_u->hdr_als_name[i]), ps); 
		}
		for (i = 0; i < r_u->num_aliases2; i++)
		{
			sprintf(tmp, "als_str[%02d]  ", i);
			smb_io_unistr2("", &(r_u->uni_als_name[i]), r_u->hdr_als_name[i].buffer, ps); 
		}
	}

	prs_align(ps);

	prs_uint32("num_als_usrs1", ps, &(r_u->num_als_usrs1));
	prs_uint32("ptr_als_usrs ", ps, &(r_u->ptr_als_usrs ));
	prs_uint32("num_als_usrs2", ps, &(r_u->num_als_usrs2));

	if (r_u->ptr_als_usrs != 0 && r_u->num_als_usrs1 != 0)
	{
		for (i = 0; i < r_u->num_als_usrs2; i++)
		{
			sprintf(tmp, "als_usrs[%02d]  ", i);
			prs_uint32(tmp, ps, &(r_u->num_als_usrs[i]));
		}
	}

	prs_uint32("status", ps, &(r_u->status));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_samr_q_open_user(SAMR_Q_OPEN_USER *q_u,
				POLICY_HND *pol,
				uint32 unk_0, uint32 rid)
{
	if (q_u == NULL) return;

	DEBUG(5,("samr_make_q_open_user\n"));

	memcpy(&q_u->domain_pol, pol, sizeof(q_u->domain_pol));
	
	q_u->unknown_0 = unk_0;
	q_u->user_rid  = rid;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_open_user(char *desc,  SAMR_Q_OPEN_USER *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_open_user");
	

	prs_align(ps);

	smb_io_pol_hnd("domain_pol", &(q_u->domain_pol), ps); 
	prs_align(ps);

	prs_uint32("unknown_0", ps, &(q_u->unknown_0));
	prs_uint32("user_rid ", ps, &(q_u->user_rid ));

	prs_align(ps);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_open_user(char *desc,  SAMR_R_OPEN_USER *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_open_user");
	

	prs_align(ps);

	smb_io_pol_hnd("user_pol", &(r_u->user_pol), ps); 
	prs_align(ps);

	prs_uint32("status", ps, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_QUERY_USERGROUPS structure.
********************************************************************/
void make_samr_q_query_usergroups(SAMR_Q_QUERY_USERGROUPS *q_u,
				POLICY_HND *hnd)
{
	if (q_u == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_query_usergroups\n"));

	memcpy(&(q_u->pol), hnd, sizeof(q_u->pol));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_query_usergroups(char *desc,  SAMR_Q_QUERY_USERGROUPS *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_query_usergroups");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_u->pol), ps); 
	prs_align(ps);
}

/*******************************************************************
makes a SAMR_R_QUERY_USERGROUPS structure.
********************************************************************/
void make_samr_r_query_usergroups(SAMR_R_QUERY_USERGROUPS *r_u,
		uint32 num_gids, DOM_GID *gid, uint32 status)
{
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_query_usergroups\n"));

	if (status == 0x0)
	{
		r_u->ptr_0        = 1;
		r_u->num_entries  = num_gids;
		r_u->ptr_1        = 1;
		r_u->num_entries2 = num_gids;

		r_u->gid = gid;

#if 0
		int i;
		for (i = 0; i < num_gids && i < LSA_MAX_GROUPS; i++)
		{
			r_u->gid[i].g_rid = gid[i].g_rid;
			r_u->gid[i].attr  = gid[i].attr ;
		}
#endif

	}
	else
	{
		r_u->ptr_0       = 0;
		r_u->num_entries = 0;
		r_u->ptr_1       = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_query_usergroups(char *desc,  SAMR_R_QUERY_USERGROUPS *r_u, prs_struct *ps)
{
	int i;
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_query_usergroups");
	

	prs_align(ps);

	prs_uint32("ptr_0       ", ps, &(r_u->ptr_0      ));

	if (r_u->ptr_0 != 0)
	{
		prs_uint32("num_entries ", ps, &(r_u->num_entries));
		prs_uint32("ptr_1       ", ps, &(r_u->ptr_1      ));

		if (r_u->num_entries != 0)
		{
			prs_uint32("num_entries2", ps, &(r_u->num_entries2));

			for (i = 0; i < r_u->num_entries2; i++)
			{
				smb_io_gid("", &(r_u->gid[i]), ps);
			}
		}
	}
	prs_uint32("status", ps, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_QUERY_USERINFO structure.
********************************************************************/
void make_samr_q_query_userinfo(SAMR_Q_QUERY_USERINFO *q_u,
				POLICY_HND *hnd, uint16 switch_value)
{
	if (q_u == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_query_userinfo\n"));

	memcpy(&(q_u->pol), hnd, sizeof(q_u->pol));
	q_u->switch_value = switch_value;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_query_userinfo(char *desc,  SAMR_Q_QUERY_USERINFO *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_query_userinfo");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_u->pol), ps); 
	prs_align(ps);

	prs_uint16("switch_value", ps, &(q_u->switch_value)); /* 0x0015 or 0x0011 */

	prs_align(ps);
}

/*******************************************************************
makes a SAM_USER_INFO_10 structure.
********************************************************************/
void make_sam_user_info11(SAM_USER_INFO_11 *usr,
				NTTIME *expiry,
				char *mach_acct,
				uint32 rid_user,
				uint32 rid_group,
				uint16 acct_ctrl)
				
{
	int len_mach_acct;
	if (usr == NULL || expiry == NULL || mach_acct == NULL) return;

	DEBUG(5,("make_samr_r_unknown_24\n"));

	len_mach_acct = strlen(mach_acct);

	memcpy(&(usr->expiry),expiry, sizeof(usr->expiry)); /* expiry time or something? */
	bzero(usr->padding_1, sizeof(usr->padding_1)); /* 0 - padding 24 bytes */

	make_uni_hdr(&(usr->hdr_mach_acct), len_mach_acct, len_mach_acct, 4);  /* unicode header for machine account */
	usr->padding_2 = 0;               /* 0 - padding 4 bytes */

	usr->ptr_1        = 1;            /* pointer */
	bzero(usr->padding_3, sizeof(usr->padding_3)); /* 0 - padding 32 bytes */
	usr->padding_4    = 0;            /* 0 - padding 4 bytes */

	usr->ptr_2        = 1;            /* pointer */
	usr->padding_5    = 0;            /* 0 - padding 4 bytes */

	usr->ptr_3        = 1;          /* pointer */
	bzero(usr->padding_6, sizeof(usr->padding_6)); /* 0 - padding 32 bytes */

	usr->rid_user     = rid_user; 
	usr->rid_group    = rid_group;

	usr->acct_ctrl    = acct_ctrl;
	usr->unknown_3    = 0x0000;

	usr->unknown_4    = 0x003f;       /* 0x003f      - 16 bit unknown */
	usr->unknown_5    = 0x003c;       /* 0x003c      - 16 bit unknown */

	bzero(usr->padding_7, sizeof(usr->padding_7)); /* 0 - padding 16 bytes */
	usr->padding_8    = 0;            /* 0 - padding 4 bytes */
	
	make_unistr2(&(usr->uni_mach_acct), mach_acct, len_mach_acct);  /* unicode string for machine account */

	bzero(usr->padding_9, sizeof(usr->padding_9)); /* 0 - padding 48 bytes */
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void sam_io_user_info11(char *desc,  SAM_USER_INFO_11 *usr, prs_struct *ps)
{
	if (usr == NULL) return;

	prs_debug(ps, desc, "samr_io_r_unknown_24");
	

	prs_align(ps);

	prs_uint8s (False, "padding_0", ps, usr->padding_0, sizeof(usr->padding_0)); 

	smb_io_time("time", &(usr->expiry), ps); 

	prs_uint8s (False, "padding_1", ps, usr->padding_1, sizeof(usr->padding_1));

	smb_io_unihdr ("unihdr", &(usr->hdr_mach_acct), ps); 
	prs_uint32(        "padding_2", ps, &(usr->padding_2));

	prs_uint32(        "ptr_1    ", ps, &(usr->ptr_1    ));
	prs_uint8s (False, "padding_3", ps, usr->padding_3, sizeof(usr->padding_3));
	prs_uint32(        "padding_4", ps, &(usr->padding_4));

	prs_uint32(        "ptr_2    ", ps, &(usr->ptr_2    ));
	prs_uint32(        "padding_5", ps, &(usr->padding_5));

	prs_uint32(        "ptr_3    ", ps, &(usr->ptr_3    ));
	prs_uint8s (False, "padding_6", ps, usr->padding_6, sizeof(usr->padding_6));

	prs_uint32(        "rid_user ", ps, &(usr->rid_user ));
	prs_uint32(        "rid_group", ps, &(usr->rid_group));
	prs_uint16(        "acct_ctrl", ps, &(usr->acct_ctrl));
	prs_uint16(        "unknown_3", ps, &(usr->unknown_3));
	prs_uint16(        "unknown_4", ps, &(usr->unknown_4));
	prs_uint16(        "unknown_5", ps, &(usr->unknown_5));

	prs_uint8s (False, "padding_7", ps, usr->padding_7, sizeof(usr->padding_7));
	prs_uint32(        "padding_8", ps, &(usr->padding_8));
	
	smb_io_unistr2("unistr2", &(usr->uni_mach_acct), True, ps); 
	prs_align(ps);

	prs_uint8s (False, "padding_9", ps, usr->padding_9, sizeof(usr->padding_9));
}

/*************************************************************************
 make_sam_user_info15

 unknown_3 = 0x00ff ffff
 unknown_5 = 0x0002 0000
 unknown_6 = 0x0000 04ec 

 *************************************************************************/
void make_sam_user_info15(SAM_USER_INFO_15 *usr,

	NTTIME *logon_time,
	NTTIME *logoff_time,
	NTTIME *kickoff_time,
	NTTIME *pass_last_set_time,
	NTTIME *pass_can_change_time,
	NTTIME *pass_must_change_time,

	char *user_name,
	char *full_name,
	char *home_dir,
	char *dir_drive,
	char *profile_path,
	char *logon_script,
	char *description,

	uint16 logon_count,
	uint16 bad_pw_count,

	uint32 unknown_0,
	uint32 unknown_1,

	uint32 user_rid,
	uint32 group_rid,
	uint16 acb_info, 

	uint32 unknown_3,
	uint16 logon_divs,
	LOGON_HRS *hrs,
	uint32 unknown_5,
	uint32 unknown_6)
{
	int len_user_name    = user_name    != NULL ? strlen(user_name   ) : 0;
	int len_full_name    = full_name    != NULL ? strlen(full_name   ) : 0;
	int len_dir_drive    = dir_drive    != NULL ? strlen(dir_drive   ) : 0;
	int len_home_dir     = home_dir     != NULL ? strlen(home_dir    ) : 0;
	int len_profile_path = profile_path != NULL ? strlen(profile_path) : 0;
	int len_logon_script = logon_script != NULL ? strlen(logon_script) : 0;
	int len_description  = description  != NULL ? strlen(description ) : 0;

	usr->logon_time            = *logon_time;
	usr->logoff_time           = *logoff_time;
	usr->kickoff_time          = *kickoff_time;
	usr->pass_last_set_time    = *pass_last_set_time;
	usr->pass_can_change_time  = *pass_can_change_time;
	usr->pass_must_change_time = *pass_must_change_time;

	make_uni_hdr(&(usr->hdr_user_name   ), len_user_name   , len_user_name   , len_user_name    != 0);
	make_uni_hdr(&(usr->hdr_full_name   ), len_full_name   , len_full_name   , len_full_name    != 0);
	make_uni_hdr(&(usr->hdr_dir_drive   ), len_dir_drive   , len_dir_drive   , len_dir_drive    != 0);
	make_uni_hdr(&(usr->hdr_home_dir    ), len_home_dir    , len_home_dir    , len_home_dir     != 0);
	make_uni_hdr(&(usr->hdr_profile_path), len_profile_path, len_profile_path, len_profile_path != 0);
	make_uni_hdr(&(usr->hdr_logon_script), len_logon_script, len_logon_script, len_logon_script != 0);
	make_uni_hdr(&(usr->hdr_description ), len_description , len_description , len_description  != 0);

	usr->logon_count = logon_count;
	usr->bad_pw_count = bad_pw_count;

	usr->ptr_padding2 = 1;
	usr->unknown_0 = unknown_0;

	usr->ptr_padding3 = 1;
	usr->unknown_1 = unknown_1;

	usr->ptr_unknown6 = 1;
	bzero(usr->unknown_2, sizeof(usr->unknown_2));

	usr->user_rid  = user_rid;
	usr->group_rid = group_rid;
	usr->acb_info = acb_info;
	usr->unknown_3 = unknown_3;

	usr->logon_divs = logon_divs;
	usr->ptr_logon_hrs = hrs ? 1 : 0;
	usr->unknown_5 = unknown_5;

	bzero(usr->padding1, sizeof(usr->padding1));

	make_unistr2(&(usr->uni_user_name   ), user_name   , len_user_name   );
	make_unistr2(&(usr->uni_full_name   ), full_name   , len_full_name   );
	make_unistr2(&(usr->uni_dir_drive   ), dir_drive   , len_dir_drive   );
	make_unistr2(&(usr->uni_home_dir    ), home_dir    , len_home_dir    );
	make_unistr2(&(usr->uni_profile_path), profile_path, len_profile_path);
	make_unistr2(&(usr->uni_logon_script), logon_script, len_logon_script);
	make_unistr2(&(usr->uni_description ), description , len_description );

	bzero(usr->padding2, sizeof(usr->padding2));
	usr->padding3 = 0;
	
	usr->unknown_6 = unknown_6;
	usr->padding4 = 0;

	if (hrs)
	{
		memcpy(&(usr->logon_hrs), hrs, sizeof(usr->logon_hrs));
	}
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void sam_io_user_info15(char *desc,  SAM_USER_INFO_15 *usr, prs_struct *ps)
{
	if (usr == NULL) return;

	prs_debug(ps, desc, "lsa_io_user_info");
	

	prs_align(ps);
	
	smb_io_time("time", &(usr->logon_time)           , ps); /* logon time */
	smb_io_time("time", &(usr->logoff_time)          , ps); /* logoff time */
	smb_io_time("time", &(usr->kickoff_time)         , ps); /* kickoff time */
	smb_io_time("time", &(usr->pass_last_set_time)   , ps); /* password last set time */
	smb_io_time("time", &(usr->pass_can_change_time) , ps); /* password can change time */
	smb_io_time("time", &(usr->pass_must_change_time), ps); /* password must change time */

	smb_io_unihdr("unihdr", &(usr->hdr_user_name)   , ps); /* username unicode string header */
	smb_io_unihdr("unihdr", &(usr->hdr_full_name)   , ps); /* user's full name unicode string header */
	smb_io_unihdr("unihdr", &(usr->hdr_home_dir)    , ps); /* home directory unicode string header */
	smb_io_unihdr("unihdr", &(usr->hdr_profile_path), ps); /* profile path unicode string header */
	smb_io_unihdr("unihdr", &(usr->hdr_logon_script), ps); /* logon script unicode string header */
	smb_io_unihdr("unihdr", &(usr->hdr_dir_drive)   , ps); /* home directory drive unicode string header */
	smb_io_unihdr("unihdr", &(usr->hdr_description) , ps); /* home directory drive unicode string header */

	prs_uint16("logon_count   ", ps, &(usr->logon_count ));  /* logon count */
	prs_uint16("bad_pw_count  ", ps, &(usr->bad_pw_count)); /* bad password count */

	prs_uint32("ptr_padding2", ps, &(usr->ptr_padding2));
	prs_uint32("unknown_0", ps, &(usr->unknown_0));

	prs_uint32("ptr_padding3", ps, &(usr->ptr_padding3));
	prs_uint32("unknown_1", ps, &(usr->unknown_1));

	prs_uint32("ptr_unknown6", ps, &(usr->ptr_unknown6));
	prs_uint8s (False, "unknown_2", ps, usr->unknown_2, sizeof(usr->unknown_2));

	prs_uint32("user_rid      ", ps, &(usr->user_rid     ));       /* User ID */
	prs_uint32("group_rid     ", ps, &(usr->group_rid    ));      /* Group ID */
	prs_uint16("acb_info      ", ps, &(usr->acb_info     ));      /* Group ID */
	prs_align(ps);

	prs_uint32("unknown_3     ", ps, &(usr->unknown_3    ));
	prs_uint16("logon_divs    ", ps, &(usr->logon_divs   ));     /* logon divisions per week */
	prs_align(ps);
	prs_uint32("ptr_logon_hrs ", ps, &(usr->ptr_logon_hrs));
	prs_uint32("unknown_5     ", ps, &(usr->unknown_5    ));

	prs_uint8s (False, "padding1      ", ps, usr->padding1, sizeof(usr->padding1));

	/* here begins pointed-to data */

	smb_io_unistr2("unistr2", &(usr->uni_user_name)   , usr->hdr_user_name   .buffer, ps); /* username unicode string */
	smb_io_unistr2("unistr2", &(usr->uni_full_name)   , usr->hdr_full_name   .buffer, ps); /* user's full name unicode string */
	smb_io_unistr2("unistr2", &(usr->uni_dir_drive)   , usr->hdr_dir_drive   .buffer, ps); /* home directory drive unicode string */
	smb_io_unistr2("unistr2", &(usr->uni_home_dir)    , usr->hdr_home_dir    .buffer, ps); /* home directory unicode string */
	smb_io_unistr2("unistr2", &(usr->uni_profile_path), usr->hdr_profile_path.buffer, ps); /* profile path unicode string */
	smb_io_unistr2("unistr2", &(usr->uni_logon_script), usr->hdr_logon_script.buffer, ps); /* logon script unicode string */
	smb_io_unistr2("unistr2", &(usr->uni_description ), usr->hdr_description .buffer, ps); /* user description unicode string */

	prs_uint8s (False, "padding2", ps, usr->padding2, sizeof(usr->padding2));

	prs_uint32("padding3      ", ps, &(usr->padding3   ));

	prs_uint32("unknown_6     ", ps, &(usr->unknown_6  ));
	prs_uint32("padding4      ", ps, &(usr->padding4   ));

	if (usr->ptr_logon_hrs)
	{
		smb_io_logon_hrs("logon_hrs", &(usr->logon_hrs)   , ps);
	}
}



/*******************************************************************
makes a SAMR_R_QUERY_USERINFO structure.
********************************************************************/
void make_samr_r_query_userinfo(SAMR_R_QUERY_USERINFO *r_u,
				uint16 switch_value, void *info, uint32 status)
				
{
	if (r_u == NULL || info == NULL) return;

	DEBUG(5,("make_samr_r_query_userinfo\n"));

	r_u->ptr = 0;
	r_u->switch_value = 0;

	if (status == 0)
	{
		r_u->switch_value = switch_value;

		switch (switch_value)
		{
			case 0x10:
			{
				r_u->ptr = 1;
				r_u->info.id10 = (SAM_USER_INFO_10*)info;

				break;
			}

			case 0x11:
			{
				r_u->ptr = 1;
				r_u->info.id11 = (SAM_USER_INFO_11*)info;

				break;
			}

			case 0x15:
			{
				r_u->ptr = 1;
				r_u->info.id15 = (SAM_USER_INFO_15*)info;

				break;
			}

			default:
			{
				DEBUG(4,("make_samr_r_query_aliasinfo: unsupported switch level\n"));
				break;
			}
		}
	}

	r_u->status = status;         /* return status */
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_query_userinfo(char *desc,  SAMR_R_QUERY_USERINFO *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_query_userinfo");
	

	prs_align(ps);

	prs_uint32("ptr         ", ps, &(r_u->ptr         ));
	prs_uint16("switch_value", ps, &(r_u->switch_value));
	prs_align(ps);

	if (r_u->ptr != 0 && r_u->switch_value != 0)
	{
		switch (r_u->switch_value)
		{
/*
			case 0x10:
			{
				if (r_u->info.id10 != NULL)
				{
					sam_io_user_info10("", r_u->info.id10, ps);
				}
				else
				{
					DEBUG(2,("samr_io_r_query_userinfo: info pointer not initialised\n"));
					return NULL;
				}
				break;
			}
			case 0x11:
			{
				if (r_u->info.id11 != NULL)
				{
					sam_io_user_info11("", r_u->info.id11, ps);
				}
				else
				{
					DEBUG(2,("samr_io_r_query_userinfo: info pointer not initialised\n"));
					return NULL;
				}
				break;
			}
*/
			case 0x15:
			{
				if (r_u->info.id15 != NULL)
				{
					sam_io_user_info15("", r_u->info.id15, ps);
				}
				else
				{
					DEBUG(2,("samr_io_r_query_userinfo: info pointer not initialised\n"));
					return;
				}
				break;
			}
			default:
			{
				DEBUG(2,("samr_io_r_query_userinfo: unknown switch level\n"));
				break;
			}
				
		}
	}

	prs_uint32("status", ps, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_UNKNOWN_21 structure.
********************************************************************/
void make_samr_q_unknown_21(SAMR_Q_UNKNOWN_21 *q_c,
				POLICY_HND *hnd, uint16 unk_1, uint16 unk_2)
{
	if (q_c == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_unknown_21\n"));

	memcpy(&(q_c->group_pol), hnd, sizeof(q_c->group_pol));
	q_c->unknown_1 = unk_1;
	q_c->unknown_2 = unk_2;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_21(char *desc,  SAMR_Q_UNKNOWN_21 *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_unknown_21");
	

	prs_align(ps);

	smb_io_pol_hnd("group_pol", &(q_u->group_pol), ps); 
	prs_align(ps);

	prs_uint16("unknown_1", ps, &(q_u->unknown_1));
	prs_uint16("unknown_2", ps, &(q_u->unknown_2));
}

/*******************************************************************
makes a SAMR_Q_UNKNOWN_13 structure.
********************************************************************/
void make_samr_q_unknown_13(SAMR_Q_UNKNOWN_13 *q_c,
				POLICY_HND *hnd, uint16 unk_1, uint16 unk_2)
{
	if (q_c == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_unknown_13\n"));

	memcpy(&(q_c->alias_pol), hnd, sizeof(q_c->alias_pol));
	q_c->unknown_1 = unk_1;
	q_c->unknown_2 = unk_2;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_13(char *desc,  SAMR_Q_UNKNOWN_13 *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_unknown_13");
	

	prs_align(ps);

	smb_io_pol_hnd("alias_pol", &(q_u->alias_pol), ps); 
	prs_align(ps);

	prs_uint16("unknown_1", ps, &(q_u->unknown_1));
	prs_uint16("unknown_2", ps, &(q_u->unknown_2));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_32(char *desc,  SAMR_Q_UNKNOWN_32 *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_unknown_32");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(q_u->pol), ps); 
	prs_align(ps);

	smb_io_unihdr ("", &(q_u->hdr_mach_acct), ps); 
	smb_io_unistr2("", &(q_u->uni_mach_acct), q_u->hdr_mach_acct.buffer, ps); 

	prs_align(ps);

	prs_uint32("acct_ctrl", ps, &(q_u->acct_ctrl));
	prs_uint16("unknown_1", ps, &(q_u->unknown_1));
	prs_uint16("unknown_2", ps, &(q_u->unknown_2));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_unknown_32(char *desc,  SAMR_R_UNKNOWN_32 *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_unknown_32");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(r_u->pol), ps); 
	prs_align(ps);

	prs_uint32("status", ps, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_CONNECT structure.
********************************************************************/
void make_samr_q_connect(SAMR_Q_CONNECT *q_u,
				char *srv_name, uint32 unknown_0)
{
	int len_srv_name = strlen(srv_name);

	if (q_u == NULL) return;

	DEBUG(5,("make_q_connect\n"));

	/* make PDC server name \\server */
	make_unistr2(&(q_u->uni_srv_name), srv_name, len_srv_name);  

	/* example values: 0x0000 0002 */
	q_u->unknown_0 = unknown_0; 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_connect(char *desc,  SAMR_Q_CONNECT *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_connect");
	

	prs_align(ps);

	prs_uint32("ptr_srv_name", ps, &(q_u->ptr_srv_name));
	smb_io_unistr2("", &(q_u->uni_srv_name), q_u->ptr_srv_name, ps); 

	prs_align(ps);

	prs_uint32("unknown_0   ", ps, &(q_u->unknown_0   ));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_connect(char *desc,  SAMR_R_CONNECT *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_connect");
	

	prs_align(ps);

	smb_io_pol_hnd("connect_pol", &(r_u->connect_pol), ps); 
	prs_align(ps);

	prs_uint32("status", ps, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_OPEN_ALIAS structure.
********************************************************************/
void make_samr_q_open_alias(SAMR_Q_OPEN_ALIAS *q_u,
				uint32 unknown_0, uint32 rid)
{
	if (q_u == NULL) return;

	DEBUG(5,("make_q_open_alias\n"));

	/* example values: 0x0000 0008 */
	q_u->unknown_0 = unknown_0; 

	q_u->rid_alias = rid; 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_open_alias(char *desc,  SAMR_Q_OPEN_ALIAS *q_u, prs_struct *ps)
{
	if (q_u == NULL) return;

	prs_debug(ps, desc, "samr_io_q_open_alias");
	

	prs_align(ps);

	prs_uint32("unknown_0", ps, &(q_u->unknown_0));
	prs_uint32("rid_alias", ps, &(q_u->rid_alias));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_open_alias(char *desc,  SAMR_R_OPEN_ALIAS *r_u, prs_struct *ps)
{
	if (r_u == NULL) return;

	prs_debug(ps, desc, "samr_io_r_open_alias");
	

	prs_align(ps);

	smb_io_pol_hnd("pol", &(r_u->pol), ps); 
	prs_align(ps);

	prs_uint32("status", ps, &(r_u->status));
}

