
/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#ifdef SYSLOG
#undef SYSLOG
#endif

#include "../includes.h"

extern int DEBUGLEVEL;


extern struct pipe_id_info pipe_names[];

/*******************************************************************
 uses SMBreadX to get rest of rpc data
 ********************************************************************/
static BOOL rpc_read(struct cli_state *cli, int t_idx, uint16 fnum,
				prs_struct *rdata, uint32 data_to_read)
{
	uint32 data_offset = rdata->data->data_used;
	int size = 512;
	int num_read;
	char *data = rdata->data->data;
	uint32 err;
	data += rdata->data->data_used;

	if (data_offset + data_to_read > rdata->data->data_size)
	{
		mem_grow_data(&rdata->data, True, rdata->data->data_used + data_to_read);
		DEBUG(5,("rpc_read: grow buffer to %d\n", rdata->data->data_used));
	}

	do /* read data using SMBreadX */
	{
		if (size > data_to_read) size = data_to_read;

		if (data_offset + size > rdata->data->data_size)
		{
			mem_grow_data(&rdata->data, True, rdata->data->data_used + size);
			DEBUG(5,("rpc_read: grow buffer to %d\n", rdata->data->data_used));
		}

		num_read = cli_readx(cli, t_idx, fnum, data, data_offset, size);

		DEBUG(5,("rpc_read: read offset: %d read: %d to read: %d\n",
				  data_offset, num_read, data_to_read));

		data_to_read -= num_read;
		data_offset  += num_read;
		data         += num_read;

		if (cli_error(cli, NULL, &err)) return False;

	} while (num_read > 0 && data_to_read > 0); /* && err == (0x80000000 | STATUS_BUFFER_OVERFLOW)); */

	mem_realloc_data(rdata->data, data_offset);

	DEBUG(5,("rpc_read: data supposedly left to read:0x%x\n", data_to_read));

	return data_to_read == 0;
}

/****************************************************************************
 checks the header
 ****************************************************************************/
static BOOL rpc_check_hdr(prs_struct *rdata, uint8 expected_pkt_type,
				BOOL *first, BOOL *last, uint32 *len)
{
	RPC_HDR    rhdr;
	RPC_HDR_RR rhdr_rr;

	smb_io_rpc_hdr   ("rpc_hdr   ", &rhdr   , rdata);
	smb_io_rpc_hdr_rr("rpc_hdr_rr", &rhdr_rr, rdata);

	prs_align(rdata);

	if (!rdata->offset || rdata->offset != 0x18)
	{
		DEBUG(5,("cli_pipe: error in rpc header\n"));
		return False;
	}

	(*first) = IS_BITS_SET_ALL(rhdr.flags, RPC_FLG_FIRST);
	(*last ) = IS_BITS_SET_ALL(rhdr.flags, RPC_FLG_LAST );

	(*len) = rhdr.frag_len - rdata->data->data_used;

	if (expected_pkt_type != 0xff && rhdr.pkt_type != expected_pkt_type)
	{
		DEBUG(4,("rpc_check_header: expected packet type received\n"));
		return False;
	}

	return True;
}

/****************************************************************************
 send data on an rpc pipe, which *must* be in one fragment.
 receive response data from an rpc pipe, which may be large...

 read the first fragment: unfortunately have to use SMBtrans for the first
 bit, then SMBreadX for subsequent bits.

 if first fragment received also wasn't the last fragment, continue
 getting fragments until we _do_ receive the last fragment.

 [note: from a data abstraction viewpoint, this function is marginally
        complicated by the return side of cli_api_pipe getting in the way
        (i.e, the SMB header stuff).  the proper way to do this is to split
        cli_api_pipe down into receive / transmit.  oh, and split cli_readx
        down.  in other words, state-based (kernel) techniques...]

 ****************************************************************************/
BOOL rpc_api_pipe(struct cli_state *cli, int t_idx,
				uint16 cmd, uint16 fnum,
				prs_struct *param , prs_struct *data,
				prs_struct *rparam, prs_struct *rdata)
{
	uint32 len;

	uint16 setup[2]; /* only need 2 uint16 setup parameters */
	uint32 err;
	BOOL first = True;
	BOOL last  = True;

	/* prepare return data and params */

	rparam->align = 4;
	rparam->io = True;
	rparam->depth = 0;

	rdata->align = 4;
	rdata->io = True;
	rdata->depth = 0;

	/* create setup parameters. */
	setup[0] = cmd; 
	setup[1] = fnum; /* pipe file handle.  got this from an SMBcreateX. */

	/* send the data: receive a response. */
	if (!cli_api_pipe(cli, t_idx, "\\PIPE\\\0\0\0", 8,

	            param != NULL ? param->data->data_used : 0,
	            data  != NULL ? data ->data->data_used : 0,
	            2,

	            0,
	            data  != NULL ? 1024 : 0 ,

	            &(rparam->data->data_used), &(rdata->data->data_used),

	            param != NULL ? param->data->data : NULL,
	            data  != NULL ? data ->data->data : NULL,
	            setup,

	            &(rparam->data->data), &(rdata->data->data)))
	{
		DEBUG(5, ("cli_pipe: return critical error\n"));
		return False;
	}

	if (cli_error(cli, NULL, &err)) return False;

	if (rdata->data->data == NULL) return False;

	/**** parse the header: check it's a response record */

	rdata->offset = 0;
	if (!rpc_check_hdr(rdata, 0xff, &first, &last, &len)) return False;
	
	/* check if data to be sent back was too large for one SMB. */
	/* err status is only informational: the _real_ check is on the length */
	if (len < rdata->data->data_used) /* || err == (0x80000000 | STATUS_BUFFER_OVERFLOW)) */
	{
		if (!rpc_read(cli, t_idx, fnum, rdata, len)) return False;
	}

	/* only one rpc fragment, and it has been read */
	if (first && last) return True;

	while (!last) /* read more fragments until we get the last one */
	{
		RPC_HDR    rhdr;
		RPC_HDR_RR rhdr_rr;
		int num_read;
		prs_struct hps;
		struct mem_buf hdr_buf;
		char tmp_data[0x18];

		mem_create(&hdr_buf, tmp_data, 0x18, 0);
		
		hps.data = &hdr_buf;
		hps.offset = 0;
		hps.align = 4;
		hps.io = True;
		hps.depth = 0;

		num_read = cli_readx(cli, t_idx, fnum, hdr_buf.data, 0, 0x18);
		DEBUG(5,("rpc_api_pipe: read header (size:%d)\n", num_read));

		if (num_read != 0x18) return False;

		smb_io_rpc_hdr   ("rpc_hdr   ", &rhdr   , &(hps));
		smb_io_rpc_hdr_rr("rpc_hdr_rr", &rhdr_rr, &(hps));

		if (cli_error(cli, NULL, &err)) return False;

		first = IS_BITS_SET_ALL(rhdr.flags, RPC_FLG_FIRST);
		last  = IS_BITS_SET_ALL(rhdr.flags, RPC_FLG_LAST );

		if (first)
		{
			DEBUG(4,("rpc_api_pipe: wierd rpc header received\n"));
			return False;
		}

		len = rhdr.frag_len - hps.offset;
		if (!rpc_read(cli, t_idx, fnum, rdata, len)) return False;
	}

	return True;
}

/*******************************************************************
 creates a DCE/RPC request

 - initialises the parse structure.
 - dynamically allocates the header data structure
 - caller is expected to free the header data structure once used.

 ********************************************************************/
static BOOL create_rpc_request(prs_struct *rhdr, uint8 op_num, int data_len)
{
	RPC_HDR_RR hdr_rr;
	RPC_HDR    hdr;

	make_rpc_hdr   (&hdr   , RPC_REQUEST, RPC_FLG_FIRST | RPC_FLG_LAST,
	                get_rpc_call_id(), data_len + 0x18);
	make_rpc_hdr_rr(&hdr_rr, data_len, op_num);

	rhdr->align = 4;
	rhdr->depth = 0;
	rhdr->io = False;
	rhdr->data = NULL;
	rhdr->offset = 0;
	
	mem_buf_init(&(rhdr->data), 0);
	mem_alloc_data(rhdr->data, 0x18);

	rhdr->data->offset.start = 0;
	rhdr->data->offset.end   = 0x18;

	smb_io_rpc_hdr   ("hdr"   , &hdr   , rhdr);
	smb_io_rpc_hdr_rr("hdr_rr", &hdr_rr, rhdr);

	if (rhdr->data == NULL || rhdr->offset != 0x18) return False;

	rhdr->data->offset.start = 0;
	rhdr->data->offset.end   = rhdr->offset;

	return True;
}


/****************************************************************************
 send a request on an rpc pipe.
 ****************************************************************************/
BOOL rpc_api_pipe_req(struct cli_state *cli, int t_idx, uint16 fnum,
				uint8 op_num,
				prs_struct *data, prs_struct *rdata)
{
	BOOL ret;

	/* fudge this, at the moment: create the header; memcpy the data.  oops. */
	prs_struct hdr;
	int data_len = data->data->offset.end + 0x18;

	create_rpc_request(&hdr, op_num, data_len);
	mem_realloc_data(hdr.data, data_len);
	mem_buf_copy(mem_data(&(data->data), 0x18), hdr.data, 0, data_len - 0x18);

	ret = rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &hdr, NULL, rdata);

	prs_mem_free(&hdr);

	return ret;
}


/****************************************************************************
do an rpc bind
****************************************************************************/
BOOL rpc_pipe_set_hnd_state(struct cli_state *cli, int t_idx,
				char *pipe_name, uint16 fnum, uint16 device_state)
{
	prs_struct param;
	prs_struct rdata;
	prs_struct rparam;
	BOOL state_set = False;

	if (pipe_name == NULL) return False;

	prs_init(&param , 2, 4, 0            , False, 0);
	prs_init(&rdata , 0, 4, SAFETY_MARGIN, True , 0);
	prs_init(&rparam, 0, 4, SAFETY_MARGIN, True , 0);

	DEBUG(5,("Bind RPC Pipe[%x]: %s - device state:%x\n",
	          fnum, pipe_name, device_state));

	/* create data parameters: device state */
	SSVAL(param.data, 0, device_state);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0001, fnum,
				&param , NULL,
				&rparam, &rdata))
	{
		DEBUG(5, ("cli_pipe: return OK\n"));
		state_set = True;
	}

	prs_mem_free(&param );
	prs_mem_free(&rparam);
	prs_mem_free(&rdata );

	return state_set;
}

/****************************************************************************
 check the rpc bind acknowledge response
****************************************************************************/
static BOOL valid_pipe_name(char *pipe_name,
				RPC_IFACE *abstract, RPC_IFACE *transfer)
{
	int pipe_idx = 0;

	while (pipe_names[pipe_idx].client_pipe != NULL)
	{
		if (strcmp(pipe_name, pipe_names[pipe_idx].client_pipe ) == 0)
		{
			DEBUG(5,("Bind Abstract Syntax: "));	
			dump_data(5, (uchar*)&(pipe_names[pipe_idx].abstr_syntax), sizeof(pipe_names[pipe_idx].abstr_syntax));
			DEBUG(5,("Bind Transfer Syntax: "));
			dump_data(5, (uchar*)&(pipe_names[pipe_idx].trans_syntax), sizeof(pipe_names[pipe_idx].trans_syntax));

			/* copy the required syntaxes out so we can do the right bind */
			memcpy(transfer, &(pipe_names[pipe_idx].trans_syntax), sizeof(pipe_names[pipe_idx].trans_syntax));
			memcpy(abstract, &(pipe_names[pipe_idx].abstr_syntax), sizeof(pipe_names[pipe_idx].abstr_syntax));

			return True;
		}
		pipe_idx++;
	};

	DEBUG(5,("Bind RPC Pipe[%s] unsupported\n", pipe_name));
	return False;
}

/****************************************************************************
 check the rpc bind acknowledge response
****************************************************************************/
static BOOL check_bind_response(RPC_HDR_BA *hdr_ba, char *pipe_name, RPC_IFACE *transfer)
{
	int i = 0;

	while ((pipe_names[i].client_pipe != NULL))
	{
		DEBUG(6,("bind_rpc_pipe: searching pipe name: client:%s server:%s\n",
				  pipe_names[i].client_pipe , pipe_names[i].server_pipe ));

		if ((strcmp(pipe_name, pipe_names[i].client_pipe ) == 0))
		{
			if (strcmp(hdr_ba->addr.str, pipe_names[i].server_pipe ) == 0)
			{
				DEBUG(5,("bind_rpc_pipe: server pipe_name found: %s\n",
						pipe_names[i].server_pipe ));
				break;
			}
			else
			{
				DEBUG(2,("bind_rpc_pipe: pipe_name %s != expected pipe %s\n",
						pipe_names[i].server_pipe , hdr_ba->addr.str));
				return False;
			}
		}
		else
		{
			i++;
		}
	}

	if (pipe_names[i].server_pipe == NULL)
	{
		DEBUG(2,("bind_rpc_pipe: pipe name %s unsupported\n", hdr_ba->addr.str));
		return False;
	}

	/* check the transfer syntax */
	if (!((hdr_ba->transfer.version == transfer->version) &&
	      (memcmp(hdr_ba->transfer.data, transfer->data,
						sizeof(transfer->version)) ==0)))
	{
		DEBUG(2,("bind_rpc_pipe: transfer syntax differs\n"));
		return False;
	}
	
	/* lkclXXXX only accept one result: check the result(s) */
	if (hdr_ba->res.num_results != 0x1 || hdr_ba->res.result != 0)
	{
		DEBUG(2,("bind_rpc_pipe: bind denied results: %d reason: %x\n",
					  hdr_ba->res.num_results,
					  hdr_ba->res.reason));
	}
		
	DEBUG(5,("bind_rpc_pipe: accepted!\n"));
	return True;
}

/****************************************************************************
do an rpc bind
****************************************************************************/
BOOL rpc_pipe_bind(struct cli_state *cli, int t_idx, char *pipe_name, uint16 fnum, 
				RPC_IFACE *abstract, RPC_IFACE *transfer)
{
	prs_struct data;
	prs_struct rdata;
	prs_struct rparam;

	RPC_HDR    hdr;
	RPC_HDR_RB hdr_rb;

    BOOL valid_ack = False;

	if (pipe_name == NULL || abstract == NULL || transfer == NULL) return False;

	DEBUG(5,("Bind RPC Pipe[%x]: %s\n", fnum, pipe_name));

	if (!valid_pipe_name(pipe_name, abstract, transfer)) return False;

	prs_init(&data  , 1024, 4, SAFETY_MARGIN, False, 0);
	prs_init(&rdata , 0   , 4, SAFETY_MARGIN, True , 0);
	prs_init(&rparam, 0   , 4, SAFETY_MARGIN, True , 0);

	/* create the request RPC_HDR_RB */
	make_rpc_hdr_rb(&hdr_rb, 
	                0x1630, 0x1630, 0x0,
	                0x1, 0x0, 0x1,
					abstract, transfer);

	/* stream the bind request data */
	data.offset = 0x10;
	smb_io_rpc_hdr_rb("", &hdr_rb,  &data);
	data.data->data_used = data.offset;

	/* create the request RPC_HDR */
	make_rpc_hdr(&hdr, RPC_BIND, 0x0, get_rpc_call_id(), data.offset - 0x10);

	/* stream the header into data */
	data.offset = 0;
	smb_io_rpc_hdr("", &hdr, &data);

	data.offset = data.data->data_used;
	data.data->offset.start = 0;
	data.data->offset.end   = data.offset;

	/* send data on \PIPE\.  receive a response */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparam, &rdata))
	{
		RPC_HDR_BA hdr_ba;

		DEBUG(5, ("rpc_api_pipe: return OK\n"));

		smb_io_rpc_hdr_ba("", &hdr_ba, &rdata);

		if (rdata.offset != 0) valid_ack = check_bind_response(&hdr_ba, pipe_name, transfer);
	}

	prs_mem_free(&data  );
	prs_mem_free(&rdata );
	prs_mem_free(&rparam);

	return valid_ack;
}

