/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../includes.h"

extern int DEBUGLEVEL;


/*******************************************************************
makes a SAMR_Q_CLOSE_HND structure.
********************************************************************/
void make_samr_q_close_hnd(SAMR_Q_CLOSE_HND *q_c, POLICY_HND *hnd)
{
	if (q_c == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_close_hnd\n"));

	memcpy(&(q_c->pol), hnd, sizeof(q_c->pol));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_close_hnd(char *desc, BOOL io, SAMR_Q_CLOSE_HND *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_close_hnd %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_close_hnd(char *desc, BOOL io, SAMR_R_CLOSE_HND *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_close_hnd %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(r_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_samr_q_open_domain(SAMR_Q_OPEN_DOMAIN *q_u,
				POLICY_HND *pol, uint32 rid, char *sid)
{
	if (q_u == NULL) return;

	DEBUG(5,("samr_make_q_open_domain\n"));

	memcpy(&q_u->pol, pol, sizeof(q_u->pol));
	q_u->rid = rid;
	make_dom_sid2(&(q_u->dom_sid), sid);
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_open_domain(char *desc, BOOL io, SAMR_Q_OPEN_DOMAIN *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_open_domain %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("rid", depth, buf, q, io, &(q_u->rid));

	smb_io_dom_sid2("sid", io, &(q_u->dom_sid), buf, q, depth); 
	buf_align(buf, q);
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_open_domain(char *desc, BOOL io, SAMR_R_OPEN_DOMAIN *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_open_domain %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(r_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_samr_q_unknown_3(SAMR_Q_UNKNOWN_3 *q_u,
				POLICY_HND *pol, uint32 switch_value)
{
	if (q_u == NULL) return;

	DEBUG(5,("samr_make_q_unknown_3\n"));

	memcpy(&q_u->pol, pol, sizeof(q_u->pol));
	q_u->switch_value = switch_value;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_3(char *desc, BOOL io, SAMR_Q_UNKNOWN_3 *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_unknown_3 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint16("switch_value", depth, buf, q, io, &(q_u->switch_value));
	buf_align(buf, q);
}

/*******************************************************************
 makes a SAMR_R_UNKNOWN3 structure.
 ********************************************************************/
void make_dom_sid3(DOM_SID3 *sid3, uint16 unk_0, uint16 unk_1, char *sid)
{
	if (sid3 == NULL) return;

	/* sid->len is _calculated_ in sam_io_dom_sid3 */
	sid3->unknown_0 = unk_0;
	sid3->unknown_1 = unk_1;
	make_dom_sid(&(sid3->sid), sid);
}

/*******************************************************************
reads or writes a SAM_SID3 structure.

this one's odd, because the length (in bytes) is specified at the beginning.
the length _includes_ the length of the length, too :-)

********************************************************************/
void sam_io_dom_sid3(char *desc, BOOL io, DOM_SID3 *sid3, struct mem_buffer *buf, int *q, int depth)
{
	int ptr_len = 0;
	int old_align = 0;

	if (sid3 == NULL) return;

	old_align = buf->align;
	buf->align = 0;

	DEBUG(5,("%s%06x sam_io_dom_sid3 %s\n", tab_depth(depth), *q, desc));
	depth++;

	if (io) 
	{
		/* reading.  do the length later */
		buf_uint16("len", depth, buf, q, io, &(sid3->len));
	}
	else
	{
		/* storing */
		*q += 2;
	}

	ptr_len = *q;

	buf_uint16("unknown_0", depth, buf, q, io, &(sid3->unknown_0));
	buf_uint16("unknown_1", depth, buf, q, io, &(sid3->unknown_1));
	smb_io_dom_sid("", io, &(sid3->sid), buf, q, depth); 

	if (!io) /* storing not reading.  do the length, now. */
	{
		uint16 len = 2 + (*q) - ptr_len;
		buf_uint16("len", depth, buf, &ptr_len, io, &len);
	}

	*q = ptr_len + sid3->len;

	buf->align = old_align;
}

/*******************************************************************
makes a SAMR_R_UNKNOWN3 structure.

unknown_2   : 0x0001
unknown_3   : 0x8004

unknown_4,5 : 0x0000 0014

unknown_6   : 0x0002
unknown_7   : 0x5800 or 0x0070

********************************************************************/
void make_sam_sid_stuff(SAM_SID_STUFF *stf,
				uint16 unknown_2, uint16 unknown_3,
				uint32 unknown_4, uint16 unknown_6, uint16 unknown_7,
				int num_sid3s, DOM_SID3 sid3[MAX_SAM_SIDS])
{
	stf->unknown_2 = unknown_2;
	stf->unknown_3 = unknown_3;

	bzero(stf->padding1, sizeof(stf->padding1));

	stf->unknown_4 = unknown_4;
	stf->unknown_5 = unknown_4;

	stf->unknown_6 = unknown_6;
	stf->unknown_7 = unknown_7;

	stf->num_sids  = num_sid3s;

	stf->padding2  = 0x0000;

	memcpy(stf->sid, sid3, sizeof(DOM_SID3) * num_sid3s);
}

/*******************************************************************
reads or writes a SAM_SID_STUFF structure.
********************************************************************/
void sam_io_sid_stuff(char *desc, BOOL io, SAM_SID_STUFF *stf, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (stf == NULL) return;

	DEBUG(5,("make_sam_sid_stuff\n"));

	buf_uint16("unknown_2", depth, buf, q, io, &(stf->unknown_2));
	buf_uint16("unknown_3", depth, buf, q, io, &(stf->unknown_3));

	buf_uint8s(False, "padding1", depth, buf, q, io, stf->padding1, sizeof(stf->padding1)); 
	
	buf_uint32("unknown_4", depth, buf, q, io, &(stf->unknown_4));
	buf_uint32("unknown_5", depth, buf, q, io, &(stf->unknown_5));
	buf_uint16("unknown_6", depth, buf, q, io, &(stf->unknown_6));
	buf_uint16("unknown_7", depth, buf, q, io, &(stf->unknown_7));

	buf_uint32("num_sids ", depth, buf, q, io, &(stf->num_sids ));
	buf_uint16("padding2 ", depth, buf, q, io, &(stf->padding2 ));

	for (i = 0; i < stf->num_sids; i++)
	{
		sam_io_dom_sid3("", io, &(stf->sid[i]), buf, q, depth); 
	}
}

/*******************************************************************
reads or writes a SAMR_R_UNKNOWN3 structure.
********************************************************************/
void make_samr_r_unknown_3(SAMR_R_UNKNOWN_3 *r_u,
				uint16 unknown_2, uint16 unknown_3,
				uint32 unknown_4, uint16 unknown_6, uint16 unknown_7,
				int num_sid3s, DOM_SID3 sid3[MAX_SAM_SIDS],
				uint32 status)
{
	if (r_u == NULL) return;

	DEBUG(5,("samr_make_r_unknown_3\n"));

	r_u->ptr_0 = 0;
	r_u->ptr_1 = 0;

	if (status == 0x0)
	{
		r_u->ptr_0 = 1;
		r_u->ptr_1 = 1;
		make_sam_sid_stuff(&(r_u->sid_stuff), unknown_2, unknown_3,
	               unknown_4, unknown_6, unknown_7,
	               num_sid3s, sid3);
	}

	r_u->status = status;
}


/*******************************************************************
reads or writes a SAMR_R_UNKNOWN_3 structure.

this one's odd, because the daft buggers use a different mechanism
for writing out the array of sids. they put the number of sids in
only one place: they've calculated the length of each sid and jumped
by that amount.  then, retrospectively, the length of the whole buffer
is put at the beginning of the data stream.

wierd.  

********************************************************************/
void samr_io_r_unknown_3(char *desc, BOOL io, SAMR_R_UNKNOWN_3 *r_u, struct mem_buffer *buf, int *q, int depth)
{
	int ptr_len0;
	int ptr_len1;
	int ptr_sid_stuff = 0;

	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_unknown_3 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_0         ", depth, buf, q, io, &(r_u->ptr_0         ));

	if (io) 
	{
		/* reading.  do the length later */
		buf_uint32("sid_stuff_len0", depth, buf, q, io, &(r_u->sid_stuff_len0));
	}
	else
	{
		/* storing */
		ptr_len0 = *q; *q += 4;
	}

	if (r_u->ptr_0 != 0)
	{
		buf_uint32("ptr_1         ", depth, buf, q, io, &(r_u->ptr_1         ));
		if (io)
		{
			/* reading.  do the length later */
			buf_uint32("sid_stuff_len1", depth, buf, q, io, &(r_u->sid_stuff_len1));
		}
		else
		{
			/* storing */
			ptr_len1 = *q; *q += 4;
		}

		if (r_u->ptr_1 != 0)
		{
			ptr_sid_stuff = *q;
			sam_io_sid_stuff("", io, &(r_u->sid_stuff), buf, q, depth); 
		}
	}

	if (!io) /* storing not reading.  do the length, now. */
	{
		if (ptr_sid_stuff != 0)
		{
			uint32 sid_stuff_len = *q - ptr_sid_stuff;
			int old_len = *q;

			*q = ptr_len0;
			buf_uint32("sid_stuff_len0", depth, buf, q, io, &sid_stuff_len); 

			*q = ptr_len1;
			buf_uint32("sid_stuff_len1", depth, buf, q, io, &sid_stuff_len);

			*q = old_len;
		}
	}

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_ENUM_DOM_USERS structure.
********************************************************************/
void make_samr_q_enum_dom_users(SAMR_Q_ENUM_DOM_USERS *q_e, POLICY_HND *pol,
				uint16 req_num_entries, uint16 unk_0,
				uint16 acb_mask, uint16 unk_1, uint32 size)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_enum_dom_users\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->req_num_entries = req_num_entries; /* zero indicates lots */
	q_e->unknown_0 = unk_0; /* this gets returned in the response */
	q_e->acb_mask  = acb_mask;
	q_e->unknown_1 = unk_1;
	q_e->max_size = size;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_enum_dom_users(char *desc, BOOL io, SAMR_Q_ENUM_DOM_USERS *q_e, struct mem_buffer *buf, int *q, int depth)
{
	if (q_e == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_enum_dom_users %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_e->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint16("req_num_entries", depth, buf, q, io, &(q_e->req_num_entries));
	buf_uint16("unknown_0      ", depth, buf, q, io, &(q_e->unknown_0      ));

	buf_uint16("acb_mask       ", depth, buf, q, io, &(q_e->acb_mask       ));
	buf_uint16("unknown_1      ", depth, buf, q, io, &(q_e->unknown_1      ));

	buf_uint32("max_size       ", depth, buf, q, io, &(q_e->max_size       ));

	buf_align(buf, q);
}


/*******************************************************************
makes a SAMR_R_ENUM_DOM_USERS structure.
********************************************************************/
void make_samr_r_enum_dom_users(SAMR_R_ENUM_DOM_USERS *r_u,
		uint16 total_num_entries, uint16 unk_0,
		uint32 num_sam_entries, struct smb_passwd pass[MAX_SAM_ENTRIES], uint32 status)
{
	int i;

	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_enum_dom_users\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	r_u->total_num_entries = total_num_entries;
	r_u->unknown_0         = unk_0;

	if (total_num_entries > 0)
	{
		r_u->ptr_entries1 = 1;
		r_u->ptr_entries2 = 1;
		r_u->num_entries2 = num_sam_entries;
		r_u->num_entries3 = num_sam_entries;

		for (i = 0; i < num_sam_entries; i++)
		{
			char *acct_name = pass[i].smb_name;
			int len_acct_name = strlen(acct_name);

			make_sam_entry(&(r_u->sam[i]), acct_name, pass[i].smb_userid);
			make_unistr2(&(r_u->uni_acct_name[i]), acct_name, len_acct_name);
		}

		r_u->num_entries4 = num_sam_entries;
	}
	else
	{
		r_u->ptr_entries1 = 0;
		r_u->num_entries2 = num_sam_entries;
		r_u->ptr_entries2 = 1;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_enum_dom_users(char *desc, BOOL io, SAMR_R_ENUM_DOM_USERS *r_u, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_enum_dom_users %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint16("total_num_entries", depth, buf, q, io, &(r_u->total_num_entries));
	buf_uint16("unknown_0        ", depth, buf, q, io, &(r_u->unknown_0        ));
	buf_uint32("ptr_entries1", depth, buf, q, io, &(r_u->ptr_entries1));

	if (r_u->total_num_entries != 0 && r_u->ptr_entries1 != 0)
	{
		buf_uint32("num_entries2", depth, buf, q, io, &(r_u->num_entries2));
		buf_uint32("ptr_entries2", depth, buf, q, io, &(r_u->ptr_entries2));
		buf_uint32("num_entries3", depth, buf, q, io, &(r_u->num_entries3));

		for (i = 0; i < r_u->num_entries2; i++)
		{
			buf_grow(buf, *q);
			smb_io_sam_entry("", io, &(r_u->sam[i]), buf, q, depth);
		}

		for (i = 0; i < r_u->num_entries2; i++)
		{
			buf_grow(buf, *q);
			smb_io_unistr2("", io, &(r_u->uni_acct_name[i]), r_u->sam[i].hdr_name.buffer, buf, q, depth);
		}

		buf_align(buf, q);

		buf_uint32("num_entries4", depth, buf, q, io, &(r_u->num_entries4));
	}

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_ENUM_DOM_ALIASES structure.
********************************************************************/
void make_samr_q_enum_dom_aliases(SAMR_Q_ENUM_DOM_ALIASES *q_e, POLICY_HND *pol, uint32 size)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_enum_dom_aliases\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->unknown_0 = 0;
	q_e->max_size = size;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_enum_dom_aliases(char *desc, BOOL io, SAMR_Q_ENUM_DOM_ALIASES *q_e, struct mem_buffer *buf, int *q, int depth)
{
	if (q_e == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_enum_dom_aliases %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_e->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("unknown_0", depth, buf, q, io, &(q_e->unknown_0));
	buf_uint32("max_size ", depth, buf, q, io, &(q_e->max_size ));

	buf_align(buf, q);
}


/*******************************************************************
makes a SAMR_R_ENUM_DOM_ALIASES structure.
********************************************************************/
void make_samr_r_enum_dom_aliases(SAMR_R_ENUM_DOM_ALIASES *r_u,
		uint32 num_sam_entries, struct smb_passwd grps[MAX_SAM_ENTRIES],
		uint32 status)
{
	int i;

	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_enum_dom_aliases\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	r_u->num_entries  = num_sam_entries;

	if (num_sam_entries > 0)
	{
		r_u->ptr_entries  = 1;
		r_u->num_entries2 = num_sam_entries;
		r_u->ptr_entries2 = 1;
		r_u->num_entries3 = num_sam_entries;

		for (i = 0; i < num_sam_entries; i++)
		{
			char *grp_name = grps[i].smb_name;
			int len_grp_name = strlen(grp_name);

			make_sam_entry(&(r_u->sam[i]), grp_name, grps[i].smb_userid);
			make_unistr2(&(r_u->uni_grp_name[i]), grp_name, len_grp_name);
		}

		r_u->num_entries4 = num_sam_entries;
	}
	else
	{
		r_u->ptr_entries = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_enum_dom_aliases(char *desc, BOOL io, SAMR_R_ENUM_DOM_ALIASES *r_u, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_enum_dom_aliases %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries", depth, buf, q, io, &(r_u->num_entries));
	buf_uint32("ptr_entries", depth, buf, q, io, &(r_u->ptr_entries));

	if (r_u->num_entries != 0 && r_u->ptr_entries != 0)
	{
		buf_uint32("num_entries2", depth, buf, q, io, &(r_u->num_entries2));
		buf_uint32("ptr_entries2", depth, buf, q, io, &(r_u->ptr_entries2));
		buf_uint32("num_entries3", depth, buf, q, io, &(r_u->num_entries3));

		for (i = 0; i < r_u->num_entries; i++)
		{
			smb_io_sam_entry("", io, &(r_u->sam[i]), buf, q, depth);
		}

		for (i = 0; i < r_u->num_entries; i++)
		{
			smb_io_unistr2("", io, &(r_u->uni_grp_name[i]), r_u->sam[i].hdr_name.buffer, buf, q, depth);
		}

		buf_align(buf, q);

		buf_uint32("num_entries4", depth, buf, q, io, &(r_u->num_entries4));
	}

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_QUERY_DISPINFO structure.
********************************************************************/
void make_samr_q_query_dispinfo(SAMR_Q_QUERY_DISPINFO *q_e, POLICY_HND *pol,
				uint16 switch_level, uint32 start_idx, uint32 size)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_query_dispinfo\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->switch_level = switch_level;

	q_e->unknown_0 = 0;
	q_e->start_idx = start_idx;
	q_e->unknown_1 = 0x000007d0;
	q_e->max_size  = size;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_query_dispinfo(char *desc, BOOL io, SAMR_Q_QUERY_DISPINFO *q_e, struct mem_buffer *buf, int *q, int depth)
{
	if (q_e == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_query_dispinfo %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_e->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint16("switch_level", depth, buf, q, io, &(q_e->switch_level));
	buf_uint16("unknown_0   ", depth, buf, q, io, &(q_e->unknown_0   ));
	buf_uint32("start_idx   ", depth, buf, q, io, &(q_e->start_idx   ));
	buf_uint32("unknown_1   ", depth, buf, q, io, &(q_e->unknown_1   ));
	buf_uint32("max_size    ", depth, buf, q, io, &(q_e->max_size    ));

	buf_align(buf, q);
}


/*******************************************************************
makes a SAM_INFO_2 structure.
********************************************************************/
void make_sam_info_2(SAM_INFO_2 *sam, uint32 acb_mask,
		uint32 start_idx, uint32 num_sam_entries,
		struct smb_passwd pass[MAX_SAM_ENTRIES])
{
	int i;
	int entries_added;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_info_2\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	for (i = start_idx, entries_added = 0; i < num_sam_entries; i++)
	{
		if (IS_BITS_SET_ALL(pass[i].acct_ctrl, acb_mask))
		{
			char *acct_name = pass[i].smb_name;
			char *acct_desc = NULL;

			make_sam_entry2(&(sam->sam[entries_added]), start_idx + entries_added + 1,
			                &pass[i], acct_desc);
			make_sam_str2  (&(sam->str[entries_added]), acct_name, acct_desc);

			entries_added++;
		}
	}

	sam->num_entries   = entries_added;
	sam->ptr_entries   = 1;
	sam->num_entries2  = entries_added;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void sam_io_sam_info_2(char *desc, BOOL io, SAM_INFO_2 *sam, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (sam == NULL) return;

	DEBUG(5,("%s%06x sam_io_sam_info_2%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries  ", depth, buf, q, io, &(sam->num_entries  ));
	buf_uint32("ptr_entries  ", depth, buf, q, io, &(sam->ptr_entries  ));

	buf_uint32("num_entries2 ", depth, buf, q, io, &(sam->num_entries2 ));

	for (i = 0; i < sam->num_entries; i++)
	{
		smb_io_sam_entry2("", io, &(sam->sam[i]), buf, q, depth);
	}

	for (i = 0; i < sam->num_entries; i++)
	{
		smb_io_sam_str2 ("", io, &(sam->str[i]),
							 sam->sam[i].hdr_srv_name.buffer,
							 sam->sam[i].hdr_srv_desc.buffer,
							 buf, q, depth);
	}
}


/*******************************************************************
makes a SAM_INFO_1 structure.
********************************************************************/
void make_sam_info_1(SAM_INFO_1 *sam, uint32 acb_mask,
		uint32 start_idx, uint32 num_sam_entries,
		struct smb_passwd pass[MAX_SAM_ENTRIES])
{
	int i;
	int entries_added;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_info_1\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	for (i = start_idx, entries_added = 0; i < num_sam_entries; i++)
	{
		if (IS_BITS_SET_ALL(pass[i].acct_ctrl, acb_mask))
		{
			char *acct_name = pass[i].smb_name;
			char *full_name = "<full name>";
			char *acct_desc = "<account description>";

			make_sam_entry1(&(sam->sam[entries_added]), start_idx + entries_added + 1,
			                &pass[i], full_name, acct_desc);
			make_sam_str1  (&(sam->str[entries_added]), acct_name, full_name, acct_desc);

			entries_added++;
		}
	}

	sam->num_entries   = entries_added;
	sam->ptr_entries   = 1;
	sam->num_entries2  = entries_added;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void sam_io_sam_info_1(char *desc, BOOL io, SAM_INFO_1 *sam, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (sam == NULL) return;

	DEBUG(5,("%s%06x sam_io_sam_info_1%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries  ", depth, buf, q, io, &(sam->num_entries  ));
	buf_uint32("ptr_entries  ", depth, buf, q, io, &(sam->ptr_entries  ));

	buf_uint32("num_entries2 ", depth, buf, q, io, &(sam->num_entries2 ));

	for (i = 0; i < sam->num_entries; i++)
	{
		smb_io_sam_entry1("", io, &(sam->sam[i]), buf, q, depth);
	}

	for (i = 0; i < sam->num_entries; i++)
	{
		smb_io_sam_str1 ("", io, &(sam->str[i]),
							 sam->sam[i].hdr_acct_name.buffer,
							 sam->sam[i].hdr_user_name.buffer,
							 sam->sam[i].hdr_user_desc.buffer,
							 buf, q, depth);
	}
}


/*******************************************************************
makes a SAMR_R_QUERY_DISPINFO structure.
********************************************************************/
void make_samr_r_query_dispinfo(SAMR_R_QUERY_DISPINFO *r_u,
		uint16 switch_level, SAM_INFO_CTR *ctr, uint32 status)
{
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_query_dispinfo\n"));

	if (status == 0x0)
	{
		r_u->unknown_0 = 0x0000001;
		r_u->unknown_1 = 0x0000001;
	}
	else
	{
		r_u->unknown_0 = 0x0;
		r_u->unknown_1 = 0x0;
	}

	r_u->switch_level = switch_level;
	r_u->ctr = ctr;
	r_u->status = status;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_query_dispinfo(char *desc, BOOL io, SAMR_R_QUERY_DISPINFO *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_query_dispinfo %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("unknown_0    ", depth, buf, q, io, &(r_u->unknown_0    ));
	buf_uint32("unknown_1    ", depth, buf, q, io, &(r_u->unknown_1    ));
	buf_uint16("switch_level ", depth, buf, q, io, &(r_u->switch_level ));

	buf_align(buf, q);

	switch (r_u->switch_level)
	{
		case 0x1:
		{
			sam_io_sam_info_1("users", io, r_u->ctr->sam.info1, buf, q, depth);
			break;
		}
		case 0x2:
		{
			sam_io_sam_info_2("servers", io, r_u->ctr->sam.info2, buf, q, depth);
			break;
		}
		default:
		{
			DEBUG(5,("samr_io_r_query_dispinfo: unknown switch value\n"));
			break;
		}
	}

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_ENUM_DOM_GROUPS structure.
********************************************************************/
void make_samr_q_enum_dom_groups(SAMR_Q_ENUM_DOM_GROUPS *q_e, POLICY_HND *pol,
				uint16 switch_level, uint32 start_idx, uint32 size)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_enum_dom_groups\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->switch_level = switch_level;

	q_e->unknown_0 = 0;
	q_e->start_idx = start_idx;
	q_e->unknown_1 = 0x000007d0;
	q_e->max_size  = size;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_enum_dom_groups(char *desc, BOOL io, SAMR_Q_ENUM_DOM_GROUPS *q_e, struct mem_buffer *buf, int *q, int depth)
{
	if (q_e == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_enum_dom_groups %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_e->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint16("switch_level", depth, buf, q, io, &(q_e->switch_level));
	buf_uint16("unknown_0   ", depth, buf, q, io, &(q_e->unknown_0   ));
	buf_uint32("start_idx   ", depth, buf, q, io, &(q_e->start_idx   ));
	buf_uint32("unknown_1   ", depth, buf, q, io, &(q_e->unknown_1   ));
	buf_uint32("max_size    ", depth, buf, q, io, &(q_e->max_size    ));

	buf_align(buf, q);
}


/*******************************************************************
makes a SAMR_R_ENUM_DOM_GROUPS structure.
********************************************************************/
void make_samr_r_enum_dom_groups(SAMR_R_ENUM_DOM_GROUPS *r_u,
		uint32 start_idx, uint32 num_sam_entries,
		struct smb_passwd pass[MAX_SAM_ENTRIES],
		uint32 status)
{
	int i;
	int entries_added;

	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_enum_dom_groups\n"));

	if (num_sam_entries >= MAX_SAM_ENTRIES)
	{
		num_sam_entries = MAX_SAM_ENTRIES;
		DEBUG(5,("limiting number of entries to %d %s\n", num_sam_entries));
	}

	if (status == 0x0)
	{
		for (i = start_idx, entries_added = 0; i < num_sam_entries; i++)
		{
			char *grp_name = pass[i].smb_name;
			char *grp_desc = "<account description>";

			entries_added++;

			make_sam_entry3(&(r_u->sam[i]), start_idx + entries_added,
			                &pass[i], grp_desc);
			make_sam_str3  (&(r_u->str[i]), grp_name, grp_desc);
		}

		if (entries_added > 0)
		{
			r_u->unknown_0 = 0x0000492;
			r_u->unknown_1 = 0x000049a;
		}
		else
		{
			r_u->unknown_0 = 0x0;
			r_u->unknown_1 = 0x0;
		}
		r_u->switch_level  = 3;
		r_u->num_entries   = entries_added;
		r_u->ptr_entries   = 1;
		r_u->num_entries2  = entries_added;
	}
	else
	{
		r_u->switch_level = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_enum_dom_groups(char *desc, BOOL io, SAMR_R_ENUM_DOM_GROUPS *r_u, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_enum_dom_groups %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("unknown_0    ", depth, buf, q, io, &(r_u->unknown_0    ));
	buf_uint32("unknown_1    ", depth, buf, q, io, &(r_u->unknown_1    ));
	buf_uint32("switch_level ", depth, buf, q, io, &(r_u->switch_level ));

	if (r_u->switch_level != 0)
	{
		buf_uint32("num_entries  ", depth, buf, q, io, &(r_u->num_entries  ));
		buf_uint32("ptr_entries  ", depth, buf, q, io, &(r_u->ptr_entries  ));

		buf_uint32("num_entries2 ", depth, buf, q, io, &(r_u->num_entries2 ));

		for (i = 0; i < r_u->num_entries; i++)
		{
			smb_io_sam_entry3("", io, &(r_u->sam[i]), buf, q, depth);
		}

		for (i = 0; i < r_u->num_entries; i++)
		{
			smb_io_sam_str3 ("", io, &(r_u->str[i]),
			                     r_u->sam[i].hdr_grp_name.buffer,
			                     r_u->sam[i].hdr_grp_desc.buffer,
			                     buf, q, depth);
		}
	}

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_QUERY_ALIASINFO structure.
********************************************************************/
void make_samr_q_query_aliasinfo(SAMR_Q_QUERY_ALIASINFO *q_e,
				POLICY_HND *pol,
				uint16 switch_level)
{
	if (q_e == NULL || pol == NULL) return;

	DEBUG(5,("make_q_query_aliasinfo\n"));

	memcpy(&(q_e->pol), pol, sizeof(*pol));

	q_e->switch_level = switch_level;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_query_aliasinfo(char *desc, BOOL io, SAMR_Q_QUERY_ALIASINFO *q_e, struct mem_buffer *buf, int *q, int depth)
{
	if (q_e == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_query_aliasinfo %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_e->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint16("switch_level", depth, buf, q, io, &(q_e->switch_level));
}


/*******************************************************************
makes a SAMR_R_QUERY_ALIASINFO structure.
********************************************************************/
void make_samr_r_query_aliasinfo(SAMR_R_QUERY_ALIASINFO *r_u,
		uint16 switch_value, char *acct_desc,
		uint32 status)
{
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_query_aliasinfo\n"));

	r_u->ptr = 0;

	if (status == 0)
	{
		r_u->switch_value = switch_value;

		switch (switch_value)
		{
			case 3:
			{
				int acct_len = acct_desc ? strlen(acct_desc) : 0;

				r_u->ptr = 1;

				make_uni_hdr(&(r_u->alias.info3.hdr_acct_desc), acct_len , acct_len, acct_desc ? 1 : 0);
				make_unistr2(&(r_u->alias.info3.uni_acct_desc), acct_desc, acct_len);

				break;
			}

			default:
			{
				DEBUG(4,("make_samr_r_query_aliasinfo: unsupported switch level\n"));
				break;
			}
		}
	}

	r_u->status = status;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_query_aliasinfo(char *desc, BOOL io, SAMR_R_QUERY_ALIASINFO *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_query_aliasinfo %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr         ", depth, buf, q, io, &(r_u->ptr         ));
	
	if (r_u->ptr != 0)
	{
		buf_uint16("switch_value", depth, buf, q, io, &(r_u->switch_value));
		buf_align(buf, q);

		if (r_u->switch_value != 0)
		{
			switch (r_u->switch_value)
			{
				case 3:
				{
					smb_io_unihdr ("", io, &(r_u->alias.info3.hdr_acct_desc), buf, q, depth);
					smb_io_unistr2("", io, &(r_u->alias.info3.uni_acct_desc), r_u->alias.info3.hdr_acct_desc.buffer, buf, q, depth);
					break;
				}
				default:
				{
					DEBUG(4,("samr_io_r_query_aliasinfo: unsupported switch level\n"));
					break;
				}
			}
		}
	}

	buf_align(buf, q);

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

/*******************************************************************
reads or writes a SAMR_Q_LOOKUP_IDS structure.
********************************************************************/
void samr_io_q_lookup_ids(char *desc, BOOL io, SAMR_Q_LOOKUP_IDS *q_u, struct mem_buffer *buf, int *q, int depth)
{
	fstring tmp;
	int i;

	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_lookup_ids %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("num_sids1", depth, buf, q, io, &(q_u->num_sids1));
	buf_uint32("ptr      ", depth, buf, q, io, &(q_u->ptr      ));
	buf_uint32("num_sids2", depth, buf, q, io, &(q_u->num_sids2));

	for (i = 0; i < q_u->num_sids2; i++)
	{
		sprintf(tmp, "ptr[%02d]", i);
		buf_uint32(tmp, depth, buf, q, io, &(q_u->ptr_sid[i]));
	}

	for (i = 0; i < q_u->num_sids2; i++)
	{
		if (q_u->ptr_sid[i] != 0)
		{
			sprintf(tmp, "sid[%02d]", i);
			smb_io_dom_sid2(tmp, io, &(q_u->sid[i]), buf, q, depth); 
		}
	}

	buf_align(buf, q);
}


/*******************************************************************
makes a SAMR_R_LOOKUP_IDS structure.
********************************************************************/
void make_samr_r_lookup_ids(SAMR_R_LOOKUP_IDS *r_u,
		uint32 num_rids, uint32 *rid, uint32 status)
{
	int i;
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_lookup_ids\n"));

	if (status == 0x0)
	{
		r_u->num_entries  = num_rids;
		r_u->ptr = 1;
		r_u->num_entries2 = num_rids;

		for (i = 0; i < num_rids; i++)
		{
			r_u->rid[i] = rid[i];
		}
	}
	else
	{
		r_u->num_entries  = 0;
		r_u->ptr = 0;
		r_u->num_entries2 = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_lookup_ids(char *desc, BOOL io, SAMR_R_LOOKUP_IDS *r_u, struct mem_buffer *buf, int *q, int depth)
{
	fstring tmp;
	int i;
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_lookup_ids %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries", depth, buf, q, io, &(r_u->num_entries));
	buf_uint32("ptr        ", depth, buf, q, io, &(r_u->ptr        ));
	buf_uint32("num_entries2", depth, buf, q, io, &(r_u->num_entries2));

	if (r_u->num_entries != 0)
	{

		for (i = 0; i < r_u->num_entries2; i++)
		{
			sprintf(tmp, "rid[%02d]", i);
			buf_uint32(tmp, depth, buf, q, io, &(r_u->rid[i]));
		}
	}

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_lookup_names(char *desc, BOOL io, SAMR_Q_LOOKUP_NAMES *q_u, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_lookup_names %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("num_rids1", depth, buf, q, io, &(q_u->num_rids1));
	buf_uint32("rid      ", depth, buf, q, io, &(q_u->rid      ));
	buf_uint32("ptr      ", depth, buf, q, io, &(q_u->ptr      ));
	buf_uint32("num_rids2", depth, buf, q, io, &(q_u->num_rids2));

	for (i = 0; i < q_u->num_rids2; i++)
	{
		smb_io_unihdr ("", io, &(q_u->hdr_user_name[i]), buf, q, depth); 
	}
	for (i = 0; i < q_u->num_rids2; i++)
	{
		smb_io_unistr2("", io, &(q_u->uni_user_name[i]), q_u->hdr_user_name[i].buffer, buf, q, depth); 
	}

	buf_align(buf, q);
}


/*******************************************************************
makes a SAMR_R_LOOKUP_NAMES structure.
********************************************************************/
void make_samr_r_lookup_names(SAMR_R_LOOKUP_NAMES *r_u,
		uint32 num_rids, uint32 *rid, uint32 status)
{
	int i;
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_lookup_names\n"));

	if (status == 0x0)
	{
		r_u->num_entries  = num_rids;
		r_u->undoc_buffer = 1;
		r_u->num_entries2 = num_rids;

		for (i = 0; i < num_rids; i++)
		{
			make_dom_rid3(&(r_u->dom_rid[i]), rid[i]);
		}

		r_u->num_entries3 = num_rids;
	}
	else
	{
		r_u->num_entries  = 0;
		r_u->undoc_buffer = 0;
		r_u->num_entries2 = 0;
		r_u->num_entries3 = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_lookup_names(char *desc, BOOL io, SAMR_R_LOOKUP_NAMES *r_u, struct mem_buffer *buf, int *q, int depth)
{
	int i;
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_lookup_names %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries ", depth, buf, q, io, &(r_u->num_entries ));
	buf_uint32("undoc_buffer", depth, buf, q, io, &(r_u->undoc_buffer));
	buf_uint32("num_entries2", depth, buf, q, io, &(r_u->num_entries2));

	if (r_u->num_entries != 0)
	{
		for (i = 0; i < r_u->num_entries2; i++)
		{
			smb_io_dom_rid3("", io, &(r_u->dom_rid[i]), buf, q, depth);
		}

	}

	buf_uint32("num_entries3", depth, buf, q, io, &(r_u->num_entries3));

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_UNKNOWN_12 structure.
********************************************************************/
void make_samr_q_unknown_12(SAMR_Q_UNKNOWN_12 *q_u,
		POLICY_HND *pol, uint32 rid,
		uint32 num_gids, uint32 *gid)
{
	int i;
	if (q_u == NULL) return;

	DEBUG(5,("make_samr_r_unknwon_12\n"));

	memcpy(&(q_u->pol), pol, sizeof(*pol));

	q_u->num_gids1 = num_gids;
	q_u->rid       = rid;
	q_u->ptr       = 0;
	q_u->num_gids2 = num_gids;

	for (i = 0; i < num_gids; i++)
	{
		q_u->gid[i] = gid[i];
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_12(char *desc, BOOL io, SAMR_Q_UNKNOWN_12 *q_u, struct mem_buffer *buf, int *q, int depth)
{
	int i;
	fstring tmp;

	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_unknown_12 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("num_gids1", depth, buf, q, io, &(q_u->num_gids1));
	buf_uint32("rid      ", depth, buf, q, io, &(q_u->rid      ));
	buf_uint32("ptr      ", depth, buf, q, io, &(q_u->ptr      ));
	buf_uint32("num_gids2", depth, buf, q, io, &(q_u->num_gids2));

	for (i = 0; i < q_u->num_gids2; i++)
	{
		sprintf(tmp, "gid[%02d]  ", i);
		buf_uint32(tmp, depth, buf, q, io, &(q_u->gid[i]));
	}

	buf_align(buf, q);
}


/*******************************************************************
makes a SAMR_R_UNKNOWN_12 structure.
********************************************************************/
void make_samr_r_unknown_12(SAMR_R_UNKNOWN_12 *r_u,
		uint32 num_aliases, fstring *als_name, uint32 *num_als_usrs,
		uint32 status)
{
	int i;
	if (r_u == NULL || als_name == NULL || num_als_usrs == NULL) return;

	DEBUG(5,("make_samr_r_unknown_12\n"));

	if (status == 0x0)
	{
		r_u->num_aliases1 = num_aliases;
		r_u->ptr_aliases  = 1;
		r_u->num_aliases2 = num_aliases;

		r_u->num_als_usrs1 = num_aliases;
		r_u->ptr_als_usrs  = 1;
		r_u->num_als_usrs2 = num_aliases;

		for (i = 0; i < num_aliases; i++)
		{
			int als_len = als_name[i] != NULL ? strlen(als_name[i]) : 0;
			make_uni_hdr(&(r_u->hdr_als_name[i]), als_len    , als_len, als_name[i] ? 1 : 0);
			make_unistr2(&(r_u->uni_als_name[i]), als_name[i], als_len);
			r_u->num_als_usrs[i] = num_als_usrs[i];
		}
	}
	else
	{
		r_u->num_aliases1 = num_aliases;
		r_u->ptr_aliases  = 0;
		r_u->num_aliases2 = num_aliases;

		r_u->num_als_usrs1 = num_aliases;
		r_u->ptr_als_usrs  = 0;
		r_u->num_als_usrs2 = num_aliases;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_unknown_12(char *desc, BOOL io, SAMR_R_UNKNOWN_12 *r_u, struct mem_buffer *buf, int *q, int depth)
{
	int i;
	fstring tmp;
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_unknown_12 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_aliases1", depth, buf, q, io, &(r_u->num_aliases1));
	buf_uint32("ptr_aliases ", depth, buf, q, io, &(r_u->ptr_aliases ));
	buf_uint32("num_aliases2", depth, buf, q, io, &(r_u->num_aliases2));

	if (r_u->ptr_aliases != 0 && r_u->num_aliases1 != 0)
	{
		for (i = 0; i < r_u->num_aliases2; i++)
		{
			sprintf(tmp, "als_hdr[%02d]  ", i);
			smb_io_unihdr ("", io, &(r_u->hdr_als_name[i]), buf, q, depth); 
		}
		for (i = 0; i < r_u->num_aliases2; i++)
		{
			sprintf(tmp, "als_str[%02d]  ", i);
			smb_io_unistr2("", io, &(r_u->uni_als_name[i]), r_u->hdr_als_name[i].buffer, buf, q, depth); 
		}
	}

	buf_align(buf, q);

	buf_uint32("num_als_usrs1", depth, buf, q, io, &(r_u->num_als_usrs1));
	buf_uint32("ptr_als_usrs ", depth, buf, q, io, &(r_u->ptr_als_usrs ));
	buf_uint32("num_als_usrs2", depth, buf, q, io, &(r_u->num_als_usrs2));

	if (r_u->ptr_als_usrs != 0 && r_u->num_als_usrs1 != 0)
	{
		for (i = 0; i < r_u->num_als_usrs2; i++)
		{
			sprintf(tmp, "als_usrs[%02d]  ", i);
			buf_uint32(tmp, depth, buf, q, io, &(r_u->num_als_usrs[i]));
		}
	}

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_samr_q_open_user(SAMR_Q_OPEN_USER *q_u,
				POLICY_HND *pol,
				uint32 unk_0, uint32 rid)
{
	if (q_u == NULL) return;

	DEBUG(5,("samr_make_q_open_user\n"));

	memcpy(&q_u->domain_pol, pol, sizeof(q_u->domain_pol));
	
	q_u->unknown_0 = unk_0;
	q_u->user_rid  = rid;
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_open_user(char *desc, BOOL io, SAMR_Q_OPEN_USER *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_open_user %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("domain_pol", io, &(q_u->domain_pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("unknown_0", depth, buf, q, io, &(q_u->unknown_0));
	buf_uint32("user_rid ", depth, buf, q, io, &(q_u->user_rid ));

	buf_align(buf, q);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_open_user(char *desc, BOOL io, SAMR_R_OPEN_USER *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_open_user %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("user_pol", io, &(r_u->user_pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_QUERY_USERGROUPS structure.
********************************************************************/
void make_samr_q_query_usergroups(SAMR_Q_QUERY_USERGROUPS *q_u,
				POLICY_HND *hnd)
{
	if (q_u == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_query_usergroups\n"));

	memcpy(&(q_u->pol), hnd, sizeof(q_u->pol));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_query_usergroups(char *desc, BOOL io, SAMR_Q_QUERY_USERGROUPS *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_query_usergroups %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);
}

/*******************************************************************
makes a SAMR_R_QUERY_USERGROUPS structure.
********************************************************************/
void make_samr_r_query_usergroups(SAMR_R_QUERY_USERGROUPS *r_u,
		uint32 num_gids, DOM_GID *gid, uint32 status)
{
	if (r_u == NULL) return;

	DEBUG(5,("make_samr_r_query_usergroups\n"));

	if (status == 0x0)
	{
		r_u->ptr_0        = 1;
		r_u->num_entries  = num_gids;
		r_u->ptr_1        = 1;
		r_u->num_entries2 = num_gids;

		r_u->gid = gid;

#if 0
		int i;
		for (i = 0; i < num_gids && i < LSA_MAX_GROUPS; i++)
		{
			r_u->gid[i].g_rid = gid[i].g_rid;
			r_u->gid[i].attr  = gid[i].attr ;
		}
#endif

	}
	else
	{
		r_u->ptr_0       = 0;
		r_u->num_entries = 0;
		r_u->ptr_1       = 0;
	}

	r_u->status = status;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_query_usergroups(char *desc, BOOL io, SAMR_R_QUERY_USERGROUPS *r_u, struct mem_buffer *buf, int *q, int depth)
{
	int i;
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_query_usergroups %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_0       ", depth, buf, q, io, &(r_u->ptr_0      ));

	if (r_u->ptr_0 != 0)
	{
		buf_uint32("num_entries ", depth, buf, q, io, &(r_u->num_entries));
		buf_uint32("ptr_1       ", depth, buf, q, io, &(r_u->ptr_1      ));

		if (r_u->num_entries != 0)
		{
			buf_uint32("num_entries2", depth, buf, q, io, &(r_u->num_entries2));

			for (i = 0; i < r_u->num_entries2; i++)
			{
				smb_io_gid("", io, &(r_u->gid[i]), buf, q, depth);
			}
		}
	}
	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_QUERY_USERINFO structure.
********************************************************************/
void make_samr_q_query_userinfo(SAMR_Q_QUERY_USERINFO *q_u,
				POLICY_HND *hnd, uint16 switch_value)
{
	if (q_u == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_query_userinfo\n"));

	memcpy(&(q_u->pol), hnd, sizeof(q_u->pol));
	q_u->switch_value = switch_value;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_query_userinfo(char *desc, BOOL io, SAMR_Q_QUERY_USERINFO *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_query_userinfo %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint16("switch_value", depth, buf, q, io, &(q_u->switch_value)); /* 0x0015 or 0x0011 */

	buf_align(buf, q);
}

/*******************************************************************
makes a SAM_USER_INFO_10 structure.
********************************************************************/
void make_sam_user_info11(SAM_USER_INFO_11 *usr,
				NTTIME *expiry,
				char *mach_acct,
				uint32 rid_user,
				uint32 rid_group,
				uint16 acct_ctrl)
				
{
	int len_mach_acct;
	if (usr == NULL || expiry == NULL || mach_acct == NULL) return;

	DEBUG(5,("make_samr_r_unknown_24\n"));

	len_mach_acct = strlen(mach_acct);

	memcpy(&(usr->expiry),expiry, sizeof(usr->expiry)); /* expiry time or something? */
	bzero(usr->padding_1, sizeof(usr->padding_1)); /* 0 - padding 24 bytes */

	make_uni_hdr(&(usr->hdr_mach_acct), len_mach_acct, len_mach_acct, 4);  /* unicode header for machine account */
	usr->padding_2 = 0;               /* 0 - padding 4 bytes */

	usr->ptr_1        = 1;            /* pointer */
	bzero(usr->padding_3, sizeof(usr->padding_3)); /* 0 - padding 32 bytes */
	usr->padding_4    = 0;            /* 0 - padding 4 bytes */

	usr->ptr_2        = 1;            /* pointer */
	usr->padding_5    = 0;            /* 0 - padding 4 bytes */

	usr->ptr_3        = 1;          /* pointer */
	bzero(usr->padding_6, sizeof(usr->padding_6)); /* 0 - padding 32 bytes */

	usr->rid_user     = rid_user; 
	usr->rid_group    = rid_group;

	usr->acct_ctrl    = acct_ctrl;
	usr->unknown_3    = 0x0000;

	usr->unknown_4    = 0x003f;       /* 0x003f      - 16 bit unknown */
	usr->unknown_5    = 0x003c;       /* 0x003c      - 16 bit unknown */

	bzero(usr->padding_7, sizeof(usr->padding_7)); /* 0 - padding 16 bytes */
	usr->padding_8    = 0;            /* 0 - padding 4 bytes */
	
	make_unistr2(&(usr->uni_mach_acct), mach_acct, len_mach_acct);  /* unicode string for machine account */

	bzero(usr->padding_9, sizeof(usr->padding_9)); /* 0 - padding 48 bytes */
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void sam_io_user_info11(char *desc, BOOL io, SAM_USER_INFO_11 *usr, struct mem_buffer *buf, int *q, int depth)
{
	if (usr == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_unknown_24 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint8s (False, "padding_0", depth, buf, q, io, usr->padding_0, sizeof(usr->padding_0)); 

	smb_io_time("time", io, &(usr->expiry), buf, q, depth); 

	buf_uint8s (False, "padding_1", depth, buf, q, io, usr->padding_1, sizeof(usr->padding_1));

	smb_io_unihdr ("unihdr", io, &(usr->hdr_mach_acct), buf, q, depth); 
	buf_uint32(        "padding_2", depth, buf, q, io, &(usr->padding_2));

	buf_uint32(        "ptr_1    ", depth, buf, q, io, &(usr->ptr_1    ));
	buf_uint8s (False, "padding_3", depth, buf, q, io, usr->padding_3, sizeof(usr->padding_3));
	buf_uint32(        "padding_4", depth, buf, q, io, &(usr->padding_4));

	buf_uint32(        "ptr_2    ", depth, buf, q, io, &(usr->ptr_2    ));
	buf_uint32(        "padding_5", depth, buf, q, io, &(usr->padding_5));

	buf_uint32(        "ptr_3    ", depth, buf, q, io, &(usr->ptr_3    ));
	buf_uint8s (False, "padding_6", depth, buf, q, io, usr->padding_6, sizeof(usr->padding_6));

	buf_uint32(        "rid_user ", depth, buf, q, io, &(usr->rid_user ));
	buf_uint32(        "rid_group", depth, buf, q, io, &(usr->rid_group));
	buf_uint16(        "acct_ctrl", depth, buf, q, io, &(usr->acct_ctrl));
	buf_uint16(        "unknown_3", depth, buf, q, io, &(usr->unknown_3));
	buf_uint16(        "unknown_4", depth, buf, q, io, &(usr->unknown_4));
	buf_uint16(        "unknown_5", depth, buf, q, io, &(usr->unknown_5));

	buf_uint8s (False, "padding_7", depth, buf, q, io, usr->padding_7, sizeof(usr->padding_7));
	buf_uint32(        "padding_8", depth, buf, q, io, &(usr->padding_8));
	
	smb_io_unistr2("unistr2", io, &(usr->uni_mach_acct), True, buf, q, depth); 
	buf_align(buf, q);

	buf_uint8s (False, "padding_9", depth, buf, q, io, usr->padding_9, sizeof(usr->padding_9));
}

/*************************************************************************
 make_sam_user_info15

 unknown_3 = 0x00ff ffff
 unknown_5 = 0x0002 0000
 unknown_6 = 0x0000 04ec 

 *************************************************************************/
void make_sam_user_info15(SAM_USER_INFO_15 *usr,

	NTTIME *logon_time,
	NTTIME *logoff_time,
	NTTIME *kickoff_time,
	NTTIME *pass_last_set_time,
	NTTIME *pass_can_change_time,
	NTTIME *pass_must_change_time,

	char *user_name,
	char *full_name,
	char *home_dir,
	char *dir_drive,
	char *profile_path,
	char *logon_script,
	char *description,

	uint16 logon_count,
	uint16 bad_pw_count,

	uint32 unknown_0,
	uint32 unknown_1,

	uint32 user_rid,
	uint32 group_rid,
	uint16 acb_info, 

	uint32 unknown_3,
	uint16 logon_divs,
	LOGON_HRS *hrs,
	uint32 unknown_5,
	uint32 unknown_6)
{
	int len_user_name    = user_name    != NULL ? strlen(user_name   ) : 0;
	int len_full_name    = full_name    != NULL ? strlen(full_name   ) : 0;
	int len_dir_drive    = dir_drive    != NULL ? strlen(dir_drive   ) : 0;
	int len_home_dir     = home_dir     != NULL ? strlen(home_dir    ) : 0;
	int len_profile_path = profile_path != NULL ? strlen(profile_path) : 0;
	int len_logon_script = logon_script != NULL ? strlen(logon_script) : 0;
	int len_description  = description  != NULL ? strlen(description ) : 0;

	usr->logon_time            = *logon_time;
	usr->logoff_time           = *logoff_time;
	usr->kickoff_time          = *kickoff_time;
	usr->pass_last_set_time    = *pass_last_set_time;
	usr->pass_can_change_time  = *pass_can_change_time;
	usr->pass_must_change_time = *pass_must_change_time;

	make_uni_hdr(&(usr->hdr_user_name   ), len_user_name   , len_user_name   , len_user_name    != 0);
	make_uni_hdr(&(usr->hdr_full_name   ), len_full_name   , len_full_name   , len_full_name    != 0);
	make_uni_hdr(&(usr->hdr_dir_drive   ), len_dir_drive   , len_dir_drive   , len_dir_drive    != 0);
	make_uni_hdr(&(usr->hdr_home_dir    ), len_home_dir    , len_home_dir    , len_home_dir     != 0);
	make_uni_hdr(&(usr->hdr_profile_path), len_profile_path, len_profile_path, len_profile_path != 0);
	make_uni_hdr(&(usr->hdr_logon_script), len_logon_script, len_logon_script, len_logon_script != 0);
	make_uni_hdr(&(usr->hdr_description ), len_description , len_description , len_description  != 0);

	usr->logon_count = logon_count;
	usr->bad_pw_count = bad_pw_count;

	usr->ptr_padding2 = 1;
	usr->unknown_0 = unknown_0;

	usr->ptr_padding3 = 1;
	usr->unknown_1 = unknown_1;

	usr->ptr_unknown6 = 1;
	bzero(usr->unknown_2, sizeof(usr->unknown_2));

	usr->user_rid  = user_rid;
	usr->group_rid = group_rid;
	usr->acb_info = acb_info;
	usr->unknown_3 = unknown_3;

	usr->logon_divs = logon_divs;
	usr->ptr_logon_hrs = hrs ? 1 : 0;
	usr->unknown_5 = unknown_5;

	bzero(usr->padding1, sizeof(usr->padding1));

	make_unistr2(&(usr->uni_user_name   ), user_name   , len_user_name   );
	make_unistr2(&(usr->uni_full_name   ), full_name   , len_full_name   );
	make_unistr2(&(usr->uni_dir_drive   ), dir_drive   , len_dir_drive   );
	make_unistr2(&(usr->uni_home_dir    ), home_dir    , len_home_dir    );
	make_unistr2(&(usr->uni_profile_path), profile_path, len_profile_path);
	make_unistr2(&(usr->uni_logon_script), logon_script, len_logon_script);
	make_unistr2(&(usr->uni_description ), description , len_description );

	bzero(usr->padding2, sizeof(usr->padding2));
	usr->padding3 = 0;
	
	usr->unknown_6 = unknown_6;
	usr->padding4 = 0;

	if (hrs)
	{
		memcpy(&(usr->logon_hrs), hrs, sizeof(usr->logon_hrs));
	}
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void sam_io_user_info15(char *desc, BOOL io, SAM_USER_INFO_15 *usr, struct mem_buffer *buf, int *q, int depth)
{
	if (usr == NULL) return;

	DEBUG(5,("%s%06x lsa_io_user_info %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_time("time", io, &(usr->logon_time)           , buf, q, depth); /* logon time */
	smb_io_time("time", io, &(usr->logoff_time)          , buf, q, depth); /* logoff time */
	smb_io_time("time", io, &(usr->kickoff_time)         , buf, q, depth); /* kickoff time */
	smb_io_time("time", io, &(usr->pass_last_set_time)   , buf, q, depth); /* password last set time */
	smb_io_time("time", io, &(usr->pass_can_change_time) , buf, q, depth); /* password can change time */
	smb_io_time("time", io, &(usr->pass_must_change_time), buf, q, depth); /* password must change time */

	smb_io_unihdr("unihdr", io, &(usr->hdr_user_name)   , buf, q, depth); /* username unicode string header */
	smb_io_unihdr("unihdr", io, &(usr->hdr_full_name)   , buf, q, depth); /* user's full name unicode string header */
	smb_io_unihdr("unihdr", io, &(usr->hdr_home_dir)    , buf, q, depth); /* home directory unicode string header */
	smb_io_unihdr("unihdr", io, &(usr->hdr_profile_path), buf, q, depth); /* profile path unicode string header */
	smb_io_unihdr("unihdr", io, &(usr->hdr_logon_script), buf, q, depth); /* logon script unicode string header */
	smb_io_unihdr("unihdr", io, &(usr->hdr_dir_drive)   , buf, q, depth); /* home directory drive unicode string header */
	smb_io_unihdr("unihdr", io, &(usr->hdr_description) , buf, q, depth); /* home directory drive unicode string header */

	buf_uint16("logon_count   ", depth, buf, q, io, &(usr->logon_count ));  /* logon count */
	buf_uint16("bad_pw_count  ", depth, buf, q, io, &(usr->bad_pw_count)); /* bad password count */

	buf_uint32("ptr_padding2", depth, buf, q, io, &(usr->ptr_padding2));
	buf_uint32("unknown_0", depth, buf, q, io, &(usr->unknown_0));

	buf_uint32("ptr_padding3", depth, buf, q, io, &(usr->ptr_padding3));
	buf_uint32("unknown_1", depth, buf, q, io, &(usr->unknown_1));

	buf_uint32("ptr_unknown6", depth, buf, q, io, &(usr->ptr_unknown6));
	buf_uint8s (False, "unknown_2", depth, buf, q, io, usr->unknown_2, sizeof(usr->unknown_2));

	buf_uint32("user_rid      ", depth, buf, q, io, &(usr->user_rid     ));       /* User ID */
	buf_uint32("group_rid     ", depth, buf, q, io, &(usr->group_rid    ));      /* Group ID */
	buf_uint16("acb_info      ", depth, buf, q, io, &(usr->acb_info     ));      /* Group ID */
	buf_align(buf, q);

	buf_uint32("unknown_3     ", depth, buf, q, io, &(usr->unknown_3    ));
	buf_uint16("logon_divs    ", depth, buf, q, io, &(usr->logon_divs   ));     /* logon divisions per week */
	buf_align(buf, q);
	buf_uint32("ptr_logon_hrs ", depth, buf, q, io, &(usr->ptr_logon_hrs));
	buf_uint32("unknown_5     ", depth, buf, q, io, &(usr->unknown_5    ));

	buf_uint8s (False, "padding1      ", depth, buf, q, io, usr->padding1, sizeof(usr->padding1));

	/* here begins pointed-to data */

	smb_io_unistr2("unistr2", io, &(usr->uni_user_name)   , usr->hdr_user_name   .buffer, buf, q, depth); /* username unicode string */
	smb_io_unistr2("unistr2", io, &(usr->uni_full_name)   , usr->hdr_full_name   .buffer, buf, q, depth); /* user's full name unicode string */
	smb_io_unistr2("unistr2", io, &(usr->uni_dir_drive)   , usr->hdr_dir_drive   .buffer, buf, q, depth); /* home directory drive unicode string */
	smb_io_unistr2("unistr2", io, &(usr->uni_home_dir)    , usr->hdr_home_dir    .buffer, buf, q, depth); /* home directory unicode string */
	smb_io_unistr2("unistr2", io, &(usr->uni_profile_path), usr->hdr_profile_path.buffer, buf, q, depth); /* profile path unicode string */
	smb_io_unistr2("unistr2", io, &(usr->uni_logon_script), usr->hdr_logon_script.buffer, buf, q, depth); /* logon script unicode string */
	smb_io_unistr2("unistr2", io, &(usr->uni_description ), usr->hdr_description .buffer, buf, q, depth); /* user description unicode string */

	buf_uint8s (False, "padding2", depth, buf, q, io, usr->padding2, sizeof(usr->padding2));

	buf_uint32("padding3      ", depth, buf, q, io, &(usr->padding3   ));

	buf_uint32("unknown_6     ", depth, buf, q, io, &(usr->unknown_6  ));
	buf_uint32("padding4      ", depth, buf, q, io, &(usr->padding4   ));

	if (usr->ptr_logon_hrs)
	{
		smb_io_logon_hrs("logon_hrs", io, &(usr->logon_hrs)   , buf, q, depth);
	}
}



/*******************************************************************
makes a SAMR_R_QUERY_USERINFO structure.
********************************************************************/
void make_samr_r_query_userinfo(SAMR_R_QUERY_USERINFO *r_u,
				uint16 switch_value, void *info, uint32 status)
				
{
	if (r_u == NULL || info == NULL) return;

	DEBUG(5,("make_samr_r_query_userinfo\n"));

	r_u->ptr = 0;
	r_u->switch_value = 0;

	if (status == 0)
	{
		r_u->switch_value = switch_value;

		switch (switch_value)
		{
			case 0x10:
			{
				r_u->ptr = 1;
				r_u->info.id10 = (SAM_USER_INFO_10*)info;

				break;
			}

			case 0x11:
			{
				r_u->ptr = 1;
				r_u->info.id11 = (SAM_USER_INFO_11*)info;

				break;
			}

			case 0x15:
			{
				r_u->ptr = 1;
				r_u->info.id15 = (SAM_USER_INFO_15*)info;

				break;
			}

			default:
			{
				DEBUG(4,("make_samr_r_query_aliasinfo: unsupported switch level\n"));
				break;
			}
		}
	}

	r_u->status = status;         /* return status */
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_query_userinfo(char *desc, BOOL io, SAMR_R_QUERY_USERINFO *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_query_userinfo %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr         ", depth, buf, q, io, &(r_u->ptr         ));
	buf_uint16("switch_value", depth, buf, q, io, &(r_u->switch_value));
	buf_align(buf, q);

	if (r_u->ptr != 0 && r_u->switch_value != 0)
	{
		switch (r_u->switch_value)
		{
/*
			case 0x10:
			{
				if (r_u->info.id10 != NULL)
				{
					sam_io_user_info10("", io, r_u->info.id10, buf, q, depth);
				}
				else
				{
					DEBUG(2,("samr_io_r_query_userinfo: info pointer not initialised\n"));
					return NULL;
				}
				break;
			}
			case 0x11:
			{
				if (r_u->info.id11 != NULL)
				{
					sam_io_user_info11("", io, r_u->info.id11, buf, q, depth);
				}
				else
				{
					DEBUG(2,("samr_io_r_query_userinfo: info pointer not initialised\n"));
					return NULL;
				}
				break;
			}
*/
			case 0x15:
			{
				if (r_u->info.id15 != NULL)
				{
					sam_io_user_info15("", io, r_u->info.id15, buf, q, depth);
				}
				else
				{
					DEBUG(2,("samr_io_r_query_userinfo: info pointer not initialised\n"));
					return;
				}
				break;
			}
			default:
			{
				DEBUG(2,("samr_io_r_query_userinfo: unknown switch level\n"));
				break;
			}
				
		}
	}

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_UNKNOWN_21 structure.
********************************************************************/
void make_samr_q_unknown_21(SAMR_Q_UNKNOWN_21 *q_c,
				POLICY_HND *hnd, uint16 unk_1, uint16 unk_2)
{
	if (q_c == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_unknown_21\n"));

	memcpy(&(q_c->group_pol), hnd, sizeof(q_c->group_pol));
	q_c->unknown_1 = unk_1;
	q_c->unknown_2 = unk_2;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_21(char *desc, BOOL io, SAMR_Q_UNKNOWN_21 *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_unknown_21 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("group_pol", io, &(q_u->group_pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint16("unknown_1", depth, buf, q, io, &(q_u->unknown_1));
	buf_uint16("unknown_2", depth, buf, q, io, &(q_u->unknown_2));
}

/*******************************************************************
makes a SAMR_Q_UNKNOWN_13 structure.
********************************************************************/
void make_samr_q_unknown_13(SAMR_Q_UNKNOWN_13 *q_c,
				POLICY_HND *hnd, uint16 unk_1, uint16 unk_2)
{
	if (q_c == NULL || hnd == NULL) return;

	DEBUG(5,("make_samr_q_unknown_13\n"));

	memcpy(&(q_c->alias_pol), hnd, sizeof(q_c->alias_pol));
	q_c->unknown_1 = unk_1;
	q_c->unknown_2 = unk_2;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_13(char *desc, BOOL io, SAMR_Q_UNKNOWN_13 *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_unknown_13 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("alias_pol", io, &(q_u->alias_pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint16("unknown_1", depth, buf, q, io, &(q_u->unknown_1));
	buf_uint16("unknown_2", depth, buf, q, io, &(q_u->unknown_2));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_unknown_32(char *desc, BOOL io, SAMR_Q_UNKNOWN_32 *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_unknown_32 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(q_u->pol), buf, q, depth); 
	buf_align(buf, q);

	smb_io_unihdr ("", io, &(q_u->hdr_mach_acct), buf, q, depth); 
	smb_io_unistr2("", io, &(q_u->uni_mach_acct), q_u->hdr_mach_acct.buffer, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("acct_ctrl", depth, buf, q, io, &(q_u->acct_ctrl));
	buf_uint16("unknown_1", depth, buf, q, io, &(q_u->unknown_1));
	buf_uint16("unknown_2", depth, buf, q, io, &(q_u->unknown_2));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_unknown_32(char *desc, BOOL io, SAMR_R_UNKNOWN_32 *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_unknown_32 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(r_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}


/*******************************************************************
makes a SAMR_Q_CONNECT structure.
********************************************************************/
void make_samr_q_connect(SAMR_Q_CONNECT *q_u,
				char *srv_name, uint32 unknown_0)
{
	int len_srv_name = strlen(srv_name);

	if (q_u == NULL) return;

	DEBUG(5,("make_q_connect\n"));

	/* make PDC server name \\server */
	make_unistr2(&(q_u->uni_srv_name), srv_name, len_srv_name);  

	/* example values: 0x0000 0002 */
	q_u->unknown_0 = unknown_0; 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_connect(char *desc, BOOL io, SAMR_Q_CONNECT *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_connect %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_srv_name", depth, buf, q, io, &(q_u->ptr_srv_name));
	smb_io_unistr2("", io, &(q_u->uni_srv_name), q_u->ptr_srv_name, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("unknown_0   ", depth, buf, q, io, &(q_u->unknown_0   ));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_connect(char *desc, BOOL io, SAMR_R_CONNECT *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_connect %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(r_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

/*******************************************************************
makes a SAMR_Q_OPEN_ALIAS structure.
********************************************************************/
void make_samr_q_open_alias(SAMR_Q_OPEN_ALIAS *q_u,
				uint32 unknown_0, uint32 rid)
{
	if (q_u == NULL) return;

	DEBUG(5,("make_q_open_alias\n"));

	/* example values: 0x0000 0008 */
	q_u->unknown_0 = unknown_0; 

	q_u->rid_alias = rid; 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_q_open_alias(char *desc, BOOL io, SAMR_Q_OPEN_ALIAS *q_u, struct mem_buffer *buf, int *q, int depth)
{
	if (q_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_q_open_alias %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("unknown_0", depth, buf, q, io, &(q_u->unknown_0));
	buf_uint32("rid_alias", depth, buf, q, io, &(q_u->rid_alias));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void samr_io_r_open_alias(char *desc, BOOL io, SAMR_R_OPEN_ALIAS *r_u, struct mem_buffer *buf, int *q, int depth)
{
	if (r_u == NULL) return;

	DEBUG(5,("%s%06x samr_io_r_open_alias %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_pol_hnd("pol", io, &(r_u->pol), buf, q, depth); 
	buf_align(buf, q);

	buf_uint32("status", depth, buf, q, io, &(r_u->status));
}

