
/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../includes.h"
#include "../trans2.h"
#include "../nterr.h"

extern int DEBUGLEVEL;




#define TRANS_SYNT_V2             \
{                                 \
	{                             \
		0x04, 0x5d, 0x88, 0x8a,   \
		0xeb, 0x1c, 0xc9, 0x11,   \
		0x9f, 0xe8, 0x08, 0x00,   \
		0x2b, 0x10, 0x48, 0x60    \
	}, 0x02                       \
}                                 \

#define SYNT_NETLOGON_V2          \
{                                 \
	{                             \
		0x04, 0x5d, 0x88, 0x8a,   \
		0xeb, 0x1c, 0xc9, 0x11,   \
		0x9f, 0xe8, 0x08, 0x00,   \
		0x2b, 0x10, 0x48, 0x60    \
	}, 0x02                       \
}                                 \

#define SYNT_WKSSVC_V1            \
{                                 \
	{                             \
		0x98, 0xd0, 0xff, 0x6b,   \
		0x12, 0xa1, 0x10, 0x36,   \
		0x98, 0x33, 0x46, 0xc3,   \
		0xf8, 0x7e, 0x34, 0x5a    \
	}, 0x01                       \
}                                 \

#define SYNT_SRVSVC_V3            \
{                                 \
	{                             \
		0xc8, 0x4f, 0x32, 0x4b,   \
		0x70, 0x16, 0xd3, 0x01,   \
		0x12, 0x78, 0x5a, 0x47,   \
		0xbf, 0x6e, 0xe1, 0x88    \
	}, 0x03                       \
}                                 \

#define SYNT_LSARPC_V0            \
{                                 \
	{                             \
		0x78, 0x57, 0x34, 0x12,   \
		0x34, 0x12, 0xcd, 0xab,   \
		0xef, 0x00, 0x01, 0x23,   \
		0x45, 0x67, 0x89, 0xab    \
	}, 0x00                       \
}                                 \

#define SYNT_SAMR_V1              \
{                                 \
	{                             \
		0x78, 0x57, 0x34, 0x12,   \
		0x34, 0x12, 0xcd, 0xab,   \
		0xef, 0x00, 0x01, 0x23,   \
		0x45, 0x67, 0x89, 0xac    \
	}, 0x01                       \
}                                 \

#define SYNT_NETLOGON_V1          \
{                                 \
	{                             \
		0x78, 0x56, 0x34, 0x12,   \
		0x34, 0x12, 0xcd, 0xab,   \
		0xef, 0x00, 0x01, 0x23,   \
		0x45, 0x67, 0xcf, 0xfb    \
	}, 0x01                       \
}                                 \

#define SYNT_WINREG_V1            \
{                                 \
	{                             \
		0x01, 0xd0, 0x8c, 0x33,   \
		0x44, 0x22, 0xf1, 0x31,   \
		0xaa, 0xaa, 0x90, 0x00,   \
		0x38, 0x00, 0x10, 0x03    \
	}, 0x01                       \
}                                 \

#define SYNT_NONE_V0              \
{                                 \
	{                             \
		0x00, 0x00, 0x00, 0x00,   \
		0x00, 0x00, 0x00, 0x00,   \
		0x00, 0x00, 0x00, 0x00,   \
		0x00, 0x00, 0x00, 0x00    \
	}, 0x00                       \
}                                 \

struct pipe_id_info pipe_names [] =
{
	/* client pipe , abstract syntax , server pipe   , transfer syntax */
	{ PIPE_LSARPC  , SYNT_LSARPC_V0  , PIPE_LSASS    , TRANS_SYNT_V2 },
	{ PIPE_SAMR    , SYNT_SAMR_V1    , PIPE_LSASS    , TRANS_SYNT_V2 },
	{ PIPE_NETLOGON, SYNT_NETLOGON_V1, PIPE_LSASS    , TRANS_SYNT_V2 },
	{ PIPE_SRVSVC  , SYNT_SRVSVC_V3  , PIPE_NTSVCS   , TRANS_SYNT_V2 },
	{ PIPE_WKSSVC  , SYNT_WKSSVC_V1  , PIPE_NTSVCS   , TRANS_SYNT_V2 },
	{ PIPE_WINREG  , SYNT_WINREG_V1  , PIPE_WINREG   , TRANS_SYNT_V2 },
	{ NULL         , SYNT_NONE_V0    , NULL          , SYNT_NONE_V0      }
};

/********************************************************************
 rpc pipe call id 
 ********************************************************************/
uint32 get_rpc_call_id(void)
{
	static uint32 call_id = 1;
	return ++call_id;
}

/* this function is due to be replaced */
void initrpcreply(char *inbuf, char *q)
{
	uint32 callid;

	SCVAL(q, 0, 5); q++; /* RPC version 5 */
	SCVAL(q, 0, 0); q++; /* minor version 0 */
	SCVAL(q, 0, 2); q++; /* RPC response packet */
	SCVAL(q, 0, 3); q++; /* first frag + last frag */
	RSIVAL(q, 0, 0x10000000); q += 4; /* packed data representation */
	RSSVAL(q, 0, 0); q += 2; /* fragment length, fill in later */
	SSVAL(q, 0, 0); q += 2; /* authentication length */
	callid = RIVAL(inbuf, 12);
	RSIVAL(q, 0, callid); q += 4; /* call identifier - match incoming RPC */
	SIVAL(q, 0, 0x18); q += 4; /* allocation hint (no idea) */
	SSVAL(q, 0, 0); q += 2; /* presentation context identifier */
	SCVAL(q, 0, 0); q++; /* cancel count */
	SCVAL(q, 0, 0); q++; /* reserved */
}

/* this function is due to be replaced */
void endrpcreply(char *inbuf, char *q, int datalen, int rtnval, int *rlen)
{
	SSVAL(q, 8, datalen + 4);
	SIVAL(q,0x10,datalen+4-0x18); /* allocation hint */
	SIVAL(q, datalen, rtnval);
	*rlen = datalen + 4;
}

/* Group and User RID username mapping function */
BOOL name_to_rid(char *user_name, uint32 *u_rid, uint32 *g_rid)
{
    struct passwd *pw = Get_Pwnam(user_name, False, NULL);

	if (u_rid == NULL || g_rid == NULL || user_name == NULL)
	{
		return False;
	}

    if (!pw)
	{
      DEBUG(1,("Username %s is invalid on this system\n", user_name));
      return False;
    }

	if (user_in_list(user_name, lp_domain_guest_users()))
	{
		*u_rid = DOMAIN_USER_RID_GUEST;
	}
	else if (user_in_list(user_name, lp_domain_admin_users()))
	{
		*u_rid = DOMAIN_USER_RID_ADMIN;
	}
	else
	{
		/* turn the unix UID into a Domain RID.  this is what the posix
		   sub-system does (adds 1000 to the uid) */
		*u_rid = (uint32)(pw->pw_uid + 1000);
	}

	/* absolutely no idea what to do about the unix GID to Domain RID mapping */
	*g_rid = (uint32)(pw->pw_gid + 1000);

	return True;
}


/* BIG NOTE: this function only does SIDS where the identauth is not >= 2^32 */
char *dom_sid_to_string(DOM_SID *sid)
{
  static pstring sidstr;
  char subauth[16];
  int i;
  uint32 ia = (sid->id_auth[5]) +
              (sid->id_auth[4] << 8 ) +
              (sid->id_auth[3] << 16) +
              (sid->id_auth[2] << 24);

  sprintf(sidstr, "S-%d-%d", sid->sid_rev_num, ia);

  for (i = 0; i < sid->num_auths; i++)
  {
    sprintf(subauth, "-%d", sid->sub_auths[i]);
    strcat(sidstr, subauth);
  }

  DEBUG(5,("dom_sid_to_string returning %s\n", sidstr));
  return sidstr;
}

int make_dom_sid2s(char *sids_str, DOM_SID2 *sids, int max_sids)
{
	char *ptr;
	pstring s2;
	int count;

	DEBUG(4,("make_dom_sid2s: %s\n", sids_str));

	if (sids_str == NULL || *sids_str == 0) return 0;

	for (count = 0, ptr = sids_str; next_token(&ptr, s2, NULL) && count < max_sids; count++) 
	{
		make_dom_sid2(&sids[count], s2);
	}

	return count;
}

/* array lookup of well-known RID aliases.  the purpose of these escapes me.. */
/* XXXX this structure should not have the well-known RID groups added to it,
   i.e the DOMAIN_GROUP_RID_ADMIN/USER/GUEST.  */
rid_name domain_alias_rids[] = 
{
	{ DOMAIN_ALIAS_RID_ADMINS       , "admins" },
	{ DOMAIN_ALIAS_RID_USERS        , "users" },
	{ DOMAIN_ALIAS_RID_GUESTS       , "guests" },
	{ DOMAIN_ALIAS_RID_POWER_USERS  , "power_users" },

	{ DOMAIN_ALIAS_RID_ACCOUNT_OPS  , "account_ops" },
	{ DOMAIN_ALIAS_RID_SYSTEM_OPS   , "system_ops" },
	{ DOMAIN_ALIAS_RID_PRINT_OPS    , "print_ops" },
	{ DOMAIN_ALIAS_RID_BACKUP_OPS   , "backup_ops" },
	{ DOMAIN_ALIAS_RID_REPLICATOR   , "replicator" },
	{ 0                             , NULL }
};

/* array lookup of well-known Domain RID groups. */
rid_name domain_group_rids[] = 
{
	{ DOMAIN_GROUP_RID_ADMINS       , "domain admins" },
	{ DOMAIN_GROUP_RID_USERS        , "domain users" },
	{ DOMAIN_GROUP_RID_GUESTS       , "domain guests" },
	{ 0                             , NULL }
};

int make_dom_gids(char *gids_str, DOM_GID *gids)
{
	char *ptr;
	pstring s2;
	int count;

	DEBUG(4,("make_dom_gids: %s\n", gids_str));

	if (gids_str == NULL || *gids_str == 0) return 0;

	for (count = 0, ptr = gids_str; next_token(&ptr, s2, NULL) && count < LSA_MAX_GROUPS; count++) 
	{
		/* the entries are of the form GID/ATTR, ATTR being optional.*/
		char *attr;
		uint32 rid = 0;
		int i;

		attr = strchr(s2,'/');
		if (attr) *attr++ = 0;
		if (!attr || !*attr) attr = "7"; /* default value for attribute is 7 */

		/* look up the RID string and see if we can turn it into a rid number */
		for (i = 0; domain_alias_rids[i].name != NULL; i++)
		{
			if (strequal(domain_alias_rids[i].name, s2))
			{
				rid = domain_alias_rids[i].rid;
				break;
			}
		}

		if (rid == 0) rid = atoi(s2);

		if (rid == 0)
		{
			DEBUG(1,("make_dom_gids: unknown well-known alias RID %s/%s\n",
			          s2, attr));
			count--;
		}
		else
		{
			gids[count].g_rid = rid;
			gids[count].attr  = atoi(attr);

			DEBUG(5,("group id: %d attr: %d\n",
			          gids[count].g_rid,
			          gids[count].attr));
		}
	}

	return count;
}

/*******************************************************************
 gets a domain user's groups
 ********************************************************************/
void get_domain_user_groups(char *domain_groups, char *user)
{
	pstring tmp;

	if (domain_groups == NULL || user == NULL) return;

	/* any additional groups this user is in.  e.g power users */
	pstrcpy(domain_groups, lp_domain_groups());

	/* can only be a user or a guest.  cannot be guest _and_ admin */
	if (user_in_list(user, lp_domain_guest_users()))
	{
		sprintf(tmp, " %ld/7 ", DOMAIN_GROUP_RID_GUESTS);
		strcat(domain_groups, tmp);

		DEBUG(3,("domain guest access %s granted\n", tmp));
	}
	else
	{
		sprintf(tmp, " %ld/7 ", DOMAIN_GROUP_RID_USERS);
		strcat(domain_groups, tmp);

		DEBUG(3,("domain user access %s granted\n", tmp));

		if (user_in_list(user, lp_domain_admin_users()))
		{
			sprintf(tmp, " %ld/7 ", DOMAIN_GROUP_RID_ADMINS);
			strcat(domain_groups, tmp);

			DEBUG(3,("domain admin access %s granted\n", tmp));
		}
	}
}

/*******************************************************************
 creates a DCE/RPC request
 ********************************************************************/
void create_rpc_request(struct mem_buffer *rdata, int *rdata_off, uint32 call_id, uint8 op_num, int data_len)
{
	RPC_HDR_RR hdr;

	make_rpc_hdr_rr(&hdr, RPC_REQUEST, call_id, data_len, op_num);
	smb_io_rpc_hdr_rr("", False, &hdr, rdata, rdata_off, 0);
}

/*******************************************************************
 creates a DCE/RPC reply
 ********************************************************************/
void create_rpc_reply(struct mem_buffer *rdata, int *rdata_off, uint32 call_id, int data_len)
{
	RPC_HDR_RR hdr;

	make_rpc_hdr_rr(&hdr, RPC_RESPONSE, call_id, data_len, 0);
	smb_io_rpc_hdr_rr("", False, &hdr, rdata, rdata_off, 0);
}


/****************************************************************************
 send a request on an rpc pipe.  receive a response, which may be large...
 ****************************************************************************/
BOOL rpc_api_pipe(struct cli_state *cli, int t_idx,
				uint16 cmd, uint16 fnum,
				struct mem_buffer *param , struct mem_buffer *data,
				struct mem_buffer *rparam, struct mem_buffer *rdata)
{
	RPC_HDR rhdr;
	int p = 0;
	uint16 setup[2]; /* only need 2 uint16 setup parameters */
	uint32 err;

	/* create setup parameters. */
	setup[0] = cmd; 
	setup[1] = fnum; /* pipe file handle.  got this from an SMBcreateX. */

	/* send the data: receive a response. */
	if (!cli_api_pipe(cli, t_idx, "\\PIPE\\\0\0\0", 8,

	            param != NULL ? param->data_used : 0,
	            data  != NULL ? data ->data_used : 0,
	            2,

	            0,
	            data  != NULL ? 1024 : 0 ,

	            &(rparam->data_used), &(rdata->data_used),

	            param != NULL ? param->data : NULL,
	            data  != NULL ? data ->data : NULL,
	            setup,

	            &(rparam->data), &(rdata->data)))
	{
		DEBUG(5, ("cli_pipe: return critical error\n"));
		return False;
	}

	if (rdata->data != NULL)
	{
		smb_io_rpc_hdr("rpc_hdr", True, &rhdr, rdata, &p, 0);
		buf_align(rdata, &p);

		if (!p || p != 0x10)
		{
			DEBUG(5,("cli_pipe: error in rpc header\n"));
			return False;
		}
	}

	/* check if data to be sent back was too large for one SMB. */

	if (cli_error(cli, NULL, &err)) return False;

	if (err == (0x80000000 | NT_STATUS_ACCESS_VIOLATION))
	{
		int read_offset;
		int num_read = 0;
		int offset = 0;
		int size = 1024;

		DEBUG(5,("rpc_api_pipe: SMB buffer overflow (size: %d\n", rhdr.frag_len));

		do /* loop to read the rest of the buffer, using SMBreadX */
		{
			rdata->data_used += num_read;
			offset += num_read;
			read_offset = rdata->data_used;

			buf_grow(rdata, rdata->data_used + size);

			num_read = cli_readx(cli, t_idx, fnum, &(rdata->data[read_offset]), read_offset, size);

		} while (rdata->data_used < rhdr.frag_len && num_read > 0 &&
		         !cli_error(cli, NULL, NULL));
	}

	return !cli_error(cli, NULL, NULL);
}


/*******************************************************************
 receives a netlogon pipe and responds.
 ********************************************************************/
BOOL api_rpcTNP(char *rpc_name, struct api_struct *api_rpc_cmds,
				int cnum, int uid,
				struct mem_buffer *data ,
				struct mem_buffer *rdata)
{
	int i;
	RPC_HDR_RR hdr;
	int data_off = 0;
	int rdata_off = 0;

	if (data == NULL || data->data == NULL)
	{
		DEBUG(2,("%s: NULL data received\n", rpc_name));
		return False;
	}

	smb_io_rpc_hdr_rr("", True, &hdr, data, &data_off, 0);

	DEBUG(4,("%s: op %x - ", rpc_name, hdr.opnum));

	for (i = 0; api_rpc_cmds[i].name; i++)
	{
		if (api_rpc_cmds[i].opnum == hdr.opnum &&
		    api_rpc_cmds[i].fn)
		{
			DEBUG(3,("name: %s\n", api_rpc_cmds[i].name));
			break;
		}
	}

	if (api_rpc_cmds[i].name == NULL)
	{
		DEBUG(4, ("unknown\n"));
		return False;
	}

	rdata_off = 0x18; /* length of rpc header */
	api_rpc_cmds[i].fn(uid, data, &data_off, rdata, &rdata_off);

	if (rdata_off == 0x18 || (rdata_off == 0)) return False;

	/* now we know the length, do the rpc header */
	rdata->data_used = rdata_off;
	rdata_off = 0;
	create_rpc_reply(rdata, &rdata_off, hdr.hdr.call_id, rdata->data_used);

    DEBUG(10,("called %s\n", rpc_name));

	return rdata->data_used > 0 && rdata_off > 0;
}

