
/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#ifdef SYSLOG
#undef SYSLOG
#endif

#include "../includes.h"

extern int DEBUGLEVEL;


extern struct pipe_id_info pipe_names[];

/****************************************************************************
do an rpc bind
****************************************************************************/
BOOL rpc_pipe_set_hnd_state(struct cli_state *cli, int t_idx,
				char *pipe_name, uint16 fnum, uint16 device_state)
{
	struct mem_buffer param;
	struct mem_buffer rdata;
	struct mem_buffer rparam;
	BOOL state_set = False;

	if (pipe_name == NULL) return False;

	buf_init(&rdata , 4, SAFETY_MARGIN);
	buf_init(&rparam, 4, SAFETY_MARGIN);
	buf_init(&param , 4, 0);
	buf_alloc(&param, 2);

	DEBUG(5,("Bind RPC Pipe[%x]: %s - device state:%x\n",
	          fnum, pipe_name, device_state));

	/* create data parameters: device state */
	SSVAL(param.data, 0, device_state);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0001, fnum,
				&param , NULL,
				&rparam, &rdata))
	{
		DEBUG(5, ("cli_pipe: return OK\n"));
		state_set = True;
	}

	buf_free(&param );
	buf_free(&rparam);
	buf_free(&rdata );

	return state_set;
}

/****************************************************************************
 check the rpc bind acknowledge response
****************************************************************************/
static BOOL valid_pipe_name(char *pipe_name,
				RPC_IFACE *abstract, RPC_IFACE *transfer)
{
	int pipe_idx = 0;

	while (pipe_names[pipe_idx].client_pipe != NULL)
	{
		if (strcmp(pipe_name, pipe_names[pipe_idx].client_pipe ) == 0)
		{
			DEBUG(5,("Bind Abstract Syntax: "));	
			dump_data(5, &(pipe_names[pipe_idx].abstr_syntax), sizeof(pipe_names[pipe_idx].abstr_syntax));
			DEBUG(5,("Bind Transfer Syntax: "));
			dump_data(5, &(pipe_names[pipe_idx].trans_syntax), sizeof(pipe_names[pipe_idx].trans_syntax));

			/* copy the required syntaxes out so we can do the right bind */
			memcpy(transfer, &(pipe_names[pipe_idx].trans_syntax), sizeof(pipe_names[pipe_idx].trans_syntax));
			memcpy(abstract, &(pipe_names[pipe_idx].abstr_syntax), sizeof(pipe_names[pipe_idx].abstr_syntax));

			return True;
		}
		pipe_idx++;
	};

	DEBUG(5,("Bind RPC Pipe[%s] unsupported\n", pipe_name));
	return False;
}

/****************************************************************************
 check the rpc bind acknowledge response
****************************************************************************/
static BOOL check_bind_response(RPC_HDR_BA *hdr_ba, char *pipe_name, RPC_IFACE *transfer)
{
	int i = 0;

	while ((pipe_names[i].client_pipe != NULL))
	{
		DEBUG(6,("bind_rpc_pipe: searching pipe name: client:%s server:%s\n",
				  pipe_names[i].client_pipe , pipe_names[i].server_pipe ));

		if ((strcmp(pipe_name, pipe_names[i].client_pipe ) == 0))
		{
			if (strcmp(hdr_ba->addr.str, pipe_names[i].server_pipe ) == 0)
			{
				DEBUG(5,("bind_rpc_pipe: server pipe_name found: %s\n",
						pipe_names[i].server_pipe ));
				break;
			}
			else
			{
				DEBUG(2,("bind_rpc_pipe: pipe_name %s != expected pipe %s\n",
						pipe_names[i].server_pipe , hdr_ba->addr.str));
				return False;
			}
		}
		else
		{
			i++;
		}
	}

	if (pipe_names[i].server_pipe == NULL)
	{
		DEBUG(2,("bind_rpc_pipe: pipe name %s unsupported\n", hdr_ba->addr.str));
		return False;
	}

	/* check the transfer syntax */
	if (!((hdr_ba->transfer.version == transfer->version) &&
	      (memcmp(hdr_ba->transfer.data, transfer->data,
						sizeof(transfer->version)) ==0)))
	{
		DEBUG(2,("bind_rpc_pipe: transfer syntax differs\n"));
		return False;
	}
	
	/* lkclXXXX only accept one result: check the result(s) */
	if (hdr_ba->res.num_results != 0x1 || hdr_ba->res.result != 0)
	{
		DEBUG(2,("bind_rpc_pipe: bind denied results: %d reason: %x\n",
					  hdr_ba->res.num_results,
					  hdr_ba->res.reason));
	}
		
	DEBUG(5,("bind_rpc_pipe: accepted!\n"));
	return True;
}

/****************************************************************************
do an rpc bind
****************************************************************************/
BOOL rpc_pipe_bind(struct cli_state *cli, int t_idx, char *pipe_name, uint16 fnum, 
				RPC_IFACE *abstract, RPC_IFACE *transfer)
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparam;
	int p = 0;

	RPC_HDR    hdr;
	RPC_HDR_RB hdr_rb;

    BOOL valid_ack = False;

	if (pipe_name == NULL || abstract == NULL || transfer == NULL) return False;

	DEBUG(5,("Bind RPC Pipe[%x]: %s\n", fnum, pipe_name));

	if (!valid_pipe_name(pipe_name, abstract, transfer)) return False;

	buf_init(&data  , 4, SAFETY_MARGIN);
	buf_init(&rdata , 4, SAFETY_MARGIN);
	buf_init(&rparam, 4, SAFETY_MARGIN);

	buf_alloc(&data, 1024);

	/* create the request RPC_HDR_RB */
	make_rpc_hdr_rb(&hdr_rb, 
	                0x1630, 0x1630, 0x0,
	                0x1, 0x0, 0x1,
					abstract, transfer);

	/* stream the bind request data */
	p = 0x10;
	smb_io_rpc_hdr_rb("", False, &hdr_rb,  &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR */
	make_rpc_hdr(&hdr, RPC_BIND, 0x0, get_rpc_call_id(), data.data_used - 0x10);

	/* stream the header into data */
	p = 0x0;
	smb_io_rpc_hdr("", False, &hdr,  &data, &p, 0);

	/* send data on \PIPE\.  receive a response */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparam, &rdata))
	{
		RPC_HDR_BA hdr_ba;
		int hdr_len;
		int pkt_len;

		DEBUG(5, ("rpc_api_pipe: return OK\n"));

		p = 0;
		smb_io_rpc_hdr("", True, &hdr, &rdata, &p, 0);
		buf_align(&rdata, &p);

		hdr_len = p;

#if 0
		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("bind_rpc_pipe: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
			p = 0;
		}

#endif
		smb_io_rpc_hdr_ba("", True, &hdr_ba, &rdata, &p, 0);

		pkt_len = p;

#if 0
		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("bind_rpc_pipe: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
			p = 0;
		}
#endif
		if (p) valid_ack = check_bind_response(&hdr_ba, pipe_name, transfer);

	}

	buf_free(&data  );
	buf_free(&rdata );
	buf_free(&rparam);

	return valid_ack;
}

