/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../includes.h"
#include "../nterr.h"

extern int DEBUGLEVEL;

/*******************************************************************
makes an NET_Q_LOGON_CTRL2 structure.
********************************************************************/
void make_q_logon_ctrl2(NET_Q_LOGON_CTRL2 *q_l, char *server_name,
			uint32 function_code)
{
	if (q_l == NULL) return;

	DEBUG(5,("make_q_logon_ctrl2\n"));

	q_l->ptr = 1; /* undocumented pointer */
	make_unistr2 (&(q_l->uni_server_name), server_name, strlen(server_name));

	q_l->function_code = function_code; /* should only be 0x1 */
	q_l->query_level   = function_code; /* should only be 0x1 */
	q_l->switch_value  = function_code; /* should only be 0x1 */
}

/*******************************************************************
reads or writes an NET_Q_LOGON_CTRL2 structure.
********************************************************************/
void net_io_q_logon_ctrl2(char *desc, BOOL io, NET_Q_LOGON_CTRL2 *q_l, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_l == NULL) return;

	DEBUG(5,("%s%06x net_io_q_logon_ctrl2%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr          ", depth, buf, q, io, &(q_l->ptr          ));

	smb_io_unistr2 ("", io, &(q_l->uni_server_name), q_l->ptr, buf, q, depth);

	buf_align(buf, q);

	buf_uint32("function_code", depth, buf, q, io, &(q_l->function_code));
	buf_uint32("query_level  ", depth, buf, q, io, &(q_l->query_level  ));
	buf_uint32("switch_value ", depth, buf, q, io, &(q_l->switch_value ));
}

/*******************************************************************
makes an NET_R_LOGON_CTRL2 structure.
********************************************************************/
void make_r_logon_ctrl2(NET_R_LOGON_CTRL2 *r_l, uint32 query_level,
				uint32 flags, uint32 pdc_status, uint32 logon_attempts,
				uint32 tc_status, char *trusted_domain_name)
{
	if (r_l == NULL) return;

	DEBUG(5,("make_r_logon_ctrl2\n"));

	r_l->switch_value  = query_level; /* should only be 0x1 */

	switch (query_level)
	{
		case 1:
		{
			r_l->ptr = 1; /* undocumented pointer */
			make_netinfo_1(&(r_l->logon.info1), flags, pdc_status);	
			r_l->status = 0;

			break;
		}
		case 2:
		{
			r_l->ptr = 1; /* undocumented pointer */
			make_netinfo_2(&(r_l->logon.info2), flags, pdc_status,
			               tc_status, trusted_domain_name);	
			r_l->status = 0;

			break;
		}
		case 3:
		{
			r_l->ptr = 1; /* undocumented pointer */
			make_netinfo_3(&(r_l->logon.info3), flags, logon_attempts);	
			r_l->status = 0;

			break;
		}
		default:
		{
			DEBUG(2,("make_r_logon_ctrl2: unsupported switch value %d\n",
				r_l->switch_value));
			r_l->ptr = 0; /* undocumented pointer */

			/* take a guess at an error code... */
			r_l->status = NT_STATUS_INVALID_INFO_CLASS;

			break;
		}
	}
}

/*******************************************************************
reads or writes an NET_R_LOGON_CTRL2 structure.
********************************************************************/
void net_io_r_logon_ctrl2(char *desc, BOOL io, NET_R_LOGON_CTRL2 *r_l, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_l == NULL) return;

	DEBUG(5,("%s%06x net_io_r_logon_ctrl2%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_uint32("switch_value ", depth, buf, q, io, &(r_l->switch_value ));
	buf_uint32("ptr          ", depth, buf, q, io, &(r_l->ptr          ));

	if (r_l->ptr != 0)
	{
		switch (r_l->switch_value)
		{
			case 1:
			{
				smb_io_netinfo_1("", io, &(r_l->logon.info1), buf, q, depth);
				break;
			}
			case 2:
			{
				smb_io_netinfo_2("", io, &(r_l->logon.info2), buf, q, depth);
				break;
			}
			case 3:
			{
				smb_io_netinfo_3("", io, &(r_l->logon.info3), buf, q, depth);
				break;
			}
			default:
			{
				DEBUG(2,("net_io_r_logon_ctrl2: unsupported switch value %d\n",
					r_l->switch_value));
				break;
			}
		}
	}

	buf_uint32("status       ", depth, buf, q, io, &(r_l->status       ));
}

/*******************************************************************
makes an NET_R_TRUST_DOM_LIST structure.
********************************************************************/
void make_r_trust_dom(NET_R_TRUST_DOM_LIST *r_t,
			uint32 num_doms, char *dom_name)
{
	int i = 0;

	if (r_t == NULL) return;

	DEBUG(5,("make_r_trust_dom\n"));

	for (i = 0; i < MAX_TRUST_DOMS; i++)
	{
		r_t->uni_trust_dom_name[i].uni_str_len = 0;
		r_t->uni_trust_dom_name[i].uni_max_len = 0;
	}
	if (num_doms > MAX_TRUST_DOMS) num_doms = MAX_TRUST_DOMS;

	for (i = 0; i < num_doms; i++)
	{
		fstring domain_name;
		fstrcpy(domain_name, dom_name);
		strupper(domain_name);
		make_unistr2(&(r_t->uni_trust_dom_name[i]), domain_name, strlen(domain_name));
		/* the use of UNISTR2 here is non-standard. */
		r_t->uni_trust_dom_name[i].undoc = 0x1;
	}
	
	r_t->status = 0;
}

/*******************************************************************
reads or writes an NET_R_TRUST_DOM_LIST structure.
********************************************************************/
void net_io_r_trust_dom(char *desc, BOOL io, NET_R_TRUST_DOM_LIST *r_t, struct mem_buffer *buf, int *q,  int depth)
{
	int i;
	if (r_t == NULL) return;

	DEBUG(5,("%s%06x net_io_r_trust_dom%s\n", tab_depth(depth), *q, desc));
	depth++;

	for (i = 0; i < MAX_TRUST_DOMS; i++)
	{
		if (r_t->uni_trust_dom_name[i].uni_str_len == 0) break;
		smb_io_unistr2("", io, &(r_t->uni_trust_dom_name[i]), True, buf, q, depth);
	}

	buf_uint32("status", depth, buf, q, io, &(r_t->status));
}

/*******************************************************************
makes an NET_Q_TRUST_DOM_LIST structure.
********************************************************************/
void make_q_trust_dom(NET_Q_TRUST_DOM_LIST *q_l, char *server_name,
			uint32 function_code)
{
	if (q_l == NULL) return;

	DEBUG(5,("make_q_trust_dom\n"));

	make_unistr2 (&(q_l->uni_server_name), server_name, strlen(server_name));

	q_l->function_code = function_code; /* should only be 0x31 */
}

/*******************************************************************
reads or writes an NET_Q_TRUST_DOM_LIST structure.
********************************************************************/
void net_io_q_trust_dom(char *desc, BOOL io, NET_Q_TRUST_DOM_LIST *q_l, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_l == NULL) return;

	DEBUG(5,("%s%06x net_io_q_trust_dom%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_uint32("ptr          ", depth, buf, q, io, &(q_l->ptr          ));
	smb_io_unistr2 ("", io, &(q_l->uni_server_name), q_l->ptr, buf, q, depth);

	buf_align(buf, q);

	buf_uint32("function_code", depth, buf, q, io, &(q_l->function_code));
}

/*******************************************************************
makes an NET_Q_REQ_CHAL structure.
********************************************************************/
void make_q_req_chal(NET_Q_REQ_CHAL *q_c,
				char *logon_srv, char *logon_clnt,
				DOM_CHAL *clnt_chal)
{
	if (q_c == NULL) return;

	DEBUG(5,("make_q_req_chal: %d\n", __LINE__));

	q_c->undoc_buffer = 1; /* don't know what this buffer is */

	make_unistr2(&(q_c->uni_logon_srv ), logon_srv , strlen(logon_srv ));
	make_unistr2(&(q_c->uni_logon_clnt), logon_clnt, strlen(logon_clnt));

	memcpy(q_c->clnt_chal.data, clnt_chal->data, sizeof(clnt_chal->data));

	DEBUG(5,("make_q_req_chal: %d\n", __LINE__));
}

/*******************************************************************
reads or writes an NET_Q_REQ_CHAL structure.
********************************************************************/
void net_io_q_req_chal(char *desc, BOOL io, NET_Q_REQ_CHAL *q_c, struct mem_buffer *buf, int *q,  int depth)
{
	int old_align;
	if (q_c == NULL) return;

	DEBUG(5,("%s%06x net_io_q_req_chal%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
    
	buf_uint32("undoc_buffer", depth, buf, q, io, &(q_c->undoc_buffer));

	smb_io_unistr2("", io, &(q_c->uni_logon_srv), True, buf, q, depth); /* logon server unicode string */
	smb_io_unistr2("", io, &(q_c->uni_logon_clnt), True, buf, q, depth); /* logon client unicode string */

	old_align = buf->align;
	buf->align = 0;
	/* client challenge is _not_ aligned after the unicode strings */
	smb_io_chal("", io, &(q_c->clnt_chal), buf, q, depth); /* client challenge */
	buf->align = old_align;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_r_req_chal(char *desc, BOOL io, NET_R_REQ_CHAL *r_c, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_c == NULL) return;

	DEBUG(5,("%s%06x net_io_r_req_chal%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
    
	smb_io_chal("", io, &(r_c->srv_chal), buf, q, depth); /* server challenge */

	buf_uint32("status", depth, buf, q, io, &(r_c->status));
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_q_auth_2(NET_Q_AUTH_2 *q_a,
		char *logon_srv, char *acct_name, uint16 sec_chan, char *comp_name,
		DOM_CHAL *clnt_chal, uint32 clnt_flgs)
{
	if (q_a == NULL) return;

	DEBUG(5,("make_q_auth_2: %d\n", __LINE__));

	make_log_info(&(q_a->clnt_id), logon_srv, acct_name, sec_chan, comp_name);
	memcpy(q_a->clnt_chal.data, clnt_chal->data, sizeof(clnt_chal->data));
	q_a->clnt_flgs.neg_flags = clnt_flgs;

	DEBUG(5,("make_q_auth_2: %d\n", __LINE__));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_q_auth_2(char *desc, BOOL io, NET_Q_AUTH_2 *q_a, struct mem_buffer *buf, int *q,  int depth)
{
	int old_align;
	if (q_a == NULL) return;

	DEBUG(5,("%s%06x net_io_q_auth_2%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
    
	smb_io_log_info ("", io, &(q_a->clnt_id), buf, q, depth); /* client identification info */
	/* client challenge is _not_ aligned */
	old_align = buf->align;
	buf->align = 0;
	smb_io_chal     ("", io, &(q_a->clnt_chal), buf, q, depth); /* client-calculated credentials */
	buf->align = old_align;
	smb_io_neg_flags("", io, &(q_a->clnt_flgs), buf, q, depth);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_r_auth_2(char *desc, BOOL io, NET_R_AUTH_2 *r_a, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_a == NULL) return;

	DEBUG(5,("%s%06x net_io_r_auth_2%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
    
	smb_io_chal     ("", io, &(r_a->srv_chal), buf, q, depth); /* server challenge */
	smb_io_neg_flags("", io, &(r_a->srv_flgs), buf, q, depth);

	buf_uint32("status", depth, buf, q, io, &(r_a->status));
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_q_srv_pwset(NET_Q_SRV_PWSET *q_s, char sess_key[16],
		char *logon_srv, char *acct_name, uint16 sec_chan, char *comp_name,
		DOM_CRED *cred, char nt_cypher[16])
{
	if (q_s == NULL || cred == NULL) return;

	DEBUG(5,("make_q_srv_pwset\n"));

	make_clnt_info(&(q_s->clnt_id), logon_srv, acct_name, sec_chan, comp_name, cred);

	memcpy(q_s->pwd, nt_cypher, sizeof(q_s->pwd)); 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_q_srv_pwset(char *desc, BOOL io, NET_Q_SRV_PWSET *q_s, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_s == NULL) return;

	DEBUG(5,("%s%06x net_io_q_srv_pwset%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
    
	smb_io_clnt_info("", io, &(q_s->clnt_id), buf, q, depth); /* client identification/authentication info */
	buf_uint8s (False, "pwd", depth, buf, q, io, q_s->pwd, 16); /* new password - undocumented */
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_r_srv_pwset(char *desc, BOOL io, NET_R_SRV_PWSET *r_s, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_s == NULL) return;

	DEBUG(5,("%s%06x net_io_r_srv_pwset%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
    
	smb_io_cred("", io, &(r_s->srv_cred), buf, q, depth); /* server challenge */

	buf_uint32("status", depth, buf, q, io, &(r_s->status));
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_q_sam_logon(char *desc, BOOL io, NET_Q_SAM_LOGON *q_l, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_l == NULL) return;

	DEBUG(5,("%s%06x net_io_q_sam_logon%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_sam_info("", io, &(q_l->sam_id), buf, q, depth);           /* domain SID */
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_r_sam_logon(char *desc, BOOL io, NET_R_SAM_LOGON *r_l, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_l == NULL) return;

	DEBUG(5,("%s%06x net_io_r_sam_logon%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_uint32("buffer_creds", depth, buf, q, io, &(r_l->buffer_creds)); /* undocumented buffer pointer */
	smb_io_cred("", io, &(r_l->srv_creds), buf, q, depth); /* server credentials.  server time stamp appears to be ignored. */

	buf_uint16("switch_value", depth, buf, q, io, &(r_l->switch_value));
	buf_align(buf, q);

	if (r_l->switch_value != 0)
	{
		lsa_io_user_info3("", io, r_l->user, buf, q, depth);
	}

	buf_uint32("auth_resp   ", depth, buf, q, io, &(r_l->auth_resp)); /* 1 - Authoritative response; 0 - Non-Auth? */

	buf_uint32("status      ", depth, buf, q, io, &(r_l->status));

	buf_align(buf, q);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_q_sam_logoff(char *desc, BOOL io, NET_Q_SAM_LOGOFF *q_l, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_l == NULL) return;

	DEBUG(5,("%s%06x net_io_q_sam_logoff%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_sam_info("", io, &(q_l->sam_id), buf, q, depth);           /* domain SID */
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void net_io_r_sam_logoff(char *desc, BOOL io, NET_R_SAM_LOGOFF *r_l, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_l == NULL) return;

	DEBUG(5,("%s%06x net_io_r_sam_logoff%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("buffer_creds", depth, buf, q, io, &(r_l->buffer_creds)); /* undocumented buffer pointer */
	smb_io_cred("", io, &(r_l->srv_creds), buf, q, depth); /* server credentials.  server time stamp appears to be ignored. */

	buf_uint32("status      ", depth, buf, q, io, &(r_l->status));
}


