/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../includes.h"
#include "../nterr.h"

extern int DEBUGLEVEL;


/*******************************************************************
makes an LSA_Q_OPEN_POL structure.
********************************************************************/
void make_q_open_pol(LSA_Q_OPEN_POL *r_q, char *server_name,
			uint32 attributes, uint32 sec_qos,
			uint32 desired_access)
{
	if (r_q == NULL) return;

	DEBUG(5,("make_open_pol\n"));

	r_q->ptr = 1; /* undocumented pointer */

	make_unistr2 (&(r_q->uni_server_name), server_name, strlen(server_name));
	make_obj_attr(&(r_q->attr           ), attributes, sec_qos);

	r_q->des_access = desired_access;
}

/*******************************************************************
reads or writes an LSA_Q_OPEN_POL structure.
********************************************************************/
void lsa_io_q_open_pol(char *desc, BOOL io, LSA_Q_OPEN_POL *r_q, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_q == NULL) return;

	DEBUG(5,("%s%06x lsa_io_q_open_pol%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_uint32("ptr       ", depth, buf, q, io, &(r_q->ptr       ));

	smb_io_unistr2 ("", io, &(r_q->uni_server_name), r_q->ptr, buf, q, depth);
	smb_io_obj_attr("", io, &(r_q->attr           ), buf, q, depth);

	buf_uint32("des_access", depth, buf, q, io, &(r_q->des_access));
}

/*******************************************************************
reads or writes an LSA_R_OPEN_POL structure.
********************************************************************/
void lsa_io_r_open_pol(char *desc, BOOL io, LSA_R_OPEN_POL *r_p, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_p == NULL) return;

	DEBUG(5,("%s%06x lsa_io_r_open_pol%s\n", tab_depth(depth), *q, desc));
	depth++;


	smb_io_pol_hnd("", io, &(r_p->pol), buf, q, depth);

	buf_uint32("status", depth, buf, q, io, &(r_p->status));
}

/*******************************************************************
makes an LSA_Q_QUERY_INFO structure.
********************************************************************/
void make_q_query(LSA_Q_QUERY_INFO *q_q, POLICY_HND *hnd, uint16 info_class)
{
	if (q_q == NULL || hnd == NULL) return;

	DEBUG(5,("make_q_query\n"));

	memcpy(&(q_q->pol), hnd, sizeof(q_q->pol));

	q_q->info_class = info_class;
}

/*******************************************************************
reads or writes an LSA_Q_QUERY_INFO structure.
********************************************************************/
void lsa_io_q_query(char *desc, BOOL io, LSA_Q_QUERY_INFO *q_q, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_q == NULL) return;

	DEBUG(5,("%s%06x lsa_io_q_query%s\n", tab_depth(depth), *q, desc));
	depth++;

	smb_io_pol_hnd("", io, &(q_q->pol), buf, q, depth);

	buf_uint16("info_class", depth, buf, q, io, &(q_q->info_class));
}

/*******************************************************************
reads or writes an LSA_Q_ENUM_TRUST_DOM structure.
********************************************************************/
void lsa_io_q_enum_trust_dom(char *desc, BOOL io, LSA_Q_ENUM_TRUST_DOM *q_e, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_e == NULL) return;

	DEBUG(5,("%s%06x lsa_io_q_enum_trust_dom%s\n", tab_depth(depth), *q, desc));
	depth++;


	smb_io_pol_hnd("", io, &(q_e->pol), buf, q, depth);

	buf_uint32("enum_context ", depth, buf, q, io, &(q_e->enum_context ));
	buf_uint32("preferred_len", depth, buf, q, io, &(q_e->preferred_len));
}

/*******************************************************************
makes an LSA_R_ENUM_TRUST_DOM structure.
********************************************************************/
void make_r_enum_trust_dom(LSA_R_ENUM_TRUST_DOM *r_e,
				uint32 enum_context, char *domain_name, char *domain_sid,
				uint32 status)
{
	if (r_e == NULL) return;

	DEBUG(5,("make_r_enum_trust_dom\n"));

	r_e->enum_context = enum_context;

	if (status == 0)
	{
		int len_domain_name = strlen(domain_name);

		r_e->num_domains  = 1;
		r_e->ptr_enum_domains = 1;
		r_e->num_domains2 = 1;

		make_uni_hdr2(&(r_e->hdr_domain_name ), len_domain_name, len_domain_name, 4);
		make_unistr2 (&(r_e->uni_domain_name ), domain_name, len_domain_name);
		make_dom_sid2(&(r_e->other_domain_sid), domain_sid);
	}
	else
	{
		r_e->num_domains = 0;
		r_e->ptr_enum_domains = 0;
	}

	r_e->status = status;
}

/*******************************************************************
reads or writes an LSA_R_ENUM_TRUST_DOM structure.
********************************************************************/
void lsa_io_r_enum_trust_dom(char *desc, BOOL io, LSA_R_ENUM_TRUST_DOM *r_e, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_e == NULL) return;

	DEBUG(5,("%s%06x lsa_io_r_enum_trust_dom%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_uint32("enum_context    ", depth, buf, q, io, &(r_e->enum_context    ));
	buf_uint32("num_domains     ", depth, buf, q, io, &(r_e->num_domains     ));
	buf_uint32("ptr_enum_domains", depth, buf, q, io, &(r_e->ptr_enum_domains));

	if (r_e->ptr_enum_domains != 0)
	{
		buf_uint32("num_domains2", depth, buf, q, io, &(r_e->num_domains2));
		smb_io_unihdr2 ("", io, &(r_e->hdr_domain_name ), buf, q, depth);
		smb_io_unistr2 ("", io, &(r_e->uni_domain_name ), r_e->hdr_domain_name.buffer, buf, q, depth);
		smb_io_dom_sid2("", io, &(r_e->other_domain_sid), buf, q, depth);
	}

	buf_uint32("status", depth, buf, q, io, &(r_e->status));
}

/*******************************************************************
makes an LSA_Q_CLOSE structure.
********************************************************************/
void make_lsa_q_close(LSA_Q_CLOSE *q_c, POLICY_HND *hnd)
{
	if (q_c == NULL || hnd == NULL) return;

	DEBUG(5,("make_lsa_q_close\n"));

	memcpy(&(q_c->pol), hnd, sizeof(q_c->pol));
}


/*******************************************************************
reads or writes an LSA_Q_CLOSE structure.
********************************************************************/
void lsa_io_q_close(char *desc, BOOL io, LSA_Q_CLOSE *q_c, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_c == NULL) return;

	DEBUG(5,("%s%06x lsa_io_q_close%s\n", tab_depth(depth), *q, desc));
	depth++;

	smb_io_pol_hnd("", io, &(q_c->pol), buf, q, depth);
}

/*******************************************************************
makes an LSA_R_CLOSE structure.
********************************************************************/
void make_lsa_r_close(LSA_R_CLOSE *q_r, POLICY_HND *hnd)
{
	if (q_r == NULL || hnd == NULL) return;

	DEBUG(5,("make_lsa_r_close\n"));

	memcpy(&(q_r->pol), hnd, sizeof(q_r->pol));
}


/*******************************************************************
reads or writes an LSA_R_CLOSE structure.
********************************************************************/
void lsa_io_r_close(char *desc, BOOL io, LSA_R_CLOSE *r_c, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_c == NULL) return;

	DEBUG(5,("%s%06x lsa_io_r_close%s\n", tab_depth(depth), *q, desc));
	depth++;

	smb_io_pol_hnd("", io, &(r_c->pol), buf, q, depth);

	buf_uint32("status", depth, buf, q, io, &(r_c->status));
}

/*******************************************************************
reads or writes an LSA_Q_QUERY_INFO structure.
********************************************************************/
void lsa_io_r_query(char *desc, BOOL io, LSA_R_QUERY_INFO *r_q, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_q == NULL) return;

	DEBUG(5,("%s%06x lsa_io_r_query%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_uint32("undoc_buffer", depth, buf, q, io, &(r_q->undoc_buffer));

	if (r_q->undoc_buffer != 0)
	{
		buf_uint16("info_class", depth, buf, q, io, &(r_q->info_class));

		switch (r_q->info_class)
		{
			case 3:
			{
				smb_io_dom_query_3("", io, &(r_q->dom.id3), buf, q, depth);
				break;
			}
			case 5:
			{
				smb_io_dom_query_5("", io, &(r_q->dom.id3), buf, q, depth);
				break;
			}
			default:
			{
				/* PANIC! */
				break;
			}
		}
	}

	buf_uint32("status", depth, buf, q, io, &(r_q->status));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_q_lookup_sids(char *desc, BOOL io, LSA_Q_LOOKUP_SIDS *q_s, struct mem_buffer *buf, int *q,  int depth)
{
	int i;

	if (q_s == NULL) return;

	DEBUG(5,("%s%06x lsa_io_q_lookup_sids%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
    smb_io_pol_hnd("", io, &(q_s->pol_hnd), buf, q, depth); /* policy handle */

	buf_uint32("num_entries          ", depth, buf, q, io, &(q_s->num_entries));
	buf_uint32("buffer_dom_sid       ", depth, buf, q, io, &(q_s->buffer_dom_sid)); /* undocumented domain SID buffer pointer */
	buf_uint32("buffer_dom_name      ", depth, buf, q, io, &(q_s->buffer_dom_name)); /* undocumented domain name buffer pointer */

	for (i = 0; i < q_s->num_entries; i++)
	{	
		fstring temp;
		sprintf(temp, "buffer_lookup_sids[%d] ", i);
		buf_uint32(temp, depth, buf, q, io, &(q_s->buffer_lookup_sids[i])); /* undocumented domain SID pointers to be looked up. */
	}

	for (i = 0; i < q_s->num_entries; i++)
	{
		smb_io_dom_sid2("", io, &(q_s->dom_sids[i]), buf, q, depth); /* domain SIDs to be looked up. */
	}

	buf_uint8s (False, "undoc                ", depth, buf, q, io, q_s->undoc, 16);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_r_lookup_sids(char *desc, BOOL io, LSA_R_LOOKUP_SIDS *r_s, struct mem_buffer *buf, int *q,  int depth)
{
	int i;

	if (r_s == NULL) return;

	DEBUG(5,("%s%06x lsa_io_r_lookup_sids%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_dom_r_ref("", io, &(r_s->dom_ref), buf, q, depth); /* domain reference info */

	buf_uint32("num_entries ", depth, buf, q, io, &(r_s->num_entries));
	buf_uint32("undoc_buffer", depth, buf, q, io, &(r_s->undoc_buffer));
	buf_uint32("num_entries2", depth, buf, q, io, &(r_s->num_entries2));

	for (i = 0; i < r_s->num_entries2; i++)
	{
		smb_io_dom_str_sid("", io, &(r_s->str_sid[i]), buf, q, depth); /* domain SIDs being looked up */
	}

	buf_uint32("num_entries3", depth, buf, q, io, &(r_s->num_entries3));

	buf_uint32("status      ", depth, buf, q, io, &(r_s->status));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_q_lookup_rids(char *desc, BOOL io, LSA_Q_LOOKUP_RIDS *q_r, struct mem_buffer *buf, int *q,  int depth)
{
	int i;

	if (q_r == NULL) return;

	DEBUG(5,("%s%06x lsa_io_q_lookup_rids%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
    smb_io_pol_hnd("", io, &(q_r->pol_hnd), buf, q, depth); /* policy handle */

	buf_uint32("num_entries    ", depth, buf, q, io, &(q_r->num_entries));
	buf_uint32("num_entries2   ", depth, buf, q, io, &(q_r->num_entries2));
	buf_uint32("buffer_dom_sid ", depth, buf, q, io, &(q_r->buffer_dom_sid)); /* undocumented domain SID buffer pointer */
	buf_uint32("buffer_dom_name", depth, buf, q, io, &(q_r->buffer_dom_name)); /* undocumented domain name buffer pointer */

	for (i = 0; i < q_r->num_entries; i++)
	{
		smb_io_dom_name("", io, &(q_r->lookup_name[i]), buf, q, depth); /* names to be looked up */
	}

	buf_uint8s (False, "undoc          ", depth, buf, q, io, q_r->undoc, UNKNOWN_LEN);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_r_lookup_rids(char *desc, BOOL io, LSA_R_LOOKUP_RIDS *r_r, struct mem_buffer *buf, int *q,  int depth)
{
	int i;

	if (r_r == NULL) return;

	DEBUG(5,("%s%06x lsa_io_r_lookup_rids%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_dom_r_ref("", io, &(r_r->dom_ref), buf, q, depth); /* domain reference info */

	buf_uint32("num_entries ", depth, buf, q, io, &(r_r->num_entries));
	buf_uint32("undoc_buffer", depth, buf, q, io, &(r_r->undoc_buffer));
	buf_uint32("num_entries2", depth, buf, q, io, &(r_r->num_entries2));

	for (i = 0; i < r_r->num_entries2; i++)
	{
		smb_io_dom_rid2("", io, &(r_r->dom_rid[i]), buf, q, depth); /* domain RIDs being looked up */
	}

	buf_uint32("num_entries3", depth, buf, q, io, &(r_r->num_entries3));

	buf_uint32("status      ", depth, buf, q, io, &(r_r->status));
}

/*************************************************************************
 make_lsa_user_info3
 *************************************************************************/
void make_lsa_user_info3(LSA_USER_INFO_3 *usr,

	NTTIME *logon_time,
	NTTIME *logoff_time,
	NTTIME *kickoff_time,
	NTTIME *pass_last_set_time,
	NTTIME *pass_can_change_time,
	NTTIME *pass_must_change_time,

	char *user_name,
	char *full_name,
	char *logon_script,
	char *profile_path,
	char *home_dir,
	char *dir_drive,

	uint16 logon_count,
	uint16 bad_pw_count,

	uint32 user_id,
	uint32 group_id,
	uint32 num_groups,
	DOM_GID *gids,
	uint32 user_flgs,

	char sess_key[16],

	char *logon_srv,
	char *logon_dom,

	char *dom_sid,
	char *other_sids)
{
	/* only cope with one "other" sid, right now. */
	/* need to count the number of space-delimited sids */
	int i;
	int num_other_sids = 0;

	int len_user_name    = strlen(user_name   );
	int len_full_name    = strlen(full_name   );
	int len_logon_script = strlen(logon_script);
	int len_profile_path = strlen(profile_path);
	int len_home_dir     = strlen(home_dir    );
	int len_dir_drive    = strlen(dir_drive   );

	int len_logon_srv    = strlen(logon_srv);
	int len_logon_dom    = strlen(logon_dom);

	usr->ptr_user_info = 1; /* yes, we're bothering to put USER_INFO data here */

	usr->logon_time            = *logon_time;
	usr->logoff_time           = *logoff_time;
	usr->kickoff_time          = *kickoff_time;
	usr->pass_last_set_time    = *pass_last_set_time;
	usr->pass_can_change_time  = *pass_can_change_time;
	usr->pass_must_change_time = *pass_must_change_time;

	make_uni_hdr(&(usr->hdr_user_name   ), len_user_name   , len_user_name   , 4);
	make_uni_hdr(&(usr->hdr_full_name   ), len_full_name   , len_full_name   , 4);
	make_uni_hdr(&(usr->hdr_logon_script), len_logon_script, len_logon_script, 4);
	make_uni_hdr(&(usr->hdr_profile_path), len_profile_path, len_profile_path, 4);
	make_uni_hdr(&(usr->hdr_home_dir    ), len_home_dir    , len_home_dir    , 4);
	make_uni_hdr(&(usr->hdr_dir_drive   ), len_dir_drive   , len_dir_drive   , 4);

	usr->logon_count = logon_count;
	usr->bad_pw_count = bad_pw_count;

	usr->user_id = user_id;
	usr->group_id = group_id;
	usr->num_groups = num_groups;
	usr->buffer_groups = 1; /* indicates fill in groups, below, even if there are none */
	usr->user_flgs = user_flgs;

	if (sess_key != NULL)
	{
		memcpy(usr->user_sess_key, sess_key, sizeof(usr->user_sess_key));
	}
	else
	{
		bzero(usr->user_sess_key, sizeof(usr->user_sess_key));
	}

	make_uni_hdr(&(usr->hdr_logon_srv), len_logon_srv, len_logon_srv, 4);
	make_uni_hdr(&(usr->hdr_logon_dom), len_logon_dom, len_logon_dom, 4);

	usr->buffer_dom_id = dom_sid ? 1 : 0; /* yes, we're bothering to put a domain SID in */

	bzero(usr->padding, sizeof(usr->padding));

	num_other_sids = make_dom_sid2s(other_sids, usr->other_sids, LSA_MAX_SIDS);

	usr->num_other_sids = num_other_sids;
	usr->buffer_other_sids = num_other_sids != 0 ? 1 : 0; 
	
	make_unistr2(&(usr->uni_user_name   ), user_name   , len_user_name   );
	make_unistr2(&(usr->uni_full_name   ), full_name   , len_full_name   );
	make_unistr2(&(usr->uni_logon_script), logon_script, len_logon_script);
	make_unistr2(&(usr->uni_profile_path), profile_path, len_profile_path);
	make_unistr2(&(usr->uni_home_dir    ), home_dir    , len_home_dir    );
	make_unistr2(&(usr->uni_dir_drive   ), dir_drive   , len_dir_drive   );

	usr->num_groups2 = num_groups;
	for (i = 0; i < num_groups; i++)
	{
		usr->gids[i] = gids[i];
	}

	make_unistr2(&(usr->uni_logon_srv), logon_srv, len_logon_srv);
	make_unistr2(&(usr->uni_logon_dom), logon_dom, len_logon_dom);

	make_dom_sid2(&(usr->dom_sid), dom_sid);
	/* "other" sids are set up above */
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_user_info3(char *desc, BOOL io, LSA_USER_INFO_3 *usr, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (usr == NULL) return;

	DEBUG(5,("%s%06x lsa_io_lsa_user_info %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("ptr_user_info ", depth, buf, q, io, &(usr->ptr_user_info));

	if (usr->ptr_user_info != 0)
	{
		smb_io_time("time", io, &(usr->logon_time)           , buf, q, depth); /* logon time */
		smb_io_time("time", io, &(usr->logoff_time)          , buf, q, depth); /* logoff time */
		smb_io_time("time", io, &(usr->kickoff_time)         , buf, q, depth); /* kickoff time */
		smb_io_time("time", io, &(usr->pass_last_set_time)   , buf, q, depth); /* password last set time */
		smb_io_time("time", io, &(usr->pass_can_change_time) , buf, q, depth); /* password can change time */
		smb_io_time("time", io, &(usr->pass_must_change_time), buf, q, depth); /* password must change time */

		smb_io_unihdr("unihdr", io, &(usr->hdr_user_name)   , buf, q, depth); /* username unicode string header */
		smb_io_unihdr("unihdr", io, &(usr->hdr_full_name)   , buf, q, depth); /* user's full name unicode string header */
		smb_io_unihdr("unihdr", io, &(usr->hdr_logon_script), buf, q, depth); /* logon script unicode string header */
		smb_io_unihdr("unihdr", io, &(usr->hdr_profile_path), buf, q, depth); /* profile path unicode string header */
		smb_io_unihdr("unihdr", io, &(usr->hdr_home_dir)    , buf, q, depth); /* home directory unicode string header */
		smb_io_unihdr("unihdr", io, &(usr->hdr_dir_drive)   , buf, q, depth); /* home directory drive unicode string header */

		buf_uint16("logon_count   ", depth, buf, q, io, &(usr->logon_count ));  /* logon count */
		buf_uint16("bad_pw_count  ", depth, buf, q, io, &(usr->bad_pw_count)); /* bad password count */

		buf_uint32("user_id       ", depth, buf, q, io, &(usr->user_id      ));       /* User ID */
		buf_uint32("group_id      ", depth, buf, q, io, &(usr->group_id     ));      /* Group ID */
		buf_uint32("num_groups    ", depth, buf, q, io, &(usr->num_groups   ));    /* num groups */
		buf_uint32("buffer_groups ", depth, buf, q, io, &(usr->buffer_groups)); /* undocumented buffer pointer to groups. */
		buf_uint32("user_flgs     ", depth, buf, q, io, &(usr->user_flgs    ));     /* user flags */

		buf_uint8s (False, "user_sess_key", depth, buf, q, io, usr->user_sess_key, 16); /* unused user session key */

		smb_io_unihdr("unihdr", io, &(usr->hdr_logon_srv), buf, q, depth); /* logon server unicode string header */
		smb_io_unihdr("unihdr", io, &(usr->hdr_logon_dom), buf, q, depth); /* logon domain unicode string header */

		buf_uint32("buffer_dom_id ", depth, buf, q, io, &(usr->buffer_dom_id)); /* undocumented logon domain id pointer */
		buf_uint8s (False, "padding       ", depth, buf, q, io, usr->padding, 40); /* unused padding bytes? */

		buf_uint32("num_other_sids", depth, buf, q, io, &(usr->num_other_sids)); /* 0 - num_sids */
		buf_uint32("buffer_other_sids", depth, buf, q, io, &(usr->buffer_other_sids)); /* NULL - undocumented pointer to SIDs. */
		
		smb_io_unistr2("unistr2", io, &(usr->uni_user_name)   , usr->hdr_user_name   .buffer, buf, q, depth); /* username unicode string */
		smb_io_unistr2("unistr2", io, &(usr->uni_full_name)   , usr->hdr_full_name   .buffer, buf, q, depth); /* user's full name unicode string */
		smb_io_unistr2("unistr2", io, &(usr->uni_logon_script), usr->hdr_logon_script.buffer, buf, q, depth); /* logon script unicode string */
		smb_io_unistr2("unistr2", io, &(usr->uni_profile_path), usr->hdr_profile_path.buffer, buf, q, depth); /* profile path unicode string */
		smb_io_unistr2("unistr2", io, &(usr->uni_home_dir)    , usr->hdr_home_dir    .buffer, buf, q, depth); /* home directory unicode string */
		smb_io_unistr2("unistr2", io, &(usr->uni_dir_drive)   , usr->hdr_dir_drive   .buffer, buf, q, depth); /* home directory drive unicode string */

		buf_align(buf, q);
		buf_uint32("num_groups2   ", depth, buf, q, io, &(usr->num_groups2));        /* num groups */
		for (i = 0; i < usr->num_groups2; i++)
		{
			smb_io_gid("", io, &(usr->gids[i]), buf, q, depth); /* group info */
		}

		smb_io_unistr2("unistr2", io, &( usr->uni_logon_srv), usr->hdr_logon_srv.buffer, buf, q, depth); /* logon server unicode string */
		smb_io_unistr2("unistr2", io, &( usr->uni_logon_dom), usr->hdr_logon_srv.buffer, buf, q, depth); /* logon domain unicode string */

		smb_io_dom_sid2("", io, &(usr->dom_sid), buf, q, depth);           /* domain SID */

		for (i = 0; i < usr->num_other_sids; i++)
		{
			smb_io_dom_sid2("", io, &(usr->other_sids[i]), buf, q, depth); /* other domain SIDs */
		}
	}
}

