/* Test the performance of SMB servers ... A start based on some ideas I had

   Version 1.0

   Copyright (C) Richard Sharpe 1996

*/

/* This program allows you to read or write files on an SMB server and it
   measures the performance of those reads and writes ...

   We take 10 (this number is settable from command line) measurements
   and list them all so we can average. 

   We read or write 10 * 1024 * 1024 bytes (my measure of a MB). If reading
   we ensure the file exists first.

   We split the write up into 1K, 2K, 4K, 8K, 16K, and 32K chunks and measure
   each lot. We have to take care to use a TCON against Windows NT 3.51.

   Command line is:

   -v means verbose
   -l means logon, username and password needed ...
   -u means user
   -p means password
   -r sets the repeat count
   -s sets the file service to try against
   -w means check write perf. Read is the default.

*/

#include <sys/types.h>
#include <unistd.h>
#include <time.h>
#include "smblib.h"

#define TEST_REPEAT_COUNT 10
#define TEST_FILE_NAME "\\testfile.dat"
#define TEST_USERNAME "fred"
#define TEST_PASSWORD "frednurk"
#define TEST_FILE_SERVICE "msfax"
#define TEST_WRITE_SIZE 4096

int verbose    = FALSE;
int logon      = FALSE;
int test_write = FALSE;
int total_file_size = 10 * 1024 * 1024;

char username[80] = "", password[80] = "", file_service[128] = "";
char file_name[80] = "", server_name[80] = "";

int prot_count = 0;
char **prots_to_use;

int repeat_count = 0;

void usage()

{

  fprintf(stderr, "test_perf -l -v -u <username> -p <password> -r <count> -s");
  fprintf(stderr, " <file-service> \\\n");
  fprintf(stderr, "          -w -f <filename> -c <chunk size> <server>\n");
  fprintf(stderr, "\t-l means logon\n");
  fprintf(stderr, "\t-v means verbose\n");
  fprintf(stderr, "\t-u <username> provides the username for logon\n");
  fprintf(stderr, "\t-p <password> provides the password for logon\n");
  fprintf(stderr, "\t-r <count> provides the repeat count of tests to do\n");
  fprintf(stderr, "\t-s <service> provides the service to use\n");
  fprintf(stderr, "\t-f <filename> provides the file name to use\n");
  fprintf(stderr, "\t-c <chunk size> sets the chunk size\n");
  fprintf(stderr, "\t<server> provides the server to connect to.\n");
}

char *SMB_Prots[] = {"PC NETWORK PROGRAM 1.0", 
			    "MICROSOFT NETWORKS 1.03",
			    "MICROSOFT NETWORKS 3.0",
			    "LANMAN1.0",
			    "LM1.2X002",
		            "LANMAN2.1",
			    "NT LM 0.12",
			    "NT LANMAN 1.0",
			    NULL};

void *connect_server(char *server, char *user, char *password)

{ void *con;
  int prot, SMB_Error;
  char **use_prots, err_string[1025];

  fprintf(stderr, "Connecting to server \\\\%s ...\n", server);

  con = SMB_Connect_Server(NULL, server);

  if (con == NULL) { /* Error ... */

    fprintf(stderr, "Unable to connect to server \"%s\" ... \n", server);
    SMB_Get_Error_Msg(SMB_Get_Last_Error(), 
		      err_string, sizeof(err_string) - 1);
    printf("Error in calling: %s ...\n", err_string);
    exit(1);

  }

  if (verbose == TRUE) 
    fprintf(stderr, "Connected to server \\\\%s ...\n", server);

  /* Now do a negotiate ... */

  if (verbose == TRUE) 
    fprintf(stderr, "Now we negotiate a protocol ... \n");

  if (prot_count > 0) /* Then use what asked for */
    use_prots = prots_to_use;
  else
    use_prots = SMB_Prots;

  if (SMB_Negotiate(con, use_prots) < 0) { /* An error */

    fprintf(stderr, "Unable to negotiate a protocol of the %i offered.\n",
	    prot_count);
    if (SMB_Get_Last_Error() == SMBlibE_Remote) {

      SMB_Error = SMB_Get_Last_SMB_Err();
      SMB_Get_SMB_Error_Msg(SMBlib_Error_Class(SMB_Error),
			    SMBlib_Error_Code(SMB_Error),
			    err_string,
			    sizeof(err_string) - 1);

    }
    else {
      SMB_Get_Error_Msg(SMB_Get_Last_Error(), err_string, sizeof(err_string) - 1);
    }


    printf("  %s ...\n", err_string);
    exit(1);

  }

  if (verbose == TRUE); {

    prot = SMB_Get_Protocol_IDX(con);

    fprintf(stderr, "  The protocol accepted was %i, %s, of %i offered.\n", 
	    prot, use_prots[prot], prot_count);

  }

  /* Now we do a logon ... */

  if (logon == TRUE) {

    if (verbose == TRUE) 
      fprintf(stderr, "Now try to logon as %s ...\n", user);

    if (SMB_Logon_Server(con, user, password) < 0) {

      fprintf(stderr, "Unable to logon to server ...\n");

      if (SMB_Get_Last_Error() == SMBlibE_Remote) {

	SMB_Error = SMB_Get_Last_SMB_Err();
	SMB_Get_SMB_Error_Msg(SMBlib_Error_Class(SMB_Error),
			      SMBlib_Error_Code(SMB_Error),
			      err_string,
			      sizeof(err_string) - 1);

      }
      else {
	SMB_Get_Error_Msg(SMB_Get_Last_Error(), err_string, 
			  sizeof(err_string) - 1);
      }

      fprintf(stderr, "  %s\n", err_string);
      fprintf(stderr, "This error ignored. We will keep going and hope ...\n");
      
    }
  }
  else {

    fprintf(stderr, "We will not try to logon ...\n");

  }

  return(con);

}

main(int argc, char* argv[])

{ extern char *optarg;
  extern int optind;
  int opt, i, start_time, end_time, write_chunk_size, writes, SMB_Error;
  int actual, this_write;
  void *con, *tree, *file;
  char file_name_tmp[128], *write_chunk, err_string[1025];

  write_chunk_size = TEST_WRITE_SIZE;

  /* Parse the parameters */

  while ((opt = getopt(argc, argv, "lvwf:p:u:s:r:c:")) != EOF) {

    switch (opt) {
    case 'c':
      write_chunk_size = atoi(optarg);
      break;

    case 'l':     /* do a logon and tree connect ... */
      logon = TRUE;
      break;

    case 'v':     /* Verbose? */
      verbose = TRUE;
      break;

    case 'w':

      test_write = TRUE;
      break;

    case 'f':    /* Get file name to use */

      strncpy(file_name, optarg, sizeof(file_name) -1);
      break;

    case 'p':     /* Get password */

      strncpy(password, optarg, sizeof(password) - 1);
      break;

    case 'u':     /* Pick up the user name */

      strncpy(username, optarg, sizeof(username) - 1);
      break;

    case 's':

      strncpy(file_service, optarg, sizeof(file_service) - 1);
      break;

    case 'r':
      repeat_count = atoi(optarg);
      break;

    default:

      usage();
      exit(1);
      break;
    }

  }

  if (optind < argc) { /* More paramenter ... pick up server name */

    strncpy(server_name, argv[optind], sizeof(server_name) - 1);

  }
  else {

    usage();
    exit(1);

  }

  /* Check parameters */

  if (repeat_count <= 0) 
    repeat_count = TEST_REPEAT_COUNT;

  if (file_name[0] == 0) 
    strncpy(file_name, TEST_FILE_NAME, sizeof(file_name) - 1);

  if (file_name[0] != '\\') { /* Put a slosh on */

    strncpy(file_name_tmp, file_name, sizeof(file_name_tmp) - 1);
    strncpy(file_name, "\\", sizeof(file_name) - 1);
    strncat(file_name, file_name_tmp, 
	    sizeof(file_name) - strlen(file_name) - 1);
  }

  if (username[0] == 0)
    strncpy(username, TEST_USERNAME, sizeof(username) - 1);

  if (password[0] == 0)
    strncpy(password, TEST_PASSWORD, sizeof(password) - 1);

  if (file_service[0] == 0)
    strncpy(file_service, TEST_FILE_SERVICE, sizeof(file_service) - 1);


  /* Go to it ... */

  if (verbose == TRUE) {

    fprintf(stderr, "User: %s \nPassword: %s \nFile Service: %s \n",
	    username, password, file_service);
  
    fprintf(stderr, "Server: %s\nFilename: %s\nRepeat count: %i\n\n",
	    server_name, file_name, repeat_count);

  }

  /* Well, now we have the parameters we want, we go to it */

  con = connect_server(server_name, username, password);

  /* Now connect to the file serice */

  strncpy(file_name_tmp, file_service, sizeof(file_name_tmp) - 1);
  strncpy(file_service, "\\\\", sizeof(file_name_tmp) - 1);
  strncat(file_service, server_name, 
	  sizeof(file_service) - strlen(file_service) - 1);
  strncat(file_service, "\\", 
	  sizeof(file_service) - strlen(file_service) - 1);
  strncat(file_service, file_name_tmp, 
	  sizeof(file_service) - strlen(file_service) - 1);

  if (verbose == TRUE)
    fprintf(stderr, "Now we connect to service: %s\n", file_service);

  tree = SMB_TreeConnect(con, NULL, file_service, "", "A:");

  if (tree == NULL) { /* We failed, print an error message and exit */

    if (SMB_Get_Last_Error() == SMBlibE_Remote) {

      SMB_Error = SMB_Get_Last_SMB_Err();
      SMB_Get_SMB_Error_Msg(SMBlib_Error_Class(SMB_Error),
			    SMBlib_Error_Code(SMB_Error),
			    err_string,
			    sizeof(err_string) - 1);

    }
    else {
      SMB_Get_Error_Msg(SMB_Get_Last_Error(), err_string, sizeof(err_string) - 1);

    }

    fprintf(stderr, "Error connecting to service: %s\n  %s\n", 
	    file_service, err_string);

    exit(1);
  }

  if (verbose == TRUE) {
    fprintf(stderr, "Connected to service: %s\n", file_service);
    fprintf(stderr, "The Max Buffer Size is: %i\n", SMB_Get_Tree_MBS(tree));
  }

  if (test_write == TRUE) { /* Read or write? */


  }
  else { /* Read, but we create the file first, then read it ... */

    /* Now open the file requested for write, as we need to write it */

    if (verbose == TRUE)
      fprintf(stderr, "Now we Create file: %s for writing.\n", file_name);

    file = SMB_Create(tree, NULL, file_name, SMB_FA_ORD);

    if (file == NULL) { /* We failed, print an error message and exit */

      if (SMB_Get_Last_Error() == SMBlibE_Remote) {

	SMB_Error = SMB_Get_Last_SMB_Err();
	SMB_Get_SMB_Error_Msg(SMBlib_Error_Class(SMB_Error),
			      SMBlib_Error_Code(SMB_Error),
			      err_string,
			      sizeof(err_string) - 1);

      }
      else {
	SMB_Get_Error_Msg(SMB_Get_Last_Error(), err_string, sizeof(err_string) - 1);

      }
      
      fprintf(stderr, "Error creating file:\n  %s\n", err_string);
      
      exit(1);
    }

    if (verbose == TRUE)
      fprintf(stderr, "File: %s created for writing...\n", file_name);

    /* Now we write the file */

    write_chunk = (char *)malloc(write_chunk_size);

    /* Should check for an error */

    for (i = 0; i < write_chunk_size; i++) {

      write_chunk[i] = i;

    }

    writes = total_file_size / write_chunk_size;

    if (verbose == TRUE) 
      fprintf(stderr, "Total writes: %i, write chunk size: %i\n", writes,
	      write_chunk_size);

    time(&start_time);

    while (total_file_size > 0) {

      if (total_file_size >= write_chunk_size)
	this_write = write_chunk_size;
      else
	this_write = total_file_size;

      actual = SMB_Write(file, write_chunk, this_write);

      if (actual < 0) {

	if (SMB_Get_Last_Error() == SMBlibE_Remote) {

	  SMB_Error = SMB_Get_Last_SMB_Err();
	  SMB_Get_SMB_Error_Msg(SMBlib_Error_Class(SMB_Error),
				SMBlib_Error_Code(SMB_Error),
				err_string,
				sizeof(err_string) - 1);

	}
	else {
	  SMB_Get_Error_Msg(SMB_Get_Last_Error(), err_string, sizeof(err_string) - 1);

	}

	fprintf(stderr, "Error writing file:\n  %s\n", err_string);

	exit(1);

	
      }
      
      if (actual < this_write) {

	fprintf(stderr, "Actual write was smaller than asked for!\n");

      }

      total_file_size = total_file_size - this_write;

    }

    time(&end_time);

    fprintf(stderr, "Total time to write: %i seconds\n", end_time - start_time);
    /* Now loop through reading the file ... and time it ... */




  }

  if (verbose == TRUE)
    fprintf(stderr, "All done, disconnecting...\n");

  SMB_Discon(con, FALSE);

}


