/* UNIX RFCNB (RFC1001/RFC1002) NEtBIOS implementation

   Version 1.0
   RFCNB IO Routines ...

   Copyright (C) Richard Sharpe 1996

*/

/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "std-includes.h"
#include "rfcnb-priv.h"
#include "rfcnb-util.h"
#include "rfcnb-io.h"
#include <sys/uio.h>


/* Discard the rest of an incoming packet as we do not have space for it
   in the buffer we allocated or were passed ...                         */

int RFCNB_Discard_Rest(struct RFCNB_Con *con, int len)

{ char temp[100];   /* Read into here */
  int rest, this_read, bytes_read;

  /* len is the amount we should read */

  rest = len;

  while (rest > 0) {

    this_read = (rest > sizeof(temp)?sizeof(temp):rest);

    bytes_read = read(con -> fd, temp, this_read);

    if (bytes_read <= 0) { /* Error so return */

      if (bytes_read < 0) 
	RFCNB_errno = RFCNBE_BadRead;
      else
	RFCNB_errno = RFCNBE_ConGone;

      return(RFCNBE_Bad);

    }
    
    rest = rest - bytes_read;

  }

  return(0);

}


/* Send an RFCNB packet to the connection.

   We just send each of the blocks linked together ... 

   If we can, try to send it as one iovec ... 

*/

int RFCNB_Put_Pkt(struct RFCNB_Con *con, struct RFCNB_Pkt *pkt, int len)

{ int len_sent, tot_sent, this_len;
  struct RFCNB_Pkt *pkt_ptr;
  char *this_data;
  int i;
  struct iovec io_list[10];          /* We should never have more      */
                                     /* If we do, this will blow up ...*/

  /* Try to send the data ... We only send as many bytes as len claims */
  /* We should try to stuff it into an IOVEC and send as one write     */


  pkt_ptr = pkt;
  len_sent =  tot_sent = 0;             /* Nothing sent so far */
  i = 0;

  while ((pkt_ptr != NULL) & (i < 10)) {  /* Watch that magic number! */

    this_len = pkt_ptr -> len;
    this_data = pkt_ptr -> data;
    if ((tot_sent + this_len) > len) 
      this_len = len - tot_sent;        /* Adjust so we don't send too much */

    /* Now plug into the iovec ... */

    io_list[i].iov_len = this_len;
    io_list[i].iov_base = this_data;
    i++;

    tot_sent += this_len;

    if (tot_sent == len) break;   /* Let's not send too much */

    pkt_ptr = pkt_ptr -> next;

  }

    if ((len_sent = writev(con -> fd, io_list, i)) < 0) { /* An error */

      con -> errno = errno;
      RFCNB_errno = RFCNBE_BadWrite;
      return(-1);

    }

  /* Should check that all sent, but for now just return */

#ifdef DEBUG

  RFCNB_Print_Pkt(stderr, "sent", pkt, len_sent); /* Print what send ... */

#endif

  return(len_sent);

}

/* Read an RFCNB packet off the connection. 

   We read the first 4 bytes, that tells us the length, then read the
   rest. We should implement a timeout, but we don't just yet 

*/


int RFCNB_Get_Pkt(struct RFCNB_Con *con, struct RFCNB_Pkt *pkt, int len)

{ int read_len, pkt_len;
  struct RFCNB_Hdr hdr;
  struct RFCNB_Pkt *pkt_frag;
  int more, this_time, offset, frag_len, this_len;

  /* Read that header straight into the buffer */

  if (len < sizeof(RFCNB_Hdr) & (len > 0)) { /* What a bozo */

    errno = ENOSPC;
#ifdef DEBUG
    fprintf(stderr, "Trying to read less than a packet:");
    perror("");
#endif
    return(RFCNBE_Bad);

  }

  if ((read_len = read(con -> fd, &hdr, sizeof(hdr))) < 0) { /* Problems */
#ifdef DEBUG
    fprintf(stderr, "Reading the packet, we got:");
    perror("");
#endif
    return(-1);

  }
 
  /* Now we check out what we got */

  if (read_len == 0) { /* Connection closed, send back eof?  */

#ifdef DEBUG
    fprintf(stderr, "Connection closed reading\n");
#endif DEBUG

    RFCNB_errno = RFCNBE_ConGone;
    return(RFCNBE_Bad);

  }

  /* What if we got less than or equal to a hdr size in bytes? */

  if (read_len < sizeof(hdr)) { /* We got a small packet */

    /* Now we need to copy the hdr portion we got into the supplied packet */

    if (len == 0) { /* Get space for the pkt */

      pkt -> data = (char *)malloc(read_len);

      if (pkt -> data == NULL) {

	RFCNB_errno = RFCNBE_NoSpace;
	return(RFCNBE_Bad);

      }
    }

    memcpy(pkt -> data, (char *)&hdr, read_len);  /*Copy data */

#ifdef DEBUG
    RFCNB_Print_Pkt(stderr, "rcvd", pkt, read_len);
#endif

    return(read_len);

  }

  /* Now, if we got at least a hdr size, alloc space for rest, if we need it */

  pkt_len = ntohs(hdr.len);

  if (len == 0) { /* Allocate space */

    pkt -> data = (char *)malloc(pkt_len + sizeof(hdr));

    if (pkt == NULL) {

      RFCNB_errno = RFCNBE_NoSpace;
      return(RFCNBE_Bad);
    }
  }

  /* Now copy in the hdr */

  memcpy(pkt -> data, (char *)&hdr, sizeof(hdr));

  /* Get the rest of the packet ... first figure out how big our buf is? */
  /* And make sure that we handle the fragments properly ... Sure should */
  /* use an iovec ...                                                    */

  more = (len<pkt_len ? len - sizeof(pkt) : pkt_len); this_time = 0;

  /* We read for each fragment ... */

  pkt_frag = (pkt -> len == read_len?pkt -> next:pkt);
  offset = (pkt -> len == read_len?0:sizeof(hdr));
  frag_len = pkt_frag -> len;
  this_len = (more < len ? more : len - sizeof(hdr));

  while (more > 0) {

    if ((this_time = read(con -> fd, (pkt_frag -> data) + offset, this_len)) <= 0) { /* Problems */

      if (read_len < 0)
	RFCNB_errno = RFCNBE_BadRead;
      else
	RFCNB_errno = RFCNBE_ConGone;

      return(RFCNBE_Bad);

    }

    read_len = read_len + this_time;  /* How much have we read ... */

    /* Now set up the next part */

    if (pkt_frag -> next == NULL) break;       /* That's it here */

    pkt_frag = pkt_frag -> next;
    frag_len = pkt_frag -> len;
    offset = 0;

    more = more - this_time;

  }

#ifdef DEBUG

  RFCNB_Print_Pkt(stderr, "rcvd", pkt, read_len + sizeof(hdr));

#endif

  if (read_len < pkt_len) {  /* Discard the rest */

    return(RFCNB_Discard_Rest(con, pkt_len - read_len));

  }

  return(read_len + sizeof(RFCNB_Hdr));
}

