/* UNIX RFCNB (RFC1001/RFC1002) NetBIOS implementation

   Version 1.0
   RFCNB Utility Routines ...

   Copyright (C) Richard Sharpe 1996

*/

/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "std-includes.h"
#include "rfcnb-priv.h"
#include "rfcnb-util.h"
#include "rfcnb-io.h"

/* Convert name and pad to 16 chars as needed */
/* Name 1 is a C string with null termination, name 2 may not be */
/* If SysName is true, then put a <00> on end, else space>       */

void RFCNB_CvtPad_Name(char *name1, char *name2)

{ char c, c1, c2;
  int i, len;

  len = strlen(name1);

  for (i = 0; i < 16; i++) {

    if (i >= len) {

     c1 = 'C'; c2 = 'A'; /* CA is a space */
 
    } else {

      c = toupper(name1[i]);     
      c1 = (char)((int)c/16 + (int)'A');
      c2 = (char)((int)c%16 + (int)'A');
    }

    name2[i*2] = c1;
    name2[i*2+1] = c2;

  }

  name2[32] = 0;   /* Put in the nll ...*/

}

/* Converts an Ascii NB Name (16 chars) to an RFCNB Name (32 chars)
   Uses the encoding in RFC1001. Each nibble of byte is added to 'A'
   to produce the next byte in the name.

   This routine assumes that AName is 16 bytes long and that NBName has 
   space for 32 chars, so be careful ... 

*/

void RFCNB_AName_To_NBName(char *AName, char *NBName)

{ char c, c1, c2;
  int i;

  for (i=0; i < 16; i++) {

    c = AName[i];

    c1 = (char)((c >> 4) + 'A');
    c2 = (char)((c & 0xF) + 'A');

    NBName[i*2] = c1;
    NBName[i*2+1] = c2;
  }

  NBName[32] = 0; /* Put in a null */

}

/* Do the reverse of the above ... */

void RFCNB_NBName_To_AName(char *NBName, char *AName)

{ char c, c1, c2;
  int i;

  for (i=0; i < 16; i++) {

    c1 = NBName[i*2];
    c2 = NBName[i*2+1];

    c = (char)(((int)c1 - (int)'A') * 16 + ((int)c2 - (int)'A'));

    AName[i] = c;

  }

  AName[i] = 0;   /* Put a null on the end ... */

}

/* Print a string of bytes in HEX etc */

void RFCNB_Print_Hex(FILE *fd, struct RFCNB_Pkt *pkt, int Offset)

{ char c, c1, c2, outbuf1[33];
  int i, j;
  struct RFCNB_Pkt *pkt_ptr = pkt;
  static char Hex_List[17] = "0123456789ABCDEF";

  j = 0;

  while (pkt_ptr != NULL) {

    for (i = 0; i < ((pkt_ptr -> len) - Offset); i++) {

      c = pkt_ptr -> data[i + Offset];
      c1 = Hex_List[c >> 4];
      c2 = Hex_List[c & 0xF];

      outbuf1[j++] = c1; outbuf1[j++] = c2;

      if (j == 32){ /* Print and reset */
	outbuf1[j] = 0;
	fprintf(fd, "    %s\n", outbuf1);
	j = 0;
      }

    }

    Offset = 0;
    pkt_ptr = pkt_ptr -> next;

  }

  fprintf(fd, "\n");

}

/* Get a packet of size n */

struct RFCNB_Pkt *RFCNB_Alloc_Pkt(int n)

{ RFCNB_Pkt *pkt;

  if ((pkt = (struct RFCNB_Pkt *)malloc(sizeof(struct RFCNB_Pkt))) == NULL) {

    RFCNB_errno = RFCNBE_NoSpace;
    return(NULL);

  }

  pkt -> next = NULL;
  pkt -> len = n;

  if (n == 0) return(pkt);

  if ((pkt -> data = (char *)malloc(n)) == NULL) {

    RFCNB_errno = RFCNBE_NoSpace;
    free(pkt);
    return(NULL);

  }

  return(pkt);

}

/* Free up a packet */

int RFCNB_Free_Pkt(struct RFCNB_Pkt *pkt)

{ struct RFCNB_Pkt *pkt_next;

  while (pkt != NULL) {

    pkt_next = pkt -> next;

    free(pkt);

    pkt = pkt_next;

  }

}

/* Print an RFCNB packet */

void RFCNB_Print_Pkt(FILE *fd, char *dirn, struct RFCNB_Pkt *pkt, int len)

{ char lname[17];

  /* We assume that the first fragment is the RFCNB Header  */
  /* We should loop through the fragments printing them out */

  fprintf(fd, "RFCNB Pkt %s:", dirn);

  switch (((struct RFCNB_Hdr *)pkt -> data) -> type) {

    case RFCNB_SESSION_MESSAGE: 

      fprintf(fd, "SESSION MESSAGE: Length = %i\n", 
	          ntohs(((struct RFCNB_Hdr *)pkt -> data) -> len));
      RFCNB_Print_Hex(fd, pkt, sizeof(struct RFCNB_Hdr));
      break;

    case RFCNB_SESSION_REQUEST:

      fprintf(fd, "SESSION REQUEST: Length = %i\n",
		  ntohs(((struct RFCNB_Hdr *)pkt -> data) -> len));
      RFCNB_NBName_To_AName(((struct RFCNB_Sess_Pkt *)pkt -> data) -> called_name, lname);
      fprintf(fd, "  Called Name: %s\n", lname);
      RFCNB_NBName_To_AName(((struct RFCNB_Sess_Pkt *)pkt -> data) -> calling_name, lname);
      fprintf(fd, "  Calling Name: %s\n", lname);

      break;

    case RFCNB_SESSION_ACK:

      fprintf(fd, "RFCNB SESSION ACK: Length = %i\n",
		  ntohs(((struct RFCNB_Hdr *)pkt -> data) -> len));

      break;

    case RFCNB_SESSION_REJ:
      fprintf(fd, "RFCNB SESSION REJECT: Length = %i\n", 
		  ntohs(((struct RFCNB_Hdr *)pkt -> data) -> len));

      if (ntohs(((struct RFCNB_Hdr *)pkt -> data) -> len) < 1) {
	fprintf(fd, "   Protocol Error, short Reject packet!\n");
      }
      else {
	fprintf(fd, "   Error = %x\n", ((struct RFCNB_Nack_Pkt *)pkt -> data) -> error);
      }

      break;

    case RFCNB_SESSION_RETARGET:

      fprintf(fd, "RFCNB SESSION RETARGET: Length = %i\n",
		  ntohs(((struct RFCNB_Hdr *)pkt -> data) -> len));

      /* Print out the IP address etc and the port? */

      break;

    default:

      break;
  }

}

/* Resolve a name into an address */

int RFCNB_Name_To_IP(char *host, struct in_addr *Dest_IP)

{ int addr;         /* Assumes IP4, 32 bit network addresses */
  struct hostent *hp;

        /* Use inet_addr to try to convert the address */

  if ((addr = inet_addr(host)) == INADDR_NONE) { /* Oh well, a good try :-) */

        /* Now try a name look up with gethostbyname */

    if ((hp = gethostbyname(host)) == NULL) { /* Not in DNS */

        /* Try NetBIOS name lookup, how the hell do we do that? */

      return -1;       /* Ignore for now ... :-) */

    }
    else {  /* We got a name */

       memcpy((void *)Dest_IP, (void *)hp -> h_addr_list[0], sizeof(struct in_addr));

    }
  }
  else { /* It was an IP address */

    memcpy((void *)Dest_IP, (void *)&addr, sizeof(struct in_addr));

  }

  return 0;

}

/* Disconnect the TCP connection to the server */

int RFCNB_Close(int socket)

{

  close(socket);

  /* If we want to do error recovery, here is where we put it */

  return 0;

}

/* Connect to the server specified in the IP address.
   Not sure how to handle socket options etc.         */

int RFCNB_IP_Connect(struct in_addr Dest_IP, int port)

{ struct sockaddr_in Socket;
  int fd;

  /* Create a socket */

  if ((fd = socket(PF_INET, SOCK_STREAM, 0)) < 0) { /* Handle the error */

    return(-1);
    } 

  bzero((char *)&Socket, sizeof(Socket));
  memcpy((char *)&Socket.sin_addr, (char *)&Dest_IP, sizeof(Dest_IP));

  Socket.sin_port = htons(port);
  Socket.sin_family = PF_INET;

  /* Now connect to the destination */

  if (connect(fd, (struct sockaddr *)&Socket, sizeof(Socket)) < 0) { /* Error */

    close(fd);
    return(-1);
    }

  return(fd);

}

/* handle the details of establishing the RFCNB session with remote 
   end 

*/

int RFCNB_Session_Req(struct RFCNB_Con *con, 
		      char *Called_Name, 
		      char *Calling_Name,
		      BOOL *redirect,
		      struct in_addr *Dest_IP,
		      int * port)

{ RFCNB_Sess_Pkt *sess_pkt;

  /* Response packet should be no more than 9 bytes, make 16 jic */

  char ln1[16], ln2[16], n1[32], n2[32];
  int len;
  struct RFCNB_Pkt *pkt, *res_pkt;

  /* We build and send the session request, then read the response */

  pkt = RFCNB_Alloc_Pkt(sizeof(RFCNB_Sess_Pkt));

  if (pkt == NULL) {

    return(RFCNBE_NoSpace);  /* Leave the error that RFCNB_Alloc_Pkt gives) */

  }

  sess_pkt = (RFCNB_Sess_Pkt *)(pkt -> data); 

  sess_pkt -> type = RFCNB_SESSION_REQUEST;
  sess_pkt -> flags = 0;
  sess_pkt -> length = htons(sizeof(RFCNB_Sess_Pkt)-sizeof(struct RFCNB_Hdr));
  sess_pkt -> n1_len = 32;
  sess_pkt -> n2_len = 32;

  RFCNB_CvtPad_Name(Called_Name, sess_pkt -> called_name);
  RFCNB_CvtPad_Name(Calling_Name, sess_pkt -> calling_name);

  /* Now send the packet */

#ifdef DEBUG

  fprintf(stderr, "Sending packet: ");
  
#endif

  if ((len = RFCNB_Put_Pkt(con, pkt, sizeof(RFCNB_Sess_Pkt))) < 0) {

    return(-1);       /* Should be able to write that lot ... */

    }

  /* Now get a response pkt */

  res_pkt = RFCNB_Alloc_Pkt(0);       /* Let lower layer deal with size */

  if (res_pkt == NULL) {

    return(RFCNBE_Bad);

  }

#ifdef DEBUG

  fprintf(stderr, "Getting packet.\n");

#endif

  if ((len = RFCNB_Get_Pkt(con, res_pkt, 0)) < 0) {

    return(RFCNBE_Bad);

    }

  /* Now analyze the packet ... */

  switch (((struct RFCNB_Hdr *)(res_pkt -> data)) -> type) {

    case RFCNB_SESSION_REJ:         /* Didnt like us ... too bad */

      return(RFCNBE_Bad);
      break;

    case RFCNB_SESSION_ACK:        /* Got what we wanted ...      */

      return(0);
      break;

    case RFCNB_SESSION_RETARGET:   /* Go elsewhere                */

      *redirect = TRUE;       /* Copy port and ip addr       */

      memcpy(Dest_IP, ((struct RFCNB_Retarget_Pkt *)(res_pkt -> data)) -> dest_ip, sizeof(struct in_addr));
      *port = ((struct RFCNB_Retarget_Pkt *)(res_pkt -> data)) -> port;

      return(0);
      break;

    default:  /* A protocol error */

      RFCNB_errno = RFCNBE_ProtErr;
      return(RFCNBE_Bad);
      break;
    }
}









