/* 
   Unix SMB/CIFS implementation.

   local testing of talloc routines.

   Copyright (C) Andrew Tridgell 2004
   
     ** NOTE! The following LGPL license applies to the talloc
     ** library. This does NOT imply that all of Samba is released
     ** under the LGPL
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "replace.h"
#include "system/time.h"
#include "talloc.h"

struct torture_context;

static struct timeval timeval_current(void)
{
	struct timeval tv;
	gettimeofday(&tv, NULL);
	return tv;
}

static double timeval_elapsed(struct timeval *tv)
{
	struct timeval tv2 = timeval_current();
	return (tv2.tv_sec - tv->tv_sec) + 
	       (tv2.tv_usec - tv->tv_usec)*1.0e-6;
}

#if _SAMBA_BUILD_==3
#ifdef malloc
#undef malloc
#endif
#ifdef strdup
#undef strdup
#endif
#endif

#define CHECK_SIZE(ptr, tsize) do { \
	if (talloc_total_size(ptr) != (tsize)) { \
		printf(__location__ " failed: wrong '%s' tree size: got %u  expected %u\n", \
		       #ptr, \
		       (unsigned)talloc_total_size(ptr), \
		       (unsigned)tsize); \
		talloc_report_full(ptr, stdout); \
		return false; \
	} \
} while (0)

#define CHECK_BLOCKS(ptr, tblocks) do { \
	if (talloc_total_blocks(ptr) != (tblocks)) { \
		printf(__location__ " failed: wrong '%s' tree blocks: got %u  expected %u\n", \
		       #ptr, \
		       (unsigned)talloc_total_blocks(ptr), \
		       (unsigned)tblocks); \
		talloc_report_full(ptr, stdout); \
		return false; \
	} \
} while (0)

#define CHECK_PARENT(ptr, parent) do { \
	if (talloc_parent(ptr) != (parent)) { \
		printf(__location__ " failed: '%s' has wrong parent: got %p  expected %p\n", \
		       #ptr, \
		       talloc_parent(ptr), \
		       (parent)); \
		talloc_report_full(ptr, stdout); \
		talloc_report_full(parent, stdout); \
		talloc_report_full(NULL, stdout); \
		return false; \
	} \
} while (0)


/*
  test references 
*/
static bool test_ref1(void)
{
	void *root, *p1, *p2, *ref, *r1;

	printf("TESTING SINGLE REFERENCE FREE\n");

	root = talloc_named_const(NULL, 0, "root");
	p1 = talloc_named_const(root, 1, "p1");
	p2 = talloc_named_const(p1, 1, "p2");
	talloc_named_const(p1, 1, "x1");
	talloc_named_const(p1, 2, "x2");
	talloc_named_const(p1, 3, "x3");

	r1 = talloc_named_const(root, 1, "r1");	
	ref = talloc_reference(r1, p2);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 5);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(r1, 2);

	printf("Freeing p2\n");
	talloc_free(p2);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 5);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(r1, 1);

	printf("Freeing p1\n");
	talloc_free(p1);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(r1, 1);

	printf("Freeing r1\n");
	talloc_free(r1);
	talloc_report_full(NULL, stdout);

	printf("Testing NULL\n");
	if (talloc_reference(root, NULL)) {
		return false;
	}

	CHECK_BLOCKS(root, 1);

	CHECK_SIZE(root, 0);

	talloc_free(root);

	return true;
}

/*
  test references 
*/
static bool test_ref2(void)
{
	void *root, *p1, *p2, *ref, *r1;

	printf("TESTING DOUBLE REFERENCE FREE\n");

	root = talloc_named_const(NULL, 0, "root");
	p1 = talloc_named_const(root, 1, "p1");
	talloc_named_const(p1, 1, "x1");
	talloc_named_const(p1, 1, "x2");
	talloc_named_const(p1, 1, "x3");
	p2 = talloc_named_const(p1, 1, "p2");

	r1 = talloc_named_const(root, 1, "r1");	
	ref = talloc_reference(r1, p2);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 5);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(r1, 2);

	printf("Freeing ref\n");
	talloc_free(ref);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 5);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(r1, 1);

	printf("Freeing p2\n");
	talloc_free(p2);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 4);
	CHECK_BLOCKS(r1, 1);

	printf("Freeing p1\n");
	talloc_free(p1);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(r1, 1);

	printf("Freeing r1\n");
	talloc_free(r1);
	talloc_report_full(root, stdout);

	CHECK_SIZE(root, 0);

	talloc_free(root);

	return true;
}

/*
  test references 
*/
static bool test_ref3(void)
{
	void *root, *p1, *p2, *ref, *r1;

	printf("TESTING PARENT REFERENCE FREE\n");

	root = talloc_named_const(NULL, 0, "root");
	p1 = talloc_named_const(root, 1, "p1");
	p2 = talloc_named_const(root, 1, "p2");
	r1 = talloc_named_const(p1, 1, "r1");
	ref = talloc_reference(p2, r1);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 2);
	CHECK_BLOCKS(p2, 2);
	CHECK_BLOCKS(r1, 1);

	printf("Freeing p1\n");
	talloc_free(p1);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p2, 2);
	CHECK_BLOCKS(r1, 1);

	printf("Freeing p2\n");
	talloc_free(p2);
	talloc_report_full(root, stdout);

	CHECK_SIZE(root, 0);

	talloc_free(root);

	return true;
}

/*
  test references 
*/
static bool test_ref4(void)
{
	void *root, *p1, *p2, *ref, *r1;

	printf("TESTING REFERRER REFERENCE FREE\n");

	root = talloc_named_const(NULL, 0, "root");
	p1 = talloc_named_const(root, 1, "p1");
	talloc_named_const(p1, 1, "x1");
	talloc_named_const(p1, 1, "x2");
	talloc_named_const(p1, 1, "x3");
	p2 = talloc_named_const(p1, 1, "p2");

	r1 = talloc_named_const(root, 1, "r1");	
	ref = talloc_reference(r1, p2);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 5);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(r1, 2);

	printf("Freeing r1\n");
	talloc_free(r1);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 5);
	CHECK_BLOCKS(p2, 1);

	printf("Freeing p2\n");
	talloc_free(p2);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 4);

	printf("Freeing p1\n");
	talloc_free(p1);
	talloc_report_full(root, stdout);

	CHECK_SIZE(root, 0);

	talloc_free(root);

	return true;
}


/*
  test references 
*/
static bool test_unlink1(void)
{
	void *root, *p1, *p2, *ref, *r1;

	printf("TESTING UNLINK\n");

	root = talloc_named_const(NULL, 0, "root");
	p1 = talloc_named_const(root, 1, "p1");
	talloc_named_const(p1, 1, "x1");
	talloc_named_const(p1, 1, "x2");
	talloc_named_const(p1, 1, "x3");
	p2 = talloc_named_const(p1, 1, "p2");

	r1 = talloc_named_const(p1, 1, "r1");	
	ref = talloc_reference(r1, p2);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 7);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(r1, 2);

	printf("Unreferencing r1\n");
	talloc_unlink(r1, p2);
	talloc_report_full(root, stdout);

	CHECK_BLOCKS(p1, 6);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(r1, 1);

	printf("Freeing p1\n");
	talloc_free(p1);
	talloc_report_full(root, stdout);

	CHECK_SIZE(root, 0);

	talloc_free(root);

	return true;
}

static int fail_destructor(void *ptr)
{
	return -1;
}

/*
  miscellaneous tests to try to get a higher test coverage percentage
*/
static bool test_misc(void)
{
	void *root, *p1;
	char *p2;
	double *d;
	const char *name;

	printf("TESTING MISCELLANEOUS\n");

	root = talloc_new(NULL);

	p1 = talloc_size(root, 0x7fffffff);
	if (p1) {
		printf("failed: large talloc allowed\n");
		return false;
	}

	p1 = talloc_strdup(root, "foo");
	talloc_increase_ref_count(p1);
	talloc_increase_ref_count(p1);
	talloc_increase_ref_count(p1);
	CHECK_BLOCKS(p1, 1);
	CHECK_BLOCKS(root, 2);
	talloc_free(p1);
	CHECK_BLOCKS(p1, 1);
	CHECK_BLOCKS(root, 2);
	talloc_unlink(NULL, p1);
	CHECK_BLOCKS(p1, 1);
	CHECK_BLOCKS(root, 2);
	p2 = talloc_strdup(p1, "foo");
	if (talloc_unlink(root, p2) != -1) {
		printf("failed: talloc_unlink() of non-reference context should return -1\n");
		return false;
	}
	if (talloc_unlink(p1, p2) != 0) {
		printf("failed: talloc_unlink() of parent should succeed\n");
		return false;
	}
	talloc_free(p1);
	CHECK_BLOCKS(p1, 1);
	CHECK_BLOCKS(root, 2);

	name = talloc_set_name(p1, "my name is %s", "foo");
	if (strcmp(talloc_get_name(p1), "my name is foo") != 0) {
		printf("failed: wrong name after talloc_set_name(my name is foo) - '%s'=>'%s'\n",
			(name?name:"NULL"), talloc_get_name(p1));
		return false;
	}
	CHECK_BLOCKS(p1, 2);
	CHECK_BLOCKS(root, 3);

	talloc_set_name_const(p1, NULL);
	if (strcmp(talloc_get_name(p1), "UNNAMED") != 0) {
		printf("failed: wrong name after talloc_set_name(NULL) - '%s'\n",
			talloc_get_name(p1));
		return false;
	}
	CHECK_BLOCKS(p1, 2);
	CHECK_BLOCKS(root, 3);
	

	if (talloc_free(NULL) != -1) {
		printf("talloc_free(NULL) should give -1\n");
		return false;
	}

	talloc_set_destructor(p1, fail_destructor);
	if (talloc_free(p1) != -1) {
		printf("Failed destructor should cause talloc_free to fail\n");
		return false;
	}
	talloc_set_destructor(p1, NULL);

	talloc_report(root, stdout);


	p2 = (char *)talloc_zero_size(p1, 20);
	if (p2[19] != 0) {
		printf("Failed to give zero memory\n");
		return false;
	}
	talloc_free(p2);

	if (talloc_strdup(root, NULL) != NULL) {
		printf("failed: strdup on NULL should give NULL\n");
		return false;
	}

	p2 = talloc_strndup(p1, "foo", 2);
	if (strcmp("fo", p2) != 0) {
		printf("failed: strndup doesn't work\n");
		return false;
	}
	p2 = talloc_asprintf_append(p2, "o%c", 'd');
	if (strcmp("food", p2) != 0) {
		printf("failed: talloc_asprintf_append doesn't work\n");
		return false;
	}
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(p1, 3);

	p2 = talloc_asprintf_append(NULL, "hello %s", "world");
	if (strcmp("hello world", p2) != 0) {
		printf("failed: talloc_asprintf_append doesn't work\n");
		return false;
	}
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(p1, 3);
	talloc_free(p2);

	d = talloc_array(p1, double, 0x20000000);
	if (d) {
		printf("failed: integer overflow not detected\n");
		return false;
	}

	d = talloc_realloc(p1, d, double, 0x20000000);
	if (d) {
		printf("failed: integer overflow not detected\n");
		return false;
	}

	talloc_free(p1);
	CHECK_BLOCKS(root, 1);

	p1 = talloc_named(root, 100, "%d bytes", 100);
	CHECK_BLOCKS(p1, 2);
	CHECK_BLOCKS(root, 3);
	talloc_unlink(root, p1);

	p1 = talloc_init("%d bytes", 200);
	p2 = talloc_asprintf(p1, "my test '%s'", "string");
	if (strcmp(p2, "my test 'string'") != 0) {
		printf("failed: talloc_asprintf(\"my test '%%s'\", \"string\") gave: \"%s\"\n", p2);
		return false;
	}
	CHECK_BLOCKS(p1, 3);
	CHECK_SIZE(p2, 17);
	CHECK_BLOCKS(root, 1);
	talloc_unlink(NULL, p1);

	p1 = talloc_named_const(root, 10, "p1");
	p2 = (char *)talloc_named_const(root, 20, "p2");
	(void)talloc_reference(p1, p2);
	talloc_report_full(root, stdout);
	talloc_unlink(root, p2);
	talloc_report_full(root, stdout);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(p1, 2);
	CHECK_BLOCKS(root, 3);
	talloc_unlink(p1, p2);
	talloc_unlink(root, p1);

	p1 = talloc_named_const(root, 10, "p1");
	p2 = (char *)talloc_named_const(root, 20, "p2");
	(void)talloc_reference(NULL, p2);
	talloc_report_full(root, stdout);
	talloc_unlink(root, p2);
	talloc_report_full(root, stdout);
	CHECK_BLOCKS(p2, 1);
	CHECK_BLOCKS(p1, 1);
	CHECK_BLOCKS(root, 2);
	talloc_unlink(NULL, p2);
	talloc_unlink(root, p1);

	/* Test that talloc_unlink is a no-op */

	if (talloc_unlink(root, NULL) != -1) {
		printf("failed: talloc_unlink(root, NULL) == -1\n");
		return false;
	}

	talloc_report(root, stdout);
	talloc_report(NULL, stdout);

	CHECK_SIZE(root, 0);

	talloc_free(root);

	CHECK_SIZE(NULL, 0);

	talloc_enable_leak_report();
	talloc_enable_leak_report_full();

	return true;
}


/*
  test realloc
*/
static bool test_realloc(void)
{
	void *root, *p1, *p2;

	printf("TESTING REALLOC\n");

	root = talloc_new(NULL);

	p1 = talloc_size(root, 10);
	CHECK_SIZE(p1, 10);

	p1 = talloc_realloc_size(NULL, p1, 20);
	CHECK_SIZE(p1, 20);

	talloc_new(p1);

	p2 = talloc_realloc_size(p1, NULL, 30);

	talloc_new(p1);

	p2 = talloc_realloc_size(p1, p2, 40);

	CHECK_SIZE(p2, 40);
	CHECK_SIZE(root, 60);
	CHECK_BLOCKS(p1, 4);

	p1 = talloc_realloc_size(NULL, p1, 20);
	CHECK_SIZE(p1, 60);

	talloc_increase_ref_count(p2);
	if (talloc_realloc_size(NULL, p2, 5) != NULL) {
		printf("failed: talloc_realloc() on a referenced pointer should fail\n");
		return false;
	}
	CHECK_BLOCKS(p1, 4);

	talloc_realloc_size(NULL, p2, 0);
	talloc_realloc_size(NULL, p2, 0);
	CHECK_BLOCKS(p1, 3);

	if (talloc_realloc_size(NULL, p1, 0x7fffffff) != NULL) {
		printf("failed: oversize talloc should fail\n");
		return false;
	}

	talloc_realloc_size(NULL, p1, 0);

	CHECK_BLOCKS(root, 1);
	CHECK_SIZE(root, 0);

	talloc_free(root);

	return true;
}

/*
  test realloc with a child
*/
static bool test_realloc_child(void)
{
	void *root;
	struct el2 {
		const char *name;
	} *el2;	
	struct el1 {
		int count;
		struct el2 **list, **list2, **list3;
	} *el1;

	printf("TESTING REALLOC WITH CHILD\n");

	root = talloc_new(NULL);

	el1 = talloc(root, struct el1);
	el1->list = talloc(el1, struct el2 *);
	el1->list[0] = talloc(el1->list, struct el2);
	el1->list[0]->name = talloc_strdup(el1->list[0], "testing");

	el1->list2 = talloc(el1, struct el2 *);
	el1->list2[0] = talloc(el1->list2, struct el2);
	el1->list2[0]->name = talloc_strdup(el1->list2[0], "testing2");

	el1->list3 = talloc(el1, struct el2 *);
	el1->list3[0] = talloc(el1->list3, struct el2);
	el1->list3[0]->name = talloc_strdup(el1->list3[0], "testing2");
	
	el2 = talloc(el1->list, struct el2);
	el2 = talloc(el1->list2, struct el2);
	el2 = talloc(el1->list3, struct el2);

	el1->list = talloc_realloc(el1, el1->list, struct el2 *, 100);
	el1->list2 = talloc_realloc(el1, el1->list2, struct el2 *, 200);
	el1->list3 = talloc_realloc(el1, el1->list3, struct el2 *, 300);

	talloc_free(root);

	return true;
}


/*
  test type checking
*/
static bool test_type(void)
{
	void *root;
	struct el1 {
		int count;
	};
	struct el2 {
		int count;
	};
	struct el1 *el1;

	printf("TESTING talloc type checking\n");

	root = talloc_new(NULL);

	el1 = talloc(root, struct el1);

	el1->count = 1;

	if (talloc_get_type(el1, struct el1) != el1) {
		printf("type check failed on el1\n");
		return false;
	}
	if (talloc_get_type(el1, struct el2) != NULL) {
		printf("type check failed on el1 with el2\n");
		return false;
	}
	talloc_set_type(el1, struct el2);
	if (talloc_get_type(el1, struct el2) != (struct el2 *)el1) {
		printf("type set failed on el1 with el2\n");
		return false;
	}

	talloc_free(root);

	return true;
}

/*
  test steal
*/
static bool test_steal(void)
{
	void *root, *p1, *p2;

	printf("TESTING STEAL\n");

	root = talloc_new(NULL);

	p1 = talloc_array(root, char, 10);
	CHECK_SIZE(p1, 10);

	p2 = talloc_realloc(root, NULL, char, 20);
	CHECK_SIZE(p1, 10);
	CHECK_SIZE(root, 30);

	if (talloc_steal(p1, NULL) != NULL) {
		printf("failed: stealing NULL should give NULL\n");
		return false;
	}

	if (talloc_steal(p1, p1) != p1) {
		printf("failed: stealing to ourselves is a nop\n");
		return false;
	}
	CHECK_BLOCKS(root, 3);
	CHECK_SIZE(root, 30);

	talloc_steal(NULL, p1);
	talloc_steal(NULL, p2);
	CHECK_BLOCKS(root, 1);
	CHECK_SIZE(root, 0);

	talloc_free(p1);
	talloc_steal(root, p2);
	CHECK_BLOCKS(root, 2);
	CHECK_SIZE(root, 20);
	
	talloc_free(p2);

	CHECK_BLOCKS(root, 1);
	CHECK_SIZE(root, 0);

	talloc_free(root);

	p1 = talloc_size(NULL, 3);
	talloc_report_full(NULL, stdout);
	CHECK_SIZE(NULL, 3);
	talloc_free(p1);

	return true;
}

/*
  test move
*/
static bool test_move(void)
{
	void *root;
	struct t_move {
		char *p;
		int *x;
	} *t1, *t2;
	printf("TESTING MOVE\n");

	root = talloc_new(NULL);

	t1 = talloc(root, struct t_move);
	t2 = talloc(root, struct t_move);
	t1->p = talloc_strdup(t1, "foo");
	t1->x = talloc(t1, int);
	*t1->x = 42;

	t2->p = talloc_move(t2, &t1->p);
	t2->x = talloc_move(t2, &t1->x);
	if (t1->p != NULL || t1->x != NULL ||
	    strcmp(t2->p, "foo") ||
	    *t2->x != 42) {
		printf("talloc move failed\n");
		return false;
	}

	talloc_free(root);

	return true;
}

/*
  test talloc_realloc_fn
*/
static bool test_realloc_fn(void)
{
	void *root, *p1;

	printf("TESTING talloc_realloc_fn\n");

	root = talloc_new(NULL);

	p1 = talloc_realloc_fn(root, NULL, 10);
	CHECK_BLOCKS(root, 2);
	CHECK_SIZE(root, 10);
	p1 = talloc_realloc_fn(root, p1, 20);
	CHECK_BLOCKS(root, 2);
	CHECK_SIZE(root, 20);
	p1 = talloc_realloc_fn(root, p1, 0);
	CHECK_BLOCKS(root, 1);
	CHECK_SIZE(root, 0);

	talloc_free(root);


	return true;
}


static bool test_unref_reparent(void)
{
	void *root, *p1, *p2, *c1;

	printf("TESTING UNREFERENCE AFTER PARENT FREED\n");

	root = talloc_named_const(NULL, 0, "root");
	p1 = talloc_named_const(root, 1, "orig parent");
	p2 = talloc_named_const(root, 1, "parent by reference");

	c1 = talloc_named_const(p1, 1, "child");
	talloc_reference(p2, c1);

	CHECK_PARENT(c1, p1);

	talloc_free(p1);

	CHECK_PARENT(c1, p2);

	talloc_unlink(p2, c1);

	CHECK_SIZE(root, 1);

	talloc_free(p2);
	talloc_free(root);

	return true;
}

/*
  measure the speed of talloc versus malloc
*/
static bool test_speed(void)
{
	void *ctx = talloc_new(NULL);
	unsigned count;
	struct timeval tv;

	printf("MEASURING TALLOC VS MALLOC SPEED\n");

	tv = timeval_current();
	count = 0;
	do {
		void *p1, *p2, *p3;
		p1 = talloc_size(ctx, count);
		p2 = talloc_strdup(p1, "foo bar");
		p3 = talloc_size(p1, 300);
		talloc_free(p1);
		count += 3;
	} while (timeval_elapsed(&tv) < 5.0);

	printf("talloc: %.0f ops/sec\n", count/timeval_elapsed(&tv));

	talloc_free(ctx);

	tv = timeval_current();
	count = 0;
	do {
		void *p1, *p2, *p3;
		p1 = malloc(count);
		p2 = strdup("foo bar");
		p3 = malloc(300);
		free(p1);
		free(p2);
		free(p3);
		count += 3;
	} while (timeval_elapsed(&tv) < 5.0);

	printf("malloc: %.0f ops/sec\n", count/timeval_elapsed(&tv));

	return true;	
}


static bool test_lifeless(void)
{
	void *top = talloc_new(NULL);
	char *parent, *child; 
	void *child_owner = talloc_new(NULL);

	printf("TESTING TALLOC_UNLINK LOOP\n");

	parent = talloc_strdup(top, "parent");
	child = talloc_strdup(parent, "child");  
	(void)talloc_reference(child, parent);
	(void)talloc_reference(child_owner, child); 
	talloc_report_full(top, stdout);
	talloc_unlink(top, parent);
	talloc_free(child);
	talloc_report_full(top, stdout);
	talloc_free(top);
	talloc_free(child_owner);
	talloc_free(child);

	return true;
}

static int loop_destructor_count;

static int test_loop_destructor(char *ptr)
{
	printf("loop destructor\n");
	loop_destructor_count++;
	return 0;
}

static bool test_loop(void)
{
	void *top = talloc_new(NULL);
	char *parent;
	struct req1 {
		char *req2, *req3;
	} *req1;

	printf("TESTING TALLOC LOOP DESTRUCTION\n");
	parent = talloc_strdup(top, "parent");
	req1 = talloc(parent, struct req1);
	req1->req2 = talloc_strdup(req1, "req2");  
	talloc_set_destructor(req1->req2, test_loop_destructor);
	req1->req3 = talloc_strdup(req1, "req3");
	(void)talloc_reference(req1->req3, req1);
	talloc_report_full(top, stdout);
	talloc_free(parent);
	talloc_report_full(top, stdout);
	talloc_report_full(NULL, stdout);
	talloc_free(top);

	if (loop_destructor_count != 1) {
		printf("FAILED TO FIRE LOOP DESTRUCTOR\n");
		return false;
	}
	loop_destructor_count = 0;

	return true;
}

static int fail_destructor_str(char *ptr)
{
	return -1;
}

static bool test_free_parent_deny_child(void)
{
	void *top = talloc_new(NULL);
	char *level1;
	char *level2;
	char *level3;

	printf("TESTING TALLOC FREE PARENT DENY CHILD\n");
	level1 = talloc_strdup(top, "level1");
	level2 = talloc_strdup(level1, "level2");
	level3 = talloc_strdup(level2, "level3");

	talloc_set_destructor(level3, fail_destructor_str);
	talloc_free(level1);
	talloc_set_destructor(level3, NULL);

	CHECK_PARENT(level3, top);

	talloc_free(top);

	return true;
}

static bool test_talloc_ptrtype(void)
{
	bool ret = true;
	void *top = talloc_new(NULL);
	struct struct1 {
		int foo;
		int bar;
	} *s1, *s2, **s3, ***s4;
	const char *location1;
	const char *location2;
	const char *location3;
	const char *location4;

	printf("TESTING TALLOC PTRTYPE\n");
	s1 = talloc_ptrtype(top, s1);location1 = __location__;

	if (talloc_get_size(s1) != sizeof(struct struct1)) {
		printf("%s: talloc_ptrtype() allocated the wrong size %lu "
		       "(should be %lu)\n",
			__location__, (unsigned long)talloc_get_size(s1),
		       (unsigned long)sizeof(struct struct1));
		ret = false;
	}

	if (strcmp(location1, talloc_get_name(s1)) != 0) {
		printf("%s: talloc_ptrtype() sets the wrong name '%s' (should be '%s')\n",
			__location__, talloc_get_name(s1), location1);
		ret = false;
	}

	s2 = talloc_array_ptrtype(top, s2, 10);location2 = __location__;

	if (talloc_get_size(s2) != (sizeof(struct struct1) * 10)) {
		printf("%s: talloc_array_ptrtype() allocated the wrong size "
		       "%lu (should be %lu)\n",
			__location__, (unsigned long)talloc_get_size(s2),
		       (unsigned long)(sizeof(struct struct1)*10));
		ret = false;
	}

	if (strcmp(location2, talloc_get_name(s2)) != 0) {
		printf("%s: talloc_array_ptrtype() sets the wrong name '%s' (should be '%s')\n",
			__location__, talloc_get_name(s2),
		       location2);
		ret = false;
	}

	s3 = talloc_array_ptrtype(top, s3, 10);location3 = __location__;

	if (talloc_get_size(s3) != (sizeof(struct struct1 *) * 10)) {
		printf("%s: talloc_array_ptrtype() allocated the wrong size "
		       "%lu (should be %lu)\n",
			__location__, (unsigned long)talloc_get_size(s3),
		       (unsigned long)(sizeof(struct struct1 *)*10));
		ret = false;
	}

	if (strcmp(location3, talloc_get_name(s3)) != 0) {
		printf("%s: talloc_array_ptrtype() sets the wrong name '%s' (should be '%s')\n",
			__location__, talloc_get_name(s3), location3);
		ret = false;
	}

	s4 = talloc_array_ptrtype(top, s4, 10);location4 = __location__;

	if (talloc_get_size(s4) != (sizeof(struct struct1 **) * 10)) {
		printf("%s: talloc_array_ptrtype() allocated the wrong size "
		       "%lu (should be %lu)\n",
			__location__, (unsigned long)talloc_get_size(s4),
		       (unsigned long)(sizeof(struct struct1 **)*10));
		ret = false;
	}

	if (strcmp(location4, talloc_get_name(s4)) != 0) {
		printf("%s: talloc_array_ptrtype() sets the wrong name '%s' (should be '%s')\n",
			__location__, talloc_get_name(s4), location4);
		ret = false;
	}

	talloc_free(top);

	return ret;
}

static bool test_autofree(void)
{
#if _SAMBA_BUILD_>=4
	/* 
	 * we can't run this inside smbtorture in samba4
	 * as smbtorture uses talloc_autofree_context()
	 */
	printf("SKIPPING TALLOC AUTOFREE CONTEXT (not supported from smbtorture)\n");
#else
	void *p;

	printf("TESTING TALLOC AUTOFREE CONTEXT\n");

	p = talloc_autofree_context();
	talloc_free(p);

	p = talloc_autofree_context();
	talloc_free(p);
#endif
	return true;
}

bool torture_local_talloc(struct torture_context *torture) 
{
	bool ret = true;

	talloc_disable_null_tracking();
	talloc_enable_null_tracking();

	ret &= test_ref1();
	ret &= test_ref2();
	ret &= test_ref3();
	ret &= test_ref4();
	ret &= test_unlink1();
	ret &= test_misc();
	ret &= test_realloc();
	ret &= test_realloc_child();
	ret &= test_steal();
	ret &= test_move();
	ret &= test_unref_reparent();
	ret &= test_realloc_fn();
	ret &= test_type();
	ret &= test_lifeless();
	ret &= test_loop();
	ret &= test_free_parent_deny_child();
	ret &= test_talloc_ptrtype();
	if (ret) {
		ret &= test_speed();
	}
	ret &= test_autofree();

	return ret;
}



#if _SAMBA_BUILD_<4
 int main(void)
{
	if (!torture_local_talloc(NULL)) {
		printf("ERROR: TESTSUITE FAILED\n");
		return -1;
	}
	return 0;
}
#endif
