# Samba Build System
# - create output for Makefile
#
#  Copyright (C) Stefan (metze) Metzmacher 2004
#  Copyright (C) Jelmer Vernooij 2005
#  Released under the GNU GPL

package smb_build::makefile;
use smb_build::env;
use strict;

use base 'smb_build::env';

sub new($$$)
{
	my ($myname, $config, $mkfile) = @_;
	my $self = new smb_build::env($config);
	
	bless($self, $myname);

	$self->{manpages} = [];
	$self->{sbin_progs} = [];
	$self->{bin_progs} = [];
	$self->{static_libs} = [];
	$self->{shared_libs} = [];
	$self->{headers} = [];
	$self->{pc_files} = [];
	$self->{proto_headers} = [];
	$self->{output} = "";

	$self->{mkfile} = $mkfile;

	$self->output("#!gmake\n");
	$self->output("################################################\n");
	$self->output("# Autogenerated by build/smb_build/makefile.pm #\n");
	$self->output("################################################\n");
	$self->output("\n");

	$self->output("default: all\n\n");

	$self->_prepare_path_vars();
	$self->_prepare_compiler_linker();
	$self->output(".SUFFIXES: .x .c .et .y .l .d .o .h .h.gch .a .so .1 .1.xml .3 .3.xml .5 .5.xml .7 .7.xml .8 .8.xml .ho\n");
	$self->_prepare_hostcc_rule();
	$self->_prepare_std_CC_rule("c","o",'$(PICFLAG)',"Compiling","Rule for std objectfiles");
	$self->_prepare_std_CC_rule("h","h.gch",'$(PICFLAG)',"Precompiling","Rule for precompiled headerfiles");

	return $self;
}

sub output($$)
{
	my ($self, $text) = @_;

	$self->{output} .= $text;
}

sub _prepare_path_vars($)
{
	my ($self) = @_;

	$self->output(<< "__EOD__"
prefix = $self->{config}->{prefix}
exec_prefix = $self->{config}->{exec_prefix}
selftest_prefix = $self->{config}->{selftest_prefix}
VPATH = $self->{config}->{srcdir}
srcdir = $self->{config}->{srcdir}
builddir = $self->{config}->{builddir}

BASEDIR = $self->{config}->{prefix}
BINDIR = $self->{config}->{bindir}
SBINDIR = $self->{config}->{sbindir}
datadir = $self->{config}->{datadir}
LIBDIR = $self->{config}->{libdir}
MODULESDIR = $self->{config}->{libdir}
INCLUDEDIR = $self->{config}->{includedir}
CONFIGDIR = $self->{config}->{configdir}
localstatedir = $self->{config}->{localstatedir}
SWATDIR = $self->{config}->{swatdir}
VARDIR = $self->{config}->{localstatedir}
LOGFILEBASE = $self->{config}->{logfilebase}
NCALRPCDIR = $self->{config}->{localstatedir}/ncalrpc
LOCKDIR = $self->{config}->{lockdir}
PIDDIR = $self->{config}->{piddir}
MANDIR = $self->{config}->{mandir}
PRIVATEDIR = $self->{config}->{privatedir}

__EOD__
);
}

sub _prepare_compiler_linker($)
{
	my ($self) = @_;

	my $devld_local = "";
	my $devld_install = "";

	$self->{duplicate_build} = 0;
	if ($self->{config}->{LIBRARY_OUTPUT_TYPE} eq "SHARED_LIBRARY") {
		if ($self->{developer}) {
			$self->{duplicate_build} = 1;
			$devld_local = " -Wl,-rpath,\$(builddir)/bin";
		}
		$devld_install = " -Wl,-rpath-link,\$(builddir)/bin";
	}

	$self->output(<< "__EOD__"
SHELL=$self->{config}->{SHELL}

PERL=$self->{config}->{PERL}

CPP=$self->{config}->{CPP}
CPPFLAGS=$self->{config}->{CPPFLAGS}

CC=$self->{config}->{CC}
CFLAGS=-I\$(srcdir)/include -I\$(srcdir) -I\$(srcdir)/lib -D_SAMBA_BUILD_ -DHAVE_CONFIG_H $self->{config}->{CFLAGS} \$(CPPFLAGS)
PICFLAG=$self->{config}->{PICFLAG}
HOSTCC=$self->{config}->{HOSTCC}

LD=$self->{config}->{LD} 
LDFLAGS=$self->{config}->{LDFLAGS} -L\$(builddir)/bin
LOCAL_LINK_FLAGS=$devld_local
INSTALL_LINK_FLAGS=$devld_install

STLD=$self->{config}->{AR}
STLD_FLAGS=-rc -L\$(builddir)/bin

SHLD=$self->{config}->{CC}
SHLD_FLAGS=$self->{config}->{LDSHFLAGS} -L\$(builddir)/bin
SONAMEFLAG=$self->{config}->{SONAMEFLAG}
SHLIBEXT=$self->{config}->{SHLIBEXT}

XSLTPROC=$self->{config}->{XSLTPROC}

LEX=$self->{config}->{LEX}
YACC=$self->{config}->{YACC}
YAPP=$self->{config}->{YAPP}
PIDL_ARGS=$self->{config}->{PIDL_ARGS}

GCOV=$self->{config}->{GCOV}

DEFAULT_TEST_TARGET=$self->{config}->{DEFAULT_TEST_TARGET}

__EOD__
);
}

sub _prepare_mk_files($)
{
	my $self = shift;
	my @tmp = ();

	
	foreach (@smb_build::config_mk::parsed_files) {
		s/ .*$//g;
		push (@tmp, $_);
	}

	$self->output("MK_FILES = " . array2oneperline(\@tmp) . "\n");
}

sub _prepare_dummy_MAKEDIR($)
{
	my ($self) = @_;

	$self->output(<< '__EOD__'
dynconfig.o: dynconfig.c Makefile
	@echo Compiling $*.c
	@$(CC) $(CFLAGS) $(PICFLAG) $(PATH_FLAGS) -c $< -o $@
__EOD__
);
	if ($self->{config}->{BROKEN_CC} eq "yes") {
		$self->output('	-mv `echo $@ | sed \'s%^.*/%%g\'` $@
');
	}
	$self->output("\n");
}

sub _prepare_std_CC_rule($$$$$$)
{
	my ($self,$src,$dst,$flags,$message,$comment) = @_;

	$self->output(<< "__EOD__"
# $comment
.$src.$dst:
	\@echo $message \$\*.$src
	\@\$(CC) `script/cflags.sh \$\@` \$(CFLAGS) $flags -c \$\*.$src -o \$\@
__EOD__
);
	if ($self->{config}->{BROKEN_CC} eq "yes") {
		$self->output('	-mv `echo $@ | sed \'s%^.*/%%g\'` $@
');
	}

	$self->output("\n");
}

sub _prepare_hostcc_rule($)
{
	my ($self) = @_;
	
	$self->output(<< "__EOD__"
.c.ho:
	\@echo Compiling \$\*.c with host compiler
	\@\$(HOSTCC) `script/cflags.sh \$\@` \$(CFLAGS) -c \$\*.c -o \$\@
__EOD__
);
	if ($self->{config}->{BROKEN_CC} eq "yes") {
		$self->output('	-mv `echo $@ | sed \'s%^.*/%%g\' -e \'s%\.ho$$%.o%\'` $@
');
	}

	$self->output("\n");
}

sub array2oneperline($)
{
	my $array = shift;
	my $output = "";

	foreach (@$array) {
		next unless defined($_);

		$output .= " \\\n\t\t$_";
	}

	return $output;
}

sub _prepare_list($$$)
{
	my ($self,$ctx,$var) = @_;

	my $tmplist = array2oneperline($ctx->{$var});
	return if ($tmplist eq "");

	$self->output("$ctx->{TYPE}\_$ctx->{NAME}_$var =$tmplist\n");
}

sub SharedLibrary($$)
{
	my ($self,$ctx) = @_;

	push (@{$self->{shared_libs}}, "bin/$ctx->{LIBRARY_REALNAME}");
	push (@{$self->{shared_modules}}, "bin/$ctx->{LIBRARY_REALNAME}");

	$self->_prepare_list($ctx, "OBJ_LIST");
	$self->_prepare_list($ctx, "CFLAGS");
	$self->_prepare_list($ctx, "DEPEND_LIST");
	$self->_prepare_list($ctx, "LINK_LIST");
	$self->_prepare_list($ctx, "LINK_FLAGS");

	push(@{$self->{all_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)");
		
	if ($ctx->{NOPROTO} eq "NO") {
		push(@{$self->{proto_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST\)");
	}

	$self->output(<< "__EOD__"
#

bin/$ctx->{LIBRARY_REALNAME}: \$($ctx->{TYPE}_$ctx->{NAME}_DEPEND_LIST) \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST) bin/.dummy
	\@echo Linking \$\@
	\@\$(SHLD) \$(SHLD_FLAGS) -o \$\@ \\
		\$($ctx->{TYPE}_$ctx->{NAME}_LINK_FLAGS) \\
		\$($ctx->{TYPE}_$ctx->{NAME}_LINK_LIST)

__EOD__
);
	if (defined($ctx->{LIBRARY_SONAME})) {
	    $self->output(<< "__EOD__"
# Symlink $ctx->{LIBRARY_SONAME}
bin/$ctx->{LIBRARY_SONAME}: bin/$ctx->{LIBRARY_REALNAME} bin/.dummy
	\@echo Symlink \$\@
	\@ln -sf $ctx->{LIBRARY_REALNAME} \$\@
# Symlink $ctx->{LIBRARY_NAME}
bin/$ctx->{LIBRARY_NAME}: bin/$ctx->{LIBRARY_SONAME} bin/.dummy
	\@echo Symlink \$\@
	\@ln -sf $ctx->{LIBRARY_SONAME} \$\@

__EOD__
);
	}
}

sub MergedObj($$)
{
	my ($self,$ctx) = @_;

	return unless $ctx->{TARGET};

	$self->_prepare_list($ctx, "OBJ_LIST");
	$self->_prepare_list($ctx, "CFLAGS");
	$self->_prepare_list($ctx, "DEPEND_LIST");

	push(@{$self->{all_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)");
		
	if ($ctx->{NOPROTO} eq "NO") {
		push(@{$self->{proto_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST\)");
	}

	$self->output("$ctx->{TARGET}: \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)\n");

	$self->output("\t\@echo \"Pre-Linking $ctx->{TYPE} $ctx->{NAME}\"\n");
	$self->output("\t@\$(LD) -r \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST) -o $ctx->{TARGET}\n");
	$self->output("\n");
}

sub ObjList($$)
{
	my ($self,$ctx) = @_;

	return unless $ctx->{TARGET};

	push(@{$self->{all_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)");
		
	if ($ctx->{NOPROTO} eq "NO") {
		push(@{$self->{proto_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST\)");
	}

	$self->_prepare_list($ctx, "OBJ_LIST");
	$self->_prepare_list($ctx, "CFLAGS");
	$self->_prepare_list($ctx, "DEPEND_LIST");
	$self->output("$ctx->{TARGET}: ");
	$self->output("\$($ctx->{TYPE}_$ctx->{NAME}_DEPEND_LIST) \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)\n");
	$self->output("\t\@touch $ctx->{TARGET}\n");
}

sub StaticLibrary($$)
{
	my ($self,$ctx) = @_;

	push (@{$self->{static_libs}}, $ctx->{OUTPUT});

	$self->_prepare_list($ctx, "OBJ_LIST");
	$self->_prepare_list($ctx, "CFLAGS");

	$self->_prepare_list($ctx, "DEPEND_LIST");
	$self->_prepare_list($ctx, "LINK_LIST");
	$self->_prepare_list($ctx, "LINK_FLAGS");

	push(@{$self->{all_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)");
		
	if ($ctx->{NOPROTO} eq "NO") {
		push(@{$self->{proto_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST\)");
	}

	$self->output(<< "__EOD__"
#
$ctx->{TARGET}: \$($ctx->{TYPE}_$ctx->{NAME}_DEPEND_LIST) \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST) bin/.dummy
	\@echo Linking \$@
	\@\$(STLD) \$(STLD_FLAGS) \$@ \\
		\$($ctx->{TYPE}_$ctx->{NAME}_LINK_LIST)

__EOD__
);
}

sub Header($$)
{
	my ($self,$ctx) = @_;

	foreach (@{$ctx->{PUBLIC_HEADERS}}) {
		push (@{$self->{headers}}, "$ctx->{BASEDIR}/$_");
	}
}

sub Binary($$)
{
	my ($self,$ctx) = @_;

	my $installdir;
	
	if ($self->{duplicate_build}) {
		$installdir = "bin/install";
	} else {
		$installdir = "bin";
	}

	push(@{$self->{all_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)");
		
	if ($ctx->{NOPROTO} eq "NO") {
		push(@{$self->{proto_objs}}, "\$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST\)");
	}

	unless (defined($ctx->{INSTALLDIR})) {
	} elsif ($ctx->{INSTALLDIR} eq "SBINDIR") {
		push (@{$self->{sbin_progs}}, "$installdir/$ctx->{BINARY}");
	} elsif ($ctx->{INSTALLDIR} eq "BINDIR") {
		push (@{$self->{bin_progs}}, "$installdir/$ctx->{BINARY}");
	}

	push (@{$self->{binaries}}, "bin/$ctx->{BINARY}");

	$self->_prepare_list($ctx, "OBJ_LIST");
	$self->_prepare_list($ctx, "CFLAGS");
	$self->_prepare_list($ctx, "DEPEND_LIST");
	$self->_prepare_list($ctx, "LINK_LIST");
	$self->_prepare_list($ctx, "LINK_FLAGS");

	if ($self->{duplicate_build}) {
	$self->output(<< "__EOD__"
#
bin/$ctx->{BINARY}: bin/.dummy \$($ctx->{TYPE}_$ctx->{NAME}_DEPEND_LIST) \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)
	\@echo Linking \$\@
	\@\$(CC) \$(LDFLAGS) -o \$\@ \$(LOCAL_LINK_FLAGS) \\
		\$\($ctx->{TYPE}_$ctx->{NAME}_LINK_LIST) \\
		\$\($ctx->{TYPE}_$ctx->{NAME}_LINK_FLAGS)

__EOD__
);
	}

$self->output(<< "__EOD__"
$installdir/$ctx->{BINARY}: \$($ctx->{TYPE}_$ctx->{NAME}_DEPEND_LIST) \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)
	\@echo Linking \$\@
	\@\$(CC) \$(LDFLAGS) -o \$\@ \$(INSTALL_LINK_FLAGS) \\
		\$\($ctx->{TYPE}_$ctx->{NAME}_LINK_LIST) \\
		\$\($ctx->{TYPE}_$ctx->{NAME}_LINK_FLAGS) 

__EOD__
);
}

sub Manpage($$)
{
	my ($self,$ctx) = @_;

	my $dir = $ctx->{BASEDIR};
	
	$dir =~ s/^\.\///g;

	push (@{$self->{manpages}}, "$dir/$ctx->{MANPAGE}");
}

sub PkgConfig($$)
{
	my ($self,$ctx) = @_;
	
	my $link_name = $ctx->{NAME};

	$link_name =~ s/^LIB//g;
	$link_name = lc($link_name);

	if (not defined($ctx->{DESCRIPTION})) {
		warn("$ctx->{NAME} has not DESCRIPTION set, not generating .pc file");
		return;
	}

	my $path = "$ctx->{BASEDIR}/$link_name.pc";

	push (@{$self->{pc_files}}, $path);

	smb_build::env::PkgConfig($self,
		$path,
		$link_name,
		$ctx->{OUTPUT},
		join(' ', @{$ctx->{CFLAGS}}), 
		"$ctx->{MAJOR_VERSION}.$ctx->{MINOR_VERSION}.$ctx->{RELEASE_VERSION}",
		$ctx->{DESCRIPTION}
	); 
}

sub ProtoHeader($$)
{
	my ($self,$ctx) = @_;

	my $dir = $ctx->{BASEDIR};

	$dir =~ s/^\.\///g;

	my $comment = "";
	if (defined($ctx->{PUBLIC_PROTO_HEADER})) {
		$comment.= " and $dir/$ctx->{PUBLIC_PROTO_HEADER}";
		push (@{$self->{proto_headers}}, "$dir/$ctx->{PUBLIC_PROTO_HEADER}");
	} else {
		$ctx->{PUBLIC_PROTO_HEADER} = $ctx->{PRIVATE_PROTO_HEADER};
	}	
	push (@{$self->{proto_headers}}, "$dir/$ctx->{PRIVATE_PROTO_HEADER}");

	$self->output("$dir/$ctx->{PUBLIC_PROTO_HEADER}: \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST:.o=.c)\n");
	$self->output("\t\@echo \"Creating $dir/$ctx->{PRIVATE_PROTO_HEADER}$comment\"\n");

	$self->output("\t\@\$(PERL) \$(srcdir)/script/mkproto.pl --private=$dir/$ctx->{PRIVATE_PROTO_HEADER} --public=$dir/$ctx->{PUBLIC_PROTO_HEADER} \$($ctx->{TYPE}_$ctx->{NAME}_OBJ_LIST)\n\n");
}

sub write($$)
{
	my ($self,$file) = @_;

	$self->output("MANPAGES = ".array2oneperline($self->{manpages})."\n");
	$self->output("BIN_PROGS = " . array2oneperline($self->{bin_progs}) . "\n");
	$self->output("SBIN_PROGS = " . array2oneperline($self->{sbin_progs}) . "\n");
	$self->output("BINARIES = " . array2oneperline($self->{binaries}) . "\n");
	$self->output("STATIC_LIBS = " . array2oneperline($self->{static_libs}) . "\n");
	$self->output("SHARED_LIBS = " . array2oneperline($self->{shared_libs}) . "\n");
	$self->output("PUBLIC_HEADERS = " . array2oneperline($self->{headers}) . "\n");
	$self->output("PC_FILES = " . array2oneperline($self->{pc_files}) . "\n");
	$self->output("ALL_OBJS = " . array2oneperline($self->{all_objs}) . "\n");
	$self->output("PROTO_OBJS = " . array2oneperline($self->{proto_objs}) .  "\n");
	$self->output("PROTO_HEADERS = " . array2oneperline($self->{proto_headers}) .  "\n");
	$self->output("SHARED_MODULES = " . array2oneperline($self->{shared_modules}) . "\n");

	$self->_prepare_mk_files();

	if ($self->{developer}) {
		$self->output(<<__EOD__

#-include \$(ALL_OBJS:.o=.d)

__EOD__
);
	}

	$self->_prepare_dummy_MAKEDIR();

	$self->output($self->{mkfile});

	open(MAKEFILE,">$file") || die ("Can't open $file\n");
	print MAKEFILE $self->{output};
	close(MAKEFILE);

	print __FILE__.": creating $file\n";
}

1;
