/* server.c -- pam_smb IPC server code 
   
   Copyright (c) Dave Airlie 2000
   airlied@samba.org

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

#include "config.h"
#include "constants.h"
#include "pam_smb_queue.h"
#include "cache.h"

#include "tdb.h"

#include <fcntl.h>
#include <time.h>
#include <sys/time.h>

#include "pamsmbd.h"

void sighup_handler(int signum);
void sigusr2_handler(int signum);
void sigterm_handler(int signum);

char map_file[255];
char cache_db_file[255];
char unix2nt_db_file[255];
char nt2unix_db_file[255];

int mid;
int debug=1;

TDB_CONTEXT *cache_db;
int cur_cache_size=0;

domain_list smb_domlist;
char *defaultdomain;
int use_map=0;

int main(void)
{
  int key, res;
  struct sigaction sighup_act, sigusr2_act, sigterm_act, sigalrm_act;
  struct itimerval cachetimer;
  
  openlog("pamsmbd", LOG_PID, LOG_AUTHPRIV);

  /* set up signal handlers */
  /* SIGHUP re-read configuration file */
  /* SIGUSR2 dumps cache to /tmp */ 
  /* SIGTERM close msgqueue and ends program */

  sigalrm_act.sa_handler=sigalrm_handler;
  sigalrm_act.sa_flags=0;
  sigaction(SIGALRM, &sigalrm_act, NULL);
  sighup_act.sa_handler=sighup_handler;
  sighup_act.sa_flags=0;
  sigaction(SIGHUP, &sighup_act, NULL);
  sigusr2_act.sa_handler=sigusr2_handler;
  sigusr2_act.sa_flags=0;
  sigaction(SIGUSR2, &sigusr2_act, NULL);
  sigterm_act.sa_handler=sigterm_handler;
  sigterm_act.sa_flags=0;
  sigaction(SIGTERM, &sigterm_act, NULL);
  sigaction(SIGINT, &sigterm_act, NULL);
  sigaction(SIGQUIT, &sigterm_act, NULL);

  switch(fork())
	 {
	case 0:
		break;
	case -1:
		printf("error in fork\n");
		exit(-1);
	default:
		exit(0);
	}

  setsid();

  cachetimer.it_interval.tv_sec=CACHE_CHECK;
  cachetimer.it_interval.tv_usec=0;
  cachetimer.it_value.tv_sec=CACHE_CHECK;
  cachetimer.it_value.tv_usec=0;
  
  setitimer(ITIMER_REAL, &cachetimer, NULL);
  
  strncpy(map_file, DEFAULT_MAP_FILE, 255);
  strncpy(cache_db_file, DB_DIR, 255);
  strncat(cache_db_file, DEFAULT_CACHE_DB, 255);
  strncpy(nt2unix_db_file, DB_DIR, 255);
  strncat(nt2unix_db_file, DEFAULT_NT2UNIX_DB, 255);
  strncpy(unix2nt_db_file, DB_DIR, 255);
  strncat(unix2nt_db_file, DEFAULT_UNIX2NT_DB, 255);

  /* Read configuration */
  smb_readpamconf(&smb_domlist);

  defaultdomain=smb_domlist.controllers[0].domain;

  map_user_initdb();
  res=map_user_read_map_file(map_file);
  if (res==0)
    {
      if (debug)
	syslog(LOG_AUTHPRIV | LOG_DEBUG, "Using map file %s\n", map_file);
      use_map=1;
    }
  serv_initcachedb();

  /* Setup the SYSV IPC msg queue */
  mid=setup_queue();

  /* Start reading the queue */
  do_queue(mid);

  closelog();
  return 0;
}

/*
 * get entry in domain list and fill out servers 
 */
int get_domain_servers(usercache_data *entry)
{
  int x;
  
  for (x=0; x<smb_domlist.numdomains; x++)
    {
      if(strcasecmp(entry->nt_domain, smb_domlist.controllers[x].domain)==0)
	break;
    }

  if (x==smb_domlist.numdomains)
    return -1;
  
  if (smb_domlist.controllers[x].numservers==0)
    return -1;
  if (smb_domlist.controllers[x].numservers>=1)
    strcpy(entry->pdc_nbname, smb_domlist.controllers[x].servers[0].sername);
  if (smb_domlist.controllers[x].numservers>=2)
    strcpy(entry->bdc_nbname, smb_domlist.controllers[x].servers[1].sername);

  return 0;
}

/*
 * setup_queue()
 *
 * Set up the SYSV message queue 
 */
int setup_queue(void)
{
  int mid;
  int key=PAMSMBKEY;
  
  /* if queue exists remove it and re-create it */
  if((mid=msgget(key, 0)) != -1)
    {
      msgctl(mid, IPC_RMID, NULL);
    }
#ifdef PAM_SMB_ROOT_ONLY  
  mid=msgget(key, IPC_CREAT | 0600);
#else
  mid=msgget(key, IPC_CREAT | 0666);
#endif
  if (mid==-1)
    {
      perror("msgget: ");
      exit(-1);
    }

  return mid;
}

/*
 * do_queue()
 *
 * Process a message in the SYSV message queue and return a reply
 * function to get queue entries and validate them 
 *
 */
int do_queue(int mid)
{
  int n;
  int ok=0, res;
  MESSAGE msg;
  int user_num;

  int cache_status;
  int valid=0;
  int position;
  
  usercache_data internal_entry, *newentry=&internal_entry;

  while(1)
    {
      ok=0;
      do {
	n=msgrcv(mid, (struct msgbuf *)&msg, sizeof(msg), SERVER, 0);
	if (n==-1)
	  {
	    if (errno!=EINTR)
	      {		
		perror("Server: msgrcv");
		msgctl(mid, IPC_RMID, NULL);
		exit(-1);
	      }
	  }
	else
	  ok=1;
      } while (ok==0);
      
      memset(newentry,0, sizeof(usercache_data));

      strncpy(newentry->nt_user, msg.username, MAX_NT_LEN);
      strncpy(newentry->ux_user, msg.username, MAX_UU_LEN);
      strncpy(newentry->password, msg.password, MAX_PASS_LEN);

      /* copy in the domain from the remote end or default if none passed */
      if (strlen(msg.domain))
	strncpy(newentry->nt_domain, msg.domain, MAX_NTDOM_LEN);
      else
	strncpy(newentry->nt_domain, defaultdomain, MAX_NTDOM_LEN);

      if (use_map)
	map_user(newentry);

      /* Check cache for user code */
      /* Cache will *always* be checked on UNIX username */
      user_num = cur_cache_size;
      position=0;
      ok=0;
    
      cache_status=check_entry_in_cache(newentry);

      if (cache_status!=CACHE_OK)
	{
	  
	  /* Need to dig entry for domain out of conf file list */
	  res=get_domain_servers(newentry);

	  if (debug) 
	    syslog(LOG_AUTHPRIV | LOG_DEBUG, "server: remote auth user %s %s %s %s %s\n", newentry->ux_user, newentry->nt_user, newentry->nt_domain, newentry->pdc_nbname, newentry->bdc_nbname);

	  /* this is the actual validation routine call here <------------*/
	  if (res==0)
	    valid=Valid_MapUser(newentry);
	  else 
	    {
	      syslog(LOG_AUTHPRIV | LOG_ERR, "no or incorrect domain record for domain %s in pam_smb configuration", newentry->nt_domain);
	      valid=1;
	    }

	  if (valid==0)
	    {
	      add_entry_to_cache(newentry);
	    }
	}
      else
	valid=0;
      
      msg.return_code = valid;

      msg.msg_to=msg.msg_fm;
      msg.msg_fm=SERVER;

      strncpy(msg.username, newentry->ux_user, MAX_UU_LEN);
      memset(msg.password, 0, MAX_PASS_LEN);
      memset(msg.domain, 0, MAX_NTDOM_LEN);
      n=msgsnd(mid, (struct msgbuf *)&msg, sizeof(msg), 0);
      if (n==-1) {
	perror("server : msgsnd");
	exit(-1);
      }
    }
}

/*
 * sighup_handler()
 *
 * Handler for SIGHUP signal, re-reads config file 
 *
 */

void sighup_handler(int signum)
{
  smb_readpamconf(&smb_domlist);
  syslog(LOG_AUTHPRIV | LOG_NOTICE, "Got SIGHUP : reloading conf file\n");
  return;
}

/*
 * sigusr2_handler()
 *
 * Handler for SIGUSR2 signal, dumps num users in cache
 *
 */
void sigusr2_handler(int signum)
{
  
  syslog(LOG_AUTHPRIV | LOG_NOTICE, "Got SIGUSR2: number of users in cache is %d",cur_cache_size);
  
  return;
}


/*
 * sigterm_handler() 
 *
 * Handler for SIGTERM signal, cleans up before exit
 *
 */
void sigterm_handler(int signum)
{
  /* kill the IPC msg queue */
  int n;
  
  n=msgctl(mid ,IPC_RMID, NULL);
  if (n!=0)
    {
      syslog(LOG_AUTHPRIV | LOG_ERR, "Error killing msg queue");
      closelog();
      exit(-1);
    }
  tdb_close(cache_db);
  closelog();
  exit(0);

}



