/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   NBT netbios routines and daemon - version 2
   Copyright (C) Andrew Tridgell 1994-1998
   Copyright (C) Luke Kenneth Casson Leighton 1994-1998
   Copyright (C) Jeremy Allison 1994-1998
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/

#include "includes.h"

extern int DEBUGLEVEL;

extern char **my_netbios_names;
extern pstring myname;
extern fstring myworkgroup;

extern uint16 samba_nb_type; /* Samba's NetBIOS type. */

/****************************************************************************
 Fail funtion when registering my netbios names.
  **************************************************************************/

static void my_name_register_failed(struct subnet_record *subrec,
                              struct response_record *rrec, struct nmb_name *nmbname)
{
  DEBUG(0,("my_name_register_failed: Failed to register my name %s on subnet %s.\n",
            namestr(nmbname), subrec->subnet_name));
}

/****************************************************************************
  Add my workgroup and my given names to the subnet lists.
  Also add the magic Samba names.
  **************************************************************************/

BOOL register_my_workgroup_and_names()
{
  struct subnet_record *subrec;
  struct work_record *work;
  int i;

  for(subrec = FIRST_SUBNET; subrec; subrec = NEXT_SUBNET_INCLUDING_UNICAST(subrec))
  {
    /* Create the workgroup on the subnet. */
    if((work = create_workgroup_on_subnet(subrec, myworkgroup, PERMANENT_TTL)) == NULL)
    {
      DEBUG(0,("register_my_workgroup_and_names: Failed to create my workgroup %s on subnet %s. \
Exiting.\n", myworkgroup, subrec->subnet_name));
      return False;
    }

    /* Each subnet entry, except for the remote_announce_broadcast subnet
       and the wins_server_subnet has the magic Samba names. */
    add_samba_names_to_subnet(subrec);

    /* Register all our names including aliases. */
    for (i=0; my_netbios_names[i]; i++) 
    {
      register_name(subrec, my_netbios_names[i],0x20,samba_nb_type,
                    NULL,
                    my_name_register_failed, NULL);
      register_name(subrec, my_netbios_names[i],0x03,samba_nb_type,
                    NULL,
                    my_name_register_failed, NULL);
      register_name(subrec, my_netbios_names[i],0x00,samba_nb_type,
                    NULL,
                    my_name_register_failed, NULL);
    }

    /* Initiate election processing, register the workgroup names etc. */
    initiate_myworkgroup_startup(subrec, work);
  }

  /* If we are not a WINS client, we still need to add the magic Samba
     names and the netbios names to the unicast subnet directly. This is
     to allow unicast node status requests and queries to still work
     in a broadcast only environment. */

  if(we_are_a_wins_client() == False)
  {
    add_samba_names_to_subnet(unicast_subnet);

    for (i=0; my_netbios_names[i]; i++)
    {
      add_name_to_subnet(unicast_subnet, my_netbios_names[i],0x20,samba_nb_type, PERMANENT_TTL,
                     SELF_NAME, 1, &FIRST_SUBNET->myip);

      add_name_to_subnet(unicast_subnet, my_netbios_names[i],0x3,samba_nb_type, PERMANENT_TTL,
                     SELF_NAME, 1, &FIRST_SUBNET->myip);

      add_name_to_subnet(unicast_subnet, my_netbios_names[i],0x0,samba_nb_type, PERMANENT_TTL,
                     SELF_NAME, 1, &FIRST_SUBNET->myip);
    }
  }

  return True;
}

/****************************************************************************
  Remove all the names we registered.
**************************************************************************/

void release_my_names()
{
  struct subnet_record *subrec;

  for (subrec = FIRST_SUBNET; subrec; subrec = NEXT_SUBNET_INCLUDING_UNICAST(subrec))
  {
    struct name_record *namerec, *nextnamerec;

    for (namerec = subrec->namelist; namerec; namerec = nextnamerec)
    {
      nextnamerec = namerec->next;
      if ((namerec->source == SELF_NAME) && !NAME_IS_DEREGISTERING(namerec))
        release_name(subrec, namerec, standard_success_release,
                     NULL, NULL);
    }
  }
}

/*******************************************************************
  Refresh our registered names.
  ******************************************************************/

void refresh_my_names(time_t t)
{
  struct subnet_record *subrec;

  for (subrec = FIRST_SUBNET; subrec; subrec = NEXT_SUBNET_INCLUDING_UNICAST(subrec))
  {
    struct name_record *namerec;
	  
    for (namerec = subrec->namelist; namerec; namerec = namerec->next)
    {
      /* Each SELF name has an individual time to be refreshed. */
      if ((namerec->source == SELF_NAME) && (namerec->refresh_time < t) && 
          (namerec->death_time != PERMANENT_TTL))
      {
        /* We cheat here and pretend the refresh is going to be
           successful & update the refresh times. This stops
           multiple refresh calls being done. We actually
           deal with refresh failure in the fail_fn.
         */
        refresh_name(subrec, namerec, NULL, NULL, NULL);
	namerec->death_time += lp_max_ttl();
	namerec->refresh_time += lp_max_ttl();
      }
    }
  }
}
