/*
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   30 July 96: David.Chappell@mail.trincoll.edu
   Expanded multiple workgroup domain master browser support.

*/

#include "includes.h"

extern int ServerNMB;
extern int ClientDGRAM;

extern int DEBUGLEVEL;
extern BOOL CanRecurse;

extern int ClientDGRAM;

extern struct in_addr ipgrp;

/* this is our domain/workgroup/server database */
extern struct subnet_record *subnetlist;

extern time_t StartupTime;

extern BOOL updatedlists;



/****************************************************************************
tell a server to become a backup browser
state - 0x01 become backup instead of master
      - 0x02 remove all entries in browse list and become non-master
      - 0x04 stop master browser service altogether. NT ignores this 

**************************************************************************/
void reset_server(struct work_record *work, struct nmb_name *name,
                  int state, struct in_addr ip)
{
  char outbuf[20];
  char *p;

  struct nmb_name src_name;

  bzero(outbuf,sizeof(outbuf));
  p = outbuf;

  CVAL(p,0) = ANN_ResetBrowserState;
  CVAL(p,2) = state; 
  p += 2;

  DEBUG(2,("sending reset to %s %s of state %d\n",
       name,inet_ntoa(ip),state));

  make_nmb_name(&src_name, brse_server_alias(work->token), 0x20, work->scope);

  send_mailslot_reply(BROWSE_MAILSLOT,ClientDGRAM,outbuf,PTR_DIFF(p,outbuf),
				&src_name,*iface_ip(ip),
				name,ip);
}


/****************************************************************************
tell a server to become a backup browser
**************************************************************************/
void tell_become_backup(void)
{
  /* XXXX note: this function is currently unsuitable for use, as it
     does not properly check that a server is in a fit state to become
     a backup browser before asking it to be one.
   */

  struct subnet_record *d;
  for (d = subnetlist; d; d = d->next)
    {
      struct work_record *work;
      for (work = d->workgrouplist; work; work = work->next)
    {
      struct server_record *s;
      int num_servers = 0;
      int num_backups = 0;
      
      for (s = work->serverlist; s; s = s->next)
      {
          struct nmb_name from, to;

          if (s->serv.type & SV_TYPE_DOMAIN_ENUM) continue;
          
          num_servers++;
          
          if (strequal(brse_server_alias(work->token), s->serv.name)) continue;
          
          if (s->serv.type & SV_TYPE_BACKUP_BROWSER) {
        num_backups++;
        continue;
          }
          
          if (s->serv.type & SV_TYPE_MASTER_BROWSER) continue;
          
          if (!(s->serv.type & SV_TYPE_POTENTIAL_BROWSER)) continue;
          
          DEBUG(3,("num servers: %d num backups: %d\n", 
               num_servers, num_backups));
          
          /* make first server a backup server. thereafter make every
         tenth server a backup server */
          if (num_backups != 0 && (num_servers+9) / num_backups > 10)
        {
          continue;
        }
          
          DEBUG(2,("sending become backup to %s %s for %s\n",
               s->serv.name, inet_ntoa(d->bcast_ip),
               work->work_group));
          
          make_nmb_name(&from, s->serv.name    , 0x20, work->scope);
          make_nmb_name(&to  , work->work_group, 0x1e, work->scope);

          /* type 11 request from brse_server_alias(work->token)(20) to WG(1e) for SERVER */
          do_announce_request(s->serv.name, &from, &to,
                  ANN_BecomeBackup,
                  d->bcast_ip);
        }
    }
    }
}


/*******************************************************************
  same context: scope.
  ******************************************************************/
BOOL same_scope(struct dgram_packet *dgram, char *scope)
{
  return strequal(dgram->dest_name.scope, scope);
}


/*******************************************************************
  am I listening on a name. XXXX check the type of name as well.
  ******************************************************************/
BOOL listening_name(struct work_record *work, struct nmb_name *n)
{
  if (strequal(n->name,brse_server_alias(work->token)) ||
      strequal(n->name,work->work_group) ||
      strequal(n->name,MSBROWSE))
    {
      return(True);
    }
  
  return(False);
}


/*******************************************************************
  process a domain announcement frame

  Announce frames come in 3 types. Servers send host announcements
  (command=1) to let the master browswer know they are
  available. Master browsers send local master announcements
  (command=15) to let other masters and backups that they are the
  master. They also send domain announcements (command=12) to register
  the domain

  The comment field of domain announcements contains the master
  browser name. The servertype is used by NetServerEnum to select
  resources. We just have to pass it to smbd (via browser.dat) and let
  the client choose using bit masks.
  ******************************************************************/
static void process_announce(struct packet_struct *p,uint16 command,char *buf)
{
  struct dgram_packet *dgram = &p->packet.dgram;
  struct subnet_record *d = find_subnet(p->ip); 
  int update_count = CVAL(buf,0);

  int ttl = IVAL(buf,1)/1000;
  char *name = buf+5;
  int osmajor=CVAL(buf,21);
  int osminor=CVAL(buf,22);
  uint32 servertype = IVAL(buf,23);
  uint32 browse_type= CVAL(buf,27);
  uint32 browse_sig = CVAL(buf,29);
  char *comment = buf+31;

  struct work_record *work;
  char *work_name;
  char *serv_name = dgram->source_name.name;
  char *work_scope = dgram->dest_name.scope;
  BOOL add = False;

  comment[43] = 0;
  
  DEBUG(4,("Announce(%d) %s(%x)",command,name,name[15]));
  DEBUG(4,("%s count=%d ttl=%d OS=(%d,%d) type=%08x sig=%4x %4x comment=%s\n",
       namestr(&dgram->dest_name),update_count,ttl,osmajor,osminor,
       servertype,browse_type,browse_sig,comment));
  
  name[15] = 0;  
  
  if (dgram->dest_name.name_type == 0 && command == ANN_HostAnnouncement)
    {
      DEBUG(2,("Announce to nametype(0) not supported yet\n"));
      return;
    }

  if (command == ANN_DomainAnnouncement && 
      ((!strequal(dgram->dest_name.name, MSBROWSE)) ||
       dgram->dest_name.name_type != 0x1))
    {
      DEBUG(0,("Announce(%d) from %s should be __MSBROWSE__(1) not %s\n",
        command, inet_ntoa(p->ip), namestr(&dgram->dest_name)));
      return;
    }
  
  if (command == ANN_DomainAnnouncement) { 
    /* XXXX if we are a master browser for the workgroup work_name,
       then there is a local subnet configuration problem. only
       we should be sending out such domain announcements, because
       as the master browser, that is our job.

       stop being a master browser, and force an election. this will
       sort out the network problem. hopefully.
     */

    work_name = name;
    add = True;
  } else {
    work_name = dgram->dest_name.name;
  }

  /* we need some way of finding out about new workgroups
     that appear to be sending packets to us. The name_type checks make
     sure we don't add host names as workgroups */
  if (command == ANN_HostAnnouncement &&
      (dgram->dest_name.name_type == 0x1d ||
       dgram->dest_name.name_type == 0x1e))
  {
    add = True;
  }
  
  if (!(work = find_workgroupstruct(d,work_name,work_scope,add)))
    return;
  
  DEBUG(4, ("workgroup %s on %s\n", work->work_group, serv_name));
  
  if (!same_scope(dgram, work->scope)) return;
  
  ttl = GET_TTL(ttl);
  
  /* add them to our browse list, and update the browse.dat file */
  add_server_entry(p->timestamp,d,work,name,servertype,ttl,comment,True);
  updatedlists = True;

#if 0
  /* the tell become backup code is broken, no great harm is done by
     disabling it */
  tell_become_backup();
#endif

  /* get the local_only browse list from the local master and add it 
     to ours. */
  if (command == ANN_LocalMasterAnnouncement)
  {
    add_browser_entry(p->timestamp, serv_name,dgram->dest_name.name_type,
              work->work_group,work->scope,30,p->ip,True);
  }
}

/*******************************************************************
  process a master announcement frame
  ******************************************************************/
static void process_master_announce(struct packet_struct *p,char *buf)
{
  struct dgram_packet *dgram = &p->packet.dgram;
  struct subnet_record *d = find_subnet(p->ip);
  char *to_name = dgram->dest_name.name; /* our primary name or an alias */

  char *name = buf;
  char *work_name; 
  char *work_scope; 
  int token;
  name[15] = 0;
  
  DEBUG(3,("Master Announce from %s (%s)\n", name, inet_ntoa(p->ip)));
  
  if (!d) return;
  
  token = brse_alias_to_token(to_name);
  if (token == -1)
  {
    DEBUG(4, ("alias %s not known\n", to_name));
    return;
  }

  work_scope = brse_domain_scope(token);

  if (!same_scope(dgram,work_scope)) return;
  
  /* carry on only if we are a domain master under the server alias */
  if (!brse_domain_master(token)) return;

  /* Convert the server name by which the master browser
     called this server to the workgroup name. */
  if ((work_name = brse_domain_name(token)) == (char*)NULL)
  {
      DEBUG(4, ("process_master_announce(): no alias for \"%s\"\n", to_name));
      return;
  }          
  
  /* merge browse lists with them */
  add_browser_entry(p->timestamp,name,0x1b,work_name,work_scope,30,p->ip,True);
}

static void response_name_srv_chk(time_t timestamp, struct packet_struct *p,
                                  struct response_record *n);

/*******************************************************************
  process a receive backup list request
  
  we receive a list of servers, and we attempt to locate them all on
  our local subnet, and sync browse lists with them on the workgroup
  they are said to be in.

  XXXX NOTE: this function is in overdrive. it should not really do
  half of what it actually does (it should pick _one_ name from the
  list received and sync with it at regular intervals, rather than
  sync with them all only once!)

  ******************************************************************/
static void process_rcv_backup_list(struct packet_struct *p,char *buf)
{
  struct dgram_packet *dgram = &p->packet.dgram;
  int count = CVAL(buf,0);
  unsigned int pick;
  uint32 info = IVAL(buf,1); /* XXXX caller's incremental info */
  char *buf1;
  
  DEBUG(3,("Receive Backup ack for %s from %s total=%d info=%d\n",
       namestr(&dgram->dest_name), inet_ntoa(p->ip),
       count, info));
  
  if (count <= 0) return;
  
  pick = sys_random(count);

  /* go through the list of servers attempting to sync browse lists */
  for (buf1 = buf+5; *buf1 && count; buf1 = skip_string(buf1, 1), --count)
  {
    struct subnet_record *d;
      
    if (count != pick) continue;

    DEBUG(4,("Searching for backup browser %s...\n", buf1));
      
    if ((d = find_subnet(p->ip)) != NULL)
    {
      struct work_record *work;
      for (work = d->workgrouplist; work; work = work->next)
      {
		struct nmb_name n;

        if (!same_scope(dgram, work->scope)) continue;
  
		make_nmb_name(&n,work->work_group, 0x1d, work->scope);
        if (work->token == 0 /* token */)
        {
		  struct nmb_query_status *nmb_data;

		  nmb_data = (struct nmb_query_status *)(malloc(sizeof(*nmb_data)));

		  if (nmb_data)
		  {
            nmb_data->d = d;

            /* query WINS server for backup browser.
               XXXX we should also do a broadcast query if this fails... */
            netbios_name_query(p->timestamp,ServerNMB, response_name_srv_chk,
				  (void*)nmb_data, &n,
                  False, False, ipgrp);
          }
        }
      }
    }
  }
}


static void response_status_srv(time_t timestamp, struct packet_struct *p,
                                struct response_record *n)
{
	if (p)
	{
		DEBUG(4,("NAME_STATUS_SRV_CHK\n")); 
		response_status_check(timestamp, False, p, n);
	}
}

static void response_name_srv_chk(time_t timestamp, struct packet_struct *p,
                                  struct response_record *n)
{
	if (p)
	{
		DEBUG(4,("NAME_QUERY_SRV_CHK\n")); 
		response_name_srv(timestamp, response_status_srv, p, n);
	}
}


/****************************************************************************
  send a backup list response.
  **************************************************************************/
static void send_backup_list(struct work_record *work, struct nmb_name *src_name,
                 int token, uint32 info,
                 int name_type, struct in_addr ip)
{                     
  char outbuf[1024];
  char *p, *countptr, *nameptr;
  int count = 0;
  struct subnet_record *d;
  struct work_record *w;
  
  struct nmb_name from, to;

  make_nmb_name(&from, brse_server_alias(work->token), 0x0, work->scope);
  make_nmb_name(&to  , src_name->name, 0x0, src_name->scope);

  DEBUG(3,("sending backup list of %s to %s: %s ", 
       work->work_group, inet_ntoa(ip), namestr(&from)));
  DEBUG(3,("%s\n", namestr(&to)));
  
  if (name_type == 0x1d)
  {
      DEBUG(4,("master browser - "));
  }
  else if (name_type == 0x1b)
  {
    DEBUG(4,("domain master - "));
  }
  else
  {
    DEBUG(0,("backup request for unknown type %0x\n", name_type));
    return;
  }
  
  bzero(outbuf,sizeof(outbuf));
  p = outbuf;
  
  CVAL(p,0) = ANN_GetBackupListResp;    /* backup list response */
  
  p++;
  countptr = p;

  SIVAL(p,1,info); /* the sender's unique info */

  p += 5;
  
  nameptr = p;

  /* find all workgroups with the same name; add _all_ backup, domain
     and local master browsers into the list of names to be returned
   */

  for (d = subnetlist; d; d = d->next)
  {
    for (w = d->workgrouplist; w; w = w->next)
    {
      struct server_record *s;
      
      /* tokens are unique: identifies the workgroup for us */
      if (w->token != work->token) continue;

      for (s = w->serverlist; s; s = s->next)
      { 
          BOOL found = False;
          char *n;
            
          if (s->serv.type & SV_TYPE_DOMAIN_ENUM) continue;
            
        for (n = nameptr; n < p; n = skip_string(n, 1))
        {
          if (strequal(n, s->serv.name)) found = True;
        }
            
        if (found) continue; /* exclude names already added */
            
        /* workgroup request: include all backup browsers in the list */
        /* domain request: include all domain members in the list */
  
        if (s->serv.type & (SV_TYPE_MASTER_BROWSER|
                            SV_TYPE_DOMAIN_MASTER|
                            SV_TYPE_BACKUP_BROWSER))
        {                          
          DEBUG(4, ("%s ", s->serv.name));
            
          count++;
          strcpy(p,s->serv.name);
          strupper(p);
          p = skip_string(p,1);
        }
      }
    }
  }

  if (count == 0)
  {
      DEBUG(4, ("none\n"));
  }
  else
  {
      DEBUG(4, (" - count %d\n", count));
  }
  
  CVAL(countptr, 0) = count;

  {
    int len = PTR_DIFF(p, outbuf);
    debug_browse_data(outbuf, len);
  }
  send_mailslot_reply(BROWSE_MAILSLOT,ClientDGRAM,outbuf,PTR_DIFF(p,outbuf),
			&from,*iface_ip(ip),
			&to,ip);
}


/*******************************************************************
  process a send backup list request

  A client sends a backup list request to ask for a list of servers on
  the net that maintain server lists for a domain. A server is then
  chosen from this list to send NetServerEnum commands to to list
  available servers.

  Currently samba only sends back one name in the backup list, its
  own. For larger nets we'll have to add backups and send "become
  backup" requests occasionally.
  ******************************************************************/
static void process_send_backup_list(struct packet_struct *p,char *buf)
{
  struct dgram_packet *dgram = &p->packet.dgram;
  struct subnet_record *d;
  struct work_record *work;

  int    token = CVAL(buf,0); /* sender's key index for the workgroup */
  uint32 info  = IVAL(buf,1); /* XXXX don't know: some sort of info */
  int name_type = dgram->dest_name.name_type;

  if (name_type != 0x1b && name_type != 0x1d)
  {
    DEBUG(0,("backup request to wrong type %d from %s\n",
          name_type,inet_ntoa(p->ip)));
    return;
  }
  
  for (d = subnetlist; d; d = d->next)
  {
    for (work = d->workgrouplist; work; work = work->next)
    {
      if (!same_scope(dgram, work->scope)) continue;
  
      if (strequal(work->work_group, dgram->dest_name.name))
      {
          DEBUG(2,("sending backup list to %s %s id=%x\n",
               namestr(&dgram->dest_name),inet_ntoa(p->ip),info));
  
          send_backup_list(work,&dgram->source_name,
                   token,info,name_type,p->ip);
          return;
      }
    } 
  }
}


/*******************************************************************
  process a reset browser state

  diagnostic packet:
  0x1 - stop being a master browser and become a backup browser.
  0x2 - discard browse lists, stop being a master browser, try again.
  0x4 - stop being a master browser forever. no way. ain't gonna.
         
  ******************************************************************/
static void process_reset_browser(struct packet_struct *p,char *buf)
{
  struct dgram_packet *dgram = &p->packet.dgram;
  int state = CVAL(buf,0);

  DEBUG(1,("received diagnostic browser reset request to %s state=0x%X\n",
       namestr(&dgram->dest_name), state));

  /* stop being a master but still deal with being a backup browser */
  if (state & 0x1)
    {
      struct subnet_record *d;
      for (d = subnetlist; d; d = d->next)
    {
      struct work_record *work;
      for (work = d->workgrouplist; work; work = work->next)
        {
          if (AM_MASTER(work))
        {
          become_nonmaster(p->timestamp,d,work,
                           SV_TYPE_DOMAIN_MASTER|SV_TYPE_MASTER_BROWSER);
        }
        }
    }
    }
  
  /* XXXX documentation inconsistency: the above description does not
     exactly tally with what is implemented for state & 0x2
   */

  /* totally delete all servers and start afresh */
  if (state & 0x2)
  {
    /* remove all workgroups (and their servers) from database */
    struct subnet_record *d;
    for (d = subnetlist; d; d = d->next)
    {
      struct work_record *work;
      for (work=d->workgrouplist;work;work=remove_workgroup(d,work,True));
    }

    /* add all known workgroups back into database */
    add_workgroups_to_subnets(p->timestamp);
  }
  
  /* stop browsing altogether. i don't think this is a good idea! */
  if (state & 0x4)
  {
      DEBUG(1,("ignoring request to stop being a browser. sorry!\n"));
  }
}

/*******************************************************************
  process a announcement request

  clients send these when they want everyone to send an announcement
  immediately. This can cause quite a storm of packets!
  ******************************************************************/
static void process_announce_request(struct packet_struct *p,char *buf)
{
  struct dgram_packet *dgram = &p->packet.dgram;
  struct work_record *work;
  struct subnet_record *d = find_subnet(p->ip);
  int token = CVAL(buf,0);
  int wg_token = 0;
  char *name = buf+1;
  char *samba_alias;

  name[15] = 0;
  
  DEBUG(3,("Announce request from %s to %s token=0x%X\n",
       name,namestr(&dgram->dest_name), token));
  
  /* look up the index for this workgroup */
  wg_token = brse_add_domain(dgram->dest_name.name);
  if (wg_token == -1) return;

  /* check that samba is participating in this workgroup */
  if (!brse_workgroup_member(wg_token)) return;

  /* find samba's NetBIOS alias it operates under in this workgroup */
  if ((samba_alias = brse_server_alias(wg_token)) == (char*)NULL) return;

  /* ignore announce requests from samba under its own alias.
     this should no longer happen because code has been added to
     discard packets from ourself. */
  if (strequal(dgram->source_name.name,samba_alias)) return;
  
  if (!d) return;
  
  /* XXXX BUG: the destination name type should also be checked,
     not just the name. e.g if the name is WORKGROUP(0x1d) then
     we should only respond if we own that name */
    
  for (work = d->workgrouplist; work; work = work->next)
  {
    if (wg_token == work->token) work->needannounce = True;
  }
}


/****************************************************************************
process a browse frame
****************************************************************************/
void process_browse_packet(struct packet_struct *p,char *buf,int len)
{
  int command = CVAL(buf,0);
  debug_browse_data(buf, len);

  switch (command) 
    {
    case ANN_HostAnnouncement:
    case ANN_DomainAnnouncement:
    case ANN_LocalMasterAnnouncement:
      {
    process_announce(p,command,buf+1);
    break;
      }
      
    case ANN_AnnouncementRequest:
      {
    process_announce_request(p,buf+1);
    break;
      }
      
    case ANN_Election:
      {
    process_election(p,buf+1);
    break;
      }
      
    case ANN_GetBackupListReq:
      {
        debug_browse_data(buf, len);
    process_send_backup_list(p,buf+1);
    break;
      }
      
    case ANN_GetBackupListResp:
    {
        debug_browse_data(buf, len);
        process_rcv_backup_list(p, buf+1);
        break;
    }
      
    case ANN_ResetBrowserState:
      {
    process_reset_browser(p, buf+1);
    break;
      }
      
    case ANN_MasterAnnouncement:
      {
    process_master_announce(p,buf+1);
    break;
      }
      
    default:
      {
    struct dgram_packet *dgram = &p->packet.dgram;
    DEBUG(4,("ignoring browse packet %d from %s %s to %s\n",
         command, namestr(&dgram->source_name), 
         inet_ntoa(p->ip), namestr(&dgram->dest_name)));
      }
    }
}


