/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: nameserv.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   04 jul 96: lkcl@pires.co.uk
   module nameserv contains name server management functions
*/

#include "includes.h"

extern int ServerNMB;

extern int DEBUGLEVEL;

extern struct in_addr ipgrp;

extern struct subnet_record *subnetlist;

extern BOOL updatedlists;

extern int updatecount;

extern struct interface *local_interfaces;

extern char *source_description[];


/****************************************************************************
  add the magic samba names, useful for finding samba servers
  **************************************************************************/
void add_my_names(time_t time_now)
{
  struct subnet_record *d;
  int token;

  /* each subnet entry, including WINS pseudo-subnet, has SELF names */

  /* XXXX if there was a transport layer added to samba (ipx/spx etc) then
     there would be yet _another_ for-loop, this time on the transport type
   */

  for (d = subnetlist; d; d = d->next)
  {
    struct nmb_name nb_name;

    BOOL wins = ip_equal(d->bcast_ip,ipgrp);
    struct in_addr myip = *iface_ip(d->bcast_ip);

	int nb_type = wins ? NB_HFLAG : NB_BFLAG;

    struct nmb_ip nb;

    nb.nb_flags = nb_type|NB_ACTIVE|NB_GROUP;
    nb.source = SELF;
    nb.death_time = 0;

    for (token = 0; token < brse_num_domains(); token++)
    {
      char my_name[17];
      char wg_name[17];

      StrnCpy(wg_name,brse_domain_name(token),16);
      StrnCpy(my_name,brse_server_alias(token),16);

      if (*my_name)
      {
        if (wins)
        {
          nb_type = NB_HFLAG; 
          /* XXXX should do a multi-homed registration, with WINS server */
        }
        else
        {
          nb_type = NB_BFLAG;
          /* XXXX should do the standard registration */
        }

        nb.ip = myip;
        nb.nb_flags = nb_type|NB_ACTIVE;

        make_nmb_name(&nb_name, my_name, 0x00, NULL);
        add_my_name_entry(time_now, d, token, &nb_name,&nb);
        make_nmb_name(&nb_name, my_name, 0x20, NULL);
        add_my_name_entry(time_now, d, token, &nb_name,&nb);
        make_nmb_name(&nb_name, my_name, 0x03, NULL);
        add_my_name_entry(time_now, d, token, &nb_name,&nb);
        make_nmb_name(&nb_name, my_name, 0x1f, NULL);
        add_my_name_entry(time_now, d, token, &nb_name,&nb);
      }

      if (*wg_name && brse_domain_logons(token))
      {
       make_nmb_name(&nb_name, wg_name, 0x1c, NULL);
        if (wins)
        {
          /* need to register all local subnets with WINS server */
          struct subnet_record *d2;

          for (d2 = subnetlist; d2; d2 = d2->next)
          {
            if (!ip_equal(d2->bcast_ip,ipgrp))
            {
              nb.ip = *iface_ip(d2->bcast_ip);
              nb.nb_flags = nb_type|NB_ACTIVE|NB_GROUP;
              /* XXXX should be a multi-homed registration, with WINS server */
              add_my_name_entry(time_now, d,token,&nb_name,&nb);
            }
          }
        }
        else
        {
          nb.ip = myip;
          nb.nb_flags = nb_type|NB_ACTIVE|NB_GROUP;
          /* register locally (broadcast) */
          add_my_name_entry(time_now, d,token,&nb_name,&nb);
        }
      }
    }

    /* these names are added permanently (ttl of zero) and will NOT be
       refreshed with the WINS server
     */

    nb.ip = d->myip;
    nb.nb_flags = nb_type|NB_ACTIVE;

    make_nmb_name(&nb_name,"*",0x0,NULL);
    add_netbios_entry(time_now, &d->namelist,&d->names_last_modified,
                      &nb_name,&nb,wins,False);

    make_nmb_name(&nb_name,"__SAMBA__",0x0,NULL);
    add_netbios_entry(time_now, &d->namelist,&d->names_last_modified,
                      &nb_name,&nb,wins,False);

    make_nmb_name(&nb_name,"__SAMBA__",0x20,NULL);
    add_netbios_entry(time_now, &d->namelist,&d->names_last_modified,
                      &nb_name,&nb,wins,False);

  }

  /* check becoming a domain master under all browser aliases */

  if ((d = find_subnet(ipgrp)))
  {
    for (token = 0; token < brse_num_domains(); token++)
    {
      if (brse_domain_master(token))
      {
        char *work_name  = brse_domain_name (token);
        char *work_scope = brse_domain_scope(token);
        if (work_name)
        {
          struct work_record *work;
          work = find_workgroupstruct(d, work_name,work_scope,True);
          if (work && work->state == MST_NONE)
          {
            work->state = MST_DOMAIN_NONE;
            become_master(time_now, d, work);
          }
        }
      }
    }
  }
}



/****************************************************************************
  remove all the samba names... from a WINS server if necessary.
  **************************************************************************/
void remove_my_names(time_t time_now)
{
    struct subnet_record *d;

    for (d = subnetlist; d; d = d->next)
    {
        struct name_record *n, *next;

        for (n = d->namelist; n; n = next)
        {
            int i;

            next = n->next;
            for (i = n->num_ips; n && i >= 0; i--)
            {
                if (n->ip_flgs[i].source == SELF)
                {
                    /* get all SELF names removed from the WINS server's database */
                    /* XXXX note: problem occurs if this removes the wrong one! */

                    remove_name_entry(time_now,d,-1,&n->name,n->ip_flgs[i].ip);
                }
            }
        }
    }
}


/* a name registration / release structure */
struct nmb_reg_rel
{
	struct subnet_record *d;
	int work_token;
};

static void response_name_release(time_t timestamp, struct packet_struct *p,
                                  struct response_record *n);

/****************************************************************************
  remove an entry from the name list

  note: the name will _always_ be removed
  XXXX at present, the name is removed _even_ if a WINS server says keep it.

  ****************************************************************************/
void remove_name_entry(time_t time_now, struct subnet_record *d, int token,
				struct nmb_name *name, struct in_addr ip)
{
  /* XXXX BUG: if samba is offering WINS support, it should still broadcast
      a de-registration packet to the local subnet before removing the
      name from its local-subnet name database. */

  struct name_record n;
  struct name_record *n2=NULL;
  struct nmb_reg_rel *nmb_data;
  int idx;
  BOOL wins = ip_equal(d->bcast_ip,ipgrp);

  memcpy(&n.name, name, sizeof(*name));

  if ((n2 = find_name_search(&d, &n.name, FIND_SELF, ip, &idx)))
  {
    /* check name isn't already being de-registered */
    if (NAME_DEREG(n2->ip_flgs[idx].nb_flags))
      return;

    /* mark the name as in the process of deletion. */
    n2->ip_flgs[idx].nb_flags &= NB_DEREG;
  }

  if (!n2) return;

  /* remove the name immediately. even if the spec says we should
     first try to release them, this is too dangerous with our current
     name structures as otherwise we will end up replying to names we
     don't really own

     XXXX solution is to ensure that name queries/status for names with DEREG
     are negative. also, should ensure that name is not used internally.

   */  
  remove_netbios_name(time_now,&d->namelist,&d->names_last_modified,
                      &n.name,n2->ip_flgs[idx].ip,False);

  /* local interface: release them on the network */

  nmb_data = (struct nmb_reg_rel*)(malloc(sizeof(*nmb_data)));

  if (nmb_data)
  {
    nmb_data->d = d;
	nmb_data->work_token = token;

    /* response_name_release() will be called when the name is released */
    netbios_name_release(time_now, ServerNMB, response_name_release,
		     nmb_data, name, &n2->ip_flgs[idx], 1,
			 !wins, True, d->bcast_ip);
  }
}


/****************************************************************************
  response for a reg release received / not received.
  **************************************************************************/
static void response_name_release(time_t timestamp, struct packet_struct *p,
                                  struct response_record *n)
{
	struct nmb_reg_rel *nmb_data = (struct nmb_reg_rel*)n->nmb_data;
	struct nmb_ip found;
	struct nmb_name *ans_name = &n->name;

	BOOL success = False;

	putip(&found.ip, &n->send_ip);
	found.nb_flags = n->nb.nb_flags;

	if (!p)
	{
		/* if no response received, and we are using a broadcast release
		   method, it must be OK for us to release the name: nobody objected 
		   on that subnet. if we are using a WINS server, then the WINS
		   server must be dead or deaf.
		*/
		if (n->bcast)
		{
			/* broadcast method: implicit acceptance of the name release
			   by not receiving any objections. */

			success = True;
		}
		else
		{
			DEBUG(1,("WINS server did not respond to name release!\n"));
		}
	}
	else
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		int rcode = nmb->header.rcode;
		char *rdata = nmb->answers->rdata;

		ans_name = &nmb->answers->rr_name;

		DEBUG(4,("NAME_RELEASE\n")); 

		if (n->num_msgs > 1)
		{
			DEBUG(1,("more than one release name response received!\n"));
			return;
		}

		if (rcode == 0 && rdata)
		{
			/* copy the netbios flags and the ip address out of reply data */
			found.nb_flags = rdata[0];
			putip((char*)&found.ip,&rdata[2]);
			
			success = True;
		}

		DEBUG(4,("response name release received\n"));
	}

	if (success)
	{
		/* NOTE: we only release our own names at present */
		if (ismyip(found.ip))
		{
			name_unregister_work(timestamp, nmb_data->d,nmb_data->work_token,
			                     ans_name,found.ip);
		}
		else
		{
			DEBUG(2,("name release for different ip! %s %s\n",
			inet_ntoa(found.ip),namestr(ans_name)));
		}
	}
	else
	{
		DEBUG(2,("name release for %s rejected\n",namestr(ans_name)));

		/* XXXX PANIC! what to do if it's one of samba's own names? */
		/* XXXX should we release the name anyway? */
	}
}

static void response_name_register(time_t timestamp, struct packet_struct *p,
                                   struct response_record *n);

/****************************************************************************
  initiate adding an entry to the netbios name list for a local interface
  ****************************************************************************/
void add_my_name_entry(time_t time_now, struct subnet_record *d, int token,
                       struct nmb_name *name,struct nmb_ip *nb)
{
  struct nmb_reg_rel *nmb_data;
  int nmb_type = NMB_REG;
  BOOL wins;

  if (!d) return;

  wins = ip_equal(d->bcast_ip,ipgrp);

  /* set up the data to be registered */
  nb->death_time = nb->death_time ? nb->death_time : GET_TTL(0);

  DEBUG(3,("add_my_name: %s %s %2x\n",
            namestr(name),inet_ntoa(nb->ip),nb->nb_flags));

  /* not that it particularly matters, but if the SELF name already
     exists, it must be re-registered, rather than just registered
   */

  /* multi-homed registration for wins netbios names if there are
     three or more (including the wins pseudo-subnet) local interfaces
   */
  if (wins && get_num_subnets() > 2) nmb_type = NMB_REG_MULTI;

  /* refreshing a name that already exists? */
  if (find_name(d->namelist, name, SELF)) nmb_type = NMB_REG_REFRESH;

  nmb_data = (struct nmb_reg_rel*)(malloc(sizeof(*nmb_data)));

  if (nmb_data)
  {
    nmb_data->d = d;
	nmb_data->work_token = token;

    /* response_name_register() will be called to deal with registration */
    netbios_name_register(time_now, ServerNMB, nmb_type, response_name_register,
                 nmb_data, name, nb, 1,
                 !wins, True, d->bcast_ip);
  }
}


/****************************************************************************
  response for a reg request received / not received
  **************************************************************************/
static void response_name_register(time_t timestamp, struct packet_struct *p,
                                   struct response_record *n)
{
	struct nmb_reg_rel *nmb_data = (struct nmb_reg_rel*)n->nmb_data;
	struct nmb_ip found;
	struct nmb_name *ans_name = &n->name;

	BOOL success = False;

	memcpy(&found,&n->nb,sizeof(n->nb));

	if (!p)
	{
		/* if no response received, and we are using a broadcast registration
		   method, it must be OK for us to register the name: nobody objected 
		   on that subnet. if we are using a WINS server, then the WINS
		   server must be dead or deaf.
		*/
		if (n->bcast)
		{
			/* broadcast method: implicit acceptance of the name registration
			   by not receiving any objections. */

			success = True;
		}
		else
		{
			/* received no response. rfc1001.txt states that after retrying,
			   we should assume the WINS server is dead, and fall back to
			   broadcasting (see bits about M nodes: can't find any right
			   now) */

			DEBUG(1,("WINS server did not respond to name registration!\n"));
		}
	}
	else
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		int rcode = nmb->header.rcode;
		char *rdata = nmb->answers->rdata;
		found.death_time = nmb->answers->ttl;

		ans_name = &nmb->answers->rr_name;

		if (n->num_msgs > 1)
		{
			DEBUG(1,("more than one register name response received!\n"));
			return;
		}

		if (rcode == 0 && rdata)
		{
			/* copy netbios flags and ip address out of the reply data */
			found.nb_flags = rdata[0];
			putip((char*)&found.ip,&rdata[2]);
			 
			success = True;
		}

	}
	if (success)
	{
		DEBUG(4,("response name reg received: %s\n",namestr(ans_name)));

		/* IMPORTANT: see expire_netbios_response_entries() */

		name_register_work(timestamp, nmb_data->d, nmb_data->work_token,
		               ans_name, &found, n->bcast);
	}
	else
	{
		DEBUG(3,("name registration for %s rejected!\n",namestr(ans_name)));

		/* oh dear. we have problems. possibly unbecome a master browser. */
		name_unregister_work(timestamp, nmb_data->d,nmb_data->work_token,
							 ans_name,found.ip);
	}
}


/****************************************************************************
  add a domain entry. creates a workgroup, if necessary, and adds the domain
  to the named a workgroup.
  ****************************************************************************/
struct subnet_record *add_subnet_entry(time_t time_now,
				struct in_addr bcast_ip, struct in_addr mask_ip,
                       char *work_name, char *work_scope,
						BOOL add, BOOL lmhosts)
{
	struct subnet_record *d;
	struct in_addr ip;

	DEBUG(4,("add_subnet_entry: %s %s\n", work_name,inet_ntoa(bcast_ip)));

	if (zero_ip(bcast_ip)) bcast_ip = *iface_bcast(bcast_ip);

	ip = *iface_ip(bcast_ip);

	/* add the domain into our domain database */
	if ((d = find_subnet(bcast_ip)) || (d = make_subnet(bcast_ip, mask_ip)))
	{
		struct work_record *work = find_workgroupstruct(d,work_name,
		                                                work_scope,add);
		int nb_type = ip_equal(d->bcast_ip, ipgrp) ? NB_HFLAG : NB_BFLAG;

		struct nmb_name n;
		struct nmb_ip nb;

		if (!work) return NULL;

		nb.nb_flags = nb_type|NB_ACTIVE|NB_GROUP;
		nb.ip = ip;
		nb.source = SELF;
		nb.death_time = 0;

		make_nmb_name(&n, work_name, 0x0,work->scope);
		add_my_name_entry(time_now, d,work->token,&n,&nb);

		if (brse_workgroup_member(work->token))
		{
			/* add samba server name to workgroup list. don't add
			lmhosts server entries to local interfaces */

			pstring comment;
			char *my_name    = brse_server_alias  (work->token);
			char *my_comment = brse_server_comment(work->token);

			StrnCpy(comment, my_comment, 43);

			add_server_entry(time_now, d,work,my_name,
			                 work->ServerType | SV_TYPE_LOCAL_LIST_ONLY,
			                 0,comment,True);

			DEBUG(3,("Added server name entry %s at %s\n",
			          work_name,inet_ntoa(bcast_ip)));

			/* add WORKGROUP(1e) and WORKGROUP(00) entries into name database */

			make_nmb_name(&n, work_name,0x1e,work->scope);
			add_my_name_entry(time_now, d,work->token,&n,&nb);

		}

		return d;
	}

	return NULL;
}


/*******************************************************************
  write out browse.dat
  ******************************************************************/
void write_browse_list(time_t time_now)
{
  struct subnet_record *d;
  pstring fname,fnamenew;
  FILE *f;

  static time_t lasttime = 0;

  if (!lasttime) lasttime = time_now;
  if (!updatedlists || time_now - lasttime < 5) return;
  
  lasttime = time_now;
  updatedlists = False;
  updatecount++;
  
  dump_workgroups();
  
  strcpy(fname,lp_lockdir());
  trim_string(fname,NULL,"/");
  strcat(fname,"/");
  strcat(fname,SERVER_LIST);
  strcpy(fnamenew,fname);
  strcat(fnamenew,".");
  
  f = fopen(fnamenew,"w");
  
  if (!f)
    {
      DEBUG(4,("Can't open %s - %s\n",fnamenew,strerror(errno)));
      return;
    }
  
  for (d = subnetlist; d ; d = d->next)
    {
      struct work_record *work;
      for (work = d->workgrouplist; work ; work = work->next)
    {
      struct server_record *s;
      for (s = work->serverlist; s ; s = s->next)
        {
          fstring tmp;
          
          /* don't list domains I don't have a master for */
          if ((s->serv.type & SV_TYPE_DOMAIN_ENUM) && !s->serv.comment[0])
        {
          continue;
        }
          
          /* output server details, plus what workgroup/domain
         they're in. without the domain information, the
         combined list of all servers in all workgroups gets
         sent to anyone asking about any workgroup! */
          
          sprintf(tmp, "\"%s\"", s->serv.name);
          fprintf(f, "%-25s ", tmp);
          fprintf(f, "%08x ", s->serv.type);
          sprintf(tmp, "\"%s\" ", s->serv.comment);
          fprintf(f, "%-30s", tmp);
          fprintf(f, "\"%s\"\n", work->work_group);
        }
    }
    }
  
  fclose(f);
  unlink(fname);
  chmod(fnamenew,0644);
  rename(fnamenew,fname);   
  DEBUG(3,("Wrote browse list %s\n",fname));
}


/****************************************************************************
  add the default workgroups into my domain
  **************************************************************************/
void add_workgroup_to_subnet(time_t time_now,
				char *group, char *scope,
				struct in_addr bcast_ip, struct in_addr mask_ip)
{
  if (group && *group != '*')
  {
    DEBUG(4,("add_wg_to_subnet: %s %s\n", group, inet_ntoa(bcast_ip)));
    add_subnet_entry(time_now, bcast_ip,mask_ip,group,scope,True, False);
  }
}


/****************************************************************************
  add the default workgroup into my domain
  **************************************************************************/
void add_workgroups_to_subnets(time_t time_now)
{
  struct interface *i;

  /* add or find domain on our local subnet, in the default workgroup */

  for (i = local_interfaces; i; i = i->next)
  {
    int token;
    for (token = 0; token < brse_num_domains(); token++)
    {
      add_workgroup_to_subnet(time_now,
                              brse_domain_name (token),
                              brse_domain_scope(token),
                              i->bcast,i->nmask);
    }
  }
}


/****************************************************************************
  dump a copy of the name table
  **************************************************************************/
void dump_names(time_t time_now)
{
	struct subnet_record *d;

	for (d = subnetlist; d; d = d->next)
	{
		write_netbios_names(time_now, inet_ntoa(d->bcast_ip),
		                    d->namelist, &d->names_last_modified, False);
	}
}


/*******************************************************************
  expires or refreshes old names in the namelist

  if the name is a samba SELF name, it must be refreshed rather than
  removed.
  ******************************************************************/
void check_expire_names(time_t time_now)
{
    struct name_record *n;
    struct name_record *next;
    struct subnet_record *d;

    /* expire old names */
    for (d = subnetlist; d; d = d->next)
    {
        for (n = d->namelist; n; n = next)
        {
			int i;
            next = n->next;
                              
            for (i = 0; n && i < n->num_ips; i++)
            {
                if (n->ip_flgs[i].death_time &&
				    n->ip_flgs[i].death_time < time_now)
                {
                    if (n->ip_flgs[i].source == SELF)
                    {
                        /* refresh the samba name. if this refresh fails
                           for any reason, it will be deleted.
                         */
                        DEBUG(3,("Refresh SELF name %s\n", namestr(&n->name)));

						n->ip_flgs[i].death_time = 0;
                        add_my_name_entry(time_now,d,-1,
						                  &n->name,&n->ip_flgs[i]);
                        continue;
                    }         
                    n = remove_name(time_now,
					                &d->namelist, &d->names_last_modified,
					                n, n->ip_flgs[i].ip,False);
                    i--; /* just removed one: do same index on next iteration */
                }
			}
        }
    }
}


/****************************************************************************
  find a name in the domain database namelist 
  search can be any of:
  FIND_SELF - look exclusively for names the samba server has added for itself
  FIND_LOCAL - look for names in the local subnet record.
  FIND_WINS - look for names in the WINS record
  **************************************************************************/
struct name_record *find_name_search(struct subnet_record **d,
            struct nmb_name *name,
            int search, struct in_addr ip, int *i)
{
    struct name_record *n;

    DEBUG(4,("find_name_search: %s %s search %x\n",
                        namestr(name), inet_ntoa(ip), search));
    if (d == NULL) return NULL; /* bad error! */
    
    if (search & FIND_LOCAL)
    {
        if (*d != NULL)
        {
            DEBUG(4,("find_name on local: %s search %x\n",
                        namestr(name), search));
			if ((n = find_name((*d)->namelist, name, search)) != NULL)
            {
				*i = find_name_idx(n, ip, search);
    			if (*i == -1) *i = 0;

				return n;
            }
        }
    }

    if ((search & FIND_WINS) != FIND_WINS) return NULL;

    /* find WINS subnet record. */
    *d = find_subnet(ipgrp);

    if (*d == NULL) return NULL;

    DEBUG(4,("find_name on WINS: %s search %x\n",
                        namestr(name), search));
    
	if ((n = find_name((*d)->namelist, name, search)) != NULL)
	{
		*i = find_name_idx(n, ip, search);
    	if (*i == -1) *i = 0;
	}
	return n;
}


