/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: namequery.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   09 oct 96: lkcl@pires.co.uk
   created module namequery containing NetBIOS query reply 

*/

#include "includes.h"

extern int DEBUGLEVEL;


/***************************************************************************
reply to a name query.

with broadcast name queries:

	- only reply if the query is for one of YOUR names. all other machines on
	  the network will be doing the same thing (that is, only replying to a
	  broadcast query if they own it)

	- NEVER send a negative response to a broadcast query. no-one else will!

with directed name queries:

	- you are expected to respond with either a negative or positive response.

****************************************************************************/
void reply_name_query(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  struct nmb_name *question = &nmb->question.question_name;
  BOOL bcast = nmb->header.nm_flags.bcast;
  BOOL wins  = nmb->header.nm_flags.recursion_desired;
  int rcode = 0;
  struct in_addr retip;
  char rdata[6];
  struct subnet_record *d = NULL;
  BOOL success = True;
  struct name_record *n;
  int idx;

  /* directed queries are for WINS server: broadcasts are local SELF queries.
     the exception is Domain Master names.  */

  if (!(d = find_req_subnet(p->ip, False)))
  {
    DEBUG(3,("name query: %s not known\n", inet_ntoa(p->ip)));
    success = False;
  }

  DEBUG(3,("Name query "));
  
  if (wins && !bcast)
  {
    DEBUG(2,("nmbd received packet intended for winsd!\n"));
    return;
  }

  n = find_name_search(&d,question,FIND_WINS|FIND_LOCAL|FIND_SELF,p->ip,&idx);

  if (!n || n->ip_flgs[idx].source != SELF)
  {
      if (bcast) return; /* never reply negative response to bcasts */
      success = False;
  }
  
  /* if the IP is 0 then substitute my IP */
  if (zero_ip(retip))
  {
    DEBUG(0,("ERROR in nmbd name query: zero IP address\n"));
  }

  if (success)
  {
    int i;
    DEBUG(3,("OK "));
    for (i = 0; i < n->num_ips; i++)
    {
      rdata[6*i+0] = n->ip_flgs[i].nb_flags;
      rdata[6*i+1] = 0;
      putip(&rdata[6*i+2],(char *)&n->ip_flgs[i].ip);
      DEBUG(3,("- %s %2x ",inet_ntoa(n->ip_flgs[i].ip),n->ip_flgs[i].nb_flags));
    }
    DEBUG(3,("\n"));      
  }
  else
  {
    DEBUG(3,("UNKNOWN\n"));      
  }
  
  /* see rfc1002.txt 4.2.14 for name query RCODES */
  rcode = success ? RCODE_QUERY_OK : RCODE_QUERY_NAM_ERR;
  
  reply_netbios_packet(p,nmb->header.name_trn_id,
			   rcode,NMB_QUERY,0,False,True,
		       &nmb->question.question_name,
               RR_TYPE_NBIP, RR_CLASS_IP,
		       0,
		       rdata, success ? n->num_ips * 6 : 0);
}
