/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   30 July 96: David.Chappell@mail.trincoll.edu
   Expanded multiple workgroup domain master browser support.

*/

#include "includes.h"
#include "smb.h"

extern int ServerNMB;

extern int DEBUGLEVEL;

extern struct in_addr ipgrp;

/* this is our browse master/backup cache database */
static struct browse_cache_record *browserlist = NULL;


/***************************************************************************
  add a browser into the list
  **************************************************************************/
static void add_browse_cache(struct browse_cache_record *b)
{
  struct browse_cache_record *b2;

  if (!browserlist)
    {
      browserlist = b;
      b->prev = NULL;
      b->next = NULL;
      return;
    }
  
  for (b2 = browserlist; b2->next; b2 = b2->next) ;
  
  b2->next = b;
  b->next = NULL;
  b->prev = b2;
}


/*******************************************************************
  remove old browse entries
  ******************************************************************/
void expire_browse_cache(time_t t)
{
  struct browse_cache_record *b;
  struct browse_cache_record *nextb;
  
  /* expire old entries in the serverlist */
  for (b = browserlist; b; b = nextb)
    {
      if (b->synced && b->sync_time < t)
    {
      DEBUG(3,("Removing dead cached browser %s\n",b->name));
      nextb = b->next;
      
      if (b->prev) b->prev->next = b->next;
      if (b->next) b->next->prev = b->prev;
      
      if (browserlist == b) browserlist = b->next; 
      
      free(b);
    }
      else
    {
      nextb = b->next;
    }
    }
}


/****************************************************************************
  add a browser entry
  ****************************************************************************/
struct browse_cache_record *add_browser_entry(time_t time_now,
				char *name, int type,
				char *wg, char *work_scope,
				time_t ttl, struct in_addr ip, BOOL local)
{
  BOOL newentry=False;
  
  struct browse_cache_record *b;

  /* search for the entry: if it's already in the cache, update that entry */
  for (b = browserlist; b; b = b->next)
    {
      if (ip_equal(ip,b->ip) && strequal(b->group, wg)) break;
    }
  
  if (b && b->synced)
    {
      /* entries get left in the cache for a while. this stops sync'ing too
     often if the network is large */
      DEBUG(4, ("browser %s %s %s already sync'd at time %d\n",
        b->name, b->group, inet_ntoa(b->ip), b->sync_time));
      return NULL;
    }
  
  if (!b)
    {
      newentry = True;
      b = (struct browse_cache_record *)malloc(sizeof(*b));
      
      if (!b) return(NULL);
      
      bzero((char *)b,sizeof(*b));
    }
  
  /* update the entry */
  ttl = time_now+ttl;
  
  StrnCpy(b->name ,name      ,sizeof(b->name )-1);
  StrnCpy(b->group,wg        ,sizeof(b->group)-1);
  StrnCpy(b->scope,work_scope,sizeof(b->scope)-1);
  strupper(b->name);
  strupper(b->group);
  
  b->ip     = ip;
  b->type   = type;
  b->local  = local; /* local server list sync or complete sync required */
  
  if (newentry || ttl < b->sync_time) 
    b->sync_time = ttl;
  
  if (newentry)
    {
      b->synced = False;
      add_browse_cache(b);
      
      DEBUG(3,("Added cache entry %s %s(%2x) %s ttl %d\n",
           wg, name, type, inet_ntoa(ip),ttl));
    }
  else
    {
      DEBUG(3,("Updated cache entry %s %s(%2x) %s ttl %d\n",
           wg, name, type, inet_ntoa(ip),ttl));
    }
  
  return(b);
}


static void response_sync_local(time_t timestamp, struct packet_struct *p,
                                struct response_record *n);

static void response_sync_remote(time_t timestamp, struct packet_struct *p,
                                 struct response_record *n);


/****************************************************************************
find a server responsible for a workgroup, and sync browse lists
**************************************************************************/
static void start_sync_browse_entry(time_t time_now,
				struct browse_cache_record *b)
{                     
  struct subnet_record *d;
  struct work_record *work;

  if ((d = find_subnet(b->ip)) == (struct subnet_record *)NULL ) return;
  
  if (!(work = find_workgroupstruct(d, b->group, b->scope,False))) return;
    
  /* only sync if we are the master */
  if (AM_MASTER(work))
  {
    
      /* first check whether the group we intend to sync with exists. if it
         doesn't, the server must have died. o dear. */
    
	  struct nmb_query_status *nmb_data;

	  nmb_data = (struct nmb_query_status *)(malloc(sizeof(*nmb_data)));

	  if (nmb_data)
	  {
        struct nmb_name n;
		nmb_data->d = d;
		make_nmb_name(&n,b->group, 0x20, b->scope);

        netbios_name_query(time_now, ServerNMB,
                       b->local?response_sync_local:response_sync_remote,
			           (void*)nmb_data, &n,
                       False,False,b->ip);
      }
  }
    
  b->synced = True;
}


/****************************************************************************
  response from a name query to sync browse lists 
  ****************************************************************************/
static void response_query_sync(time_t timestamp, BOOL local,
			struct packet_struct *p, struct response_record *n)
{
	if (p)
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		struct nmb_name *ans_name = &nmb->answers->rr_name;
		int rcode = nmb->header.rcode;
		char *rdata = nmb->answers->rdata;

		struct nmb_query_status *nbd = (struct nmb_query_status*)n->nmb_data;

		struct nmb_ip found;
		struct nmb_ip *data = NULL;
		
		if (rcode == 0 && rdata)
		{
			/* copy the netbios flags and the ip address out the reply data */
			found.nb_flags = rdata[0];
			putip((char*)&found.ip,&rdata[2]);
			
			data = &found;
		}

		DEBUG(4, ("Name query SYNC at %s ip %s - ",
			  namestr(&n->name), inet_ntoa(n->send_ip)));

		if (!name_equal(&n->name, ans_name))
		{
			/* someone gave us the wrong name as a reply. oops. */
			DEBUG(4,("unexpected name received: %s\n", namestr(ans_name)));
			return;
		}

		if (data)
		{
			struct work_record *work = NULL;

			if (!ip_equal(n->send_ip, data->ip))
			{
				/* someone gave us the wrong ip as a reply. oops. */
				DEBUG(4,("expected ip: %s\n", inet_ntoa(n->send_ip)));
				DEBUG(4,("unexpected ip: %s\n", inet_ntoa(data->ip)));
				return;
			}

			DEBUG(4, (" OK: %s\n", inet_ntoa(data->ip)));

			if ((work = find_workgroupstruct(nbd->d, ans_name->name,
														 ans_name->scope,False)))
			{
				/* the server is there: sync quick before it (possibly) dies! */
				sync_browse_lists(timestamp, nbd->d,work,
								ans_name->name,ans_name->name_type,
								data->ip, local);
			}
		}
		else
		{
			DEBUG(4, (" NEGATIVE RESPONSE!\n"));
		}
    }
}

static void response_sync_local(time_t timestamp, struct packet_struct *p,
                                struct response_record *n)
{
	if (p)
	{
		DEBUG(4,("RESPONSE QUERY SYNC - LOCAL\n"));
		response_query_sync(timestamp, True, p, n);
	}
}


static void response_sync_remote(time_t timestamp, struct packet_struct *p,
                                 struct response_record *n)
{
	if (p)
	{
		DEBUG(4,("RESPONSE QUERY SYNC - REMOTE\n"));
		response_query_sync(timestamp, False, p, n);
	}
}


/****************************************************************************
  search through browser list for an entry to sync with
  **************************************************************************/
void do_browser_lists(time_t time_now)
{
  struct browse_cache_record *b;
  static time_t last = 0;
  
  if (time_now-last < 20) return; /* don't do too many of these at once! */
                           /* XXXX equally this period should not be too long
                              the server may die in the intervening gap */
  
  last = time_now;
  
  /* pick any entry in the list, preferably one whose time is up */
  for (b = browserlist; b && b->next; b = b->next)
    {
      if (b->sync_time < time_now && b->synced == False) break;
    }
  
  if (b && !b->synced)
  {
    /* sync with the selected entry then remove some dead entries */
    start_sync_browse_entry(time_now, b);
    expire_browse_cache(time_now - 60);
  }

}

