/*
 *   fs/cifs/transport.c
 *
 *   Copyright (c) International Business Machines  Corp., 2002
 *   Author(s): Steve French (sfrench@us.ibm.com)
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation; either version 2.1 of the License, or
 *   (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <linux/fs.h>
#include <linux/list.h>
#include <linux/wait.h>
#include <linux/net.h>
#include <linux/version.h>
#include <asm/uaccess.h>
#include <asm/processor.h>
#include "cifspdu.h"
#include "cifsglob.h"
#include "cifsproto.h"
#include "cifs_debug.h"


struct mid_q_entry *AllocMidQEntry(struct smb_hdr *smb_buffer,
				   struct cifsSesInfo *ses)
{
	struct mid_q_entry *temp;

/* BB add spinlock to protect midq for each session BB */
	if (ses == NULL) {
		cERROR(1, ("\nNull session passed in to AllocMidQEntry "));
		return NULL;
	}
	temp = kmalloc(sizeof(struct mid_q_entry), GFP_KERNEL);
	if (temp == NULL)
		return temp;
	else {
		memset(temp, 0, sizeof(struct mid_q_entry));
		temp->mid = smb_buffer->Mid;	/* always LE */
		temp->pid = current->pid;
		temp->command = smb_buffer->Command;
		cFYI(1, ("\nFor smb_command %d", temp->command));
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,18))
		get_fast_time(&temp->when_sent);
#else
		do_gettimeofday(&temp->when_sent);
#endif
		temp->ses = ses;
		temp->tsk = current;
	}
	if (ses->status == CifsGood) {
		list_add_tail(&temp->qhead, &ses->server->pending_mid_q);
		atomic_inc(&midCount);
	} else {		/* BB add reconnect code here BB */

		cERROR(1,
		       ("\nNeed to reconnect after session died to server\n"));
		if (temp)
			kfree(temp);
		return NULL;
	}
	return temp;
}

void DeleteMidQEntry(struct mid_q_entry *midEntry)
{
	/* BB add spinlock to protect midq for each session BB */
	if (midEntry->resp_buf)
		kfree(midEntry->resp_buf);
	list_del(&midEntry->qhead);
	kfree(midEntry);
	atomic_dec(&midCount);
}

int smb_send(struct socket *ssocket, struct smb_hdr *smb_buffer,
	     unsigned int smb_buf_length, struct sockaddr *sin)
{
	int rc = 0;
	struct msghdr smb_msg;
	struct iovec iov;
	mm_segment_t temp_fs;

/*  ssocket->sk->allocation = GFP_BUFFER; *//* BB is this spurious? */
	iov.iov_base = smb_buffer;
	iov.iov_len = smb_buf_length + 4;

	smb_msg.msg_name = sin;
	smb_msg.msg_namelen = sizeof(struct sockaddr);
	smb_msg.msg_iov = &iov;
	smb_msg.msg_iovlen = 1;
	smb_msg.msg_control = NULL;
	smb_msg.msg_controllen = 0;
	smb_msg.msg_flags = MSG_DONTWAIT + MSG_NOSIGNAL;	/* BB add more flags? */

	/* smb header is converted in header_assemble. bcc and rest of SMB word area, 
	   and byte area if necessary, is converted to littleendian in cifssmb.c and RFC1001 
	   len is converted to bigendian in smb_send */
	if (smb_buf_length > 12)
		smb_buffer->Flags2 = cpu_to_le16(smb_buffer->Flags2);
	smb_buffer->smb_buf_length =
	    cpu_to_be32(smb_buffer->smb_buf_length);
	cFYI(1, ("\nSending smb of length %d ", smb_buf_length));
	dump_smb(smb_buffer, smb_buf_length + 4);

	temp_fs = get_fs();	/* we must turn off socket api parm checking */
	set_fs(get_ds());
	rc = sock_sendmsg(ssocket, &smb_msg, smb_buf_length + 4);

	set_fs(temp_fs);

	if (rc < 0) {
		cERROR(1,
		       ("\nError %d sending data on socket to server.\n",
			rc));
	} else
		rc = 0;

	return rc;
}

int SendReceive(const unsigned int xid, struct cifsSesInfo *ses,
		struct smb_hdr *in_buf, struct smb_hdr *out_buf,
		int *pbytes_returned, const int long_op)
{
	int rc = 0;
	int receive_len;
	long timeout;
	struct mid_q_entry *midQ;

	midQ = AllocMidQEntry(in_buf, ses);
	if (midQ == NULL)
		return -EIO;	/* reconnect should be done, if possible, in AllocMidQEntry */
	/* BB should we convert more fields to little endian here such as flags2 ? */
	midQ->resp_buf = out_buf;	/* BB won't work - need to have buffers allocated by socket receiver thread */
	if (in_buf->smb_buf_length > CIFS_MAX_MSGSIZE + MAX_CIFS_HDR_SIZE) {
		cERROR(1,
		       ("\nIllegal length, greater than maximum frame, %d ",
			in_buf->smb_buf_length));
		DeleteMidQEntry(midQ);
		return -EIO;
	}
	rc = smb_send(ses->server->ssocket, in_buf, in_buf->smb_buf_length,
		      (struct sockaddr *) &(ses->server->sockAddr));

	cFYI(1, ("\ncifs smb_send rc %d", rc));
	/* BB add code to wait for response and copy to out_buf */
	if (long_op > 1)	/* writes past end of file can take a looooooong time */
		timeout = 300 * HZ;
	else if (long_op == 1)
		timeout = 60 * HZ;
	else
		timeout = 15 * HZ;
	/* wait for 15 seconds or until woken up due to response arriving or due to last connection to this server being unmounted */
	timeout =
	    interruptible_sleep_on_timeout(&ses->server->response_q,
					   timeout);
	cFYI(1, (" with timeout %ld and Out_buf: %x  midQ->resp_buf: %x ",
		 timeout, (int) out_buf, (int) midQ->resp_buf));
	cFYI(1,
	     ("and Length: %x\n",
	      be32_to_cpu(midQ->resp_buf->smb_buf_length)));
	if (signal_pending(current)) {
		cERROR(1, (KERN_ERR "\nCIFS: caught signal"));
		DeleteMidQEntry(midQ);
		return -EINTR;
	} else {
		receive_len = be32_to_cpu(midQ->resp_buf->smb_buf_length);
	}

	if (timeout == 0) {
		cFYI(1,
		     ("\nTimeout on receive. Assume response SMB is invalid.\n"));
		rc = -ETIMEDOUT;
	} else if (receive_len > CIFS_MAX_MSGSIZE + MAX_CIFS_HDR_SIZE) {
		cERROR(1,
		       ("\nFrame too large received.  Length: %d  Xid: %d\n",
			receive_len, xid));
		rc = -EIO;
	} else {		/* rcvd frame is ok */

		if (midQ->resp_buf && out_buf) {
			memcpy(out_buf, midQ->resp_buf,
			       receive_len +
			       4 /* include 4 byte RFC1001 header */ );
			/* convert the length back to a form that we can use */

			dump_smb(out_buf, 92);

			out_buf->smb_buf_length =
			    be32_to_cpu(out_buf->smb_buf_length);
			if (out_buf->smb_buf_length > 12)
				out_buf->Flags2 =
				    le16_to_cpu(out_buf->Flags2);
			if (out_buf->smb_buf_length > 28)
				out_buf->Pid = le16_to_cpu(out_buf->Pid);
			if (out_buf->smb_buf_length > 28)
				out_buf->PidHigh =
				    le16_to_cpu(out_buf->PidHigh);


			*pbytes_returned = out_buf->smb_buf_length;

			/* BB special case reconnect tid and reconnect uid here? */
			rc = map_smb_to_linux_error(out_buf);	/* BB watch endianness here BB */

            /* convert ByteCount if necessary */
            if(receive_len >= sizeof(struct smb_hdr) - 4 /* do not count RFC1001 header */ + (2 * out_buf->WordCount) + 2 /* bcc */)
                BCC(out_buf) = le16_to_cpu(BCC(out_buf));
		} else
			rc = -EIO;
	}

	DeleteMidQEntry(midQ);	/* BB what if process is killed ? - BB add background daemon to clean up Mid entries from killed processes BB test killing process with active mid */
	return rc;
}
