/*
 *   fs/cifs/cifsfs.c
 *
 *   Copyright (c) International Business Machines  Corp., 2002
 *   Author(s): Steve French (sfrench@us.ibm.com)
 *
 *   Common Internet FileSystem (CIFS) client
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation; either version 2.1 of the License, or
 *   (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/* Note that BB means BUGBUG (ie something to fix eventually) */


#include <linux/module.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/version.h>
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,7))
#include <linux/locks.h>
#endif
#include <linux/list.h>
#include "cifsfs.h"
#include "cifspdu.h"
#define DECLARE_GLOBALS_HERE
#include "cifsglob.h"
#include "cifsproto.h"
#include "cifs_debug.h"
#include "cifs_fs_sb.h"
#include <linux/mm.h>
#define CIFS_MAGIC_NUMBER 0xFF534D42	/* the first four bytes of all SMB PDUs */

extern struct file_system_type cifs_fs_type;

int cifsFYI = 0;
int cifsERROR = 1;
int traceSMB = 0;
unsigned int oplockEnabled = 0;
unsigned int multiuser_mount = 0;

extern int cifs_mount(struct super_block *, struct cifs_sb_info *, char *);
extern int cifs_umount(struct super_block *, struct cifs_sb_info *);
void cifs_proc_init(void);
void cifs_proc_clean(void);

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
struct super_block *cifs_read_super(struct super_block *sb,
				    void *data, int silent)
{
	struct inode *inode;
	struct cifs_sb_info *cifs_sb;
	int rc = 0;

	/* dump_mem(" cifs_fs_type ", &cifs_fs_type, sizeof(cifs_fs_type)); */
	cifs_sb = (struct cifs_sb_info *) &sb->u;
	cifs_sb->local_nls = load_nls_default();	/* needed for ASCII cp to Unicode converts */
	rc = cifs_mount(sb, cifs_sb, data);

	if (rc) {
		if (!silent)
			cERROR(1,
			       ("cifs_mount failed w/return code = %d\n",
				rc));
		goto out_mount_failed;
	}

	sb->s_magic = CIFS_MAGIC_NUMBER;
	sb->s_op = &cifs_super_ops;
	sb->s_blocksize = CIFS_MAX_MSGSIZE;	/* BB check SMBSessSetup negotiated size */
	sb->s_blocksize_bits = 10;	/* 2**10 = CIFS_MAX_MSGSIZE */
	inode = iget(sb, ROOT_I);

	if (!inode)
		goto out_no_root;

	sb->s_root = d_alloc_root(inode);

	if (!sb->s_root)
		goto out_no_root;

	return sb;

      out_no_root:
	cEVENT(1, ("cifs_read_super: get root inode failed\n"));
	if (inode)
		iput(inode);

/*	rc = cifs_umount(sb);  BB is CIFS unmount routine needed? */
	if (rc) {
		cERROR(1,
		       ("cifs_umount failed with return code %d\n", rc));
	}
      out_mount_failed:
	sb->s_dev = 0;
	return 0;
}
#else
static int cifs_read_super(struct super_block *sb, void *data, int silent)
{
	struct inode *inode;
	struct cifs_sb_info *cifs_sb;
	int rc = 0;

	/* dump_mem(" cifs_fs_type ", &cifs_fs_type, sizeof(cifs_fs_type)); */
	cifs_sb = (struct cifs_sb_info *) &sb->u;
	cifs_sb->local_nls = load_nls_default();	/* needed for ASCII cp to Unicode converts */
	rc = cifs_mount(sb, cifs_sb, data);

	if (rc) {
		if (!silent)
			cERROR(1,
			       ("cifs_mount failed w/return code = %d\n",
				rc));
		goto out_mount_failed;
	}

	sb->s_magic = CIFS_MAGIC_NUMBER;
	sb->s_op = &cifs_super_ops;
	sb->s_blocksize = CIFS_MAX_MSGSIZE;	/* BB check SMBSessSetup negotiated size */
	sb->s_blocksize_bits = 10;	/* 2**10 = CIFS_MAX_MSGSIZE */
	inode = iget(sb, ROOT_I);

	if (!inode)
		goto out_no_root;

	sb->s_root = d_alloc_root(inode);

	if (!sb->s_root)
		goto out_no_root;

	return 0;

      out_no_root:
	cEVENT(1, ("cifs_read_super: get root inode failed\n"));
	if (inode)
		iput(inode);

/*	rc = cifs_umount(sb);  BB is CIFS unmount routine needed? */
	if (rc) {
		cERROR(1,
		       ("cifs_umount failed with return code %d\n", rc));
	}
      out_mount_failed:
	return -EINVAL;
}
#endif

void cifs_put_super(struct super_block *sb)
{
	int rc = 0;
	struct cifs_sb_info *cifs_sb;

	cFYI(1, ("In cifs_put_super\n"));
	cifs_sb = (struct cifs_sb_info *) &sb->u;
	rc = cifs_umount(sb, cifs_sb);	/* BB add in missing function Make sure to free waiting procs by preventing new sends on session if this tcon is the last one and also waking those blocked on server->response_q before exiting.  Also remember to only kill the SMB session (ses struct) by sending uloggoffX if this is the last tconn on the SMB session and kill the socket (tcp session) if this is the last SMB session on the socket BB (tcp_server_info struct */
	if (rc) {
		cERROR(1,
		       ("cifs_umount failed with return code %d\n", rc));
	}
	unload_nls(cifs_sb->local_nls);
	cifs_sb->local_nls = NULL;
	return;
}

int cifs_statfs(struct super_block *sb, struct statfs *buf)
{
	int xid, rc;
	struct cifs_sb_info *cifs_sb;
	struct cifsTconInfo *pTcon;

	xid = GetXid();

	cifs_sb = (struct cifs_sb_info *) &(sb->u);
	pTcon = cifs_sb->tcon;

	buf->f_type = CIFS_MAGIC_NUMBER;

	/* instead could get the real value via SMB_QUERY_FS_ATTRIBUTE_INFO */
	buf->f_namelen = PATH_MAX;	/* PATH_MAX may be too long - it would presumably be the
					   length of total path, note that some servers may be 
					   able to support more than this, but best to be safe
					   since Win2k among others will not work with long filenames */
	buf->f_files = 0;	/* undefined */
	buf->f_ffree = -1;	/* unlimited */

	rc = CIFSSMBQFSInfo(xid, pTcon, buf, cifs_sb->local_nls);

	/*     
	   int f_type;
	   __fsid_t f_fsid;
   int f_namelen;  *//* BB get this/these from info put in tcon struct at mount time with call to QFSAttrInfo */

	return 0;		/* always return success? what if volume is no longer available? */
}

struct address_space_operations cifs_addr_ops = {
	/* none needed yet */

};

struct super_operations cifs_super_ops = {

/*clear_inode:  cifs_clear_inode,
  write_inode:  cifs_write_inode,*//* probably unneeded in our case */

	read_inode:cifs_read_inode,
/* put_inode:    cifs_put_inode, *//* BB do not need to know about decrement of usage count ? */
/* delete_inode: cifs_delete_inode,  *//* BB add back in delete_inode code when safe distributed inode caching - ie FindNotify added */
	put_super:cifs_put_super,
	statfs:cifs_statfs,

	/*  umount_begin: cifs_umount_begin,
	   dirty_inode:  cifs_dirty_inode,
   remount_fs:   cifs_remount_fs, *//* TBD */
};

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
static DECLARE_FSTYPE(cifs_fs_type, "cifs", cifs_read_super,
		      0 /* BUGBUG FS_ flags go here */ );
struct inode_operations cifs_dir_inode_ops = {	/* divide the following into file vs. dir inode operations */
	create:cifs_create,
	lookup:cifs_lookup,
	unlink:cifs_unlink,
	link:cifs_hardlink,
	mkdir:cifs_mkdir,
	rmdir:cifs_rmdir,
	rename:cifs_rename,
	revalidate:cifs_revalidate,
	setattr:cifs_setattr,
	symlink:cifs_symlink,
};

struct inode_operations cifs_file_inode_ops = {
	revalidate:cifs_revalidate,
	setattr:cifs_setattr,
	rename:cifs_rename,
};

#else
static struct super_block *cifs_get_sb(struct file_system_type *fs_type,
				       int flags, char *dev_name,
				       void *data)
{
	cFYI(1, ("\nDevname: %s flags: %d ", dev_name, flags));
	return get_sb_nodev(fs_type, flags, data, cifs_read_super);
}

static struct file_system_type cifs_fs_type = {
	owner:THIS_MODULE,
	name:"cifs",
	get_sb:cifs_get_sb,
};
struct inode_operations cifs_dir_inode_ops = {	/* divide the following into file vs. dir inode operations */
	create:cifs_create,
	lookup:cifs_lookup,
	unlink:cifs_unlink,
	link:cifs_hardlink,
	mkdir:cifs_mkdir,
	rmdir:cifs_rmdir,
	rename:cifs_rename,
/*	revalidate:cifs_revalidate,   */
	setattr:cifs_setattr,
	symlink:cifs_symlink,
};

struct inode_operations cifs_file_inode_ops = {
/*	revalidate:cifs_revalidate, */
	setattr:cifs_setattr,
	rename:cifs_rename,
};
#endif


struct inode_operations cifs_symlink_inode_ops = {
	readlink:cifs_readlink,
	follow_link:cifs_follow_link,
	/* BB add the following two eventually */
	/* revalidate:     cifs_revalidate,
	   setattr:        cifs_notify_change, */
};

struct file_operations cifs_file_ops = {
	read:cifs_read,
	write:cifs_write,
	open:cifs_open,
	release:cifs_close,
	lock:cifs_lock,
	fsync:cifs_fsync,
};

struct file_operations cifs_dir_ops = {
/* open:        cifs_dir_open, *//* turns out to be unneeded - we FindFirst, FindNext and close search in cifs_readdir */
	readdir:cifs_readdir,
	release:cifs_closedir,
};

static int __init init_cifs(void)
{
#if CONFIG_PROC_FS
	cifs_proc_init();
#endif
	INIT_LIST_HEAD(&GlobalServerList);	/* BB not implemented yet */
	INIT_LIST_HEAD(&GlobalSMBSessionList);
	INIT_LIST_HEAD(&GlobalTreeConnectionList);

/*
 *  Initialize Global counters
 */
	atomic_set(&sesInfoAllocCount, 0);
	atomic_set(&tconInfoAllocCount, 0);
	atomic_set(&bufAllocCount, 0);
	atomic_set(&midCount, 0);
	GlobalTotalActiveXid = 0;
	GlobalMaxActiveXid = 0;

	return register_filesystem(&cifs_fs_type);
}

static void __exit exit_cifs(void)
{
	cFYI(0, ("\nIn unregister ie exit_cifs"));
#if CONFIG_PROC_FS
	cifs_proc_clean();
#endif
/* BB check to see if there is additional memory freeing we need to do here e.g. freeing from list_heads etc. BB */
	unregister_filesystem(&cifs_fs_type);
}

MODULE_AUTHOR("Steve French <sfrench@us.ibm.com>");
MODULE_LICENSE("GPL");		/* combination of LGPL + GPL source behaves as GPL */
MODULE_DESCRIPTION
    ("VFS support for Common Internet File System servers such as Windows, Samba and those that comply with the SNIA CIFS Specification");
module_init(init_cifs)
    module_exit(exit_cifs)
