
#include "yodl.h"

static int escapechar (char const *str)         /* convert escaped char */
{
    switch (*str)
    {
        case 'a':           return ('\a');
        case 'b':           return ('\b');
        case 'f':           return ('\f');
        case 'n':           return ('\n');
        case 'r':           return ('\r');
        case 't':           return ('\t');
        case 'v':           return ('\v');
        default:            return (*str);
    }
}

static int string_to_char (char const *str)     /* convert 'charspec' */
{
    if (*str != '\'')
        return (-1);
        
    str++;
    if (! *str)
        return (-1);
    
    if (*str != '\\')
        return (*str);
        
    str++;
    if (! *str)
        return (-1);
        
    return (escapechar (str));
}

static char *string_to_string (char const *str) /* convert escape string */
{
    char
        *ret = 0;
        
    while (*str)
    {
        if (*str == '\\')                       /* escape sequence */
        {
            str++;
            ret = str_addchar (ret, escapechar (str));
        }
        else
            ret = str_addchar (ret, *str);
            
        if (*str)
            str++;
    }
    
    return (ret);
}

void parse_table (char *table, CHARTAB *dest)
{
    char
        *line,
        **lines = 0,
	**charp = 0,
        *newchar,
	**redefp = 0,
        *newredef;
    int 
        i,
        chindex,
        redeflen,
        nlines = 0,
	ncharp = 0,
	nredefp = 0;
        
    line = strtok (table, "\n");
    while (line)
    {
        lines = strtab_add (lines, &nlines, line);
        line = strtok (0, "\n");
    }
    
    for (i = 0; i < nlines; i++)
    {
        newchar = strtok (lines [i], "=");
        if (! newchar)
            error_gram (builtin [mac_definechartable], 
                        "missing character specifier in line %s", lines [i]);
        while (isspace (*newchar))
            newchar++;
        newredef = strtok (0, "=");
        if (! newredef)
            error_gram (builtin [mac_definechartable], 
                        "missing redefinition specifier in line %s", lines [i]);
        while (isspace (*newredef))
            newredef++;
                        
        charp = strtab_add (charp, &ncharp, newchar);
        redefp = strtab_add (redefp, &nredefp, newredef);
    }
    
    for (i = 0; i < nlines; i++)
    {
        if ( (chindex = string_to_char (charp [i])) == -1 )
            error_gram (builtin [mac_definechartable], 
                        "at %s: expected 'c' character specifier", charp [i]);
                        
        if (*redefp [i] != '"' || ! *(redefp[i] + 1))
            error_gram (builtin [mac_definechartable], 
                        "redefinition %s: must have form \"redef\"", redefp);
        strcpy (redefp [i], redefp [i] + 1);
        redeflen = strlen (redefp [i]) - 1;
        if (redefp [i] [redeflen] != '"')
            error_gram (builtin [mac_definechartable], 
                        "redefinition %s: unterminated string", redefp [i]);
        redefp [i] [redeflen] = '\0';

        (*dest)[chindex] = string_to_string (redefp [i]);
    }

    strtab_free (lines, nlines);
    strtab_free (charp, ncharp);
    strtab_free (redefp, nredefp);
}

void gram_definechartable ()
{
    char
        *tabname,                               /* table name */
	*table;                                 /* table itself */
    int
        i;                                      /* index in chartab */
    CHARTAB
        *dest;                                  /* new table to define */
    static char
        buf [2];                                /* conversion buf */
          
                                                
    tabname =                                   /* retrieve table name */
        gram_parlist (builtin [mac_definechartable], 0);
    gram_onename (builtin [mac_definechartable], tabname);
    
    message (3, "%s %s\n", builtin [mac_definechartable], tabname);
    
    while (lextok == tok_space ||               /* skip spaces, newlines */
           lextok == tok_newline
	  )
        lexer ();
            
    table =                                     /* retrieve table itself */
        gram_parlist (builtin [mac_definechartable], 0);
    if (! table || ! *table)                    /* may not be empty */
        error_gram (builtin [mac_definechartable],
                    "empty table definition for table %s", tabname);
    
                                                /* table already defined? */
    if (strtab_find (chartabname, nchartab, tabname) != -1)
        error_gram (builtin [mac_definechartable], 
                    "table %s already defined", tabname);

    i = curchartab - chartab;                   /* remember current */
                    
                                                /* make new table */
    chartabname = strtab_add (chartabname, &nchartab, tabname);
    chartab = (CHARTAB *) xrealloc (chartab, nchartab * sizeof (CHARTAB));

    if (curchartab)                             /* restore current */
        curchartab = chartab + i;
    
    dest = chartab + nchartab - 1;              /* destination of new */

    for (i = 0; i < 256; i++)                   /* reset table to defaults */
    {
        buf [0] = (char) i;
        (*dest) [i] = xstrdup (buf);
    }
    
    parse_table (table, dest);
    
    free (tabname);
    free (table);
}
