/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   Password and authentication handling
   Copyright (C) Andrew Tridgell 1992-1998
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

extern int DEBUGLEVEL;
extern int Protocol;
extern struct in_addr ipzero;

/* users from session setup */
static pstring session_users="";

extern pstring global_myname;
extern fstring global_myworkgroup;

/* this holds info on user ids that are already validated for this VC */
static user_struct *validated_users;
static int next_vuid = VUID_OFFSET;
static int num_validated_vuids;

/****************************************************************************
check if a uid has been validated, and return an pointer to the user_struct
if it has. NULL if not. vuid is biased by an offset. This allows us to
tell random client vuid's (normally zero) from valid vuids.
****************************************************************************/
user_struct *get_valid_user_struct(uint16 vuid)
{
	user_struct *usp;
	int count=0;

	if (vuid == UID_FIELD_INVALID)
		return NULL;

	for (usp=validated_users;usp;usp=usp->next,count++) {
		if (vuid == usp->vuid) {
			if (count > 10) {
				DLIST_PROMOTE(validated_users, usp);
			}
			return usp;
		}
	}

	return NULL;
}

/****************************************************************************
invalidate a uid
****************************************************************************/
void invalidate_vuid(uint16 vuid)
{
	user_struct *vuser = get_valid_user_struct(vuid);

	if (vuser == NULL)
		return;

	session_yield(vuid);

	DLIST_REMOVE(validated_users, vuser);

	safe_free(vuser->groups);
	delete_nt_token(&vuser->nt_user_token);
	safe_free(vuser);
	num_validated_vuids--;
}

/****************************************************************************
invalidate all vuid entries for this process
****************************************************************************/
void invalidate_all_vuids(void)
{
	user_struct *usp, *next=NULL;

	for (usp=validated_users;usp;usp=next) {
		next = usp->next;
		
		invalidate_vuid(usp->vuid);
	}
}

/****************************************************************************
return a validated username
****************************************************************************/
char *validated_username(uint16 vuid)
{
	user_struct *vuser = get_valid_user_struct(vuid);
	if (vuser == NULL)
		return 0;
	return(vuser->user.unix_name);
}

/****************************************************************************
return a validated domain
****************************************************************************/
char *validated_domain(uint16 vuid)
{
	user_struct *vuser = get_valid_user_struct(vuid);
	if (vuser == NULL)
		return 0;
	return(vuser->user.domain);
}


/****************************************************************************
 Create the SID list for this user.
****************************************************************************/

NT_USER_TOKEN *create_nt_token(uid_t uid, gid_t gid, int ngroups, gid_t *groups, BOOL is_guest)
{
	extern DOM_SID global_sid_World;
	extern DOM_SID global_sid_Network;
	extern DOM_SID global_sid_Builtin_Guests;
	extern DOM_SID global_sid_Authenticated_Users;
	NT_USER_TOKEN *token;
	DOM_SID *psids;
	int i, psid_ndx = 0;
	size_t num_sids = 0;
	fstring sid_str;

	if ((token = (NT_USER_TOKEN *)malloc( sizeof(NT_USER_TOKEN) ) ) == NULL)
		return NULL;

	ZERO_STRUCTP(token);

	/* We always have uid/gid plus World and Network and Authenticated Users or Guest SIDs. */
	num_sids = 5 + ngroups;

	if ((token->user_sids = (DOM_SID *)malloc( num_sids*sizeof(DOM_SID))) == NULL) {
		free(token);
		return NULL;
	}

	psids = token->user_sids;

	/*
	 * Note - user SID *MUST* be first in token !
	 * se_access_check depends on this.
	 */

	uid_to_sid( &psids[psid_ndx++], uid);

	/*
	 * Primary group SID is second in token. Convention.
	 */

	gid_to_sid( &psids[psid_ndx++], gid);

	/* Now add the group SIDs. */

	for (i = 0; i < ngroups; i++) {
		if (groups[i] != gid) {
			gid_to_sid( &psids[psid_ndx++], groups[i]);
		}
	}

	/*
	 * Finally add the "standard" SIDs.
	 * The only difference between guest and "anonymous" (which we
	 * don't really support) is the addition of Authenticated_Users.
	 */

	sid_copy( &psids[psid_ndx++], &global_sid_World);
	sid_copy( &psids[psid_ndx++], &global_sid_Network);

	if (is_guest)
		sid_copy( &psids[psid_ndx++], &global_sid_Builtin_Guests);
	else
		sid_copy( &psids[psid_ndx++], &global_sid_Authenticated_Users);

	token->num_sids = psid_ndx;

	/* Dump list of sids in token */

	for (i = 0; i < token->num_sids; i++) {
		DEBUG(5, ("user token sid %s\n", 
			  sid_to_string(sid_str, &token->user_sids[i])));
	}

	return token;
}

/****************************************************************************
register a uid/name pair as being valid and that a valid password
has been given. vuid is biased by an offset. This allows us to
tell random client vuid's (normally zero) from valid vuids.
****************************************************************************/

int register_vuid(uid_t uid,gid_t gid, char *unix_name, char *requested_name, 
		  char *domain,BOOL guest)
{
	user_struct *vuser = NULL;
	struct passwd *pwfile; /* for getting real name from passwd file */

	/* Ensure no vuid gets registered in share level security. */
	if(lp_security() == SEC_SHARE)
		return UID_FIELD_INVALID;

	/* Limit allowed vuids to 16bits - VUID_OFFSET. */
	if (num_validated_vuids >= 0xFFFF-VUID_OFFSET)
		return UID_FIELD_INVALID;

	if((vuser = (user_struct *)malloc( sizeof(user_struct) )) == NULL) {
		DEBUG(0,("Failed to malloc users struct!\n"));
		return UID_FIELD_INVALID;
	}

	ZERO_STRUCTP(vuser);

	DEBUG(10,("register_vuid: (%u,%u) %s %s %s guest=%d\n", (unsigned int)uid, (unsigned int)gid,
				unix_name, requested_name, domain, guest ));

	/* Allocate a free vuid. Yes this is a linear search... :-) */
	while( get_valid_user_struct(next_vuid) != NULL ) {
		next_vuid++;
		/* Check for vuid wrap. */
		if (next_vuid == UID_FIELD_INVALID)
			next_vuid = VUID_OFFSET;
	}

	DEBUG(10,("register_vuid: allocated vuid = %u\n", (unsigned int)next_vuid ));

	vuser->vuid = next_vuid;
	vuser->uid = uid;
	vuser->gid = gid;
	vuser->guest = guest;
	fstrcpy(vuser->user.unix_name,unix_name);
	fstrcpy(vuser->user.smb_name,requested_name);
	fstrcpy(vuser->user.domain,domain);

	vuser->n_groups = 0;
	vuser->groups  = NULL;

	/* Find all the groups this uid is in and store them. 
		Used by become_user() */
	initialise_groups(unix_name, uid, gid);
	get_current_groups( &vuser->n_groups, &vuser->groups);

	/* Create an NT_USER_TOKEN struct for this user. */
	vuser->nt_user_token = create_nt_token(uid,gid, vuser->n_groups, vuser->groups, guest);

	next_vuid++;
	num_validated_vuids++;

	DLIST_ADD(validated_users, vuser);

	DEBUG(3,("uid %d registered to name %s\n",(int)uid,unix_name));

	DEBUG(3, ("Clearing default real name\n"));
	if ((pwfile=sys_getpwnam(vuser->user.unix_name))!= NULL) {
		DEBUG(3, ("User name: %s\tReal name: %s\n",vuser->user.unix_name,pwfile->pw_gecos));
		fstrcpy(vuser->user.full_name, pwfile->pw_gecos);
	}

	if (!session_claim(vuser->vuid)) {
		DEBUG(1,("Failed to claim session for vuid=%d\n", vuser->vuid));
		invalidate_vuid(vuser->vuid);
		return -1;
	}

	return vuser->vuid;
}


/****************************************************************************
add a name to the session users list
****************************************************************************/
void add_session_user(char *user)
{
  fstring suser;
  StrnCpy(suser,user,sizeof(suser)-1);

  if (!Get_Pwnam(suser,True)) return;

  if (suser && *suser && !in_list(suser,session_users,False))
    {
      if (strlen(suser) + strlen(session_users) + 2 >= sizeof(pstring))
	DEBUG(1,("Too many session users??\n"));
      else
	{
	  pstrcat(session_users," ");
	  pstrcat(session_users,suser);
	}
    }
}


/****************************************************************************
check if a username is valid
****************************************************************************/
BOOL user_ok(char *user,int snum)
{
	char **valid, **invalid;
	BOOL ret;

	valid = invalid = NULL;
	ret = True;

	if (lp_invalid_users(snum)) {
		lp_list_copy(&invalid, lp_invalid_users(snum));
		if (invalid && lp_list_substitute(invalid, "%S", lp_servicename(snum))) {
			ret = !user_in_list(user, invalid);
		}
	}
	if (invalid) lp_list_free (&invalid);

	if (ret && lp_valid_users(snum)) {
		lp_list_copy(&valid, lp_valid_users(snum));
		if (valid && lp_list_substitute(valid, "%S", lp_servicename(snum))) {
			ret = user_in_list(user,valid);
		}
	}
	if (valid) lp_list_free (&valid);

	if (ret && lp_onlyuser(snum)) {
		char **user_list = lp_list_make (lp_username(snum));
		if (user_list && lp_list_substitute(user_list, "%S", lp_servicename(snum))) {
			ret = user_in_list(user, user_list);
		}
		if (user_list) lp_list_free (&user_list);
	}

	return(ret);
}

/****************************************************************************
validate a group username entry. Return the username or NULL
****************************************************************************/
static char *validate_group(char *group,char *password,int pwlen,int snum)
{
#ifdef HAVE_NETGROUP
	{
		char *host, *user, *domain;
		setnetgrent(group);
		while (getnetgrent(&host, &user, &domain)) {
			if (user) {
				if (user_ok(user, snum) && 
				    password_ok(user,password,pwlen)) {
					endnetgrent();
					return(user);
				}
			}
		}
		endnetgrent();
	}
#endif
  
#ifdef HAVE_GETGRENT
	{
		struct group *gptr;
		setgrent();
		while ((gptr = (struct group *)getgrent())) {
			if (strequal(gptr->gr_name,group))
				break;
		}

		/*
		 * As user_ok can recurse doing a getgrent(), we must
		 * copy the member list into a pstring on the stack before
		 * use. Bug pointed out by leon@eatworms.swmed.edu.
		 */

		if (gptr) {
			pstring member_list;
			char *member;
			size_t copied_len = 0;
			int i;

			*member_list = '\0';
			member = member_list;

			for(i = 0; gptr->gr_mem && gptr->gr_mem[i]; i++) {
				size_t member_len = strlen(gptr->gr_mem[i]) + 1;
				if( copied_len + member_len < sizeof(pstring)) { 

					DEBUG(10,("validate_group: = gr_mem = %s\n", gptr->gr_mem[i]));

					safe_strcpy(member, gptr->gr_mem[i], sizeof(pstring) - copied_len - 1);
					copied_len += member_len;
					member += copied_len;
				} else {
					*member = '\0';
				}
			}

			endgrent();

			member = member_list;
			while (*member) {
				static fstring name;
				fstrcpy(name,member);
				if (user_ok(name,snum) &&
				    password_ok(name,password,pwlen)) {
					endgrent();
					return(&name[0]);
				}

				DEBUG(10,("validate_group = member = %s\n", member));

				member += strlen(member) + 1;
			}
		} else {
			endgrent();
			return NULL;
		}
	}
#endif
	return(NULL);
}

/****************************************************************************
 Check for authority to login to a service with a given username/password.
 Note this is *NOT* used when logging on using sessionsetup_and_X.
****************************************************************************/

BOOL authorise_login(int snum,char *user,char *password, int pwlen, 
		     BOOL *guest,BOOL *force,uint16 vuid)
{
	BOOL ok = False;
	user_struct *vuser = get_valid_user_struct(vuid);

#if DEBUG_PASSWORD
	DEBUG(100,("authorise_login: checking authorisation on user=%s pass=%s\n",
			user,password));
#endif

	*guest = False;
  
	if (GUEST_ONLY(snum))
		*force = True;

	if (!GUEST_ONLY(snum) && (lp_security() > SEC_SHARE)) {

		/*
		 * We should just use the given vuid from a sessionsetup_and_X.
		 */

		if (!vuser) {
			DEBUG(1,("authorise_login: refusing user %s with no session setup\n",
					user));
			return False;
		}

		if (!vuser->guest && user_ok(vuser->user.unix_name,snum)) {
			fstrcpy(user,vuser->user.unix_name);
			*guest = False;
			DEBUG(3,("authorise_login: ACCEPTED: validated uid ok as non-guest \
(user=%s)\n", user));
			return True;
		}
	}
 
	/* there are several possibilities:
		1) login as the given user with given password
		2) login as a previously registered username with the given password
		3) login as a session list username with the given password
		4) login as a previously validated user/password pair
		5) login as the "user =" user with given password
		6) login as the "user =" user with no password (guest connection)
		7) login as guest user with no password

		if the service is guest_only then steps 1 to 5 are skipped
	*/

	if (!(GUEST_ONLY(snum) && GUEST_OK(snum))) {
		/* check the given username and password */
		if (!ok && (*user) && user_ok(user,snum)) {
			ok = password_ok(user,password, pwlen);
			if (ok)
				DEBUG(3,("authorise_login: ACCEPTED: given username (%s) password ok\n",
						user ));
		}

		/* check for a previously registered guest username */
		if (!ok && (vuser != 0) && vuser->guest) {	  
			if (user_ok(vuser->user.unix_name,snum) &&
					password_ok(vuser->user.unix_name, password, pwlen)) {
				fstrcpy(user, vuser->user.unix_name);
				vuser->guest = False;
				DEBUG(3,("authorise_login: ACCEPTED: given password with registered user %s\n", user));
				ok = True;
			}
		}

		/* now check the list of session users */
		if (!ok) {
			char *auser;
			char *user_list = strdup(session_users);
			if (!user_list)
				return(False);

			for (auser=strtok(user_list,LIST_SEP); !ok && auser;
									auser = strtok(NULL,LIST_SEP)) {
				fstring user2;
				fstrcpy(user2,auser);
				if (!user_ok(user2,snum))
					continue;
		  
				if (password_ok(user2,password, pwlen)) {
					ok = True;
					fstrcpy(user,user2);
					DEBUG(3,("authorise_login: ACCEPTED: session list username (%s) \
and given password ok\n", user));
				}
			}

			free(user_list);
		}

		/* check for a previously validated username/password pair */
		if (!ok && (lp_security() > SEC_SHARE) && (vuser != 0) && !vuser->guest &&
							user_ok(vuser->user.unix_name,snum)) {
			fstrcpy(user,vuser->user.unix_name);
			*guest = False;
			DEBUG(3,("authorise_login: ACCEPTED: validated uid (%s) as non-guest\n",
				user));
			ok = True;
		}

		/* check the user= fields and the given password */
		if (!ok && lp_username(snum)) {
			char *auser;
			pstring user_list;
			StrnCpy(user_list,lp_username(snum),sizeof(pstring));

			pstring_sub(user_list,"%S",lp_servicename(snum));
	  
			for (auser=strtok(user_list,LIST_SEP); auser && !ok;
											auser = strtok(NULL,LIST_SEP)) {
				if (*auser == '@') {
					auser = validate_group(auser+1,password,pwlen,snum);
					if (auser) {
						ok = True;
						fstrcpy(user,auser);
						DEBUG(3,("authorise_login: ACCEPTED: group username \
and given password ok (%s)\n", user));
					}
				} else {
					fstring user2;
					fstrcpy(user2,auser);
					if (user_ok(user2,snum) && password_ok(user2,password,pwlen)) {
						ok = True;
						fstrcpy(user,user2);
						DEBUG(3,("authorise_login: ACCEPTED: user list username \
and given password ok (%s)\n", user));
					}
				}
			}
		}
	} /* not guest only */

	/* check for a normal guest connection */
	if (!ok && GUEST_OK(snum)) {
		fstring guestname;
		StrnCpy(guestname,lp_guestaccount(snum),sizeof(guestname)-1);
		if (Get_Pwnam(guestname,True)) {
			fstrcpy(user,guestname);
			ok = True;
			DEBUG(3,("authorise_login: ACCEPTED: guest account and guest ok (%s)\n",
					user));
		} else {
			DEBUG(0,("authorise_login: Invalid guest account %s??\n",guestname));
		}
		*guest = True;
	}

	if (ok && !user_ok(user,snum)) {
		DEBUG(0,("authorise_login: rejected invalid user %s\n",user));
		ok = False;
	}

	return(ok);
}
