package ch.swisslife.ibgm.mc2;
import java.awt.*;
import java.util.*;

/** Component like label, but with multiple lines. Good for messages which span over more than one line. 
 * Taken from "Java in a Nutshell" by David Flanagan, O'Reilly &Assoc.
 * @author David Flanagan
 * @author Michael Grob for RA/SL
 * @version 1.01 09/04/98
 */
public class MultiLineLabel extends Canvas {
	public static final int LEFT = 0; 	// layout variables
	public static final int CENTER = 1;
	public static final int RIGHT = 2;
	protected String[] lines;		// lines of text
	protected int num_lines = 7;		// number of lines
	protected int margin_width;		// left and right margins
	protected int margin_height;		// top and bottom margins
	protected int line_height;		// size of font
	protected int line_ascent;		// ascend of font
	protected int [] line_widths;		// width of each line
	protected int max_width;		// width of widest line
	protected int alignment = LEFT;		// alignment of text

	
	/*Puts the message into an array of lines.
	 * 
	 * @param label	String with the text to be displayed
	 */
	protected void newLabel(String label) {
		StringTokenizer t = new StringTokenizer(label, "\n");
		num_lines = t.countTokens();
		lines = new String[num_lines];
		line_widths = new int[num_lines];
		for(int i = 0; i < num_lines; i++) {
			lines[i] = t.nextToken();
		}
	}


	/** Gives back the size of the font, the width of each line.
	 */
	protected void measure() {
		FontMetrics fm = this.getFontMetrics(this.getFont());
		// if no font metrics available yet, go back
		if (fm == null) return;

		line_height = fm.getHeight();
		line_ascent = fm.getAscent();
		max_width =400;
		for(int i = 0; i < num_lines; i++) {
			line_widths[i] = fm.stringWidth(lines[i]);
			if (line_widths[i] > max_width) max_width = line_widths[i];
		}
	}


	/** Constructor version 1: all options.
	 *
	 * @param label		String with text
	 * @param margin_widht	int with margin width
	 * @param margin_height	int with height of margin
	 * @param alignment	int describing the alignment
	 */
	public MultiLineLabel(String label, int margin_width, int margin_height, int alignment) {
		newLabel(label);
		this.margin_width = margin_width;
		this.margin_height = margin_height;
		this.alignment = alignment;
	}

	/** Constructor version 2: margins.
	 *
	 * @param label		String with text
	 * @param margin_widht	int with margin width
	 * @param margin_height	int with height of margin
	 */
	public MultiLineLabel (String label, int width, int height) {
		this(label, width, height, LEFT);
	}

		
	/** Constructor version 3: alignment.
	 *
	 * @param label		String with text
	 * @param alignment	int describing the alignment
	 */
	public MultiLineLabel (String label, int alignment) {
		this(label, 10, 10, alignment);
	}


	/** Constructor version 4: nothing specified.
	 *
	 * @param label		String with text
	 */
	public MultiLineLabel (String label) {
		this(label, 10, 10, LEFT);
	}


	/** Sets the label
	 *
	 * @param label	String with Text to be displayed
	 */
	public void setLabel(String label) {
		newLabel(label);
		measure();
		repaint();
	}

	
	/** Sets the Font
	 *
	 * @param f	Font
	 */
	public void setFont(Font f) {
		super.setFont(f);
		measure();
		repaint();
	}


	public void setAlignment(int a) {
		alignment = a;
		repaint();
	}


	public void setMarginWidth(int mw) {
		margin_width = mw;
		repaint();
	}


	public void setMarginHeight(int mh) {
		margin_height = mh;
		repaint();
	}


	public int getAlignment() {
		return alignment;
	}


	public int getMarginWidth() {
		return margin_width;
	}


	public int getMarginHeight() {
		return margin_height;
	}


	/** Is used when our component is first called, but before it is displayed.
	 */
	public void addNotify() {
		super.addNotify();
		measure();
	}


	/** IF layout manager asks how big it should be, this is the answer.
	 *
	 * @return Dimension the size we need
	 */
	public Dimension getPreferredSize() {
		return new Dimension(max_width +2*margin_width, 7*line_height+2*margin_height);
	}


	/** If layout manager asks how big the minial space is
	 *
	 * @return Dimension Minimal space
	 */
	public Dimension getMinimumSize() {
		return this.getPreferredSize();
	}

	
	/** Paints the Component.
	 *
	 * @param g	Graphics object of the component
	 */
	public void paint(Graphics g) {
		int x, y;
		Dimension d = this.getSize();
		y = margin_height;
		x = margin_width;
		for(int i = 0; i < num_lines; i++, y +=line_height) {
			switch(alignment) {
			case LEFT:
				x = margin_width;
				break;
			case CENTER:
				x = (d.width - line_widths[i])/2;
				break;
			case RIGHT:
				x = d.width - margin_width - line_widths[i];
				break;
			}
			g.drawString(lines[i], x, y);
		}
	}
}

