# ciscoRttMonLib - implements code to query Cisco IOS devices
# Joerg.Kummer at Roche.com 16/01/04

package ciscoRttMonLib;

##################################################################
#           PLEASE MODIFY THE FOLLOWING TO INSTALL               #
##################################################################

my $SAAroot = "/usr/local/SAA";

# the path to the configuration files directory
# please include a trailing slash
my $configpath = $SAAroot."etc/";

# the path to a temporary directory used for cached router measurements and the logfile
# please include a trailing slash
my $tmppath = $SAAroot."tmp/";

# It would be nice to have latency graphs, which differentiate between
# Case 1) the measurement didn't work , e.g. no SNMP access to IOS/SAA device etc
# Case 2) the SAA measurements were performed, but none of the pings/TCP connect was successful, e.g. 100% loss
# E.g. a latency graph in MRTG could have a grey area in case 1 and a red area in case 2.
#
# However, as of MRTG 2.9.10 this is not possible, because MRTG only accepts
# 0..MaxBytes (AbsMax if defined) as legal values into the RRA. All other values and "UNKNOWN" result into
# a "U" (unknown, NaN) value being written into the RRA.
# (2.9.29 introduced UNKNOWN support).
#
# This script writes an error mesasge to stdout in case 1. 
# MRTG will try to interprete the error message, fail, log the err or message and write <NaN,NaN> into the RRA.
# In case 2, the script writes $ERRAllLost to stdout
my $ERRAllLost="-1";
my $ERRNaN="-2";

##################################################################
#                PLEASE DO NOT MODIFY AFTER THIS                 #
##################################################################

require 5.004;

use vars qw($VERSION);
use Exporter;
$VERSION = '0.2';
@ISA = qw(Exporter);
sub version () { $VERSION; };

use strict;
use Symbol;
use Carp;
use BER;
use SNMP_Session;
use SNMP_util "0.97";
use ciscoRttMonMIB "0.6";
use Getopt::Long;
use Pod::Usage;

my %opt;
my ($debug, $history);
my (@times, $sum, $sum2, $min, $max, $n, $drops);
my $scaleC=1;
my ($ioshost, $community, $entry);
my $Sioshost;
my @myparms;

sub SAAStart();
sub SAAFetch();
sub SAADelete();
sub EncodeAddress($);

sub start() {
   GetOptions(\%opt, 'man','help', 'configfile=s', 'debug') or pod2usage(2);; 

   if($opt{man})      { pod2usage(-verbose => 2);exit  0};
   if($opt{help})     { pod2usage(-verbose => 1);exit  0};

   $debug= 1==1 if $opt{debug} ;

   if(!$opt{configfile}) { pod2usage(-verbose => 1);exit -1};
   my $file=$configpath.$opt{configfile};
   print "reading configuration $file\n" if($debug);
   if (! eval `cat $file`) {
      print "error reading configuration file $file !\n";
      exit -1;
      }
   # check whether config file contains history configuration
   $history=0==1;
   for (my $i = 0; $i <= $#myparms; $i++) {
      if ($myparms[$i] =~ /^rttMon.*/) {
         $myparms[$i].=".".$entry;
         if ($myparms[$i] =~ /^rttMonHistoryAdminFilter.*/) {
            $history=0==0;
         }
      }
   }
   $Sioshost = $community."@".$ioshost.":::::2";

   &SAAStart();
}

sub delete() {
   GetOptions(\%opt, 'man','help', 'ioshost=s', 'community=s', 'entry=i',
         'configfile=s', 'debug') or pod2usage(2);

   if($opt{man})      { pod2usage(-verbose => 2);exit  0};
   if($opt{help})     { pod2usage(-verbose => 1);exit  0};

   $debug= 1==1 if $opt{debug} ;

   if (! $opt{configfile}) {
      if (! defined $opt{ioshost}) { pod2usage(-verbose => 1);exit -1};
      $ioshost=$opt{ioshost};
      $entry= $opt{entry} || 4711;
      print "SAA delete on $ioshost, entry number $entry\n" if($debug);
      $community=$opt{community} || "public";
      }
   else {
      if(!$opt{configfile}) { pod2usage(-verbose => 1);exit -1};
      my $file=$configpath.$opt{configfile};
      print "reading configuration $file for SAA delete\n" if($debug);
      if (! eval `cat $file`) {
         print "error reading configuration file $file !\n";
         exit -1;
         }
      }
   $Sioshost = $community."@".$ioshost.":::::2";

   print "$Sioshost $entry\n";

   &SAADelete();
}

sub fetchstart($) {
   my $restart=shift;

   if ($restart) {
      GetOptions(\%opt, 'man','help', 'ioshost=s', 'community=s', 'entry=i', 
         'configfile=s', 'history',
         'debug', 'cache=i', 'log=s', 'line0|0=s',
         'line1|1=s',  'line2|2=s', 'line3|3=s', 'line4|4=s',
         'line5|5=s', 'line6|6=s', 'line7|7=s', 'line8|8=s',
         'line9|9=s') or pod2usage(2);
      }
   else {
         GetOptions(\%opt, 'man','help', 'configfile=s',
         'debug', 'cache=i', 'log=s', 'line0|0=s',
         'line1|1=s',  'line2|2=s', 'line3|3=s', 'line4|4=s',
         'line5|5=s', 'line6|6=s', 'line7|7=s', 'line8|8=s',
         'line9|9=s') or pod2usage(2);
   }

   if($opt{man})      { pod2usage(-verbose => 2);exit  0};
   if($opt{help})     { pod2usage(-verbose => 1);exit  0};

   $debug= 1==1 if $opt{debug} ;
   my $log= 1==1 if $opt{log};
   my $logfile;
   if ($log) {
      $logfile=">>".$tmppath.$opt{log};
      print "logfile is $logfile \n" if ($debug);
   }

   if ($restart && ! $opt{configfile}) {
      if (! defined $opt{ioshost}) { pod2usage(-verbose => 1);exit -1};
      $ioshost=$opt{ioshost};
      $entry= $opt{entry} || 4711;
      print "SAA restart on $ioshost, entry number $entry\n" if($debug);
      $community=$opt{community} || "public";
      $history=$opt{history};
      }
   else {
      if(!$opt{configfile}) { pod2usage(-verbose => 1);exit -1};
      my $file=$configpath.$opt{configfile};
      print "reading configuration $file\n" if($debug);
      if (! eval `cat $file`) {
         print "error reading configuration file $file !\n";
         exit -1;
         }
      # check whether config file contains history configuration
      $history=0==1;
      for (my $i = 0; $i <= $#myparms; $i++) {
         if ($myparms[$i] =~ /^rttMon.*/) {
            $myparms[$i].=".".$entry;
            if ($myparms[$i] =~ /^rttMonHistoryAdminFilter.*/) {
               $history=0==0;
            }
         }
      }
   }

   $Sioshost = $community."@".$ioshost.":::::2";

   ########################################################         
   # end of parameter / config reading
   # get data now

   my $cache=$opt{cache} || 270;
   my $cachefile=$tmppath.$ioshost."-".$entry;
   my $cacheupdate;

   $n=0;
   $sum=0;
   $sum2=0;
   $max=0; 
   $min=999999999999;
   $drops=0;
  
   my $cacheage= -M $cachefile; 
   $cacheage*=24*60*60 if $cacheage;
   if (-e $cachefile && $cacheage<$cache) {
      print "reading cache $cachefile (age is $cacheage secs)\n" if ($debug);
      # use cached data
      open (CACHE,"<".$cachefile);
      if ($history) {
         $drops=<CACHE>; chop $drops if($drops);
         while (<CACHE>) { 
            chop; 
            push @times,$_; 
            $sum+=$_;
            $sum2+=$_*$_;
            $min=$_ if ($min>$_);
            $max=$_ if ($max<$_);
            $n++;
         }
      }
      else {
         $n=<CACHE>; chop $ n if($n);
         $drops=<CACHE>; chop $drops if($drops);
         $sum=<CACHE>; chop $sum if($sum);
         $min=<CACHE>; chop $min if($min);
         $max=<CACHE>; chop $max if($max);
         $sum2=<CACHE>; chop $sum2 if($sum2);
      }
      close (CACHE);
      $cacheupdate=0==1;
   }
   else {
      $SNMP_Session::suppress_warnings =  10; # be silent

      print "SAAFetch " if ($debug); 
      if (! defined &SAAFetch()) {
         if ($restart && ! $opt{configfile}) {
            print "SAA entry $entry not found on $ioshost\n";
            exit -1;
            }
         else {
            &SAAStart();
            print ("SAA entry $entry configured on $ioshost\n") if ($debug);
            # print avoids MRTG errors
            print "$ERRNaN\n$ERRNaN\n";
            exit 0;
         }
      }
      $SNMP_Session::suppress_warnings = 0; # report errors

      $cacheupdate=0==0;
      if ($restart) {
         print "SAARestart " if ($debug); &SAARestart();
         }
      else {
         print "SAADelete " if ($debug); &SAADelete();
         print "SAAStart " if ($debug); &SAAStart();
      }
      print "-- SNMP done\n" if ($debug);
   }

   ########################################################         
   # do all calculations

   my $med=0;
   if ($history) {
      if ($n>0) {
         # determine median value
         @times = sort { $a <=> $b } @times;
         if ($#times % 2 != 0) {
            $med= @times[int($#times/2)+1];
            $med+=@times[int($#times/2)];
            $med/=2;
            }
         else {
            $med=@times[int($#times/2)];
         }
      }
   }
   else {
      $med=$ERRNaN;
   }

   my $avg=0; my $var=0; my $stddev=0; my $cov=0;
   if ($n>1) {
      $avg=$sum/$n;
      $var=($n*$sum2-$sum*$sum)/($n*($n-1));
      $stddev=sqrt($var);
      $cov=$stddev/$avg;
      }
   else {
     $avg=$sum;
     $var=0; $stddev=0; $cov=0;
   }

   my $ams=$avg-$stddev;
   my $aps=$avg+$stddev;
   my $dpr;
   my $dn=$n;
   my $ddrops=$drops;

   my $ava=1;
   my $avp=100;

   if (($drops+$n)>0) {
      $dpr=$drops/($drops+$n)*100 if (($drops+$n)>0);
      if ($n==0) {
         $avg=$med=$min=$max=$var=$stddev=$cov=$ams=$aps=$ERRAllLost;
         # setting it to 0 would be correct, but using a low value is easier to configure in MRTG (no Option withzeros required)
         $ava=0.0001;$avp=0.0001;
      } 
   }
   else {
      $dpr=$dn=$ddrops=$avg=$med=$min=$max=$var=$stddev=$cov=$ams=$aps=$ava=$ERRNaN;
   }

   ########################################################         
   # create output

   if ($log) {
      open (LOG, $logfile);
      $_=gmtime();
      print LOG "$_ $ioshost $entry "; 
      if ($cacheupdate){print LOG "measured ";} else {print LOG "cached ";} 
   }
   # print requested values in order indicated on CLI
   my $i=0;
   while ($i<=15) {
      my $k="line".$i;

      my $str;
      if (defined $opt{$k} && $opt{$k} ne "") {
         $_=$opt{$k};
         print LOG $_," " if ($log);
         {
            $str= $dn          , last if /^n/;
            $str= $ddrops      , last if /drp/;
            $str= $dpr         , last if /dpr/;
            $str= $ava         , last if /ava/;
            $str= $avp         , last if /avp/;
            $str= $med         , last if /med/;
            $str= $avg         , last if /avg/;
            $str= $min         , last if /min/;
            $str= $max         , last if /max/;
            $str= $var         , last if /var/;
            $str= $stddev      , last if /std/;
            $str= $cov         , last if /cov/;
            $str= $ams         , last if /ams/;
            $str= $aps         , last if /aps/;
            $str= $_;
         } 
      print "$str\n"; 
      print LOG $str," " if ($log);
      }
      $i++;
   } 
   if ($log) {
      print LOG "\n"; 
      close(LOG);
   }

   # write cache, if required
   if ($cacheupdate) {
      open (CACHE, ">".$cachefile);
      print "updating cache $cachefile\n" if ($debug);
      if ($history) {
         print CACHE "$drops\n";
         foreach my $tm (@times) {
            print CACHE "$tm\n";
         }
      }
      else {
         print CACHE "$n\n";
         print CACHE "$drops\n";
         print CACHE "$sum\n";
         print CACHE "$min\n";
         print CACHE "$max\n";
         print CACHE "$sum2\n";
      }
      close (CACHE);
   }

   ########################################################         
   # create "data dump" for debugging

   if ($debug) {
   print ("----------- Data Dump ----------\n");
      if ($history) {
         print ("History values:\n   ");
         foreach my $tmp (@times) {
            print ("$tmp ");
         }
         print ("\n");
      }

      print "Loss-related values:\n   n(n): $dn, drops(drp): $ddrops, drop percentage(dpr): $dpr%, availability(ava): $ava,  availability percentage(avp): $avp%\n";
      print "Latency-related values:\n   median(med): $med, average(avg): $avg, minimum(min): $min, maximum(max): $max, ";
      print "variance(var): $var, standard deviation(std): $stddev, coefficient of variation(cov):";
      print " $cov, avg-std(ams): ",$ams, ", avg+std(aps): ",$aps,"\n";

      print "Error definitions:\n",'   $ERRAllLost: ',$ERRAllLost,' $ERRNaN: ',$ERRNaN,"\n";
   }
}

# I am still not certain, whether SAADelete() or SAASafeDelete() causes less IOS problems.
# The comment on rttMonCtrlAdminStatus in the Cisco rttMon says
# This object can be set to 'destroy' from any value at any time."
# However, I started using SAASafeDelete() after a series of sensational IOS crashes, for which I couldn't pinpoint a cause.

sub SAADelete(){
   # delete any old config, set conceptual row to 6:destroy
   &snmpset($Sioshost, "rttMonCtrlAdminStatus.$entry",'integer',6);
}

sub SAASafeDelete(){
   # reset(1), orderlyStop(2), immediateStop(3), pending(4), inactive(5), active(6), restart(7)
   &snmpset($Sioshost, "rttMonCtrlOperState.$entry",'integer',3);

   # wait max 30 secs until entry transitions to inactive
   sleep(1);
   my $secs=0;
   (my $response) = &snmpget($Sioshost, "rttMonCtrlOperState.$entry");
   while ( ($response!=5) && ($secs++<=15) ) {
      (my $response) = &snmpget($Sioshost, "rttMonCtrlOperState.$entry");
      sleep(2);
   }

   # 1:active 2:notInService 3:notReady 4:createAndGo 5:createAndWait 6:destroy
   &snmpset($Sioshost, "rttMonCtrlAdminStatus.$entry",'integer',2);
   # delete any old config
   &snmpset($Sioshost, "rttMonCtrlAdminStatus.$entry",'integer',6);
}

sub SAAStart() {
   # 4:createAndGo 
   push @myparms, "rttMonCtrlAdminStatus.$entry",'integer',4;
   if (! defined &snmpset($Sioshost, @myparms)) {
      print ("SAA entry start failed !\n$ERRNaN\n");
      exit -1;
   }
}

sub SAARestart() {
   # clear measurement data, restart measurement 
   # reset(1), orderlyStop(2), immediateStop(3), pending(4), inactive(5), active(6), restart(7)
   &snmpset($Sioshost, "rttMonCtrlOperState.$entry", 'integer', 7);
}

sub SAAFetch() {
   if ($history) {
      # snmpmaptable walks two columns of rttMonHistoryCollectionTable
      # - "rttMonHistoryCollectionCompletionTime.$entry", 
      # - "rttMonHistoryCollectionSense.$entry"
      # The code in the sub() argument is executed for each index value snmptable walks

      my $result =snmpmaptable ($Sioshost,
         sub () {
            my ($index, $rtt, $status) = @_;
            if ($status==1){
               $rtt=$rtt/$scaleC;
               push @times, $rtt;
               $sum+=$rtt;
               $sum2+=$rtt*$rtt;
               $min=$rtt if ($min>$rtt);
               $max=$rtt if ($max<$rtt);
               $n++;
               }
            else {
                $drops++;
               }
            },
         "rttMonHistoryCollectionCompletionTime.$entry",
         "rttMonHistoryCollectionSense.$entry");

      return $result;
      }
   else {
      # Find out the time this entry was modified - modt, in order to use it as an index below.
      # Unfortunately this value only seems to be available as an OID - not as a value, i.e.
      # a snmpwalk/get-bulk-request must be used to walk an area containing the OID
      my $modt;
      (my $result) = snmpmaptable ($Sioshost,
         sub () {
            my ($index, $comp) = @_;
            if ($index=~/(\d+)\..*/){
            $modt=$1; $n=$comp}; },
      "rttMonStatsCaptureCompletions.$entry");
      if (! defined $modt || ! defined $result) {
         print "no stats found on $ioshost, entry $entry\n$ERRNaN\n" if ($debug);
         return undef;
      } 

      my $timout=0;
      my $noconns=0;
      my $sum2lo=0; 
      my $sum2hi=0;
      # now we know the modt (and n), so we can ask for specific statistical values collected
      my @ret=&snmpget($Sioshost, 
         "rttMonStatsCaptureSumCompletionTime.$entry.$modt.1.1.1",
         "rttMonStatsCaptureSumCompletionTime2Low.$entry.$modt.1.1.1",
         "rttMonStatsCaptureSumCompletionTime2High.$entry.$modt.1.1.1",
         "rttMonStatsCaptureCompletionTimeMax.$entry.$modt.1.1.1",
         "rttMonStatsCaptureCompletionTimeMin.$entry.$modt.1.1.1",
         "rttMonStatsCollectTimeouts.$entry.$modt.1.1",
         "rttMonStatsCollectNoConnections.$entry.$modt.1.1");
      return undef if (! @ret);
      ($sum, $sum2lo, $sum2hi, $max, $min, $timout, $noconns)= @ret;
      $drops=$timout+$noconns;
      $sum=$sum/$scaleC;
      $min=$min/$scaleC;
      $max=$max/$scaleC;
      $sum2 = ($sum2lo+$sum2hi*4294967296)/$scaleC/$scaleC;
   }
   return 0==0;
}

sub EncodeAddress($) {
# returns IP address in the format required by rttMon; works with DNS
# samples
#    my addr=EncodeAddress("10.12.22.123");
#    my addr=EncodeAddress("www.cisco.com");
   my $host=shift;
   $_=$host;
   if (!/^([0-9]|\.)+/) {
      (my $name, my $aliases, my $addrtype, my $length, my @addrs) = gethostbyname ($host);
      $host=join('.',(unpack("C4",$addrs[0])));
   }
   my @octets=split(/\./,$host);
   return pack ("CCCC", @octets);
}

# return 1 to indicate that module loaded ok..
1;

