/*
 * Copyright (c) 2002-2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: smtpc.c,v 1.167 2005/10/25 23:33:42 ca Exp $")

#include "sm/common.h"
#include "sm/assert.h"
#include "sm/error.h"
#include "sm/io.h"
#include "sm/sysexits.h"
#include "sm/str.h"
#include "sm/string.h"
#include "sm/limits.h"
#include "sm/net.h"
#include "statethreads/st.h"
#include "sm/ctype.h"
#include "sm/memops.h"
#include "sm/signal.h"
#include "sm/wait.h"
#include "sm/qmgrcomm.h"
#include "sm/hostname.h"
#include "sm/misc.h"
#include "smtpc.h"
#include "sm/da.h"
#include "sm/das.h"
#include "c2q.h"
#include "sm/log.h"
#include "sm/version.h"
#define SMTPC_LOG_DEFINES 1
#include "log.h"
#include "sm/resource.h"
#include "sm/smtpdef.h"
#include "sm/sm-conf.h"
#include "sm/sm-conf-prt.h"
#include "sm/sccnfdef.h"
#include "sm/confsetpath.h"

#if SM_HEAP_CHECK
extern SM_DEBUG_T SmHeapCheck;
# define HEAP_CHECK (SmHeapCheck > 0)
# define HEAP_REPORT do { if (HEAP_CHECK) sm_heap_report(smioerr, SmHeapCheck);} while (0)
#else /* SM_HEAP_CHECK */
# define HEAP_CHECK	false
# define HEAP_REPORT
#endif /* SM_HEAP_CHECK */

extern sm_stream_T SmStThrNetIO;

/*
**  Client configuration parameters
*/

/* Log files */
#define PID_FILE    "sc.pid"

/*
**  Number of file descriptors needed to handle one client session
**  1 client communication
**  1 data file
*/

#define FD_PER_THREAD 2

/*
**  Global data
*/

static sc_ctx_T		 Sc_ctx;	/* SMTPC context */

/* some of the following data should be in sc_ctx */

uint			 conns;
uint			 busy = 0;
ulong			 total = 0;

static int	 my_index = -1;	/* Current process index */
static pid_t	 my_pid = -1;	/* Current process pid   */

static pid_t	*vp_pids;	/* Array of VP pids */

static st_netfd_t	sig_pipe[2];	/* Signal pipe */

/* context for SMTPC/QMGR communication subsystem */
c2q_ctx_T	c2q_ctx;

/*
**  Configuration flags/parameters
*/

/* see below: sm_log_setfp_fd(..., Errfp, SMIOOUT_FILENO); */
static sm_file_T	*Errfp = smiolog;


#if 0
static bool rpools = false;	/* use rpools... */
#endif

/*
**  Forward declarations.
**  Many functions here just exit on error, change them to return
**  an error code instead.
*/

static void	 sc_usage(const char *progname);
static void	 sc_parse_arguments(sc_ctx_P _sc_ctx, int argc, char *argv[]);
#if 0
static void	 sc_start_daemon(void);
#endif
static void	 sc_set_thread_throttling(sc_ctx_P _sc_ctx);
static void	 sc_start_processes(sc_ctx_P _sc_ctx);
#if 0
static void	 wdog_sighandler(int signo);
#endif
static void	 sc_child_sighandler(int signo);
static void	 sc_install_sighandlers(void);
static void	 sc_start_threads(sc_ctx_P _sc_ctx);
static void	 sc_process_signals(sc_ctx_P _sc_ctx);

static void	 sc_signal(int sig, void (*handler)(int));

static sm_ret_T		 sc_init0(sc_ctx_P _sc_ctx);
static sm_ret_T		 sc_init1(sc_ctx_P _sc_ctx);
static uint		 get_nthr_id(sc_ctx_P _sc_ctx);
static void		 sc_dump_info(sc_ctx_P _sc_ctx);

sm_ret_T	 sc_hdl_session(sc_t_ctx_P _sc_t_ctx);
static sm_ret_T	 sc_timeout_session(sc_t_ctx_P _sc_t_ctx);
static sm_ret_T	 sc_shutdown_session(sc_t_ctx_P _sc_t_ctx, bool _fast);
static void	 sc_load_configs(sc_ctx_P _sc_ctx);

/*
**  Fatal error unrelated to a system call.
**  Print a message and terminate.
*/

static void
sc_err_quit(int ex_code, const char *fmt,...)
{
	va_list ap;

	va_start(ap, fmt);
	if (Sc_ctx.scc_lctx == NULL)
	{
		sm_io_vfprintf(smioerr, fmt, ap);
		sm_io_putc(smioerr, '\n');
	}
	else
	{
		sm_log_vwrite(Sc_ctx.scc_lctx,
			SC_LCAT_INIT, SC_LMOD_CONFIG,
			SM_LOG_ERR, 0,
			fmt, ap);
	}
	va_end(ap);
	exit(ex_code);
}

/*
**  MAIN -- SMTPC main
**
**	Parameters:
**		argc -- number of arguments
**		argv -- vector of arguments
**
**	Returns:
**		exit code
*/

int
main(int argc, char *argv[])
{
	sm_ret_T ret;
	char *prg;

	prg = argv[0];
	sm_memzero(&Sc_ctx, sizeof(Sc_ctx));
	if (getuid() == 0 || geteuid() == 0)
	{
		sc_err_quit(EX_USAGE, SM_DONTRUNASROOT, prg);
		exit(EX_USAGE);
	}

	ret = sc_init0(&Sc_ctx);
	if (sm_is_err(ret))
	{
		sm_log_write(Sc_ctx.scc_lctx,
			SC_LCAT_INIT, SC_LMOD_CONFIG,
			SM_LOG_ERR, 0,
			"sev=ERROR, func=main, initialization=failed, init0=%m"
			, ret);
		exit(sm_is_perm_err(ret) ? EX_SOFTWARE : EX_OSERR);
	}

	/* Parse command-line options */
	sc_parse_arguments(&Sc_ctx, argc, argv);

	if (Sc_ctx.scc_cnf.sc_cnf_debug > 1)
	{
		sm_log_write(Sc_ctx.scc_lctx,
			SC_LCAT_INIT, SC_LMOD_CONFIG,
			SM_LOG_DEBUG, 10,
			"sev=DBG, func=main, uid=%ld, gid=%ld, euid=%ld, egid=%ld\n"
			, (long) getuid(), (long) getgid(), (long) geteuid()
			, (long) getegid());
	}

	/* Allocate array of server pids */
	vp_pids = sm_zalloc(Sc_ctx.scc_cnf.sc_cnf_vp_count * sizeof(pid_t));
	if (vp_pids == NULL)
		sc_err_quit(EX_TEMPFAIL,
			"sev=ERROR, func=main, malloc=failed, error=%s",
			strerror(errno));

#if 0
	/* Start the daemon */
	if (SC_IS_CFLAG(&Sc_ctx, SCC_CFL_BACKGROUND))
		sc_start_daemon();
#endif /* 0 */

	/* Initialize the ST library */
	if (st_init() < 0)
		sc_err_quit(EX_OSERR,
			"sev=ERROR, func=main, initialization=failed, st_init=failed");

	/* Set thread throttling parameters */
	sc_set_thread_throttling(&Sc_ctx);

	ret = sc_init1(&Sc_ctx);
	if (sm_is_err(ret))
		sc_err_quit(sm_is_perm_err(ret) ? EX_SOFTWARE : EX_OSERR,
			"sev=ERROR, func=main, initialization=failed, sc_init1=%m"
			, ret);

	/* Start server processes (VPs) */
	sc_start_processes(&Sc_ctx);

	/* Turn time caching on */
	st_timecache_set(1);

	/* Install signal handlers */
	sc_install_sighandlers();

	/* Load configuration from config files */
	sc_load_configs(&Sc_ctx);

	/* Start all threads */
	sc_start_threads(&Sc_ctx);

	/* Become a signal processing thread */
	sc_process_signals(&Sc_ctx);

	/* NOTREACHED */
	return 1;
}

/*
**  SC_USAGE -- sc_usage
**
**	Parameters:
**		progname -- program name
**
**	Returns:
**		none (doesn't return)
*/

static void
sc_usage(const char *progname)
{
	sm_io_fprintf(smioerr,
		"Usage: %s [options]\n"
		"%s SMTP/LMTP client\n"
		"Possible options:\n"
		"-D                    Run in background\n"
		"-d n                  Debug level\n"
		"-f name               Name of configuration file [none]\n"
		"-h                    Print this message\n"
#if SM_HEAP_CHECK
		"-H n                  Set heap check level\n"
#endif
		"-i id                 Set id\n"
		"-L socket             Socket location for LMTP [%s]\n"
		"-N name               Name of smtpc section in configuration file to use []\n"
		"-O timeout            I/O timeout [%d]\n"
		"-p num_processes      Create specified number of processes\n"
		"-P port               Connect to port [%d]\n"
		"-S C=name             Directory for CDB [%s]\n"
		"-S q=name             Socket for communication with QMGR [%s]\n"
		"-t min_thr:max_thr    Specify thread limits\n"
		"-v loglevel           Set loglevel\n"
		"-V                    Show version\n"
		"-w timeout            Time to wait for QMGR to be ready\n"
		, progname
		, SMX_VERSION_STR
		, LMTPSOCK		/* -L */
		, SC_IO_TIMEOUT		/* -O */
		, SMTPC_PORT		/* -P */
		, Sc_ctx.scc_cnf.sc_cnf_cdb_base	/* -S C */
		, Sc_ctx.scc_cnf.sc_cnf_smtpcsock	/* -S q */
		);
	exit(EX_USAGE);
}

/*
**  SC_SHOWVERSION -- show version information
**
**	Parameters:
**		sc_ctx -- SMTP Client context
**		progname -- program name
**		cnf -- configuration file name
**		level -- how much detail?
**
**	Returns:
**		usual error code
*/

static sm_ret_T
sc_showversion(sc_ctx_P sc_ctx, const char *progname, const char *cnf, uint level)
{
	char *prefix;

	prefix = "";
	if (level >= SM_SV_RD_CONF)
		prefix = "# ";
	sm_io_fprintf(smioout, "%s%s: %s\n", prefix, progname, SMX_VERSION_STR);
	if (SM_SHOW_VERSION(level))
		sm_io_fprintf(smioout, "%sversion=0x%08x\n",
			prefix, SMX_VERSION);
#if SM_USE_TLS
	if (SM_SHOW_LIBS(level))
		(void) sm_tlsversionprt(smioout);
#endif
	if (SM_SHOW_OPTIONS(level))
	{
		sm_io_fprintf(smioout, "compile time options:\n"
#if SC_DEBUG
			"SC_DEBUG\n"
#endif
#if SM_USE_TLS
			"SM_USE_TLS\n"
#endif
			);
	}
	if (SM_SHOW_RD_CONF(level) && cnf != NULL && *cnf != '\0')
	{
		sm_io_fprintf(smioout,
			"# configuration-file=%s\n\n", cnf);
		(void) sm_conf_prt_conf(sc_global_defs,
				&(sc_ctx->scc_cnf), smioout);
	}
	else if (SM_SHOW_DFLT_CONF(level))
	{
		sm_io_fprintf(smioout, "# default configuration:\n\n");
		(void) sm_conf_prt_dflt(sc_global_defs, 0, smioout);
	}
	else if (SM_SHOW_ALL_CONF(level))
	{
		sm_io_fprintf(smioout,
			"# configuration including flags.\n"
			"# note: this data cannot be used as configuration file\n"
			"# but it shows the available flags.\n"
			);
		(void) sm_conf_prt_dflt(sc_global_defs, SMC_FLD_FLAGS, smioout);
	}
	sm_io_flush(smioout);
	return sm_error_perm(SM_EM_SMTPC, SM_E_DONTSTART);
}

/*
**  SC_PARSE_ARGUMENTS -- parse arguments
**
**	Parameters:
**		sc_ctx -- SMTPC context
**		argc -- number of arguments
**		argv -- vector of arguments
**
**	Returns:
**		none.
*/

static void
sc_parse_arguments(sc_ctx_P sc_ctx, int argc, char *argv[])
{
	extern char *optarg;
	int opt, h;
	uint u, showversion;
	char *c;

	showversion = 0;
	while ((opt = getopt(argc, argv, "Dd:F:f:H:hi:l:L:N:O:P:p:rS:t:Vv:w:"))
		!= -1)
	{
		switch (opt)
		{
		  case 'D':
			SC_SET_CFLAG(sc_ctx, SCC_CFL_BACKGROUND);
			break;
		  case 'd':
			sc_ctx->scc_cnf.sc_cnf_debug = atoi(optarg);
			break;
		  case 'F':
			sc_ctx->scc_cnf.sc_cnf_confflags = atoi(optarg);
			break;
		  case 'f':
			c = strdup(optarg);
			if (c == NULL)
			{
				sc_err_quit(EX_OSERR,
					"sev=ERROR, func=sc_parse_arguments, argument=\"%@s\", strdup=%m"
					, optarg, sm_err_temp(errno));
			}
			sc_ctx->scc_cnf.sc_cnf_conffile = c;
			h = sc_read_cnf(sc_ctx, c,
					&(sc_ctx->scc_cnf.sc_cnf_smc));
			if (h != 0)
				sc_err_quit(EX_USAGE,
					"sev=ERROR, func=sc_parse_arguments, status=invalid_configuration_file, error=%m"
					, h);
			break;

		  case 'H':
#if SM_HEAP_CHECK
			SmHeapCheck = atoi(optarg);
			if (SmHeapCheck < 0)
				sc_err_quit(EX_USAGE,
					"sev=ERROR, func=sc_parse_arguments, SmHeapCheck=%s, status=invalid_value"
					, optarg);
#endif /* SM_HEAP_CHECK */
			break;
		  case 'i':
			sc_ctx->scc_cnf.sc_cnf_id = strtol(optarg, &c, 10);
			break;
		  case 'l':
#if 0
			/* currently unused */
			sc_ctx->scc_cnf.sc_cnf_logdir = optarg;
#endif
			break;
		  case 'L':
			c = strdup(optarg);
			if (c == NULL)
				sc_err_quit(EX_OSERR,
					"sev=ERROR, func=sc_parse_arguments, argument=\"%@s\", strdup=%m"
					, optarg, sm_err_temp(errno));
			sc_ctx->scc_cnf.sc_cnf_lmtpsock = c;
			break;
		  case 'N':
			if (sc_ctx->scc_cnf.sc_cnf_conffile != NULL)
				sc_err_quit(EX_USAGE,
					"sev=ERROR, func=sc_parse_arguments, -N must be before -f");
			c = strdup(optarg);
			if (c == NULL)
			{
				sc_err_quit(EX_OSERR,
					"sev=ERROR, func=sc_parse_arguments, argument=\"%@s\", strdup=%m"
					, optarg, sm_err_temp(errno));
			}
			sc_ctx->scc_cnf.sc_cnf_section = c;
			break;

		  case 'O':
			h = atoi(optarg);
			if (h <= 0)
				sc_err_quit(EX_USAGE,
					"sev=ERROR, func=sc_parse_arguments, timeout=%@s, status=invalid_value"
					, optarg);
			sc_ctx->scc_cnf.sc_cnf_timeout = (sm_intvl_T) h;
			break;
		  case 'p':
			h = atoi(optarg);
			if (h < 1)
				sc_err_quit(EX_USAGE,
					"sev=ERROR, func=sc_parse_arguments, processes=%s, status=invalid_value"
					, optarg);
			sc_ctx->scc_cnf.sc_cnf_vp_count = h;
			break;
		  case 'P':
			sc_ctx->scc_cnf.sc_cnf_port = atoi(optarg);
			break;
		  case 'r':
#if 0
			rpools = true;
#endif
			break;

		  case 'S':
			if (optarg != NULL && ISALPHA(optarg[0])
			    && optarg[1] == '=' && optarg[2] != '\0')
				h = 2;
			else
			{
				sc_usage(argv[0]);
				/* NOTREACHED */
				break;	/* shut up stupid compiler */
			}
			c = NULL;
			switch (optarg[0])
			{
			  case 'C':
			  case 'q':
				c = strdup(optarg + h);
				if (c == NULL)
				{
					sc_err_quit(EX_OSERR,
						"sev=ERROR, func=sc_parse_arguments, argument=\"%@s\", strdup=%m"
						, optarg + h
						, sm_err_temp(errno));
				}
				break;
			  default:
				sc_usage(argv[0]);
				break;
			}
			switch (optarg[0])
			{
			  case 'C':
				sc_ctx->scc_cnf.sc_cnf_cdb_base = c;
				break;
			  case 'q':
				sc_ctx->scc_cnf.sc_cnf_smtpcsock = c;
				break;
			}
			c = NULL;
			break;

		  case 't':
			u = strtoul(optarg, &c, 10);
			if (*c++ == ':')
				MAX_THREADS(sc_ctx) = strtoul(c, NULL, 10);
			if (u <= 0 || MAX_THREADS(sc_ctx) <= 0)
				sc_err_quit(EX_USAGE,
					"sev=ERROR, func=sc_parse_arguments, threads=%s, status=invalid_value"
					, optarg);
			MAX_WAIT_THREADS(sc_ctx) = u;
			break;
		  case 'v':
			sc_ctx->scc_cnf.sc_cnf_loglevel =
				(uint) atoi(optarg);
			break;
		  case 'V':
			++showversion;
			break;
		  case 'w':
			sc_ctx->scc_cnf.sc_cnf_wait4srv =
				(uint) atoi(optarg);
			break;
		  case 'h':
		  case '?':
		  default:
			sc_usage(argv[0]);
		}
	}

	if (showversion > 0)
	{
		(void) sc_showversion(sc_ctx, argv[0],
			sc_ctx->scc_cnf.sc_cnf_conffile, showversion);
		exit(0);
	}
#if 0
	if (sc_ctx->scc_cnf.sc_cnf_logdir == NULL &&
	    SC_IS_CFLAG(sc_ctx, SCC_CFL_BACKGROUND))
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_INIT, SC_LMOD_CONFIG,
			SM_LOG_ERR, 0,
			"sev=ERROR, func=sc_parse_arguments, status=logging_directory_is_required");
		sc_usage(argv[0]);
	}
#endif /* 0 */

	/* Already checked at startup */
	if (getuid() == 0 || geteuid() == 0)
		sc_err_quit(EX_USAGE,
			"sev=ERROR, func=sc_parse_arguments, Sorry Dave, I can't do that.");

	if (sc_ctx->scc_cnf.sc_cnf_vp_count <= 0)
		sc_ctx->scc_cnf.sc_cnf_vp_count = 1;

}

/*
**  SC_SET_THREAD_THROTTLING -- set appropriate values for number of threads
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		none.
*/

static void
sc_set_thread_throttling(sc_ctx_P sc_ctx)
{
	uint max_wait_threads;

	/*
	**  For simplicity, the minimal size of thread pool is considered
	**  as a maximum number of spare threads (max_wait_threads) that
	**  will be created upon server startup. The pool size can grow up
	**  to the max_threads value. Note that this is a per listening
	**  socket limit. It is also possible to limit the total number of
	**  threads for all sockets rather than impose a per socket limit.
	**
	**  Calculate total values across all processes.
	**  All numbers are per listening socket.
	*/

	/* just a shorter name; will be reassigned before return */
	max_wait_threads = MAX_WAIT_THREADS(sc_ctx);
	if (max_wait_threads == 0)
		max_wait_threads = MAX_WAIT_THREADS_DEFAULT *
				sc_ctx->scc_cnf.sc_cnf_vp_count;

	/* Assuming that each client session needs FD_PER_THREAD fds */
	if (MAX_THREADS(sc_ctx) == 0)
		MAX_THREADS(sc_ctx) = (st_getfdlimit() *
					sc_ctx->scc_cnf.sc_cnf_vp_count) /
					FD_PER_THREAD;
	if (max_wait_threads > MAX_THREADS(sc_ctx))
		max_wait_threads = MAX_THREADS(sc_ctx);

	/*
	**  Now calculate per-process values.
	*/

	if (max_wait_threads % sc_ctx->scc_cnf.sc_cnf_vp_count)
		max_wait_threads = max_wait_threads /
				sc_ctx->scc_cnf.sc_cnf_vp_count + 1;
	else
		max_wait_threads = max_wait_threads /
				sc_ctx->scc_cnf.sc_cnf_vp_count;
	if (MAX_THREADS(sc_ctx) % sc_ctx->scc_cnf.sc_cnf_vp_count)
		MAX_THREADS(sc_ctx) = MAX_THREADS(sc_ctx) /
				sc_ctx->scc_cnf.sc_cnf_vp_count + 1;
	else
		MAX_THREADS(sc_ctx) = MAX_THREADS(sc_ctx) /
				sc_ctx->scc_cnf.sc_cnf_vp_count;

	if (MIN_WAIT_THREADS(sc_ctx) > max_wait_threads)
		MIN_WAIT_THREADS(sc_ctx) = max_wait_threads;

	MAX_WAIT_THREADS(sc_ctx) = max_wait_threads;
}

/*
**  SC_START_PROCESSES -- start processes
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		none
*/

static void
sc_start_processes(sc_ctx_P sc_ctx)
{
	(void)sc_ctx;
	my_index = 0;
	my_pid = getpid();
	return;
}

/*
**  SC_INSTALL_SIGHANDLERS -- install sighandlers
**
**	Parameters:
**		none
**
**	Returns:
**		none
*/

static void
sc_install_sighandlers(void)
{
	sigset_t mask;
	int p[2];

	/* Create signal pipe */
	if (pipe(p) < 0)
		sc_err_quit(EX_OSERR,
			"sev=ERROR, func=sc_install_sighandlers, index=%d, pid=%d, pipe()=failed, error=%m"
			, my_index, my_pid, sm_err_temp(errno));
	if ((sig_pipe[0] = st_netfd_open(p[0])) == NULL ||
	    (sig_pipe[1] = st_netfd_open(p[1])) == NULL)
		sc_err_quit(EX_OSERR,
			"sev=ERROR, func=sc_install_sighandlers, index=%d, pid=%d, st_netfd_open(pipe)=failed, error=%m"
			 , my_index, my_pid, sm_err_temp(errno));

	/* Install signal handlers */
	sc_signal(SIGTERM, sc_child_sighandler);	/* terminate */
	sc_signal(SIGHUP,  sc_child_sighandler);	/* restart   */
	sc_signal(SIGUSR1, sc_child_sighandler);	/* dump info */
	sc_signal(SIGUSR2, sc_child_sighandler);	/* dump info */

	/* Unblock signals */
	sigemptyset(&mask);
	sigaddset(&mask, SIGTERM);
	sigaddset(&mask, SIGHUP);
	sigaddset(&mask, SIGUSR1);
	sigaddset(&mask, SIGUSR2);
	sigprocmask(SIG_UNBLOCK, &mask, NULL);
}

/*
**  SC_CHILD_SIGHANDLER -- signal handler for children
**
**	Parameters:
**		signo -- signal number
**
**	Returns:
**		none
*/

static void
sc_child_sighandler(int signo)
{
	int err, fd;

	err = errno;
	fd = st_netfd_fileno(sig_pipe[1]);

	/* write() is async-safe */
	if (write(fd, &signo, sizeof(int)) != sizeof(int))
		sc_err_quit(EX_OSERR,
			"sev=ERROR, func=sc_child_sighandler, index=%d, pid=%d, func=ss_child_sighandler, write=failed, error=%m"
			, my_index, my_pid, sm_err_temp(errno));
	errno = err;
}

/*
**  SC_PROCESS_SIGNALS -- signal processing thread.
**	Note: this is not called by a signal handler, hence any function
**		can be used here.
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		none
*/

static void
sc_process_signals(sc_ctx_P sc_ctx)
{
	int signo;
	sm_ret_T ret;

	for (;;)
	{
		/* Read the next signal from the signal pipe */
		if (st_read(sig_pipe[0], &signo, sizeof(int), (st_utime_t)-1)
		    != sizeof(int))
			sc_err_quit(EX_OSERR,
				"sev=ERROR, func=sc_process_signals, index=%d, pid=%d, func=ss_process_signals, st_read()=failed, error=%m"
				, my_index, my_pid, sm_err_temp(errno));

		switch (signo)
		{
		  case SIGHUP:
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INTERN, SC_LMOD_SIGNAL,
				SM_LOG_INFO, 9,
				"sev=INFO, func=sc_process_signals, index=%d, pid=%d, signal=SIGHUP, action=reloading_configuration"
				, my_index, my_pid);

			sc_load_configs(&Sc_ctx);
			break;
		  case SIGTERM:
			/* "signal" shutdown to threads */
			SC_SET_FLAG(&Sc_ctx, SCC_FL_SHUTDOWN);

			/*
			**  Terminate ungracefully since it is generally not
			**  known how long it will take to gracefully complete
			**  all client sessions.
			*/

			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INTERN, SC_LMOD_SIGNAL,
				SM_LOG_INFO, 9,
				"sev=INFO, func=sc_process_signals, index=%d, pid=%d, signal=SIGTERM, action=terminating"
				, my_index, my_pid);
			HEAP_REPORT;
			(void) sm_log_destroy(Sc_ctx.scc_lctx);
			exit(0);
		  case SIGUSR1:
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INTERN, SC_LMOD_SIGNAL,
				SM_LOG_INFO, 9,
				"sev=INFO, func=sc_process_signals, index=%d, pid=%d, signal=SIGUSR1, action=printinfo"
				, my_index, my_pid);

			sc_dump_info(&Sc_ctx);
			HEAP_REPORT;
			break;
		  case SIGUSR2:
			/* Reopen log files - needed for log rotation */
			ret = sm_log_reopen(Sc_ctx.scc_lctx);
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INTERN, SC_LMOD_SIGNAL,
				SM_LOG_INFO, 9,
				"sev=INFO, func=sc_process_signals, index=%d, pid=%d, signal=SIGUSR2, action=reopened_logfiles"
				, my_index, my_pid);

			break;
		  default:
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INTERN, SC_LMOD_SIGNAL,
				SM_LOG_INFO, 8,
				"sev=INFO, func=sc_process_signals, index=%d, pid=%d, signal=%d, action=ignored"
				, my_index, my_pid, signo);
		}
	}

	/* NOTREACHED */
	return;
}

/*
**  SC_START_THREADS -- start threads
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		none; may exit if no threads have been started.
*/

static void
sc_start_threads(sc_ctx_P sc_ctx)
{
	uint n, threads;

	/* Create connections handling threads */
	sm_log_write(sc_ctx->scc_lctx,
		SC_LCAT_INIT, SC_LMOD_INTERN,
		SM_LOG_INFO, 10,
		"sev=INFO, func=sc_start_threads, index=%d, pid=%d, threads=%d"
		, my_index, my_pid, MAX_WAIT_THREADS(sc_ctx));
	WAIT_THREADS(sc_ctx) = 0;
	IDLE_THREADS(sc_ctx) = 0;
	BUSY_THREADS(sc_ctx) = 0;
	MAXUSED_THREADS(sc_ctx) = 0;
	RQST_COUNT(sc_ctx) = 0;
#if SC_STATS
	MAIL_COUNT(sc_ctx) = 0;
	RCPT_COUNT(sc_ctx) = 0;
	TA_CNT_OK(sc_ctx) = 0;
	RCPT_CNT_OK(sc_ctx) = 0;
#endif /* SC_STATS */
	threads = 0;
	for (n = 0; n < MAX_WAIT_THREADS(sc_ctx); n++)
	{
		if (st_thread_create(sc_hdl_requests, (void *) sc_ctx, 0, 0)
			    == NULL)
		{
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INIT, SC_LMOD_INTERN,
				SM_LOG_ERR, 2,
				"sev=ERROR, func=sc_start_threads, index=%d, pid=%d, wait=%u, idle=%u, busy=%u, st_thread_create()=failed"
				, my_index, my_pid
				, WAIT_THREADS(sc_ctx)
				, IDLE_THREADS(sc_ctx)
				, BUSY_THREADS(sc_ctx)
				);
		}
		else
		{
			WAIT_THREADS(sc_ctx)++;
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INIT, SC_LMOD_INTERN,
				SM_LOG_INFO, 14,
				"sev=INFO, func=sc_start_threads, index=%d, pid=%d, wait=%u, idle=%u, busy=%u, sc_hdl_requests=created"
				, my_index, my_pid
				, WAIT_THREADS(sc_ctx)
				, IDLE_THREADS(sc_ctx)
				, BUSY_THREADS(sc_ctx)
				);
			threads++;
		}
	}
	if (threads == 0)
	{
		c2q_stop(&c2q_ctx);
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_INIT, SC_LMOD_INTERN,
			SM_LOG_ERR, 0,
			"sev=ERROR, func=sc_start_threads, status=no_thread_started__terminating");
		exit(EX_OSERR);
	}
}

/*
**  GET_NTHR_ID -- get a free thread id
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		<MAX_THREADS(sc_ctx): free thread id
**		>=MAX_THREADS(sc_ctx): no unused thread available
**			Notice: maybe use >=0: ok, <0: error?
*/

static uint
get_nthr_id(sc_ctx_P sc_ctx)
{
	uint i;

	for (i = 0; i < MAX_THREADS(sc_ctx); i++)
	{
		if ((sc_ctx->scc_scts)[i] == NULL)
			break;
	}
	return i;
}

/*
**  SC_INIT0 -- initialize SMTPC context
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sc_init0(sc_ctx_P sc_ctx)
{
	sm_ret_T ret;

	SM_REQUIRE(sc_ctx != NULL);
	ret = sm_log_create(NULL, &(sc_ctx->scc_lctx), &(sc_ctx->scc_lcfg));
	if (sm_is_err(ret))
		return ret;
	ret = sm_log_setfp_fd(sc_ctx->scc_lctx, Errfp, SMIOLOG_FILENO);
	if (sm_is_err(ret))
		return ret;

	sc_ctx->scc_cnf.sc_cnf_cdb_base = "";
	sc_ctx->scc_cnf.sc_cnf_smtpcsock = smsmtpcsock;
	sc_ctx->scc_cnf.sc_cnf_timeout = SC_IO_TIMEOUT;
	sc_ctx->scc_cnf.sc_cnf_lmtpsock = LMTPSOCK;
	sc_ctx->scc_cnf.sc_cnf_port = SMTPC_PORT;
	sc_ctx->scc_cnf.sc_cnf_vp_count = 1;
	sc_ctx->scc_cnf.sc_cnf_loglevel = UINT_MAX;
	sc_ctx->scc_cnf.sc_cnf_wait4srv = 1;
	sc_ctx->scc_cnf.sc_cnf_id = SMTPC_ID;
	MAX_WAIT_THREADS(sc_ctx) = 0;
	MIN_WAIT_THREADS(sc_ctx) = 2;

#if SM_USE_TLS
	ret = sm_tlsversionok();
	if (sm_is_err(ret))
		return ret;
#endif /* SM_USE_TLS */

	return ret;
}

/*
**  SC_INIT1 -- initialize SMTPC context
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sc_init1(sc_ctx_P sc_ctx)
{
	sm_ret_T ret;
	size_t l;

	SM_REQUIRE(sc_ctx != NULL);
	ret = sm_log_setdebuglevel(sc_ctx->scc_lctx,
			sc_ctx->scc_cnf.sc_cnf_loglevel);
	if (sm_is_err(ret))
		return ret;
	l = MAX_THREADS(sc_ctx) * sizeof(*(sc_ctx->scc_scts));
	sc_ctx->scc_scts = sm_zalloc(l);
	if (sc_ctx->scc_scts == NULL)
		return sm_error_temp(SM_EM_SMTPC, ENOMEM);

	ret = sm_myhostname(&(sc_ctx->scc_hostname));
	if (sm_is_err(ret))
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_INIT, SC_LMOD_CONFIG,
			SM_LOG_ERR, 1,
			"sev=ERROR, func=sc_init1, status=cannot_determine_my_hostname, ret=%m"
			, ret);
		return ret;
	}

	ret = cdb_start(sc_ctx->scc_cnf.sc_cnf_cdb_base,
			&(sc_ctx->scc_cdb_ctx));
	if (sm_is_err(ret))
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_INIT, SC_LMOD_CONFIG,
			SM_LOG_ERR, 1,
			"sev=ERROR, func=sc_init1, cdb_start=%m", ret);
		return ret;
	}

	ret = c2q_init(sc_ctx, &c2q_ctx, sc_ctx->scc_cnf.sc_cnf_id,
			MAX_THREADS(sc_ctx));
	if (sm_is_err(ret))
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_INIT, SC_LMOD_CONFIG,
			SM_LOG_ERR, 1,
			"sev=ERROR, func=sc_init1, c2q_init=%m", ret);
		return ret;
	}
#if SM_USE_TLS
	ret = sm_init_tls_library(&(sc_ctx->scc_tlsl_ctx));
	if (sm_is_success(ret))
	{
		const char *cert, *key, *dsa_cert, *dsa_key, *certpath, *cacert;
		char confdir[PATH_MAX];

		ret = sm_dirname(sc_ctx->scc_cnf.sc_cnf_conffile, confdir,
			sizeof(confdir));
		if (sm_is_err(ret))
		{
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INIT, SC_LMOD_CONFIG,
				SM_LOG_ERR, 1,
				"sev=ERROR, func=sc_init1, sm_dirname=%m", ret);
			return ret;
		}

		SM_GEN_MAP_PATH(cert, sc_ctx->scc_cnf.sc_cnf_cert_abs, confdir,
			sc_ctx->scc_cnf.sc_cnf_cert, SM_MTA_CERT, enomem);
		SM_GEN_MAP_PATH(key, sc_ctx->scc_cnf.sc_cnf_key_abs, confdir,
			sc_ctx->scc_cnf.sc_cnf_key, SM_MTA_KEY, enomem);
		SM_GEN_MAP_PATH(dsa_cert, sc_ctx->scc_cnf.sc_cnf_dsa_cert_abs,
			confdir, sc_ctx->scc_cnf.sc_cnf_dsa_cert, NULL, enomem);
		SM_GEN_MAP_PATH(dsa_key, sc_ctx->scc_cnf.sc_cnf_dsa_key_abs,
			confdir, sc_ctx->scc_cnf.sc_cnf_dsa_key, NULL, enomem);
		SM_GEN_MAP_PATH(certpath, sc_ctx->scc_cnf.sc_cnf_certpath_abs,
			confdir, sc_ctx->scc_cnf.sc_cnf_certpath, SM_MTA_CERTP,
			enomem);
		SM_GEN_MAP_PATH(cacert, sc_ctx->scc_cnf.sc_cnf_cacert_abs,
			confdir, sc_ctx->scc_cnf.sc_cnf_cacert, SM_MTA_CACERT,
			enomem);
		ret = sm_inittls(sc_ctx->scc_tlsl_ctx, &sc_ctx->scc_ssl_ctx,
			TLS_I_CLT, false, cert, key, dsa_cert, dsa_key,
			certpath, cacert, NULL);
		if (sm_is_success(ret))
			SC_SET_FLAG(sc_ctx, SCC_FL_TLS_OK);
		else
		{
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_INIT, SC_LMOD_CONFIG,
				SM_LOG_INFO, 9,
				"sev=INFO, func=sc_init1, sm_inittls=%m", ret);
		}
	}
	ret = SM_SUCCESS;	/* fake it */
#endif /* SM_USE_TLS */

	SC_SET_FLAG(sc_ctx, SCC_FL_INIT);
	return ret;

#if SM_USE_TLS
  enomem:
	return sm_error_temp(SM_EM_SMTPC, ENOMEM);
#endif
}

/*
**  SC_HDL_REQUESTS -- handle requests from QMGR (thread)
**
**	Parameters:
**		arg -- SMTPC context
**
**	Returns:
**		none (NULL)
*/

void *
sc_hdl_requests(void *arg)
{
	uint thr_id;
	int r;
	sc_ctx_P sc_ctx;
	sc_t_ctx_P sc_t_ctx;
	sm_ret_T ret;
	sc_sess_P sess;

	sess = NULL;
	sc_ctx = (sc_ctx_P) arg;
	SM_REQUIRE(sc_ctx != NULL);

	/* get a free thread id */
	thr_id = get_nthr_id(sc_ctx);
	if (thr_id >= MAX_THREADS(sc_ctx))
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 1,
			"sev=ERROR, func=sc_hdl_requests, process=%d, pid=%d, thr_id=%u, max_threads=%u, status=cannot_find_thread_slot"
			, my_index, my_pid, thr_id, MAX_THREADS(sc_ctx));
SM_ASSERT(0);
		WAIT_THREADS(sc_ctx)--;
		return NULL;		/* question: some error?? */
	}

	/* allocate thread context */
	ret = sc_t_ctx_new(sc_ctx, &sc_t_ctx, thr_id);
	if (sm_is_err(ret))
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 1,
			"sev=ERROR, func=sc_hdl_requests, process=%d, pid=%d, sc_t_ctx_new=%m"
			, my_index, my_pid, ret);
		WAIT_THREADS(sc_ctx)--;
		return NULL;		/* question: some error?? */
	}

	/* allocate session context */
	ret = sc_sess_new(&sess, sc_t_ctx);
	if (sm_is_err(ret))
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 1,
			"sev=ERROR, func=sc_hdl_requests, process=%d, pid=%d, sc_sess_new=%m"
			, my_index, my_pid, ret);
		(void) sc_t_ctx_free(sc_ctx, sc_t_ctx);
		WAIT_THREADS(sc_ctx)--;
		return NULL;		/* question: some error?? */
	}

	sc_t_ctx->sct_status = SC_T_FREE;
	if (sc_ctx->scc_cnf.sc_cnf_debug > 3)
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_INFO, 14,
			"sev=INFO, func=sc_hdl_requests, thr_id=%u, status=FREE, sc_t_ctx->sct_sess=%p, waitthr=%u, busythr=%u"
			, thr_id, sc_t_ctx->sct_sess, WAIT_THREADS(sc_ctx)
			, BUSY_THREADS(sc_ctx));
	}

	/* WAIT_THREADS(sc_ctx)++; */

	/* notify c2q thread that we are ready to go */
	if (SC_IS_FLAG(sc_ctx, SCC_FL_INIT) &&
	    !SC_IS_FLAG(sc_ctx, SCC_FL_COMMOK))
	{
		SC_SET_FLAG(sc_ctx, SCC_FL_COMMOK);
		r = st_cond_signal(c2q_ctx.c2q_cond_rd);
		if (r == 0)
			SC_SET_FLAG(sc_ctx, SCC_FL_NOTIFIED);
		else
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_ERR, 4,
				"sev=ERROR, func=sc_hdl_requests, thr_id=%u, st_cond_signal=%x"
				, thr_id, r);
	}

	while (AVAIL_THREADS(sc_ctx) <= MAX_WAIT_THREADS(sc_ctx)
	       || sc_t_ctx->sct_status != SC_T_FREE)
	{
		/* wait for a request from QMGR (indirectly) */
		r = st_cond_timedwait(sc_t_ctx->sct_cond_rd,
				SEC2USEC(REQUEST_TIMEOUT));
		if (SC_IS_FLAG(sc_ctx, SCC_FL_SHUTDOWN))
		{
			/* shutdown session and terminate */
			ret = sc_shutdown_session(sc_t_ctx, false);
			SC_LEV_DPRINTF(sc_t_ctx->sct_sc_ctx, 1, (smioerr,
				"func=sc_hdl_requests, sc_t_ctx=%p, shutdown=%m\n"
				, sc_t_ctx, ret));
			break;
		}
		if (r == -1)
		{
			if (errno == EINTR)
				continue;
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_INFO,
				sc_t_ctx->sct_status == SC_T_BUSY ? 9 : 19,
				"sev=INFO, func=sc_hdl_requests, thr_id=%u, status=timeout_waiting_for_QMGR, thread_stat=%d"
				, thr_id, sc_t_ctx->sct_status);

			/*
			**  "race condition": the wait could have timed out,
			**  but the thread just got work, i.e., it is marked
			**  as busy by sc_rcb_from_qmgr().
			**  this seems like a side effect from the statethreads
			**  implementation: the thread is not activated as
			**  long as another thread is running.
			*/

			if (sc_t_ctx->sct_status != SC_T_BUSY)
			{
				ret = sc_timeout_session(sc_t_ctx);
				continue;
			}
		}

		/* alread set in sc_rcb_from_qmgr() */
		/* sc_t_ctx->sct_status = SC_T_BUSY; */

		WAIT_THREADS(sc_ctx)--;
		BUSY_THREADS(sc_ctx)++;
		if (MAXUSED_THREADS(sc_ctx) < BUSY_THREADS(sc_ctx))
			MAXUSED_THREADS(sc_ctx) = BUSY_THREADS(sc_ctx);
		SM_ASSERT(WAIT_THREADS(sc_ctx) >= IDLE_THREADS(sc_ctx));
		if (AVAIL_THREADS(sc_ctx) < MIN_WAIT_THREADS(sc_ctx)
		    && TOTAL_THREADS(sc_ctx) < MAX_THREADS(sc_ctx))
		{
			/* Create another spare thread */
			if (st_thread_create(sc_hdl_requests,
						(void *) sc_ctx, 0, 0) == NULL)
			{
				sm_log_write(sc_ctx->scc_lctx,
					SC_LCAT_CLIENT, SC_LMOD_CLIENT,
					SM_LOG_ERR, 1,
					"sev=ERROR, func=sc_hdl_requests, process=%d, pid=%d, status=cannot_create_thread"
					, my_index, my_pid);
			}
			else
			{
				WAIT_THREADS(sc_ctx)++;
				sm_log_write(sc_ctx->scc_lctx,
					SC_LCAT_INIT, SC_LMOD_INTERN,
					SM_LOG_INFO, 14,
					"sev=INFO, func=sc_hdl_requests, index=%d, pid=%d, wait=%u, idle=%u, busy=%u, sc_hdl_requests=created"
					, my_index, my_pid
					, WAIT_THREADS(sc_ctx)
					, IDLE_THREADS(sc_ctx)
					, BUSY_THREADS(sc_ctx)
					);
			}
		}

		ret = sc_hdl_session(sc_t_ctx);

/*  sess_error: */
		WAIT_THREADS(sc_ctx)++;
		BUSY_THREADS(sc_ctx)--;
		if (sm_is_err(ret))
		{
			/* complain? or did sc_hdl_session() do that? */
			break;	/* question: some error?? */
		}

		/* this thread is only free if there is no active session */
		if (sc_t_ctx->sct_sess == NULL ||
		    sc_t_ctx->sct_sess->scse_state == SCSE_ST_NONE)
			sc_t_ctx->sct_status = SC_T_FREE;
		else
		{
			IDLE_THREADS(sc_ctx)++;
			sc_t_ctx->sct_status = SC_T_IDLE;
		}
	}
	/* XXX preserve ret from possible error above? */

	ret = sc_sess_free(sc_t_ctx->sct_sess, sc_t_ctx);
	if (sm_is_err(ret))
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 4,
			"sev=ERROR, func=sc_hdl_requests, thr_id=%u, sc_sess_free=%m"
			, thr_id, ret);

	ret = sc_t_ctx_free(sc_ctx, sc_t_ctx);
	if (sm_is_err(ret))
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 4,
			"sev=ERROR, func=sc_hdl_requests, thr_id=%u, sc_t_ctx_free=%m"
			, thr_id, ret);

	if (sc_ctx->scc_cnf.sc_cnf_debug > 3)
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_INFO, 18,
			"sev=INFO, func=sc_hdl_requests, thr_id=%u, status=terminates, ret=%m, busythreads=%u, waitthreads=%u"
			, thr_id, ret, BUSY_THREADS(sc_ctx)
			, WAIT_THREADS(sc_ctx));
	WAIT_THREADS(sc_ctx)--;
	return NULL;
}

/*
**  HANDLE_SESSION -- handle SMTPC session/transaction
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sc_hdl_session(sc_t_ctx_P sc_t_ctx)
{
	uint32_t v, l, rt;
	sm_ret_T ret, res;
	bool gotport;
	off_t off;
	sc_sess_P sc_sess;
	sc_ta_P sc_ta;
	sm_str_P rcpt_pa;
	rcpt_idx_T rcpt_idx;
	sc_rcpt_P rcpt;
	sc_ctx_P sc_ctx;

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	SM_REQUIRE(sc_sess != NULL);
	sc_ctx = sc_t_ctx->sct_sc_ctx;

	SC_LEV_DPRINTF(sc_ctx, 3, (smioerr, "func=sc_hdl_session, id=%s, state=%x\n", sc_sess->scse_id, sc_sess->scse_state));
	ret = SM_SUCCESS;
	gotport = false;
	SCSE_CLR_FLAG(sc_sess, SCSE_FL_SND_ST);
	SCSE_SET_FLAG(sc_sess, SCSE_FL_SE_CLS2QMGR);

	/* always get server IP address */
	/* fixme: server IP address; must match structure later on! */
	ret = sm_rcb_get3uint32(sc_t_ctx->sct_rcb, &l, &rt, &v);
	if (sm_is_err(ret) || l != 4)
		goto error;
	if (rt == RT_Q2C_SRVPORT)
	{
		sc_sess->scse_rmt_addr.sin.sin_port = htons((short) v);
		gotport = true;
		ret = sm_rcb_get3uint32(sc_t_ctx->sct_rcb, &l, &rt, &v);
	}
	if (rt == RT_Q2C_DA_IDX)
	{
		sc_sess->scse_da_idx = v;
		ret = sm_rcb_get3uint32(sc_t_ctx->sct_rcb, &l, &rt, &v);
	}

	SC_LEV_DPRINTF(sc_ctx, 5, (smioerr, "func=sc_hdl_session, l=%d, rt=%x, v=%x, ret=%m\n", l, rt, v, ret));
	if (sm_is_err(ret) || l != 4 ||
	    (rt != RT_Q2C_SRVIPV4 && rt != RT_Q2C_LMTP))
		goto error;

	/*
	**  NOTE: v is used below, do NOT transfer data after RT_Q2C_SRVIPV4
	**  that is read before this section!
	*/

	if (sc_sess->scse_state == SCSE_ST_NEW)
	{
		/* get connection data */

		/*
		**  HACK ahead: address LMTP_IPV4_ADDR is "special",
		**  RT_Q2C_LMTP is not yet used!
		**  da_idx can be used to select the "protocol".
		*/

		if (rt == RT_Q2C_LMTP || v == LMTP_IPV4_ADDR
		    || sc_sess->scse_da_idx == DA_IDX_LMTP_UNIX)
		{
			sc_sess->scse_rmt_addr.sunix.sun_family = AF_UNIX;
			if (strlcpy(sc_sess->scse_rmt_addr.sunix.sun_path,
				sc_ctx->scc_cnf.sc_cnf_lmtpsock,
				sizeof(sc_sess->scse_rmt_addr.sunix.sun_path))
			    >= sizeof(sc_sess->scse_rmt_addr.sunix.sun_path))
				goto error;
#if HAVE_SOCK_UN_SUN_LEN
			sc_sess->scse_rmt_addr.sunix.sun_len =
				strlen(sc_ctx->scc_cnf.sc_cnf_lmtpsock);
#endif
			/* the following hacks should depend on da_idx! */

			/* HACK turn on LMTP */
			SCSE_SET_FLAG(sc_sess, SCSE_FL_LMTP);
			SCSE_SET_FLAG(sc_sess, SCSE_FL_LWR);

			/* HACK Add Return-Path: */
			SCSE_SET_FLAG(sc_sess, SCSE_FL_RETPATH);
		}
		else if (sc_ctx->scc_cnf.sc_cnf_sink != 0)
		{
			sc_sess->scse_rmt_addr.sin.sin_addr.s_addr =
				sc_ctx->scc_cnf.sc_cnf_sink;
			sc_sess->scse_rmt_addr.sin.sin_family = AF_INET;
#if 0
			/* should smar port override connect_to default port? */
			if (!gotport)
#endif
				sc_sess->scse_rmt_addr.sin.sin_port =
					htons(sc_ctx->scc_cnf.sc_cnf_port);
		}
		else
		{
			sc_sess->scse_rmt_addr.sin.sin_addr.s_addr = v;
			sc_sess->scse_rmt_addr.sin.sin_family = AF_INET;
			if (!gotport)
				sc_sess->scse_rmt_addr.sin.sin_port =
					htons(sc_ctx->scc_cnf.sc_cnf_port);
		}

		/* use LMTP? */
		if (sc_sess->scse_da_idx == DA_IDX_LMTP_UNIX ||
		    sc_sess->scse_da_idx == DA_IDX_LMTP_INET)
			SCSE_SET_FLAG(sc_sess, SCSE_FL_LMTP);

		/* port? more? */

		ret = sc_sess_open(sc_t_ctx);
		if (ret != SM_SUCCESS)
		{
			if (!SCSE_IS_FLAG(sc_sess, SCSE_FL_LOGGED))
			{
				sm_log_write(sc_ctx->scc_lctx,
					SC_LCAT_CLIENT, SC_LMOD_CLIENT,
					SM_LOG_WARN, 8,
					"sev=WARN, func=sc_hdl_session, id=%s, state=%d, flags=%x, sc_sess_open=%m"
					, sc_sess->scse_id, sc_sess->scse_state
					, sc_sess->scse_flags, ret);
				SCSE_SET_FLAG(sc_sess, SCSE_FL_LOGGED);
			}

			/* send reply codes back, HACK */
			ret = sc_c2q(sc_t_ctx, RT_C2Q_SESTAT, ret, &c2q_ctx);
			SCSE_CLR_FLAG(sc_sess, SCSE_FL_SE_CLS2QMGR);

			/* ret is used as return value for this function */
			goto error;
		}
	}
#if 0
	else
	{
		/* compare  v with sc_sess->scse_rmt_addr.sin.sin_addr.s_addr */
	}
#endif /* 0 */

	SC_LEV_DPRINTF(sc_ctx, 3, (smioerr, "func=sc_hdl_session, after open: state=%d, flags=%x, ret=%m\n", sc_sess->scse_state, sc_sess->scse_flags, ret));
	if (sc_sess->scse_state != SCSE_ST_OPEN)
	{
		/* session isn't open for some reason: close it and stop */
		goto closesess;
	}

	if (SCSE_IS_FLAG(sc_sess, SCSE_FL_CLOSE))
		goto closesess;

	/* read transaction data */

	sc_ta = sc_sess->scse_ta;
	SC_LEV_DPRINTF(sc_ctx, 3, (smioerr, "func=sc_hdl_session, da_sess=%s, ta=%p\n", sc_sess->scse_id, sc_ta));

	if (sc_ta == NULL)
		ret = sc_ta_new(&sc_ta, sc_sess);
	else
		ret = sc_ta_clr(sc_ta);
	if (sm_is_err(ret))
		goto closesess;
	SCTA_SET_STATE(sc_ta, SCTA_INIT);

	/* get transaction data */
	ret = sm_rcb_get2uint32(sc_t_ctx->sct_rcb, &l, &rt);
	SC_LEV_DPRINTF(sc_ctx, 4, (smioerr, "func=sc_hdl_session, ta: l=%d, rt=%x,  ret=%m\n", l, rt, ret));
	if (sm_is_err(ret) || l != SMTP_STID_SIZE ||
	    (rt != RT_Q2C_NTAID && rt != RT_Q2C_NTAIDB && rt != RT_Q2C_NTAIDDB
	     && rt != RT_Q2C_NTAIDD))
		goto abortsess;
	if (rt == RT_Q2C_NTAIDD)
		SCTA_SET_FLAG(sc_ta, SCTA_FL_DELAY);
	else if (rt == RT_Q2C_NTAIDB)
		SCTA_SET_FLAG(sc_ta, SCTA_FL_BOUNCE);
	else if (rt == RT_Q2C_NTAIDDB)
		SCTA_SET_FLAG(sc_ta, SCTA_FL_DBOUNCE);

	ret = sm_rcb_getn(sc_t_ctx->sct_rcb, (uchar *) sc_ta->scta_id,
			SMTP_STID_SIZE);
	if (sm_is_err(ret))
		goto abortsess;

	ret = sm_rcb_get2uint32(sc_t_ctx->sct_rcb, &l, &rt);
	if (sm_is_err(ret) || l != SMTP_STID_SIZE || rt != RT_Q2C_SSTAID)
		goto abortsess;
	ret = sm_rcb_getn(sc_t_ctx->sct_rcb, (uchar *) sc_ta->scta_ssta_id,
			SMTP_STID_SIZE);
	if (sm_is_err(ret))
		goto abortsess;

	ret = sm_rcb_get3off_t(sc_t_ctx->sct_rcb, &l, &rt, &off);
	SC_LEV_DPRINTF(sc_ctx, 4, (smioerr, "func=sc_hdl_session, where=msg_size, l=%d, rt=%x, off=%lu, ret=%m\n", l, rt, (ulong) off, ret));
	if (sm_is_err(ret) || l != SIZEOF_OFF_T || rt != RT_Q2C_SIZE_B)
		goto abortsess;
	sc_ta->scta_msg_sz_b = off;

	ret = sm_rcb_get2uint32(sc_t_ctx->sct_rcb, &l, &rt);
	SC_LEV_DPRINTF(sc_ctx, 4, (smioerr, "func=sc_hdl_session, where=mail, l=%d, rt=%x,  ret=%m\n", l, rt, ret));
	if (sm_is_err(ret) || rt != RT_Q2C_MAIL)
		goto abortsess;
	ret = sc_mail_new(sc_ta);
	if (sm_is_err(ret))
		goto abortsess;
	ret = sm_rcb_getnstr(sc_t_ctx->sct_rcb,
			&(sc_ta->scta_mail->scm_pa), l);
	if (sm_is_err(ret))
		goto abortsess;

	ret = sm_rcb_get2uint32(sc_t_ctx->sct_rcb, &l, &rt);
	SC_LEV_DPRINTF(sc_ctx, 4, (smioerr, "func=sc_hdl_session, where=cdb, l=%d, rt=%x,  ret=%m\n", l, rt, ret));
	if (sm_is_err(ret) ||
	    ((rt != RT_Q2C_CDBID) && (rt != RT_Q2C_CDBID_NO)
	     && (rt != RT_Q2C_CDBID_HDR))
	   )
		goto abortsess;
	ret = sm_rcb_getn0str(sc_t_ctx->sct_rcb, &(sc_ta->scta_cdb_id), l);
	if (sm_is_err(ret))
	{
SC_LEV_DPRINTF(sc_ctx, 4, (smioerr, "sev=ERROR, func=sc_hdl_session, where=cdb_id, l=%d, rt=%x, ret=%m\n", l, rt, ret));
		goto abortsess;
	}
	if (rt == RT_Q2C_CDBID_NO)
		SCTA_SET_FLAG(sc_ta, SCTA_FL_NO_BODY);
	else if (rt == RT_Q2C_CDBID_HDR)
		SCTA_SET_FLAG(sc_ta, SCTA_FL_HDR_ONLY);

	ret = sm_rcb_get3uint32(sc_t_ctx->sct_rcb, &l, &rt, &v);
	if (sm_is_err(ret) || l != 4 || rt != RT_Q2C_FLAGS)
		goto abortsess;
	sc_ta->scta_flags |= v;
SC_LEV_DPRINTF(sc_ctx, 1, (smioerr, "sev=INFO, func=sc_hdl_session, flags=%#x\n", v));

	do
	{
		ret = sm_rcb_get3uint32(sc_t_ctx->sct_rcb, &l, &rt, &v);
		SC_LEV_DPRINTF(sc_ctx, 4, (smioerr, "func=sc_hdl_session, where=rcptidx, l=%d, rt=%x, v=%d, ret=%m\n", l, rt, v, ret));
		if (sm_is_err(ret) || l != 4 || rt != RT_Q2C_RCPT_IDX)
			goto abortsess;
		rcpt_idx = v;

		ret = sm_rcb_get2uint32(sc_t_ctx->sct_rcb, &l, &rt);
		SC_LEV_DPRINTF(sc_ctx, 4, (smioerr, "func=sc_hdl_session, where=rcpt, l=%d, rt=%x,  ret=%m\n", l, rt, ret));
		if (sm_is_err(ret) || rt != RT_Q2C_RCPT)
			goto abortsess;

		ret = sm_rcb_getnstr(sc_t_ctx->sct_rcb, &rcpt_pa, l);
		if (sm_is_err(ret))
			goto abortsess;
		if (SCSE_IS_FLAG(sc_sess, SCSE_FL_LWR))
			sm_str2lower(rcpt_pa);

		/* add recipient to list */
		ret = sc_rcpts_new(sc_ta, rcpt_pa, rcpt_idx, &rcpt);
		if (sm_is_err(ret))
			goto abortsess;
		SC_LEV_DPRINTF(sc_ctx, 4, (smioerr,
			"func=sc_hdl_sess, ta-id=%s, rcptaddr=%N, l=%d, nrctps=%d\n",
			(char *) (sc_ta->scta_id), rcpt_pa, l,
			sc_ta->scta_rcpts_tot));

	/* Only one recipient allowed for bounces! */
	} while (!SM_RCB_ISEOB(sc_t_ctx->sct_rcb)
		 && !SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN));

	if (SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN))
	{
		ret = sm_rcb_get2uint32(sc_t_ctx->sct_rcb, &l, &rt);
		if (sm_is_err(ret) || rt != RT_Q2C_B_MSG)
			goto abortsess;

		ret = sm_rcb_getnstr(sc_t_ctx->sct_rcb, &sc_ta->scta_b_msg, l);
		if (sm_is_err(ret))
			goto abortsess;
	}
	else if (SCTA_IS_FLAG(sc_ta, SCTA_FL_NO_BODY|SCTA_FL_HDR_ONLY))
	{
		/*
		**  It's not a bounce but no body to send?
		**  Treat this as inconsistent for now.
		**  Later on this might become a "probe", but that should
		**  use an explicit flag so DATA isn't even tried.
		*/

		ret = sm_error_perm(SM_EM_SMTPC, SM_E_UNEXPECTED);
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 5,
			"sev=ERROR, func=sc_hdl_requests, da_sess=%s, da_ta=%s, status=no_bounce_and_no_body"
			, sc_sess->scse_id, sc_ta->scta_id);
		goto abortsess;
	}

	/* fake message size if it's a bounce without body */
	if (SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN) &&
	    SCTA_IS_FLAG(sc_ta, SCTA_FL_NO_BODY|SCTA_FL_HDR_ONLY))
	{
		sc_ta->scta_msg_sz_b = 0;
	}

	if (SCSE_IS_CAP(sc_sess, SCSE_CAP_SIZE) && sc_sess->scse_max_sz_b > 0
	    && sc_ta->scta_msg_sz_b > 0
	    && sc_ta->scta_msg_sz_b > sc_sess->scse_max_sz_b)
	{
		/* message too big */
		ret = SMTP_R_PERM;
		sc_ta->scta_err_state = DA_TA_ERR_SIZE_I;
	}
	else
	{
		/* repeat transactions */
		ret = sc_one_ta(sc_t_ctx);
	}

	/* need to deal with 421 */
	if (ret == SMTP_R_SSD || SCSE_IS_FLAG(sc_sess, SCSE_FL_ONE))
		SCSE_SET_FLAG(sc_sess, SCSE_FL_CLOSE);

	ret = sc_c2q(sc_t_ctx, RT_C2Q_TASTAT, ret, &c2q_ctx);
	if (!sm_is_err(ret))
		SCSE_SET_FLAG(sc_sess, SCSE_FL_SND_ST);
	else
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 5,
			"sev=ERROR, func=sc_hdl_session, da_sess=%s, da_ta=%s, ss_ta=%s, sc_c2q=%m"
			, sc_sess->scse_id, sc_ta->scta_id, sc_ta->scta_ssta_id
			, ret);

	if (SCSE_IS_FLAG(sc_sess, SCSE_FL_CLOSE))
	{
  closesess:
		{
			bool sendstatus;

			/* free transaction */
			sc_ta_free(sc_t_ctx->sct_sess->scse_ta);
			sc_ta = sc_t_ctx->sct_sess->scse_ta = NULL;

			/* close session */
			sendstatus = SCSE_IS_FLAG(sc_sess, SCSE_FL_SND_ST);
			SCSE_CLR_FLAG(sc_sess, SCSE_FL_SE_CLS2QMGR);
			ret = sc_sess_close(sc_t_ctx);
			sc_sess_clr(sc_sess);
			if (sendstatus)
			{
				res = sc_rcb_send(sc_t_ctx->sct_rcb, &c2q_ctx);
				if (sm_is_err(res))
				{
					sm_log_write(sc_ctx->scc_lctx,
						SC_LCAT_CLIENT, SC_LMOD_CLIENT,
						SM_LOG_ERR, 6,
						"sev=ERROR, func=sc_hdl_session, thr_id=%u, sc_rcb_send=%x"
						, sc_t_ctx->sct_thr_id
						, res);
					goto error;
				}
			}
			if (sm_is_err(ret))
				goto error2;
			/* sc_t_ctx->sct_status = SC_T_FREE; */
		}
	}
	else if (SCSE_IS_FLAG(sc_sess, SCSE_FL_SND_ST))
	{
		res = sc_rcb_send(sc_t_ctx->sct_rcb, &c2q_ctx);
		if (sm_is_err(res))
		{
			sm_log_write(sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_ERR, 6,
				"sev=ERROR, func=sc_hdl_session, sc_rcb_send=%x"
				, res);
			goto error;
		}
		SCSE_CLR_FLAG(sc_sess, SCSE_FL_SND_ST);
	}

	/* XXX send appropriate status code to QMGR! */

	(void) sm_rcb_close_decn(sc_t_ctx->sct_rcb);
	return ret;

  abortsess:
	sm_log_write(sc_ctx->scc_lctx,
		SC_LCAT_CLIENT, SC_LMOD_CLIENT,
		SM_LOG_ERR, 9,
		"sev=ERROR, func=sc_hdl_session, status=abort, l=%d, rt=%x, sc_ta=%p, ret=%m"
		, l, rt, sc_ta, ret);
	sc_ta_free(sc_ta);
	sc_ta = sc_t_ctx->sct_sess->scse_ta = NULL;
	(void) sc_sess_close(sc_t_ctx);
	sc_t_ctx->sct_status = SC_T_FREE;

  error:
	/* XXX send appropriate status code to QMGR! */
	sc_sess_clr(sc_sess);
	sc_t_ctx_clr(sc_t_ctx);
  error2:
	return ret;
}

/*
**  SC_TIMEOUT_SESSION -- timeout occurred while waiting for QMGR: close session
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sc_timeout_session(sc_t_ctx_P sc_t_ctx)
{
	sm_ret_T ret;
	sc_sess_P sc_sess;

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	if (sc_sess != NULL && sc_sess->scse_state != SCSE_ST_NONE)
	{
		/*
		**  XXX add a "last used" timestamp? close session only if a
		**  "session timeout" value is exceeded?
		*/

		if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug > 3)
		{
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_INFO, 17,
				"sev=INFO, func=sc_timeout_session, sess=%p, da_sess=%s"
				, sc_sess, sc_sess->scse_id);
		}

		if (SCSE_IS_FLAG(sc_sess, SCSE_FL_SE_CLS2QMGR))
		{
			sc_sess->scse_err_st = 0;
			ret = sc_c2q(sc_t_ctx, RT_C2Q_SECLSD,
				0 /* SESS_TIMEOUT? */, &c2q_ctx);
			SCSE_CLR_FLAG(sc_sess, SCSE_FL_SE_CLS2QMGR);
		}

		/* close session */
		ret = sc_sess_close(sc_t_ctx);
		if (sm_is_err(ret))
			goto error;
		sc_sess_clr(sc_sess);
	}
	if (sc_t_ctx->sct_status == SC_T_BUSY)
	{
		sc_t_ctx->sct_status = SC_T_FREE;
		WAIT_THREADS(sc_t_ctx->sct_sc_ctx)++;
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_WARN, 9,
			"sev=WARN, func=sc_timeout_session, sess=%p, waitthr=%u, busythr=%u, total=%u"
			, sc_sess, WAIT_THREADS(sc_t_ctx->sct_sc_ctx)
			, BUSY_THREADS(sc_t_ctx->sct_sc_ctx)
			, TOTAL_THREADS(sc_t_ctx->sct_sc_ctx));
	}
	sc_t_ctx_clr(sc_t_ctx);
	return SM_SUCCESS;

  error:
	sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
		SC_LCAT_CLIENT, SC_LMOD_CLIENT,
		SM_LOG_ERR, 8,
		"sev=ERROR, func=sc_timeout_session, ret=%m", ret);
	sc_t_ctx_clr(sc_t_ctx);
	return ret;
}

/*
**  SC_SHUTDOWN_SESSION -- shutdown session because the system is shutting down
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**		fast -- shutdown fast
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sc_shutdown_session(sc_t_ctx_P sc_t_ctx, bool fast)
{
	sm_ret_T ret;
	sc_sess_P sc_sess;

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	ret = SM_SUCCESS;
	if (!fast && sc_sess != NULL && sc_sess->scse_state != SCSE_ST_NONE)
	{
		/* close session */
		ret = sc_sess_close(sc_t_ctx);
		if (sm_is_err(ret))
		{
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_ERR, 7,
				"sev=ERROR, func=sc_shutdown_session, sc_sess_close=%x"
				, ret);
		}
	}
	sc_sess_clr(sc_sess);
	return ret;
}

/*
**  SC_LOAD_CONFIGS -- load configs; fixme: this does nothing right now.
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		none
*/

static void
sc_load_configs(sc_ctx_P sc_ctx)
{
	sm_log_write(sc_ctx->scc_lctx,
		SC_LCAT_INIT, SC_LMOD_CONFIG,
		SM_LOG_INFO, 12,
		"sev=INFO, func=sc_load_configs, index=%d, pid=%d, status=configuration_reload_not_yet_implemented"
		, my_index, my_pid);
}

/*
**  SC_SIGNAL -- install signal handler
**
**	Parameters:
**		sig -- signal
**		handler -- signal handler
**
**	Returns:
**		none
*/

static void
sc_signal(int sig, void (*handler)(int))
{
	struct sigaction sa;

	sa.sa_handler = handler;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags = 0;
	sigaction(sig, &sa, NULL);
}

/*
**  SC_DUMP_INFO -- print some status information
**
**	Parameters:
**		sc_ctx -- SMTPC context
**
**	Returns:
**		none
*/

static void
sc_dump_info(sc_ctx_P sc_ctx)
{
	char *buf;
	int len, s, l;
	ssize_t b;
	struct rusage rusage;

	s = 512;
	if ((buf = sm_malloc(s)) == NULL)
	{
		sm_log_write(sc_ctx->scc_lctx,
			SC_LCAT_INIT, SC_LMOD_CONFIG,
			SM_LOG_ERR, 1,
			"sev=ERROR, func=sc_dump_info, bytes=%d, malloc=failed",
			s);
		return;
	}

	len = sm_snprintf(buf, s, "\n\nProcess #%d (pid %d):\n", my_index,
			(int)my_pid);
	l = sm_snprintf(buf + len, s - len,
		"Max threads         %u\n"
		"Min waiting threads %u\n"
		"Max waiting threads %u\n"
		"Wait threads        %u\n"
		"Idle threads        %u\n"
		"Busy threads        %u\n"
		"Max used threads    %u\n"
		"Requests            %u\n"
#if SC_STATS
		"Transactions        %u\n"
		"Recipients          %u\n"
		"Transactions ok     %u\n"
		"Recipients ok       %u\n"
#endif /* SC_STATS */

		, MAX_THREADS(&(Sc_ctx))
		, MIN_WAIT_THREADS(&Sc_ctx)
		, MAX_WAIT_THREADS(&Sc_ctx)
		, WAIT_THREADS(&(Sc_ctx))
		, IDLE_THREADS(&(Sc_ctx))
		, BUSY_THREADS(&(Sc_ctx))
		, MAXUSED_THREADS(&(Sc_ctx))
		, RQST_COUNT(&(Sc_ctx))
#if SC_STATS
		, MAIL_COUNT(&(Sc_ctx))
		, RCPT_COUNT(&(Sc_ctx))
		, TA_CNT_OK(&(Sc_ctx))
		, RCPT_CNT_OK(&(Sc_ctx))
#endif /* SC_STATS */
		);
	if (l < s - len)
		len += l;

	l = getrusage(RUSAGE_SELF, &rusage);
	if (l == 0 && len < s)
	{
		l = sm_snprintf(buf + len, s - len,
			"ru_utime=   %7ld.%07ld\n"
			"ru_stime=   %7ld.%07ld\n"
			"ru_maxrss=  %7ld\n"
			"ru_ixrss=   %7ld\n"
			"ru_idrss=   %7ld\n"
			"ru_isrss=   %7ld\n"
			"ru_minflt=  %7ld\n"
			"ru_majflt=  %7ld\n"
			"ru_nswap=   %7ld\n"
			"ru_inblock= %7ld\n"
			"ru_oublock= %7ld\n"
			"ru_msgsnd=  %7ld\n"
			"ru_msgrcv=  %7ld\n"
			"ru_nsignals=%7ld\n"
			"ru_nvcsw=   %7ld\n"
			"ru_nivcsw=  %7ld\n"
			, rusage.ru_utime.tv_sec
			, rusage.ru_utime.tv_usec
			, rusage.ru_stime.tv_sec
			, rusage.ru_stime.tv_usec
			, rusage.ru_maxrss
			, rusage.ru_ixrss
			, rusage.ru_idrss
			, rusage.ru_isrss
			, rusage.ru_minflt
			, rusage.ru_majflt
			, rusage.ru_nswap
			, rusage.ru_inblock
			, rusage.ru_oublock
			, rusage.ru_msgsnd
			, rusage.ru_msgrcv
			, rusage.ru_nsignals
			, rusage.ru_nvcsw
			, rusage.ru_nivcsw
			);
		if (l < s - len)
			len += l;
	}
	sm_io_write(smioerr, (uchar *) buf, len, &b);
	sm_free_size(buf, s);
}
