/*
 * Copyright (c) 2002-2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: smclt.c,v 1.188 2005/10/28 05:50:52 ca Exp $")
#include "sm/assert.h"
#include "sm/error.h"
#include "sm/string.h"
#include "sm/str.h"
#include "sm/io.h"
#include "sm/ctype.h"
#include "sm/net.h"
#include "sm/cdb.h"
#include "sm/reccom.h"
#include "sm/da.h"
#include "sm/tls.h"
#include "sm/tlsbio.h"
#include "smtpc.h"
#include "log.h"

#include "statethreads/st.h"
#if SM_STATETHREADS_DEBUG
/* HACK for debugging, otherwise structs are hidden */
# include "statethreads/common.h"
#endif

static sm_ret_T	 sc_command(sc_t_ctx_P _sc_t_ctx, int _phase);

/*
**  macros for sc_rd_reply() phase
**  This is bit of a hack: it combines a state with a flag
*/

#define SC_PIPELINE_OK	0x0100		/* don't need to read reply now */
#define SC_PHASE_OTHER	(0|SC_PIPELINE_OK)	/* no special treatment */
#define SC_PHASE_INIT	1			/* initial 220 greeting */
#define SC_PHASE_EHLO	2			/* EHLO response */
#define SC_PHASE_TLS	3			/* STARTTLS response */
#define SC_PHASE_MAIL	(4|SC_PIPELINE_OK)	/* MAIL response */
#define SC_PHASE_RCPT	(5|SC_PIPELINE_OK)	/* RCPT response */
#define SC_PHASE_DATA	6			/* DATA response */
#define SC_PHASE_LDAT	7			/* LMTP DATA response */
#define SC_PHASE_DOT	8			/* response to final dot */
#define SC_PHASE_QUIT	9			/* QUIT */
#define SC_PHASE_NOREPLY	10		/* no reply requested */

#define sc_pipeline_ok(phase)	(((phase) & SC_PIPELINE_OK) != 0)

/* 3 digits reply code plus ' ' or '-' */
#define SMTP_REPLY_OFFSET	4

/*
**  SC_EHLO_OPTIONS -- read server capabilities
**
**	Parameters:
**		sc_sess -- SMTPC session context
**
**	Returns:
**		usual sm_error code
**		(currently only SM_SUCCESS)
*/

static sm_ret_T
sc_ehlo_options(sc_sess_P sc_sess)
{
	char *opt;

	SM_REQUIRE(sc_sess != NULL);

	/* "250 " + something useful: at least 4 characters */
	if (sm_str_getlen(sc_sess->scse_rd) < 8)
		return SM_SUCCESS;
	if (sm_str_rd_elem(sc_sess->scse_rd, 0) != '2' ||
	    sm_str_rd_elem(sc_sess->scse_rd, 1) != '5' ||
	    sm_str_rd_elem(sc_sess->scse_rd, 2) != '0' ||
	    (sm_str_rd_elem(sc_sess->scse_rd, 3) != ' ' &&
	     sm_str_rd_elem(sc_sess->scse_rd, 3) != '-'))
		return SM_SUCCESS;	/* error?? */

#define SMTPC_IS_CAPN(str)	(sm_strcaseeqn(opt, (str), strlen(str)))
#define SMTPC_IS_CAP(str)	(sm_strcaseeq(opt, (str)))

	opt = (char *) sm_str_getdata(sc_sess->scse_rd);
	SM_REQUIRE(opt != NULL);
	opt += SMTP_REPLY_OFFSET;

	if (SMTPC_IS_CAP("pipelining"))
		SCSE_SET_CAP(sc_sess, SCSE_CAP_PIPELINING);
	else if (SMTPC_IS_CAP("8bitmime"))
		SCSE_SET_CAP(sc_sess, SCSE_CAP_8BITMIME);
	else if (SMTPC_IS_CAP("size"))
		SCSE_SET_CAP(sc_sess, SCSE_CAP_SIZE);
	else if (SMTPC_IS_CAPN("size "))
	{
		ulonglong_T max_sz_b;	/* off_t? */
		char *endptr;

		errno = 0;

		/* 5 == strlen("size ") */
		max_sz_b = sm_strtoull(opt + 5, &endptr, 10);
		if (!((max_sz_b == ULLONG_MAX && errno == ERANGE) ||
		      endptr == opt + 5))
		{
			SCSE_SET_CAP(sc_sess, SCSE_CAP_SIZE);
			sc_sess->scse_max_sz_b = max_sz_b;
		}
		/* else: silently ignore ... */
	}
	else if (SMTPC_IS_CAP("enhancedstatuscodes"))
		SCSE_SET_CAP(sc_sess, SCSE_CAP_ENHSTAT);
	else if (SMTPC_IS_CAPN("auth "))
		SCSE_SET_CAP(sc_sess, SCSE_CAP_AUTH);
	else if (SMTPC_IS_CAP("starttls"))
		SCSE_SET_CAP(sc_sess, SCSE_CAP_STARTTLS);
	return SM_SUCCESS;
}

/*
**  SC_TALKINGTOMYSELF -- does the server greet us with my own name?
**
**	Parameters:
**		sc_sess -- SMTPC session context
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sc_talkingtomyself(sc_sess_P sc_sess)
{
	sm_str_P hn;
	size_t l;

	SM_REQUIRE(sc_sess != NULL);
	if (SCSE_IS_FLAG(sc_sess, SCSE_FL_LMTP|SCSE_FL_NOTTM))
		return SM_SUCCESS;

	/* "250 " + something useful: at least 1 characters */
	if (sm_str_getlen(sc_sess->scse_rd) < 5)
		return SM_SUCCESS;
	if (sm_str_rd_elem(sc_sess->scse_rd, 0) != '2' ||
	    sm_str_rd_elem(sc_sess->scse_rd, 1) != '5' ||
	    sm_str_rd_elem(sc_sess->scse_rd, 2) != '0' ||
	    (sm_str_rd_elem(sc_sess->scse_rd, 3) != ' ' &&
	     sm_str_rd_elem(sc_sess->scse_rd, 3) != '-'))
		return SM_SUCCESS;	/* error?? */

	hn = sc_sess->scse_sct_ctx->sct_sc_ctx->scc_hostname;
	l = sm_str_getlen(hn) + SMTP_REPLY_OFFSET;
	if (sm_str_getlen(sc_sess->scse_rd) >= l
	    && (strncasecmp((char *) (sm_str_getdata(sc_sess->scse_rd) +
					SMTP_REPLY_OFFSET),
		(char *) sm_str_getdata(hn), l - SMTP_REPLY_OFFSET) == 0)
	    && (sm_str_rd_elem(sc_sess->scse_rd, l) == ' '
		|| sm_str_rd_elem(sc_sess->scse_rd, l) == '\r')
	    )
	{
		return sm_error_perm(SM_EM_SMTPC, SM_E_TTMYSELF);
	}
	return SM_SUCCESS;
}

/*
**  SC_RD_REPLY -- read reply/replies from SMTP server
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**		phase -- phase of SMTP dialogue
**
**	Returns:
**		SMTP reply code (2xy -> SMTP_OK) or error code
**
**	Side Effects:
**		If the server responded:
**		  sc_sess->scse_str contains entire response (provided it is
**		    big enough to hold all data);
**		  sc_sess->scse_rd contains last line of response
**		  (identical for single line responses).
*/

static sm_ret_T
sc_rd_reply(sc_t_ctx_P sc_t_ctx, int phase)
{
	sm_ret_T ret, ttmyself;
	uint thr_id;
	int rcode;
	bool firstline;
	bool used;	/* reply has been used for MAIL/RCPT? */
	sc_sess_P sc_sess;
#if SC_PIPELINING
	sc_ta_P sc_ta;
#endif
	sm_ret_T lmtp_ret;

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	SM_REQUIRE(sc_sess != NULL);
	thr_id = sc_t_ctx->sct_thr_id;
	ttmyself = SM_SUCCESS;
#if SC_PIPELINING
	sc_ta = sc_sess->scse_ta;

  again:
#endif
	lmtp_ret = SMTPC_NO_REPLY;
	firstline = true;
	used = false;
	rcode = -1;
	sm_str_clr(sc_sess->scse_str);
	do
	{
#if SC_PIPELINING
		/* also check how much data has been sent? */
		if (sc_pipeline_ok(phase)
		    && SCSE_IS_CAP(sc_sess, SCSE_CAP_PIPELINING)
		    && !sm_io_getinfo(sc_sess->scse_fp, SM_IO_IS_READABLE,
					NULL))
			return SMTPC_NO_REPLY;
#endif /* SC_PIPELINING */

		sm_str_clr(sc_sess->scse_rd);
		if (sc_sess->scse_fp == NULL ||
		    SCSE_IS_FLAG(sc_sess, SCSE_FL_IO_ERR))
			ret = EOF;
		else
			ret = sm_fgetline0(sc_sess->scse_fp, sc_sess->scse_rd);
		/* timeout error: EOF */

#if SC_DEBUG
		if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug > 3)
		{
			ssize_t b;

			/*
			**  multiple statements to print one entry...
			**  non-trivial to convert to logging.
			*/

			sm_io_fprintf(smioerr,
				"thread=%u, da_sess=%s, read [len=%d, res=%r]: ",
				thr_id, sc_sess->scse_id, sm_str_getlen(sc_sess->scse_rd), ret);
#if SC_PIPELINING
			if (sc_ta != NULL)
				sm_io_fprintf(smioerr,
					"[flags=%x, rcpts_rcvd=%d, snt=%d]: ",
					sc_ta->scta_state,
					sc_ta->scta_rcpts_rcvd,
					sc_ta->scta_rcpts_snt);
#endif /* SC_PIPELINING */

			/*
			**  WARNING: data directly from server!
			**  May contain "bogus" characters.
			*/

			sm_io_write(smioerr, sm_str_data(sc_sess->scse_rd),
				sm_str_getlen(sc_sess->scse_rd), &b);
			sm_io_flush(smioerr);
		}
#endif /* SC_DEBUG */
		if (sm_is_err(ret))
		{
			if (!SCSE_IS_FLAG(sc_sess, SCSE_FL_LOGGED) &&
			    !SCSE_IS_FLAG(sc_sess, SCSE_FL_IO_ERR))
			{
				sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
					SC_LCAT_CLIENT, SC_LMOD_CLIENT,
					SM_LOG_WARN,
					SCSE_IS_FLAG(sc_sess, SCSE_FL_SSD)
						? 14 : 9,
					"sev=WARN, func=sc_rd_reply, thread=%u, da_sess=%s, phase=%#x, ret=%m",
					thr_id, sc_sess->scse_id, phase, ret);
			}

			/* I/O error (always??) */
			SCSE_SET_FLAG(sc_sess, SCSE_FL_IO_ERR|SCSE_FL_LOGGED);
			goto fail;
		}
		if (phase == SC_PHASE_INIT)
		{
			/* question: perform a "loops back to me" check?? */
		}
		else if (phase == SC_PHASE_EHLO)
		{
			if (firstline)
			{
				/* get server capabilities */
				ttmyself = sc_talkingtomyself(sc_sess);
			}
			else
			{
				/* get server capabilities */
				ret = sc_ehlo_options(sc_sess);
			}
		}
		else if (sm_is_success(ret) &&
			 IS_SMTP_REPLY_STR(sc_sess->scse_rd, 0))
		{
			int r;

			r = SMTP_REPLY_STR2VAL(sc_sess->scse_rd, 0);
			if (!firstline && rcode != r)
			{
				sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
					SC_LCAT_CLIENT, SC_LMOD_CLIENT,
					SM_LOG_WARN, 9,
					"sev=WARN, func=sc_rd_reply, thread=%u, da_sess=%s, phase=%#x, status=inconsistent_multiline_reply_codes, previous=%d, current=%d"
					, thr_id, sc_sess->scse_id, phase
					, rcode, r);
			}
			rcode = r;
		}
		/* other cases? */

		/* bogus response? check every line, not just last */
		if (sm_is_success(ret) &&
		    !IS_SMTP_REPLY_STR(sc_sess->scse_rd, 0))
			ret = SMTPC_PROT;

		sm_str_cat(sc_sess->scse_str, sc_sess->scse_rd);
		firstline = false;
	} while (sm_is_success(ret) && sm_str_getlen(sc_sess->scse_rd) > 3 &&
		 sm_str_rd_elem(sc_sess->scse_rd, 3) == '-');
	if (sm_is_err(ttmyself))
		ret = ttmyself;
	if (sm_is_err(ret) || sm_str_getlen(sc_sess->scse_rd) < 3)
		goto fail;

	/*
	**  Note: in case of a multi-line reply sc_sess->scse_rd contains
	**  now the last line... we may have to store all of them for
	**  better error handling, i.e., to return the complete reply
	**  to a user in case of a DSN or for logging.
	*/

#if SC_DEBUG
	if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug > 3
	    && phase == SC_PHASE_EHLO)
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_INFO, 17,
			"sev=INFO, func=sc_rd_reply, thread=%u, da_sess=%s, capabilities=%x",
			thr_id, sc_sess->scse_id, sc_sess->scse_cap);
#endif /* SC_DEBUG */

	if (sm_str_rd_elem(sc_sess->scse_rd, 0) == '2')
		ret = SMTP_OK;
	else if (!IS_SMTP_REPLY_STR(sc_sess->scse_rd, 0))
		ret = SMTPC_PROT;
	else
	{
		ret = SMTP_REPLY_STR2VAL(sc_sess->scse_rd, 0);
		if (ret == SMTP_R_SSD)
			SCSE_SET_FLAG(sc_sess, SCSE_FL_SSD);
	}

#if SC_PIPELINING
	if (SCSE_IS_CAP(sc_sess, SCSE_CAP_PIPELINING) &&
	    (phase == SC_PHASE_RCPT
	     || phase == SC_PHASE_DATA
	     || phase == SC_PHASE_LDAT))
	{
		SM_IS_SC_TA(sc_ta);
		if (!SCTA_IS_STATE(sc_ta, SCTA_MAIL_R))
		{
			/* reply for MAIL */
			SCTA_SET_STATE(sc_ta, SCTA_MAIL_R);
			sc_ta->scta_mail->scm_st = ret;
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_INFO, 8,
				"sev=INFO, func=sc_rd_reply, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, mail=%@S, stat=%m, reply=%@S",
				thr_id, sc_sess->scse_id,
				sc_ta->scta_id, sc_ta->scta_ssta_id,
				sc_ta->scta_mail->scm_pa, ret,
				sc_sess->scse_rd);
			if (ret != SMTP_OK)
			{
				sc_ta->scta_mail->scm_reply = sm_str_dup(
							sc_ta->scta_rpool,
							sc_sess->scse_rd);
				sc_ta->scta_err_state = DA_TA_ERR_MAIL_S;
				sc_ta->scta_status = ret;
				SCTA_SET_STATE(sc_ta, smtp_is_reply_fail(ret)
					? SCTA_FAIL_PERM : SCTA_FAIL_TEMP);
			}
			ret = SMTPC_NO_REPLY;
			used = true;
		}
		else if (sc_ta->scta_rcpts_rcvd < sc_ta->scta_rcpts_snt)
		{
			sc_rcpt_P sc_rcpt;

			sc_rcpt = sc_ta->scta_rcpt_p;
			SM_ASSERT(sc_rcpt != SC_RCPTS_END(&(sc_ta->scta_rcpts)));
			if (ret == SMTP_OK)
			{
				sc_rcpt->scr_st = ret;
#if SC_STATS
				++RCPT_CNT_OK(sc_t_ctx->sct_sc_ctx);
#endif
			}

			/*
			**  Set error state only if there wasn't a transaction
			**  error yet, i.e., if MAIL was successful, otherwise
			**  the RCPT commands are irrelevant anyway.
			*/

			else if (ret != SMTP_OK &&
				 !SCTA_IS_STATE(sc_ta,
					SCTA_FAIL_PERM|SCTA_FAIL_TEMP))
			{
				sc_rcpt->scr_st = ret;
				sc_rcpt->scr_reply = sm_str_dup(sc_ta->scta_rpool,
							sc_sess->scse_rd);
				if (sc_ta->scta_err_state == DA_ERR_NONE)
					sc_ta->scta_err_state = DA_TA_ERR_RCPT_S;
				SCTA_SET_STATE(sc_ta, smtp_is_reply_fail(ret)
					? SCTA_R_PERM : SCTA_R_TEMP);
				if (smtp_is_reply_temp(ret))
				{
					/*
					**  Save temporary failure state for
					**  later perusal if necessary
					*/

					sc_ta->scta_status = ret;
				}

			}
			if (ret != SMTPC_NO_REPLY &&
			    !SCTA_IS_STATE(sc_ta,
					SCTA_FAIL_PERM|SCTA_FAIL_TEMP) &&
			    !SCR_IS_FLAG(sc_rcpt, SCR_FL_LOGGED))
			{
				sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
					SC_LCAT_CLIENT, SC_LMOD_CLIENT,
					SM_LOG_INFO, 8,
					"sev=INFO, func=sc_rd_reply, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, rcpt=%@S, stat=%m, reply=%@S",
					thr_id, sc_sess->scse_id,
					sc_ta->scta_id, sc_ta->scta_ssta_id,
					sc_rcpt->scr_pa, ret,
					sc_sess->scse_rd);
				SCR_SET_FLAG(sc_rcpt, SCR_FL_LOGGED);
			}

#if SC_DEBUG
			if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug > 3)
				sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
					SC_LCAT_CLIENT, SC_LMOD_CLIENT,
					SM_LOG_INFO, 15,
					"sev=INFO, func=sc_rd_reply, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, rcpt=%@S, idx=%d, stat=%m, scta_rcpts_rcvd=%d, scta_rcpts_snt=%d, reply=%p, err_st=%x, state=%x",
					thr_id, sc_sess->scse_id,
					sc_ta->scta_id, sc_ta->scta_ssta_id,
					sc_rcpt->scr_pa,
					sc_rcpt->scr_idx, ret,
					sc_ta->scta_rcpts_rcvd,
					sc_ta->scta_rcpts_snt,
					sc_rcpt->scr_reply,
					sc_ta->scta_err_state,
					sc_ta->scta_state);
#endif /* SC_DEBUG */

			if (ret == SMTP_OK)
				++sc_ta->scta_rcpts_ok;
			++sc_ta->scta_rcpts_rcvd;
			sc_ta->scta_rcpt_p = SC_RCPTS_NEXT(sc_rcpt);
			ret = SMTPC_NO_REPLY;
			if (sc_ta->scta_rcpts_rcvd == sc_ta->scta_rcpts_snt)
				SCTA_SET_STATE(sc_ta, SCTA_RCPT_R);
			used = true;
		}

		/* need to collect all replies? */
		if (phase == SC_PHASE_DATA)
		{
			if (used)
				goto again;
			SCTA_SET_STATE(sc_ta, SCTA_DATA_R);
		}
	}
#endif /* SC_PIPELINING */

	if (phase == SC_PHASE_LDAT)
	{
		sc_rcpt_P sc_rcpt;

		sc_rcpt = sc_ta->scta_rcpt_p;
		SM_ASSERT(sc_rcpt != SC_RCPTS_END(&(sc_ta->scta_rcpts)));

		/* only collect replies for "OK" recipients */
		while (sc_rcpt != SC_RCPTS_END(&(sc_ta->scta_rcpts))
		       && sc_rcpt->scr_st != SMTP_OK)
			sc_rcpt = SC_RCPTS_NEXT(sc_rcpt);

#if SC_DEBUG
		if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug > 3)
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_DEBUG, 15,
				"sev=DBG, func=sc_rd_reply, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, LMTP_rcpt=%S, idx=%d, stat=%m, rcpts_lmtp=%d",
				thr_id, sc_sess->scse_id,
				sc_ta->scta_id, sc_ta->scta_ssta_id,
				sc_rcpt->scr_pa,
				sc_rcpt->scr_idx, ret,
				sc_ta->scta_rcpts_lmtp);
#endif /* SC_DEBUG */

		if (sc_rcpt == SC_RCPTS_END(&(sc_ta->scta_rcpts)))
		{
			/* went beyond the list... */
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_ERR, 1,
				"sev=ERROR, func=sc_rd_reply, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, where=LMTP, status=end_of_list",
				thr_id, sc_sess->scse_id,
				sc_ta->scta_id, sc_ta->scta_ssta_id);
			return ret;
		}

		sc_rcpt->scr_st = ret;
		if (ret != SMTP_OK)
		{
			sc_rcpt->scr_reply = sm_str_dup(sc_ta->scta_rpool,
					sc_sess->scse_rd);
			if (sc_ta->scta_err_state == DA_ERR_NONE)
				sc_ta->scta_err_state = DA_TA_ERR_RCPT_S;
			--sc_ta->scta_rcpts_ok;
		}

		/* Hack: use the "lowest" value for lmtp_ret */
		if (lmtp_ret > ret)
			lmtp_ret = ret;
		--sc_ta->scta_rcpts_lmtp;
		sc_ta->scta_rcpt_p = SC_RCPTS_NEXT(sc_rcpt);

		/* all responses collected? */
		if (sc_ta->scta_rcpts_lmtp == 0)
		{
			/*
			**  RFC2033 4.2: there is no separate response
			**  for DATA, but only responses for previously
			**  accepted recipients, hence we don't set
			**  used, but use lmtp_ret as return value.
			*/

			SCTA_SET_STATE(sc_ta, SCTA_L_RCPT_R);
			ret = lmtp_ret;
		}
		else
		{
			used = true;
			ret = SMTPC_NO_REPLY;
		}
		if (used)
			goto again;
		SCTA_SET_STATE(sc_ta, SCTA_DATA_R);
	}

#if SC_DEBUG
	if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug > 3)
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_INFO, 15,
			"sev=INFO, func=sc_rd_reply, thread=%u, da_sess=%s, phase=%d, stat=%m",
			thr_id, sc_sess->scse_id, phase, ret);
#endif /* SC_DEBUG */
	return ret;

  fail:
	return ret;
}

/*
**  SC_GEN_DSN_HDR -- generate bounce header
**	on successful return scse_str contains MIME delimiter for this TA.
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**
**	Returns:
**		usual sm_error code
**
**	Side Effects:
**		on success scse_wr contains a header for a bounce message
**		uses scse_str as temporary string;
**		on successful return scse_str contains the MIME delimiter!
*/

static sm_ret_T
sc_gen_dsn_hdr(sc_t_ctx_P sc_t_ctx)
{
	sm_ret_T ret;
	time_T now;
	sc_sess_P sc_sess;
	sc_ta_P sc_ta;
	sc_rcpt_P sc_rcpt;

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	SM_REQUIRE(sc_sess != NULL);
	sc_ta = sc_sess->scse_ta;
	SM_IS_SC_TA(sc_ta);

	if (!SCTA_IS_FLAG(sc_ta, SCTA_FL_DBOUNCE) &&
	    !SCTA_IS_FLAG(sc_ta, SCTA_FL_BOUNCE) &&
	    !SCTA_IS_FLAG(sc_ta, SCTA_FL_DELAY))
		return SM_SUCCESS;


	now = st_time();
	sm_str_clr(sc_sess->scse_str);
	ret = arpadate(&now, sc_sess->scse_str);
	sc_rcpt = SC_RCPTS_FIRST(&(sc_ta->scta_rcpts));

	/* Send header for bounce */
	/* fixme: Put this in a library function */
	/* requires hostname (Hack!) */
	if (sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
				"From: Mailer-Daemon@")) ||
	    sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
				sc_t_ctx->sct_sc_ctx->scc_hostname)) ||
	    sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
				"\r\n"
				"Date: ")) ||
	    sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
				sc_sess->scse_str)) ||
	    sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
				"\r\nSubject: ")))
		goto error;

	if (sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
		SCTA_IS_FLAG(sc_ta, SCTA_FL_DELAY)
		? "Delayed mail\r\n"
		: SCTA_IS_FLAG(sc_ta, SCTA_FL_BOUNCE)
		? "Undeliverable mail\r\n"
		: "Double Bounce\r\n")))
			goto error;

	if (sm_is_err(ret = sm_str_scat(sc_sess->scse_wr, "To: ")) ||
	    sm_is_err(ret = sm_str_cat(sc_sess->scse_wr, sc_rcpt->scr_pa)) ||
	    sm_is_err(ret = sm_str_scat(sc_sess->scse_wr, "\r\n")))
		goto error;

	/* create MIME delimiter in scse_str */
	sm_str_clr(sc_sess->scse_str);
	if (sm_is_err(ret = sm_str_cat(sc_sess->scse_str,
				sc_t_ctx->sct_sc_ctx->scc_hostname)) ||
	    sm_is_err(ret = sm_str_put(sc_sess->scse_str, '/')) ||
	    sm_is_err(ret = sm_str_scat(sc_sess->scse_str, sc_ta->scta_id)))
		goto error;

	/* Add MIME header here... */
	if (SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN_MIME) &&
	    (SCTA_IS_FLAG(sc_ta, SCTA_FL_HDR_ONLY) ||
	     !SCTA_IS_FLAG(sc_ta, SCTA_FL_NO_BODY)))
	{
		if (sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
				"MIME-Version: 1.0\r\n"
				"Content-Type: multipart/mixed; boundary=\""))
		   || sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
						sc_sess->scse_str))
		   || sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
				"\"\r\n\r\n"
				"This is a MIME-encapsulated message\r\n\r\n--"))
		   || sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
						sc_sess->scse_str))
		   || sm_is_err(ret = sm_str_scat(sc_sess->scse_wr, "\r\n")))
			goto error;
	}

	if (sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
		SCTA_IS_FLAG(sc_ta, SCTA_FL_DELAY)
		? "\r\nHi! This is the sendmail X MTA.\r\n"
		  "A mail from you has not yet been successfully delivered.\r\n"
		  "It will be tried again, so you do not need to resend it!\r\n"
		  "See below for details.\r\n\r\n"
		: SCTA_IS_FLAG(sc_ta, SCTA_FL_BOUNCE)
		? "\r\nHi! This is the sendmail X MTA. "
		  "I'm sorry to inform you that a mail\r\n"
		  "from you could not be delivered. "
		  "See below for details.\r\n\r\n"
		: "\r\nHi! This is the sendmail X MTA. "
		  "I couldn't deliver a bounce message;\r\n"
		  "see below for details.\r\n\r\n")))
	{
		goto error;
	}

	return SM_SUCCESS;

  error:
	sm_str_clr(sc_sess->scse_str);
	return ret;
}

/*
**  SC_DATA -- send data
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**
**	Returns:
**		>=0: SMTP reply code
**		<0: error code
*/

static sm_ret_T
sc_data(sc_t_ctx_P sc_t_ctx)
{
	int c;
	size_t wrt, offset;
	ssize_t byteswritten;
	sm_ret_T ret;
	uint eoh_state;
#if 0
	uint eot_state;
#endif
	off_t total_size, cdb_rd;
	sc_sess_P sc_sess;
	sc_ta_P sc_ta;
	sm_file_T *cfp;
	cdb_ctx_P cdb_ctx;
	static SM_DECL_EOH;
#if 0
	static SM_DECL_EOT;
#endif

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	SM_REQUIRE(sc_sess != NULL);
	sc_ta = sc_sess->scse_ta;
	SM_IS_SC_TA(sc_ta);
	cfp = sc_sess->scse_fp;
	total_size = 0;
	eoh_state = 0;
#if 0
	eot_state = 0;
#endif
	cdb_ctx = sc_t_ctx->sct_sc_ctx->scc_cdb_ctx;

#if SC_PIPELINING
	if (sc_ta->scta_rcpts_ok == 0)
	{
		const uchar eod[] = ".\r\n";

		/* just send a single dot, see RFC 2920, 3.1 */
		ret = sm_io_write(cfp, eod, 3, &byteswritten);
		if (sm_is_err(ret))
			goto error;
		if (byteswritten > 0)
			total_size += byteswritten;
	}
	else
	{
#endif /* SC_PIPELINING */

	/* Compose bounce message (header, text from QMGR) */
	if (SCTA_IS_FLAG(sc_ta, SCTA_FL_DELAY|SCTA_FL_BOUNCE|SCTA_FL_DBOUNCE))
	{
		sm_str_P str_wr;

		sm_str_clr(sc_sess->scse_wr);
		if (SCSE_IS_FLAG(sc_sess, SCSE_FL_RETPATH))
		{
			if (sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
						"Return-Path: ")) ||
			    sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
					sc_ta->scta_mail->scm_pa)) ||
			    sm_is_err(sm_str_scat(sc_sess->scse_wr, "\r\n")))
				goto error;
		}

		/* Create header for bounce */
		ret = sc_gen_dsn_hdr(sc_t_ctx);
		if (sm_is_err(ret))
			goto error;
		ret = sc_command(sc_t_ctx, SC_PHASE_NOREPLY);
		if (sm_is_err(ret))
			goto error;

		/* Hack: save wr string and replace it temporarily */
		str_wr = sc_sess->scse_wr;
		sc_sess->scse_wr = sc_ta->scta_b_msg;
		ret = sc_command(sc_t_ctx, SC_PHASE_NOREPLY);
		sc_sess->scse_wr = str_wr;
		if (sm_is_err(ret))
			goto error;
		total_size += sm_str_getlen(sc_ta->scta_b_msg);
		sm_str_clr(sc_sess->scse_wr);

		if (!SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN_MIME) &&
		    SCTA_IS_FLAG(sc_ta, SCTA_FL_HDR_ONLY))
		{
			if (sm_is_err(ret = sm_str_scopy(sc_sess->scse_wr,
					"\r\nThe headers of your original mail follow:\r\n")))
				goto error;
		}
		else if (!SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN_MIME) &&
			 !SCTA_IS_FLAG(sc_ta, SCTA_FL_NO_BODY))
		{
			if (sm_is_err(ret = sm_str_scopy(sc_sess->scse_wr,
					"\r\nYour original mail follows:\r\n")))
				goto error;
		}

		if (SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN_MIME) &&
		    (SCTA_IS_FLAG(sc_ta, SCTA_FL_HDR_ONLY) ||
		     !SCTA_IS_FLAG(sc_ta, SCTA_FL_NO_BODY)))
		{
			if (sm_is_err(ret = sm_str_scopy(sc_sess->scse_wr,
					"\r\n--"))
			   || sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
					sc_sess->scse_str))
			   || sm_is_err(ret = sm_str_scat(sc_sess->scse_wr,
					"\r\n"
					"Content-Type: message/rfc822"
					"\r\n\r\n")))
				goto error;
		}

		if (sm_str_getlen(sc_sess->scse_wr))
		{
			ret = sc_command(sc_t_ctx, SC_PHASE_NOREPLY);
			if (sm_is_err(ret))
				goto error;
			sm_str_clr(sc_sess->scse_wr);
		}

	}

	if (SCTA_IS_FLAG(sc_ta, SCTA_FL_NO_BODY))
	{
		if (sm_is_err(ret = sm_str_scopy(sc_sess->scse_wr,
					"\r\n.\r\n")))
			goto error;
		ret = sc_command(sc_t_ctx, SC_PHASE_NOREPLY);
		if (sm_is_err(ret))
			goto error;
	}
	else
	{
		SM_ASSERT(sc_ta->scta_cdb_fp != NULL);

		if (SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN_MIME))
		{
			off_t cdb_sz_b;

			ret = sm_io_getinfo(sc_ta->scta_cdb_fp, SM_IO_WHAT_SIZE,
					&cdb_sz_b);
			if (sc_ta->scta_msg_sz_b == 0)
				sc_ta->scta_msg_sz_b = cdb_sz_b;
			else if (cdb_sz_b != sc_ta->scta_msg_sz_b)
			{
				sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
					SC_LCAT_CLIENT, SC_LMOD_CLIENT,
					SM_LOG_ERROR, 8,
					"sev=ERROR, func=sc_data, status=size_mismatch, cdb_sz_b=%lu, msg_sz_b=%lu, ret=%d",
					(ulong) cdb_sz_b, (ulong) sc_ta->scta_msg_sz_b,
					ret);
			}
		}

		if (SCSE_IS_FLAG(sc_sess, SCSE_FL_RETPATH) &&
		    !SCTA_IS_FLAG(sc_ta,
				SCTA_FL_DELAY|SCTA_FL_BOUNCE|SCTA_FL_DBOUNCE))
		{
			if (sm_is_err(ret = sm_str_scopy(sc_sess->scse_wr,
						"Return-Path: ")) ||
			    sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
					sc_ta->scta_mail->scm_pa)) ||
			    sm_is_err(sm_str_scat(sc_sess->scse_wr, "\r\n")))
				goto error;
			ret = sc_command(sc_t_ctx, SC_PHASE_NOREPLY);
			if (sm_is_err(ret))
				goto error;
		}

		/* read data and send it out... */
		cdb_rd = 0;
		do
		{
			/* get new buffer */
			c = sm_rget(sc_ta->scta_cdb_fp);
			if (c == SM_IO_EOF)
				break;

			/* +1 because we got the first char already */
			wrt = f_r(*(sc_ta->scta_cdb_fp)) + 1;
			cdb_rd += wrt;

			/* check whether eoh is in this block */
			if (SCTA_IS_FLAG(sc_ta, SCTA_FL_HDR_ONLY)
			    && eoh_state < SM_EOH_LEN)
			{
				uchar *p;

				p = f_bfbase(*sc_ta->scta_cdb_fp);
				offset = 0;
				do
				{
					if (c == eoh[eoh_state])
						++eoh_state;
					else
					{
						eoh_state = 0;
						if (c == eoh[eoh_state])
							++eoh_state;
					}

#if 0
					if (c == eot[eot_state])
						++eot_state;
					else
					{
						eot_state = 0;
						if (c == eot[eot_state])
							++eot_state;
					}
#endif

/* new, but broken?
					if (offset < wrt)
						c = p[offset++];
 old follows: */
					if (eoh_state < SM_EOH_LEN &&
					    offset < wrt)
						c = p[offset++];
					else
						break;
				} while (eoh_state < SM_EOH_LEN
					 && offset < wrt);

				/*
				**  Found end of header?  If yes: set the
				**  number of bytes to write; the buffer
				**  MUST end with \r\n such that the final
				**  dot is properly recognized (see below).
				*/

				if (eoh_state >= SM_EOH_LEN)
				{
					SCTA_SET_FLAG(sc_ta, SCTA_FL_CUT_HDR);
					wrt = offset;
				}
			}

			/*
			**  For a MIME DSN the last 3 bytes (.\r\n)
			**  of cdb must not be sent.
			*/

			if (SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN_MIME) &&
			    !SCTA_IS_FLAG(sc_ta, SCTA_FL_CDB_CUT) &&
			    cdb_rd + 3 >= sc_ta->scta_msg_sz_b
			   )
			{
				size_t cutoff;

				SM_ASSERT(sc_ta->scta_msg_sz_b >= SM_EOT_LEN);
				SM_ASSERT(cdb_rd <= sc_ta->scta_msg_sz_b);
				cutoff = sc_ta->scta_msg_sz_b - cdb_rd + 3;
				if (wrt > cutoff)
					wrt -= cutoff;
				else
					wrt = 0;
				SCTA_SET_FLAG(sc_ta, SCTA_FL_CUT_DOT);
			}

			/*
			**  Write the current buffer.
			*/

			offset = 0;
			do
			{
				ret = sm_io_write(cfp,
					f_bfbase(*sc_ta->scta_cdb_fp) + offset,
					wrt, &byteswritten);
#if SC_DEBUG
				if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug
				    > 3)
					sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
						SC_LCAT_CLIENT, SC_LMOD_CLIENT,
						SM_LOG_INFO, 18,
						"sev=INFO, func=sc_data, wrt=%d, written=%d, ret=%m",
						wrt, byteswritten, ret);
#endif /* SC_DEBUG */
				if (sm_is_err(ret))
					goto error;
				if (byteswritten > 0)
				{
					total_size += byteswritten;
					offset += byteswritten;
				}

				/* paranoia... should this be just an if ()? */
				SM_ASSERT(wrt >= byteswritten);
				if (wrt > byteswritten)
					sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
						SC_LCAT_CLIENT, SC_LMOD_CLIENT,
						SM_LOG_INFO, 8,
						"sev=INFO, func=sc_data, wrt=%d, written=%d, ret=%m",
						wrt, byteswritten, ret);

				wrt -= byteswritten;

			} while (wrt > 0);

			if (SCTA_IS_FLAG(sc_ta, SCTA_FL_CUT_HDR) ||
			    SCTA_IS_FLAG(sc_ta, SCTA_FL_CUT_DOT))
			{
				/* the data above ended with \r\n (eoh) */
				if (SCTA_IS_FLAG(sc_ta, SCTA_FL_DSN_MIME))
				{
					if (sm_is_err(ret =
						sm_str_scopy(sc_sess->scse_wr,
							"\r\n--"))
					   || sm_is_err(ret =
						sm_str_cat(sc_sess->scse_wr,
							sc_sess->scse_str))
					   || sm_is_err(ret =
						sm_str_scat(sc_sess->scse_wr,
							"--\r\n.\r\n")))
						goto error;
				}
				else
				{
					ret = sm_str_scopy(sc_sess->scse_wr,
						".\r\n");
				}

				if (sm_is_err(ret))
					goto error;
				ret = sc_command(sc_t_ctx, SC_PHASE_NOREPLY);
				if (sm_is_err(ret))
					goto error;
				c = SM_IO_EOF;	/* force end of loop */
			}

		} while (c != SM_IO_EOF);
		ret = cdb_close(cdb_ctx, sc_ta->scta_cdb_fp);
		if (sm_is_err(ret))
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_ERR, 8,
				"sev=ERROR, func=sc_data, cdb_close=%m"
				, ret);
		sc_ta->scta_cdb_fp = NULL;
	}

#if SC_PIPELINING
	}
#endif

	SCTA_SET_STATE(sc_ta, SCTA_DOT_S);
	ret = sc_rd_reply(sc_t_ctx,
			SCSE_IS_FLAG(sc_sess, SCSE_FL_LMTP)
			? SC_PHASE_LDAT : SC_PHASE_DOT);
	if (ret != SMTP_OK && sm_str_getlen(sc_sess->scse_str) > 0)
	{
		sc_ta->scta_reply = sm_str_dup(sc_ta->scta_rpool,
						sc_sess->scse_str);
	}
#if SC_STATS
	else
		++TA_CNT_OK(sc_t_ctx->sct_sc_ctx);
#endif

	sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
		SC_LCAT_CLIENT, SC_LMOD_CLIENT,
		SM_LOG_INFO, 8,
		"sev=INFO, func=sc_data, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, where=final_dot, size=%lu, stat=%m, reply=%@S",
		sc_t_ctx->sct_thr_id, sc_sess->scse_id,
		sc_ta->scta_id, sc_ta->scta_ssta_id,
		(ulong) total_size, ret, sc_sess->scse_str);
	return ret;

  error:
	/* we have to abort the connection */
	if (sc_ta->scta_cdb_fp != NULL)
	{
		(void) cdb_close(cdb_ctx, sc_ta->scta_cdb_fp);
		sc_ta->scta_cdb_fp = NULL;
	}
	if (cfp != NULL)
	{
		(void) sm_io_close(cfp);
		sc_sess->scse_fp = NULL;
		sc_sess->scse_state = SCSE_ST_CLOSED;	/* error state? */
	}
	if (sc_ta->scta_err_state == DA_ERR_NONE ||
	    (da_ta_err_cmd(sc_ta->scta_err_state) == DA_TA_ERR_RCPT &&
	     sc_ta->scta_rcpts_ok > 0))
		sc_ta->scta_err_state = DA_TA_ERR_BODY_I;
	return ret;
}

/*
**  SC_COMMAND -- send one SMTP command, read reply (unless turned off)
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**		phase -- phase of SMTP dialogue
**
**	Returns:
**		SMTP reply code (2xy -> SMTP_OK) or error code
*/

static sm_ret_T
sc_command(sc_t_ctx_P sc_t_ctx, int phase)
{
	sm_ret_T ret;
	ssize_t b;
	size_t l;
	uint thr_id;
	sc_sess_P sc_sess;

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	SM_REQUIRE(sc_sess != NULL);
	thr_id = sc_t_ctx->sct_thr_id;

	l = sm_str_getlen(sc_sess->scse_wr);
#if SC_DEBUG
	if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug > 3)
	{
		/* multiple print statements... log? */
		sm_io_fprintf(smioerr, "thread=%u, da_sess=%s, send[%d]: ", thr_id, sc_sess->scse_id, l);
		sm_io_write(smioerr, sm_str_data(sc_sess->scse_wr), l, &b);
		sm_io_fprintf(smioerr, "\n");
		sm_io_flush(smioerr);
	}
#endif /* SC_DEBUG */

	do
	{
		ret = sm_io_write(sc_sess->scse_fp,
				sm_str_data(sc_sess->scse_wr), l, &b);
		if (sm_is_err(ret))
		{
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_WARN, 11,
				"sev=WARN, func=sc_command, thread=%u, da_sess=%s, where=write, n=%d, r=%d, ret=%m",
				thr_id, sc_sess->scse_id, l, (int) b, ret);

			/* I/O error (always??) */
			SCSE_SET_FLAG(sc_sess, SCSE_FL_IO_ERR|SCSE_FL_LOGGED);
			return ret;
		}

		/* paranoia... should this be just an if (..)? */
		SM_ASSERT(l >= b);
		l -= b;
	} while (l > 0);

#if SC_PIPELINING
	if (!SCSE_IS_CAP(sc_sess, SCSE_CAP_PIPELINING) ||
	    !sc_pipeline_ok(phase))
	{
#endif
		ret = sm_io_flush(sc_sess->scse_fp);
		if (sm_is_err(ret))
		{
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_WARN, 11,
				"sev=WARN, func=sc_command, thread=%u, da_sess=%s, where=flush, n=%d, ret=%m",
				thr_id, sc_sess->scse_id, (int) b, ret);
			return ret;
		}
#if SC_PIPELINING
	}
#endif

	if (phase == SC_PHASE_NOREPLY)
		return SM_SUCCESS;
	return sc_rd_reply(sc_t_ctx, phase);
}

/*
**  SC_ONE_TA -- perform one SMTPC transaction
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**
**	Returns:
**		SMTP reply code or error code
*/

sm_ret_T
sc_one_ta(sc_t_ctx_P sc_t_ctx)
{
	uint thr_id;
	sm_ret_T ret;
	sc_sess_P sc_sess;
	sc_ta_P sc_ta;
	sc_rcpt_P sc_rcpt, sc_rcpt_nxt;
	sc_rcpts_P sc_rcpts;

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	SM_REQUIRE(sc_sess != NULL);
	sc_ta = sc_sess->scse_ta;
	SM_IS_SC_TA(sc_ta);
	ret = SM_SUCCESS;
	thr_id = sc_t_ctx->sct_thr_id;
	sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
		SC_LCAT_CLIENT, SC_LMOD_CLIENT,
		SM_LOG_INFO, 14,
		"sev=INFO, func=sc_one_ta, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, where=start_transaction, ta_state=%x",
		thr_id, sc_sess->scse_id, sc_ta->scta_id, sc_ta->scta_ssta_id,
		sc_ta->scta_state);

	/* already closed? check also state? */
	if (sc_sess->scse_fp == NULL)
		goto done;

	if (!SCTA_IS_FLAG(sc_ta, SCTA_FL_NO_BODY))
	{
		/* open cdb entry */
		ret = cdb_open(sc_t_ctx->sct_sc_ctx->scc_cdb_ctx,
				sm_str_getdata(sc_ta->scta_cdb_id),
				sc_ta->scta_cdb_fp,
				SM_IO_RDONLY, /*size*/ 0, /*hints*/ 0);
		if (sm_is_err(ret))
		{
			/*
			**  todo: send error code back to QMGR?? Don't retry
			**  if file does not exist? (some bozo may have
			**  removed it or there was a communication problem
			**  between qmgr and smtps)
			*/

			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_ERR, 4,
				"sev=ERROR, func=sc_one_ta, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, cdb=%N, cdb_open=%m",
				sc_t_ctx->sct_thr_id, sc_sess->scse_id,
				sc_ta->scta_id, sc_ta->scta_ssta_id,
				sc_ta->scta_cdb_id, ret);
			sc_ta->scta_err_state = DA_TA_ERR_CDB_I;
			if (sm_error_value(ret) == ENOENT)
			{
				ret = SMTP_CDB_PERM;
				goto fail;
			}
			else
			{
				/*
				**  Treat every other error as temp??
				**  Note: this returns the cdb_open() error
				**  code directly to QMGR which might not
				**  understand it... fixme?
				*/

				sc_ta->scta_status = SMTP_CDB_TEMP;
				SCTA_SET_STATE(sc_ta, SCTA_FAIL_TEMP);
				goto done;
			}
		}
	}

	/*
	**  How can we simplify this? Create some constant str and just
	**  use sm_str_catv()?
	**
	**  What about PIPELINING?
	**  Decouple sending of commands and reading of replies.
	**
	**  How to treat errors in this code? That is, if some operation
	**  in smtpc fails, what should be returned to QMGR?
	**  Answer: an internal error, see sm/da.h
	*/

	if (SCSE_IS_CAP(sc_sess, SCSE_CAP_SIZE) && sc_ta->scta_msg_sz_b > 0)
	{
		sm_str_clr(sc_sess->scse_wr);
		if (sm_str_printf(sc_sess->scse_wr,
			"MAIL From:%S SIZE=%lu\r\n"
			, sc_ta->scta_mail->scm_pa
			, sc_ta->scta_msg_sz_b) <= 18)
		{
			sc_ta->scta_err_state = DA_TA_ERR_MAIL_I;
			goto fail;
		}
	}
	else if (sm_is_err(ret = sm_str_scopy(sc_sess->scse_wr, "MAIL From:"))
		|| sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
					sc_ta->scta_mail->scm_pa))
		|| sm_is_err(sm_str_scat(sc_sess->scse_wr, "\r\n")))
	{
		sc_ta->scta_err_state = DA_TA_ERR_MAIL_I;
		goto fail;
	}
	ret = sc_command(sc_t_ctx, SC_PHASE_MAIL);
#if SC_STATS
	++MAIL_COUNT(sc_t_ctx->sct_sc_ctx);
#endif
	sc_ta->scta_mail->scm_st = ret;
	if (ret != SMTPC_NO_REPLY)
	{
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_INFO, 8,
			"sev=INFO, func=sc_one_ta, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, mail=%@S, stat=%m, reply=%@S",
			thr_id, sc_sess->scse_id, sc_ta->scta_id,
			sc_ta->scta_ssta_id, sc_ta->scta_mail->scm_pa, ret,
			sc_sess->scse_str);
	}
	if (!SMTPC_OK_REPLY(ret))
	{
		sc_ta->scta_mail->scm_reply = sm_str_dup(NULL,
							sc_sess->scse_rd);
		sc_ta->scta_err_state = DA_TA_ERR_MAIL_S;
		if (ret == SMTP_R_SSD)
			goto done;
		goto fail;
	}
#if SC_PIPELINING
	if (ret == SMTPC_NO_REPLY)
		SCTA_SET_STATE(sc_ta, SCTA_MAIL_S);
	else
#endif
		SCTA_SET_STATE(sc_ta, SCTA_MAIL_S|SCTA_MAIL_R);

	sc_rcpts = &(sc_ta->scta_rcpts);
#if SC_PIPELINING
	sc_ta->scta_rcpt_p = SC_RCPTS_FIRST(sc_rcpts);
#endif
	for (sc_rcpt = SC_RCPTS_FIRST(sc_rcpts);
	     sc_rcpt != SC_RCPTS_END(sc_rcpts);
	     sc_rcpt = sc_rcpt_nxt)
	{
		sc_rcpt_nxt = SC_RCPTS_NEXT(sc_rcpt);
		if (sm_is_err(ret = sm_str_scopy(sc_sess->scse_wr,
						"RCPT To:")) ||
		    sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
						sc_rcpt->scr_pa)) ||
		    sm_is_err(sm_str_scat(sc_sess->scse_wr, "\r\n")))
		{
			if (sc_ta->scta_err_state == DA_ERR_NONE)
				sc_ta->scta_err_state = DA_TA_ERR_RCPT_I;
			goto fail;
		}
		ret = sc_command(sc_t_ctx, SC_PHASE_RCPT);
#if SC_STATS
		++RCPT_COUNT(sc_t_ctx->sct_sc_ctx);
#endif

		/* this might not be valid (PIPELINING) */
		sc_rcpt->scr_st = ret;
		if (ret != SMTPC_NO_REPLY &&
		    !SCR_IS_FLAG(sc_rcpt, SCR_FL_LOGGED))
		{
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_INFO, 8,
				"sev=INFO, func=sc_one_ta, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, rcpt=%@S, stat=%m, reply=%@S",
				thr_id, sc_sess->scse_id, sc_ta->scta_id,
				sc_ta->scta_ssta_id, sc_rcpt->scr_pa, ret,
				sc_sess->scse_str);
			SCR_SET_FLAG(sc_rcpt, SCR_FL_LOGGED);

		}
		if (!SMTPC_OK_REPLY(ret))
		{
			sc_rcpt->scr_reply = sm_str_dup(sc_ta->scta_rpool,
						sc_sess->scse_rd);
			if (sc_ta->scta_err_state == DA_ERR_NONE)
				sc_ta->scta_err_state = DA_TA_ERR_RCPT_S;
			SCTA_SET_STATE(sc_ta, smtp_is_reply_fail(ret)
					? SCTA_R_PERM : SCTA_R_TEMP);
			if (ret == SMTP_R_SSD)
				goto done;
		}
#if SC_STATS
		else if (ret == SMTP_OK)
			++RCPT_CNT_OK(sc_t_ctx->sct_sc_ctx);
#endif
		if (sm_is_err(ret))
			goto fail;
		sc_ta->scta_rcpts_snt++;

		if (ret == SMTP_OK)
		{
			sc_ta->scta_rcpts_ok++;
#if SC_PIPELINING
			if (sc_ta->scta_rcpts_rcvd == sc_ta->scta_rcpts_snt)
				SCTA_SET_STATE(sc_ta, SCTA_RCPT_S|SCTA_RCPT_R);
			else
				SCTA_SET_STATE(sc_ta, SCTA_RCPT_S);
#else
			SCTA_SET_STATE(sc_ta, SCTA_RCPT_S|SCTA_RCPT_R);
#endif
		}
#if SC_PIPELINING
		else
			SCTA_SET_STATE(sc_ta, SCTA_RCPT_S);
#endif
	}

	/* no valid recipients? */
	if (sc_ta->scta_rcpts_ok == 0
	    && !SCSE_IS_CAP(sc_sess, SCSE_CAP_PIPELINING))
		goto fail;

	ret = sm_str_scopy(sc_sess->scse_wr, "DATA\r\n");
	if (sm_is_err(ret))
	{
		if (sc_ta->scta_err_state == DA_ERR_NONE)
			sc_ta->scta_err_state = DA_TA_ERR_DATA_I;
		goto fail;
	}

	/* this will collect all outstanding results if PIPELINING is active */
	ret = sc_command(sc_t_ctx, SC_PHASE_DATA);
	if (smtp_reply_type(ret) != 3)
	{
		/*
		**  where=data is confusing for PIPELINING...
		**  state gives an indication what's going, needs to be
		**  translated to text.
		*/

#if SC_PIPELINING
		if (sc_ta->scta_rcpts_ok == 0)
			goto norcpts;
#endif
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_INFO, 8,
			"sev=INFO, func=sc_one_ta, thread=%u, da_sess=%s, da_ta=%s, ss_ta=%s, where=data, ret=%m, rcpts_ok=%d, state=%r, reply=%@S"
			, thr_id, sc_sess->scse_id, sc_ta->scta_id
			, sc_ta->scta_ssta_id, ret
			, sc_ta->scta_rcpts_ok
			, sc_ta->scta_state
			, sc_sess->scse_str);
		if (sc_ta->scta_err_state == DA_ERR_NONE ||
		    (da_ta_err_cmd(sc_ta->scta_err_state) == DA_TA_ERR_RCPT &&
		     sc_ta->scta_rcpts_ok > 0))
			sc_ta->scta_err_state = DA_TA_ERR_DATA_S;
		if (ret == SMTP_R_SSD)
			goto done;
		goto fail;
	}
	SCTA_SET_STATE(sc_ta, SCTA_DATA_S|SCTA_DATA_R);
	if (SCSE_IS_FLAG(sc_sess, SCSE_FL_LMTP))
	{
#if SC_PIPELINING
		/* reset recipient list to collect results */
		sc_ta->scta_rcpt_p = SC_RCPTS_FIRST(sc_rcpts);
#endif
		sc_ta->scta_rcpts_lmtp = sc_ta->scta_rcpts_ok;
	}

	/* Send data */
	ret = sc_data(sc_t_ctx);
	if (ret != SMTP_OK)
	{
#if SC_PIPELINING
		if (sc_ta->scta_rcpts_ok == 0)
			goto norcpts;
#endif
		if (sc_ta->scta_err_state == DA_ERR_NONE ||
		    (da_ta_err_cmd(sc_ta->scta_err_state) == DA_TA_ERR_RCPT &&
		     sc_ta->scta_rcpts_ok > 0))
			sc_ta->scta_err_state = DA_TA_ERR_DOT_S; /* really?? */
		if (ret == SMTP_R_SSD)
			goto done;
		goto fail;
	}
	SCTA_SET_STATE(sc_ta, SCTA_DOT_R);
	SCTA_SET_STATE(sc_ta, SCTA_COMPLETE);

#if 0
	/* RSET isn't necessary: MAIL starts a new transaction */
	ret = sm_str_scopy(sc_sess->scse_wr, "RSET\r\n");
	if (sm_is_err(ret))
	{
		if (sc_ta->scta_err_state == DA_ERR_NONE)
			sc_ta->scta_err_state = DA_TA_ERR_RSET_I;
		goto fail;
	}
	ret = sc_command(sc_t_ctx, SC_PHASE_OTHER);
	if (ret == SMTP_R_SSD)
		goto done;
	if (ret != SMTP_OK)
		goto fail;
#endif /* 0 */

#if SC_STATS
	++TOTAL(sc_t_ctx);
#endif
	goto done;

#if SC_PIPELINING
  norcpts:
	if (SCTA_IS_STATE(sc_ta, SCTA_FAIL_TEMP|SCTA_R_TEMP))
	{
		ret = sc_ta->scta_status;
		goto done;
	}
#endif /* SC_PIPELINING */

  fail:
	/* clean up? */
	if (sm_is_err(ret))
	{
		SCTA_SET_STATE(sc_ta, sm_is_temp_err(ret)
				? SCTA_FAIL_TEMP : SCTA_FAIL_PERM);
	}
	else if (IS_SMTP_REPLY(ret))
	{
		SCTA_SET_STATE(sc_ta, smtp_is_reply_temp(ret)
				? SCTA_FAIL_TEMP : SCTA_FAIL_PERM);
	}
	else
	{
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 8,
			"sev=ERROR, func=sc_one_ta, thread=%u, da_sess=%s, da_ta=%s, ret=%d, status=unexpected_error_code"
			, thr_id, sc_sess->scse_id, sc_ta->scta_id
			, ret);
	}
  done:
	if (sc_ta->scta_cdb_fp != NULL)
	{
		(void) cdb_close(sc_t_ctx->sct_sc_ctx->scc_cdb_ctx,
				sc_ta->scta_cdb_fp);
		sc_ta->scta_cdb_fp = NULL;
	}
	return ret;
}

/*
**  SC_SESS_OPEN -- open SMTPC session
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**
**	Returns:
**		SMTP reply code or error code
**
**	Side Effects:
**		on error sc_sess->scse_str should contain an error string
*/

sm_ret_T
sc_sess_open(sc_t_ctx_P sc_t_ctx)
{
	st_netfd_t rmt_nfd;
	int sock, r, addrlen;
	uint thr_id;
	sm_ret_T ret;
	sc_sess_P sc_sess;
#if SM_USE_TLS
	sc_ctx_P sc_ctx;
#endif

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	SM_REQUIRE(sc_sess != NULL);
	thr_id = sc_t_ctx->sct_thr_id;
	ret = SM_SUCCESS;
	sc_sess->scse_fp = NULL;
	rmt_nfd = INVALID_NETFD;
	sm_str_clr(sc_sess->scse_str);
	sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
		SC_LCAT_CLIENT, SC_LMOD_CLIENT,
		SM_LOG_INFO, 14,
		"sev=INFO, func=sc_sess_open, thread=%u, da_sess=%s, where=open",
		thr_id, sc_sess->scse_id);

#if SC_STATS
	++BUSY(sc_t_ctx);
#endif

	/* Connect to remote host */
	sock = socket(sc_sess->scse_rmt_addr.sa.sa_family, SOCK_STREAM, 0);
	if (sock < 0)
	{
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 8,
			"sev=ERROR, func=sc_sess_open, thread=%u, da_sess=%s, socket()=failed, error=%s"
			, thr_id, sc_sess->scse_id, strerror(errno));
		sc_sess->scse_err_st = DA_SE_ERR_OPEN_I;
		goto done;
	}
	ret = sm_io_open(&SmStThrIO, (void *) &sock, SM_IO_RDWR,
			&(sc_sess->scse_fp), NULL);
	if (ret != SM_SUCCESS)
	{
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 8,
			"sev=ERROR, func=sc_sess_open, thread=%u, da_sess=%s, sm_io_open()=%m, error=%s"
			, thr_id, sc_sess->scse_id, ret, strerror(errno));
		close(sock);
		sc_sess->scse_err_st = DA_SE_ERR_OPEN_I;
		goto done;
	}
	sm_io_clrblocking(sc_sess->scse_fp);
	rmt_nfd = (st_netfd_t) f_cookie(*(sc_sess->scse_fp));
	r = sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_timeout;
	ret = sm_io_setinfo(sc_sess->scse_fp, SM_IO_WHAT_TIMEOUT, &r);
	if (ret != SM_SUCCESS)
	{
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 8,
			"sev=ERROR, func=sc_sess_open, thread=%u, da_sess=%s, set_timeout()=%m, error=%s"
			, thr_id, sc_sess->scse_id, ret, strerror(errno));
		sc_sess->scse_err_st = DA_SE_ERR_OPEN_I;
		SCSE_SET_FLAG(sc_sess, SCSE_FL_IO_ERR);
		goto fail;
	}

	ret = sm_io_setinfo(sc_sess->scse_fp, SM_IO_DOUBLE, NULL);
	if (ret != SM_SUCCESS)
	{
		r = errno;
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 8,
			"sev=ERROR, func=sc_sess_open, thread=%u, da_sess=%s, set_double()=%m, error=%s"
			, thr_id, sc_sess->scse_id, ret, strerror(errno));
		sc_sess->scse_err_st = DA_SE_ERR_OPEN_I;
		SCSE_SET_FLAG(sc_sess, SCSE_FL_IO_ERR);
		goto fail;
	}

	/* fixme: put this into a library? */

	switch (sc_sess->scse_rmt_addr.sa.sa_family)
	{
	  case AF_INET:
		addrlen = sizeof(sc_sess->scse_rmt_addr.sin);
		break;
	  case AF_UNIX:
		addrlen = sizeof(sc_sess->scse_rmt_addr.sunix);
		break;
	  default:
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_ERR, 8,
			"sev=ERROR, func=sc_sess_open, thread=%u, da_sess=%s, where=connect, unsupported_family=%d"
			, thr_id, sc_sess->scse_id
			, sc_sess->scse_rmt_addr.sa.sa_family);
		sc_sess->scse_err_st = DA_SE_ERR_OPEN_I;
		SCSE_SET_FLAG(sc_sess, SCSE_FL_IO_ERR);
		goto fail;
	}

	if (st_connect(rmt_nfd, (sockaddr_P) &(sc_sess->scse_rmt_addr), addrlen,
		SEC2USEC(sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_timeout)) < 0)
	{
		r = errno;

		/* fixme: always temp?? */
		ret = sm_error_temp(SM_EM_SMTPC, r);
#if 0
		sc_sess->scse_ret = ret;
#endif
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_WARN, 8,
			"sev=WARN, func=sc_sess_open, thread=%u, da_sess=%s, where=connect, port=%d, addr=%s, error=%s",
			thr_id, sc_sess->scse_id,
			(sc_sess->scse_rmt_addr.sa.sa_family == AF_INET) ?
				ntohs(sc_sess->scse_rmt_addr.sin.sin_port): -1,
			(sc_sess->scse_rmt_addr.sa.sa_family == AF_INET) ?
				inet_ntoa(sc_sess->scse_rmt_addr.sin.sin_addr) :
				sc_sess->scse_rmt_addr.sunix.sun_path,
			strerror(r));
		sm_str_scat(sc_sess->scse_str, strerror(r));
		sc_sess->scse_err_st = DA_SE_ERR_OPEN_I;
		SCSE_SET_FLAG(sc_sess, SCSE_FL_IO_ERR|SCSE_FL_LOGGED);
		goto fail;
	}
	sc_sess->scse_state = SCSE_ST_CONNECTED;
	sc_sess->scse_rmt_fd = rmt_nfd;

	sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
		SC_LCAT_CLIENT, SC_LMOD_CLIENT,
		SM_LOG_INFO, 10,
		"sev=INFO, func=sc_sess_open, thread=%u, da_sess=%s, status=connected, port=%d, addr=%s",
		thr_id, sc_sess->scse_id,
		(sc_sess->scse_rmt_addr.sa.sa_family == AF_INET) ?
			ntohs(sc_sess->scse_rmt_addr.sin.sin_port): -1,
		(sc_sess->scse_rmt_addr.sa.sa_family == AF_INET) ?
			inet_ntoa(sc_sess->scse_rmt_addr.sin.sin_addr) :
			sc_sess->scse_rmt_addr.sunix.sun_path);

	ret = sc_rd_reply(sc_t_ctx, SC_PHASE_INIT);
	if (ret != SMTP_OK)
	{
		if (!SCSE_IS_FLAG(sc_sess, SCSE_FL_LOGGED))
		{
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_WARN, 9,
				"sev=WARN, func=sc_sess_open, thread=%u, da_sess=%s, where=connection_rd_reply, ret=%m"
				, thr_id, sc_sess->scse_id, ret);
			SCSE_SET_FLAG(sc_sess, SCSE_FL_LOGGED);
		}
		sc_sess->scse_err_st = DA_SE_ERR_GRET_R;
		if (sc_sess->scse_reply != NULL)
		{
			sm_str_clr(sc_sess->scse_reply);
			sm_str_cat(sc_sess->scse_reply, sc_sess->scse_str);
		}
		else
		{
			sc_sess->scse_reply = sm_str_dup(NULL,
							sc_sess->scse_str);
		}
		goto fail;
	}

#if 0
	/* HACK turn on LMTP */
	if (sc_t_ctx->sct_sc_ctx->scc_cnf.sc_cnf_debug == 9)
		SCSE_SET_FLAG(sc_sess, SCSE_FL_LMTP);
#endif

  ehlo:
	sc_sess->scse_cap = SCSE_CAP_NONE;
	if (sm_is_err(ret = sm_str_scopy(sc_sess->scse_wr,
				SCSE_IS_FLAG(sc_sess, SCSE_FL_LMTP)
				? "LHLO "
				: SCSE_IS_FLAG(sc_sess, SCSE_FL_EHLO)
				  ? "HELO " : "EHLO ")) ||
	    sm_is_err(ret = sm_str_cat(sc_sess->scse_wr,
				sc_t_ctx->sct_sc_ctx->scc_hostname)) ||
	    sm_is_err(ret = sm_str_scat(sc_sess->scse_wr, "\r\n")))
	{
		sc_sess->scse_err_st = SCSE_IS_FLAG(sc_sess, SCSE_FL_LMTP)
				? DA_SE_ERR_LHLO_I
				: SCSE_IS_FLAG(sc_sess, SCSE_FL_EHLO)
				  ? DA_SE_ERR_HELO_I : DA_SE_ERR_EHLO_I;
		goto fail;
	}
	ret = sc_command(sc_t_ctx, SC_PHASE_EHLO);
	if (ret == sm_error_perm(SM_EM_SMTPC, SM_E_TTMYSELF))
	{
		sc_sess->scse_err_st = DA_SE_ERR_TTMYSLEF_S;
		goto fail;
	}
	else if (ret != SMTP_OK)
	{
		if (!SCSE_IS_FLAG(sc_sess, SCSE_FL_LMTP)
		    && !SCSE_IS_FLAG(sc_sess, SCSE_FL_EHLO)
		    && !SCSE_IS_FLAG(sc_sess, SCSE_FL_IO_ERR))
		{
			SCSE_SET_FLAG(sc_sess, SCSE_FL_EHLO);
			goto ehlo;
		}

		/* distinguish between SMTP reply code and sm/error code! */
		sc_sess->scse_err_st = DA_SET_ERR_RS(
				SCSE_IS_FLAG(sc_sess, SCSE_FL_LMTP)
				? DA_SE_ERR_LHLO
				: SCSE_IS_FLAG(sc_sess, SCSE_FL_EHLO)
				  ? DA_SE_ERR_HELO : DA_SE_ERR_EHLO, ret);
		goto fail;
	}
#if SM_USE_TLS
	sc_ctx = sc_t_ctx->sct_sc_ctx;
	if (SC_IS_FLAG(sc_ctx, SCC_FL_TLS_OK) &&
	    SCSE_IS_CAP(sc_sess, SCSE_CAP_STARTTLS) &&
	    !SCSE_IS_FLAG(sc_sess, SCSE_FL_STARTTLS))
	{
		ssize_t written;

		sc_sess->scse_con = SSL_new(sc_ctx->scc_ssl_ctx);
		if (sc_sess->scse_con == NULL)
			goto notls;

		ret = sm_str_scopy(sc_sess->scse_wr, "STARTTLS\r\n");
		if (sm_is_err(ret))
		{
			SSL_free(sc_sess->scse_con);
			sc_sess->scse_con = NULL;
			goto notls;
		}
		ret = sc_command(sc_t_ctx, SC_PHASE_TLS);
		if (ret != SMTP_OK)
		{
			SSL_free(sc_sess->scse_con);
			sc_sess->scse_con = NULL;
			goto notls;
		}

		SSL_set_connect_state(sc_sess->scse_con);
		ret = tls_open(sc_sess->scse_fp, sc_sess->scse_con,
				&sc_sess->scse_fptls);
		if (sm_is_success(ret))
		{
			/* HACK */
			sc_sess->scse_fp = sc_sess->scse_fptls;
			ret = do_tls_operation(sc_sess->scse_fp,
					SSL_connect, NULL, NULL, NULL, 0,
					&written);
			if (sm_is_err(ret))
			{
				sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
					SC_LCAT_CLIENT, SC_LMOD_CLIENT,
					SM_LOG_INFO, 8,
					"sev=INFO, func=sc_sess_open, thread=%u, da_sess=%s, where=connection, starttls=%m",
					thr_id, sc_sess->scse_id, ret);
				sc_sess->scse_err_st = DA_SE_ERR_STLS_S;

				/* get error message from OpenSSL? */
				sm_str_scat(sc_sess->scse_str,
					"TLS Handshake failed");

				/* avoid further I/O, TLS handshake failed */
				SCSE_SET_FLAG(sc_sess, SCSE_FL_IO_ERR);
				goto fail;
			}
			r = 1;
			ret = sm_io_setinfo(sc_sess->scse_fp, SM_IO_DOUBLE, &r);
			if (sm_is_err(ret))
				goto fail;
			(void) tls_get_info(sc_ctx->scc_tlsl_ctx,
				sc_sess->scse_con,
				TLS_F_SRV|TLS_F_CERT_REQ,
				(sc_sess->scse_rmt_addr.sa.sa_family == AF_INET)
				? inet_ntoa(sc_sess->scse_rmt_addr.sin.sin_addr)
				: sc_sess->scse_rmt_addr.sunix.sun_path,
				sc_sess->scse_tlsi);
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_INFO, 8,
				"sev=INFO, func=sc_sess_open, thread=%u, da_sess=%s, where=connection, starttls=successful, cipher=%S, bits=%d/%d, verify=%s"
				, thr_id, sc_sess->scse_id
				, sc_sess->scse_tlsi->tlsi_cipher
				, sc_sess->scse_tlsi->tlsi_cipher_bits
				, sc_sess->scse_tlsi->tlsi_algs_bits
				, tls_vrfy2txt(sc_sess->scse_tlsi->tlsi_vrfy));

			SCSE_SET_FLAG(sc_sess, SCSE_FL_STARTTLS);
			goto ehlo;
		}
		else
		{
			/* more clean up?? */
			SSL_free(sc_sess->scse_con);
			sc_sess->scse_con = NULL;
		}
	}
  notls:
#endif /* SM_USE_TLS */
	sc_sess->scse_state = SCSE_ST_OPEN;
	ret = SM_SUCCESS;
	goto done;

  fail:
	sc_sess_close(sc_t_ctx);
	if (!SCSE_IS_FLAG(sc_sess, SCSE_FL_LOGGED))
	{
		sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
			SC_LCAT_CLIENT, SC_LMOD_CLIENT,
			SM_LOG_WARN, 8,
			"sev=WARN, func=sc_sess_open, thread=%u, da_sess=%s, where=connection, status=failed, state=0x%x, ret=%m",
			thr_id, sc_sess->scse_id, sc_sess->scse_err_st, ret);
		SCSE_SET_FLAG(sc_sess, SCSE_FL_LOGGED);
	}
	/* error state? */

  done:
	return ret;
}

/*
**  SC_SESS_CLOSE -- close SMTPC session
**
**	Parameters:
**		sc_t_ctx -- SMTPC thread context
**
**	Returns:
**		SMTP reply code or error code
*/

sm_ret_T
sc_sess_close(sc_t_ctx_P sc_t_ctx)
{
	uint thr_id;
	sm_ret_T ret;
	sc_sess_P sc_sess;

	SM_REQUIRE(sc_t_ctx != NULL);
	sc_sess = sc_t_ctx->sct_sess;
	SM_REQUIRE(sc_sess != NULL);
	thr_id = sc_t_ctx->sct_thr_id;
	sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
		SC_LCAT_CLIENT, SC_LMOD_CLIENT,
		SM_LOG_INFO, 14,
		"sev=INFO, func=sc_sess_close, thread=%u, da_sess=%s, where=close, fp=%p",
		thr_id, sc_sess->scse_id, sc_sess->scse_fp);

	RQST_COUNT(sc_t_ctx->sct_sc_ctx)++;

	/*
	**  Already closed? Check also status?
	**  Don't send QUIT if we had an I/O error in the session...
	**  What about 421?
	*/

	if (sc_sess->scse_fp != NULL &&
	    !SCSE_IS_FLAG(sc_sess, SCSE_FL_IO_ERR) &&
	    sc_sess->scse_state >= SCSE_ST_CONNECTED &&
	    sc_sess->scse_state < SCSE_ST_CLOSED &&
	    sm_is_success(ret = sm_str_scopy(sc_sess->scse_wr, "QUIT\r\n")))
	{
		ret = sc_command(sc_t_ctx, SC_PHASE_QUIT);
		if (ret != SMTP_OK)
		{
			sm_log_write(sc_t_ctx->sct_sc_ctx->scc_lctx,
				SC_LCAT_CLIENT, SC_LMOD_CLIENT,
				SM_LOG_INFO, 14,
				"sev=INFO, func=sc_sess_close, thread=%u, da_sess=%s, quit=%m",
				thr_id, sc_sess->scse_id, ret);
		}
	}

	if (sc_sess->scse_fp != NULL)
	{
		sm_io_close(sc_sess->scse_fp);
		sc_sess->scse_fp = NULL;
	}

#if 0
//	if (SCSE_IS_FLAG(sc_sess, SCSE_FL_TELL_QMGR))
//	{
//		sc_sess->scse_err_st = 0;
//		ret = sc_c2q(sc_t_ctx, RT_C2Q_SECLSD, 0 /* SESS_TIMEOUT? */,
//				&c2q_ctx);
//		SCSE_CLR_FLAG(sc_sess, SCSE_FL_TELL_QMGR);
//	}
#endif /* 0 */

	sc_sess->scse_state = SCSE_ST_CLOSED;
#if 0
//	sc_sess->scse_cap = SCSE_CAP_NONE;
//	sc_sess->scse_flags = SCSE_FL_NONE;
#endif /* 0 */
#if SC_STATS
	--BUSY(sc_t_ctx);
#endif
	return SM_SUCCESS;
}
