/*
 * Copyright (c) 2002, 2004, 2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: strstr.c,v 1.18 2005/06/02 19:00:37 ca Exp $")

#include "sm/assert.h"
#include "sm/magic.h"
#include "sm/error.h"
#include "sm/memops.h"
#include "sm/rpool.h"
#include "sm/varargs.h"
#include "sm/limits.h"
#include "sm/str.h"
#include "sm/str-int.h"

/*
**  SM_STR_SCPY -- Create a str object and copy a char * into it.
**
**	Creates a new str object of size strlen(str) and copies
**	contents of str into it (without trailing '\0').
**
**	Parameters:
**		rpool -- The rpool in which to allocate.
**		str -- '\0' terminated str to copy into new object.
**		maxlen -- Maximum length of str data.
**
**	Return:
**		New str object.
**		NULL on error.
*/

sm_str_P
sm_str_scpy(sm_rpool_P rpool, const char *char_str, uint maxlen)
{
	sm_str_P str;
	uint len;

	SM_REQUIRE(char_str != NULL);

	/* trailing '\0' is not copied */
	len = strlen(char_str);
	str = sm_str_new(rpool, len, maxlen);
	if (str == NULL)
		return NULL;
	sm_memcpy(str->sm_str_base, char_str, len);
	str->sm_str_len = len;
	return str;
}

/*
**  SM_STR_SCPYN -- Create a str object and copy n bytes from src into it.
**
**	Parameters:
**		rpool -- The rpool in which to allocate.
**		src -- Byte array to copy from.
**		n -- Number of bytes to copy from src.
**		maxlen -- Maximum length of str data.
**
**	Returns:
**		New str object.
**		NULL on error.
*/

sm_str_P
sm_str_scpyn(sm_rpool_P rpool, const char *src, uint n, uint maxlen)
{
	sm_str_P str;

	SM_REQUIRE(src != NULL);
	str = sm_str_new(rpool, n, maxlen);
	if (str == NULL)
		return NULL;
	sm_memcpy(str->sm_str_base, src, n);
	str->sm_str_len = n;
	return str;
}

/*
**  SM_STR_SCAT -- Appends a copy of a char * onto the end of a sm_str_P
**
**	Parameters:
**		str -- sm_str_P object to append onto.
**		append -- '\0' terminated char * to append.
**
**	Returns:
**		usual error code
*/

sm_ret_T
sm_str_scat(sm_str_P str, const char *append)
{
	uint append_len;

	SM_IS_BUF(str);
	SM_REQUIRE(append != NULL);

	append_len = strlen(append);
	return sm_str_scatn(str, append, append_len);
}

/*
**  SM_STR_TERM -- Make sure data is '\0' terminated.
**
**	Parameters:
**		str -- sm_str_P object to append onto.
**
**	Returns:
**		usual sm_error code; ENOMEM, SM_E_OVFLW_SC, SM_E_OVFLW_NS
*/

sm_ret_T
sm_str_term(sm_str_P str)
{
	SM_IS_BUF(str);

	if (SM_STR_LAST(str) == '\0')
		return SM_SUCCESS;
	if (str->sm_str_len == str->sm_str_size)
	{
		uint new_alloc;

		new_alloc = str->sm_str_len + 1;

		/* overflow? */
		if (new_alloc < str->sm_str_len)
			return sm_error_perm(SM_EM_STR, SM_E_OVFLW_SC);
		SM_STR_INCREASE_R(str, new_alloc);
	}
	str->sm_str_base[str->sm_str_len] = '\0';
	return SM_SUCCESS;
}
