/*
 * Copyright (c) 2002-2005 Sendmail, Inc. and its suppliers.
 *      All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 *
 *	$Id: dnstsk.h,v 1.28 2005/06/30 19:56:21 ca Exp $
 */

#ifndef SM_DNS_TSK_H
#define SM_DNS_TSK_H 1

#include "sm/generic.h"
#include "sm/types.h"
#include "sm/str.h"
#include "sm/bhtable.h"
#include "sm/dns.h"
#include "sm/log.h"
#if SM_USE_PTHREADS
# include "sm/pthread.h"
# include "sm/evthr.h"
#endif

/*
**  Not defined on HP-UX by default? How to get it?  Should be in sm/net.h?
**  requires _XOPEN_SOURCE_EXTENDED?
*/

#ifndef MSG_WAITALL
# define MSG_WAITALL	0x40
#endif

#define LOCALHOST_IP	INADDR_LOOPBACK
#define DNS_PORT	53

/* maximum number of DNS tasks */
#define SM_DNS_MAX_TSKS	4u

/* Hash table size */
#define DNS_HT_SIZE	1023
#define DNS_HT_MAX	4095

/* Maximum length of query/answer (way too large?) */
#define MAX_QUERY_SIZE	(64 * 1024)


/* list of requests can be SIMPLEQ */
TAILQ_HEAD(dns_rql_S, dns_req_S);

typedef struct dns_tsk_S	dns_tsk_T, *dns_tsk_P;


/*
**  DNS Manager Context
**
**  The DNS manager uses a request list and a hash table to store DNS requests.
**  The (incoming) request list contains (in arriving order) the list of
**  unique requests.
**  The hash table contains all requests. It is used for two purposes:
**  1. to avoid querying a DNS resolver for the same item again if
**  there is already an open query.
**  2. to lookup requests after an answer is returned by a DNS
**  resolver.
**  The key for a hash table entry is the DNS query plus its type.
**
**  The incoming request list acts as a FIFO buffer between the
**  clients that send request to this DNS library and the DNS server:
**  requests from clients are appended to the list.
**  The wait list stores entries that are waiting for answers from a
**  DNS server. It might not be necessary to keep the list sorted in
**  case the list is short such that a search for expired entries is fast.
**
**  Since there will be only one request per query, different timeouts
**  don't work well. For example: ask for MX record for A with timeout 5s,
**  then ask for MX record for A with timeout 20s: the second request
**  will not be sent to a DNS server, hence if the requests times out
**  after 6 seconds, the second request will be treated as (temp) failed too
**  even though it might have succeeded.
*/

struct dns_mgr_ctx_S
{
	/* magic? */
	uint		 dnsmgr_flags;

	sm_log_ctx_P	 dnsmgr_lctx;	/* log context */

/* XXX should these be per DNS task? */
	/* sorted list of incoming requests */
	dns_rql_T	 dnsmgr_increq_hd;

	/* sorted list of requests waiting for answers */
	dns_rql_T	 dnsmgr_waitreq_hd;

	/* hash table to store all requests */
	bht_P		 dnsmgr_req_ht;

	uint		 dnsmgr_timeout;	/* timeout for queries (s) */

	uint		 dnsmgr_ntsks;	/* number of DNS tasks */
	dns_tsk_P	 dnsmgr_dnstsks[SM_DNS_MAX_TSKS];
	uint32_t	 dnsmgr_tskstatus[SM_DNS_MAX_TSKS];

#if SM_USE_PTHREADS
	uint		 dnsmgr_ctsk;	/* current DNS task to use for query */
	sm_evthr_task_P	 dnsmgr_tsk[SM_DNS_MAX_TSKS];
	sm_evthr_task_P	 dnsmgr_cleanup;
	pthread_mutex_t	 dnsmgr_mutex;	/* for the entire context? */
#endif
};

/* states for DNS tasks, must be ordered */
#define DNSTSK_ST_NONE	0x0000u
#define DNSTSK_ST_INIT	0x0001u
#define DNSTSK_ST_OK	0x0002u

/*
**  DNS task context
*/

struct dns_tsk_S
{
	/* XXX int or statethreads socket */
	int		 dnstsk_fd;		/* socket */
	dns_mgr_ctx_P	 dnstsk_mgr;		/* DNS manager */
	uint		 dnstsk_flags;		/* operating flags */
#if 0
	/*
	**  counter for queries that timed out: if a limit is exceeded
	**  consider the server "dead" (unresponsive) and exclude it
	**  from the list
	*/

	uint		 dnstsk_timeouts;
	uint		 dnstsk_maxtimeouts;
#endif /* 0 */
	sockaddr_in_T	 dnstsk_sin;		/* socket description */

	sm_str_P	 dnstsk_rd;	/* read buffer */
	sm_str_P	 dnstsk_wr;	/* write buffer */
};

#define DNS_TSK_SHUTDOWN	0x0001u		/* shutdown on next run */
#define DNS_TSK_USETCP		0x0002u		/* use TCP instead of UDP */
#define DNS_TSK_CONNECTUDP	0x0004u		/* use connected UDP socket */
#define DNS_TSK_NOROTATE	0x0008u		/* don't rotate servers */
#define DNS_TSK_BLOCKBADHOSTS	0x0010u		/* block bad nameservers */

sm_ret_T dns_mgr_ctx_del(dns_mgr_ctx_P _dns_mgr_ctx);
sm_ret_T dns_mgr_ctx_new(uint _flags, uint _timeout, uint _htsize, uint _htlimit, dns_mgr_ctx_P *_pdns_mgr_ctx);
sm_ret_T dns_mgr_set_timeout(dns_mgr_ctx_P _dns_mgr_ctx, uint _timeout);
sm_ret_T dns_tsk_new(dns_mgr_ctx_P _dns_mgr_ctx, uint _flags, ipv4_T ipv4, dns_tsk_P *_pdns_tsk);
#if SM_USE_PTHREADS
sm_ret_T dns_tsk_start(dns_mgr_ctx_P _dns_mgr_ctx, sm_evthr_ctx_P _evthr_ctx);
sm_ret_T dns_comm_tsk(sm_evthr_task_P _tsk);
#else
/* sm_ret_T dns_comm_tsk(sm_evthr_task_P _tsk); */
#endif
sm_ret_T dns_req_add(dns_mgr_ctx_P _dns_mgr_ctx, sm_cstr_P _query, dns_type_T _type, uint _timeout, dns_callback_F *_fct, void *_ctx);
void	 dns_req_del(void *_value, void *_key, void *_ctx);

sm_ret_T dns_receive(dns_tsk_P dns_tsk);
sm_ret_T dns_send(dns_tsk_P dns_tsk);

#endif /* ! SM_DNS_TSK_H */
