/*
 * Copyright (c) 2000 Qualcomm Incorporated.  All rights reserved.
 * The file License.txt specifies the terms for use, modification,
 * and redistribution.
 *
 * Revisions:
 *
 *     08/17/00  [RCG]
 *             - Added 'bulldb-nonfatal'.
 *             - Added 'bulldb-max-tries'.
 *             - Added ability to restrict options so users can't set them.
 *
 *     06/27/00  [rcg]
 *             - Added guts.
 *
 */


#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>

#include "popper.h"
#include "utils.h"
#include "string_util.h"


#define COMMENT_CHAR    '#'
#define MAX_RECURSION   100

#ifndef MIN
#  define MIN(A,B) ( ( (A) < (B) ) ? (A) : (B) )
#endif

#ifndef MAX
#  define MAX(A,B) ( ( (A) > (B) ) ? (A) : (B) )
#endif


/*
 * The possible errors.
 */
typedef enum
{
    kER_NOFILE = 1,
    kER_LINE_LEN,
    kER_EXP_SET,
    kER_UNKN_OPT,
    kER_RESTR,
    kER_NOT_BOOL,
    kER_EXP_EQL,
    kER_BAD_VAL,
    kER_EXP_EOL,
    kER_RECURS,
    
    kER_LAST_ERROR
} error_code_type;


/*
 * The values of the recognized options.
 */
typedef enum
{
    kBULLDIR = 1,           /* -b bulldir       */
    kBULLDBNONFATAL,        /* -B               */
    kBULLDBMAXTRIES,        /*    (no flag)     */
    kDCASEUSER,             /* -c               */
    kDEBUG,                 /* -d               */
    kDRACHOST,              /* -D drac-host     */
    kANNLOGINDELAY,         /* -eLOGIN_DELAY=x  */
    kANNEXPIRE,             /* -eEXPIRE=x       */
    kCONFIGFILE,            /* -f config-file   */
    kKERBEROS,              /* -k               */
    kKERBSERVICE,           /* -K service-name  */
    kLOCKCHECK,             /* -L lock-refresh  */
    kCLEARPASS,             /* -p 0|1|2|3       */
    kREVLOOKUP,             /* -R               */  /* SENSE REVERSED! */
    kSTATS,                 /* -s               */
    kSERVER,                /* -S               */
    kTRACEFILE,             /* -t logfile       */
    kTIMEOUT,               /* -T timeout       */
    kUSEROPTS,              /* -u               */

    LAST_OPT_VAL
} opt_val_type;


/*
 * Table of recognized options.
 */
static config_table options [] =
{
/*    name                                type      restr  value                */
    { "announce-login-delay"            , CfgInt  , FALSE, kANNLOGINDELAY      },
    { "announce-expire"                 , CfgInt  , FALSE, kANNEXPIRE          },
    { "bulldir"                         , CfgStr  , FALSE, kBULLDIR            },
#ifdef    BULLDB
    { "bulldb-nonfatal"                 , CfgBool , FALSE, kBULLDBNONFATAL     },
    { "bulldb-max-tries"                , CfgInt  , FALSE, kBULLDBMAXTRIES     },
#endif /* BULLDB */
    { "clear-text-password"             , CfgInt  , TRUE , kCLEARPASS          },
    { "config-file"                     , CfgStr  , FALSE, kCONFIGFILE         },
    { "debug"                           , CfgBool , FALSE, kDEBUG              },
    { "downcase-user"                   , CfgBool , TRUE , kDCASEUSER          },
#ifdef    DRAC_AUTH
    { "drac-host"                       , CfgStr  , FALSE, kDRACHOST           },
#endif /* DRAC_AUTH */
#ifdef    KERBEROS
    { "kerberos"                        , CfgBool , TRUE , kKERBEROS           },
    { "kerberos-service"                , CfgStr  , TRUE , kKERBSERVICE        },
#endif /* KERBEROS */
    { "mail-lock-check"                 , CfgInt  , FALSE, kLOCKCHECK          },
    { "reverse-lookup"                  , CfgBool , TRUE , kREVLOOKUP          },
    { "server-mode"                     , CfgBool , FALSE, kSERVER             },
    { "statistics"                      , CfgBool , FALSE, kSTATS              },
    { "timeout"                         , CfgInt  , FALSE, kTIMEOUT            },
    { "tracefile"                       , CfgStr  , FALSE, kTRACEFILE          },
    { "user-options"                    , CfgInt  , TRUE , kUSEROPTS           },

    { NULL                              , CfgBad  , FALSE, LAST_OPT_VAL        }
};


/*
 * Globals
 */
extern int login_delay;
extern int expire;
extern int no_rev_lookup;
extern int pop_timeout;


/*
 * Prototypes
 */
static void          tilt (POP *p, error_code_type er, char *token, long len, size_t line_no, char *cfile, config_table *ctp, WHENCE);
       config_table *find_option (char *token, long len);
       long          length_to_white_space (char *token, long line_length);
       int           get_token_length (char *token, long line_length);
       void          skip_white_space (char **token, long *line_length);
       void          skip_token (char **token, long *token_len, long *line_len);
static void          show_result (POP *p, config_table *opt, int iVal, const char *sVal);
       pop_result    handle_value (POP *p, config_table *opt, char *pval, long plen, BOOL bUser);


/*
 * Log an error message.
 */
static void
tilt ( POP              *p, 
       error_code_type   er, 
       char             *token, 
       long              len, 
       size_t            line_no, 
       char             *cfile,
       config_table     *ctp,
       WHENCE )
{
    switch ( er )
    {
        case kER_NOFILE:
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "Unable to open configuration file %s: %s (%d)",
                      cfile, strerror(errno), errno );
            break;

        case kER_LINE_LEN:
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "Config file %s line %d too long",
                      cfile, line_no );
            break;

        case kER_EXP_SET:
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "Expected \"set\" or \"reset\", found \"%.*s\" "
                      "at line %d of config file %s",
                      MIN ( 128, len), token, line_no, cfile );
            break;

        case kER_UNKN_OPT:
            { /* local block */
            size_t        sz  = 0;
            char         *buf = NULL;

            for ( ctp = options; ctp->name != NULL; ctp++ )
                sz += strlen ( ctp->name ) + 2;
            buf = malloc ( sz + 1 );
            if ( buf != NULL )
            {
                buf[0] = '\0';
                for ( ctp = options; ctp->name != NULL; ctp++ )
                {
                    strcat ( buf, ctp->name );
                    strcat ( buf, ", "     );
                }
                buf [ strlen ( buf ) - 2 ] = '\0';
            }
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "Unrecognized option: \"%.*s\" at line %d of config "
                      "file %s; valid options are: %s",
                      MIN ( 128, len), token, line_no, cfile, buf );
            } /* local block */
            break;

        case kER_RESTR:
            pop_log ( p, POP_DEBUG, fn, ln, /* fn and ln are part of WHENCE */
                      "Attempt to use restricted option \"%s\" at line %d of "
                      "user config file %s",
                      ctp->name, line_no, cfile );
            break;

        case kER_NOT_BOOL:
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "%s is not a boolean option; scanning \"%.*s\" at "
                      "line %d of config file %s",
                      ctp->name, MIN ( 128, len), token, line_no, cfile );
            break;

        case kER_EXP_EQL:
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "\"=\" expected; found \"%.*s\" at "
                      "line %d of config file %s",
                      MIN ( 128, len), token, line_no, cfile );
            break;

        case kER_RECURS:
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "Config file nesting exceeds %d; will not process "
                      "config file %s",
                      MAX_RECURSION, cfile );
            break;

        case kER_EXP_EOL:
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "Expected comment or end of line; found \"%.*s\" at "
                      "line %d of config file %s",
                      MIN ( 128, len), token, line_no, cfile );
            break;

        default:
            pop_log ( p, POP_PRIORITY, fn, ln, /* fn and ln are part of WHENCE */
                      "Unknown error; scanning \"%.*s\" at "
                      "line %d of config file %s",
                      MIN ( 128, len), token, line_no, cfile );
            break;
    } /* switch ( er ) */
}


/*
 * Lookup an option table entry by name
 */
config_table *
find_option ( char *token, long len )
{
    config_table *opt = NULL;


    for ( opt = options; opt->name != NULL; opt++ )
    {
        if ( equal_strings ( opt->name, -1, token, len ) == TRUE )
            return opt;
    }

    return NULL;
}


/*
 * Return number of characters until white space.
 */
long
length_to_white_space ( char *token, long line_length )
{
    char *orig_ptr = token;


    while ( line_length > 0 && *token != ' ' && *token != '\t' )
    {
        line_length--;
        token++;
    }
    
    return ( token - orig_ptr );
}


/*
 * Return length of current token.
 */
int
get_token_length ( char *token, long line_length )
{
    char *orig_ptr = token;


    if ( line_length > 0 && CharIsAlnum ( *token ) == FALSE )
        return 1; /* non-alphanumerics are their own token */

    while ( line_length > 0 && ( CharIsAlnum ( *token ) || *token == '-' ) )
    {
        token++;
        line_length--;
    }
    
    return ( token - orig_ptr );
}


/*
 * Skips past white space.
 */
void
skip_white_space ( char **token, long *line_length )
{
    while ( *line_length > 0 && ( **token == ' ' || **token == '\t' ) )
    {
        *line_length   = *line_length   -1;
        *token         = *token         +1;
    }
}


/*
 * Skips to next token.
 */
void
skip_token ( char **token, long *token_len, long *line_len )
{
    *token    += *token_len;
    *line_len -= *token_len;
    skip_white_space ( token, line_len );
    *token_len = get_token_length ( *token, *line_len );
}


/*
 * Writes debug entry showing an item set.
 */
static void
show_result ( POP *p, config_table *opt, int iVal, const char *sVal )
{
    switch ( opt->type )
    {
        case CfgBool:
            DEBUG_LOG2 ( p, "Set %s to %s", opt->name,
                         iVal ? "true" : "false" );
            break;

        case CfgInt:
            DEBUG_LOG2 ( p, "Set %s to %d", opt->name, iVal );
            break;

        case CfgStr:
            DEBUG_LOG2 ( p, "Set %s to %.255s", opt->name, sVal );
            break;

        default:
            /* we should never get here */
            DEBUG_LOG1 ( p, "** INTERNAL ERROR processing %s **",
                         opt->name );
            break;
    }
}


/*
 * Sets a value.
 */
pop_result
handle_value ( POP *p, config_table *opt, char *pval, long plen, BOOL bUser )
{

    long              iVal       = 0;
    config_opt_type   val_type   = CfgBad;


    if ( isdigit ( (int) *pval ) )
    {
        iVal     = atol ( pval );
        val_type = CfgInt;
    }
    else
    if ( isalnum ( (int) *pval ) || ispunct ( (int) *pval )  )
    {
        val_type = CfgStr;
    }
    else
        return POP_FAILURE;

    switch ( opt->type )
    {
        case CfgBool:
            if ( val_type == CfgStr )
            {
                if ( equal_strings ( pval, plen, "true",  4 ) == TRUE )
                {
                    val_type = CfgInt;
                    iVal     = 1;
                }
                else
                if ( equal_strings ( pval, plen, "false", 5 ) == TRUE )
                {
                    val_type = CfgInt;
                    iVal     = 0;
                }
            }
            
            if ( val_type != CfgInt || iVal < 0 || iVal > 1 )
                return POP_FAILURE;
            break;

        case CfgInt:
            if ( val_type != CfgInt )
                return POP_FAILURE;
            break;

        case CfgStr:
            break; /* we'll take anything as a string */

        default:
            /* we should never get here */
            pop_log ( p, POP_PRIORITY, HERE,
                      "Invalid option type for option \"%s\"", opt->name );
            return POP_FAILURE;
            break;
    } /* switch opt->type */

    switch ( opt->value )
    {
        case kBULLDIR:
            p->bulldir = string_copy ( pval, plen );
            if ( p->bulldir == NULL )
            {
                pop_log ( p, POP_PRIORITY, HERE,
                          "Unable to allocate %ld bytes", plen );
                return POP_FAILURE;
            }
            show_result ( p, opt, 0, p->bulldir );
            break;

        case kBULLDBNONFATAL:
#ifdef    BULLDB
            p->bulldb_nonfatal  = iVal;
            show_result ( p, opt, p->bulldb_nonfatal, NULL );
#else
            pop_log ( p, POP_PRIORITY, HERE,
                      "BULLDB not set; can't set %s", opt->name );
            return POP_FAILURE;
#endif /* BULLDB */
            break;

        case kBULLDBMAXTRIES:
#ifdef    BULLDB
            p->bulldb_max_tries  = iVal;
            show_result ( p, opt, p->bulldb_max_tries, NULL );
#else
            pop_log ( p, POP_PRIORITY, HERE,
                      "BULLDB not set; can't set %s", opt->name );
            return POP_FAILURE;
#endif /* BULLDB */
            break;

        case kDCASEUSER:
            p->downcase_user = iVal;
            show_result ( p, opt, p->downcase_user, NULL );
            break;

        case kDEBUG:
            p->debug = iVal;
            show_result ( p, opt, p->debug, NULL );
            break;

        case kDRACHOST:
#ifdef    DRAC_AUTH
        /* drac host specified */
            p->drac_host = string_copy ( pval, plen );
            if ( p->drac_host == NULL )
            {
                pop_log ( p, POP_PRIORITY, HERE,
                          "Unable to allocate %ld bytes", plen );
                return POP_FAILURE;
            }
            show_result ( p, opt, 0, p->drac_host );
#else
            pop_log ( p, POP_PRIORITY, HERE, 
                      "DRAC_AUTH not set; can't set %s", opt->name ); 
            return POP_FAILURE;
#endif /* DRAC_AUTH */
            break;

        case kANNLOGINDELAY:
            login_delay = iVal;
            DEBUG_LOG1 ( p, "Announcing login_delay of %d", login_delay );
            break;

        case kANNEXPIRE:
            expire = iVal;
            DEBUG_LOG1 ( p, "Announcing expire %d", expire );
            break;

        case kCONFIGFILE:
            * (pval + plen) = '\0';
            if ( pop_config ( p, pval, bUser ) == POP_FAILURE )
                return POP_FAILURE;

        case kKERBEROS:
#ifdef KERBEROS
            p->kerberos = iVal;
            show_result ( p, opt, p->kerberos, NULL );
#else
            pop_log ( p, POP_PRIORITY, HERE,
                      "KERBEROS not set; can't set %s", opt->name );
            return POP_FAILURE;
#endif  /* KERBEROS */
            break;

        case kKERBSERVICE:
#ifdef KERBEROS
            p->kerberos_service = string_copy ( pval, plen );
            if ( p->kerberos_service == NULL )
            {
                pop_log ( p, POP_PRIORITY, HERE,
                          "Unable to allocate %ld bytes", plen );
                return POP_FAILURE;
            }
            show_result ( p, opt, 0, p->kerberos_service );
#else
            pop_log ( p, POP_PRIORITY, HERE,
                      "KERBEROS not set; can't set %s", opt->name );
            return POP_FAILURE;
#endif  /* KERBEROS */
            break;

        case kLOCKCHECK:   /*  Touch the mail lock every this many messages (crude)  */
            p->check_lock_refresh = iVal;
            DEBUG_LOG1 ( p, "Checking for maillock refresh every %i msgs", 
                         p->check_lock_refresh );
            break;

        case kCLEARPASS:
#if defined(APOP) || defined(SCRAM)
        /*  Set clear text password handling */
            switch ( iVal )
            {
                case 0:
                    p->AllowClearText = ClearTextDefault;
                    break;
                case 1:
                    p->AllowClearText = ClearTextNever;
                    break;
                case 2:
                    p->AllowClearText = ClearTextAlways;
                    break;
                case 3:
                    p->AllowClearText = ClearTextLocal;
                    break;
                default:
                    return POP_FAILURE;
            }
            show_result ( p, opt, p->AllowClearText, NULL );
#else
            pop_log ( p, POP_PRIORITY, HERE, 
                      "Neither APOP not SCRAM set; can't set %s",
                      opt->name );
            return POP_FAILURE;
#endif /* APOP or SCRAM */
        break;

        case kREVLOOKUP:
            no_rev_lookup = ( iVal ? FALSE : TRUE );
            DEBUG_LOG1 ( p, "Set reverse lookups %s", 
                         (no_rev_lookup ? "false" : "true") );
            break;

        case kSTATS:
            p->stats = iVal;
            show_result ( p, opt,  p->stats, NULL );
            break;

        case kSERVER:
            p->server_mode = iVal;
            show_result ( p, opt, p->server_mode, NULL );
            break;

        case kTRACEFILE:
            p->debug = TRUE;
            if ( p->trace != NULL )
            {
                pop_log ( p, POP_PRIORITY, HERE,
                          "Unable to log to %.*s; log file already specified",
                           (int) MIN ( 256, plen ), pval );
                return POP_FAILURE;
            }
            * (pval + plen) = '\0';
            p->trace = fopen ( pval, "a+" );
            if ( p->trace == NULL )
            {
                pop_log ( p, POP_PRIORITY, HERE,
                          "Unable to open trace file \"%s\": %s (%d)",
                          pval, STRERROR(errno), errno );
                return POP_FAILURE;
            }
            DEBUG_LOG1 ( p, "Trace and Debug destination is file \"%s\"",
                         pval );
            break;

        case kTIMEOUT:
            pop_timeout = iVal;
            show_result ( p, opt, pop_timeout, NULL );
            break;

        case kUSEROPTS:
            p->user_opts = iVal;
            show_result ( p, opt, p->user_opts, NULL );
            break;

        default:
            pop_log ( p, POP_PRIORITY, HERE,
                      "Unhandled option value: \"%s\"",
                      opt->name );
            return POP_FAILURE;
            break;
    } /* switch ( opt->value ) */

    return POP_SUCCESS;
}


/*
 * Reads a config file and processes it.
 *
 * bUser is TRUE if this is a user's own config file.
 */
pop_result
pop_config ( POP *p, char *config_file, BOOL bUser )
{
    FILE           *config     = NULL;
    size_t          line_no    = 0;
    char            buffer [ MAXLINELEN ] = "";
    char           *ptok       = NULL;
    long            line_len   = 0;
    long            tok_len    = 0;
    int             bReset     = FALSE;
    config_table   *opt        = NULL;
    pop_result      rslt       = POP_FAILURE;
    pop_result      err        = POP_FAILURE;
    static int      depth      = 0;


    DEBUG_LOG2 ( p, "Processing config file '%.256s'; bUser=%d",
                 config_file, bUser );

    if ( depth++ > MAX_RECURSION )
    {
        tilt ( p, kER_RECURS, ptok, tok_len, line_no, config_file, opt, HERE );
        goto Exit;
    }

    config = fopen ( config_file, "r" );
    if ( config == NULL )
    {
        tilt ( p, kER_NOFILE, ptok, tok_len, line_no, config_file, opt, HERE );
        goto Exit;
    }

    ptok = fgets ( buffer, MAXLINELEN -1, config );
    while ( ptok != NULL )
    {
        line_len = strlen ( ptok );
        
        /*
         * Make sure line was not too long for buffer.  Also
         * get rid of trailing newline.
         */
        if ( buffer [ line_len -1 ] != '\n' )
        {
            tilt ( p, kER_LINE_LEN, ptok, tok_len, line_no, config_file, opt, HERE );
            goto Exit;
        }
        buffer [ --line_len ] = '\0';
        
        /*
         * Skip leading white space.
         */
        skip_white_space ( &ptok, &line_len );
        DEBUG_LOG2 ( p, "..read (%lu): %.256s", line_len, ptok );
        
        /*
         * Ignore empty and comment lines.
         */
        if ( line_len != 0 && *ptok!= COMMENT_CHAR )
        { /* process the line */

            /*
             * Get token length.
             */
            tok_len = get_token_length ( ptok, line_len );

            /*
             * There should be a "set" (or even "reset" if
             * the option is boolean) before the name.
             */
            if ( equal_strings ( ptok, tok_len, "reset", 5 ) == TRUE )
                bReset = TRUE;
            else
            if ( equal_strings ( ptok, tok_len, "set",   3 ) == TRUE )
                bReset = FALSE;
            else
            {
                tilt ( p, kER_EXP_SET, ptok, tok_len, line_no, config_file, opt, HERE );
                goto Exit;
            }
            
            /*
             * Skip to next token.
             */
            skip_token ( &ptok, &tok_len, &line_len );

            /*
             * See if the option is known.
             */
            opt = find_option ( ptok, tok_len );
            if ( opt == NULL )
            {
                tilt ( p, kER_UNKN_OPT, ptok, tok_len, line_no, config_file, opt, HERE );
                goto Exit;
            }
            
            /*
             * See if a restricted option is used in a user's file
             */
            if ( bUser && opt->restricted )
            {
                tilt ( p, kER_RESTR, ptok, tok_len, line_no, config_file, opt, HERE );
                goto Exit;
            }

            /*
             * See if 'reset' used with a non-boolean option.
             */
            if ( bReset == TRUE && opt->type != CfgBool )
            {
                tilt ( p, kER_NOT_BOOL, ptok, tok_len, line_no, config_file, opt, HERE );
                goto Exit;
            }

            /*
             * See if there is an "=" followed by a value.
             */
            skip_token ( &ptok, &tok_len, &line_len );
            if ( ( tok_len == 0 || *ptok != '=' )
                 && opt->type != CfgBool )
            {
                tilt ( p, kER_EXP_EQL, ptok, tok_len, line_no, config_file, opt, HERE );
                goto Exit;
            }

            if ( tok_len == 1 && *ptok == '=' )
            {
                /*
                 * Try and handle the value.
                 */
                skip_token ( &ptok, &tok_len, &line_len );
                if ( opt->type == CfgStr )
                    tok_len = length_to_white_space ( ptok, line_len );

                err = handle_value ( p, opt, ptok, tok_len, bUser );
                if ( err == POP_FAILURE )
                {
                    tilt ( p, kER_BAD_VAL, ptok, tok_len, line_no, config_file, opt, HERE );
                    goto Exit;
                }
                skip_token ( &ptok, &tok_len, &line_len ); /* consume the token */
            }
            else
            if ( tok_len == 0 || *ptok == COMMENT_CHAR )
            { /* implicit value */
                if ( bReset )
                    err = handle_value ( p, opt, "false", 5, bUser );
                else
                    err = handle_value ( p, opt, "true",  4, bUser );
                if ( err == POP_FAILURE )
                {
                    tilt ( p, kER_BAD_VAL, ptok, tok_len, line_no, config_file, opt, HERE );
                    goto Exit;
                }
            } /* implicit value */
            else
            {
                tilt ( p, kER_EXP_EQL, ptok, tok_len, line_no, config_file, opt, HERE );
                goto Exit;
            }
            
            /*
             * Make sure there is no extra junk after the value
             */
            if ( tok_len != 0 && *ptok != COMMENT_CHAR )
            {
                tilt ( p, kER_EXP_EOL, ptok, tok_len, line_no, config_file, opt, HERE );
                goto Exit;
            }

        } /* process the line */

        /*
         * get the next line, if any
         */
        ptok = fgets ( buffer, MAXLINELEN -1, config );
        line_no++;
    } /* while ( ptok != NULL ) */
    
    rslt = POP_SUCCESS;

Exit:
    if ( rslt != POP_SUCCESS )
    { /* cleanup */
        if ( config != NULL )
        {
            fclose ( config );
            config = NULL;
        }
    } /* cleanup */

    return rslt;
}
