/* popauth.c - manipulate POP authentication DB:
 *   aka "apopauth" [same thing really] and "scramauth" [manipulate 
 *   scram entries]
 */

/*
 * Copyright (c) 1998, 1999 Qualcomm Incorporated. All rights reserved.
 *
 * Revisions:
 *    4/21/00  [Randall Gellens]
 *           - Remove __STDC__ (some compilers don't define this when extensions
 *             are enabled, leading to the erroneous conclusion that ANSI C is
 *             not supported.  These days, it is far more likely that a compiler
 *             fails to correctly handle K&R syntax than ANSI.)
 *    4/29/99  [Randall Gellens]
 *           - Don't reference auth_file when GDBM is undefined (syntax error)
 *    2/04/99  [Randall Gellens]
 *           - Don't flock() when using gdbm.
 *    2/13/98  [Rob Duwors]
 *           - Added SCRAM support.
 */

/*
 *. All fields for a given user entry are null terminated strings.
 *. APOP field come first [may be a null string if logically absent].
 *. SCRAM field come second [may be physically or logically absent].
 *. SCRAM fields are kept in base64 representation of the original 40 byte
 *   SCRAM verifier (salt, client id, and server key).
 *. Deleting a user still deletes the entire entry [APOP and SCRAM 
 *   disappear].
 *. parameter -list also shows the presence of APOP and SCRAM entries.
 *. old authentication databases and APOP password fields can be used 
 *   as is [the missing SCRAM field will be treated as null]
 *. a specific field [APOP or SCRAM] can be cleared to null by using
 *   password "*", i.e. a single star character as the new password.
 *. scramauth by default will clear the APOP field [disabling APOP 
 *   access for that user].  This can be stopped by using the -only   
 *   switch.
 *. popauth/apopauth can ONLY change the APOP password, never distrubing 
 *   the the SCRAM field.
 *. APOP password and SCRAM pass phrases are not related to each other 
 *   in any way ([a]pop|scram)auth, i.e. the APOP password can not 
 *   compromise the SCRAM verifiers, except by user choice.
 */

#include "config.h"

#ifdef SCRAM
#  include <md5.h>
#  include <hmac-md5.h>
#  include <scram.h>
#endif

#ifdef GDBM
#  include <gdbm.h>
#else
#  if HAVE_NDBM_H
#    include <ndbm.h>
#  endif
#endif

#include <sys/types.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#ifndef HAVE_BCOPY
#  define bcopy(src,dest,len) (void) (memcpy(dest,src,len))
#  define bzero(dest,len)     (void) (memset(dest, (char)NULL, len))
#  define bcmp(b1,b2,n)       memcmp(b1,b2,n)
#endif

#ifndef HAVE_INDEX
#  define index(s,c)     strchr(s,c)
#  define rindex(s,c)        strrchr(s,c)
#endif

#include <flock.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

#if HAVE_SYS_FILE_H
#  include <sys/file.h>
#endif

#include <time.h>
#define TIME_T time_t

#ifdef SCRAM
extern int encode64();
extern int decode64();
#endif

#ifdef BSDI
#  define BSD44_DBM
#endif

#ifndef HAVE_STRERROR
char *strerror();
#endif

#define UID_T   uid_t


static struct mods {
       char   *name;
} modes[] = {
#define SCRAM_AUTH  0
    "scramauth",
#define APOP_AUTH   1
    "apopauth",
#define POP_AUTH    2 
    "popauth",
#define OTHER       3
    NULL
};


static struct swit {
    char *name;
} switches[] = {
#define INITSW  0
    "init", 
#define LISTSW  1
    "list", 
#define USERSW  2
    "user", 
#define DELESW  3
    "delete", 
#define ONLYSW  4
    "only",

    NULL,
};

static char   *program;

static void
adios(const char *fmt, ...)
{
    va_list ap;

    fprintf  ( stderr, "%s: ", program );
    va_start ( ap, fmt );
    vfprintf ( stderr, fmt, ap );
    fprintf  ( stderr, "\n" );
    va_end   ( fmt );
    exit(1);
}

#ifndef HAVE_STRDUP
#include <stddef.h>

char *
strdup(str)
        char *str;
{
    int len;
    char *copy;

    len = strlen(str) + 1;
    if (!(copy = malloc((u_int)len)))
    return((char *)NULL);
    bcopy(str, copy, len);
    return(copy);
}
#endif

/*
 * Obscure password so a cleartext search doesn't come up with
 * something interesting.
 *
 */

char *
obscure(string)
char *string;
{
    unsigned char *cp, *newstr;

    cp = newstr = (unsigned char *)strdup(string);

    while (*cp) {
        *cp++ ^= 0xff;
    }

    return((char *)newstr);
}

/* Use GNU_PASS for longer passwords on systems that support termios */

#ifndef GNU_PASS
char *getpass();
#else

/* Copyright (C) 1992, 1993, 1994 Free Software Foundation, Inc.
 * This file is part of the GNU C Library.

 * The GNU C Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 */
/* It is desireable to use this bit on systems that have it.
    The only bit of terminal state we want to twiddle is echoing, which is
   done in software; there is no need to change the state of the terminal
   hardware.  */

#include <stdio.h>
#include <termios.h>
#include <unistd.h>

#ifndef TCSASOFT
#define TCSASOFT 0
#endif

#ifdef SSIZET
typedef SSIZET ssize_t;
#endif

char *
getpass (prompt)
#if defined(HPUX)
char *prompt;
#else
const char *prompt;
#endif
{
  FILE *in, *out;
  struct termios t;
  int echo_off;
  static char *buf = NULL;
  static size_t bufsize = 0;
  ssize_t nread;

  /* Try to write to and read from the terminal if we can.
     If we can't open the terminal, use stderr and stdin.  */

  in = fopen ("/dev/tty", "w+");
  if (in == NULL)
    {
      in = stdin;
      out = stderr;
    }
  else
    out = in;

  /* Turn echoing off if it is on now.  */

  if (tcgetattr (fileno (in), &t) == 0)
    {
      if (t.c_lflag & ECHO)
    {
      t.c_lflag &= ~ECHO;
      echo_off = tcsetattr (fileno (in), TCSAFLUSH|TCSASOFT, &t) == 0;
      t.c_lflag |= ECHO;
    }
      else
    echo_off = 0;
    }
  else
    echo_off = 0;

  /* Write the prompt.  */
  fputs (prompt, out);
  fflush (out);

  /* Read the password.  */
#ifdef NO_GETLINE
  bufsize = 256;
  buf = (char *)malloc(256);
  nread = (fgets(buf, (size_t)bufsize, in) == NULL) ? 1 : strlen(buf);
  rewind(in);
  fputc('\n', out);
#else
  nread = __getline (&buf, &bufsize, in);
#endif
  if (nread < 0 && buf != NULL)
    buf[0] = '\0';
  else if (buf[nread - 1] == '\n')
    /* Remove the newline.  */
    buf[nread - 1] = '\0';

  /* Restore echoing.  */
  if (echo_off)
    (void) tcsetattr (fileno (in), TCSAFLUSH|TCSASOFT, &t);

  if (in != stdin)
    /* We opened the terminal; now close it.  */
    fclose (in);

  return buf;
}
#endif

/* ARGSUSED */

main (argc, argv)
int argc;
char   *argv[];
{
    UID_T   myuid;
    int     flags,
    i,
    apoplen,
    scramlen,
    delesw   = 0,
    initsw   = 0,
    onlysw   = 0,
    passtype = 0,
    insist,
    listsw   = 0,
    popuser  = 0;
    long    clock;
    char   *bp,
    *cp,
    *usersw = NULL,
    apopfld[BUFSIZ],
    scramfld[BUFSIZ],
    outfld[BUFSIZ],
    buf[100],
    obuf[100];
    struct  passwd *pw;
    datum   key,
    value;

#ifdef SCRAM
    SCRAM_MD5_VRFY  scram_verifiers;
#endif

#ifdef GDBM
    GDBM_FILE db;
    char      auth_file[BUFSIZ];
#else 
    DBM       *db;
    char      auth_dir[BUFSIZ];
#ifndef BSD44_DBM
    char      auth_pag[BUFSIZ];
#endif
#endif
    int     f, mode;

    srandom( (unsigned int) time((TIME_T *)0) );   /* seed random with the
                              current time */

    cp = program = argv[0];
    while( *program )  if( *(program++) == '/' ) cp = program;
    program = argv[0];

    mode = 0;
    while ( modes[mode].name &&
              strcmp(cp, modes[mode].name ) )  mode++;

 
    if ( mode >= OTHER ) 
      adios("Unknown mode: %s", cp);
    else if ( mode == POP_AUTH || mode == APOP_AUTH ) {
      passtype = 0;
      onlysw   = 1;   /* popauth always change ONLY the apop 
             password */
    }
    else if ( mode == SCRAM_AUTH ) {
      passtype = 1; 
      onlysw   = 0;    /* scramauth by default also clears the apop 
              password */
    }

    argv++;
    argc--;

    while (argc > 0) {
    cp = argv[0];
    if (*cp == '-') {
        int i, v;

        i = 0;
        v = -1;
        for(i = 0; switches[i].name; i++) {
        if(strcmp(&cp[1], switches[i].name) == 0) {
            v = i;
            break;
        }
        }
        cp++;
        switch (v) {
        default:
            adios ("-%s unknown option", cp);
        case INITSW:
            initsw = 1, listsw = 0; delesw = 0;
            break;
        case LISTSW:
            listsw = 1, initsw = 0; delesw = 0;
            break;
                case ONLYSW:
                    onlysw = 1;
            break;
        case DELESW:
            delesw = 1, initsw = 0; listsw = 0;
            if (argc < 2 || argv[1][0] == '-')
            adios ("missing argument to %s", argv[0]);
            usersw = argv[1];
            argc--;
            argv++;
            break;
        case USERSW:
            if (argc < 2 || argv[1][0] == '-')
            adios ("missing argument to %s", argv[0]);
            usersw = argv[1];
            argc--;
            argv++;
            if (delesw)
            fprintf(stderr, "Warning: user '%s' will now be deleted\n", usersw);
            break;
        }
    }
    else
        adios ("usage: %s [[-init]|[-list]|[-only]|[-user name]|[-delete name]]", program);
    argc--;
    argv++;
    }


/* Expand the rest iff SCRAM or APOP and favor SCRAM to hold auth db name */

#ifdef   SCRAM
#  define  AUTHON
#  define  AUTHDB SCRAM
#else
#  ifdef   APOP
#    define  AUTHON
#    define  AUTHDB APOP
#  endif /* APOP */
#endif /* SCRAM */

#ifndef  AUTHON
   adios ("not compiled with either SCRAM or APOP options");
#else
#ifndef SCRAM
   if ( mode == SCRAM_AUTH )
     adios ("not compiled with SCRAM option");
#endif
#ifndef APOP
   if ( (mode == APOP_AUTH) || (mode == POP_AUTH) )
     adios ("not compiled with APOP option");
#endif
    myuid = getuid();

    if ((pw = getpwnam (POPUID)) == NULL)
    adios ("\"%s\": user-id unknown", POPUID);

    if (pw->pw_uid == myuid)
    popuser = 1;

    if (myuid && !popuser && (delesw || initsw || listsw || (usersw != NULL)))
    adios("Only superuser or user '%s' can perform the requested function",
        POPUID);

    if (myuid && initsw)
    adios("Only superuser can init the authentication database");

#ifdef GDBM
    (void) strncpy(auth_file, AUTHDB, sizeof(auth_file) - 1);
    auth_file[sizeof(auth_file)-1] = '\0';
#else
    (void) strncpy(auth_dir, AUTHDB, sizeof(auth_dir) - 5);
#ifdef BSD44_DBM
    (void) strcat(auth_dir, ".db");
#else
    (void) strncpy(auth_pag, AUTHDB, sizeof(auth_pag) - 5);
    (void) strcat(auth_pag, ".pag");
    (void) strcat(auth_dir, ".dir");
#endif
#endif

    if (delesw) {
    if (myuid && !popuser)
        adios ("Only root or %s may delete entries", POPUID);

#ifdef GDBM
    if ((db = gdbm_open(auth_file,512,GDBM_WRITER,0,0)) == NULL)
#else
    if ((db = dbm_open (AUTHDB, O_RDWR, 0)) == NULL)
#endif
        adios ("%s: unable to open POP authentication DB: %s (%i)", 
                   AUTHDB, strerror(errno),  __LINE__);
    
    key.dsize = strlen (key.dptr = usersw) + 1;

#ifdef GDBM
    value = gdbm_fetch(db, key);
#else
    value = dbm_fetch(db, key);
#endif
    if (value.dptr == NULL)
        adios("User '%s' not found in authentication database", usersw);

#ifdef GDBM
    if (gdbm_delete(db,key) < 0)
#else
    if (dbm_delete(db, key) < 0)
#endif
        adios("Unable to delete user '%s' from authentication database", usersw);

#ifdef GDBM
    gdbm_close (db);
#else
    dbm_close (db);
#endif

    exit (0);
    }

    if (initsw) {
    struct stat st;

    setuid(myuid);

#ifdef GDBM
    if (stat (auth_file, &st) != -1) 
#else
    if (stat (auth_dir, &st) != -1) 
#endif
    {
        char ibuf[30];
        
        printf("Really initialize POP authentication DB? ");
        if(fgets(ibuf, sizeof(ibuf), stdin) == NULL || ibuf[0] != 'y')
        exit (1);
#ifdef GDBM
        (void) unlink (auth_file);
#else
        (void) unlink (auth_dir);
# ifndef BSD44_DBM
        (void) unlink (auth_pag);
# endif
#endif
    }
#ifdef GDBM
    if ((db = gdbm_open (auth_file, 512, GDBM_WRCREAT, 0600, 0)) == NULL)
        adios ("unable to create POP authentication DB %s: %s (%i)", 
            auth_file, strerror(errno), __LINE__);
    gdbm_close (db);
    if (chown (auth_file, pw->pw_uid, pw->pw_gid) == -1)
        adios ("error setting ownership of POP authentication for %s: %s", 
        auth_file, strerror(errno));
#else
    if ((db = dbm_open (AUTHDB, O_RDWR | O_CREAT, 0600)) == NULL)
        adios ("unable to create POP authentication DB: %s", 
            strerror(errno));
    dbm_close (db);
    if (chown (auth_dir, pw->pw_uid, pw->pw_gid) == -1 
#ifndef BSD44_DBM
     || chown (auth_pag, pw->pw_uid, pw->pw_gid) == -1
#endif
        )
        adios ("error setting ownership of POP authentication DB: %s", 
        strerror(errno));
#endif

    exit (0);
    }

#ifdef GDBM
    if ((db = gdbm_open (auth_file, 512, GDBM_READER, 0, 0)) == NULL)
                adios ("unable to open POP authentication DB %s:\n     %s (%i)", 
                       auth_file, strerror(errno), __LINE__);
#else
    if ((db = dbm_open (AUTHDB, O_RDONLY, 0)) == NULL)
        adios ("unable to open POP authentication DB %s:\n     %s (%i)", 
                       AUTHDB, strerror(errno), __LINE__);
#endif

#ifdef GDBM
    if ((f = open (auth_file, listsw ? O_RDONLY : O_RDWR)) == -1)
        adios ("%s: unable to open POP authentication DB:\n    %s (%i)", 
                        auth_file,  strerror(errno), __LINE__);
#else
    if ((f = open (auth_dir, listsw ? O_RDONLY : O_RDWR)) == -1)
        adios ("%s: unable to open POP authentication DB", auth_dir);
    if (flock (f, LOCK_SH) == -1)
        adios ("%s: unable to lock POP authentication DB", auth_dir);
#endif


    if (listsw) {
    if (usersw) {
        key.dsize = strlen (key.dptr = usersw) + 1;
#ifdef GDBM
        value = gdbm_fetch (db, key);
#else
        value = dbm_fetch (db, key);
#endif
        if (value.dptr == NULL)
        adios ("no such entry in POP authentication DB");
            (strncpy(buf, key.dptr, sizeof(buf)-2))[sizeof(buf)-2] = '\0';
            strcat( buf, ":" );
        printf ("%-16s", buf );
            if ( value.dptr[0] ) printf (" APOP");
            if ( value.dsize >= (i=strlen(value.dptr)+2)
                    && value.dptr[i] ) printf (" SCRAM");
        printf ("\n");
    }
    else
#ifdef GDBM
        for (key = gdbm_firstkey (db); key.dptr; key = gdbm_nextkey (db,key)) 
#else
        for (key = dbm_firstkey (db); key.dptr; key = dbm_nextkey (db)) 
#endif
        {
               (strncpy(buf, key.dptr, sizeof(buf)-2))[sizeof(buf)-2] = '\0';
                strcat( buf, ":" );
        printf ("%-16s", buf );
#ifdef GDBM
        value = gdbm_fetch (db, key);
#else
        value = dbm_fetch (db, key);
#endif
        if (value.dptr == NULL)
            printf(" *** no information?!?\n");
        else {

                    if ( value.dptr[0] ) printf (" APOP");
                    if ( value.dsize >= (i=strlen(value.dptr)+2)
                           && value.dptr[i] ) printf (" SCRAM");
            printf ("\n");
        }
        }

#ifdef GDBM
    gdbm_close (db);
#else
    dbm_close (db);
#endif

    exit (0);
    }

    if (usersw == NULL) {
    if ((pw = getpwuid(myuid)) == NULL)
        adios("Sorry, don't know who uid %d is.", myuid);
    usersw = pw->pw_name;
    } else {
    if ((pw = getpwnam(usersw)) == NULL)
        adios("Sorry, don't know who uid %s is.", usersw);
    usersw = pw->pw_name;
    }

    fprintf( stderr, "Changing %s%s %s for %s.\n",
         onlysw? "only " : "",  
        (mode==SCRAM_AUTH) ? "SCRAM" : "APOP",
         passtype ? "pass phrase" : "password",
         usersw );

    key.dsize = strlen (key.dptr = usersw) + 1;
#ifdef GDBM
    value = gdbm_fetch (db, key);
#else
    value = dbm_fetch (db, key);
#endif
    
    if ( value.dptr ) {
      strcpy( apopfld, value.dptr ); 
      if ( value.dsize > strlen(apopfld)+1 )
        strcpy( scramfld, &value.dptr[strlen(apopfld)+1] );
      else
        scramfld[0] = '\0';
    }
    else 
      apopfld[0] = scramfld[0] = '\0';

    apoplen  = strlen(  apopfld ) + 1;
    scramlen = strlen( scramfld ) + 1;

    if ( (myuid && !popuser) 
          && (((mode==APOP_AUTH || mode==POP_AUTH) && apopfld[0] )
                ||(mode==SCRAM_AUTH && scramfld[0] )) ) {

      if ( !(i=strlen(strncpy(obuf,
             getpass(passtype?"Old pass phrase:":"Old password:"),sizeof(obuf)))) )
    adios( "Sorry, password entered incorrectly\n" );

      if ( mode == APOP_AUTH || mode == POP_AUTH ) {
    if ( ((apoplen - 1) != i) ||
        (strncmp(obuf, apopfld, i) &&
             strncmp(obuf, obscure(apopfld), i)) ) 
    adios( "Sorry, password entered incorrectly\n" );
      }
      else if ( mode == SCRAM_AUTH ) {
#ifdef SCRAM
        SCRAM_MD5_VRFY  test_verifiers;
        
        scramlen = sizeof( scram_verifiers );

        if (    decode64(   scramfld,         strlen(scramfld),
                           &scram_verifiers, &scramlen )
             || scramlen != sizeof( scram_verifiers ) ) 
       adios("%s: unable to decode SCRAM verifier from the authenication DB" );

        scram_md5_vgen( &test_verifiers,
                              scram_verifiers.salt,
                              obuf,
                              0,
                              1,
                              NULL );

        if ( memcmp( &scram_verifiers, &test_verifiers, sizeof(scram_verifiers)) ) 
#endif
          adios("Sorry, pass phrase entered incorrectly\n");
        }
    }
#ifdef GDBM
    gdbm_close (db);
#else
     dbm_close (db);
    if (flock (f, LOCK_UN) == -1)
        adios ("%s: unable to unlock POP authentication DB", auth_dir);
#endif


#ifdef  lint
    flags = 0;
#endif  /* lint */
    for (insist = 0; insist < 2; insist++) {
    int i;
    char    c;

    if (insist)
        printf ("Please use %s.\n",
            flags == 1 ? "at least one non-numeric character"
            : (passtype?"a longer pass phrase":"a longer password"));

    if (((i = strlen(strncpy(buf,
                 getpass(passtype?"New pass phrase:":"New password:"), sizeof(buf)))) == 0) ||
          !strncmp(buf, obuf, i)) {
        fprintf (stderr, passtype?"Pass phrase unchanged.\n":"Password unchanged.\n");
        exit (1);
    }

        if ( !strcmp(buf,"*") )
          break;

    flags = 0;
    for ( (cp = buf); c = *cp++; )
        if (c >= 'a' && c <= 'z')
        flags |= 2;
        else
        if (c >= 'A' && c <= 'Z')
            flags |= 4;
        else
            if (c >= '0' && c <= '9')
            flags |= 1;
            else
            flags |= 8;

    if ((flags >= 7 && i >= 4)
        || ((flags == 2 || flags == 4) && i >= 6)
        || ((flags == 3 || flags == 5 || flags == 6) && i >= 5))
        break;
    }

    
    if (strcmp(buf,
          getpass(passtype?"Retype new pass phrase:":"Retype new password:"))) {
    fprintf (stderr,
     passtype?"Mismatch -- pass phrase unchanged.\n":"Mismatch -- password unchanged.\n");
    exit (1);
    }
  
    if ( !strcmp(buf,"*") ) buf[0]='\0';

#ifdef GDBM
    if ((db = gdbm_open(auth_file, 512, GDBM_WRITER, 0, 0)) == NULL)
#else
    if ((db = dbm_open(AUTHDB, O_RDWR, 0)) == NULL)
#endif
        adios("%s: unable to open POP authentication DB:\n     %s (%i)", 
                       AUTHDB,  strerror(errno), __LINE__);

#ifdef FLOCK_EX
#ifndef GDBM
    if (flock(f, LOCK_EX) == -1)
    adios("%s: unable to lock POP authentication DB\n    %s (%i)", 
              auth_dir, strerror(errno), __LINE__); 
#endif
#endif

    key.dsize = strlen (key.dptr = usersw) + 1;

    if ( mode == APOP_AUTH || mode == POP_AUTH ) {
      strcpy( apopfld, obscure(buf) );
    }

    else if ( mode == SCRAM_AUTH ) {
      scramlen = 0;

      if ( buf[0] ) {
#ifdef SCRAM
        UINT4          key[4];
        UINT4          data[4];
        unsigned char  salt[ SCRAM_MD5_SALTSIZE ];
        int            x;
        unsigned char  digest[ HMAC_MD5_SIZE ];
    
        key[ 0 ] = (UINT4)random();   /* seeded with start up time */
        key[ 1 ] = (UINT4)time((time_t *)0);/* include current time */
    
    /* seed random with the current time to nearest second    */
    srandom( (unsigned int) time((TIME_T *)0) );
    
        key[ 3 ] = (UINT4)random();       
        key[ 4 ] = (UINT4)random();       
    
        data[ 0 ] = (UINT4)getpid();      /* pid+uid+most of user name */
        data[ 1 ] = myuid;
        strncpy( (char *)&data[2], usersw, 2*sizeof(UINT4) );
    /* personal info as data  */
    /* random()& time() as key*/
        hmac_md5( (unsigned char *)data, sizeof( data ), 
                  (unsigned char *)key, sizeof( key ),   
                   digest );
    
        /* zero out the salt           */
        /* and then fold in the digest */
    
        memset(salt, '\0', SCRAM_MD5_SALTSIZE );
        x = HMAC_MD5_SIZE;
        while( x-- )
          salt[ x % SCRAM_MD5_SALTSIZE ] ^= digest[ x ];
  
        scram_md5_vgen( &scram_verifiers,
                         salt,
                         buf,
                         0,
                         1,
                         NULL );
  
        scramlen = sizeof( scramfld );
  
        if ( encode64( &scram_verifiers, sizeof(scram_verifiers),
                          scramfld, &scramlen ) || scramlen >= sizeof(scramfld) )
#endif
           adios("%s: unable to encode SCRAM verifier for the authenication DB" );
      }

      scramfld[ scramlen ] = '\0';

      if( !onlysw ) apopfld[0] = '\0';
    }

    value.dsize = strlen( value.dptr=strcpy(outfld, apopfld) ) + 1;
    value.dsize += strlen( strcpy(&outfld[value.dsize], scramfld) ) + 1;

#ifdef GDBM
    if (gdbm_store (db, key, value, GDBM_REPLACE))
    adios ("POP authentication DB may be corrupt?!?");
    gdbm_close (db);
#else
    if (dbm_store (db, key, value, DBM_REPLACE))
    adios ("POP authentication DB may be corrupt?!?");
    dbm_close (db);
#endif
#endif

    exit (0);
}

#ifndef HAVE_STRERROR
char *
strerror(e)
    int e;
{
    extern char *sys_errlist[];
    extern int sys_nerr;

    if(e < sys_nerr)
        return(sys_errlist[e]);
    else
        return("unknown error");
}
#endif
