/*
 * Copyright (c) 2000 QUALCOMM Incorporated. All rights reserved.
 * See License.txt file for terms and conditions for modification and
 * redistribution.
 *
 * Revisions:
 *
 * 05/17/00  [rg]
 *         - Tweak mail lock refresh algorithm to avoid performance hit.
 *
 * 04/21/00  [rg]
 *         - Always use our maillock(), even if system supplies one.
 *         - Rename our maillock() to Qmaillock().
 *
 * 03/09/00  [rg]
 *         - Fix incorrect use of save_errno / save_error.
 *
 * 02/16/00  [rg]
 *         - Now calling touchlock periodically.
 *
 * 12/06/99  [rg]
 *         - Added #include "misc.h" to make sure odds and ends are properly
 *           defined.
 *         - Fixed typo (SYS_MAILOCK instead of SYS_MAILLOCK).
 *
 * 11/22/99  [rg]
 *         - fixed HAVE_MAILLOCK to be HAVE_MAILLOCK_H
 *         - Changed MAILLOCK to be SYS_MAILLOCK
 *         - Removed mailunlock macro; added #include for our own maillock.h,
 *         - Made use of maillock() unconditional.
 *
 */


/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */


#include <sys/types.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>

#include "config.h"

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#include <flock.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

#include <sys/stat.h>
#include <sys/file.h>

#include "popper.h"
#include "maillock.h"

#ifdef KEEP_TEMP_DROP
#  define UNLINK_TEMP_DROP(a)
#else
            /* Added code in pop_dropcopy.c makes unlink ok now. */
            /* s-dorner@uiuc.edu, 12/91 */
#  define UNLINK_TEMP_DROP(a) (void)unlink(a)
#endif

#if !defined(L_XTND)
#  define L_XTND SEEK_END
#endif

#ifndef L_SET
#  define L_SET SEEK_SET
#endif

#if defined(HAVE_UNISTD_H)
#  include <unistd.h>
#endif

#include "misc.h"

extern int      errno;

#define         BBSIZE          4096

static char standard_error[] =
    "Error updating primary drop. Mail left in temporary maildrop (%d)";

/* 
 *  updt:   Apply changes to a user's POP maildrop
 */

int pop_updt (p)
POP     *   p;
{
    FILE                *   md = NULL;              /*  Stream pointer for 
                                                        the user's maildrop */
    int                     mfd;                    /*  File descriptor for
                                                        above */
    char                    buffer[BBSIZE];         /*  Read buffer */

    char                    uidl_buf[128];          /*  UIDL header */
    MsgInfoList         *   mp = NULL;              /*  Pointer to message 
                                                        info list */
    register int            msg_num;                /*  Current message 
                                                        counter */
    register int            status_written;         /*  Status header field 
                                                        written */
    int                     nchar;                  /* Bytes read/written */

    long                    offset = 0;             /* New mail offset */

    char                *   result = NULL;          /* fget and fputs status */

    int                     save_errno = 0;         /* Save the error value we
                                                       are trying to print. */
    int                     rslt = 0;               /* temp use when calling */
    struct stat             mybuf;                  /*  For fstat() */
    time_t                  time_locked = 0;        /* time maillock got or touched */


#ifdef AUTO_DELETE
#  ifdef NOUPDATEONABORT
    /* 
       AUTO_DELETE causes messages which have been RETRd by the client to be
       unconditionally and automatically marked deleted.  This prevents clients
       from leaving mail on the server.  

       CAUTION: Be sure users are informed of this before enabling it, to 
       prevent lost mail.

       Note: NOUPDATEONABORT must also be defined; otherwise users may lose mail
       if they try and RETR it but the connection closes for any reason, the 
       client aborts, etc.
    */
    DEBUG_LOG0 ( p, "Auto-deleting messages which have been RETRd..." );
    for ( msg_num = 0; msg_num < p->msg_count; ++msg_num )
    {
      mp = &p->mlp [ msg_num ];
      if ( mp->retr_flag )
      {
        p->msgs_deleted++;
        mp->del_flag = 1;
        DEBUG_LOG1 ( p, ">>> msg %i automatically flagged as DELETED <<<", 
                     msg_num );
      }

    }
#  endif /* NOUPDATEONABORT */
#endif /* AUTO_DELETE */

    DEBUG_LOG0 ( p, "Performing maildrop update..." );
    DEBUG_LOG0 ( p, "Checking to see if all messages were deleted" );

    if ( p->stats ) {
        pop_log ( p, POP_PRIORITY, HERE, "Stats: %s %d %d %d %d %s %s",
                  p->user, p->msgs_deleted, p->bytes_deleted,
                  p->msg_count - p->msgs_deleted,
                  p->drop_size - p->bytes_deleted,
                  p->client, p->ipaddr );
    }

    if ( p->server_mode ) {
        if ( !p->dirty ) {
            UNLINK_TEMP_DROP ( p->temp_drop );
            DEBUG_LOG0 ( p, "Running in server mode; spool not "
                            "changed; deleting temp drop" );
            return ( POP_SUCCESS );
        }
    }
    if ( !p->server_mode && p->msgs_deleted == p->msg_count ) {
        /*
         * In a !server_mode the server process would be the sole user of
         * .user.pop so truncate and unlink is not a problem.
         *
         * Truncate before close, to avoid race condition.  
         */
        ftruncate ( fileno ( p->drop ), (OFF_T)0 );
        UNLINK_TEMP_DROP ( p->temp_drop );
        DEBUG_LOG0 ( p, "Non-server mode and all msgs deleted; "
                        "deleting temp drop" );
        fclose ( p->drop );
        return ( POP_SUCCESS );
    }

    /* 
     * Always use Qmaillock()
     */

    rslt = Qmaillock ( p->user, 1, p->trace, p->debug );
    if ( rslt != 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE, "maillock error %d on '%s': %s (%d)", 
                           rslt, p->temp_drop, STRERROR(errno), errno ) );
    time_locked = time(0);
    
    if ( !p->server_mode ) {
        DEBUG_LOG1 ( p, "(Normal mode) opening spool \"%s\"", 
                     p->drop_name );
        /*  
         * Open the user's real maildrop 
         */
        if ( ( mfd = open   ( p->drop_name, O_RDWR | O_CREAT, 0660 ) ) == -1 ||
             ( md  = fdopen ( mfd, "r+" ) ) == NULL ) {
            Qmailunlock();
            return pop_msg ( p, POP_FAILURE, HERE, standard_error, errno );
        }
    } 
    else {
        DEBUG_LOG0 ( p, "(Server mode) accessing spool" );
        mfd = fileno ( p->drop );
    }

    /*  
     * Lock the user's real mail drop 
     */
    if ( flock ( mfd, LOCK_EX ) == -1 ) {
        fclose ( (p->server_mode) ? p->drop : md );
        Qmailunlock();
        return pop_msg ( p, POP_FAILURE, HERE, "flock: '%s': %s (%d)", 
                         p->drop_name,
                         STRERROR(errno), errno );
    }

    if ( !p->server_mode )  {
        /* 
         * Go to the right places 
         */
        fseek ( p->drop, 0, SEEK_END ); 
        offset = ftell ( p->drop ); 
        DEBUG_LOG1 ( p, "Drop has %lu octets", offset );
        
        Qtouchlock();

        /*  
         * Append any messages that may have arrived during the session 
         * to the temporary maildrop 
         */
        while ( ( nchar = read ( mfd, buffer, BBSIZE ) ) > 0 ) {
            if ( nchar != write ( fileno ( p->drop ), buffer, nchar ) ) {
                nchar = -1; /* write failed */
                break ;
            }
        }
        if ( nchar ) {
            save_errno = errno;
            fclose ( md );
            Qmailunlock();
            ftruncate ( fileno ( p->drop ), (OFF_T)offset );
            fclose ( p->drop );
            return pop_msg ( p, POP_FAILURE, HERE, 
                             "Error copying new messages: '%s' : %s (%d)",
                             p->drop_name, STRERROR(save_errno), save_errno );
        }

        fflush ( md );
        rewind ( md );
        ftruncate ( mfd, (OFF_T)0 );
        lseek ( mfd, (OFF_T)0, L_SET );
        DEBUG_LOG0 ( p, "Non-server mode; copied any new msgs to temp "
                        "drop; truncated spool" );

        /* 
         * Synch stdio and the kernel for the POP drop 
         */
        rewind ( p->drop );
        lseek  ( fileno ( p->drop ), (OFF_T)0, L_SET );

        /*  Transfer messages not flagged for deletion from the temporary 
         *  maildrop to the new maildrop 
         */
        DEBUG_LOG2 ( p, "Creating new maildrop \"%s\" from \"%s\"",
                     p->drop_name, p->temp_drop );

    } /* not server mode */
    else { /* SERVER_MODE */
        fstat ( fileno ( p->drop ), &mybuf );
        if ( p->msgs_deleted == p->msg_count &&
             mybuf.st_size   == p->spool_end ) {
            /* 
             * Truncate before close, to avoid race condition.  
             */
            ftruncate ( fileno ( p->drop ), (OFF_T)0 );
            UNLINK_TEMP_DROP ( p->temp_drop );
            flock  ( mfd, LOCK_UN );
            fclose ( p->drop );
            fclose ( p->hold );
            Qmailunlock();
            DEBUG_LOG0 ( p, "Server mode; all msgs deleted and spool size "
                            "unchanged; truncating spool" );
            return ( POP_SUCCESS );
        }
        md  = p->hold;   /* Really the temp drop */
        mfd = fileno ( md );
    }

    if ( !p->server_mode || ( p->msgs_deleted != p->msg_count ) ) {
        DEBUG_LOG3 ( p, "Server mode=%i; %i out of %i msgs deleted; copying "
                        "msgs from temp drop to spool", 
                     p->server_mode, p->msgs_deleted, p->msg_count );

        for ( msg_num = 0; msg_num < p->msg_count; ++msg_num ) {

            int inheader   = 1;
            int body_lines = 0;

            /*  
             * Get a pointer to the message information list 
             */
            mp = &p->mlp [ msg_num ];
            
            /*
             * It is very important to refresh the mail lock on an
             * interval between one and three minutes.  Let's guess
             * and use ten messages.
             */
            if ( msg_num % 500 == 0 &&
                 time(0) - time_locked > 60 ) {
                Qtouchlock();
                time_locked = time(0);
            }

            if ( mp->del_flag ) {
                DEBUG_LOG1 ( p, "Message %d flagged for deletion.", mp->number );
                continue;
            }

            fseek ( p->drop, mp->offset, SEEK_SET );

            DEBUG_LOG1 ( p, "Copying message %d.", mp->number );

            /* 
             * Put the From line separator 
             */
            fgets ( buffer, MAXMSGLINELEN, p->drop );
            if ( fputs ( buffer, md ) == EOF )
                break;

            sprintf ( buffer, "%s %s", "X-UIDL:", mp->uidl_str );
            if ( fputs ( buffer, md ) == EOF )
                break;

            for ( status_written = 0, inheader = 1;
                  fgets ( buffer, MAXMSGLINELEN, p->drop );
                ) {

                if ( inheader ) { /* Header */

                    /*  
                     * A blank line signals the end of the header. 
                     */
                    if ( *buffer == '\n' ) {
#ifndef NO_STATUS
                        if ( status_written == 0 ) {
                            if ( mp->retr_flag ) {
                                sprintf ( buffer, "Status: RO\n\n" );
                            } else {
                                sprintf ( buffer, "Status: U\n\n" );
                            }
                        }
#endif

                        inheader       = 0;
                        body_lines     = 1;
                        status_written = 0;

                    } else if ( !strncasecmp ( buffer, "X-UIDL:", 7 ) ) {
                        continue;       /* Skip all existing UIDL lines */

                    } else if ( !strncasecmp ( buffer, "Status:", 7 ) ) {

                        /*  
                         * Update the message status 
                         */
                        if ( mp->retr_flag )
                            sprintf ( buffer, "Status: RO\n" );
                        status_written++;
                    }
                    /*  
                     * Save another header line 
                     */
                    if ( fputs ( buffer, md ) == EOF )
                        break;

                    } else { /* Body */ 
                        if ( ++body_lines > mp->body_lines )
                            break;
                        if ( fputs ( buffer, md ) == EOF )
                            break;
                    }
            }

            if ( ferror ( md ) ) {
                break;
            }

            if ( p->mmdf_separator ) {
                fputs ( p->mmdf_separator, md );
            }
        }

    /* flush and check for errors now!  The new mail will be writen
     * without stdio,  since we need not separate messages 
     */

        if ( ferror ( md ) ) {
            save_errno = errno;

            ftruncate ( mfd, (OFF_T)0 );
            fclose ( md );
            Qmailunlock();
            fclose ( p->drop );
#ifdef EDQUOT
            if ( save_errno == EDQUOT )
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Overquota copying messages to Mailspool."
                                 " Temp drop unchanged (%d)",
                                 save_errno );
            else
#endif
                return pop_msg ( p, POP_FAILURE, HERE, 
                                 standard_error, save_errno );
        }

        fflush ( md );
        if ( ferror ( md ) ) {
            int save_error = errno;

            ftruncate ( mfd, (OFF_T)0 );
            close ( mfd );
            Qmailunlock();
            fclose ( p->drop );
#ifdef EDQUOT
            if ( save_error == EDQUOT )
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Overquota copying messages to Mailspool. "
                                 "Temp drop unchanged (%d)",
                                 save_error );
            else
#endif
                return pop_msg ( p, POP_FAILURE, HERE, 
                                 standard_error, save_error );
        }

    } /* p->msgs_deleted != p->msg_count */

    if ( p->server_mode ) {
        DEBUG_LOG0 ( p, "(server mode)" );
        if ( mybuf.st_size > p->spool_end ) {
            /* 
             * Go to the right places 
             */
            lseek ( fileno ( p->drop ), (OFF_T)p->spool_end, L_SET ); 
            
            DEBUG_LOG1 ( p, "%lu octets were added to spool during session",
                         mybuf.st_size - p->spool_end );

            Qtouchlock();
            
            /*  
             * Append any messages that may have arrived during the session 
             * to the temporary maildrop 
             */
            while ( ( nchar = read ( fileno ( p->drop ), buffer, BBSIZE ) ) > 0 )
                if ( nchar != write ( mfd, buffer, nchar ) ) {
                    nchar = -1;
                    break ;
                }

            if ( nchar != 0 ) {
                Qmailunlock();
                ftruncate ( mfd, (OFF_T)0 );
                close ( mfd );
#ifdef EDQUOT
                if ( errno == EDQUOT ) {
                    pop_msg ( p, POP_FAILURE, HERE,
                              "Overquota: appending messages from mailspool "
                              "to temporary drop (%d)",
                              errno );
                } else
#endif
                    pop_msg ( p, POP_FAILURE, HERE,
                              "Error appending messages from mailspool to "
                              "temporary drop: %s (%d)",
                               STRERROR(errno), errno );
            }
        } /* mybuf.st_size > p->spool_end */

        rewind ( p->drop );
        ftruncate ( fileno(p->drop ), (OFF_T)0 );
        lseek ( fileno ( p->drop ), (OFF_T)0, L_SET );

        lseek ( mfd, (OFF_T)0, L_SET );

        Qtouchlock ();
        
        while ( ( nchar = read ( mfd, buffer, BBSIZE ) ) > 0 )
            if ( nchar != write ( fileno ( p->drop ), buffer, nchar ) ) {
                nchar = -1;
                break ;
            }

        if ( nchar != 0 ) {
            ftruncate ( fileno(p->drop), (OFF_T)0 );
            fclose ( p->drop );
            Qmailunlock();
            fclose ( md );
#ifdef EDQUOT
            if ( errno == EDQUOT ) {
                pop_msg ( p, POP_FAILURE, HERE,
                          "Overquota: copying messages back to mailspool (%d)", 
                          errno );
            } else
#endif
                pop_msg ( p, POP_FAILURE, HERE,
                          "Error appending messages from temporary drop "
                          "to mailspool: %s (%d)",
                          STRERROR(errno), errno );
        }

        fclose ( md );
        Qmailunlock();
        ftruncate ( mfd, (OFF_T)0 );
        UNLINK_TEMP_DROP ( p->temp_drop );
        fclose ( p->drop );
    } /* server mode */
    else {
        /* 
         * Go to start of new mail if any 
         */
        lseek ( fileno(p->drop), (OFF_T)offset, L_SET );

        Qtouchlock ();
        
        /* 
         * Copy over any new mail that arrived while processing the pop drop 
         */
        while ( (nchar = read ( fileno(p->drop), buffer, BBSIZE ) ) > 0 )
            if ( nchar != write ( mfd, buffer, nchar ) ) {
                nchar = -1;
                break ;
            }
        if ( nchar != 0 ) {
            int save_error = errno;

            ftruncate ( mfd, (OFF_T)0 );
            fclose ( md );
            Qmailunlock();
            fclose ( p->drop ) ;
#ifdef EDQUOT
            if ( save_error == EDQUOT )
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Overquota copying messages to Mailspool. "
                                 "Temp drop unchanged (%d)",
                                 save_error );
            else
#endif
                return pop_msg ( p, POP_FAILURE, HERE, 
                                 standard_error, save_error );
        }

        /*  
         * Close the maildrop and empty temporary maildrop 
         */
        fclose ( md );
        Qmailunlock();
        ftruncate ( fileno(p->drop), (OFF_T)0 );
        UNLINK_TEMP_DROP ( p->temp_drop );
        fclose ( p->drop );
    }
    return ( pop_quit ( p ) );
}



int pop_restore ( p )
POP *p;
{
  FILE    *md;
  int     mfd;
  char    buffer[BBSIZE];
  int     nchar;
  long    offset;
  int     save_errno;
  int     rslt;

    DEBUG_LOG0 ( p, "Performing maildrop restoration..." ) ;

    if ( p->server_mode )
    {
        UNLINK_TEMP_DROP ( p->temp_drop );
        return ( POP_SUCCESS );
    }

    DEBUG_LOG1 ( p, "Opening mail drop \"%s\"", p->drop_name );

    rslt = Qmaillock ( p->user, 1, p->trace, p->debug );
    if ( rslt != 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE, "maillock error %d on '%s': %s (%d)", 
                           rslt, p->drop_name, STRERROR(errno), errno ) );

    if ( ( mfd = open   ( p->drop_name, O_RDWR | O_CREAT, 0660 ) ) == -1 ||
         ( md  = fdopen ( mfd, "r+" ) ) == NULL ) {
        Qmailunlock();
        return pop_msg ( p, POP_FAILURE, HERE, standard_error, errno );
    }

    if ( flock ( mfd, LOCK_EX ) == -1 ) {
        fclose ( md );
        Qmailunlock();
        return pop_msg ( p, POP_FAILURE, HERE, "flock: '%s': %s (%d)", 
                         p->drop_name, STRERROR(errno), errno );
    }

    fseek ( p->drop, 0, SEEK_END );
    offset = ftell ( p->drop );

    /* Append any messages that may have arrived during the session 
     * to the temporary mail drop 
     */
    while ( ( nchar = read ( mfd, buffer, BBSIZE ) ) > 0 )
        if ( nchar != write ( fileno(p->drop), buffer, nchar ) ) {
            nchar = -1;
            break;
        }
        if ( nchar != 0) {
            save_errno = errno;
            fclose ( md );
            Qmailunlock();
            ftruncate ( fileno(p->drop), (OFF_T)offset );
            fclose ( p->drop );
#ifdef EDQUOT
            if ( save_errno == EDQUOT ) {
                pop_msg ( p, POP_FAILURE, HERE, 
                          "Overquota: appending messages to temporary drop (%d)",
                          save_errno );
            }
            else
#endif
            pop_msg ( p, POP_FAILURE, HERE,
                      "Error appending messages from mailspool "
                      "to temporary drop: %s (%d)",
                      STRERROR(save_errno), save_errno );
       }

       fflush ( md );
       rewind ( md );
       ftruncate ( mfd, (OFF_T)0 );
       lseek ( mfd, (OFF_T)0, L_SET );

       rewind ( p->drop );
       lseek ( fileno(p->drop), (OFF_T)0, L_SET );
       
       while ( ( nchar = read ( fileno(p->drop), buffer, BBSIZE ) ) > 0 )
            if ( nchar != write ( mfd, buffer, nchar ) ) {
                nchar = -1;
                break;
            }
       if ( nchar != 0 ) {
           save_errno = errno;
           ftruncate ( mfd, (OFF_T)0 );
           fclose ( md );
           Qmailunlock();
           fclose ( p->drop );
#ifdef EDQUOT
           if ( save_errno == EDQUOT )
               return pop_msg ( p, POP_FAILURE, HERE,
                                "Overquota copying messages to Mailspool. "
                                "Temp drop unchanged (%d)",
                                save_errno );
           else
#endif
           return pop_msg ( p, POP_FAILURE, HERE, standard_error, errno );
       }

       fclose ( md );
       Qmailunlock();
       ftruncate ( fileno(p->drop), (OFF_T)0 );
       UNLINK_TEMP_DROP ( p->temp_drop );
       fclose ( p->drop );

       return pop_quit ( p );
}








