
/*
 * Copyright (c) 2000 QUALCOMM Incorporated. All rights reserved.
 * See License.txt file for terms and conditions for modification and
 * redistribution.
 *
 * Revisions: 
 *
 * 05/17/00  [rg]
 *         - Tweak mail lock refresh algorithm to avoid performance hit.
 *
 * 04/28/00  [rg]
 *         - Avoid being fooled by a spoofed 'From ' line that follows
 *           a line which overflowed our input buffer.  [Based on patch
 *           by '3APA3A'].
 *         - Fixed some cases in which an empty .user.pop file could be
 *           left behind.
 *
 * 04/21/00  [rg]
 *         - Always use our maillock(), even if system supplies one.
 *         - Rename our maillock() to Qmaillock().
 *
 * 03/14/00  [rg]
 *         - Minor syntactical tweaks.
 *
 * 02/16/00  [rg]
 *         - Now calling touchlock periodically.
 *         - Added "or check for corrupted mail drop" to infamous
 *           'change recognition mode' message.
 *         - Check for and hide UW Folder Internal Data message if
 *           CHECK_UW_KLUDGE defined.
 *
 * 02/04/00  [rg]
 *         - Minor changes to try and appease HP C compiler.
 *
 * 02/02/00  [rg]
 *         - Return extended error response [SYS/TEMP] or [SYS/PERM]
 *           for various system errors.
 *         - Improved error message when maillock fails.
 *         - No longer reporting "Unable to process From lines" when
 *           running in normal (not server) mode and previous session 
 *           aborted due to quota exceeded or disk full.
 *         - If we are unable to copy spool to temp drop in normal (not
 *           server) mode due to disk full or over quota, and temp drop
 *           was empty or did not previously exist, we now properly zero
 *           temp drop instead of setting it to a random size.
 *
 * 01/26/00  [rg]
 *         - Fixed typo in UIDL processing.
 *
 * 01/25/00  [rg]
 *         - Added more trace code to UID processing.
 *
 * 01/07/00  [rg]
 *         - Added STRERROR
 *         - Create hashed spool directory if it doesn't exist
 *         - Check for old .pop file in old location first.
 *
 * 12/30/99  [rg]
 *         - Added retry loop if bulletin db in use.
 *
 * 12/07/99  [rg]
 *         - Added misc.h to be sure odd things that are missing on some
 *           systems are properly defined.
 *         - Fixed typo (SYS_MAILOCK instead of SYS_MAILLOCK)./
 *
 * 11/24/99  [rg]
 *         - Moved genpath() to its own file: genpath.c (and genpath.h).
 *         - Fixed HAVE_MAILLOCK to be HAVE_MAILLOCK_H
 *         - Changed MAILLOCK to be SYS_MAILLOCK
 *         - Removed MAILUNLOCK macro; added #include for our own maillock.h.
 *         - Made use of maillock() unconditional.
 *
 *  6/22/98  [py]
 *         - Added kludge for NULLs in message content (suggested by 
 *           Mordechai T.Abzug).
 *           Uidl to contain From Envelope in non-mmdf, and length check.
 *
 *  3/18/98  [py]
 *         - Reviewed the code, added DEBUG macros, corrected the 
 *           resources that are not closed.
 */

/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */


/*
 * When adding each line length into the size of the message and the maildrop,
 * increase the character count by one for the <cr> added when sending the
 * message to the mail client.  All lines sent to the client are terminated
 * with a <cr><lf>.
 */


#include <sys/types.h>
#include <errno.h>
#include <sys/errno.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <flock.h>
#include <time.h>

#include "config.h"

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif /* HAVE_UNISTD_H */

#if HAVE_SYS_UNISTD_H
#  include <sys/unistd.h>
#endif /* HAVE_SYS_UNISTD_H */

#ifndef HAVE_INDEX
#  define   index(s,c)      strchr(s,c)
#  define   rindex(s,c)     strrchr(s,c)
#endif /* HAVE_INDEX */

#if HAVE_STRINGS_H
# include <strings.h>
#endif

#include <sys/stat.h>
#include <sys/file.h>
#include <pwd.h>

#include "genpath.h"
#include "popper.h"
#include "misc.h"
#include "maillock.h"

#ifdef CHECK_UW_KLUDGE
#  define UW_KLUDGE 1
#else
#  define UW_KLUDGE 0
#endif /* CHECK_UW_KLUDGE */


#define MIN_UIDL_LENGTH 5     /* There is no particular reason to have this */
#define MAX_UIDL_LENGTH 70

/* This macro comes from Mark Crispin's c-client code */

/* Validate line
 * Accepts: pointer to candidate string to validate as a From header
 *      return pointer to end of date/time field
 *      return pointer to offset from t of time (hours of ``mmm dd hh:mm'')
 *      return pointer to offset from t of time zone (if non-zero)
 * Returns: t,ti,zn set if valid From string, else ti is NIL
 */

#define VALID(s,x,ti,zn) {                      \
  ti = 0;                               \
  if ((*s == 'F') && (s[1] == 'r') && (s[2] == 'o') && (s[3] == 'm') && \
      (s[4] == ' ')) {                          \
    for (x = s + 5; *x && *x != '\n'; x++);             \
    if (x) {                                \
      if (x - s >= 41) {                        \
    for (zn = -1; x[zn] != ' '; zn--);              \
    if ((x[zn-1] == 'm') && (x[zn-2] == 'o') && (x[zn-3] == 'r') && \
        (x[zn-4] == 'f') && (x[zn-5] == ' ') && (x[zn-6] == 'e') && \
        (x[zn-7] == 't') && (x[zn-8] == 'o') && (x[zn-9] == 'm') && \
        (x[zn-10] == 'e') && (x[zn-11] == 'r') && (x[zn-12] == ' '))\
      x += zn - 12;                         \
      }                                 \
      if (x - s >= 27) {                        \
    if (x[-5] == ' ') {                     \
      if (x[-8] == ':') zn = 0,ti = -5;             \
      else if (x[-9] == ' ') ti = zn = -9;              \
      else if ((x[-11] == ' ') && ((x[-10]=='+') || (x[-10]=='-'))) \
        ti = zn = -11;                      \
    }                               \
    else if (x[-4] == ' ') {                    \
      if (x[-9] == ' ') zn = -4,ti = -9;                \
    }                               \
    else if (x[-6] == ' ') {                    \
      if ((x[-11] == ' ') && ((x[-5] == '+') || (x[-5] == '-')))    \
        zn = -6,ti = -11;                       \
    }                               \
    if (ti && !((x[ti - 3] == ':') &&               \
            (x[ti -= ((x[ti - 6] == ':') ? 9 : 6)] == ' ') &&   \
            (x[ti - 3] == ' ') && (x[ti - 7] == ' ') &&     \
            (x[ti - 11] == ' '))) ti = 0;           \
      }                                 \
    }                                   \
  }                                 \
}

/* You are not expected to understand this macro, but read the next page if
 * you are not faint of heart.
 *
 * Known formats to the VALID macro are:
 *      From user Wed Dec  2 05:53 1992
 * BSD      From user Wed Dec  2 05:53:22 1992
 * SysV     From user Wed Dec  2 05:53 PST 1992
 * rn       From user Wed Dec  2 05:53:22 PST 1992
 *      From user Wed Dec  2 05:53 -0700 1992
 *      From user Wed Dec  2 05:53:22 -0700 1992
 *      From user Wed Dec  2 05:53 1992 PST
 *      From user Wed Dec  2 05:53:22 1992 PST
 *      From user Wed Dec  2 05:53 1992 -0700
 * Solaris  From user Wed Dec  2 05:53:22 1992 -0700
 *
 * Plus all of the above with `` remote from xxx'' after it. Thank you very
 * much, smail and Solaris, for making my life considerably more complicated.
 */


int newline  = 1;   /* set to 1 when preceding line was only a newline */
int isOflow  = 0;   /* set to 1 when current line exceeds input buffer */
int wasOflow = 0;   /* set to 1 when previous line exceeded input buffer */

/*
 *  0 for not a from line
 *  1 for is a from line
 */

/* Valid UUCP From lines:
 *
 *  From Address [tty] date
 *  From [tty] date
 *
 *  "From [tty] date" is probably valid but I'm too lazy at this
 *  time to add the code.
 *
 */
int
isfromline ( cp )
char        *cp;
{
    int ti, zn;
    char *x;

    /* If we overflow out input buffer, we can't trust an immediately
       following "\nFrom " line as a valid message separator. */
    if ( isOflow ) {
        isOflow  = 0;
        wasOflow = 1;
        return ( 0 );
    }
    if ( wasOflow ) {
        wasOflow = 0;
        return ( 0 );
    }

    /* If the previous line was not a newline then just return */
    /* 'From ' message separators are preceeded by a newline */ 
    if ( *cp == '\n' ) {
        newline = 1;
        return ( 0 );
    } else if ( !newline ) {
        return ( 0 );
    } else
        newline = 0;

    VALID ( cp, x, ti, zn );
    return ( ti != 0 );
}


/*
 * Base 80 encoding of 16byte MD5 hash. This is about the smallest encoding
 * possible for the MD5 hash using the chars allowed for UIDLs. 
 * (93 ASCII values are available and 19^93 < 2^128 < 20^93)
 * This implementation isn't exact about MSBs and LSB, but it is unique..
 */

char *
base80 ( char *output, unsigned char *digest )
{
  unsigned long v, n, j;
  int i, k;

  for ( i = 0; i < 16; ) {
    v = 0L;
    do {
      v = (v << 4) + digest[i++];
    } while ( i%4 );
    for ( k = 0; k < 5; k++ ) {
      j = v % 80;
      v = (v - j)/80;
      *output = 0x21 + j;
      /* 
       * Avoid . altgother to avoid having to stuff it 
       */
      if ( *output == '.' )
        *output = '~';
      output++;
    }
  }
  *output = '\0';
  return ( output );
}


/* 
 * Build the Msg_info_list from the temp_drop in case the previous session
 * failed to truncate. init_dropinfo and do_drop_copy differ in that
 * do_drop_copy copies all the messages from the source file to the 
 * destination file while it generates the Msg_info_list.
 */
int 
init_dropinfo ( POP *p, char *fname, FILE *fdesc, time_t time_locked )
{
    MsgInfoList    *mp;                     /* Pointer to message info list */
    int             msg_num;                /* Current message number */
    int             visible_msg_num;        /* Current visible message number */
    int             nchar;
    int             inheader;
    int             uidl_found;
    int             expecting_trailer;
    int             content_length, content_nchar, cont_len;
    char            buffer[MAXLINELEN];     /* Read buffer */
    char            frombuf[MAXLINELEN];    /* From Envelope of each
                                             * message is saved, for
                                             * adding it to UIDL
                                             * computation */
    MD5_CTX         mdContext;
    unsigned char   digest[16];
    long            unique_num = time(0);
    int             uw_hint    = 0;

    DEBUG_LOG1 ( p, "DROPINFO Checking file %s", fname );


#ifdef NULLKLUDGE
    {
    /* 
     * Eat NULLs at the beginning of the mailspool 
     */
    char tempchar;
    while ( ( tempchar = getc ( fdesc ) ) == 0 );
    ungetc ( tempchar, fdesc );
    }
#endif

    if ( mfgets ( buffer, MAXMSGLINELEN, fdesc ) == NULL ) {
        DEBUG_LOG1 ( p, "Drop file empty intially : %s", fname );
        return ( POP_SUCCESS );    
    }

    /*
     * Is this an MMDF file? 
     */
    if ( *buffer == MMDF_SEP_CHAR ) {
        p->mmdf_separator = (char *)strdup(buffer);
    } else if ( !isfromline(buffer) ) {
        DEBUG_LOG5 ( p, "newline=%d; isOflow=%d; wasOflow=%d; buffer(%u):  %.256s", 
                     newline, isOflow, wasOflow, strlen(buffer), buffer );
        return pop_msg ( p, POP_FAILURE, HERE,
                         "[SYS/PERM] Unable to process From lines "
                         "(envelopes) in %s; change recognition "
                         "mode or check for corrupted mail drop.",
                         fname );
    }

    newline = 1;
    rewind ( fdesc );

    inheader          = 0;
    msg_num           = 0;
    visible_msg_num   = 0;
    uidl_found        = 0;
    expecting_trailer = 0;
    content_length    = 0;
    content_nchar     = 0;
    cont_len          = 0;
    p->msg_count      = ALLOC_MSGS;


#ifdef NULLKLUDGE
    {
    /* 
     * Eat NULLs at the beginning of the mailspool 
     */
    char tempchar;
    while ( ( tempchar = getc ( fdesc ) ) == 0 );
    ungetc ( tempchar, fdesc );
    }
#endif

    DEBUG_LOG1 ( p, "Building the Message Information list from %s", fname );

    for ( mp = p->mlp - 1; mfgets ( buffer, MAXMSGLINELEN, fdesc ); ) {
        nchar = strlen(buffer);
        
        /*
         * It is very important to refresh the mail lock on an
         * interval between one and three minutes.  
         */
        if ( msg_num % 500 == 0 &&
             time(0) - time_locked > 60 ) {
            Qtouchlock();
            time_locked = time(0);
        }

        if ( ( content_nchar >= content_length ) &&
             ( p->mmdf_separator ? !strcmp(p->mmdf_separator, buffer) :
                                   isfromline(buffer) ) ) {
            /* 
             * ---- A message boundary ----- 
             *
             * ---- Tidy up previous message (if there was one) ---- 
             */

            if ( expecting_trailer ) {
                /* 
                 * Skip over the MMDF trailer 
                 */
                expecting_trailer = 0;
                continue;
            }

            if ( !p->mmdf_separator ) {
                strcpy(frombuf, buffer); /* Preserve From envelope */
            }
            
            /* 
             * ------ Get ready for the next message ----- 
             */
            MD5Init   ( &mdContext );
            MD5Update ( &mdContext, (unsigned char *)buffer,
                        strlen(buffer) );
            /* 
             * Include a unique number in case everything else is not 
             */
            unique_num++;
            MD5Update ( &mdContext, (unsigned char *)&unique_num, 
                        sizeof(long) );

            if ( !inheader ) {
                if ( ++msg_num > p->msg_count ) {
                    p->mlp = (MsgInfoList *) realloc ( p->mlp,
                                                       (p->msg_count += ALLOC_MSGS) 
                                                       * sizeof(MsgInfoList) );
                    if ( p->mlp == NULL ) {
                        p->msg_count = 0;
                        return pop_msg ( p, POP_FAILURE, HERE,
                                         "[SYS/TEMP] Can't build message list "
                                         "for '%s': Out of memory",
                                         p->user );
                    }
                    mp = p->mlp + msg_num - 2;
                }

                if ( msg_num != 1 ) {
                    DEBUG_LOG5 ( p, "Msg %d uidl '%s' at offset %d is %d "
                                    "octets long and has %u lines.",
                                 mp->number, mp->uidl_str, mp->offset, 
                                 mp->length, mp->lines );
                }
                else
                    DEBUG_LOG0 ( p, "Found top of first message" );
                ++mp;

            } 
            else { /* still in header */
                pop_log ( p, POP_PRIORITY, HERE,
                          "Msg %d corrupted; ignoring previous header"
                          " information.",
                          mp->number );
            }
            mp->number     = msg_num;
            mp->length     = 0;
            mp->lines      = 0;
            mp->body_lines = 0;
            mp->offset     = ftell(fdesc) - nchar;
            mp->del_flag   = FALSE;
            mp->hide_flag  = FALSE;
            mp->retr_flag  = FALSE;
            mp->orig_retr_state = FALSE;
            mp->uidl_str   = "\n";
            DEBUG_LOG2 ( p, "Msg %d being added to list %x", mp->number, p->mlp );
            inheader       = 1;
            uidl_found     = 0;
            content_nchar  = 0;
            content_length = 0;
            cont_len       = 0;
            if ( p->mmdf_separator )
                expecting_trailer = 1;

            continue;   /* Don't count message separator in total size */
        }
        
        if ( inheader ) {
            if ( *buffer == '\n' ) { /* End of headers */
                inheader = 0;
                content_length = cont_len;
                mp->body_lines = 1;  /* Count newline as the first body line */
                if ( mp->hide_flag == 0 )
                    mp->visible_num = ++visible_msg_num;
                
                if ( !uidl_found ) {
                    char    *cp;
                    int     i;

                    MD5Update ( &mdContext, (unsigned char *)frombuf, 
                                strlen(frombuf) );
                    MD5Final ( digest, &mdContext );
                    cp = mp->uidl_str = (char *)malloc((DIG_SIZE * 2) + 2);
                    cp = base80 ( cp, digest );
                    *cp++ = '\n';
                    *cp   = '\0';
                    
                    DEBUG_LOG2 ( p, "UID not found; generated UID(%d): %s",
                                    strlen(mp->uidl_str), mp->uidl_str );

                    mp->length   += strlen("X-UIDL: ") + strlen(mp->uidl_str) +1;
                    p->drop_size += strlen("X-UIDL: ") + strlen(mp->uidl_str) +1;

                    /* New UIDs do not dirty the mailspool if NO_STATUS is set.
                     * They are just recalculated each time the popper is run or
                     * LMOS is set and the mail spool is updated.
                     */
#    ifndef NO_STATUS
                    p->dirty = 1;
#    endif
                } /* !uidl_found */

            } else if ( CONTENT_LENGTH && 
                        !strncmp(buffer, "Content-Length:", 15) ) {
                cont_len = atoi(buffer + 15);
                MD5Update ( &mdContext, (unsigned char *)buffer, strlen(buffer) );
                mp->lines++;
                continue;   /* not part of the message size */
            } else if ( !uidl_found && 
                           ( !strncasecmp("Received:",   buffer,  9) ||
                             !strncasecmp("Date:",       buffer,  5) ||
                             !strncasecmp("Message-Id:", buffer, 11) ||
                             !strncasecmp("Subject:",    buffer,  8)  )
                      ) {
                MD5Update ( &mdContext, (unsigned char *)buffer,
                            strlen(buffer) );
            } else if ( !strncasecmp("X-UIDL:", buffer, 7) ) {
                if ( !uidl_found ) {
                    char *cp;
                    int len;

                    /* 
                     * Skip over header string 
                     */
                    cp = index ( buffer, ':' );
                    if ( cp != NULL ) {
                        cp++;
                        while ( *cp && ( *cp == ' ' || *cp == '\t') ) 
                            cp++;
                    } else
                        cp = "";

                    len = strlen ( cp );
                    if ( len > MIN_UIDL_LENGTH && 
                         len < MAX_UIDL_LENGTH ) {
                        uidl_found++;
                        mp->uidl_str  = (char *)strdup(cp);
                        mp->length   += nchar + 1;
                        p->drop_size += nchar + 1;
                        DEBUG_LOG1 ( p, "Found UIDL header: %s", buffer );
                    }
                    else
                        DEBUG_LOG4 ( p, "Ignoring UIDL header: %s "
                                        "(len=%d; min=%d; max=%d)",
                                     buffer, len, MIN_UIDL_LENGTH, 
                                     MAX_UIDL_LENGTH );
                } /* !uidl_found */
            
                mp->lines++;
                continue;  /* Do not include this value in the message size */
            } else if ( (strncasecmp(buffer, "Status:", 7) == 0) ) {
                if ( index(buffer, 'R') != NULL ) {
                    mp->retr_flag = TRUE;
                    mp->orig_retr_state = TRUE;
                }
            }
            if ( UW_KLUDGE && msg_num == 1 ) {
                if ( strncasecmp ( buffer, "Subject: ", 9  ) == 0   &&
                      ( strstr ( buffer, "DO NOT DELETE"             ) != NULL ||
                        strstr ( buffer, "DON'T DELETE THIS MESSAGE" ) != NULL ||
                        strstr ( buffer, "FOLDER INTERNAL DATA"      ) != NULL  
                      )
                   ) {
                    uw_hint++;
                    DEBUG_LOG2 ( p, "uw_hint now %i; UW_KLUDGE matched '%.64s'",
                                 uw_hint, buffer );
                } else
                if ( strncasecmp ( buffer, "X-IMAP: ", 8 ) == 0 ) {
                    uw_hint++;
                    DEBUG_LOG2 ( p, "uw_hint now %i; UW_KLUDGE matched '%.64s'",
                                 uw_hint, buffer );
                } else
                if ( strncasecmp ( buffer, "From: Mail System Internal Data", 31 ) == 0 ) {
                    uw_hint++;
                    DEBUG_LOG2 ( p, "uw_hint now %i; UW_KLUDGE matched '%.64s'",
                                 uw_hint, buffer );
                }
                if ( uw_hint == 2 ) {
                    mp->hide_flag = 1;
                    p->first_msg_hidden = 1;
                    DEBUG_LOG0 ( p, "UW_KLUDGE check matched 2 conditions; msg hidden" );
                }
            } /* UW_KLUDGE && msg_num == 1 */
        } /* inheader */
        else { /* not inheader */
            content_nchar += nchar;
            mp->body_lines++;
        }
           
        mp->length   += nchar + 1;
        p->drop_size += nchar + 1;
        mp->lines++;
    } /* for loop */

    p->msg_count = msg_num;
    p->visible_msg_count = visible_msg_num;

    DEBUG_LOG1 ( p, "init_dropinfo exiting; newline=%d", newline );
    return ( POP_SUCCESS );
}


/* 
 *  Used to be dropinfo:   Extract information about the POP maildrop and store 
 *  it for use by the other POP routines.
 *
 *  Now, the copy and info collection are done at the same time.
 *
 *  Parameters:
 *     p:           Pointer to POP structure.
 *     mfd:         File descriptor for spool file (p->drop_name is name).
 *     dfd:         File descriptor for temp drop  (p->temp_drop is name).
 *     time_locked  time(0) when maillock acquired.
 */
int
do_drop_copy ( POP *p, int mfd, int dfd, time_t time_locked ) 
{
    char            buffer  [ MAXLINELEN ];      /*  Read buffer */
    char            frombuf [ MAXLINELEN ];      /*  To save From Envelope */
    MsgInfoList    *mp;                          /*  Pointer to message 
                                                     info list */
    int             nchar;                       /*  Bytes written/read */
    int             inheader;                    /*  Header section flag */
    int             uidl_found;                  /*  UIDL header flag */
    int             msg_num;                     /*  Current msg number */
    int             visible_msg_num;             /*  Current visible msg num */
    int             expecting_trailer;
    int             content_length, content_nchar, cont_len;
    MD5_CTX         mdContext;
    unsigned char   digest[16];
    int             mangled_one;
    FILE           *mail_drop;                   /*  Streams for fids */
    long            unique_num = time(0);
    int             uw_hint    = 0;


    DEBUG_LOG1 ( p, "DROPCOPY: Reading the mail drop (p->msg_count = %i).",
                 p->msg_count );
    /*  
     * Acquire a stream pointer for the maildrop 
     */
    mail_drop = fdopen ( mfd, "r" );
    if ( mail_drop == NULL ) {
        return pop_msg ( p, POP_FAILURE, HERE,
                         "[SYS/TEMP] Cannot assign stream for %s: %s (%d)",
                         p->drop_name, STRERROR(errno), errno );
    }

    rewind ( mail_drop );
    newline = 1;  /* first line of file counts as having new line before it */

#ifdef NULLKLUDGE
    {/* 
      * Eat NULLs at the beginning of the mailspool 
      */
    char tempchar;
    
    while ( ( tempchar = getc ( p->drop ) ) == 0 );
    ungetc ( tempchar, p->drop );
    }
#endif

    if ( mfgets ( buffer, MAXMSGLINELEN, mail_drop ) == NULL ) {
        return ( POP_SUCCESS );
    }

    /* 
     * Is this an MMDF file? 
     */
    if ( *buffer == MMDF_SEP_CHAR ) {
        if ( !p->mmdf_separator )
            p->mmdf_separator = (char *)strdup(buffer);
    } else if ( !isfromline ( buffer ) ) {
        DEBUG_LOG5 ( p, "newline=%d; isOflow=%d; wasOflow=%d; buffer(%u):  %.256s", 
                     newline, isOflow, wasOflow, strlen(buffer), buffer );
        return pop_msg ( p, POP_FAILURE, HERE,
                            "[SYS/PERM] Unable to process From lines "
                            "(envelopes), change recognition modes or "
                            "check for corrupted mail drop." );
    }

    newline = 1;  /* first line of file counts as having new line before it */
    rewind ( mail_drop );

    /*  
     * Scan the file, loading the message information list with 
     * information about each message 
     */

    inheader          = 0;
    uidl_found        = 0;
    expecting_trailer = 0;
    msg_num           = p->msg_count;
    visible_msg_num   = p->visible_msg_count;
    content_length    = 0;
    content_nchar     = 0;
    cont_len          = 0;
    mangled_one       = 0;
    p->msg_count      = (((p->msg_count - 1) / ALLOC_MSGS) + 1) * ALLOC_MSGS;

#ifdef NULLKLUDGE
    {/* Eat NULLs at the beginning of the mailspool if any.*/
    char tempchar;
    
    while ( ( tempchar = getc ( p->drop ) ) == 0 );
    ungetc  ( tempchar, p->drop );
    }
#endif /* NULLKLUDGE */

    for ( mp = p->mlp + msg_num - 1; 
          mfgets ( buffer, MAXMSGLINELEN, mail_drop ); 
        ) {
        nchar = strlen ( buffer );
        
        /*
         * It is very important to refresh the mail lock on an
         * interval between one and three minutes. 
         */
        if ( msg_num % 500 == 0 &&
             time(0) - time_locked > 60 ) {
            Qtouchlock();
            time_locked = time(0);
        }

        if ( fputs ( buffer, p->drop)  == EOF ) {
            if ( errno == EDQUOT )
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] Unable to copy mail spool file, "
                                 "quota exceeded (%d)",
                                  errno );
            return pop_msg ( p, POP_FAILURE, HERE,
                             "[SYS/TEMP] Unable to copy mail spool file to "
                             "temp pop dropbox %s: %s (%d)",
                             p->temp_drop, STRERROR(errno), errno );
        }
    
        /* 
         * End of previous message 
         */
        
        if ( ( content_nchar >= content_length ) &&
             ( p->mmdf_separator ? !strcmp ( p->mmdf_separator, buffer )
                                 : isfromline ( buffer )
             ) 
           ) {
            /*
             * === Matched separator, tie off last message, start new one ===
             */


            if ( expecting_trailer ) {
                expecting_trailer = 0;
                continue;
            }

            if ( !p->mmdf_separator ) {
                strcpy ( frombuf, buffer );
            }

            /*
             * === Starting new message ===
             */
            MD5Init   ( &mdContext );
            MD5Update ( &mdContext, (unsigned char *)buffer, strlen(buffer) );
            /* 
             * Include a unique number in case everything else is not 
             */
            unique_num++;
            MD5Update ( &mdContext, (unsigned char *)&unique_num, sizeof(long) );

            if ( !inheader ) {
                if ( ++msg_num > p->msg_count ) {
                    p->mlp = (MsgInfoList *) realloc ( p->mlp,
                                                       (p->msg_count+=ALLOC_MSGS)
                                                       * sizeof(MsgInfoList) );
                    if ( p->mlp == NULL ) {
                        p->msg_count = 0;
                        return pop_msg ( p, POP_FAILURE, HERE,
                                        "[SYS/TEMP] Can't build message list "
                                        "for '%s': Out of memory",
                                        p->user );
                    }
                    mp = p->mlp + msg_num - 2;
                } /* ++msg_num > p->msg_count */
            if ( msg_num != 1 )
                DEBUG_LOG5 (p, "Msg %d uidl '%s' at offset %d is %d octets long"
                               " and has %u lines.",
                            mp->number, mp->uidl_str, mp->offset, mp->length,
                            mp->lines );
            ++mp;

            } else { /* still in header */
                pop_log ( p, POP_PRIORITY, HERE,
                          "Msg %d corrupted, ignoring previous header information.",
                          mp->number );
            }

            mp->number     = msg_num;
            mp->length     = 0;
            mp->lines      = 0;
            mp->body_lines = 0;
            mp->offset     = ftell(p->drop) - nchar;
            mp->del_flag   = FALSE;
            mp->hide_flag  = FALSE;
            mp->retr_flag  = FALSE;
            mp->orig_retr_state = FALSE;
            mp->uidl_str   = "\n";

            DEBUG_LOG1 ( p, "Msg %d being added to list.", mp->number );
            inheader       = 1;
            content_length = 0;
            content_nchar  = 0;
            cont_len       = 0;
            uidl_found     = 0;
            if ( p->mmdf_separator )
                expecting_trailer = 1;

            continue;   /* Do not include From separator in message size */
            } /* tie off last message, start new one */

        /* 
         * ====== Handle header and body of message here ==== 
         */
        if ( inheader ) {
            /* 
             * ===== Handle the header of a message ==== 
             */
            if ( *buffer == '\n' ) {
                /* 
                 * === Encountered transition to body ====
                 */
                inheader = 0;
                mp->body_lines = 1;
                content_length = cont_len;
                if ( mp->hide_flag == 0 )
                    mp->visible_num = ++visible_msg_num;

                if ( !uidl_found ) {
                    char *cp;
                    int  i;

                    MD5Update ( &mdContext, (unsigned char *)frombuf,
                                strlen(frombuf) );
                    MD5Final  ( digest, &mdContext );
                    cp = mp->uidl_str = (char *)malloc((DIG_SIZE * 2) + 2);
                    cp = base80 ( cp, digest );
                    *cp++ = '\n';
                    *cp   = '\0';
                    DEBUG_LOG3 ( p, "UID not found; generated UID(%d): %s [%d]",
                                    strlen(mp->uidl_str), mp->uidl_str, __LINE__ );

                    mp->length   += strlen("X-UIDL: ") + strlen(mp->uidl_str) +1;
                    p->drop_size += strlen("X-UIDL: ") + strlen(mp->uidl_str) +1;

                    /* New UIDs do not dirty the mailspool if NO_STATUS is
                     * set.  They are just recalculated each time the
                     * popper is run or LMOS is set and the mail spool
                     * is updated. 
                     */
#    ifndef NO_STATUS
                    p->dirty = 1;
#    endif
                } /* !uidl_found */

            } else 
            if ( CONTENT_LENGTH && 
                 !strncmp ( buffer, "Content-Length:", 15 ) ) {
                /*=== 
                 * Handle content_length if we do that sort of thing ===
                 */
                cont_len = atoi ( buffer + 15 );
                MD5Update ( &mdContext, (unsigned char *)buffer, strlen(buffer) );
                mp->lines++;
                continue;  /* Not included in message size */

            } else 
            if ( !uidl_found && 
                            ( !strncasecmp ( "Received:",   buffer,  9 ) ||
                              !strncasecmp ( "Date:",       buffer,  5 ) ||
                              !strncasecmp ( "Message-Id:", buffer, 11 ) ||
                              !strncasecmp ( "Subject:",    buffer,  8 )  )
                      ) {
                /* 
                 * === Accumuate certain header we might see for UIDL ===
                 */
                MD5Update ( &mdContext, (unsigned char *)buffer, 
                            strlen(buffer) );
            } else if ( !strncasecmp ( "X-UIDL:", buffer, 7 ) ) {
                /* 
                 * ==== Do the UIDL header ==== 
                 */
                if ( !uidl_found ) {
                    char *cp;
                    int len;

                    /* 
                     * Skip over header 
                     */
                    cp = index ( buffer, ':' );
                    if ( cp != NULL ) {
                        cp++;
                        while ( *cp && ( *cp == ' ' || *cp == '\t' ) ) 
                            cp++;
                    } else
                        cp = "";

                    len = strlen ( cp );
                    if ( len > MIN_UIDL_LENGTH && 
                         len < MAX_UIDL_LENGTH ) {
                        uidl_found++;
                        mp->uidl_str = (char *)strdup(cp);
                        mp->length += nchar + 1;
                        p->drop_size += nchar + 1;
                        DEBUG_LOG1 ( p, "Found UIDL header: %s", buffer );
                    }
                    else
                        DEBUG_LOG4 ( p, "Ignoring UIDL header: %s "
                                        "(len=%d; min=%d; max=%d)",
                                     buffer, len, MIN_UIDL_LENGTH, 
                                     MAX_UIDL_LENGTH );
                } /* !uidl_found */
                mp->lines++;
                continue;  /* Do not include this value in the message size */
            } 
            else if ( !strncasecmp ( buffer, "Status:", 7 ) ) {
                /* 
                 * === Do the status header === 
                 */
                if ( index(buffer, 'R') != NULL ) {
                    mp->retr_flag = TRUE;
                    mp->orig_retr_state = TRUE;
                }
            }
            if ( UW_KLUDGE && msg_num == 1 ) {
                if ( strncasecmp ( buffer, "Subject: ", 9  ) == 0   &&
                      ( strstr ( buffer, "DO NOT DELETE"             ) != NULL ||
                        strstr ( buffer, "DON'T DELETE THIS MESSAGE" ) != NULL ||
                        strstr ( buffer, "FOLDER INTERNAL DATA"      ) != NULL  
                      )
                   ) {
                    uw_hint++;
                    DEBUG_LOG2 ( p, "uw_hint now %i; UW_KLUDGE matched '%.64s'",
                                 uw_hint, buffer );
                } else
                if ( strncasecmp ( buffer, "X-IMAP: ", 8 ) == 0 ) {
                    uw_hint++;
                    DEBUG_LOG2 ( p, "uw_hint now %i; UW_KLUDGE matched '%.64s'",
                                 uw_hint, buffer );
                } else
                if ( strncasecmp ( buffer, "From: Mail System Internal Data", 31 ) == 0 ) {
                    uw_hint++;
                    DEBUG_LOG2 ( p, "uw_hint now %i; UW_KLUDGE matched '%.64s'",
                                 uw_hint, buffer );
                }
                if ( uw_hint == 2 ) {
                    mp->hide_flag = 1;
                    p->first_msg_hidden = 1;
                    DEBUG_LOG0 ( p, "UW_KLUDGE check matched 2 conditions; msg hidden" );
                }
            } /* UW_KLUDGE && msg_num == 1 */
        } /* inheader */
        else {
            /* 
             * ==== Handle the body of a message (not much to do) === 
             */
            content_nchar += nchar;
            mp->body_lines++;
        }

        mp->length   += nchar + 1;
        p->drop_size += nchar + 1;
        mp->lines++;
    } /* for loop */

    p->msg_count = msg_num;
    p->visible_msg_count = visible_msg_num;

#ifdef DEBUG
    if ( p->debug && msg_num > 0 ) {
        register int    i;
        
        for ( i = 0, mp = p->mlp; i < p->msg_count; i++, mp++ )
            pop_log ( p, POP_DEBUG, HERE,
                      "Msg %d (%d) uidl '%s' at offset %d is %d octets long"
                      "%.*s and has %u lines.",
                      mp->number, mp->visible_num, mp->uidl_str, 
                      mp->offset, mp->length,
                      (mp->hide_flag ? 9 : 0), " (hidden)",
                      mp->lines );
    }
#endif

    if ( fflush ( p->drop ) == EOF )
        return pop_msg ( p, POP_FAILURE, HERE, 
                         "[SYS/TEMP] Flush of temp pop dropbox %s failed: %s (%d)",
                         p->temp_drop, STRERROR(errno), errno );

    return ( POP_SUCCESS );
}

/* 
 *  dropcopy:   Make a temporary copy of the user's mail drop and 
 *  save a stream pointer for it.
 */

pop_dropcopy ( p, pwp )
POP        *   p;
struct passwd   * pwp;
{
    int                     mfd;                    /*  File descriptor for 
                                                        the user's maildrop */
    int                     dfd;                    /*  File descriptor for 
                                                        the SERVER maildrop */
    FILE                   *tf;                     /*  The temp file */
    int                     tfn;            
    char                    buffer [ MAXLINELEN ];  /*  Read buffer */
    OFF_T                   offset = 0;             /*  Old/New boundary */
    int                     nchar;                  /*  Bytes written/read */
    struct stat             mybuf;                  /*  For fstat() */
    char                   *ptr;
    int                     rslt;
    time_t                  time_locked;


    if ( genpath ( p->user, 
                   p->drop_name, 
                   sizeof(p->drop_name),
                   GNPH_SPOOL,
                   (p->debug ? p->trace : NULL),
                   p->debug ) < 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                          "[SYS/TEMP] Unable to get temporary drop name" ) );

    /*  
     *  Get name of temporary maildrop into which to copy the updated 
     *  maildrop. 
     *
     *  Older versions of qpopper ignored HASH_SPOOL and HOMEDIRMAIL
     *  when building the path for this file, so we first check if
     *  a file exists using the old rules.  If not, or if it is empty,
     *  we build it where it goes.
     *  
     */
    if ( genpath ( p->user, 
                   p->temp_drop, 
                   sizeof(p->temp_drop),
                   GNPH_OLDPOP,
                   (p->debug ? p->trace : NULL),
                   p->debug ) < 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE,
                              "[SYS/TEMP] Unable to get temporary drop name" ) );
    
    if ( stat ( p->temp_drop, &mybuf ) == -1 || mybuf.st_size <= 0 ) {
        if ( genpath ( p->user, 
                       p->temp_drop, 
                       sizeof(p->temp_drop),
                       GNPH_POP,
                       (p->debug ? p->trace : NULL),
                       p->debug ) < 0 )
            return ( pop_msg ( p, POP_FAILURE, HERE,
                               "[SYS/TEMP] Unable to get temporary drop name" ) );
    }
    
    DEBUG_LOG1 ( p, "Temporary maildrop name: '%s'", p->temp_drop );

#ifdef BULLDB
#  define SLEEP_INTERVAL 1 /* how long to wait between tries */
#  define MAX_ATTEMPTS  10 /* when to give up */
    if (p->bulldir) {
        char bull_db [ MAXLINELEN ];
        int  tries     = 0;
        int  db_err    = 0;
        int  try_again = 1;

#  ifdef BULLDBDIR
        sprintf(bull_db, "%s/bulldb", BULLDBDIR);
#  else /* BULLDBDIR not defined, use bull dir */
        sprintf(bull_db, "%s/bulldb", p->bulldir);
#  endif /* BULLDBDIR */

        p->bull_db = NULL;
    
        while ( try_again ) {
#  ifdef GDBM
            p->bull_db = gdbm_open (bull_db, 512, GDBM_WRCREAT, 0600, 0);
#  else /* not GDBM */
            p->bull_db = dbm_open (bull_db, O_RDWR | O_CREAT, 0600);
#  endif /* GDBM */
            if ( p->bull_db == NULL && errno == EAGAIN
                                    && tries++ <= MAX_ATTEMPTS ) {
                DEBUG_LOG3 ( p, "Bulletin database '%s' unavailable on "
                                "try #%d (user %s); sleeping",
                            bull_db, tries, p->user );
                sleep ( tries * SLEEP_INTERVAL );   /* Sleep between retries */
            } else {
                try_again = 0;
            }
        } /* while loop */

        if ( p->bull_db == NULL ) {
            pop_log (p, POP_PRIORITY, HERE, "gdbm_open failed: %s (%d)",
                    sys_errlist[errno], errno );
            return ( pop_msg ( p, POP_FAILURE, HERE, 
                               "[SYS/TEMP] Unable to open Bulletin "
                               "database; contact your administrator" ) );
        } else {
            DEBUG_LOG2 ( p, "Opened Bulletin database '%s' (checking user '%s')",
                        bull_db, p->user );
        }
        
    } /* if (p->bulldir) */
#endif /* BULLDB */

#ifdef HASH_SPOOL
    { /* HASH_SPOOL block */
    /* 
     * Make sure the path to the spool file exists 
     */
     mode_t  my_umask;
     mode_t  spool_mode;
     mode_t  temp_mode = 0x1C0;
     uid_t   spool_owner;
     gid_t   spool_group;
     char   *ptr;
     
    /*
     * Get the path name.
     */
    if ( genpath ( p->user, 
                   buffer, 
                   sizeof(buffer),
                   GNPH_PATH,
                   (p->debug ? p->trace : NULL),
                   p->debug ) < 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE,
                           "[SYS/TEMP] Unable to get spool path name" ) );

    /*
     * Get the mode of the parent directory.
     */
    if ( stat ( POP_MAILDIR, &mybuf ) != 0 )
        return pop_msg ( p, POP_FAILURE, HERE, 
                         "[SYS/TEMP] Unable to access spool directory "
                         "%s: %s (%d)",
                         POP_MAILDIR, STRERROR(errno), errno );
    /*
     * We create subdirectories with same mode and owner/group as parent.
     */
    spool_mode  = mybuf.st_mode;
    spool_owner = mybuf.st_uid;
    spool_group = mybuf.st_gid;
    DEBUG_LOG4 ( p, "Spool directory %s has mode %04o; owner %d; group %d",
                 POP_MAILDIR, spool_mode, spool_owner, spool_group );

    /*
     * Zero umask during this operation, so we can create the new
     * directories with the correct modes.
     */
    my_umask = umask ( 0000 );
    DEBUG_LOG4 ( p, "umask was %04o; now %04o (my real UID=%d; eUID=%d)", 
                 my_umask, 0000, getuid(), geteuid() );

   /*
    * In theory, we could just create the new directories with mkdir(2)
    * or mknod(2), and pass in the desired mode.  But this causes
    * problems: we can't set the sticky bit this way on Solaris, and 
    * sometimes the call fails on Linux.  So We first create the
    * directories with a temporary mode, then call chmnod(2).
    */

#  if HASH_SPOOL == 2
    /*
     * This method uses two-level hashed spool directories -- create
     * first one first.
     */
    for ( ptr = buffer + ( strlen(buffer) -1 );
          ptr > buffer && *ptr != '/';
          ptr-- );
    if ( *ptr == '/' ) {
        *ptr = '\0';
        if ( stat ( buffer, &mybuf ) == -1 && errno == ENOENT ) {
            /* The directory doesn't exit -- create it */
            if ( mkdir ( buffer, temp_mode ) == -1 )
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] Unable to create spool directory "
                                 "%s (%04o): %s (%d)",
                                 buffer, temp_mode, STRERROR(errno), errno );
            if ( chmod ( buffer, spool_mode ) == -1 )
                return pop_msg ( p, POP_FAILURE, HERE,
                                "[SYS/TEMP] Unable to chmod of %s to %04o: "
                                "%s (%d)",
                                buffer, spool_mode, STRERROR(errno), errno );
            if ( chown ( buffer, spool_owner, spool_group ) == -1 )
                return pop_msg ( p, POP_FAILURE, HERE,
                                "[SYS/TEMP] Unable to set owner/group of spool "
                                "directory %s to %d/%d: %s (%d)",
                                buffer, spool_owner, spool_group, 
                                STRERROR(errno), errno );

#    ifdef DEBUG
            if ( stat ( buffer, &mybuf ) != 0 )
                return pop_msg ( p, POP_FAILURE, HERE, 
                                "[SYS/TEMP] Unable to access newly-created "
                                "dir %s: %s (%d)",
                                buffer, STRERROR(errno), errno );
            DEBUG_LOG4 ( p, "Created %s; mode=%04o; owner=%d; group=%d",
                         buffer, mybuf.st_mode, mybuf.st_uid, mybuf.st_gid );
#    endif /* DEBUG */

        } /* directory doesn't exit */
        *ptr = '/';
    } else {
        DEBUG_LOG1 ( p, "HASH_SPOOL=2 but unable to find first dir in '%s'",
                     buffer );
    }
#  endif /* HASH_SPOOL == 2 */
    
    if ( stat ( buffer, &mybuf ) == -1 && errno == ENOENT ) {
        /* The directory doesn't exit -- create it */
        DEBUG_LOG2 ( p, "HASH_SPOOL; Creating spool path: %s (mode %04o)",
                     buffer, temp_mode );
        if ( mkdir ( buffer, temp_mode ) == -1 )
            return pop_msg ( p, POP_FAILURE, HERE,
                             "[SYS/TEMP] Unable to create spool directory "
                             "%s (%04o): %s (%d)",
                             buffer, temp_mode, STRERROR(errno), errno );
        if ( chmod ( buffer, spool_mode ) == -1 )
            return pop_msg ( p, POP_FAILURE, HERE,
                             "[SYS/TEMP] Unable to chmod of %s to %04o: %s (%d)",
                             buffer, spool_mode, STRERROR(errno), errno );
        if ( chown ( buffer, spool_owner, spool_group ) == -1 )
            return pop_msg ( p, POP_FAILURE, HERE,
                             "[SYS/TEMP] Unable to set owner/group of spool "
                             "directory %s to %d/%d: %s (%d)",
                             buffer, spool_owner, spool_group, 
                             STRERROR(errno), errno );

#    ifdef DEBUG        
        if ( stat ( buffer, &mybuf ) != 0 )
            return pop_msg ( p, POP_FAILURE, HERE, 
                             "[SYS/TEMP] Unable to access newly-created "
                             "dir %s: %s (%d)",
                             buffer, STRERROR(errno), errno );
        DEBUG_LOG4 ( p, "Created %s; mode=%04o; owner=%d; group=%d",
                     buffer, mybuf.st_mode, mybuf.st_uid, mybuf.st_gid );
#    endif /* DEBUG */

    } /* directory doesn't exit */
    umask ( my_umask );
    DEBUG_LOG1 ( p, "umask now %04o", my_umask );
    } /* HASH_SPOOL block */
#endif /* HASH_SPOOL */

    /* Here we work to make sure the user doesn't cause us to remove or
     * write over existing files by limiting how much work we do while
     * running as root.
     */

#ifdef BINMAIL_IS_SETGID
#  if BINMAIL_IS_SETGID > 1
   pwp->pw_gid = (gid_t)BINMAIL_IS_SETGID;
#  else
   if ( !stat(POP_MAILDIR, &mybuf) )
       pwp->pw_gid = mybuf.st_gid;
#  endif /* BINMAIL_IS_SETGID > 1 */
#endif /* BINMAIL_IS_SETGID */

    /* 
     * Now we run as the user. 
     */
    (void) setgid ( (GID_T)pwp->pw_gid );
    /* 
     * Set the supplementary groups list 
     */
    (void) setgroups ( 1, (GID_T *)&pwp->pw_gid ); 
    (void) setuid ( (UID_T)pwp->pw_uid );

    DEBUG_LOG2 ( p, "uid = %d, gid = %d", getuid(), getgid() );

    dfd = open ( p->temp_drop, O_RDWR | O_CREAT, 0600 );
    if ( dfd == -1 ) {
        pop_log ( p, POP_PRIORITY, HERE,
                  "[SYS/TEMP] Unable to open temporary maildrop "
                  "'%s': %s (%d)",
                  p->temp_drop, STRERROR(errno), errno );
        return pop_msg ( p, POP_FAILURE, HERE,
                         "[SYS/TEMP] Failed to create %s with uid %d, gid %d. "
                         "Change permissions.",
                          p->temp_drop, pwp->pw_uid, pwp->pw_gid );
    }

    fstat ( dfd, &mybuf );
    if ( mybuf.st_uid != pwp->pw_uid ) {
        close  ( dfd );
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                           "[SYS/PERM] Temporary drop %s not owned by %s.",
                           p->temp_drop, p->user) );
    }
#ifdef NEXT
    if ( mybuf.st_mode & (0x7) )
#else /* not NEXT */
    if ( mybuf.st_mode & (S_IWOTH | S_IROTH) )
#endif /* NEXT */
    {
        close  ( dfd );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                           "[SYS/PERM] Your temporary file %s is accessible "
                           "by others.  This must be fixed",
                           p->temp_drop ) );
    }
    /* 
     * Make sure the mailspool is not a hard link 
     */
    if ( mybuf.st_nlink != 1 ) {
        close  ( dfd );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                           "[SYS/PERM] Your temporary file appears to have "
                           "more than one link." ) );
    }

    /* 
     * If the temporary popdrop is not empty, revert to regular mode. 
     */
    if ( mybuf.st_size != 0 )
        p->server_mode = 0;

    /* 
     * Lock the temporary maildrop.  Locking is just to check for single
     * pop session per user.
     */
    if ( flock ( dfd, LOCK_EX|LOCK_NB ) == -1 ) {
        switch ( errno ) {
            case EWOULDBLOCK:
                close ( dfd );
                return pop_msg ( p, POP_FAILURE, HERE,
                                "[IN-USE] %s lock busy!  Is another session "
                                "active? (%d)",
                                p->temp_drop, errno );
            default:
                close ( dfd );
                return pop_msg ( p, POP_FAILURE, HERE, 
                                "[SYS/TEMP] flock: '%s': %s (%d)",
                                p->temp_drop, STRERROR(errno), errno );
        }
    }
    
#ifndef KEEP_TEMP_DROP
    /* check for race conditions involving unlink.  See pop_updt.c */
    /* s-dorner@uiuc.edu, 12/91 */
    { /* KEEP_TEMP_DROP block */
    struct stat byName, byFd;
    if ( stat ( p->temp_drop, &byName ) || fstat ( dfd, &byFd )
                                        || byName.st_ino != mybuf.st_ino )
    {
        flock ( dfd, LOCK_UN );
        close ( dfd );
        return pop_msg ( p, POP_FAILURE, HERE,
                         "[IN-USE] Maildrop being unlocked; try again.");
    }
    } /* KEEP_TEMP_DROP block */
#endif /* KEEP_TEMP_DROP */
    
    /*  
     * Acquire a stream pointer for the temporary maildrop 
     */
    p->drop = fdopen ( dfd, "r+" );
    if ( p->drop == NULL ) {
        flock ( dfd, LOCK_UN );
        close ( dfd );
        return pop_msg ( p, POP_FAILURE, HERE, 
                         "[SYS/TEMP] Cannot assign stream for %s: %s (%d)",
                         p->temp_drop, STRERROR(errno), errno );
    }
    /*
     * Allocate memory for message information structures and this is
     * not deleted since a failure, for some reason in this function
     * would result in process death. 
     */
    p->mlp = (MsgInfoList *)calloc((unsigned)ALLOC_MSGS,sizeof(MsgInfoList));
    if ( p->mlp == NULL ){
        flock ( dfd, LOCK_UN );
        close ( dfd );
        return pop_msg ( p, POP_FAILURE, HERE,
                         "[SYS/TEMP] Can't allocate memory for message list." );
    }
    p->msg_count = 0;
    p->visible_msg_count = 0;
    p->drop_size = 0;

    if ( mybuf.st_size != 0 ) { /* Mostly this is for regular mode. */
        DEBUG_LOG2 ( p, "Temp drop %s not empty (%lu octets)", 
                     p->temp_drop, mybuf.st_size );
        if ( init_dropinfo ( p, p->temp_drop, p->drop, time_locked ) != POP_SUCCESS ) {
            /* Occurs on temp_drop corruption */
            flock ( dfd, LOCK_UN );
            close ( dfd );
            return ( POP_FAILURE );
        }
    /* 
     * Sync with stdio; should be at end anyway 
     */
    fseek ( p->drop, 0L, SEEK_END );
    offset = ftell ( p->drop );
    DEBUG_LOG1 ( p, "Temp drop has %lu octets of old data", offset );
    } /* mybuf.st_size != 0 */

    /* 
     * Always use Qmaillock().
     */

    DEBUG_LOG0 ( p, "Getting mail lock" );
    rslt = Qmaillock ( p->user, 1,  p->trace, p->debug );
    if ( rslt != 0 ) {
        flock ( dfd, LOCK_UN );
        close ( dfd );
        return pop_msg ( p, POP_FAILURE, HERE, 
                         "[SYS/TEMP] maillock error %d: '%s'", 
                         rslt, p->drop_name );
    }
    time_locked = time(0);

    /*  Open the user's maildrop, If this fails,  no harm in assuming empty */
    /* <todo> Mail drop has to be created if one doesn't exist for server_mode.
     * Because p->drop is used for bulletins. Right now it reverts back to 
     * regular mode if drop box open fails. </todo>
     */
    mfd = open ( p->drop_name, O_RDWR );
    if ( mfd > 0 ) {
        /* 
         * Lock the maildrop 
         */
        if ( flock ( mfd, LOCK_EX ) == -1 ) {
            flock ( dfd, LOCK_UN );
            close ( dfd );
            close ( mfd );
            Qmailunlock();
            return pop_msg ( p, POP_FAILURE, HERE, 
                             "[SYS/TEMP] flock: '%s': %s (%d)", 
                             p->drop_name, STRERROR(errno), errno );
        }

        if ( !p->server_mode ) {
            /*
             * New routine to copy and get dropbox info all at the same time 
             */
            nchar = do_drop_copy ( p, mfd, dfd, time_locked );

            if ( nchar != POP_SUCCESS ) {
                /* 
                 * pop_dropcopy has integrated the info gathering pass into
                 * the copy pass so now the information of the dropfile is
                 * inconsistent if there is an error.  Now we exit instead
                 * of trying to continue with what is available.
                 */
                DEBUG_LOG1 ( p, "do_drop_copy returned error; truncating "
                                "temp drop to %lu octets", offset );
                ftruncate ( dfd, offset );
                goto bailout;
            } 
            else { /* do_drop_copy worked */
                /* 
                 * Mail transferred!  Zero the mail drop NOW,  that we
                 * do not have to do gymnastics to figure out what's new
                 * and what is old later
                 */
                ftruncate ( mfd, 0 );
                DEBUG_LOG0 ( p, "Mail copied to temp drop; zeroing spool" );
            }

            /* 
             * Unlock and close the actual mail drop 
             */
            flock ( mfd, LOCK_UN );
            close ( mfd );
        } /* not p->server_mode */
        else { /* SERVER_MODE */
            /* 
             * Save the temporary drop FILE and fid values
             */
            p->hold = p->drop;
            p->drop = fdopen ( mfd, "r+" );
            if ( p->drop == NULL ) {
                pop_msg ( p, POP_FAILURE, HERE,
                          "[SYS/TEMP] Cannot assign stream for %s: %s (%d)",
                          p->drop_name, STRERROR(errno), errno );
                goto bailout;
            }
            if ( init_dropinfo ( p, p->drop_name, p->drop, time_locked ) != POP_SUCCESS )
                goto bailout;
        }
    } /* opened maildrop */
    else {
        /* 
         * Revert to regular operation when there are no mails in
         * users mail box. This is for the sake of any bulletins, 
         * which uses p->drop (temp_drop) and to allow pop_updt to
         * copy back to original mail box.
         */
        if ( p->server_mode ) {
            p->server_mode = 0; 
        }
        DEBUG_LOG3 ( p, "Unable to open maildrop %s: %s (%d)", 
                     p->drop_name, (errno < sys_nerr) ? sys_errlist[errno] : "", errno );
    }

    if ( p->bulldir ) {
        /* 
         * Recalculate offset 
         */
        fseek ( p->drop, 0L, SEEK_END );
        offset = ftell ( p->drop );
        DEBUG_LOG1 ( p, "Recalculated offset before bulletin processing: %lu",
                     offset );
        /* 
         * In Server_mode, p->drop can be null (no mails); 
         * pop_bull requires a drop handle. </bug> 
         */
        if ( pop_bull ( p, pwp ) != POP_SUCCESS ) {
            /* 
             * Return pop drop back to what it was before the bulletins 
             */
            ftruncate ( p->server_mode ? ( (mfd == -1) ? dfd : mfd ) 
                                       : dfd, 
                        offset );
            DEBUG_LOG3 ( p, "pop_bull reported error; truncating; "
                            "server_mode=%i; offset=%lu; fd=%s",
                         p->server_mode, offset, 
                         p->server_mode ? ( (mfd == -1) ? "dfd" : "mfd" ) 
                                        : "dfd" );
        }
    }
#ifdef BULLDB
#  ifdef GDBM
    gdbm_close ( p->bull_db );
#  else /* not GDBM */
    dbm_close ( p->bull_db );
#  endif /* GDBM */
    DEBUG_LOG0 ( p, "Closed Bulletin database" );
#endif /* BULLDB */

    fseek ( p->drop, 0L, SEEK_END );
    p->spool_end = ftell ( p->drop );
    DEBUG_LOG3 ( p, "Temp drop contains %lu (%lu visible) messages in %lu octets", 
                 p->msg_count, p->visible_msg_count, p->spool_end );

#ifdef DEBUG
    if ( p->debug && p->msg_count > 0 ) {
        register int    i;
        MsgInfoList *mp; 
                           
        for ( i = 0, mp = p->mlp; i < p->msg_count; i++, mp++ )
            pop_log ( p, POP_DEBUG, HERE,
                      "Msg %d (%d) uidl '%s' at offset %d is %d octets "
                      "long%.*s and has %u lines.",
                      mp->number, mp->visible_num, 
                      mp->uidl_str, mp->offset, mp->length,
                      (mp->hide_flag ? 9 : 0), " (hidden)",
                      mp->lines );
    }
#endif /* DEBUG */
    Qmailunlock();

    if ( p->server_mode )
        flock ( mfd, LOCK_UN );

    return ( POP_SUCCESS );

bailout:
    DEBUG_LOG0 ( p, "--- bailing out ---" );
    Qmailunlock();

#ifndef KEEP_TEMP_DROP
    if ( fstat ( dfd, &mybuf ) == 0 && mybuf.st_size == 0 ) {
        unlink ( p->temp_drop ); /* didn't get to use it */
        DEBUG_LOG1 ( p, "Removed empty temp drop: %s", 
                     p->temp_drop )
    }
#endif /* KEEP_TEMP_DROP */

    flock ( mfd, LOCK_UN );
    flock ( dfd, LOCK_UN );
    close ( mfd );
    close ( dfd );

#ifdef BULLDB
    if ( p->bull_db != NULL ) {
#  ifdef GDBM
        gdbm_close ( p->bull_db );
#  else /* not GDBM */
        dbm_close  ( p->bull_db );
#  endif /* GDBM */
    }
#endif /* BULLDB */
    return ( POP_FAILURE );
}

/*
 * This function emulates fgets() except that it replaces the NULLs with
 * SPACEs
 */

char *
mfgets ( s, size, stream )
char *s;
int size;
FILE *stream;
{
    int c;
    char *p = s;
#ifdef _DEBUG
    extern POP *global_debug_p;
#endif

    if ( size <= 0 ) {
        return NULL;
    }
    isOflow = 1; /* assume the worst */
    while ( --size && ((c = getc(stream)) != EOF) ) {
        if ( (*p = (char)c) == '\0' ) 
            *p = ' ';
        if ( *p++ == '\n' ) {
            isOflow = 0; /* we didn't overflow after all */
            break;
        }
    }
    if ( p == s ) {
        isOflow = 0;
        return NULL;
    }
    *p = '\0';
#ifdef _DEBUG
    DEBUG_LOG4 ( global_debug_p, "#mfgets: isOflow=%d wasOflow=%d s=(%d) %.128s",
               isOflow, wasOflow, strlen(s), s );
#endif
    return s;
}

