/*
 * Copyright (c) 2000 QUALCOMM Incorporated. All rights reserved.
 * See License.txt file for terms and conditions for modification and
 * redistribution.
 *
 * snprintf.c: 
 *
 * Revisions: 
 *
 *  04/21/00  [rcg]
 *            -  Changed snprintf to Qsnprintf, vsnprintf to Qvsnprintf.
 *               added Qsprintf.  Code now unconditionally included.
 *            -  Fixed copy_buf() to use strlen of source string instead
 *               of value of first character.
 *
 *  04/07/00  [rcg]
 *            -  Account for sprintf not returning length of added chars
 *               on some systems.
 *
 *  01/27/00  [rcg]
 *            -  Changed snprintf to vsnprintf, added snprint as jacket.
 *
 *  01/20/00  [rcg]
 *            -  Guarded with #ifndef HAVE_SNPRINTF
 *
 */

#include <sys/types.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>

#include "config.h"
#include "snprintf.h"



/*
 * A (hopefully full enough) snprintf implementation built over sprintf.
 * Based on modified version of snprint from one extracted from K&R 'C'.
 * Reference : Unix specification version 2. Copyright 1997 Open Group.
 * null terminated string.
 *
 * Caveats :
 *    - Limited support for format modifiers.
 *    - No nth argument specification %n$.
 *    - The known formats "sdcfxXgGeEou%".
 *    - Assuming int as 4 bytes, double as 8 bytes. No support for
 *      long double.
 *
 * Parameters:
 *     s:       buffer to print to.
 *     n:       max number of bytes to print.
 *     format:  the format string.
 *
 * Returned : On success the number of bytes printed to buffer.
 *            On failure -1.
 */



#  define BUFSIZE 1024
#  define MIN(A,B) ( ( (A) < (B) ) ? (A) : (B) )
#  define MAX(A,B) ( ( (A) > (B) ) ? (A) : (B) )


/*
 * Copy chars and update pointer and remaining space count.
 *
 * Parameters:
 *    s:   pointer to char * for destination.
 *    m:   pointer to source string.
 *    l:   maximum chars to copy.
 *    n:   pointer to remaining space count.
 *
 * Results:
 *    String 'm' is appended to 's'; 's' is updated to
 *    point at terminating null; 'n' is decremented by
 *    number of characters copied.
 *
 *    Return value is number of characters copied.
 */
static int 
copy_buf ( char **s, char *m, size_t l, size_t *n )
{
    l = MIN ( strlen(m), l );
    strncpy ( *s, m, l );
    
    l = MIN ( l, strlen(m) );
    *n     -= l;
    *s     += l;
    return    l;
}


enum STATES_TAG 
{
    IN_FORM,    /* Copying format string chars */
    IN_CONV,    /* In a conversion specification */
    IN_LEN      /* In a maximum length field */
};

typedef enum STATES_TAG STATES;


int 
Qvsnprintf ( char *s, size_t n, const char *format, va_list ap )
{
    char    *sval;
    int     ival;
    double  dval;
    char    cval;
    float   fval;
    char    *p = (char *)format,       /* Pointer into format string */
             msgBuf [ BUFSIZE ],       /* Temporary store for printing
                                        *     ints, floats etc using sprintf */
             frmToken [ 64 ],          /* Temporary store for conversion
                                        *     tokens. */
            *f = NULL;                 /* Pointer into frmToken */
    STATES   nState = IN_FORM; 
    size_t   nSize  = n-1;
    long     width  = -1;              /* If we have a width minimum */
    long     limit  = -1;              /* If we have a length limiter */


    if ( s == NULL || n == 0 || format == NULL ) 
        return -1;
    
    for ( ; nSize > 0 && *p != '\0'; p++ ) {
        if ( nState == IN_FORM ) {
            if ( *p != '%' ) {
                /*
                 * Still not in a conversion spec; just copy the char
                 */
                *s++ = *p;
                nSize--;
            }
            else {
                /*
                 * Start of a conversion spec
                 */
                nState =  IN_CONV;
                width  = -1;
                limit  = -1;
                f      = frmToken;
                *f++   = *p;
            }
        } /* nState == IN_FORM */
        else { /* We're still inside a conversion spec */
            switch ( *p ) {

            case 'c':                   /* C converts char args to int */
            case 'd':
            case 'i':
            case 'o':
            case 'u':
            case 'x':
            case 'X':
                /*
                 * Ensure width or max length, if specified, does not exceed
                 * available space.
                 */
                if ( width != -1 || limit != -1 )
                    if ( MAX ( width,   limit ) >
                         MIN ( BUFSIZE, nSize )
                       )
                        return -1;

                *f++ = *p;
                *f   = '\0';
                ival = va_arg ( ap, int );
                sprintf  ( msgBuf, frmToken, ival );
                copy_buf ( &s, msgBuf, nSize, &nSize );
                nState = 0;
                break;

            case 'f':                   /* C converts float args to double */
            case 'e':
            case 'E':
            case 'g':
            case 'G':
                /*
                 * Ensure width or max length, if specified, does not exceed
                 * available space.
                 */
                if ( width != -1 || limit != -1 )
                    if ( MAX ( width,   limit ) >
                         MIN ( BUFSIZE, nSize )
                       )
                        return -1;

                *f++ = *p;
                *f   = '\0';
                dval = va_arg ( ap, double );
                sprintf  ( msgBuf, frmToken, dval );
                copy_buf ( &s, msgBuf, nSize, &nSize );
                nState = IN_FORM;
                break;

            case 's':
                /*
                 * Ensure width or max length, if specified, does not exceed
                 * available space.
                 */
                if ( width != -1 || limit != -1 )
                    if ( MAX ( width,   limit ) >
                         MIN ( BUFSIZE, nSize )
                       )
                        return -1;

                /* 
                 * Get the string pointer.  If NULL, or if the maximum length
                 * is zero, we're done with this conversion.
                 */
                sval = va_arg ( ap, char * );
                if ( sval == NULL || limit == 0 ) {
                    nState = IN_FORM;
                    break;
                }
                
                /*
                 * Make sure we have room if limit is unspecified.
                 */
                if ( limit == -1 && strlen(sval) > nSize )
                    return -1;
                
                /* 
                 * Handle minimum width
                 */
                if ( width != -1 && width > strlen(sval) ) {
                  for ( ; width > 0; width-- )
                      copy_buf ( &s, " ", nSize, &nSize );
                    }
                
                /*
                 * Handle maximum length
                 */
                if ( limit != -1 && limit < strlen(sval) )
                    copy_buf ( &s, sval, limit, &nSize );
                else
                    copy_buf ( &s, sval, nSize, &nSize );
                
                nState = IN_FORM;
                break;
            
            case '%':
                *s++ = *p;
                nSize--;
                nState = IN_FORM;
                break;
            
            case '*': 
                /*
                 * Variable min width or max length specification.
                 */
                if ( nState == IN_LEN && limit == -1 ) {
                    limit = va_arg ( ap, int );
                    sprintf ( f, "%li", limit );
                    f += strlen ( f ); /* on some systems sprintf doesn't
                                          return length of added chars */
                }
                else
                if ( nState == IN_FORM && width == -1 ) {
                    width  = va_arg ( ap, int );
                    sprintf ( f, "%li", width );
                    f += strlen ( f ); /* on some systems sprintf doesn't
                                          return length of added chars */
                }
                else
                    return -1;
                break;
            
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
                *f++ = *p;
                *f   = '\0';

                /*
                 * Width or max length specification.  If the
                 * appropriate variable is -1, this is the start;
                 * otherwise it is a subsequent digit and we just
                 * skip over it.
                 */
                if ( nState == IN_LEN && limit == -1 )
                    limit = atoi ( p );
                else
                if ( nState == IN_FORM && width == -1 )
                    width  = atoi ( p );
                break;
             
            case '.':
                /*
                 * Max length follows
                 */
                *f++ = *p;
                *f   = '\0';
                nState = IN_LEN;
                break;
            
            default:
                *f++ = *p;
                *f   = '\0';
                break;
            } /* switch */
        } /* We're still inside a format */
    } /* for loop */
    
    if ( nSize ) 
        *s++ = *p;
    return ( (n-1) - nSize );
}


int 
Qsnprintf ( char *s, size_t n, const char *format, ... )
{
    int      rslt;
    va_list  ap;                       /* Pointer into stack to extract
                                        *     parameters */
    va_start ( ap, format );
    rslt = Qvsnprintf ( s, n, format, ap );
    va_end   ( ap );
    return rslt;
}


int 
Qsprintf ( char *s, const char *format, ... )
{
    int      rslt;
    va_list  ap;                       /* Pointer into stack to extract
                                        *     parameters */
    va_start ( ap, format );
    rslt = Qvsnprintf ( s, 2147483647, format, ap );
    va_end   ( ap );
    return rslt;
}
