/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*	BSDI ifconfig.c,v 2.22 1997/01/09 16:53:02 jch Exp	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1983, 1993\n\
	The Regents of the University of California.  All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)ifconfig.c	8.2 (Berkeley) 2/16/94";
#endif /* not lint */

#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include <net/if.h>
#include <net/if_media.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/if_ppp.h>
#include <net/if_pppioctl.h>
#include <net/if_pif.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define	NSIP
#include <netns/ns.h>
#include <netns/ns_if.h>
#include <netdb.h>

#define EON
#include <netiso/iso.h>
#include <netiso/iso_var.h>
#include <sys/protosw.h>

#ifdef INET6
#include <netinet6/in6_var.h>
#include <netinet6/nd6.h>
#endif /* INET6 */

#include <err.h>
#include <errno.h>
#include <ifaddrs.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* Known address families */
struct afswitch {
	const	char *af_name;		/* Address family name */
	int	af_af;
	struct	sockaddr *af_stab[3];
					/* Table of interface addresses */
#define ADDR	0
#define MASK	1
#define DSTADDR	2
	void	(*af_status) __P((struct ifaddrs *));
	void	(*af_getaddr) __P((struct afswitch *, char *, long));
	int	(*af_cmpaddr) __P((char *, struct sockaddr *, struct sockaddr *));
	int	af_difaddr;		/* ioctl to delete an address */
	int	af_aifaddr;		/* ioctl to add a new address */
	struct	ifreq *af_ridreq;	/* ifreq structure for delete */
	struct	ifaliasreq *af_addreq;	/* ifalias structure for add */
};

struct llswitch {
	const	char *ll_name;
	u_int	ll_type;
};

struct cmd {
	const	char *c_name;		/* Argument name */
	u_int	c_parameter;		/* Flag to set or reset */
#define	NEXTARG		0xffffff	/* Next argv */
	u_int	c_flag;			/* Flag to check and set */
	void	(*c_func) __P((struct afswitch *,
			       struct cmd *,
			       char *));
	int	c_af;			/* 0, AF not required, -1, AF */
					/* required, or must match */
					/* supplied AF */
	u_int	c_ltype;		/* used by media routines */
};

struct pifflags_cmd {
	char	*pf_name;		/* name */
	int	pf_report;		/* should this be printed in status? */
	u_long	pf_reset;		/* bits to clear (mask) */
	u_long	pf_set;			/* bits to set */
};

static struct	ifaliasreq addreq;	/* For adding addresses */
static struct	ifreq dlreq;		/* For configuring link layer */
static struct	ifaddrs *ifaddrs;	/* List of address info from getifaddrs() */
static struct	iso_aliasreq iso_addreq;	/* For adding ISO addresses */
static struct	ifreq ridreq;		/* For removing old address */
#ifdef INET6
static struct	in6_ifreq in6_ridreq;	/* For removing old address */
static struct	in6_aliasreq in6_addreq; /* For adding addresses */
#endif /* INET6 */
static struct	iso_ifreq iso_ridreq;	/* For removing old ISO  address */
static u_long	ifmetric;		/* Metric */
static u_int	ifflags_set;		/* Flags to be set */
static u_int	ifflags_reset;		/* Flags to be reset */
static u_int	pifflags_set;		/* PIF flags to be set */
static u_int	pifflags_reset;		/* PIF flags to be reset */
static u_int	nsellength = 1;		/* Length of ISO selector */
static int	mflags_set;		/* Media flags to be set */
static int	mflags_clr;		/* Media flags to be reset */
static int	cur_mtype;		/* Current media type to set */
static u_int	cur_ltype;		/* Current media type to set */
					/* shadows dlreq.ifr_addr.sdl_type */
static int	minst;			/* Media instance */
static int	mflag = 0;		/* Display all media possiblities */

static int	s = -1;		/* for address changes */
static struct	pifreq pifreq;
static char	*pifname_add, *pifname_rm;
static char	*interfacename;

/* Parameters set */
enum {
	PS_ADDR =         0x01,	/* Address has been set */
	PS_DSTADDR =      0x02,	/* Destination address has been set */
	PS_BROADADDR =    0x04,	/* Broadcast address has been set */
	PS_NETMASK =      0x08,	/* Network Mask has been set */
	PS_METRIC =       0x10,	/* Metric has been set */
	PS_LINKTYPE =     0x20,	/* Link type has been set */
	PS_IPDST =        0x40,	/* IP Destination (for EON) has been set */
	PS_NSELLENGTH =   0x80,	/* NSEL length has been set (AF_ISO) */
	PS_SNPAOFFSET = 0x0100,	/* SNPA offset has been set (AF_ISO) */
	PS_ADDRMOD =    0x0200,	/* An address is being modified */
	PS_ADD =        0x0400,	/* Add an address */
	PS_REMOVE =     0x0800,	/* Remove an address */
	PS_MODIFY =     0x1000,	/* Modify an address */
	PS_ALIAS =      0x2000,	/* Add or change and alias */
	PS_DELETE =     0x4000,	/* Delete an alias */
#define	PS_ADDRATTR	(PS_ADDR | PS_DSTADDR | PS_BROADADDR | \
			 PS_NETMASK | PS_NSELLENGTH | PS_SNPAOFFSET)
	PS_MEDIATYPE =	0x8000, /* Change media type selection */
	PS_MEDIAFLAG = 0x10000, /* Media flags modified */
	PS_MEDIAINST = 0x20000, /* Media instance supplied */
#ifdef INET6
	PS_PREFIXLEN = 0x40000, /* Prefixlen specified */
	PS_SCOPEID =   0x80000,	/* Scope ID specified */
	PS_ANYCAST =   0x100000, /* Anycast address flag */
	PS_TENTATIVE = 0x200000, /* Tentative address flag */
	PS_PLTIME =    0x400000, /* Preferred lifetime */
	PS_VLTIME =    0x800000, /* Valid lifetime */
#endif /* INET6 */
} parms_set;

static void	Perror __P((const char *));
static void	adjust_nsellength __P((struct afswitch *));
static void	addpif __P((char *, char *));
static void	dopifflags __P((struct afswitch *, struct cmd *, char *));
static void	media_status_all __P((char *ifname));
static void 	media_status_current __P((char *ifname));
static char    *media_display_word __P((int, int));
static struct	ifaddrs *findaddr __P((const char *, struct sockaddr *));
static void	nonoarp __P((struct afswitch *, struct cmd *, char *));
static void	notrailers __P((struct afswitch *, struct cmd *, char *));
static int	numaddrs __P((const char *, int));
static struct	afswitch *parse_af __P((const char *));
static void	printb __P((const char *, u_int, const char *));
static void	resetifflags __P((struct afswitch *, struct cmd *, char *));
static void	setifaddr __P((struct afswitch *, struct cmd *, char *));
static void	setifbroadaddr __P((struct afswitch *, struct cmd *, char *));
static void	setifdstaddr __P((struct afswitch *, struct cmd *, char *));
static void	setifflags __P((struct afswitch *, struct cmd *, char *));
static void	setifipdst __P((struct afswitch *, struct cmd *, char *));
static void	setifnetmask __P((struct afswitch *, struct cmd *, char *));
static void	setifmetric __P((struct afswitch *, struct cmd *, char *));
static void	setlinktype __P((struct afswitch *, struct cmd *, char *));
static void	setnsellength __P((struct afswitch *, struct cmd *, char *));
static void 	setmedia __P((struct afswitch *, struct cmd *, char *));
static void	setsnpaoffset __P((struct afswitch *, struct cmd *, char *));
static void	status __P((const char *, struct afswitch *));
static void	checkpif __P((struct afswitch *, struct cmd *, char *));
static void	rmpif __P((char *, char *));
static void	usage __P((void));
static int	sa_cmpaddr __P((char *, struct sockaddr *, struct sockaddr *));

#define	SA(x)	((struct sockaddr *)(x))

/* INET */
static void	in_getaddr __P((struct afswitch *, char *, long));
static void	in_status __P((struct ifaddrs *));

#define SIN(x) ((struct sockaddr_in *)(x))

#ifdef INET6
/* INET6 */
static void	in6_getaddr __P((struct afswitch *, char *, long));
static void	in6_setmask __P((struct afswitch *, int));
static void	in6_fillscopeid __P((struct sockaddr_in6 *));
static void	in6_status __P((struct ifaddrs *));
static int	prefixlen __P((void *, size_t));
static void	setifprefixlen __P((struct afswitch *, struct cmd *, char *));
static void	setifscopeid __P((struct afswitch *, struct cmd *, char *));
static void	setip6flags __P((struct afswitch *, struct cmd *, char *));
static void	resetip6flags __P((struct afswitch *, struct cmd *, char *));
static void	setip6lifetime __P((struct afswitch *, struct cmd *, char *));
static int	in6_cmpaddr __P((char *, struct sockaddr *, struct sockaddr *));
static char	*sec2str __P((time_t));

#define SIN6(x) ((struct sockaddr_in6 *)(x))
#endif /* INET6 */

/* ISO */
static void	iso_getaddr __P((struct afswitch *, char *, long));
static void	iso_status __P((struct ifaddrs *));

#define SISO(x) ((struct sockaddr_iso *)(x))

/* Link (physical) */
static void	link_status __P((struct ifaddrs *));
static void	pif_status __P((struct ifaddrs *));

#define	SDL(x) ((struct sockaddr_dl *)(x))

/* XNS */
/*
 * XNS support liberally adapted from code written at the University of
 * Maryland principally by James O'Toole and Chris Torek.
 */

static void	xns_getaddr __P((struct afswitch *, char *, long));
static void	xns_status __P((struct ifaddrs *));

#define SNS(x) ((struct sockaddr_ns *) (x))

#ifdef INET6
static int Lflag = 0;
#endif

static struct cmd cmds[] = {
		/* Flags */
	{ "up",		IFF_UP,		0,		setifflags },
	{ "down",	IFF_UP,		0,		resetifflags },
	{ "trailers",	0,		0,		notrailers },
	{ "-trailers",	0,		0,		notrailers },
	{ "arp",	0,		0,		nonoarp },
	{ "-arp",	0,		0,		nonoarp },
	{ "debug",	IFF_DEBUG,	0,		setifflags },
	{ "-debug",	IFF_DEBUG,	0,		resetifflags },
	{ "link0",	IFF_LINK0,	0,		setifflags } ,
	{ "-link0",	IFF_LINK0,	0,		resetifflags } ,
	{ "link1",	IFF_LINK1,	0,		setifflags } ,
	{ "-link1",	IFF_LINK1,	0,		resetifflags } ,
	{ "link2",	IFF_LINK2,	0,		setifflags } ,
	{ "-link2",	IFF_LINK2,	0,		resetifflags } ,
	{ "metric",	NEXTARG,	PS_METRIC,	setifmetric },

	    /* Address attributes */
	    /* Generic */
	{ "alias",	0,		PS_ADDRMOD|PS_ALIAS },
	{ "-alias",	0,		PS_ADDRMOD|PS_ALIAS|PS_DELETE },
	{ "delete",	0,		PS_ADDRMOD|PS_ALIAS|PS_DELETE },

	{ "add",	NEXTARG,	PS_ADDRMOD|PS_ADD|PS_ADDR,
	    setifaddr },
	{ "modify",	NEXTARG,	PS_ADDRMOD|PS_MODIFY|PS_ADDR,
	    setifaddr },
	{ "remove",	NEXTARG,	PS_ADDRMOD|PS_REMOVE|PS_ADDR,
	    setifaddr },
	{ "-remove",	0,		PS_ADDRMOD|PS_REMOVE },
	{ "set",	NEXTARG,	PS_ADDR,
	    setifaddr },

	{ "broadcast",	NEXTARG,	PS_BROADADDR,	setifbroadaddr },
	{ "destination", NEXTARG,	PS_DSTADDR,	setifdstaddr },
	{ "netmask",	NEXTARG,	PS_NETMASK,	setifnetmask },
#ifdef INET6
	{ "prefixlen",	NEXTARG,	PS_PREFIXLEN,	setifprefixlen },
	{ "scopeid",	NEXTARG,	0,		setifscopeid },
	{ "anycast",	IN6_IFF_ANYCAST,  0,		setip6flags },
	{ "-anycast",	IN6_IFF_ANYCAST, 0,		resetip6flags },
	{ "autoconf",	IN6_IFF_AUTOCONF, 0, 0,		setip6flags },
	{ "-autoconf",	IN6_IFF_AUTOCONF, 0, 0,		resetip6flags },
	{ "tentative",	IN6_IFF_TENTATIVE, 0,		setip6flags },
	{ "-tentative",	IN6_IFF_TENTATIVE, 0,		resetip6flags },
	{ "deprecated",	IN6_IFF_DEPRECATED, 0,		setip6flags },
	{ "-deprecated", IN6_IFF_DEPRECATED, 0,		resetip6flags },
	{ "pltime",	NEXTARG,	PS_PLTIME,	setip6lifetime },
	{ "vltime",	NEXTARG,	PS_VLTIME,	setip6lifetime },
#endif /* INET6 */

	    /* XNS specific */
	{ "ipdst",	NEXTARG,	PS_IPDST,
	    setifipdst,	AF_NS },

	    /* ISO specific */
	{ "snpaoffset",	NEXTARG,	PS_SNPAOFFSET,
	    setsnpaoffset,	AF_ISO },
	{ "nsellength",	NEXTARG,	PS_NSELLENGTH,
	    setnsellength,	AF_ISO },

	    /* Link parameters */
	{ "linktype",	NEXTARG,	PS_LINKTYPE,	setlinktype },

	{ "media",	NEXTARG,	0,		setmedia },

	    /* PIF releated keywords */
	{ "pif",	NEXTARG,	0,		checkpif },
	{ "-pif",	NEXTARG,	0,		checkpif },
	{ "pifflags",	NEXTARG,	0,		dopifflags } ,


/* These two must be in this order and at the end of the list */	    
	{ NULL,		0,		PS_ADDR,	setifaddr },
	{ "destination address",
	    0,		PS_DSTADDR,	setifdstaddr }
};

/* Arguments for the pifflags keyword */
struct pifflags_cmd pifflags_cmd[] = {
	/* First, the enumerated values */
	{ "first-idle",		1, PIF_MASK,		PIF_FIRST_IDLE },
	{ "next-idle",		1, PIF_MASK,		PIF_NEXT_IDLE },
	{ "first-up",		1, PIF_MASK,		PIF_FIRST_UP },
	{ "next-up",		1, PIF_MASK,		PIF_NEXT_UP },

	/* Next, the individual bits for PIF_MASK */
	{ "rescan",		1, PIF_RESCAN,		PIF_RESCAN },
	{ "-rescan",		0, PIF_RESCAN,		0 },
	{ "ignore-oactive",	1, PIF_IGNORE_OACTIVE,	PIF_IGNORE_OACTIVE },
	{ "-ignore-oactive",	0, PIF_IGNORE_OACTIVE,	0 },
	{ "reserved1",		1, PIF_RESERVED1,	PIF_RESERVED1 },
	{ "-reserved1",		0, PIF_RESERVED1,	0 },
	{ "reserved2",		1, PIF_RESERVED2,	PIF_RESERVED2 },
	{ "-reserved2",		0, PIF_RESERVED2,	0 },
	{ "reserved3",		1, PIF_RESERVED3,	PIF_RESERVED3 },
	{ "-reserved3",		0, PIF_RESERVED3,	0 },

	{ "default",		0, ~0,			0, },
	{ NULL,			0, ~0,			0, }
};


static struct afswitch afs[] = {
	/* First entry is default address family */    
	{ "inet", AF_INET,
	  { &addreq.ifra_addr,
	    &addreq.ifra_mask,
	    &addreq.ifra_broadaddr },
	  in_status,	in_getaddr, sa_cmpaddr,
	  SIOCDIFADDR,	SIOCAIFADDR, 
	  &ridreq,	&addreq },
#ifdef INET6
	{ "inet6", AF_INET6,
	  { (struct sockaddr *)&in6_addreq.ifra_addr,
	    (struct sockaddr *)&in6_addreq.ifra_prefixmask,
	    (struct sockaddr *)&in6_addreq.ifra_broadaddr },
	  in6_status,	in6_getaddr, in6_cmpaddr,
	  SIOCDIFADDR_IN6,	SIOCAIFADDR_IN6,
	  (struct ifreq *)&in6_ridreq,
	  (struct ifaliasreq *)&in6_addreq },
#endif /* INET6 */
	{ "ns", AF_NS,
	  { &addreq.ifra_addr,
	    &addreq.ifra_mask,
	    &addreq.ifra_broadaddr },
	  xns_status,	xns_getaddr, sa_cmpaddr,
	  SIOCDIFADDR,	SIOCAIFADDR,
	  &ridreq,	&addreq },
	{ "iso", AF_ISO,
	  { SA(&iso_addreq.ifra_addr),
	    SA(&iso_addreq.ifra_mask),
	    SA(&iso_addreq.ifra_broadaddr) },
	  iso_status,	iso_getaddr, sa_cmpaddr,
	  SIOCDIFADDR_ISO, SIOCAIFADDR_ISO,
	  (struct ifreq *)&iso_ridreq,
	  (struct ifaliasreq *)&iso_addreq },
	{ "link",	AF_LINK,	{ NULL },	link_status },
	{ NULL },
};

static struct llswitch lls[] = {
    { "none", IFT_NONE, },
    { "other", IFT_OTHER, },
    { "1822", IFT_1822, },
    { "hdh1822", IFT_HDH1822, },
    { "x25ddn", IFT_X25DDN, },
    { "x25", IFT_X25, },
    { "ether", IFT_ETHER, },
    { "iso88023", IFT_ISO88023, },
    { "token_bus", IFT_ISO88024, },
    { "iso88024", IFT_ISO88024, },
    { "token_ring", IFT_ISO88025, },
    { "iso88025", IFT_ISO88025, },
    { "iso88026", IFT_ISO88026, },
    { "starlan", IFT_STARLAN, },
    { "pronet10", IFT_P10, },
    { "p10", IFT_P10, },
    { "pronet80", IFT_P80, },
    { "p80", IFT_P80, },
    { "hy", IFT_HY, },
    { "fddi", IFT_FDDI, },
    { "lapb", IFT_LAPB, },
    { "sdlc", IFT_SDLC, },
    { "t1", IFT_T1, },
    { "cept", IFT_CEPT, },
    { "isdnbasic", IFT_ISDNBASIC, },
    { "isdnprimary", IFT_ISDNPRIMARY, },
    { "chdlc", IFT_PTPSERIAL, },
    { "ptpserial", IFT_PTPSERIAL, },
    { "ppp", IFT_PPP, },
    { "loop", IFT_LOOP, },
    { "eon", IFT_EON, },
    { "xether", IFT_XETHER, },
    { "nsip", IFT_NSIP, },
    { "slip", IFT_SLIP, },
    { "ultra", IFT_ULTRA, },
    { "ds3", IFT_DS3, },
    { "sip", IFT_SIP, },
    { "frelay", IFT_FRELAY, },
    { "fr", IFT_FRELAY, },
    { "rs232", IFT_RS232, },
    { "para", IFT_PARA, },
    { "arcnet", IFT_ARCNET, },
    { "arcnetplus", IFT_ARCNETPLUS, },
    { "atm", IFT_ATM, },
    { "miox25", IFT_MIOX25, },
    { "sonet", IFT_SONET, },
    { "x25ple", IFT_X25PLE, },
    { "iso88022llc", IFT_ISO88022LLC, },
    { "localtalk", IFT_LOCALTALK, },
    { "smdsdxi", IFT_SMDSDXI, },
    { "frelaydce", IFT_FRELAYDCE, },
    { "v35", IFT_V35, },
    { "hssi", IFT_HSSI, },
    { "hippi", IFT_HIPPI, },
    { "modem", IFT_MODEM, },
    { "aal5", IFT_AAL5, },
    { "sonetpath", IFT_SONETPATH, },
    { "sonetvt", IFT_SONETVT, },
    { "smdsicip", IFT_SMDSICIP, },
    { "propvirtual", IFT_PROPVIRTUAL, },
    { "propmux", IFT_PROPMUX, },
    { 0, },
};

int
main(argc, argv)
	int argc;
	char *argv[];
{
	struct afswitch *afp;
	struct ifreq ifr;
	int aflag, ch, add_ifaddr, del_ifaddr, nsip_route, vflag;
	int flags;
	char *ifname;

	add_ifaddr = del_ifaddr = nsip_route = 0;

	/* Parse flags */
	aflag = vflag = 0;
	while ((ch = getopt(argc, argv, "amv"
#ifdef INET6
					"L"
#endif
			)) != EOF) {
		switch (ch) {
		case 'a':
			aflag++;
			break;

		case 'm':
			mflag++;
			break;

		case 'v':
			vflag++;
			break;

#ifdef INET6
		case 'L':
			Lflag++;
			break;
#endif

		default:
			usage();
			break;
		}
	}
	argc -= optind;
	argv += optind;

	/* Get current interface configuration */
	if (getifaddrs(&ifaddrs) < 0)
		err(1, "getifaddrs");

	/* Aflag means print status about all interfaces */
	if (aflag) {
		if (argc > 1)
			usage();
		if (argc == 1) {
			afp = parse_af(*argv);
			if (afp == NULL)
				usage();
		} else
			afp = NULL;

		/* Open a socket for media queries */
		if ((s = socket((afp ? afp : afs)->af_af, SOCK_DGRAM, 0)) < 0)
			err(1, "ifconfig: %s socket", afs->af_name);

		status(NULL, afp);
		exit(0);
	}

	/* We need at least an interface name */
	if (argc < 1)
		usage();

	/* Save interface name and make sure it is valid */
	ifname = *argv++;
	interfacename = ifname;	/* make a global copy for later use */
	argc--;
	if (numaddrs(ifname, AF_LINK) == 0)
		errx(1, "no such interface: %s", ifname);

	/* Check for address family */
	if (argc > 0) {
		afp = parse_af(*argv);
		if (afp != NULL) {
			argv++;
			argc--;
		}
	} else
		afp = NULL;

	/* Open a socket for the specified family */
	if ((s = socket((afp ? afp : afs)->af_af, SOCK_DGRAM, 0)) < 0)
		err(1, "ifconfig: %s socket", afp->af_name);

	/* No more args means a status request */
	if (argc == 0) {
		status(ifname, afp);
		exit(0);
	}

	/* Default to first address family (INET) */
	if (afp == NULL)
		afp = afs;

#ifdef INET6
	/* initialization */
	in6_addreq.ifra_lifetime.ia6t_vltime = ND6_INFINITE_LIFETIME;
	in6_addreq.ifra_lifetime.ia6t_pltime = ND6_INFINITE_LIFETIME;
#endif

	/*
	 * Parse the supplied arguments.
	 */
	while (argc > 0) {
		struct cmd *p;

		/* Lookup the current arg as a command */
		for (p = cmds; p->c_name != NULL; p++)
			if (strcasecmp(*argv, p->c_name) == 0)
				break;

		/*
		 * Address and destination address are the last
		 * elements in the list.  If we did not match a
		 * command name, assume it is the address, unless we
		 * already have an address, then it is the destination
		 * address.
		 */
		if (p->c_name == NULL && (parms_set & p->c_flag))
			p++;


		/* Supress duplicate and conflicting parameters */
		if (p->c_flag && (parms_set & p->c_flag)) {
			if (p->c_parameter == NEXTARG && argc > 1)
				errx(1, "conflicting or duplicate parameter: "
				    "%s %s",
				    *argv, argv[1]);
			else
				errx(1, "conflicting or duplicate parameter: "
				    "%s",
				    *argv);
		} else
			parms_set |= p->c_flag;

		/* Verify that this option is valid for this address */
		/* family */
		if (p->c_af != AF_UNSPEC && p->c_af != afp->af_af)
			errx(1, "'%s' not valid for %s address",
			    p->c_name == NULL ? "address" : p->c_name,
			    afp->af_name);

		if (p->c_func) {
			/* If required, fetch the next argument */
			if (p->c_parameter == NEXTARG) {
				if (argc == 1)
					errx(1, "'%s' requires argument",
					    p->c_name);
				argc--, argv++;
			}
			(*p->c_func)(afp, p, *argv);
		}
		argc--, argv++;
	}


	/*
	 * Figure out what we need to do based on the supplied info
	 */

	if (parms_set & PS_IPDST) {
		/* XNS in IP encapsulation */
		nsip_route++;
	} else if (parms_set & PS_ALIAS) {
		/* For compatibility, no checking */
		if (parms_set & PS_DELETE) {
			if (parms_set & PS_ADDR)
				memcpy(&afp->af_ridreq->ifr_addr,
				    afp->af_stab[ADDR],
				    afp->af_stab[ADDR]->sa_len);
			else
				afp->af_ridreq->ifr_addr.sa_family 
				    = afp->af_af;
			del_ifaddr++;
		} else {
			add_ifaddr++;
		}
	} else if (parms_set & PS_ADD) {
		/* 
		 * Add an address to an interface.  At least the
		 * address must be specified and it must not already
		 * be configured on the interface.
		 */
		if (!(parms_set & PS_ADDR))
			errx(1, "must specify address to be added");
		if (findaddr(ifname, afp->af_stab[ADDR]) != NULL)
			errx(1, "specified address already configured, "
			    "use modify to change attributes",
			    ifname);
		add_ifaddr++;
	} else if (parms_set & PS_REMOVE) {
		/*
		 * Delete an address from an interface.  If the
		 * address is specified, it must exist.  Otherwise
		 * there must be only one address configured.
		 */
		if (parms_set & PS_ADDR) {
			if (findaddr(ifname, afp->af_stab[ADDR]) == NULL)
				errx(1, "specified address not configured",
				    ifname);
			memcpy(&afp->af_ridreq->ifr_addr, afp->af_stab[ADDR],
			    afp->af_stab[ADDR]->sa_len);
		} else if (numaddrs(ifname, afp->af_af) > 1)
			errx(1, "must specify address to be removed");
		else
			afp->af_ridreq->ifr_addr.sa_family = afp->af_af;
		del_ifaddr++;
	} else if (parms_set & PS_MODIFY) {
		if (!(parms_set & PS_MODIFY))
			errx(1, "must specify address to be modified");
		if (findaddr(ifname, afp->af_stab[ADDR]) == NULL)
			errx(1, "specified address not configured",
			    ifname);
		add_ifaddr++;;
	} else if (parms_set & PS_ADDRATTR) {
		/*
		 * Old style set, only allow if there are not multiple
		 * addresses.
		 */
		if (numaddrs(ifname, afp->af_af) > 1)
			errx(1, "multiple %s addresses configured, "
			    "use add, modify and remove.",
			    ifname,
			    afp->af_name);
		add_ifaddr++;
		del_ifaddr++;
		afp->af_ridreq->ifr_addr.sa_family = afp->af_af;
	}

	/*
	 * Now change the interface configuration!
	 */

	/* Set and/or reset flags */
	if (ifflags_set != 0 || ifflags_reset != 0) {
		strncpy(ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
	 	if (ioctl(s, SIOCGIFFLAGS, &ifr) < 0) {
 			Perror("ioctl (SIOCGIFFLAGS)");
	 		exit(1);
	 	}

		flags = (ifr.ifr_flags & ~ifflags_reset) | ifflags_set;

		if (flags != ifr.ifr_flags) {
			ifr.ifr_flags = flags;
			if (ioctl(s, SIOCSIFFLAGS, &ifr) < 0)
				Perror("setting interface flags");
		}
	}

	/* Set and/or reset pif flags */
	if (pifflags_set || pifflags_reset) {
		bzero(&pifreq, sizeof(pifreq));
		strncpy(pifreq.pifr_name, ifname, sizeof(pifreq.pifr_name));
		if (ioctl(s, PIFIOCGFLG, &pifreq) < 0) {
 			Perror("ioctl (PIFIOCGFLG)");
	 		exit(1);
	 	}

		flags = (pifreq.pifr_flags & ~pifflags_reset) | pifflags_set;

		if (flags != pifreq.pifr_flags) {
			pifreq.pifr_flags = flags;
			if (ioctl(s, PIFIOCSFLG, &pifreq) < 0)
				Perror("setting interface pif flags");
		}
	}

	/* Set the Metric if requested */
	if (parms_set & PS_METRIC) {
		strncpy(ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
		ifr.ifr_metric = ifmetric;
		if (ioctl(s, SIOCSIFMETRIC, &ifr) < 0)
			warn("setting metric");
	}
	
	/* Set the nsellength for all addresses */
	if (afp->af_af == AF_ISO)
		adjust_nsellength(afp);

	/* Set XNS encapsulation routing */
	if (nsip_route) {
		struct nsip_req rq;
		int size = sizeof(rq);

		rq.rq_ns = addreq.ifra_addr;
		rq.rq_ip = addreq.ifra_dstaddr;

		if (setsockopt(s, 0, SO_NSIP_ROUTE, &rq, size) < 0)
			Perror("Encapsulation Routing");
	}

	/* Remove the old interface address */
	if (del_ifaddr) {
		/*
		 * When changing the interface address, we delete an
		 * address that is all zeros, which is shorthand for
		 * the first address on the interface.
		 * When deleting an alias, we need to specify and
		 * actual address.
		 */
		strncpy(afp->af_ridreq->ifr_name, ifname,
		    sizeof afp->af_ridreq->ifr_name);
		if (ioctl(s, afp->af_difaddr, afp->af_ridreq) < 0) {
			if (errno == EADDRNOTAVAIL &&
			    !(parms_set & PS_REMOVE)) {
				/* means no previous address for interface */
			} else
				Perror("ioctl (SIOCDIFADDR)");
		}
	}

	/* Set new link level info */
	if (parms_set & PS_LINKTYPE) {
		struct sockaddr_dl *dl = SDL(&dlreq.ifr_addr);
		int rs;

		if ((rs = socket(AF_LINK, SOCK_RAW, 0)) < 0)
			Perror("socket(SOCK_RAW)");

		dl->sdl_family = AF_LINK;
		dl->sdl_len = sizeof(*dl) - sizeof(dl->sdl_data) +
		    dl->sdl_nlen + dl->sdl_alen + dl->sdl_slen;

		strncpy(dlreq.ifr_name, ifname, sizeof dlreq.ifr_name);
		if (ioctl(rs, SIOCSIFADDR, &dlreq) < 0)
			Perror("ioctl (SIOCSIFADDR) for linktype");
	}

	/* Deal with media changes */
	while (parms_set & (PS_MEDIAFLAG | PS_MEDIAINST | PS_LINKTYPE | 
	    PS_MEDIATYPE)) {
		struct ifmediareq mreq;
		int mword;
		int lt;

		/* Snag current media options word */
		strncpy(mreq.ifm_name, ifname, sizeof (mreq.ifm_name));
		mreq.ifm_count = 0;
		if (ioctl(s, SIOCGIFMEDIA, &mreq) < 0) {
			/* Old/p2p are OK if only link type changed */
			if ((parms_set & (PS_MEDIAFLAG | PS_MEDIAINST | 
			    PS_MEDIATYPE)) == 0) {
				break;
			}
			err(1, "Media operations not supported on interface");
		}
		mword = mreq.ifm_current;
		/* Apply changes */
		if ((parms_set & (PS_MEDIAINST|PS_MEDIATYPE|PS_LINKTYPE)) ||
		    ((parms_set & PS_MEDIAFLAG) &&
		    (mflags_set|mflags_clr) & ~mreq.ifm_mask)) {
			/* 
			 * If we touch any non-global flags clear all the
			 * non global flags first; also do this if changing
			 * media type or link type.
			 */
			mword &= ~IFM_GMASK | mreq.ifm_mask;
		}
		if ((parms_set & PS_LINKTYPE) != 0) {
			/* Convert IFT to IFM */
			switch (cur_ltype) {
			case IFT_ETHER:
				lt = IFM_ETHER;
				break;
			case IFT_ISO88025:
				lt = IFM_TOKEN;
				break;
			case IFT_FDDI:
				lt = IFM_FDDI;
				break;
			default:
				errx(1, "Unsupported link type");
			}
			/* If changed link type must set media type as well */
			if ((parms_set & PS_MEDIATYPE) == 0 &&
			    lt != (mword & IFM_NMASK))
				errx(1, "Must set media type when changing "
				    "link type");
			mword = (mword & ~IFM_NMASK) | lt;
		}
		if (parms_set & PS_MEDIAFLAG) {
			mword |= mflags_set;
			mword &= ~mflags_clr;
		}
		if (parms_set & PS_MEDIAINST)
			mword = (mword & ~IFM_IMASK) | minst << IFM_ISHIFT;
		if (parms_set & PS_MEDIATYPE) {
			switch (cur_mtype) {
			case -1:		/* UTP */
				/* Meaning up UTP depend on link type */
				switch (mword & IFM_NMASK) {
				case IFM_ETHER:
					cur_mtype = IFM_10_T;
					break;
				case IFM_TOKEN:
					cur_mtype = IFM_TOK_UTP16;
					break;
				case IFM_FDDI:
					cur_mtype = IFM_FDDI_UTP;
					break;
				}
				/* Fall through */
			default:
				mword = (mword & ~IFM_TMASK) | cur_mtype;
			}
		}

		/* Attempt to install changed media word */
		if (mword != mreq.ifm_current) {
			strncpy(ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
			ifr.ifr_media = mword;
			if (ioctl(s, SIOCSIFMEDIA, &ifr) < 0) {
				if (errno == ENXIO)
					errx(1,
				"%s: media settings not valid: %s",
				            ifname,
					    media_display_word(mword, 1));
				err(1, "Failed to set media options");
			}
		}
		break;
	}

	/* Set the new interface address (and mask and destination */
	if (add_ifaddr) {
		strncpy(afp->af_addreq->ifra_name, ifname, IFNAMSIZ);
		if (ioctl(s, afp->af_aifaddr, afp->af_addreq) < 0)
			Perror("ioctl (SIOCAIFADDR)");
		else if ((ifflags_reset & IFF_UP) != 0) {
			/*
			 * Configuring an interface address brings it
			 * up, if we have a request to bring it down
			 * we may have to repeat it now.
			 */
			strncpy(ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
		 	if (ioctl(s, SIOCGIFFLAGS, &ifr) < 0) {
 				Perror("ioctl (SIOCGIFFLAGS)");
	 			exit(1);
		 	}

			flags = (ifr.ifr_flags & ~IFF_UP);

			if (flags != ifr.ifr_flags) {
				ifr.ifr_flags = flags;
				if (ioctl(s, SIOCSIFFLAGS, &ifr) < 0)
					Perror("setting interface flags");
		        }
		}
	}
	if (pifname_rm)
		rmpif(ifname, pifname_rm);
	if (pifname_add)
		addpif(ifname, pifname_add);

	if (vflag) {
		/* Get and display the new interface configuration */
		if (getifaddrs(&ifaddrs) < 0)
			err(1, "getifaddrs");
		status(ifname, afp);
	}

	exit(0);
}

static void
usage()
{
	fprintf(stderr,
	    "usage: ifconfig -a [-m%s] [address_family]\n"
	    "       ifconfig [-m%s] interface [address_family]\n"
	    "       ifconfig [-mv%s] interface [address_family] "
	    "[address parameters] [link parameters]\n",
#ifdef INET6
		"L", "L", "L"
#else
		"", "", ""
#endif
		);
	exit(1);
}

static struct afswitch *
parse_af(afname)
	const char *afname;
{
	struct afswitch *afp;
	
	for (afp = afs; afp->af_name != NULL; afp++)
		if (strcmp(afp->af_name, afname) == 0)
			return(afp);

	return(NULL);
}


/*ARGSUSED*/
static void
setifaddr(afp, p, addr)
	struct afswitch *afp;
	struct cmd *p;
	char *addr;
{
	(*afp->af_getaddr)(afp, addr, ADDR);
}


/*ARGSUSED*/
static void
setifdstaddr(afp, p, addr)
	struct afswitch *afp;
	struct cmd *p;
	char *addr;
{
    
	if (parms_set & PS_BROADADDR)
		errx(1, "broadcast address may not be specified "
		    "when destination address is specified");
	(*afp->af_getaddr)(afp, addr, DSTADDR);
}

static void
setifnetmask(afp, p, addr)
	struct afswitch *afp;
	struct cmd *p;
	char *addr;
{
    
	(*afp->af_getaddr)(afp, addr, MASK);
}

#ifdef INET6
static void
setifprefixlen(afp, p, arg)
	struct afswitch *afp;
	struct cmd *p;
	char *arg;
{
	unsigned long plen;
	char *ep;

	plen = strtoul(arg, &ep, 10);
	if (arg == ep)
		errx(1, "invalid prefixlen");
	if (afp->af_af == AF_INET6) {
		struct sockaddr_in6 *sin6p;
		if (plen < 0 || 128 < plen)
			errx(1, "invalid prefixlen for %s", afp->af_name);
		sin6p = (struct sockaddr_in6 *)afp->af_stab[MASK];
		in6_setmask(afp, (int)plen);
	}
#if 0
	else if (afp->af_af == AF_INET) {
		struct sockaddr_in *sinp;
		if (plen < 0 || 32 < plen)
			errx(1, "invalid prefixlen for %s", afp->af_name);
		sinp = (struct sockaddr_in *)afp->af_stab[MASK];
		prefixlen(&sinp->sin_addr, (int)plen);
	}
#endif
	else
		errx(1, "prefixlen not allowed for the AF");
}

static void
setifscopeid(afp, p, arg)
	struct afswitch *afp;
	struct cmd *p;
	char *arg;
{
	u_short scopeid;
	char *ep;

	scopeid = strtoul(arg, &ep, 0) & 0xffff;
	if (arg == ep)
		errx(1, "invalid scopeid");
	if (afp->af_af == AF_INET6) {
		struct sockaddr_in6 *sin6p;
		sin6p = (struct sockaddr_in6 *)afp->af_stab[ADDR];
		if (!IN6_IS_ADDR_SITELOCAL(&sin6p->sin6_addr))
			errx(1, "scopeid allowed only for sitelocal address");
		sin6p->sin6_scope_id = scopeid;
	} else
		errx(1, "scopeid not allowed for the AF");
}

static void
setip6flags(afp, p, arg)
	struct afswitch *afp;
	struct cmd *p;
	char *arg;
{
	if (afp->af_af != AF_INET6)
		errx(1, "address flags can be only set for inet6 addresses");

	in6_addreq.ifra_flags |= p->c_parameter;
}

static void
resetip6flags(afp, p, arg)
	struct afswitch *afp;
	struct cmd *p;
	char *arg;
{
	if (afp->af_af != AF_INET6)
		errx(1, "address flags can be only set for inet6 addresses");

	in6_addreq.ifra_flags &= ~p->c_parameter;
}

static void
setip6lifetime(afp, p, arg)
	struct afswitch *afp;
	struct cmd *p;
	char *arg;
{
	time_t newval, t;
	char *ep;

	t = time(NULL);
	newval = (time_t)strtoul(arg, &ep, 0);
	if (arg == ep)
		errx(1, "invalid %s", p->c_name);
	if (afp->af_af != AF_INET6)
		errx(1, "%s not allowed for the AF", p->c_name);
	if (p->c_flag & PS_VLTIME) {
		in6_addreq.ifra_lifetime.ia6t_expire = t + newval;
		in6_addreq.ifra_lifetime.ia6t_vltime = newval;
	} else if (p->c_flag & PS_PLTIME) {
		in6_addreq.ifra_lifetime.ia6t_preferred = t + newval;
		in6_addreq.ifra_lifetime.ia6t_pltime = newval;
	}
}

#endif /* INET6 */

static void
setifbroadaddr(afp, p, addr)
	struct afswitch *afp;
	struct cmd *p;
	char *addr;
{

	if (parms_set & PS_DSTADDR)
		errx(1, "broadcast address may not be specified "
		    "when destination address is specified");
	(*afp->af_getaddr)(afp, addr, DSTADDR);
}

static void
setifipdst(afp, p, addr)
	struct afswitch *afp;
	struct cmd *p;
	char *addr;
{

	/*
	 * We rely on the fact that IP and XNS use the same request
	 * structures.
	 */
	in_getaddr(afp, addr, DSTADDR);
}

static void
setlinktype(afp, p, type)
	struct afswitch *afp;
	struct cmd *p;
	char *type;
{
	struct llswitch *ll;
	struct sockaddr_dl *dl;

	for (ll = lls; ll->ll_name != NULL; ll++)
		if (strcasecmp(type, ll->ll_name) == 0)
			break;

	if (ll->ll_name == NULL)
		errx(1, "%s: invalid link type", type);

	dl = SDL(&dlreq.ifr_addr);
	dl->sdl_type = ll->ll_type;
	if (cur_ltype && cur_ltype != ll->ll_type)
		errx(1, "%s: link type in conflict with earlier args", type);
	cur_ltype = ll->ll_type;
}

/*ARGSUSED*/
static void
nonoarp(afp, p, param)
	struct afswitch *afp;
	struct cmd *p;
	char *param;
{
	static int fence = 0;

	if (fence++ == 0)
		warnx("ARP is no longer optional: %s", param);
}

/*ARGSUSED*/
static void
notrailers(afp, p, param)
	struct afswitch *afp;
	struct cmd *p;
	char *param;
{
	static int fence = 0;

	if (fence++ == 0)
		warnx("the sending of trailers is no longer supported: %s",
		      param);
}

static void
resetifflags(afp, p, unused)
	struct afswitch *afp;
	struct cmd *p;
	char *unused;
{

	if (ifflags_set & p->c_parameter)
		errx(1, "flag conflict: %s", p->c_name);
	ifflags_reset |= p->c_parameter;
}

static void
setifflags(afp, p, unused)
	struct afswitch *afp;
	struct cmd *p;
	char *unused;
{

	if (ifflags_reset & p->c_parameter)
		errx(1, "flag conflict: %s", p->c_name);
	ifflags_set |= p->c_parameter;
}

static void
setifmetric(afp, p, metric)
	struct afswitch *afp;
	struct cmd *p;
	char *metric;
{
	char *ep;

	errno = 0;
	ifmetric = strtoul(metric, &ep, 10);
	if (*metric == 0 || *ep != 0 ||
	   (ifmetric == ULONG_MAX && errno == EINVAL))
		errx(1, "invalid metric: %s", metric);
}

/*ARGSUSED*/
static void
checkpif(afp, p, pif)
	struct afswitch *afp;
	struct cmd *p;
	char *pif;
{
	if (*p->c_name == '-') {
		if (pifname_rm)
			errx(1, "only one -pif command allowed\n");
		else
			pifname_rm = pif;
	} else {
		if (pifname_add)
			errx(1, "only one pif command allowed\n");
		else
			pifname_add = pif;
	}
}

static void
addpif(ifname, pif)
	char *ifname;
	char *pif;
{
	bzero(&pifreq, sizeof(pifreq));
	strncpy(pifreq.pifr_name, pif, sizeof(pifreq.pifr_name));
	strncpy(pifreq.pifr_subname, ifname, sizeof(pifreq.pifr_subname));
	if (ioctl(s, PIFIOCADD, &pifreq) < 0) {
		switch(errno) {
		case EBUSY:
			errx(1, "%s: already linked to a pif", ifname);
		case EOPNOTSUPP:
			errx(1, "%s: not supported for PIF", ifname);
		default:
			Perror("ioctl (PIFIOCADD)");
		}
	}
}

static void
rmpif(ifname, pif)
	char *ifname;
	char *pif;
{
	bzero(&pifreq, sizeof(pifreq));
	strncpy(pifreq.pifr_name, pif, sizeof(pifreq.pifr_name));
	strncpy(pifreq.pifr_subname, ifname, sizeof(pifreq.pifr_subname));
	if (ioctl(s, PIFIOCDEL, &pifreq) < 0) {
		switch(errno) {
		case ENXIO:
			errx(1, "%s is not linked to %s", ifname, pif);
		default:
			Perror("ioctl(PIFIOCDEL)");
		}
	}
}

static void
dopifflags(afp, p, valstr)
	struct afswitch *afp;
	struct cmd *p;
	char *valstr;
{
	struct pifflags_cmd *pf;
	u_long set;
	char *ep, *val;

	while ((val = strsep(&valstr, ",")) != NULL) {
		for (pf = pifflags_cmd; pf->pf_name; pf++)
			if (strcasecmp(val, pf->pf_name) == 0)
				break;

		if (pf->pf_name == NULL) {
			set = strtoul(val, &ep, 0);
			if (*val == 0 || *ep != 0 ||
			    (set == ULONG_MAX && errno == EINVAL))
				errx(1, "%s: invalid argument for keyword %s",
				    val, p->c_name);
		} else
			set = pf->pf_set;

		if ((pifflags_reset | pifflags_set) & (pf->pf_reset | set))
			errx(1, "pifflags conflict: %s", val);

		pifflags_reset |= pf->pf_reset;
		pifflags_set |= set;
	}
}

#define	IFFBITS \
"\020\1UP\2BROADCAST\3DEBUG\4LOOPBACK\5POINTOPOINT\6NOTRAILERS\7RUNNING\10NOARP\
\11PROMISC\12ALLMULTI\13OACTIVE\14SIMPLEX\15LINK0\16LINK1\17LINK2\20MULTICAST"

/*
 * Print a value a la the %b format of the kernel's printf
 */
static void
printb(label, v, bits)
	const char *label;
	u_int v;
	const char *bits;
{
	int any, i;
	char c;

	any = 0;
	if (bits && *bits == 8)
		printf("%s=%o", label, v);
	else
		printf("%s=%x", label, v);
	bits++;
	if (bits) {
		printf("<");
		while ((i = *bits++)) {
			if ((v & (1 << (i-1)))) {
				if (any)
					printf(",");
				any = 1;
				for (; (c = *bits) > 32; bits++)
					printf("%c", c);
			} else
				for (; *bits > 32; bits++)
					;
		}
		printf(">");
	}
}

/*
 * Count the number of interface addresses in the given address family
 * configured on the given interface.
 */
static int
numaddrs(ifn, af)
	const char *ifn;
	int af;
{
	struct ifaddrs *ifa;
	int num_addrs;

	num_addrs = 0;
	for (ifa = ifaddrs; ifa != NULL; ifa = ifa->ifa_next)
		if ((ifn == NULL || strcmp(ifa->ifa_name, ifn) == 0) &&
		    ifa->ifa_addr->sa_family == af)
			num_addrs++;

	return(num_addrs);
}

/*
 * Find the specified address on the given interface.
 */
static struct ifaddrs *
findaddr(ifn, sa)
	const char *ifn;
	struct sockaddr *sa;
{
	struct ifaddrs *ifa;

	for (ifa = ifaddrs; ifa != NULL; ifa = ifa->ifa_next)
		if ((ifn == NULL || strcmp(ifa->ifa_name, ifn) == 0) &&
		    ifa->ifa_addr->sa_family == sa->sa_family &&
		    ifa->ifa_addr->sa_len == sa->sa_len) {
			struct afswitch *afp;

			for (afp = afs; afp->af_name != NULL; afp++)
				if (afp->af_af == ifa->ifa_addr->sa_family &&
				    (*afp->af_cmpaddr)(ifa->ifa_name,
						       ifa->ifa_addr, sa))
					goto find;
		}

  find:
	return(ifa);
}

/*
 * Print the status of the interface.  If an address family was
 * specified, show it and it only; otherwise, show them all.
 */
static void
status(ifn, afp)
	const char *ifn;
	struct afswitch *afp;
{
	struct ifaddrs *ifa, *link;
	struct afswitch *p;

	link = NULL;
	
	/* Find the first, last and linkp entries for this interface */
	for (ifa = ifaddrs; ifa != NULL; ifa = ifa->ifa_next) {
		if (ifn != NULL && strcmp(ifa->ifa_name, ifn) != 0)
			continue;

		if (afp == NULL) {
			if (ifa->ifa_addr->sa_family == AF_LINK) {
				link_status(ifa);
				continue;
			}
			for (p = afs; p->af_name != NULL; p++)
				if (p->af_af == ifa->ifa_addr->sa_family) {
					p->af_status(ifa);
					break;
				}
		} else {
			if (ifa->ifa_addr->sa_family == AF_LINK)
				link = ifa;
			if (afp->af_af == ifa->ifa_addr->sa_family) {
				if (afp->af_af != AF_LINK && link != NULL) {
					link_status(link);
					link = NULL;
				}
				afp->af_status(ifa);
			}
		}
	}
}

static void
link_status(ifa)
	struct ifaddrs *ifa;
{
	struct if_data *ifd;
	struct llswitch *ll;
	struct sockaddr_dl *sdl;
	double baudrate;
	char *multi;
	u_char *cp, *lp;

	sdl = SDL(ifa->ifa_addr);

	printf("%.*s: ", sdl->sdl_nlen, sdl->sdl_data);
	printb("flags", (u_short)ifa->ifa_flags, IFFBITS);

	printf("\n\tlink type ");
	
	for (ll = lls; ll->ll_name != NULL; ll++)
		if (ll->ll_type == sdl->sdl_type)
			break;

	if (ll->ll_name != NULL)
		printf("%s", ll->ll_name);
	else
		printf("%u", sdl->sdl_type);

	if (sdl->sdl_alen > 0) {	
		for (cp = (u_char *)LLADDR(sdl), lp = cp + sdl->sdl_alen;
		     cp < lp; cp++)
			printf("%c%x",
			    cp == (u_char *)LLADDR(sdl) ? ' ' : ':', *cp);
	}
	if ((ifd = ifa->ifa_data) == NULL)
		warnx("no if_data");
	else {
		if (ifd->ifi_metric)
			printf(" metric %lu", ifd->ifi_metric);
		if (ifd->ifi_mtu)
			printf(" mtu %lu", ifd->ifi_mtu);
		if ((baudrate = (double)ifd->ifi_baudrate)) {
			if (baudrate > 1000000000.0) {
				baudrate /= 1000000000.0;
				multi = "G";
			} else if (baudrate > 1000000.0) {
				baudrate /= 1000000.0;
				multi = "M";
			} else if (baudrate > 1000.0) {
				baudrate /= 1000.0;
				multi = "k";
			} else
				multi = "";
			printf(" speed %g%sbps", baudrate, multi);
		}
	}
	printf("\n");
	media_status_current(ifa->ifa_name);
	if (mflag)
		media_status_all(ifa->ifa_name);

	pif_status(ifa);
}

void
pif_status(ifa)
	struct ifaddrs *ifa;
{
	struct pifflags_cmd *pf;
	u_long reset;

	bzero(&pifreq, sizeof(pifreq));
	strncpy(pifreq.pifr_name, "pif0", sizeof(pifreq.pifr_name));
	strncpy(pifreq.pifr_subname, ifa->ifa_name,
		    sizeof(pifreq.pifr_subname));
	if (ioctl(s, PIFIOCGPIF, &pifreq) == 0)
		printf("\tattached to %s\n", pifreq.pifr_name);

	bzero(&pifreq, sizeof(pifreq));
	strncpy(pifreq.pifr_name, ifa->ifa_name, sizeof(pifreq.pifr_name));
	if (ioctl(s, PIFIOCGFLG, &pifreq) < 0)
		return;
	printf("\tpifflags");
	for (reset = 0, pf = pifflags_cmd; pf->pf_name; pf++) {
		if (pf->pf_report && (reset & pf->pf_reset) == 0 &&
		    (pifreq.pifr_flags & pf->pf_reset) == pf->pf_set) {
			printf(reset ? ",%s" : " %s", pf->pf_name);
			reset |= pf->pf_reset;
		}
	}
	if ((pifreq.pifr_flags &= ~reset) != 0)
		printf(" otherflags=0x%x", pifreq.pifr_flags);
	printf("\n");
}

static void
Perror(cmd)
	const char *cmd;
{

	switch (errno) {
	case ENXIO:
		errx(1, "%s: no such interface", cmd);
		break;

	case EPERM:
		errx(1, "%s: permission denied", cmd);
		break;

	default:
		err(1, "%s", cmd);
	}
}

/* INET family support */

static void
in_getaddr(afp, addr, which)
	struct afswitch *afp;
	char *addr;
	long which;
{
	struct sockaddr_in *sinp;
	struct sockaddr_in *maskp;
	struct hostent *hp;
	struct netent *np;
	u_long mask, mlen;
	char *mp, *ep;

	sinp = SIN(afp->af_stab[which]);
	sinp->sin_len = sizeof(*sinp);
	if (which != MASK)
		sinp->sin_family = AF_INET;

	/* For the primary address we recognize address/mask-length */
	if (which == ADDR && (mp = strchr(addr, '/'))) {
		*mp++ = 0;
		errno = 0;
		mlen = strtoul(mp, &ep, 10);
		if (*mp == 0 || *ep != 0 ||
		   (mlen == ULONG_MAX && errno == EINVAL) || mlen > 32)
			errx(1, "invalid mask length: %s", mp);
		if (mlen > 0) {
			mask = 0x80000000;
			while (--mlen)
				mask |= mask >> 1;
		} else
			mask = 0;
		maskp = SIN(afp->af_stab[MASK]);
		maskp->sin_addr.s_addr = htonl(mask);
		maskp->sin_len = sizeof(*sinp);
		parms_set |= PS_NETMASK;
	}

	if ((hp = gethostbyname(addr)) != NULL) {
		if (hp->h_addr_list[1] != NULL)
			errx(1, "'%s' resolves to multiple addresses", addr);
		memcpy(&sinp->sin_addr, hp->h_addr, hp->h_length);
	} else if ((np = getnetbyname(addr)) != NULL)
		sinp->sin_addr = inet_makeaddr(np->n_net, INADDR_ANY);
	else
		errx(1, "%s: bad value", addr);
}

static void
in_status(ifa)
	struct ifaddrs *ifa;
{
	struct sockaddr_in *sinp;

	sinp = SIN(ifa->ifa_addr);
	printf("\tinet %s ", inet_ntoa(sinp->sin_addr));
	if (ifa->ifa_flags & IFF_POINTOPOINT) {
		sinp = (struct sockaddr_in *)ifa->ifa_dstaddr;
		printf("--> %s ", inet_ntoa(sinp->sin_addr));
	}
	sinp = SIN(ifa->ifa_netmask);
	printf("netmask %s ", inet_ntoa(sinp->sin_addr));
	if (ifa->ifa_flags & IFF_BROADCAST) {
		sinp = SIN(ifa->ifa_broadaddr);
		printf("broadcast %s", inet_ntoa(sinp->sin_addr));
	}
	printf("\n");
}

#ifdef INET6
/* INET6 family support */

static void
in6_getaddr(afp, addr, which)
	struct afswitch *afp;
	char *addr;
	long which;
{
	struct sockaddr_in6 *sin6p;
#if defined(__KAME__) && defined(KAME_SCOPEID)
	struct addrinfo hints, *res;
	int error;
#else
	struct hostent *hp;
	struct netent *np;
#endif
	int mlen;
	char *mp, *ep;

	sin6p = SIN6(afp->af_stab[which]);
	sin6p->sin6_len = sizeof(*sin6p);
	if (which != MASK)
		sin6p->sin6_family = AF_INET6;

	/*
	 * IPv6 aggregatable address architecture support:
	 * set netmask to 64 by default
	 */
	in6_setmask(afp, 64);

	/* For the primary address we recognize address/mask-length */
	if (which == ADDR && (mp = strchr(addr, '/'))) {
		*mp++ = 0;
		errno = 0;
		mlen = strtoul(mp, &ep, 10);
		if (*mp == 0 || *ep != 0 ||
		   (mlen == ULONG_MAX && errno == EINVAL) ||
		   mlen > 8 * sizeof(struct in6_addr))
			errx(1, "invalid mask length: %s", mp);
		in6_setmask(afp, mlen);
	}

#if defined(__KAME__) && defined(KAME_SCOPEID)
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_INET6;
	hints.ai_socktype = SOCK_DGRAM;
#if 0
	hints.ai_flags = AI_NUMERICHOST;
#endif
	error = getaddrinfo(addr, "0", &hints, &res);
	if (error)
		errx(1, "%s: %s", addr, gai_strerror(error));
	if (res->ai_next)
		errx(1, "%s: resolved to multiple hosts", addr);
	if (res->ai_addrlen != sizeof(struct sockaddr_in6))
		errx(1, "%s: bad value", addr);
	memcpy(sin6p, res->ai_addr, res->ai_addrlen);
	freeaddrinfo(res);
	if (IN6_IS_ADDR_LINKLOCAL(&sin6p->sin6_addr) && sin6p->sin6_scope_id) {
		*(u_int16_t *)&sin6p->sin6_addr.s6_addr[2] =
			htons(sin6p->sin6_scope_id);
		sin6p->sin6_scope_id = 0;
	}
#else
	if ((hp = gethostbyname2(addr, AF_INET6)) != NULL) {
		if (hp->h_addr_list[1] != NULL)
			errx(1, "'%s' resolves to multiple addresses", addr);
		memcpy(&sin6p->sin6_addr, hp->h_addr, hp->h_length);
	} else if (inet_pton(AF_INET6, addr, &sin6p->sin6_addr) == 1)
		; /* okay */
	else
		errx(1, "in6_getaddr: %s: bad value", addr);
#endif
}

static void
in6_setmask(afp, prefixlen)
	struct afswitch *afp;
	int prefixlen;
{
	struct sockaddr_in6 *mask6p;
	struct in6_addr mask;
	u_char *p;

	if (prefixlen < 0 || sizeof(mask) * 8 < prefixlen)
		errx(1, "%d: bad value", prefixlen);

	p = (u_char *)&mask;
	memset(&mask, 0, sizeof(mask));
	for ( ; 8 <= prefixlen; prefixlen -= 8)
		*p++ = 0xff;
	if (prefixlen)
		*p = (0xff00 >> prefixlen) & 0xff;
	mask6p = SIN6(afp->af_stab[MASK]);
	mask6p->sin6_addr = mask;
	mask6p->sin6_len = sizeof(*mask6p);
	parms_set |= PS_NETMASK;
}

void
in6_fillscopeid(sin6)
	struct sockaddr_in6 *sin6;
{
#if defined(__KAME__) && defined(KAME_SCOPEID)
	if (IN6_IS_ADDR_LINKLOCAL(&sin6->sin6_addr)) {
		sin6->sin6_scope_id =
			ntohs(*(u_int16_t *)&sin6->sin6_addr.s6_addr[2]);
		sin6->sin6_addr.s6_addr[2] = sin6->sin6_addr.s6_addr[3] = 0;
	}
#endif
}

static void
in6_status(ifa)
	struct ifaddrs *ifa;
{
	struct sockaddr_in6 *sin6p;
	struct in6_ifreq ifr6;
	int s6;
	u_int32_t flags6;
	struct in6_addrlifetime lifetime;
	time_t t = time(NULL);
	char hbuf[NI_MAXHOST];
#ifdef NI_WITHSCOPEID
	const int niflag = NI_NUMERICHOST | NI_WITHSCOPEID;
#else
	const int niflag = NI_NUMERICHOST;
#endif

	strncpy(ifr6.ifr_name, ifa->ifa_name, sizeof(ifr6.ifr_name));
	if ((s6 = socket(AF_INET6, SOCK_DGRAM, 0)) < 0) {
		perror("ifconfig: socket");
		return;
	}
	ifr6.ifr_addr = *SIN6(ifa->ifa_addr);
	if (ioctl(s6, SIOCGIFAFLAG_IN6, &ifr6) < 0) {
		perror("ifconfig: ioctl(SIOCGIFAFLAG_IN6)");
		close(s6);
		return;
	}
	flags6 = ifr6.ifr_ifru.ifru_flags6;
	ifr6.ifr_addr = *SIN6(ifa->ifa_addr);
	memset(&lifetime, 0, sizeof(lifetime));
	if (ioctl(s6, SIOCGIFALIFETIME_IN6, &ifr6) < 0) {
		perror("ifconfig: ioctl(SIOCGIFALIFETIME_IN6)");
		close(s6);
		return;
	}
	lifetime = ifr6.ifr_ifru.ifru_lifetime;
	close(s6);

	sin6p = SIN6(ifa->ifa_addr);
	in6_fillscopeid(sin6p);
	if (getnameinfo((struct sockaddr *)sin6p, sin6p->sin6_len,
			hbuf, sizeof(hbuf), NULL, 0, niflag))
		strncpy(hbuf, "", sizeof(hbuf));	/* some message? */
	printf("\tinet6 %s ", hbuf);
	if (ifa->ifa_flags & IFF_POINTOPOINT) {
		sin6p = (struct sockaddr_in6 *)ifa->ifa_dstaddr;
		if (sin6p->sin6_family == AF_INET6) {
			in6_fillscopeid(sin6p);
			if (getnameinfo((struct sockaddr *)sin6p,
					sin6p->sin6_len, hbuf, sizeof(hbuf),
					NULL, 0, niflag)) {
				/* some message? */
				strncpy(hbuf, "", sizeof(hbuf));
			}
			printf("--> %s ", hbuf);
		}
	}
	sin6p = SIN6(ifa->ifa_netmask);
	printf("prefixlen %d ", prefixlen(&sin6p->sin6_addr,
				sizeof(sin6p->sin6_addr)));
	sin6p = SIN6(ifa->ifa_addr);
	if (sin6p->sin6_scope_id)
		printf("scopeid 0x%04x ", sin6p->sin6_scope_id);
	if ((flags6 & IN6_IFF_ANYCAST) != 0)
		printf("anycast ");
	if ((flags6 & IN6_IFF_TENTATIVE) != 0)
		printf("tentative ");
	if ((flags6 & IN6_IFF_DUPLICATED) != 0)
		printf("duplicated ");
	if ((flags6 & IN6_IFF_DETACHED) != 0)
		printf("detached ");
	if ((flags6 & IN6_IFF_DEPRECATED) != 0)
		printf("deprecated ");
	if ((flags6 & IN6_IFF_AUTOCONF) != 0)
		printf("autoconf ");
	if ((flags6 & IN6_IFF_TEMPORARY) != 0)
		printf("temporary ");
#ifdef IN6_IFF_HOME
	if ((flags6 & IN6_IFF_HOME) != 0)
		printf("home ");
#endif
	
	if (Lflag) {
		printf("pltime ");
		if (lifetime.ia6t_preferred) {
			printf("%s ", lifetime.ia6t_preferred < t
				? "0" : sec2str(lifetime.ia6t_preferred - t));
		} else
			printf("infty ");

		printf("vltime ");
		if (lifetime.ia6t_expire) {
			printf("%s ", lifetime.ia6t_expire < t
				? "0" : sec2str(lifetime.ia6t_expire - t));
		} else
			printf("infty ");
	}

	printf("\n");
}

static int
prefixlen(val, siz)
	void *val;
	size_t siz;
{
	u_char *name = (u_char *)val;
	int byte, bit, plen = 0;

	for (byte = 0; byte < siz; byte++, plen += 8)
		if (name[byte] != 0xff)
			break;
	if (byte == siz)
		return plen;
	for (bit = 7; bit != 0; bit--, plen++)
		if (!(name[byte] & (1 << bit)))
			break;
	for ( ; bit != 0; bit--)
		if (name[byte] & (1 << bit))
			return 0;
	byte++;
	for ( ; byte < siz; byte++)
		if (name[byte])
			return 0;
	return plen;
}
#endif /* INET6 */

/* XNS address family support */

static void
xns_getaddr(afp, addr, which)
	struct afswitch *afp;
	char *addr;
	long which;
{
	struct sockaddr_ns *sns;

	sns = SNS(afp->af_stab[which]);
	sns->sns_family = AF_NS;
	sns->sns_len = sizeof(*sns);
	sns->sns_addr = ns_addr(addr);
	if (which == MASK)
		printf("Attempt to set XNS netmask will be ineffectual\n");
}

static void
xns_status(ifa)
	struct ifaddrs *ifa;
{
	struct sockaddr_ns *sns;

	sns = SNS(ifa->ifa_addr);
	printf("\tns %s ", ns_ntoa(sns->sns_addr));
	if (ifa->ifa_flags & IFF_POINTOPOINT) { /* by W. Nesheim@Cornell */
		sns = SNS(ifa->ifa_dstaddr);
		printf("--> %s ", ns_ntoa(sns->sns_addr));
	}
	printf("\n");
}

/* ISO address family support */

static void
iso_getaddr(afp, addr, which)
	struct afswitch *afp;
	char *addr;
	long which;
{
	struct sockaddr_iso *siso;

	siso = SISO(afp->af_stab[which]);
	siso->siso_addr = *iso_addr(addr);
	if (which == MASK) {
		siso->siso_len = TSEL(siso) - (caddr_t)(siso);
		siso->siso_nlen = 0;
	} else {
		siso->siso_len = sizeof(*siso);
		siso->siso_family = AF_ISO;
	}
}

static void
iso_status(ifa)
	struct ifaddrs *ifa;
{
	struct sockaddr_iso *siso;

	siso = SISO(ifa->ifa_addr);
	printf("\tiso %s ", iso_ntoa(&siso->siso_addr));
	if (ifa->ifa_netmask != NULL) {
		siso = SISO(ifa->ifa_netmask);
		printf(" netmask %s ", iso_ntoa(&siso->siso_addr));
	}
	if (ifa->ifa_flags & IFF_POINTOPOINT) {
		siso = SISO(ifa->ifa_dstaddr);
		printf("--> %s ", iso_ntoa(&siso->siso_addr));
	}
	printf("\n");
}

static int
sa_cmpaddr(ifname, sa1, sa2)
	char *ifname;		/* unused */
	struct sockaddr *sa1, *sa2;
{
	return(memcmp((void *)sa1, (void *)sa2, sa1->sa_len) == 0);
}

#ifdef INET6
static int
in6_cmpaddr(ifname, sa1, sa2)
	char *ifname;
	struct sockaddr *sa1, *sa2;
{
	struct sockaddr_in6 *sa6_1, *sa6_2;

	sa6_1 = (struct sockaddr_in6 *)sa1;
	sa6_2 = (struct sockaddr_in6 *)sa2;

	if (IN6_IS_ADDR_LINKLOCAL(&sa6_1->sin6_addr)) {
		struct sockaddr_in6 sa6ll_1 = *sa6_1; /* make a copy for safety */
		struct sockaddr_in6 sa6ll_2 = *sa6_2;
		u_int16_t embedded_id =
			*(u_int16_t *)&sa6ll_1.sin6_addr.s6_addr[2];
		u_int32_t ifindex;

		/*
		 * XXX: ad-hoc hack for KAME kernel. Set the embedded ifindex
		 * to the sin6_scope_id field and clear the embedded index.
		 */
		embedded_id = ntohs(embedded_id);
		sa6ll_1.sin6_scope_id = (u_int32_t)embedded_id;
		*(u_int16_t *)&sa6ll_1.sin6_addr.s6_addr[2] = 0;

		if ((ifindex = if_nametoindex(ifname)) == 0)
			errx(1, "if_nametoindex failed in in6_cmpaddr");
		sa6ll_2.sin6_scope_id = ifindex;

		sa6_1 = &sa6ll_1;
		sa6_2 = &sa6ll_2;
	}

	return(memcmp((void *)sa6_1, (void *)sa6_2, sa6_1->sin6_len) == 0);
}
#endif /* INET6 */

static void
setsnpaoffset(afp, p, offset)
	struct afswitch *afp;
	struct cmd *p;
	char *offset;
{
	u_long snpaoffset;
	char *ep;

	errno = 0;
	snpaoffset = strtoul(offset, &ep, 10);
	if (*offset == 0 || *ep != 0 ||
	   (snpaoffset == ULONG_MAX && errno == EINVAL))
		errx(1, "invalid snpaoffset: %s", offset);

	iso_addreq.ifra_snpaoffset = snpaoffset;
}

static void
setnsellength(afp, p, length)
	struct afswitch *afp;
	struct cmd *p;
	char *length;
{
	char *ep;

	errno = 0;
	nsellength = strtoul(length, &ep, 10);
	if (*length == 0 || *ep != 0 ||
	   (nsellength == ULONG_MAX && errno == EINVAL))
		errx(1, "invalid nsellength: %s", length);
}

static void
fixnsel(siso)
	struct sockaddr_iso *siso;
{
	if (siso->siso_family == 0)
		return;
	siso->siso_tlen = nsellength;
}

static void
adjust_nsellength(afp)
	struct afswitch *afp;
{
	fixnsel(SISO(afp->af_stab[ADDR]));
	fixnsel(SISO(afp->af_stab[DSTADDR]));
}

/**/
/* Media support */

struct media_cmd {
	const	char *m_name;		/* Argument name */
	u_int	m_mediabits;		/* Flag to set or reset */
	u_int	m_linktype;		/* Link type */
	u_int	m_parameter;		/* TRUE if this is a flag, not a type */
};

/* Media parameters */

    /*
     * Order matters!  The first command name for a given
     * media sub-type is the one used in a status display.
     */
#define MF(key, ifm, type) { key, ifm, type, PS_MEDIAFLAG }, \
	{ "-" key, ifm, type, PS_MEDIAFLAG }
#define	MI(inst) { "inst" #inst,	inst,	0,	PS_MEDIAINST }, \
	{ "instance" #inst,	inst,	0,	PS_MEDIAINST }
#define MT(key, ifm, type) { key, ifm, type, PS_MEDIATYPE }

static struct media_cmd media_cmds[] = {
	/* Ethernet */
	MT("aui",		IFM_10_5,	IFT_ETHER),
	MT("10base5",		IFM_10_5,	IFT_ETHER),
	MT("bnc",		IFM_10_2,	IFT_ETHER),
	MT("10base2",		IFM_10_2,	IFT_ETHER),
	MT("10baseT",		IFM_10_T,	IFT_ETHER),
	MT("utp",		-1,		0),		/* sp case 1 */
	MT("100baseTX",		IFM_100_TX,	IFT_ETHER),
	MT("tx",		IFM_100_TX,	IFT_ETHER),
	MT("100baseFX",		IFM_100_FX,	IFT_ETHER),
	MT("fx",		IFM_100_FX,	IFT_ETHER),
	MT("100baseT4",		IFM_100_T4,	IFT_ETHER),
	MT("t4",		IFM_100_T4,	IFT_ETHER),
	MT("100VgAnyLan",	IFM_100_VG,	IFT_ETHER),
	MT("vg", 		IFM_100_VG,	IFT_ETHER),
	MT("anylan", 		IFM_100_VG,	IFT_ETHER),
	MT("100baseT2",		IFM_100_T2,	IFT_ETHER),
	MT("t2",		IFM_100_T2,	IFT_ETHER),

	/* Token Ring */
	MT("utp16", 		IFM_TOK_UTP16,	IFT_ISO88025),
	MT("utp4", 		IFM_TOK_UTP4,	IFT_ISO88025),
	MT("stp16", 		IFM_TOK_STP16,	IFT_ISO88025),
	MT("stp4", 		IFM_TOK_STP4,	IFT_ISO88025),
	MF("early_token_release",	IFM_TOK_ETR,	IFT_ISO88025),
	MF("etr", 		IFM_TOK_ETR,	IFT_ISO88025),
	MF("early", 		IFM_TOK_ETR,	IFT_ISO88025),
	MF("source_route",	IFM_TOK_SRCRT,	IFT_ISO88025),
	MF("srt", 		IFM_TOK_SRCRT,	IFT_ISO88025),
	MF("all_broadcast",	IFM_TOK_ALLR,	IFT_ISO88025),
	MF("allbc", 		IFM_TOK_ALLR,	IFT_ISO88025),

	/* FDDI */
	MT("multimode",		IFM_FDDI_MMF,	IFT_FDDI),
	MT("mmf",		IFM_FDDI_MMF,	IFT_FDDI),
	MT("fiber",		IFM_FDDI_MMF,	IFT_FDDI),
	MT("fibre",		IFM_FDDI_MMF,	IFT_FDDI),
	MT("singlemode",	IFM_FDDI_SMF,	IFT_FDDI),
	MT("smf",		IFM_FDDI_SMF,	IFT_FDDI),
	MT("cddi",		IFM_FDDI_UTP,	IFT_FDDI),
	MF("dual_attach",	IFM_FDDI_DA,	IFT_FDDI),
	MF("dual",		IFM_FDDI_DA,	IFT_FDDI),
	MF("da",		IFM_FDDI_DA,	IFT_FDDI),

	/* Global options */
	MF("fdx",		IFM_FDX,	0),
	MF("full_duplex",	IFM_FDX,	0),
	MF("hdx",		IFM_HDX,	0),
	MF("half_duplex",	IFM_HDX,	0),
	MF("flag0",		IFM_FLAG0,	0),
	MF("flag1",		IFM_FLAG1,	0),
	MF("flag2",		IFM_FLAG2,	0),
	MF("loopback",		IFM_LOOP,	0),
	MT("auto",		IFM_AUTO,	0),
	MT("automedia",		IFM_AUTO,	0),
	MT("nomedia",		IFM_NONE,	0),
	MT("disc",		IFM_NONE,	0),
	MT("manual",		IFM_MANUAL,	0),

	/* Instances */
	MI(0),		MI(1),		MI(2),		MI(3),
	MI(4),		MI(5),		MI(6),		MI(7),
	MI(8),		MI(9),		MI(10),		MI(11),
	MI(12),		MI(13),		MI(14),		MI(15),

	{ NULL }
};

/*
 * Parse media keywords
 */
static void
setmedia(afp, p, addr)
	struct afswitch *afp;
	struct cmd *p;
	char *addr;
{
	struct media_cmd *mp;
	char *ap, *arg;
	int *mflags, *moflags;

	ap = addr;
	while ((arg = strsep(&ap, " ,")) != NULL) {
		/* Ignore empty fields */
		if (*arg == 0)
			continue;

		for (mp = media_cmds; mp->m_name != NULL; mp++)
			if (strcasecmp(arg, mp->m_name) == 0)
				break;

		if (mp->m_name == NULL)
			errx(1, "unknown media parameters: %s", arg);

		switch (mp->m_parameter) {
		case PS_MEDIATYPE:
		case PS_MEDIAINST:
			if (parms_set & mp->m_parameter)
				errx(1, "conflicting or duplicate media "
				    "parameter: %s",
				    arg);
			break;

		default:
			break;
		}
		switch (mp->m_parameter) {
		case PS_MEDIATYPE:
			cur_mtype = mp->m_mediabits;
			if (mp->m_linktype == 0)
				break;

			if (cur_ltype != 0 && mp->m_linktype != cur_ltype)
				errx(1, "Media %s in conflict"
				    "with some media flags",
				    mp->m_name);

			cur_ltype = mp->m_linktype;
			break;

		case PS_MEDIAFLAG:
			if (mp->m_linktype != 0) {
				if (cur_ltype != 0 && 
				    mp->m_linktype != cur_ltype)
					errx(1, 
				"Media flag %s conflicts with media type",
					     mp->m_name);
				cur_ltype = mp->m_linktype;
			}

                        if (*arg == '-') {
                                mflags = &mflags_clr;
                                moflags = &mflags_set;
                        } else {
                                mflags = &mflags_set;
                                moflags = &mflags_clr;
                        }
                        if (*moflags & mp->m_mediabits)
                                errx(1, "media flag conflict: %s", mp->m_name);
                        *mflags |= mp->m_mediabits; 
			break;

		case PS_MEDIAINST:
			minst = mp->m_mediabits;
			break;

		default:
			errx(1, "internal media table error");
			break;
		}

		parms_set |= mp->m_parameter;
	}
}

/*
 * Display current interface media status
 */
static void
media_status_current(ifname)
	char *ifname;
{
	struct ifmediareq mr;

	strncpy(mr.ifm_name, ifname, sizeof(mr.ifm_name));
	mr.ifm_count = 0;
	if (ioctl(s, SIOCGIFMEDIA, &mr) < 0 || mr.ifm_count == 0)
		return;
	printf("\tmedia %s",
	    media_display_word(mr.ifm_current, 0));
	if (mr.ifm_active != mr.ifm_current)
		printf(" (%s)",
		    media_display_word(mr.ifm_active, 0));
	if ((mr.ifm_status & IFM_AVALID) != 0) {
		printf(" status ");
		switch (mr.ifm_current & IFM_NMASK) {
		case IFM_ETHER:
			if ((mr.ifm_status & IFM_ACTIVE) != 0)
				printf("active");
			else
				printf("no-carrier");
			break;
		case IFM_TOKEN:
		case IFM_FDDI:
			if ((mr.ifm_status & IFM_ACTIVE) != 0)
				printf("inserted");
			else
				printf("no-ring");
			break;
		}
	}
	printf("\n");
}

/* The order in which multiple link types are displayed */
static int ntypes[] = { IFM_ETHER, IFM_TOKEN, IFM_FDDI, 0 };

/*
 * Display all media possibilities on an interface
 */
static void
media_status_all(ifname)
	char *ifname;
{
	struct ifmediareq mr;
	int *ip, *lp, *np;
	int ntype, multi_linktypes;

	/* Find out how many slots we need */
	strncpy(mr.ifm_name, ifname, sizeof(mr.ifm_name));
	mr.ifm_count = 0;
	if (ioctl(s, SIOCGIFMEDIA, &mr) < 0 || mr.ifm_count == 0)
		return;
	mr.ifm_ulist = malloc(sizeof(*mr.ifm_ulist) * mr.ifm_count);
	/* Now fetch list of options */
	if (ioctl(s, SIOCGIFMEDIA, &mr) < 0) {
		warn("Error reading media options on %s", ifname);
		return;
	}

	if (mr.ifm_count == 1 &&
	    *mr.ifm_ulist == mr.ifm_current)
		return;

	/* Does this interface support more than one link type? */
	ntype = mr.ifm_current & IFM_NMASK;
	multi_linktypes = 0;
	for (ip = mr.ifm_ulist, lp = ip + mr.ifm_count; ip < lp; ip++)
		if ((*ip & IFM_NMASK) != ntype)
			multi_linktypes++;

	if (multi_linktypes) {
		if (mr.ifm_mask & IFM_GMASK) {
			printf("\tmedia options");
			if (mr.ifm_mask & IFM_FDX)
				printf(" full_duplex");
			if (mr.ifm_mask & IFM_HDX)
				printf(" half_duplex");
			if (mr.ifm_mask & IFM_LOOP)
				printf(" loopback");
			if (mr.ifm_mask & IFM_FLAG0)
				printf(" flag0");
			if (mr.ifm_mask & IFM_FLAG1)
				printf(" flag1");
			if (mr.ifm_mask & IFM_FLAG2)
				printf(" flag2");
			printf("\n");
		}
		for (np = ntypes; *np != 0; np++)
			for (ip = mr.ifm_ulist, lp = ip + mr.ifm_count; ip < lp;
			    ip++) {
				if ((*ip & IFM_NMASK) != *np)
					continue;
				printf("\tmedia choice %s\n",
				    media_display_word(*ip, 1));
			}
	} else {
		if (mr.ifm_mask != 0) {
			printf("\tmedia options");
			switch (mr.ifm_current & IFM_NMASK) {
			case IFM_ETHER:
				break;
			case IFM_TOKEN:
				if (mr.ifm_mask & IFM_TOK_ETR)
					printf(" early_token_release");
				if (mr.ifm_mask & IFM_TOK_SRCRT)
					printf(" source_route");
				if (mr.ifm_mask & IFM_TOK_ALLR)
					printf(" all_broadcast");
				break;
			case IFM_FDDI:
				if (mr.ifm_mask & IFM_FDDI_DA)
					printf(" dual_attach");
				break;
			default:
				printf(" %8.8x", mr.ifm_mask);
			}
			if (mr.ifm_mask & IFM_FDX)
				printf(" full_duplex");
			if (mr.ifm_mask & IFM_HDX)
				printf(" half_duplex");
			if (mr.ifm_mask & IFM_LOOP)
				printf(" loopback");
			if (mr.ifm_mask & IFM_FLAG0)
				printf(" flag0");
			if (mr.ifm_mask & IFM_FLAG1)
				printf(" flag1");
			if (mr.ifm_mask & IFM_FLAG2)
				printf(" flag2");
			printf("\n");
		}
		for (ip = mr.ifm_ulist, lp = ip + mr.ifm_count; ip < lp; ip++)
			printf("\tmedia choice %s\n",
			    media_display_word(*ip, 0));
	}
}

/*
 * Display a media word in text form
 */
static char *
media_display_word(w, plink)
	int w;			/* Media word to decode */
	int plink;		/* Print link info if non-zero */
{
	struct media_cmd *mp;
	const char *desc;
	u_int ltype, ntype, stype;
	static char line[1024];
	char *lp, *ep;

	lp = line;
	ep = line + sizeof(line);

	ntype = w & IFM_NMASK;
	switch (ntype) {
	case IFM_ETHER:
		ltype = IFT_ETHER;
		break;
	case IFM_TOKEN:
		ltype = IFT_ISO88025;
		break;
	case IFM_FDDI:
		ltype = IFT_FDDI;
		break;
	default:
		(void)snprintf(lp, ep - lp, "unknown media type: %u", ntype);
		return (line);
	}
	stype = w & IFM_TMASK;

	switch (stype) {
	case IFM_AUTO:
		desc = "auto";
		break;
	case IFM_MANUAL:
		desc = "manual";
		break;
	case IFM_NONE:
		desc = "none";
		break;
	default:
		for (mp = media_cmds; mp->m_name != NULL; mp++)
			if (mp->m_parameter == PS_MEDIATYPE &&
			    mp->m_linktype == ltype && mp->m_mediabits == stype)
				break;
		desc = mp->m_name == NULL ? "unknown media" : mp->m_name;
	}
	switch (ntype) {
	case IFM_ETHER:
		if (plink)
			lp += snprintf(lp, ep - lp, "linktype ether ");
		lp += snprintf(lp, ep - lp, "%s", desc);
		break;
	case IFM_TOKEN:
		if (plink)
			lp += snprintf(lp, ep - lp, "linktype token_ring ");
		lp += snprintf(lp, ep - lp, "%s", desc);
		if (w & IFM_TOK_ETR)
			lp += snprintf(lp, ep - lp, " early_token_release");
		if (w & IFM_TOK_SRCRT)
			lp += snprintf(lp, ep - lp, " source_route");
		if (w & IFM_TOK_ALLR)
			lp += snprintf(lp, ep - lp, " all_broadcast");
		break;
	case IFM_FDDI:
		if (plink)
			lp += snprintf(lp, ep - lp, "linktype fddi ");
		lp += snprintf(lp, ep - lp, "%s", desc);
		if (w & IFM_FDDI_DA)
			lp += snprintf(lp, ep - lp, " dual_attach");
		break;
	default:
		(void)snprintf(lp, ep - lp,
		    "Undecipherable media word: %8.8x\n", w);
		return (line);
	}
	if (w & IFM_FDX)
		lp += snprintf(lp, ep - lp, " full_duplex");
	if (w & IFM_HDX)
		lp += snprintf(lp, ep - lp, " half_duplex");
	if (w & IFM_LOOP)
		lp += snprintf(lp, ep - lp, " loopback");
	if (w & IFM_FLAG0)
		lp += snprintf(lp, ep - lp, " flag0");
	if (w & IFM_FLAG1)
		lp += snprintf(lp, ep - lp, " flag1");
	if (w & IFM_FLAG2)
		lp += snprintf(lp, ep - lp, " flag2");
	if ((w & IFM_IMASK) != 0)
		lp += snprintf(lp, ep - lp,
		    " instance%d", (w & IFM_IMASK) >> IFM_ISHIFT);

	return (line);
}

#ifdef INET6
static char *
sec2str(total)
	time_t total;
{
	static char result[256];
	int days, hours, mins, secs;
	int first = 1;
	char *p = result;

	if (0) {	/*XXX*/
		days = total / 3600 / 24;
		hours = (total / 3600) % 24;
		mins = (total / 60) % 60;
		secs = total % 60;

		if (days) {
			first = 0;
			p += sprintf(p, "%dd", days);
		}
		if (!first || hours) {
			first = 0;
			p += sprintf(p, "%dh", hours);
		}
		if (!first || mins) {
			first = 0;
			p += sprintf(p, "%dm", mins);
		}
		sprintf(p, "%ds", secs);
	} else
		sprintf(p, "%u", total);

	return(result);
}
#endif /*INET6*/
