/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: pfkey.c,v 1.1.1.1.2.4.2.9 1998/12/05 11:33:02 sakane Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <net/route.h>
#include <net/pfkeyv2.h>

#include <netinet/in.h>

#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "var.h"
#include "vmbuf.h"
#include "misc.h"
#include "schedule.h"
#include "pfkey.h"
#include "pfkey_lib.h"
#include "debug.h"

int sockP;
u_int pfkey_timer = 5;
u_int pfkey_try = 3;

/*
 * PF_KEY packet handler
 *	0: success
 *	-1: fail
 */
int
pfkey_handler()
{
	struct sadb_msg *msg;
	caddr_t mhp[SADB_EXT_MAX + 1];
	struct pfkey_st *pst;
	int error = -1;

	/* receive pfkey message. */
	if ((msg = (struct sadb_msg *)pfkey_recv()) == 0) {
		plog("pfkey_handler", "pfkey_recv (%s)\n", strerror(errno));
		goto end;
	}

	/* check pfkey message. */
	memset(mhp, 0, sizeof(caddr_t) * (SADB_EXT_MAX + 1));
	if (pfkey_check(msg, mhp) != 0) {
		plog("pfkey_handler", "pfkey_check (%s)\n", strerror(errno));
		goto end;
	}

	switch (msg->sadb_msg_type) {
	case SADB_GETSPI:
		/* allocate buffer for status management of pfkey message */
		if ((pst = pfkey_new_st(mhp)) == 0) goto end;

		if (!mhp[SADB_EXT_SA]) {
			plog("pfkey_main", "invalid pointer was passed.\n");
			pfkey_free_st(pst);
			goto end;
		}
		pst->spi = ((struct sadb_sa *)mhp[SADB_EXT_SA])->sadb_sa_spi;

		if (isakmp_post_getspi(pst) < 0)
			plog("pfkey_main", "pfkey_getspi failed in isakmp.c\n");

		pfkey_free_st(pst);
		break;

	case SADB_UPDATE:
		/* If error, done in pfkey_check(). */
		plog("pfkey_main", "pfkey add was success.\n");
		break;

	case SADB_ADD:
		/* If error, done in pfkey_check(). */
		plog("pfkey_main", "pfkey update was success.\n");
		break;

	case SADB_ACQUIRE:
		/* allocate buffer for status management of pfkey message */
		if ((pst = pfkey_new_st(mhp)) == 0) goto end;

		/* start isakmp initiation by using ident exchange */
		if (isakmp_post_acquire(pst) < 0) {
			plog("pfkey_main", "pfkey_acquire failed in isakmp.c\n");
			pfkey_free_st(pst);
			goto end;
		}
		break;

	case SADB_REGISTER:
		/* XXX to be check to wait entry of register. */
		break;

#if 0
	case SADB_EXPIRE:
		if ((pst = pfkey_expire(mhp)) == 0) goto end;

		/* start isakmp initiation by using ident exchange */
		if (isakmp_pfkey(pst) < 0) goto end;
		break;
#endif

	default:
		plog("pfkey_main", "command %d isn't supported.\n",
			msg->sadb_msg_type);
		goto end;
	}

	error = 0;
end:
	if (msg) free(msg);
	return(error);
}

/*
 * PF_KEY initialization
 */
int
pfkey_init()
{
	if ((sockP = pfkey_open()) < 0)
		return(-1);

	if (pfkey_register(SADB_SATYPE_ESP) < 0) {
		plog("pfkey_init", "%s", strerror(errno));
		pfkey_close();
		return(-1);
	}

	if (pfkey_register(SADB_SATYPE_AH) < 0) {
		plog("pfkey_init", "%s", strerror(errno));
		pfkey_close();
		return(-1);
	}

#if 0
	if (pfkey_promisc_toggle(1) < 0) {
		pfkey_close();
		return(-1);
	}
#endif

	return(0);
}

