/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    its contributors.
 * 4. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* KAME @(#)$Id: pfkey_debug.c,v 1.1.1.1.2.4.2.5 1998/09/14 22:25:44 sakane Exp $ */

#if defined(IPSEC_DEBUG)

#include <sys/types.h>
#include <sys/param.h>
#if defined(KERNEL)
#include <sys/systm.h>
#include <sys/mbuf.h>
#endif /* defined(KERNEL) */
#include <sys/socket.h>

#include <net/route.h>

#include <netinet6/ipsec.h>
#include <netinet/in.h>
#include <netinet6/in6.h>

#include <netkey/keyv2.h>
#include <netkey/keydb.h>
#include <netkey/key_debug.h>

#if !defined(KERNEL)
#include <stdio.h>
#endif /* defined(KERNEL) */

/* NOTE: host byte order */

/* %%%: about struct sadb_msg */
void
kdebug_sadb(base)
	struct sadb_msg *base;
{
	struct sadb_ext *ext;
	int tlen, extlen;

	/* sanity check */
	if (base == 0)
		panic("kdebug_sadb: NULL pointer was passed.\n");

	printf("sadb_msg{ version=%u type=%u errno=%u satype=%u\n",
	    base->sadb_msg_version, base->sadb_msg_type,
	    base->sadb_msg_errno, base->sadb_msg_satype);
	printf("  len=%u ext=0x%04x seq=0x%08x pid=%u }\n",
	    base->sadb_msg_len, base->sadb_msg_reserved,
	    base->sadb_msg_seq, base->sadb_msg_pid);

	tlen = _UNUNIT64(base->sadb_msg_len) - sizeof(struct sadb_msg);
	ext = (struct sadb_ext *)((caddr_t)base + sizeof(struct sadb_msg));

	while (tlen > 0) {
		printf("sadb_ext{ len=%u type=%u }\n",
		    ext->sadb_ext_len, ext->sadb_ext_type);

		if (ext->sadb_ext_len == 0) {
			printf("kdebug_sadb: invalid ext_len=0 was passed.\n");
			return;
		}

		switch (ext->sadb_ext_type) {
		case SADB_EXT_SA:
			kdebug_sadb_sa(ext);
			break;
		case SADB_EXT_LIFETIME_CURRENT:
		case SADB_EXT_LIFETIME_HARD:
		case SADB_EXT_LIFETIME_SOFT:
			kdebug_sadb_lifetime(ext);
			break;
		case SADB_EXT_ADDRESS_SRC:
		case SADB_EXT_ADDRESS_DST:
		case SADB_EXT_ADDRESS_PROXY:
			kdebug_sadb_address(ext);
			break;
		case SADB_EXT_KEY_AUTH:
		case SADB_EXT_KEY_ENCRYPT:
			kdebug_sadb_key(ext);
			break;
		case SADB_EXT_IDENTITY_SRC:
		case SADB_EXT_IDENTITY_DST:
		case SADB_EXT_SENSITIVITY:
		case SADB_EXT_PROPOSAL:
			break;
		case SADB_EXT_SUPPORTED_AUTH:
		case SADB_EXT_SUPPORTED_ENCRYPT:
			kdebug_sadb_supported(ext);
			break;
		case SADB_EXT_SPIRANGE:
		case SADB_X_EXT_KMPRIVATE:
			break;
		case SADB_X_EXT_POLICY:
			kdebug_sadb_policy(ext);
			break;
		default:
			printf("kdebug_sadb: invalid ext_type %u was passed.\n",
			    ext->sadb_ext_type);
			return;
		}

		extlen = _UNUNIT64(ext->sadb_ext_len);
		tlen -= extlen;
		ext = (struct sadb_ext *)((caddr_t)ext + extlen);
	}

	return;
}

void
kdebug_sadb_supported(ext)
	struct sadb_ext *ext;
{
	struct sadb_supported *sup = (struct sadb_supported *)ext;

	/* sanity check */
	if (ext == 0)
		printf("kdebug_sadb_supported: NULL pointer was passed.\n");

	printf("sadb_sup{ len=%d }\n", sup->sadb_supported_len);

	return;
}

void
kdebug_sadb_lifetime(ext)
	struct sadb_ext *ext;
{
	struct sadb_lifetime *lft = (struct sadb_lifetime *)ext;

	/* sanity check */
	if (ext == 0)
		printf("kdebug_sadb_lifetime: NULL pointer was passed.\n");

	printf("sadb_lifetime{ alloc=%u, bytes=%u\n",
		lft->sadb_lifetime_allocations,
		(u_int32_t)lft->sadb_lifetime_bytes);
	printf("  addtime=%u, usetime=%u }\n",
		(u_int32_t)lft->sadb_lifetime_addtime,
		(u_int32_t)lft->sadb_lifetime_usetime);

	return;
}

void
kdebug_sadb_sa(ext)
	struct sadb_ext *ext;
{
	struct sadb_sa *sa = (struct sadb_sa *)ext;

	/* sanity check */
	if (ext == 0)
		panic("kdebug_sadb_sa: NULL pointer was passed.\n");

	printf("sadb_sa{ spi=0x%08lx replay=%u state=%u\n",
	    ntohl(sa->sadb_sa_spi), sa->sadb_sa_replay, sa->sadb_sa_state);
	printf("  auth=%u encrypt=%u flags=0x%08x }\n",
	    sa->sadb_sa_auth, sa->sadb_sa_encrypt, sa->sadb_sa_flags);

	return;
}

void
kdebug_sadb_address(ext)
	struct sadb_ext *ext;
{
	struct sadb_address *addr = (struct sadb_address *)ext;

	/* sanity check */
	if (ext == 0)
		panic("kdebug_sadb_address: NULL pointer was passed.\n");

	printf("sadb_address{ proto=%u prefixlen=%u reserved=%02x %02x }\n",
	    addr->sadb_address_proto, addr->sadb_address_prefixlen,
	    ((caddr_t)&addr->sadb_address_reserved)[0],
	    ((caddr_t)&addr->sadb_address_reserved)[1]);

	kdebug_sockaddr((struct sockaddr *)((caddr_t)ext + sizeof(*addr)));

	return;
}

void
kdebug_sadb_key(ext)
	struct sadb_ext *ext;
{
	struct sadb_key *key = (struct sadb_key *)ext;

	/* sanity check */
	if (ext == 0)
		panic("kdebug_sadb_key: NULL pointer was passed.\n");

	printf("sadb_key{ bits=%u reserved=%u }\n",
	    key->sadb_key_bits, key->sadb_key_reserved);
	printf("  key=");
#if 1
	kdebug_pdump((caddr_t)((caddr_t)key + sizeof(struct sadb_key)),
	             key->sadb_key_bits >> 3,
	             KDUMP_HEX);
#else
	kdebug_pdump((caddr_t)((caddr_t)key + sizeof(struct sadb_key)),
	             _UNUNIT64(key->sadb_key_len) - sizeof(struct sadb_key),
	             KDUMP_HEX);
#endif
	printf("\n");
	return;
}

void
kdebug_sadb_policy(ext)
	struct sadb_ext *ext;
{
	struct sadb_x_policy *pl = (struct sadb_x_policy *)ext;

	/* sanity check */
	if (ext == 0)
		panic("kdebug_sadb_x_policy: NULL pointer was passed.\n");

	printf("sadb_x_policy{ type=%u", pl->sadb_x_policy_type);
	if (pl->sadb_x_policy_type == SADB_X_PL_IPSEC)
		printf(" esp/t=%u ah/t=%u esp/n=%u ah/n=%u }\n",
		    pl->sadb_x_policy_esp_trans,
		    pl->sadb_x_policy_ah_trans,
		    pl->sadb_x_policy_esp_network,
		    pl->sadb_x_policy_ah_network);
	else
		printf(" }\n");

	return;
}

/* %%%: about SPD and SAD */
void
kdebug_secpc(sp)
	struct secpc *sp;
{
	/* sanity check */
	if (sp == 0)
		panic("kdebug_secpc: NULL pointer was passed.\n");

	printf("secpc{ refcnt=%u state=0x%02x\n", sp->refcnt, sp->state);

	kdebug_secindex(&sp->idx);

	switch ((u_int32_t)sp->pl) {
	case 0:
		printf("  type=discard }\n");
		break;
	case ~0:
		printf("  type=bypass }\n");
		break;
	default:
		printf("  type=ipsec %u:%u:%u:%u }\n",
			sp->pl->level[SADB_X_ESP_TRANS],
			sp->pl->level[SADB_X_AH_TRANS],
			sp->pl->level[SADB_X_ESP_NETWORK],
			sp->pl->level[SADB_X_AH_NETWORK]);
		break;
	}

	return;
}

void
kdebug_secasb(sab)
	struct secasb *sab;
{
	/* sanity check */
	if (sab == 0)
		panic("kdebug_secasb: NULL pointer was passed.\n");

	printf("secasb{ refcnt=%u state=0x%02x\n", sab->refcnt, sab->state);

	kdebug_secindex(&sab->idx);

	return;
}

void
kdebug_secindex(idx)
	struct secindex *idx;
{
	/* sanity check */
	if (idx == 0)
		panic("kdebug_secindex: NULL pointer was passed.\n");

	printf("secindex{ family=%u prefs=%u prefd=%d\n",
		idx->family, idx->prefs, idx->prefd);

	if (idx->src)
		kdebug_pdump(idx->src, _SALENBYAF(idx->family), KDUMP_HEX);
	if (idx->dst)
		kdebug_pdump(idx->dst, _SALENBYAF(idx->family), KDUMP_HEX);

	printf("  proto=%u ports=%u portd=%d }\n",
		idx->proto, idx->ports, idx->portd);

	return;
}

void
kdebug_secas(sa)
	struct secas *sa;
{
	/* sanity check */
	if (sa == 0)
		panic("kdebug_secas: NULL pointer was passed.\n");

	printf("secas{ refcnt=%u state=0x%02x type=0x%02x auth=%u enc=%u\n",
	    sa->refcnt, sa->state, sa->type, sa->alg_auth, sa->alg_enc);
	printf("  spi=0x%08lx flags=%u\n", ntohl(sa->spi), sa->flags);

	if (sa->key_auth) kdebug_sadb_key((struct sadb_ext *)sa->key_auth);
	if (sa->key_enc) kdebug_sadb_key((struct sadb_ext *)sa->key_enc);
	if (sa->iv) kdebug_pdump(sa->iv, sa->ivlen ? sa->ivlen : 8, KDUMP_HEX);
	if (sa->proxy) kdebug_sockaddr(sa->proxy);
	if (sa->replay) kdebug_secreplay(sa->replay);

	return;
}

void
kdebug_secreplay(rpl)
	struct secreplay *rpl;
{
	/* sanity check */
	if (rpl == 0)
		panic("kdebug_secreplay: NULL pointer was passed.\n");

	printf(" secreplay{ count=0x%08x wsize=%u seq=0x%08x lastseq=0x%08x }\n",
	    rpl->count, rpl->wsize, rpl->seq, rpl->lastseq);

	return;
}

void
kdebug_sockaddr(addr)
	struct sockaddr *addr;
{
	caddr_t ad;

	/* sanity check */
	if (addr == 0)
		panic("kdebug_sockaddr: NULL pointer was passed.\n");

	switch (addr->sa_family) {
	case PF_INET:
	{
		struct sockaddr_in *in = (struct sockaddr_in *)addr;
		printf("sockaddr{ len=%u family=%u port=%u }\n",
		    in->sin_len, in->sin_family, in->sin_port);
		ad = (caddr_t)&in->sin_addr;
	}
		break;
	case PF_INET6:
	{
		struct sockaddr_in6 *in6 = (struct sockaddr_in6 *)addr;
		ad = (caddr_t)&in6->sin6_addr;
	}
		break;
	default:
		return;
	}

	kdebug_pdump(ad, _ALENBYAF(addr->sa_family), KDUMP_HEX);

	return;
}

#if defined(KERNEL)
void
kdebug_mbufhdr(m)
	struct mbuf *m;
{
	/* sanity check */
	if (m == 0)
		panic("debug_mbufhdr: NULL pointer was passed.\n");

	printf("mbuf(0x%08x){ m_len=%u m_type=0x%02x m_flags=0x%02x\n",
	    m, m->m_len, m->m_type, m->m_flags);
	printf("  m_pkthdr.len=%d m_pkthdr.rcvif=0x%x }\n",
	    m->m_pkthdr.len, m->m_pkthdr.rcvif);

	return;
}

void
kdebug_mbuf(m0)
	struct mbuf *m0;
{
	struct mbuf *m = m0;
	int i, j;

	kdebug_mbufhdr(m);
	printf("  m_data=\n");
	for (j = 0; m; m = m->m_next) {
		for (i = 0; i < m->m_len; i++) {
			if (i != 0 && i % 32 == 0) printf("\n");
			if (i % 4 == 0) printf(" ");
			printf("%02x", mtod(m, u_char *)[i]);
			j++;
		}
	}

	printf("\n");

	return;
}
#endif /* defined(KERNEL) */

void
kdebug_pdump(buf, len, mode)
	caddr_t buf;
	int len;
	int mode;
{
	int i;

	if (mode == KDUMP_BIN) {
		for (i = 0; i < len; i++)
			printf("%c", (unsigned char)buf[i]);
		printf("\n");
	} else { /* HEX */
		for (i = 0; i < len; i++) {
			if (i != 0 && i % 32 == 0) printf("\n");
			if (i % 4 == 0) printf(" ");
			printf("%02x", (unsigned char)buf[i]);
		}
		if (i % 32 != 0) printf("\n");
	}

}

#endif /* defined(IPSEC_DEBUG) */
