/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    its contributors.
 * 4. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: isakmp.c,v 1.1.1.1.2.9.2.12 1998/09/14 11:56:31 sakane Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <net/route.h>
#include <net/pfkeyv2.h>
#include <netkey/keydb.h>

#include <netinet/in.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include "var.h"
#include "vmbuf.h"
#include "misc.h"
#include "schedule.h"
#include "cfparse.h"
#include "isakmp.h"
#include "isakmp_var.h"
#include "handler.h"
#include "oakley.h"
#include "ipsec_doi.h"
#include "crypto.h"
#include "pfkey.h"
#include "pfkey_lib.h"
#include "admin.h"
#include "debug.h"

char *local_secret;
vchar_t oakley_prime768;
vchar_t oakley_prime1024;
static u_char r_ck0[] = { 0,0,0,0,0,0,0,0 }; /* used to verify the r_ck. */
static u_char msgid0[] = { 0,0,0,0 }; /* used to verify the msgid. */

u_int isakmp_try = ISAKMP_TRY_DEFAULT;
u_int isakmp_timer = ISAKMP_TIMER_DEFAULT;
u_int isakmp_nonce_size = DEFAULTNONCESIZE;
u_int local_secret_size = DEFAULTSECRETSIZE;
u_int isakmp_pad_lword = MAXPADLWORD;
int isakmp_randpad = 0;
int isakmp_pad_check = 0;
int isakmp_pad_exclone = 0;

static int isakmp_phase1 __P((vchar_t *, struct sockaddr *, struct sockaddr *));
static int isakmp_phase2 __P((vchar_t *, struct sockaddr *));
static int isakmp_validate_pl2 __P((vchar_t *, struct sockaddr *));
static int isakmp_validate_pl1 __P((vchar_t *, struct sockaddr *));

/* quick mode */
static int isakmp_quick_r2 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph2 *));
static int isakmp_quick_i2 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph2 *));
static int isakmp_quick_r1 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph2 *));
static int isakmp_quick_i1 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph2 *));
static vchar_t *isakmp_quick_ir1mx __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph2 *));
static int isakmp_compute_keymat __P((struct isakmp_ph2 *));
static vchar_t *isakmp_compute_keymat_x __P((struct isakmp_ph2 *,
				struct ipsec_sa *));
static vchar_t *isakmp_compute_hash3 __P((struct isakmp_ph2 *));
static vchar_t *isakmp_compute_hash2 __P((struct isakmp_ph2 *, vchar_t *));
static vchar_t *isakmp_compute_hash1 __P((struct isakmp_ph2 *, vchar_t *));

/* main mode */
static int isakmp_ident_i4 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_ident_r3 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_ident_i3 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_ident_r2 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_ident_i2 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_ident_r1 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_ident_i1 __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static vchar_t *isakmp_ident_ir3mx __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static vchar_t *isakmp_ident_ir2mx __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_compute_skeyids __P((struct isakmp_ph1 *));
static int isakmp_compute_enckey __P((struct isakmp_ph1 *));
static int isakmp_compute_iv __P((struct isakmp_ph1 *));
static int isakmp_compute_iv2 __P((struct isakmp_ph2 *));
static vchar_t *isakmp_compute_hash __P((struct isakmp_ph1 *, int));
static vchar_t *isakmp_prf __P((vchar_t *, vchar_t *, struct isakmp_ph1 *));
static vchar_t *isakmp_hash __P((vchar_t *, struct isakmp_ph1 *));
static int isakmp_dh_compute __P((u_int, vchar_t *, vchar_t *, vchar_t *,
				vchar_t *, vchar_t **));
static int isakmp_dh_generate __P((u_int, vchar_t **, vchar_t **, vchar_t **));
static vchar_t *isakmp_do_decrypt __P((struct isakmp_ph1 *, vchar_t *,
				vchar_t *, vchar_t *));
static vchar_t *isakmp_do_encrypt __P((struct isakmp_ph1 *, vchar_t *,
				vchar_t *, vchar_t *));
static int isakmp_padlen __P((int));
static int isakmp_set_cookie __P((char *, struct sockaddr *));
static int isakmp_id2isa __P((struct isakmp_ph1 *, struct isakmp_pl_id *));
static int isakmp_kn2isa __P((struct isakmp_ph1 *, struct isakmp_pl_ke *,
				struct isakmp_pl_nonce *));

/* information exchange */
static int isakmp_inf_nrecv __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_inf_drecv __P((vchar_t *, struct sockaddr *,
				struct isakmp_ph1 *));
static int isakmp_inf_send __P((int));

static int isakmp_check_des_weakkey __P((vchar_t *));

/* ipsec_doi */
extern vchar_t *ipsecdoi_make_mysa(struct isakmp_cf_sa *, u_int32);
extern int ipsecdoi_sap2isa(caddr_t, struct ipsecdoi_sa *);

/*
 * main processing to handle isakmp payload
 */
int
isakmp_main(msg, from, my)
	vchar_t *msg;
	struct sockaddr *from, *my;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;

	/* validate header */
	if (isakmp_validate_pl1(msg, from) < 0)
		return(-1);

	/* whether phase 1 or phase 2 */
	if (memcmp((char *)&isakmp->msgid, msgid0, sizeof(msgid_t)) == 0) {
		/* i.e. phase 1 */
		if (isakmp_phase1(msg, from, my) < 0)
			return(-1);
	} else {
		/* i.e. phase 2 */
		if (isakmp_phase2(msg, from) < 0)
			return(-1);
	}

	return(0);
}

struct isakmp_ph1 *
isakmp_begin_ident(cfp, my, remote)
	struct isakmp_conf *cfp;
	struct sockaddr *my, *remote;
{
	struct isakmp_ph1 *iph1;
	isakmp_index index;
	int error = -1;

	/* create isakmp index */
	memset((char *)&index, 0, sizeof(index));
	isakmp_set_cookie((char *)&index, remote);

	/* add new entry to isakmp status table */
	if ((iph1 = isakmp_new_ph1(&index)) == 0) goto end;

	iph1->dir = INITIATOR;
	iph1->cfp = cfp;

	/* set my address */
	if ((iph1->local = (struct sockaddr *)malloc(my->sa_len)) == 0) {
		plog("isakmp_begin_ident", "malloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy((caddr_t)iph1->local, (caddr_t)my, my->sa_len);

	/* begin ident mode */
	/* XXX: variable ? */
	if (isakmp_ident_i1((vchar_t *)0, remote, iph1) < 0) goto end;

	error = 0;

end:
	if (error) iph1 = 0;
	return(iph1);
}

int
isakmp_begin_quick(iph1, pst)
	struct isakmp_ph1 *iph1;
	struct pfkey_st *pst;
{
	struct isakmp_ph2 *iph2 = 0;
	int error = -1;

	/* add new entry to isakmp status table */
	iph1->msgid2++; /* XXX */
	if ((iph2 = isakmp_new_ph2(iph1, (msgid_t *)&iph1->msgid2)) == 0)
		goto end;

	iph2->dir = INITIATOR;
	iph2->pst = pst;

	if (memcmp(_INADDRBYAF(pst->dst),
	           _INADDRBYAF(iph1->remote), _ALENBYAF(pst->dst))) {
		iph2->proxy = EXCHANGE_PROXY;
		/* XXX: to be written the proxy processing */
		goto end;
	} else
		iph2->proxy = EXCHANGE_MYSELF;

	if (isakmp_compute_iv2(iph2) < 0) {
		if (isakmp_free_ph2(iph2) < 0) {
			goto end; /* XXX */
		}
		goto end;
	}

	/* begin quick mode */
	if (isakmp_quick_i1((vchar_t *)0, iph1->cfp->remote, iph2) < 0) goto end;

	error = 0;
end:
	return(error);
}
/*
 * handling phase 1.
 */
static int
isakmp_phase1(msg, from, my)
	vchar_t *msg;
	struct sockaddr *from, *my;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	isakmp_index *index = (isakmp_index *)isakmp;
	struct isakmp_ph1 *iph1 = 0;
	vchar_t *buf = 0;
	int error = -1;

	/* search isakmp status record of phase 1 */
	iph1 = isakmp_ph1byindex(index);

	/* If set encryption bit. */
	if (ISSET(isakmp->flags, ISAKMP_FLAG_E)) {

		/* error if not exists. */
		if (iph1 == 0) {
			isakmp_inf_send(ISAKMP_NTYPE_PAYLOAD_MALFORMED);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_phase1",
			        "unknown packet, no ISAKMP-SA exists.\n"));
			return(-1);
		}

		/* decrypt packet */
		if ((buf = isakmp_do_decrypt(iph1, msg, iph1->iv, iph1->ive)) == 0)
			return(-1);

	} else
		/* not encrypted */
		buf = msg;

	/* validate pre-check payload, general headers */
	if (isakmp_validate_pl2(buf, from) < 0)
		goto end;

	switch (isakmp->etype) {
	case ISAKMP_ETYPE_IDENT:

		/* error if not exists. */
		if (isakmp->np != ISAKMP_NPTYPE_SA && iph1 == 0) {
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_phase1",
			        "unknown packet, expecting SA payload.\n"));
			goto end;
		}

		switch (isakmp->np) {
		case ISAKMP_NPTYPE_SA:

			/* 
			 * It must be SA payload that next payload type of ISAKMP header
			 * which is the 1st or 2nd exchange in phase 1 negociation.
			 */

			/* search isakmp status table and error if exists. */
			if (iph1 != 0) {
				YIPSDEBUG(DEBUG_NOTIFY,
				    plog2(from, "isakmp_phase1",
				        "duplicate packet, may be processed.\n"));
				goto end;
			}
	
			/* search isakmp status table by index in which r_ck is zero */
			iph1 = isakmp_ph1byindex0(index);

			if (iph1 == 0) {

				/* check responder's cookie whether 0 or not. */
				if (memcmp((char *)&isakmp->r_ck, r_ck0, sizeof(cookie_t)) != 0) {
					isakmp_inf_send(ISAKMP_NTYPE_INVALID_COOKIE);
					YIPSDEBUG(DEBUG_NOTIFY,
					    plog2(from, "isakmp_phase1",
					        "unknown packet, expecting r_ck to be 0.\n"));
					goto end;
				}

				/* This packet is from initiator as 1st exchange. */

				/* add new entry to isakmp status table. */
				if ((iph1 = isakmp_new_ph1(index)) == 0)
					goto end;

				iph1->dir = RESPONDER;

				/* set my address */
				if ((iph1->local = (struct sockaddr *)malloc(sizeof(*my))) == 0) {
					plog("isakmp_phase1", "malloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy((caddr_t)iph1->local, (caddr_t)my,
					sizeof(*my));

				/* set remote address */
				if ((iph1->remote = (struct sockaddr *)malloc(sizeof(*iph1->remote))) == 0) {
					plog("isakmp_phase1", "malloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy((caddr_t)iph1->remote, (caddr_t)from,
					sizeof(*iph1->remote));

				/* It's responder's 1st exchange in phase 1. */
				if (isakmp_ident_r1(buf, from, iph1) < 0)
					goto end;

			} else {

				/* This packet is from responder as 2nd exchange. */

				/* It's initiator's 2nd exchange in phase 1. */
				/* check exchange type with isakmp status record. */
				if (isakmp->etype != iph1->etype) {
					isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
					YIPSDEBUG(DEBUG_NOTIFY,
					    plog2(from, "isakmp_phase1",
					        "invalid exchange type %d, expecting %d.\n",
					        isakmp->etype, iph1->etype));
					goto end;
				}
	
				/* Has this packet been processed or unexpected ? */
				if (iph1->dir != INITIATOR || iph1->status != ISAKMP_PH1_1) {
					YIPSDEBUG(DEBUG_NOTIFY,
					    plog2(from, "isakmp_phase1",
					        "packet unexpected or is old.\n"));
					goto end;
				}

				/* turn off schedule */
				sched_kill(iph1->sc);
	
				if (isakmp_ident_i2(buf, from, iph1) < 0)
					goto end;
			}
			break;

		case ISAKMP_NPTYPE_KE:

			/* check exchange type */
			if (isakmp->etype != iph1->etype) {
				isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
				YIPSDEBUG(DEBUG_NOTIFY,
				    plog2(from, "isakmp_phase1",
				        "invalid exchange type %d, expecting %d.\n",
				        isakmp->etype, iph1->etype));
				goto end;
			}

			/* Has this packet been processed or unexpected ? */
			if ((iph1->dir == INITIATOR && iph1->status != ISAKMP_PH1_2)
			 || (iph1->dir == RESPONDER && iph1->status != ISAKMP_PH1_1)) {
				YIPSDEBUG(DEBUG_NOTIFY,
				    plog2(from, "isakmp_phase1",
				        "packet unexpected or is old.\n"));
				goto end;
			}

			/* turn off schedule */
			sched_kill(iph1->sc);

			if (iph1->dir == INITIATOR) {
				if (isakmp_ident_i3(buf, from, iph1) < 0)
					goto end;

			} else { /* RESPONDER */
				if (isakmp_ident_r2(buf, from, iph1) < 0)
					goto end;

				/* generate SKEYIDs & IV & final cipher key */
				if (isakmp_compute_skeyids(iph1) < 0) goto end;
				if (isakmp_compute_enckey(iph1) < 0) goto end;
				if (isakmp_compute_iv(iph1) < 0) goto end;
			}
			break;

		case ISAKMP_NPTYPE_ID:
			/* check exchange type */
			if (isakmp->etype != iph1->etype) {
				isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
				YIPSDEBUG(DEBUG_NOTIFY,
				    plog2(from, "isakmp_phase1",
				        "invalid exchange type %d, expecting %d.\n",
				        isakmp->etype, iph1->etype));
				goto end;
			}

			/* Has this packet been processed or unexpected ? */
			if ((iph1->dir == INITIATOR && iph1->status != ISAKMP_PH1_3)
			 || (iph1->dir == RESPONDER && iph1->status != ISAKMP_PH1_2)) {
				YIPSDEBUG(DEBUG_NOTIFY,
				    plog2(from, "isakmp_phase1",
				        "packet unexpected or is old.\n"));
				goto end;
			}

			/* turn off schedule */
			sched_kill(iph1->sc);

			if (iph1->dir == INITIATOR) {
				if (isakmp_ident_i4(buf, from, iph1) < 0)
					goto end;

			} else { /* RESPONDER */
				if (isakmp_ident_r3(buf, from, iph1) < 0)
					goto end;
			}
			break;

		default:
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_phase1",
			        "invalid next payload type %d.\n", isakmp->np));
			goto end;
		}
		break;

	case ISAKMP_ETYPE_INF:
		switch (isakmp->np) {
		case ISAKMP_NPTYPE_N:
			if (isakmp_inf_nrecv(buf, from, iph1) < 0)
				goto end;

		case ISAKMP_NPTYPE_D:
			if (isakmp_inf_drecv(buf, from, iph1) < 0)
				goto end;

		default:
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_phase1",
			        "invalid next payload type %d.\n", isakmp->np));
			goto end;
		}
		break;

	case ISAKMP_ETYPE_AGG:
	case ISAKMP_ETYPE_BASE:
	case ISAKMP_ETYPE_AUTH:
	case ISAKMP_ETYPE_NEWGRP:
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase1",
		        "exchange type %d unsupported.\n", isakmp->etype));
		goto end;

	case ISAKMP_ETYPE_QUICK:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase1",
		        "invalied exchange type %d, expecting to be phase 1.\n",
		        isakmp->etype));
		goto end;

	case ISAKMP_ETYPE_NONE:
	default:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase1",
		        "invalid exchange type %d.\n", isakmp->etype));
		goto end;
	}

	error = 0;

end:
	/* free buffer allocated to decrypt. */
	if (ISSET(isakmp->flags, ISAKMP_FLAG_E) && buf != 0) {
		free(buf);
	}

	return(error);
}

/*
 * handling phase 2.
 */
static int
isakmp_phase2(msg, from)
	vchar_t *msg;
	struct sockaddr *from;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	isakmp_index *index = (isakmp_index *)isakmp;
	msgid_t *msgid = &isakmp->msgid;
	struct isakmp_ph1 *iph1 = 0;
	struct isakmp_ph2 *iph2 = 0;
	vchar_t *buf = 0;
	int fnew = 0; /* flag which is used if doing free `*iph2'. */
	int error = -1;

	/* If set encryption bit. */
	if (! ISSET(isakmp->flags, ISAKMP_FLAG_E)) {
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase2",
		        "unknown packet, expecting packet encypted as phase 2.\n"));
		return(-1);
	}

	/* search isakmp status record of phase 1. */
	/* error if not exists. */
	if ((iph1 = isakmp_ph1byindex(index)) == 0) {
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase2",
		        "unknown packet, no ISAKMP-SA exists.\n"));
		return(-1);
	}

	/* check status of phase 1 whether negotiated or not. */
	if (iph1->status != ISAKMP_PH2_N) {
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase2",
		        "unknown packet, no ISAKMP-SA negotiated exists.\n"));
		return(-1);
	}

	/* search isakmp phase 2 stauts record. */
	if ((iph2 = isakmp_ph2bymsgid(iph1, msgid)) == 0) {

		/* new negotiation in phase 2, and compute IV */
		if ((iph2 = isakmp_new_ph2(iph1, msgid)) == 0)
			return(-1);

		fnew = 1; /* to do free later */
		iph2->dir = RESPONDER;

		if (isakmp_compute_iv2(iph2) < 0)
			goto end;
	}

	/* decrypt packet */
	if ((buf = isakmp_do_decrypt(iph1, msg, iph2->iv, iph2->ive)) == 0)
		goto end;

	/* validate pre-check payload, general headers */
	if (isakmp_validate_pl2(buf, from) < 0)
		goto end;

	switch (isakmp->etype) {
	case ISAKMP_ETYPE_QUICK:

		/* error if not exists. */
		if (iph1 == 0 || iph2 == 0) {
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_phase2",
			        "received unknown packet, must be negotiated phase 1.\n"));
			goto end;
		}
	
		/* check status in isakmp phase 1 status */
		if (iph1->status != ISAKMP_PH2_N) {
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_phase2",
			        "invalid status, phase 1 hasn't finished.\n"));
			goto end;
		}

		/* search isakmp phase 2 status record */
		if (iph2->dir == RESPONDER && iph2->status == ISAKMP_PH2_N) {

			/* clear flag because of doing free in quick_r1(). */
			fnew = 0;

			/* begin Phase 2 session as responder */
			if (isakmp_quick_r1(buf, from, iph2) < 0) goto end;

		} else {

			/* Has this packet been processed or unexpected ? */
			if ((iph2->dir == INITIATOR && iph2->status != ISAKMP_PH2_1)
			 || (iph2->dir == RESPONDER && iph2->status != ISAKMP_PH2_1)) {
				YIPSDEBUG(DEBUG_NOTIFY,
				    plog2(from, "isakmp_phase2", "received packet unexpected.\n"));
				goto end;
			}

			/* turn off schedule */
			sched_kill(iph2->sc);
	
			if (iph2->dir == INITIATOR) {
				if (isakmp_quick_i2(buf, from, iph2) < 0) goto end;
			} else { /* i.e. RESPONDER */
				if (isakmp_quick_r2(buf, from, iph2) < 0) goto end;
			}
		}
		break;

	case ISAKMP_ETYPE_INF:
		switch (isakmp->np) {
		case ISAKMP_NPTYPE_N:
			if (isakmp_inf_nrecv(buf, from, iph1) < 0)
				goto end;

		case ISAKMP_NPTYPE_D:
			if (isakmp_inf_drecv(buf, from, iph1) < 0)
				goto end;

		default:
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_phase2",
			        "invalid next payload type %d.\n", isakmp->np));
			goto end;
		}
		break;

	case ISAKMP_ETYPE_NEWGRP:
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase2",
		        "exchange type %d unsupported.\n", isakmp->etype));
		goto end;

	case ISAKMP_ETYPE_IDENT:
	case ISAKMP_ETYPE_AGG:
	case ISAKMP_ETYPE_BASE:
	case ISAKMP_ETYPE_AUTH:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase2",
		        "invalied exchange type %d, expecting to be phase 2.\n",
		        isakmp->etype));
		goto end;

	case ISAKMP_ETYPE_NONE:
	default:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_phase2",
		        "invalid exchange type.\n", isakmp->etype));
		goto end;
	}

	error = 0;

end:
	if (error && fnew)
		if (iph2) isakmp_free_ph2(iph2);

	/* free buffer allocated to decrypt. */
	free(buf);

	return(error);
}

/*
 * validate ISAKMP payload.
 *   see 5.2 ISAKMP Header Processing.
 */
static int
isakmp_validate_pl2(buf, from)
	vchar_t *buf;
	struct sockaddr *from;
{
	struct isakmp *isakmp = (struct isakmp *)buf->v;
	struct isakmp_gen *gen;
	int tlen, plen;

	tlen = buf->l - sizeof(struct isakmp);

	if (tlen <= sizeof(struct isakmp_gen)) {
		isakmp_inf_send(ISAKMP_NTYPE_PAYLOAD_MALFORMED);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_validate_pl2",
		        "invalid length of payload\n"));
		return(-1);
	}

	plen = sizeof(struct isakmp);
	gen = (struct isakmp_gen *)buf->v;

	/* validate general headers */
	do {
		gen = (struct isakmp_gen *)((char *)gen + plen);

		tlen -= ntohs(gen->len);

		if (gen->reserved != 0) {
			isakmp_inf_send(ISAKMP_NTYPE_PAYLOAD_MALFORMED);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_validate_pl2",
			        "invalid general header.\n"));
			return(-1);
		}

		switch (gen->np) {
		case ISAKMP_NPTYPE_NONE:
			if (isakmp_pad_check && tlen != 0) {
				isakmp_inf_send(ISAKMP_NTYPE_PAYLOAD_MALFORMED);
				YIPSDEBUG(DEBUG_NOTIFY,
				    plog2(from, "isakmp_validate_pl2",
				        "invalid length of payload.\n"));
				return(-1);
			}
			break;

		case ISAKMP_NPTYPE_SA:
		case ISAKMP_NPTYPE_KE:
		case ISAKMP_NPTYPE_ID:
		case ISAKMP_NPTYPE_CERT:
		case ISAKMP_NPTYPE_CR:
		case ISAKMP_NPTYPE_HASH:
		case ISAKMP_NPTYPE_SIG:
		case ISAKMP_NPTYPE_NONCE:
		case ISAKMP_NPTYPE_N:
		case ISAKMP_NPTYPE_D:
			if (tlen < sizeof(struct isakmp_gen)) {
				isakmp_inf_send(ISAKMP_NTYPE_PAYLOAD_MALFORMED);
				YIPSDEBUG(DEBUG_NOTIFY,
				    plog2(from, "isakmp_validate_pl2",
				        "invalid length of payload.\n"));
				return(-1);
			}
			break;

		case ISAKMP_NPTYPE_P:
		case ISAKMP_NPTYPE_T:
		default:
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_validate_pl2",
			        "invalid next payload type %d.\n", gen->np));
			return(-1);
		}

		plen = ntohs(gen->len);

	} while (gen->np != ISAKMP_NPTYPE_NONE);

	return(0);
}

/*
 * validate ISAKMP header.
 *   see 5.2 ISAKMP Header Processing.
 */
static int
isakmp_validate_pl1(buf, from)
	vchar_t *buf;
	struct sockaddr *from;
{
	struct isakmp *isakmp = (struct isakmp *)buf->v;
	int len = ntohl(isakmp->len);

	/* validate total length */
	if (len != buf->l) {
		isakmp_inf_send(ISAKMP_NTYPE_PAYLOAD_MALFORMED);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_validate_pl1",
		        "invalid length of payload.\n"));
		return(-1);
	}

	/* validate the type of next payload */
	switch (isakmp->np) {
	case ISAKMP_NPTYPE_SA:
	case ISAKMP_NPTYPE_KE:
	case ISAKMP_NPTYPE_ID:
	case ISAKMP_NPTYPE_N:
	case ISAKMP_NPTYPE_D:
	case ISAKMP_NPTYPE_HASH:
		break;

	case ISAKMP_NPTYPE_CERT:
	case ISAKMP_NPTYPE_CR:
	case ISAKMP_NPTYPE_SIG:
	case ISAKMP_NPTYPE_NONCE:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_validate_pl1",
		        "valid next payload type, but %d unsupported.\n", isakmp->np));
		return(-1);

	case ISAKMP_NPTYPE_NONE:
	case ISAKMP_NPTYPE_P:
	case ISAKMP_NPTYPE_T:
	default:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_validate_pl1",
		        "invalid next payload type %d.\n", isakmp->np));
		return(-1);
	}

	/* Check the Major and Minor Version fields. */
	if (((isakmp->v_maj << 4) | isakmp->v_min) <
	    ((ISAKMP_MAJOR_VERSION << 4) | ISAKMP_MINOR_VERSION)) {

		if (isakmp->v_maj < ISAKMP_MAJOR_VERSION) {
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_MAJOR_VERSION);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_validate_pl1",
			        "invalid major version %d.\n", isakmp->v_maj));
			return(-1);
		}

		if (isakmp->v_min < ISAKMP_MINOR_VERSION) {
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_MINOR_VERSION);
			YIPSDEBUG(DEBUG_NOTIFY,
			    plog2(from, "isakmp_validate_pl1",
			    "invalid minor version %d.\n", isakmp->v_min));
			return(-1);
		}
	}

	switch (isakmp->etype) {
	case ISAKMP_ETYPE_IDENT:
	case ISAKMP_ETYPE_QUICK:
		break;

	case ISAKMP_ETYPE_AGG:
	case ISAKMP_ETYPE_NONE:
	case ISAKMP_ETYPE_BASE:
	case ISAKMP_ETYPE_AUTH:
	case ISAKMP_ETYPE_NEWGRP:
	case ISAKMP_ETYPE_INF:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_validate_pl1",
		        "exchange type %d unsupported.\n", isakmp->etype));
		return(-1);

	default:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_EXCHANGE_TYPE);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_validate_pl1",
		    "invalid exchange type.\n", isakmp->etype));
		return(-1);
	}

	/* check the Flags field. */
	if (isakmp->flags & ~(ISAKMP_FLAG_E | ISAKMP_FLAG_C)) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_FLAGS);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_validate_pl1",
		        "invalid flag 0x%02x.\n", isakmp->flags));
		return(-1);
	}

	/* If set commit bit. */
	if (ISSET(isakmp->flags, ISAKMP_FLAG_C)) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_FLAGS);
		YIPSDEBUG(DEBUG_NOTIFY,
		    plog2(from, "isakmp_validate_pl1",
		        "ISAKMP_FLAG_C unsupported.\n"));
		return(-1);
	}

	return(0);
}

/* %%%
 * Quick Mode
 */
/*
 * receive HDR*, HASH(3) from initiator, and set SA to kernel.
 */
static int
isakmp_quick_r2(msg, from, iph2)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph2 *iph2;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_hash *hash = 0;
	vchar_t *tmp = 0;
	char *r_hash;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_quick_r2", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_HASH) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_quick_r2",
		    "received invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_HASH);
		goto end;
	}

	hash = (struct isakmp_pl_hash *)((char *)isakmp + sizeof(*isakmp));
	if (hash->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_quick_r2",
		    "received invalid next payload type %d, expecting %d.\n",
		    hash->h.np, ISAKMP_NPTYPE_NONE);
		goto end;
	}
	hash->h.len = ntohs(hash->h.len);

	/* validate HASH(3) */
	r_hash = (char *)hash + sizeof(*hash);

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_quick_r2", "received HASH(3)\n"));
	YIPSDEBUG(DEBUG_DHASH,
	    pdump(r_hash, hash->h.len - sizeof(*hash), YDUMP_HEX));

	if ((tmp = isakmp_compute_hash3(iph2)) == 0) goto end;
	if (memcmp(tmp->v, r_hash, tmp->l)) {
		plog2(from, "isakmp_quick_r2", "HASH(3) mismatch.\n");
		goto end;
	}
	vfree(tmp);

	/* compute both of KEYMATs */
	if (isakmp_compute_keymat(iph2) < 0) goto end;

	/* Do UPDATE */
	plog("isakmp_quick_r2", "call update\n");
	if (f_local) ;
	else
	if ((pfkey_update(0,
			iph2->ph1->local, iph2->ph1->remote,
			iph2->isa)) < 0) {
		plog("isakmp_quick_r2", "update failed.\n");
		goto end;
	}

	/* Do ADD */
	plog("isakmp_quick_r2", "call add\n");
	if (f_local) ;
	else
	if ((pfkey_add(iph2->isa->proto_id,
			iph2->ph1->remote, iph2->ph1->local,
			iph2->isa)) < 0) {
		plog("isakmp_quick_r2", "add failed.\n");
		goto end;
	}

	plog2(from, "isakmp_quick_r2",
	    "get SA values for IPsec, %s.\n",
	        isakmp_pindex(&iph2->ph1->index, &iph2->msgid));

	error = 0;

end:
	if (iph2) (void)isakmp_free_ph2(iph2);
	return(error);
}

/*
 * receive HDR*;HASH(2);SA;Nr;[KE];[IDci,IDcr] from responder,
 * and send HDR*;HASH(3)
 */
static int
isakmp_quick_i2(msg, from, iph2)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph2 *iph2;
{
	vchar_t *buf = 0, *body = 0;
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_gen *gen;
	struct ipsecdoi_sa *sa = 0;
	struct isakmp_pl_hash *hash = 0;
	struct isakmp_pl_nonce *nonce = 0;
	char *p, *r_hash;
	int tlen, np;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_quick_i2", "begin.\n"));

	/* create buffer for validating HASH(2) */
	tlen = ntohl(isakmp->len) - sizeof(*isakmp);
	if ((buf = vmalloc(tlen)) == 0) {
		plog("isakmp_quick_i2", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = buf->v;

	/* validate the type of next payload */
	np = isakmp->np;
	gen = (struct isakmp_gen *)((char *)isakmp + sizeof(*isakmp));

	tlen = 0;

	while (np != ISAKMP_NPTYPE_NONE) {

		if (np == ISAKMP_NPTYPE_HASH) {
			hash = (struct isakmp_pl_hash *)gen;
			buf->l -= ntohs(gen->len); /* adjust length of buffer */
		} else {
			switch (np) {
			case ISAKMP_NPTYPE_SA:
				sa = (struct ipsecdoi_sa *)gen;
				break;
			case ISAKMP_NPTYPE_NONCE:
				if ((iph2->nonce_p =
				        vmalloc(ntohs(gen->len) - sizeof(*gen))) == 0) {
					plog("isakmp_quick_i2", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(iph2->nonce_p->v,
					(caddr_t)gen + sizeof(*gen),
					iph2->nonce_p->l);
				break;
			case ISAKMP_NPTYPE_KE:
				if ((iph2->dhpub_p =
				        vmalloc(ntohs(gen->len) - sizeof(*gen))) == 0) {
					plog("isakmp_quick_i1", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(iph2->dhpub_p->v,
					(caddr_t)gen + sizeof(*gen),
					iph2->dhpub_p->l);
				break;
			case ISAKMP_NPTYPE_ID:
				plog2(from, "isakmp_quick_i2",
				    "next payload type %d isn't supported.\n", np);
				goto end;
			default:
				isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
				plog2(from, "isakmp_quick_i2",
				    "received invalid next payload type %d.\n", np);
				goto end;
			}

			memcpy(p, (caddr_t)gen, ntohs(gen->len));
			p += ntohs(gen->len);
			tlen += ntohs(gen->len); /* compute true length of payload. */
		}

		/* adjust payload length */
		gen->len = ntohs(gen->len);

		/* next payload */
		np = gen->np;
		gen = (struct isakmp_gen *)((char *)gen + gen->len);
	}

	if (hash == 0 || sa == 0 || iph2->nonce_p == 0) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_quick_i2", "received short payload.\n");
		goto end;
	}

	/* validate HASH(2) */
	r_hash = (char *)hash + sizeof(*hash);

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_quick_i2", "received HASH(2)\n"));
	YIPSDEBUG(DEBUG_DHASH,
	    pdump(r_hash, hash->h.len - sizeof(*hash), YDUMP_HEX));

	buf->l = tlen; /* adjust length */

	if ((body = isakmp_compute_hash2(iph2, buf)) == 0) goto end;
	if (memcmp(body->v, r_hash, body->l)) {
		plog2(from, "isakmp_quick_i2", "HASH(2) mismatch.\n");
		goto end;
	}
	vfree(body);
	body = 0;
	vfree(buf);

	/* XXX check SA returned. */
	/* check(); to compare iph2->sa->v and sa. */

    {
	/* set responder's SPI */
	vchar_t *spi;
	
	/* save SPI */
	spi = iph2->isa->spi;

	/* save SA to use */
	if (ipsecdoi_sap2isa((caddr_t)iph2->isa, sa) < 0) goto end; 

	/* set SPI values */
	iph2->isa->spi_p = iph2->isa->spi;
	iph2->isa->spi = spi;
    }

	/* generate HASH(3) */
	if ((iph2->hash = isakmp_compute_hash3(iph2)) == 0) goto end;

	/* create buffer for isakmp payload */
	tlen = sizeof(*isakmp) + sizeof(*gen) + iph2->hash->l;
	if ((buf = vmalloc(tlen)) == 0) { 
		plog("isakmp_quick_i2", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/* create isakmp header */
	memcpy(buf->v, (caddr_t)isakmp, sizeof(*isakmp));
	isakmp = (struct isakmp *)buf->v;
	isakmp->np = ISAKMP_NPTYPE_HASH;
	isakmp->len = htons(tlen);
	memcpy((caddr_t)&isakmp->msgid, (caddr_t)&iph2->msgid,
		sizeof(isakmp->msgid));
	p = buf->v + sizeof(*isakmp);

	/* create HASH(3) payload */
	gen = (struct isakmp_gen *)p;
	gen->np = ISAKMP_NPTYPE_NONE;
	gen->len = htons(sizeof(*gen) + iph2->hash->l);
	p += sizeof(*gen);

	memcpy(p, iph2->hash->v, iph2->hash->l);

	/* encoding */
	if ((body = isakmp_do_encrypt(iph2->ph1, buf, iph2->ive, iph2->iv)) == 0)
		goto end;

	vfree(buf);

	buf = body;

	/* send HDR*;HASH(3) */
	if (isakmp_send(iph2->ph1, buf) < 0) goto end;

	/* XXX: How resend ? */

	/* compute both of KEYMATs */
	if (isakmp_compute_keymat(iph2) < 0) goto end;

	/* Do UPDATE */
	plog("isakmp_quick_i2", "call update\n");
	if (f_local) ;
	else {
		if (iph2->pst->seq != 0) {
			/* begin by SADB_ACQUIRE */
			if ((pfkey_update(iph2->pst->seq,
					iph2->ph1->local, iph2->ph1->remote,
					iph2->isa)) < 0) {
				plog("isakmp_quick_i2", "update failed.\n");
				goto end;
			}
		} else {
			/* begin by SADB_EXPIRE */
			if ((pfkey_add(iph2->pst->satype,
					iph2->ph1->local, iph2->ph1->remote,
					iph2->isa)) < 0) {
				plog("isakmp_quick_i2", "add failed.\n");
				goto end;
			}
		}
	}

	/* Do ADD */
	plog("isakmp_quick_i2", "call add\n");
	if (f_local) ;
	else
	if ((pfkey_add(iph2->pst->satype,
			iph2->ph1->remote, iph2->ph1->local,
			iph2->isa)) < 0) {
		plog("isakmp_quick_i2", "add failed.\n");
		goto end;
	}

	plog2(from, "isakmp_quick_i2",
	    "get SA values for IPsec, %s.\n",
	        isakmp_pindex(&iph2->ph1->index, &iph2->msgid));

	error = 0;

end:
	if (iph2) (void)isakmp_free_ph2(iph2);
	return(error);
}

/*
 * receive HDR*;HASH(1);SA;Ni;[KE];[IDci,IDCr] from initiator,
 * and send HDR*;HASH(2);SA;Nr;[KE];[IDci,IDCr]
 */
static int
isakmp_quick_r1(msg, from, iph2)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph2 *iph2;
{
	vchar_t *buf = 0, *body = 0;
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_gen *gen;
	struct isakmp_pl_hash *hash = 0;
	char *p, *r_hash;
	int tlen, np;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_quick_r1", "begin.\n"));

	/* create buffer for validating HASH(1) */
	tlen = ntohl(isakmp->len) - sizeof(*isakmp);
	if ((buf = vmalloc(tlen)) == 0) {
		plog("isakmp_quick_r1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = buf->v;

	/* validate the type of next payload */
	np = isakmp->np;
	gen = (struct isakmp_gen *)((char *)isakmp + sizeof(*isakmp));

	/* initialize */
	iph2->id_p = 0;
	tlen = 0;

	while (np != ISAKMP_NPTYPE_NONE) {

		if (np == ISAKMP_NPTYPE_HASH) {
			hash = (struct isakmp_pl_hash *)gen;
			buf->l -= ntohs(gen->len); /* adjust length of buffer */
		} else {
			switch (np) {
			case ISAKMP_NPTYPE_SA:
				if ((iph2->sa =
	        			vmalloc(ntohs(gen->len))) == 0) {
					plog("isakmp_quick_r1", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(iph2->sa->v, (caddr_t)gen, iph2->sa->l);
				((struct ipsecdoi_sa *)iph2->sa->v)->h.len = iph2->sa->l;
				break;

			case ISAKMP_NPTYPE_NONCE:
				if ((iph2->nonce_p =
				        vmalloc(ntohs(gen->len) - sizeof(*gen))) == 0) {
					plog("isakmp_quick_r1", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(iph2->nonce_p->v,
					(caddr_t)gen + sizeof(*gen),
					iph2->nonce_p->l);
				break;

			case ISAKMP_NPTYPE_KE:
				if ((iph2->dhpub_p =
				        vmalloc(ntohs(gen->len) - sizeof(*gen))) == 0) {
					plog("isakmp_quick_r1", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(iph2->dhpub_p->v,
					(caddr_t)gen + sizeof(*gen),
					iph2->dhpub_p->l);
				break;

			case ISAKMP_NPTYPE_ID:
				if (iph2->id_p == 0) {
					/* for IDci */
					if ((iph2->id_p =
				   	     vmalloc(ntohs(gen->len) - sizeof(*gen))) == 0) {
						plog("isakmp_quick_r1", "vmalloc (%s)\n", strerror(errno));
						goto end;
					}
					memcpy(iph2->id_p->v,
						(caddr_t)gen + sizeof(*gen),
						iph2->id_p->l);
				} else {
					/* for IDci */
					if ((iph2->id =
				   	     vmalloc(ntohs(gen->len) - sizeof(*gen))) == 0) {
						plog("isakmp_quick_r1", "vmalloc (%s)\n", strerror(errno));
						goto end;
					}
					memcpy(iph2->id->v,
						(caddr_t)gen + sizeof(*gen),
						iph2->id->l);
				}
				break;

			default:
				isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
				plog2(from, "isakmp_quick_r1",
				    "received invalid next payload type %d.\n", np);
				goto end;
			}

			memcpy(p, (caddr_t)gen, ntohs(gen->len));
			p += ntohs(gen->len);
			tlen += ntohs(gen->len); /* compute true length of payload. */
		}

		/* adjust payload length */
		gen->len = ntohs(gen->len);

		/* next payload */
		np = gen->np;
		gen = (struct isakmp_gen *)((char *)gen + gen->len);
	}

	if (hash == 0 || iph2->sa == 0 || iph2->nonce_p == 0) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_quick_r1", "received short payload.\n");
		goto end;
	}

	/* validate HASH(1) */
	r_hash = (char *)hash + sizeof(*hash);

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_quick_r1", "received HASH(1)\n"));
	YIPSDEBUG(DEBUG_DHASH,
	    pdump(r_hash, hash->h.len - sizeof(*hash), YDUMP_HEX));

	buf->l = tlen; /* adjust length */

	if ((body = isakmp_compute_hash1(iph2, buf)) == 0) goto end;
	if (memcmp(body->v, r_hash, body->l)) {
		plog2(from, "isakmp_quick_r1", "HASH(1) mismatch.\n");
		goto end;
	}
	vfree(body);
	body = 0;
	vfree(buf);

	/* XXX select SA to use. */

	/* XXX check ID payload */

	/* save SA to use */
	if ((iph2->isa =
	        CALLOC(sizeof(struct ipsec_sa), struct ipsec_sa *)) == 0) {
		plog("isakmp_quick_r1", "calloc (%s)\n", strerror(errno));
		goto end;
	}
	if (ipsecdoi_sap2isa((caddr_t)iph2->isa, (struct ipsecdoi_sa *)iph2->sa->v) < 0) goto end; 

	/* replace SPI values set by ipsecdoi_sap2isa(). */
	iph2->isa->spi_p = iph2->isa->spi;
	iph2->isa->spi = 0;

    {
	u_int32_t spi;

	/* Do GETSPI */
	/* XXX to be handled ID payload */
	if (f_local) spi = 0x12345678;
	else
	if ((spi = pfkey_getspi(0, iph2->isa->proto_id,
			iph2->ph1->local, iph2->ph1->remote)) == 0) {
		plog("isakmp_quick_r1", "getspi failed.\n");
		goto end;
	}
	plog("isakmp_quick_r1", "called getspi spi=0x%0x\n", spi);

	/* set SPI values */
	/* XXX */
	memcpy((caddr_t)(iph2->sa->v
			+ sizeof(struct ipsecdoi_sa)
			+ sizeof(struct isakmp_pl_p)),
		(caddr_t)&spi, sizeof(spi));

	if ((iph2->isa->spi = vmalloc(sizeof(u_int32_t))) == 0) {
		plog("isakmp_quick_r1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy(iph2->isa->spi->v, (caddr_t)&spi, iph2->isa->spi->l);
    }

	/* generate NONCE value */
	if ((iph2->nonce = eay_set_random(isakmp_nonce_size)) == 0) goto end;

	/* generate KE value if need */
	if (iph2->ph1->cfp->pfs == PFS_NEED) {
		/* generate DH public value */
			/* XXX: grp should be from iph2. */
		if (isakmp_dh_generate(iph2->ph1->isa->dhgrp,
			&iph2->dhp, &iph2->dhpub, &iph2->dhpriv) < 0) goto end;
	}

	/* create SA;NONCE payload, and KE if need */
	tlen = iph2->sa->l + sizeof(*gen) + iph2->nonce->l
		+ ((iph2->ph1->cfp->pfs == PFS_NEED) ?
				(sizeof(*gen) + iph2->dhpub->l) : 0);
		+ ((iph2->id_p != 0) ? (iph2->id_p->l + iph2->id->l) : 0);

	if ((body = vmalloc(tlen)) == 0) { 
		plog("isakmp_quick_r1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = body->v;

	((struct ipsecdoi_sa *)iph2->sa->v)->h.len = htons(iph2->sa->l);
	memcpy(p, iph2->sa->v, iph2->sa->l);
	((struct ipsecdoi_sa *)iph2->sa->v)->h.len = iph2->sa->l;
	((struct isakmp_gen *)p)->np = ISAKMP_NPTYPE_NONCE;
	p += iph2->sa->l;

	gen = (struct isakmp_gen *)p;
	gen->np = ((iph2->ph1->cfp->pfs == PFS_NEED) ?
		ISAKMP_NPTYPE_KE : ISAKMP_NPTYPE_NONE);
	gen->len = htons(sizeof(*gen) + iph2->nonce->l);
	p += sizeof(*gen);
	memcpy(p, iph2->nonce->v, iph2->nonce->l);
	p += iph2->nonce->l;

	/* add KE payload if need. */
	if (iph2->ph1->cfp->pfs == PFS_NEED) {
		gen = (struct isakmp_gen *)p;
		gen->np = ((iph2->id_p == 0) ? ISAKMP_NPTYPE_NONE : ISAKMP_NPTYPE_ID);
		gen->len = htons(sizeof(*gen) + iph2->dhpub->l);
		p += sizeof(*gen);
		memcpy(p, iph2->dhpub->v, iph2->dhpub->l);
	}

	/* add ID payloads received. */
	if (iph2->id_p != 0) {
		gen = (struct isakmp_gen *)p;
		gen->np = ISAKMP_NPTYPE_ID;
		gen->len = htons(sizeof(*gen) + iph2->id_p->l);
		p += sizeof(*gen);
		memcpy(p, iph2->id_p->v, iph2->id_p->l);

		gen = (struct isakmp_gen *)p;
		gen->np = ISAKMP_NPTYPE_NONE;
		gen->len = htons(sizeof(*gen) + iph2->id->l);
		p += sizeof(*gen);
		memcpy(p, iph2->id->v, iph2->id->l);
	}

	/* compute HASH(2) */
	if ((iph2->hash = isakmp_compute_hash2(iph2, body)) == 0) goto end;

	/* send isakmp payload */
	if ((buf = isakmp_quick_ir1mx(body, from, iph2)) == 0) goto end;

	/* change status of isakmp status entry */
	iph2->status = ISAKMP_PH2_1;

	/* add to the schedule to resend, and seve back pointer. */
	iph2->sc = sched_add(isakmp_timer, isakmp_resend_ph2,
	                     isakmp_try, isakmp_timeout_ph2,
	                     (caddr_t)iph2, (caddr_t)buf);

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_quick_r1", "sendto HDR*;HASH(2);SA;Nr;[KE].\n"));

	error = 0;

end:
	if (error) {
		if (iph2) (void)isakmp_free_ph2(iph2);
	}
	if (body) vfree(body);

	return(error);
}

/*
 * begin Quick Mode as initiator , and send HDR*;HASH(1);SA;Ni;[KE];[IDci,IDCr]
 */
static int
isakmp_quick_i1(msg, from, iph2)
	vchar_t *msg; /* must be null pointer */
	struct sockaddr *from;
	struct isakmp_ph2 *iph2;
{
	vchar_t *buf = 0, *body = 0;
	struct isakmp_ph1 *xph1 = iph2->ph1;
	struct ipsecdoi_sa *sa;
	struct isakmp_gen *gen;
	struct isakmp *isakmp;
	char *p;
	int tlen;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_quick_i1", "begin.\n"));

    {
	u_int32_t spi;

	/* Do GETSPI */
	if (f_local) spi = 0x87654321;
	else
	if ((spi = pfkey_getspi(iph2->pst->seq,
				iph2->pst->satype,
				iph2->ph1->local, iph2->ph1->remote)) == 0) {
		plog("isakmp_quick_i1", "getspi failed.\n");
		goto end;
	}
	plog("isakmp_quick_i1", "called getspi spi=0x%0x\n", spi);

	/* make my ipsec sa */
	if ((iph2->sa = ipsecdoi_make_mysa(&xph1->cfp->ph[1]->sa, spi)) < 0)
		goto end;

	YIPSDEBUG(DEBUG_SA,
	    plog("isakmp_quick_i1", "SA payload len=%d\n", iph2->sa->l));
	YIPSDEBUG(DEBUG_DSA, pdump(iph2->sa->v, iph2->sa->l, YDUMP_HEX));

	/* save SPI */
	if ((iph2->isa =
	        CALLOC(sizeof(struct ipsec_sa), struct ipsec_sa *)) == 0) {
		plog("isakmp_quick_i1", "calloc (%s)\n", strerror(errno)); 
		goto end;
	}

	if ((iph2->isa->spi = vmalloc(sizeof(u_int32_t))) == 0) {
		plog("isakmp_quick_i1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy(iph2->isa->spi->v, (caddr_t)&spi, iph2->isa->spi->l);
    }

	/* generate NONCE value */
	if ((iph2->nonce = eay_set_random(isakmp_nonce_size)) == 0) goto end;

	/* generate KE value if need */
	if (iph2->ph1->cfp->pfs == PFS_NEED) {
		/* generate DH public value */
			/* XXX: grp should be from iph2. */
		if (isakmp_dh_generate(iph2->ph1->isa->dhgrp,
			&iph2->dhp, &iph2->dhpub, &iph2->dhpriv) < 0) goto end;
	}

	/* create SA;NONCE payload, and KE if need */
	tlen = iph2->sa->l + sizeof(*gen) + iph2->nonce->l
		+ ((iph2->ph1->cfp->pfs == PFS_NEED) ?
				(sizeof(*gen) + iph2->dhpub->l) : 0);
	if ((body = vmalloc(tlen)) == 0) { 
		plog("isakmp_quick_i1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = body->v;

	memcpy(p, iph2->sa->v, iph2->sa->l);
	((struct isakmp_gen *)p)->np = ISAKMP_NPTYPE_NONCE;
	p += iph2->sa->l;

	gen = (struct isakmp_gen *)p;
	gen->np = ((iph2->ph1->cfp->pfs == PFS_NEED) ?
		ISAKMP_NPTYPE_KE : ISAKMP_NPTYPE_NONE);
	gen->len = htons(sizeof(*gen) + iph2->nonce->l);
	p += sizeof(*gen);
	memcpy(p, iph2->nonce->v, iph2->nonce->l);
	p += iph2->nonce->l;

	/* add KE payload if need. */
	if (iph2->ph1->cfp->pfs == PFS_NEED) {
		gen = (struct isakmp_gen *)p;
		gen->np = ISAKMP_NPTYPE_NONE;
		gen->len = htons(sizeof(*gen) + iph2->dhpub->l);
		p += sizeof(*gen);
		memcpy(p, iph2->dhpub->v, iph2->dhpub->l);
	}

	/* XXX IDci, IDcr */

	/* compute HASH(1) */
	if ((iph2->hash = isakmp_compute_hash1(iph2, body)) == 0) goto end;

	/* send isakmp payload */
	if ((buf = isakmp_quick_ir1mx(body, from, iph2)) == 0) goto end;

	/* change status of isakmp status entry */
	iph2->status = ISAKMP_PH2_1;

	/* add to the schedule to resend, and seve back pointer. */
	iph2->sc = sched_add(isakmp_timer, isakmp_resend_ph2,
	                     isakmp_try, isakmp_timeout_ph2,
	                     (caddr_t)iph2, (caddr_t)buf);

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_quick_i1", "sendto HDR*;HASH(1);SA;Nr;[KE].\n"));

	error = 0;

end:
	if (error) {
		if (iph2) (void)isakmp_free_ph2(iph2);
	}
	if (body) vfree(body);

	return(error);
}

/*
 * create HASH, body (SA, NONCE) payload with isakmp header.
 */
static vchar_t *
isakmp_quick_ir1mx(body, from, iph2)
	vchar_t *body;
	struct sockaddr *from;
	struct isakmp_ph2 *iph2;
{
	struct isakmp_ph1 *xph1 = iph2->ph1;
	struct isakmp *isakmp;
	vchar_t *buf = 0, *new = 0;
	char *p;
	int tlen;
	struct isakmp_gen *gen;
	int error = -1;

	/* create buffer for isakmp payload */
	tlen = sizeof(*isakmp) + sizeof(*gen) + iph2->hash->l + body->l;
	if ((buf = vmalloc(tlen)) == 0) { 
		plog("isakmp_quick_ir1mx", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = buf->v;

	/* create isakmp header */
	isakmp = (struct isakmp *)buf->v;
	memcpy((caddr_t)&isakmp->i_ck, (caddr_t)&xph1->index.i_ck,
		sizeof(cookie_t));
	memcpy((caddr_t)&isakmp->r_ck, (caddr_t)&xph1->index.r_ck,
		sizeof(cookie_t));
	isakmp->np    = ISAKMP_NPTYPE_HASH;
	isakmp->v_maj = ISAKMP_MAJOR_VERSION;
	isakmp->v_min = ISAKMP_MINOR_VERSION;
	isakmp->etype = ISAKMP_ETYPE_QUICK;
	isakmp->flags = ISAKMP_FLAG_E;
	memcpy((caddr_t)&isakmp->msgid, (caddr_t)&iph2->msgid,
		sizeof(isakmp->msgid));
	isakmp->len   = htonl(tlen);
	p += sizeof(*isakmp);

	/* create HASH payload */
	gen = (struct isakmp_gen *)p;
	gen->np = ISAKMP_NPTYPE_SA;
	gen->len = htons(sizeof(*gen) + iph2->hash->l);
	p += sizeof(*gen);

	memcpy(p, iph2->hash->v, iph2->hash->l);
	p += iph2->hash->l;

	/* create SA;NONCE payload */
	memcpy(p, body->v, body->l);

	/* encoding */
	if ((new = isakmp_do_encrypt(xph1, buf, iph2->ive, iph2->iv)) == 0)
		goto end;

	vfree(buf);

	buf = new;

	/* send HDR*;HASH(1);SA;Nr to responder */
	if (isakmp_send(iph2->ph1, buf) < 0) goto end;

#if 0 /* emulation traffic congestion */
	if (isakmp_send(iph2->ph1, buf) < 0) goto end;
#endif

	/* XXX: synchronization IV */
	memcpy(iph2->ivd->v, iph2->iv->v, iph2->iv->l);

	error = 0;

end:
	if (error && buf) {
		vfree(buf);
		buf = 0;
	}

	return(buf);
}

/*
 * compute KEYMAT
 *   see seciton 5.5 Phase 2 - Quick Mode in isakmp-oakley-05.
 */
static int
isakmp_compute_keymat(iph2)
	struct isakmp_ph2 *iph2;
{
	int error = -1;

	/* compute sharing secret of DH when PFS */
	if (iph2->ph1->cfp->pfs == PFS_NEED && iph2->dhpub_p) {
		if (isakmp_dh_compute(iph2->ph1->isa->dhgrp, iph2->dhp, iph2->dhpub,
				iph2->dhpriv, iph2->dhpub_p, &iph2->dhgxy) < 0)
			goto end;
	}

	/* compute keymat */
	if ((iph2->isa->keymat = isakmp_compute_keymat_x(iph2, iph2->isa)) == 0)
		goto end;

	YIPSDEBUG(DEBUG_KEY,
	    plog("isakmp_compute_keymat", "compute KEYMAT.\n"));
	YIPSDEBUG(DEBUG_DKEY, pvdump(iph2->isa->keymat));

	error = 0;

end:
	return(error);
}

/*
 * compute KEYMAT.
 * KEYMAT = prf(SKEYID_d, protocol | SPI | Ni_b | Nr_b).
 * If PFS is desired and KE payloads were exchanged,
 *   KEYMAT = prf(SKEYID_d, g(qm)^xy | protocol | SPI | Ni_b | Nr_b)
 */
static vchar_t *
isakmp_compute_keymat_x(iph2, sa)
	struct isakmp_ph2 *iph2;
	struct ipsec_sa *sa;
{
	vchar_t *buf = 0, *res = 0, *bp;
	char *p;
	int len;
	int error = -1;
	int pfs = 0;

	pfs = ((iph2->ph1->cfp->pfs == PFS_NEED && iph2->dhgxy) ? 1 : 0);
	
	len = (pfs ? iph2->dhgxy->l : 0)
		+ 1 + sa->spi->l + iph2->nonce->l + iph2->nonce_p->l;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_keymat_x", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	p = buf->v;

	/* if PFS */
	if (pfs) {
		memcpy(p, iph2->dhgxy->v, iph2->dhgxy->l);
		p += iph2->dhgxy->l;
	}

	p[0] = sa->proto_id;
	p += 1;

	bp = (iph2->ph1->dir == INITIATOR ? iph2->isa->spi : iph2->isa->spi_p);
	memcpy(p, bp->v, bp->l);
	p += sa->spi->l;

	bp = (iph2->ph1->dir == INITIATOR ? iph2->nonce : iph2->nonce_p);
	memcpy(p, bp->v, bp->l);
	p += bp->l;

	bp = (iph2->ph1->dir == INITIATOR ? iph2->nonce_p : iph2->nonce);
	memcpy(p, bp->v, bp->l);
	p += bp->l;

	/* compute IV */
	YIPSDEBUG(DEBUG_DKEY,
	    plog("isakmp_compute_keymat_x", "body for computing KEYMAT.\n"));
	YIPSDEBUG(DEBUG_DKEY, pvdump(buf));
	if ((res = isakmp_prf(iph2->ph1->skeyid_d,
				buf, iph2->ph1)) == 0) goto end;

	/* XXX should be algorithm independent routine. */
	if (iph2->isa->enc_t == IPSECDOI_ESP_3DES) {
		vchar_t *res_e = 0;
		int len = res->l;

		if ((res_e = isakmp_prf(iph2->ph1->skeyid_d,
					buf, iph2->ph1)) == 0) goto end;

		if (VREALLOC(res, len * 2) == 0) {
			perror("vrealloc");
			vfree(res_e);
			goto end;
		}

		memcpy(res->v + len, res_e->v, len);
		res->l = len * 2;
	}

	error = 0;

end:
	if (error) {
		vfree(res);
	}

	if (buf) vfree(buf);

	return(res);
}

/*
 * compute HASH(3)
 *   see seciton 5.5 Phase 2 - Quick Mode in isakmp-oakley-05.
 */
static vchar_t *
isakmp_compute_hash3(iph2)
	struct isakmp_ph2 *iph2;
{
	vchar_t *buf = 0, *res = 0, *bp;
	struct isakmp_ph1 *xph1 = iph2->ph1;
	char *p;
	int len;
	int error = -1;

	/* create buffer */
	len = 1 + sizeof(msgid_t) + iph2->nonce->l + iph2->nonce_p->l;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_hash3", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/*
	 * HASH(3) = prf(SKEYID_a, 0 | M-ID | Ni_b | Nr_b)
	 */
	p = buf->v;

	p[0] = 0;
	p += 1;

	memcpy(p, (caddr_t)&iph2->msgid, sizeof(msgid_t));
	p += sizeof(msgid_t);

	bp = (iph2->ph1->dir == INITIATOR ? iph2->nonce : iph2->nonce_p);
	memcpy(p, bp->v, bp->l);
	p += bp->l;

	bp = (iph2->ph1->dir == INITIATOR ? iph2->nonce_p : iph2->nonce);
	memcpy(p, bp->v, bp->l);

	/* compute HASH */
	if ((res = isakmp_prf(xph1->skeyid_a, buf, xph1)) == 0) goto end;

	error = 0;

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_compute_hash3", "compute HASH(3).\n"));
	YIPSDEBUG(DEBUG_DHASH, pvdump(res));

end:
	if (buf) vfree(buf);
	return(res);
}

/*
 * compute HASH(2)
 *   see seciton 5.5 Phase 2 - Quick Mode in isakmp-oakley-05.
 */
static vchar_t *
isakmp_compute_hash2(iph2, body)
	struct isakmp_ph2 *iph2;
	vchar_t *body;
{
	vchar_t *buf = 0, *res = 0, *bp;
	struct isakmp_ph1 *xph1 = iph2->ph1;
	char *p;
	int len;
	int error = -1;

	bp = (iph2->ph1->dir == INITIATOR ? iph2->nonce : iph2->nonce_p);

	/* create buffer */
	len = sizeof(msgid_t) + bp->l + body->l;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_hash2", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/*
	 * HASH(2) = prf(SKEYID_a, M-ID | Ni_b | SA | Nr [ | KE ] [ | IDci | IDcr ])
	 */
	p = buf->v;

	memcpy(p, (caddr_t)&iph2->msgid, sizeof(msgid_t));
	p += sizeof(msgid_t);

	memcpy(p, bp->v, bp->l);
	p += bp->l;

	memcpy(p, body->v, body->l);

	YIPSDEBUG(DEBUG_DHASH, pvdump(buf));

	/* compute HASH */
	if ((res = isakmp_prf(xph1->skeyid_a, buf, xph1)) == 0) goto end;

	error = 0;

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_compute_hash2", "compute HASH(2).\n"));
	YIPSDEBUG(DEBUG_DHASH, pvdump(res));

end:
	if (buf) vfree(buf);
	return(res);
}

/*
 * compute HASH(1)
 *   see seciton 5.5 Phase 2 - Quick Mode in isakmp-oakley-05.
 */
static vchar_t *
isakmp_compute_hash1(iph2, body)
	struct isakmp_ph2 *iph2;
	vchar_t *body;
{
	vchar_t *buf = 0, *res = 0;
	struct isakmp_ph1 *xph1 = iph2->ph1;
	char *p;
	int len;
	int error = -1;

	/* create buffer */
	len = sizeof(msgid_t) + body->l;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_hash1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/*
	 * HASH(1) = prf(SKEYID_a, M-ID | SA | Ni [ | KE ] [ | IDci | IDcr ])
	 */
	p = buf->v;

	memcpy(buf->v, (caddr_t)&iph2->msgid, sizeof(msgid_t));
	p += sizeof(msgid_t);

	memcpy(p, body->v, body->l);

	/* compute HASH */
	if ((res = isakmp_prf(xph1->skeyid_a, buf, xph1)) == 0) goto end;

	error = 0;

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_compute_hash1", "compute HASH(1).\n"));
	YIPSDEBUG(DEBUG_DHASH, pvdump(res));

end:
	if (buf) vfree(buf);
	return(res);
}

/* %%%
 * Identity Protecion Exchange (Main Mode)
 */
/*
 * receive HDR*;ID;HASH from responder, and do QUICK mode.
 */
static int
isakmp_ident_i4(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_id *id;
	struct isakmp_pl_hash *hash;
	vchar_t *v_hash = 0;
	char *p, *r_hash;
	int tlen;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_ident_i4", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_ID) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_i4",
		    "received invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_ID);
		goto end;
	}

	id = (struct isakmp_pl_id *)((char *)isakmp + sizeof(*isakmp));
	if (id->h.np != ISAKMP_NPTYPE_HASH) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_i4",
		    "received invalid next payload type %d, expecting %d.\n",
		    id->h.np, ISAKMP_NPTYPE_HASH);
		goto end;
	}
	id->h.len = ntohs(id->h.len);

	hash = (struct isakmp_pl_hash *)((char *)id + id->h.len);
	if (hash->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_i4",
		    "received invalid next payload type %d, expecting %d.\n",
		    hash->h.np, ISAKMP_NPTYPE_NONE);
		goto end;
	}
	hash->h.len = ntohs(hash->h.len);

	/* validate HASH */
	r_hash = (char *)hash + sizeof(*hash);

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_ident_i4", "received HASH\n"));
	YIPSDEBUG(DEBUG_DHASH,
	    pdump(r_hash, hash->h.len - sizeof(*hash), YDUMP_HEX));

	/* save responder's id */
	if (isakmp_id2isa(iph1, id) < 0) goto end;

	/* compute HASH to validate */
	if ((v_hash = isakmp_compute_hash(iph1, VALIDATE)) == 0) goto end;
	if (memcmp(v_hash->v, r_hash, v_hash->l)) {
		plog("isakmp_ident_i4", "HASH mismatch.\n");
		goto end;
	}
	vfree(v_hash);

	/* XXX: compare address in stauts structure and address in ID payload */
	YIPSDEBUG(DEBUG_MISC,
		INET_NTOP(iph1->remote, _addr1_);
		plog("isakmp_ident_i4", "remote addr=%s.\n", _addr1_);
		plog("isakmp_ident_i4", "ID ");
		pdump(((caddr_t)id + sizeof(*id)),
			id->h.len - sizeof(*id), YDUMP_HEX));

	/* XXX: synchronization IV */
	memcpy(iph1->ivd->v, iph1->ive->v, iph1->iv->l);
	memcpy(iph1->iv->v, iph1->ive->v, iph1->iv->l);

	/* change status of isakmp status entry */
	iph1->status = ISAKMP_PH2_N;

	/* add to the schedule to expire, and seve back pointer. */
	iph1->sc = sched_add(iph1->isa->ldur, 0,
	                    1, isakmp_expire,
	                    (caddr_t)iph1, (caddr_t)0);

	plog2(from, "isakmp_ident_i4",
	    "established ISAKMP-SA, %s.\n", isakmp_pindex(&iph1->index, 0));

	error = 0;

end:
	if (error) {
		if (iph1) (void)isakmp_free_ph1(iph1);
	}
	return(error);
}

/*
 * receive HDR*;ID;HASH from initiator, and retrun HDR*;ID;HASH
 */
static int
isakmp_ident_r3(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	vchar_t *buf = 0;
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_id *id;
	struct isakmp_pl_hash *hash;
	vchar_t *v_hash = 0;
	char *p, *r_hash;
	int tlen;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_ident_r3", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_ID) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_r3",
		    "received invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_ID);
		goto end;
	}

	id = (struct isakmp_pl_id *)((char *)isakmp + sizeof(*isakmp));
	if (id->h.np != ISAKMP_NPTYPE_HASH) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_r3",
		    "received invalid next payload type %d, expecting %d.\n",
		    id->h.np, ISAKMP_NPTYPE_HASH);
		goto end;
	}
	id->h.len = ntohs(id->h.len);

	hash = (struct isakmp_pl_hash *)((char *)id + id->h.len);
	if (hash->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_r3",
		    "received invalid next payload type %d, expecting %d.\n",
		    hash->h.np, ISAKMP_NPTYPE_NONE);
		goto end;
	}
	hash->h.len = ntohs(hash->h.len);

	/* validate HASH */
	r_hash = (char *)hash + sizeof(*hash);

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_ident_r3", "received HASH\n"));
	YIPSDEBUG(DEBUG_DHASH,
	    pdump(r_hash, hash->h.len - sizeof(*hash), YDUMP_HEX));

	/* save initiator's id */
	if (isakmp_id2isa(iph1, id) < 0) goto end;

	/* compute HASH to validate */
	if ((v_hash = isakmp_compute_hash(iph1, VALIDATE)) == 0) goto end;
	if (memcmp(v_hash->v, r_hash, v_hash->l)) {
		plog2(from, "isakmp_ident_r3", "HASH mismatch.\n");
		goto end;
	}
	vfree(v_hash);

	/* XXX: compare address in stauts structure and address in ID payload */
	YIPSDEBUG(DEBUG_MISC,
		INET_NTOP(iph1->remote, _addr1_);
		plog("isakmp_ident_r3", "remote addr=%s.\n", _addr1_);
		plog("isakmp_ident_i4", "ID ");
		pdump(((caddr_t)id + sizeof(*id)),
			id->h.len - sizeof(*id), YDUMP_HEX));

	/* generate HASH to send */
	if ((iph1->hash = isakmp_compute_hash(iph1, GENERATE)) == 0) goto end;

	/* create HDR;ID;HASH payload */
	if ((buf = isakmp_ident_ir3mx(msg, from, iph1)) == 0) goto end;

	/* change status of isakmp status entry */
	iph1->status = ISAKMP_PH2_N;

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_ident_r3", "sendto HDR*;ID;HASH.\n"));

#if 0 /* XXX: How resend ? */
	/* add to the schedule to resend, and seve back pointer. */
	iph1->sc = sched_add(isakmp_timer, isakmp_resend_ph1,
	                    isakmp_try, isakmp_timeout_ph1,
	                    (caddr_t)iph1, (caddr_t)buf);
#else
	/* add to the schedule to expire, and seve back pointer. */
	iph1->sc = sched_add(iph1->isa->ldur, 0,
	                    1, isakmp_expire,
	                    (caddr_t)iph1, (caddr_t)0);

	plog2(from, "isakmp_ident_r3",
	    "established ISAKMP-SA, %s.\n", isakmp_pindex(&iph1->index, 0));

#endif

	error = 0;

end:
	if (error) {
		if (iph1) (void)isakmp_free_ph1(iph1);
	}
	return(error);
}

/*
 * receive HDR;KE;NONCE from responder, and retrun HDR*;ID;HASH
 */
static int
isakmp_ident_i3(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	vchar_t *buf = 0;
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_ke *ke;
	struct isakmp_pl_nonce *nonce;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_ident_i3", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_KE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_i3",
		    "received invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_KE);
		goto end;
	}

	ke = (struct isakmp_pl_ke *)((char *)isakmp + sizeof(*isakmp));
	if (ke->h.np != ISAKMP_NPTYPE_NONCE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_i3",
		    "received invalid next payload type %d, expecting %d.\n",
		    ke->h.np, ISAKMP_NPTYPE_NONCE);
		goto end;
	}
	ke->h.len = ntohs(ke->h.len);

	nonce = (struct isakmp_pl_nonce *)((char *)ke + ke->h.len);
	if (nonce->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_i3",
		    "received invalid next payload type %d, expecting %d.\n",
		    nonce->h.np, ISAKMP_NPTYPE_NONE);
		goto end;
	}
	nonce->h.len = ntohs(nonce->h.len);

	/* commit responder's ke, nonce for use */
	if (isakmp_kn2isa(iph1, ke, nonce) < 0) goto end;

	/* generate SKEYIDs & IV & final cipher key */
	if (isakmp_compute_skeyids(iph1) < 0) goto end;
	if (isakmp_compute_enckey(iph1) < 0) goto end;
	if (isakmp_compute_iv(iph1) < 0) goto end;

	/* generate HASH to send */
	if ((iph1->hash = isakmp_compute_hash(iph1, GENERATE)) == 0) goto end;

	/* create HDR;ID;HASH payload */
	if ((buf = isakmp_ident_ir3mx(msg, from, iph1)) == 0) goto end;

	/* change status of isakmp status entry */
	iph1->status = ISAKMP_PH1_3;

	/* add to the schedule to resend, and seve back pointer. */
	iph1->sc = sched_add(isakmp_timer, isakmp_resend_ph1,
	                    isakmp_try, isakmp_timeout_ph1,
	                    (caddr_t)iph1, (caddr_t)buf);

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_ident_i3", "sendto HDR*;KE;NONCE.\n"));

	error = 0;

end:
	if (error) {
		if (iph1) (void)isakmp_free_ph1(iph1);
	}
	return(error);
}

/*
 * receive HDR;KE;NONCE from initiator, and retrun HDR;KE;NONCE.
 */
static int
isakmp_ident_r2(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	vchar_t *buf = 0;
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_ke *ke;
	struct isakmp_pl_nonce *nonce;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_ident_r2", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_KE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_r2",
		    "received invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_KE);
		goto end;
	}

	ke = (struct isakmp_pl_ke *)((char *)isakmp + sizeof(*isakmp));
	if (ke->h.np != ISAKMP_NPTYPE_NONCE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_r2",
		    "received invalid next payload type %d, expecting %d.\n",
		    ke->h.np, ISAKMP_NPTYPE_NONCE);
		goto end;
	}
	ke->h.len = ntohs(ke->h.len);

	nonce = (struct isakmp_pl_nonce *)((char *)ke + ke->h.len);
	if (nonce->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_r2",
		    "received invalid next payload type %d, expecting %d.\n",
		    nonce->h.np, ISAKMP_NPTYPE_NONE);
		goto end;
	}
	nonce->h.len = ntohs(nonce->h.len);

	/* commit initiator's ke, nonce for use */
	if (isakmp_kn2isa(iph1, ke, nonce) < 0) goto end;

	/* generate DH public value */
	if (isakmp_dh_generate(iph1->isa->dhgrp,
		&iph1->dhp, &iph1->dhpub, &iph1->dhpriv) < 0) goto end;

	/* generate NONCE value */
	if ((iph1->nonce = eay_set_random(isakmp_nonce_size)) == 0) goto end;

	/* create HDR;KE;NONCE payload */
	if ((buf = isakmp_ident_ir2mx(msg, from, iph1)) == 0) goto end;

	/* change status of isakmp status entry */
	iph1->status = ISAKMP_PH1_2;

	/* add to the schedule to resend, and seve back pointer. */
	iph1->sc = sched_add(isakmp_timer, isakmp_resend_ph1,
	                    isakmp_try, isakmp_timeout_ph1,
	                    (caddr_t)iph1, (caddr_t)buf);

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_ident_r2", "sendto HDR;KE;NONCE.\n"));

	error = 0;

end:
	if (error) {
		if (iph1) (void)isakmp_free_ph1(iph1);
	}
	return(error);
}

/*
 * receive HDR;SA from responder, and retrun HDR;KE;NONCE.
 */
static int
isakmp_ident_i2(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	vchar_t *buf = 0;
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_sa *sa;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_ident_i2", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_SA) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_i2",
		    "received invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_SA);
		goto end;
	}

	sa = (struct isakmp_pl_sa *)((char *)isakmp + sizeof(*isakmp));
	if (sa->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_i2",
		    "received invalid next payload type %d, expecting %d.\n",
		    sa->h.np, ISAKMP_NPTYPE_NONE);
		goto end;
	}
	sa->h.len = ntohs(sa->h.len);

	/* check received proposals to use */
	/* XXX */

	/* modify index of the isakmp status record */
	memcpy((caddr_t)&iph1->index.r_ck, (caddr_t)&isakmp->r_ck,
		sizeof(cookie_t));

	/* commit sa for use */
	if ((iph1->isa =
	        CALLOC(sizeof(struct oakley_sa), struct oakley_sa *)) == 0) {
		plog("isakmp_ident_i2", "calloc (%s)\n", strerror(errno));
		goto end;
	}
	/* XXX: to be brunched off by situation. */
	if (ipsecdoi_sap2isa((caddr_t)iph1->isa, (struct ipsecdoi_sa *)sa) < 0) goto end; 

	/* save SA payload minus general header */
	if ((iph1->sa = vmalloc(sa->h.len - sizeof(struct isakmp_gen))) == 0) {
		plog("isakmp_ident_i2", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy(iph1->sa->v, (caddr_t)sa + sizeof(struct isakmp_gen),
		iph1->sa->l);

	/* generate DH public value */
	if (isakmp_dh_generate(iph1->isa->dhgrp,
		&iph1->dhp, &iph1->dhpub, &iph1->dhpriv) < 0) goto end;

	/* generate NONCE value */
	/* XXX: configurable */
	if ((iph1->nonce = eay_set_random(isakmp_nonce_size)) == 0) goto end;

	/* create HDR;KE;NONCE payload */
	if ((buf = isakmp_ident_ir2mx(msg, from, iph1)) == 0) goto end;

	/* change status of isakmp status entry */
	iph1->status = ISAKMP_PH1_2;

	/* add to the schedule to resend, and seve back pointer. */
	iph1->sc = sched_add(isakmp_timer, isakmp_resend_ph1,
	                    isakmp_try, isakmp_timeout_ph1,
	                    (caddr_t)iph1, (caddr_t)buf);

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_ident_i2", "sendto HDR;KE;NONCE.\n"));

	error = 0;

end:
	if (error) {
		if (iph1) (void)isakmp_free_ph1(iph1);
	}
	return(error);
}

/*
 * receive HDR;SA from initiator, and retrun appropreate HDR;SA.
 */
static int
isakmp_ident_r1(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	vchar_t *buf = 0;
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_sa *sa;
	isakmp_index index;
	char *p;
	int tlen;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_ident_r1", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_SA) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_r1",
		    "received invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_SA);
		goto end;
	}

	sa = (struct isakmp_pl_sa *)((char *)isakmp + sizeof(*isakmp));
	if (sa->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_ident_r1",
		    "received invalid next payload type %d, expecting %d.\n",
		    sa->h.np, ISAKMP_NPTYPE_NONE);
		goto end;
	}
	sa->h.len = ntohs(sa->h.len);

	/* check proposals and select one */
	/* XXX */

	/* commit sa for use */
	if ((iph1->isa =
	        CALLOC(sizeof(struct oakley_sa), struct oakley_sa *)) == 0) {
		plog("isakmp_ident_r1", "calloc (%s)\n", strerror(errno));
		goto end;
	}
	/* XXX: to be brunched off by situation. */
	if (ipsecdoi_sap2isa((caddr_t)iph1->isa, (struct ipsecdoi_sa *)sa) < 0) goto end; 

	/* save SA payload minus general header */
	if ((iph1->sa = vmalloc(sa->h.len - sizeof(struct isakmp_gen))) == 0) {
		plog("isakmp_ident_r1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy(iph1->sa->v, (caddr_t)sa + sizeof(struct isakmp_gen),
		iph1->sa->l);

	/* set pointer of my configuration */
	if ((iph1->cfp = isakmp_cfbypeer(from)) == &cftab) {
		if (iph1->cfp->remote == 0) {
			plog2(from, "isakmp_ident_r1",
				"couldn't find configuration.\n");
			goto end;
		}
	}

	/* set remote address */
	if ((iph1->remote =
	        (struct sockaddr *)malloc(sizeof(*iph1->remote))) == 0) {
		plog("isakmp_ident_r1", "malloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy((caddr_t)iph1->remote, (caddr_t)from,
		sizeof(*iph1->remote));

	/* save values into isakmp status */
	iph1->etype = iph1->cfp->ph[0]->etype;
	iph1->doi = iph1->cfp->ph[0]->sa.doi;
	iph1->sit = iph1->cfp->ph[0]->sa.sit;

	/* make ID payload into isakmp status */
	if ((iph1->id = iph1->cfp->ph[0]->idb) == 0) goto end;

	/* create buffer to send isakmp payload */
	tlen = sizeof(*isakmp) + sa->h.len; /* XXX */
	if ((buf = vmalloc(tlen)) == 0) { 
		plog("isakmp_ident_r1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/* create isakmp header */
	memcpy(buf->v, (caddr_t)isakmp, sizeof(*isakmp));

	/* set responder's cookie */
	isakmp = (struct isakmp *)buf->v;
	isakmp_set_cookie((char *)&isakmp->r_ck, from);

	/* modify index of the isakmp status record */
	memcpy((caddr_t)&iph1->index.r_ck, (caddr_t)&isakmp->r_ck,
		sizeof(cookie_t));

	isakmp->len = htonl(tlen);
	p = buf->v + sizeof(*isakmp);

	/* create ISAKMP-SA payload to reply */
	/* XXX */
	memcpy(p, (caddr_t)sa, sa->h.len);
	((struct isakmp_pl_sa *)p)->h.len = htons(sa->h.len);

	/* send HDR;SA to responder */
	if (isakmp_send(iph1, buf) < 0) goto end;

	/* change status of isakmp status entry */
	iph1->status = ISAKMP_PH1_1;

	/* add to the schedule to resend, and seve back pointer. */
	iph1->sc = sched_add(isakmp_timer, isakmp_resend_ph1,
	                     isakmp_try, isakmp_timeout_ph1,
	                     (caddr_t)iph1, (caddr_t)buf);

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_ident_r1", "sendto HDR;SA.\n"));

	error = 0;

end:
	if (error) {
		if (iph1) (void)isakmp_free_ph1(iph1);
		if (buf) vfree(buf);
	}
	return(error);
}

/*
 * begin Identity Protection Mode as initiator, and send HDR;SA to responder.
 */
static int
isakmp_ident_i1(msg, to, iph1)
	vchar_t *msg; /* must be null */
	struct sockaddr *to;
	struct isakmp_ph1 *iph1;
{
	vchar_t *buf = 0;
	struct isakmp *isakmp;
	struct isakmp_gen *gen;
	vchar_t *mysa;
	char *p;
	int tlen;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_ident_i1", "begin.\n"));

	/* set remote address */
	if ((iph1->remote =
	        (struct sockaddr *)malloc(sizeof(*iph1->remote))) == 0) {
		plog("isakmp_ident_i1", "malloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy((caddr_t)iph1->remote, (caddr_t)to, sizeof(*iph1->remote));

	/* save values into isakmp status */
	iph1->etype = iph1->cfp->ph[0]->etype;
	iph1->doi = iph1->cfp->ph[0]->sa.doi;
	iph1->sit = iph1->cfp->ph[0]->sa.sit;

	/* make ID payload into isakmp status */
	if ((iph1->id = iph1->cfp->ph[0]->idb) == 0) goto end;

	/* create SA payload */
	if ((mysa = ipsecdoi_make_mysa(&iph1->cfp->ph[0]->sa, 0)) < 0) goto end;

	/* create buffer to send isakmp payload */
	tlen = sizeof(*isakmp) + mysa->l;
	if ((buf = vmalloc(tlen)) == 0) {
		plog("isakmp_ident_i1", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/* create isakmp header */
	isakmp = (struct isakmp *)buf->v;
	memcpy((caddr_t)&isakmp->i_ck, (caddr_t)&iph1->index.i_ck,
		sizeof(cookie_t));
	isakmp->np    = ISAKMP_NPTYPE_SA;
	isakmp->v_maj = ISAKMP_MAJOR_VERSION;
	isakmp->v_min = ISAKMP_MINOR_VERSION;
	isakmp->etype = ISAKMP_ETYPE_IDENT;
	isakmp->flags = 0;
	memcpy((caddr_t)&isakmp->msgid, msgid0, sizeof(isakmp->msgid));

	isakmp->len = htonl(tlen);
	p = buf->v + sizeof(*isakmp);

	/* create ISAKMP-SA payload to propose */
	memcpy(p, mysa->v, mysa->l);

	/* send HDR;SA to responder */
	if (isakmp_send(iph1, buf) < 0) goto end;

	/* change status of isakmp status entry */
	iph1->status = ISAKMP_PH1_1;

	/* add to the schedule to resend, and seve back pointer. */
	iph1->sc = sched_add(isakmp_timer, isakmp_resend_ph1,
	                    isakmp_try, isakmp_timeout_ph1,
	                    (caddr_t)iph1, (caddr_t)buf);

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_ident_i1", "sendto HDR;SA.\n"));

	error = 0;

end:
	if (error) {
		if (iph1) (void)isakmp_free_ph1(iph1);
		if (buf) vfree(buf);
	}
	if (mysa) vfree(mysa);

	return(error);
}

/*
 * create ID, HASH payload with isakmp header.
 */
static vchar_t *
isakmp_ident_ir3mx(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	vchar_t *buf = 0, *new = 0;
	char *p;
	int tlen;
	struct isakmp_gen *gen;
	int error = -1;

	/* create buffer */
	tlen = sizeof(*isakmp)
	     + sizeof(*gen) + iph1->id->l
	     + sizeof(*gen) + iph1->hash->l;

	if ((buf = vmalloc(tlen)) == 0) {
		plog("isakmp_ident_ir3mx", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/* create isakmp header */
	memcpy(buf->v, (caddr_t)isakmp, sizeof(*isakmp));
	isakmp = (struct isakmp *)buf->v;
	isakmp->np = ISAKMP_NPTYPE_ID;
	isakmp->len = htonl(tlen);
	isakmp->flags = ISAKMP_FLAG_E;
	p = buf->v + sizeof(*isakmp);

	/* create isakmp ID payload */
	gen = (struct isakmp_gen *)p;
	gen->np = ISAKMP_NPTYPE_HASH;
	gen->len = htons(sizeof(*gen) + iph1->id->l);
	p += sizeof(*gen);

	memcpy(p, iph1->id->v, iph1->id->l);
	p += iph1->id->l;

	/* create isakmp HASH payload */
	gen = (struct isakmp_gen *)p;
	gen->np = ISAKMP_NPTYPE_NONE;
	gen->len = htons(sizeof(*gen) + iph1->hash->l);
	p += sizeof(*gen);

	memcpy(p, iph1->hash->v, iph1->hash->l);
	p += iph1->hash->l;

	/* encoding */
	if ((new = isakmp_do_encrypt(iph1, buf, iph1->ive, iph1->iv)) == 0)
		goto end;

	vfree(buf);

	buf = new;

	/* send HDR;KE;NONCE to responder */
	if (isakmp_send(iph1, buf) < 0) goto end;

	/* XXX: synchronization IV */
	memcpy(iph1->ive->v, iph1->iv->v, iph1->iv->l);
	memcpy(iph1->ivd->v, iph1->iv->v, iph1->iv->l);

	error = 0;

end:
	if (error && buf) {
		vfree(buf);
		buf = 0;
	}

	return(buf);
}

/*
 * create KE, NONCE payload with isakmp header.
 */
static vchar_t *
isakmp_ident_ir2mx(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	vchar_t *buf = 0;
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_gen *gen;
	char *p;
	int tlen;
	int error = -1;

	/* create buffer */
	tlen = sizeof(*isakmp)
	     + sizeof(*gen) + iph1->dhpub->l
	     + sizeof(*gen) + iph1->nonce->l;

	if ((buf = vmalloc(tlen)) == 0) {
		plog("isakmp_ident_ir2mx", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = buf->v;

	/* create isakmp header */
	memcpy(p, (caddr_t)isakmp, sizeof(*isakmp));
	isakmp = (struct isakmp *)buf->v;
	isakmp->np = ISAKMP_NPTYPE_KE;
	isakmp->len = htonl(tlen);
	p += sizeof(*isakmp);

	/* create isakmp KE payload */
	gen = (struct isakmp_gen *)p;
	gen->np = ISAKMP_NPTYPE_NONCE;
	gen->len = htons(sizeof(*gen) + iph1->dhpub->l);
	p += sizeof(*gen);

	memcpy(p, iph1->dhpub->v, iph1->dhpub->l);
	p += iph1->dhpub->l;

	/* create isakmp NONCE payload */
	gen = (struct isakmp_gen *)p;
	gen->np = ISAKMP_NPTYPE_NONE;
	gen->len = htons(sizeof(*gen) + iph1->nonce->l);
	p += sizeof(*gen);

	memcpy(p, iph1->nonce->v, iph1->nonce->l);

	/* send HDR;KE;NONCE to responder */
	if (isakmp_send(iph1, buf) < 0) goto end;

	error = 0;

end:
	if (error) {
		if (buf) vfree(buf);
		buf = 0;
	}
	return(buf);
}

/*
 * compute skeyids
 * see seciton 5. Exchanges in isakmp-oakley-05.txt
 */
static int
isakmp_compute_skeyids(iph1)
	struct isakmp_ph1 *iph1;
{
	vchar_t *key, *buf = 0, *bp;
	char *p;
	int len;
	int error = -1;

	/* compute sharing secret of DH */
	if (isakmp_dh_compute(iph1->isa->dhgrp, iph1->dhp, iph1->dhpub,
			iph1->dhpriv, iph1->dhpub_p, &iph1->dhgxy) < 0)
		goto end;

	/* SKEYID */
	switch(iph1->isa->auth_t) {
	case OAKLEY_ATTR_AUTH_METHOD_PSKEY:
		/* SKEYID = prf(pre-shared-key, Ni_b | Nr_b) */
		if ((key = iph1->cfp->ph[0]->pskey) == 0) {
			plog("isakmp_compute_skeyids", "couldn't find pskey.\n");
			goto end;
		}
		YIPSDEBUG(DEBUG_MISC, plog("isakmp_compute_skeyids", "get secret.\n"));
		YIPSDEBUG(DEBUG_DMISC, pvdump(key));

		len = iph1->nonce->l + iph1->nonce_p->l;
		if ((buf = vmalloc(len)) == 0) {
			plog("isakmp_compute_skeyids", "vmalloc (%s)\n", strerror(errno));
			goto end;
		}
		p = buf->v;

		bp = (iph1->dir == INITIATOR ? iph1->nonce : iph1->nonce_p);
		memcpy(p, bp->v, bp->l);
		p += bp->l;

		bp = (iph1->dir == INITIATOR ? iph1->nonce_p : iph1->nonce);
		memcpy(p, bp->v, bp->l);
		p += bp->l;

		if ((iph1->skeyid = isakmp_prf(key, buf, iph1)) == 0) goto end;
		break;

	case OAKLEY_ATTR_AUTH_METHOD_DSS:
		/* SKEYID = prf(Ni_b | Nr_b, g^xy) */
	case OAKLEY_ATTR_AUTH_METHOD_RSA:
		/* SKEYID = prf(hash(Ni_b | Nr_b), CKY-I | CKY-R) */
	case OAKLEY_ATTR_AUTH_METHOD_RSAENC:
	case OAKLEY_ATTR_AUTH_METHOD_RSAREV:
		plog("isakmp_compute_skeyids",
		    "authentication method %d isn't supported\n", iph1->isa->auth_t);
		break;
	default:
		break;
	}

	vfree(buf);
	buf = 0;

	YIPSDEBUG(DEBUG_SKEYID,
	    plog("isakmp_compute_skeyids", "compute SKEYID.\n"));
	YIPSDEBUG(DEBUG_DSKEYID, pvdump(iph1->skeyid));

	/* SKEYID D */
	/* SKEYID_d = prf(SKEYID, g^xy | CKY-I | CKY-R | 0) */
	len = iph1->dhgxy->l + sizeof(cookie_t) * 2 + 1;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_skeyids", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = buf->v;

	memcpy(p, iph1->dhgxy->v, iph1->dhgxy->l);
	p += iph1->dhgxy->l;
	memcpy(p, (caddr_t)&iph1->index.i_ck, sizeof(cookie_t));
	p += sizeof(cookie_t);
	memcpy(p, (caddr_t)&iph1->index.r_ck, sizeof(cookie_t));
	p += sizeof(cookie_t);
	*p = 0;
	if ((iph1->skeyid_d = isakmp_prf(iph1->skeyid, buf, iph1)) == 0) goto end;

	vfree(buf);
	buf = 0;

	YIPSDEBUG(DEBUG_SKEYID,
	    plog("isakmp_compute_skeyids", "compute SKEYID_d.\n"));
	YIPSDEBUG(DEBUG_DSKEYID, pvdump(iph1->skeyid_d));

	/* SKEYID A */
	/* SKEYID_a = prf(SKEYID, SKEYID_d | g^xy | CKY-I | CKY-R | 1) */
	len = iph1->skeyid_d->l + iph1->dhgxy->l + sizeof(cookie_t) * 2 + 1;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_skeyids", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = buf->v;
	memcpy(p, iph1->skeyid_d->v, iph1->skeyid_d->l);
	p += iph1->skeyid_d->l;
	memcpy(p, iph1->dhgxy->v, iph1->dhgxy->l);
	p += iph1->dhgxy->l;
	memcpy(p, (caddr_t)&iph1->index.i_ck, sizeof(cookie_t));
	p += sizeof(cookie_t);
	memcpy(p, (caddr_t)&iph1->index.r_ck, sizeof(cookie_t));
	p += sizeof(cookie_t);
	*p = 1;
	if ((iph1->skeyid_a = isakmp_prf(iph1->skeyid, buf, iph1)) == 0) goto end;

	vfree(buf);
	buf = 0;

	YIPSDEBUG(DEBUG_SKEYID,
	    plog("isakmp_compute_skeyids", "compute SKEYID_a.\n"));
	YIPSDEBUG(DEBUG_DSKEYID, pvdump(iph1->skeyid_a));

	/* SKEYID_e = prf(SKEYID, SKEYID_a | g^xy | CKY-I | CKY-R | 2) */
	len = iph1->skeyid_a->l + iph1->dhgxy->l + sizeof(cookie_t) * 2 + 1;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_skeyids", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = buf->v;
	memcpy(p, iph1->skeyid_a->v, iph1->skeyid_a->l);
	p += iph1->skeyid_a->l;
	memcpy(p, iph1->dhgxy->v, iph1->dhgxy->l);
	p += iph1->dhgxy->l;
	memcpy(p, (caddr_t)&iph1->index.i_ck, sizeof(cookie_t));
	p += sizeof(cookie_t);
	memcpy(p, (caddr_t)&iph1->index.r_ck, sizeof(cookie_t));
	p += sizeof(cookie_t);
	*p = 2;
	if ((iph1->skeyid_e = isakmp_prf(iph1->skeyid, buf, iph1)) == 0) goto end;

	vfree(buf);
	buf = 0;

	YIPSDEBUG(DEBUG_SKEYID,
	    plog("isakmp_compute_skeyids", "compute SKEYID_e.\n"));
	YIPSDEBUG(DEBUG_DSKEYID, pvdump(iph1->skeyid_e));

	error = 0;

end:
	if (buf) vfree(buf);
	return(error);
}

/*
 * compute final encryption key.
 * see Appendix B.
 */
static int
isakmp_compute_enckey(iph1)
	struct isakmp_ph1 *iph1;
{
	char *p;
	u_int keylen, prflen;
	vchar_t *buf = 0, *res, *bp;
	int cplen, reslen;
	int error = -1;

	switch (iph1->isa->enc_t) {
	case OAKLEY_ATTR_ENC_ALG_DES:
		keylen = 8;
		break;
	case OAKLEY_ATTR_ENC_ALG_3DES:
		keylen = 24;
		break;
	case OAKLEY_ATTR_ENC_ALG_IDEA:
	case OAKLEY_ATTR_ENC_ALG_BL:
	case OAKLEY_ATTR_ENC_ALG_RC5:
	case OAKLEY_ATTR_ENC_ALG_CAST:
	default:
		plog("isakmp_compute_keye",
		    "encryption algoritym %d isn't supported.\n", iph1->isa->enc_t);
		goto end;
	}

	switch (iph1->isa->prf_t) {
	default:
		switch (iph1->isa->hash_t) {
		case OAKLEY_ATTR_HASH_ALG_MD5:
			prflen = 16;
			break;
		case OAKLEY_ATTR_HASH_ALG_SHA:
			prflen = 20;
			break;
		default:
			plog("isakmp_compute_enckey",
			    "hash type %d isn't supported.\n", iph1->isa->hash_t);
			return(0);
			break;
		}
	}

	if ((iph1->key = vmalloc(keylen)) == 0) {
		plog("isakmp_compute_enckey", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = iph1->key->v;

	if (prflen < keylen) {
		/*
		 * i.g.
		 * Ka = K1 | K2 | K3
		 * and
		 * K1 = prf(SKEYID_e, 0)
		 * K2 = prf(SKEYID_e, K1)
		 * K3 = prf(SKEYID_e, K2)
		 */
		if ((buf = vmalloc(prflen)) == 0) {
			plog("isakmp_compute_enckey", "vmalloc (%s)\n", strerror(errno));
			goto end;
		}
		buf->v[0] = 0;
		buf->l = 1;

		reslen = keylen;

		for (reslen = keylen; reslen > 0; reslen -= prflen) {
			if ((res = isakmp_prf(iph1->skeyid_e, buf, iph1)) == 0) goto end;

			cplen = ((reslen - prflen) ? prflen : reslen);
			memcpy(p, res->v, cplen);
			p += cplen;

			memcpy(buf->v, res->v, prflen);
			vfree(res);
		}
	} else
		memcpy(p, iph1->skeyid_e->v, keylen);

	/* des weakkey check */
	switch (iph1->isa->enc_t) {
	case OAKLEY_ATTR_ENC_ALG_DES:
		if (isakmp_check_des_weakkey(iph1->key) < 0) {
			plog("isakmp_compute_enckey", "weak key generated.\n");
			goto end;
		}
		break;
	case OAKLEY_ATTR_ENC_ALG_3DES:
		if (isakmp_check_des_weakkey(iph1->key) < 0) {
			plog("isakmp_compute_enckey", "weak key generated.\n");
			goto end;
		}

		iph1->key->v += 8;
		iph1->key->l -= 8;
		if (isakmp_check_des_weakkey(iph1->key) < 0) {
			plog("isakmp_compute_enckey", "weak key generated.\n");
			goto end;
		}

		iph1->key->v += 8;
		iph1->key->l -= 8;
		if (isakmp_check_des_weakkey(iph1->key) < 0) {
			plog("isakmp_compute_enckey", "weak key generated.\n");
			goto end;
		}

		iph1->key->v -= 16;
		iph1->key->l += 16;
		break;
	}

	YIPSDEBUG(DEBUG_CRYPT,
	    plog("isakmp_compute_enckey", "compute final cipher key.\n"));
	YIPSDEBUG(DEBUG_DCRYPT, pvdump(iph1->key));

	error = 0;

end:
	if (buf) vfree(buf);

	return(error);
}

/*
 * compute IV
 * see Appendix B in isakmp-oakley-05.
 */
static int
isakmp_compute_iv(iph1)
	struct isakmp_ph1 *iph1;
{
	vchar_t *buf = 0, *bp;
	char *p;
	int len;
	int error = -1;

	len = iph1->dhpub->l + iph1->dhpub_p->l;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_iv", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/*
	 * IV = hash(g^xi | g^xr)
	 */
	p = buf->v;

	bp = (iph1->dir == INITIATOR ? iph1->dhpub : iph1->dhpub_p);
	memcpy(p, bp->v, bp->l);
	p += bp->l;

	bp = (iph1->dir == INITIATOR ? iph1->dhpub_p : iph1->dhpub);
	memcpy(p, bp->v, bp->l);
	p += bp->l;

	/* compute IV */
	if ((iph1->iv = isakmp_hash(buf, iph1)) == 0) goto end;

	/* adjust length of iv */
	iph1->iv->l = DESBLOCKLEN;

	/* create buffer to save iv */
	if ((iph1->ive = vdup(iph1->iv)) == 0
	 || (iph1->ivd = vdup(iph1->iv)) == 0) {
		plog("isakmp_compute_iv", "vdup (%s)\n", strerror(errno));
		goto end;
	}

	error = 0;

	YIPSDEBUG(DEBUG_CRYPT, plog("isakmp_compute_iv", "compute IV.\n"));
	YIPSDEBUG(DEBUG_DCRYPT, pvdump(iph1->iv));

end:
	if (buf) vfree(buf);
	return(0);
}

/*
 * compute IV for Phase 2
 * see Appendix B in isakmp-oakley-05.
 */
static int
isakmp_compute_iv2(iph2)
	struct isakmp_ph2 *iph2;
{
	struct isakmp_ph1 *iph1 = iph2->ph1;
	vchar_t *buf = 0;
	char *p;
	int len;
	int error = -1;

	/* create buffer */
	len = iph1->iv->l + sizeof(msgid_t);
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_iv2", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/*
	 * IV = hash(last CBC block of Phase 1 | M-ID)
	 */
	p = buf->v;

	memcpy(p, iph1->iv->v, iph1->iv->l);
	p += iph1->iv->l;

	memcpy(p, (caddr_t)&iph2->msgid, sizeof(msgid_t));

	/* compute IV */
	if ((iph2->iv = isakmp_hash(buf, iph1)) == 0) goto end;

	/* create buffer to save new iv */
	if ((iph2->ive = vdup(iph2->iv)) == 0
	 || (iph2->ivd = vdup(iph2->iv)) == 0) {
		plog("isakmp_compute_iv2", "vdup (%s)\n", strerror(errno));
		goto end;
	}

	error = 0;

	YIPSDEBUG(DEBUG_CRYPT,
	    plog("isakmp_compute_iv2", "compute IV for Phase 2.\n"));
	YIPSDEBUG(DEBUG_DCRYPT, pvdump(iph2->iv));

end:
	if (buf) vfree(buf);
	return(error);
}

/*
 * compute HASH
 *   see seciton 5. Exchanges in isakmp-oakley-05.
 */
static vchar_t *
isakmp_compute_hash(iph1, sw)
	struct isakmp_ph1 *iph1;
	int sw;
{
	vchar_t *buf = 0, *res = 0, *bp;
	char *p, *bp2;
	int len, bl;
	int error = -1;

	/* create buffer */
	len = iph1->dhpub->l + iph1->dhpub_p->l
	       + sizeof(cookie_t) * 2 + iph1->sa->l + iph1->id->l;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_compute_hash", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	/*
	 * HASH_I = prf(SKEYID, g^xi | g^xr | CKY-I | CKY-R | SAi_b | IDii_b )
	 * HASH_R = prf(SKEYID, g^xr | g^xi | CKY-R | CKY-I | SAi_b | IDir_b )
	 */
	p = buf->v;

	bp = (sw == GENERATE ? iph1->dhpub : iph1->dhpub_p);
	memcpy(p, bp->v, bp->l);
	p += bp->l;

	bp = (sw == GENERATE ? iph1->dhpub_p : iph1->dhpub);
	memcpy(p, bp->v, bp->l);
	p += bp->l;

	if (iph1->dir == INITIATOR)
		bp2 = (sw == GENERATE ?
		      (char *)&iph1->index.i_ck : (char *)&iph1->index.r_ck);
	else
		bp2 = (sw == GENERATE ?
		      (char *)&iph1->index.r_ck : (char *)&iph1->index.i_ck);
	bl = sizeof(cookie_t);
	memcpy(p, bp2, bl);
	p += bl;

	if (iph1->dir == INITIATOR)
		bp2 = (sw == GENERATE ?
		      (char *)&iph1->index.r_ck : (char *)&iph1->index.i_ck);
	else
		bp2 = (sw == GENERATE ?
		      (char *)&iph1->index.i_ck : (char *)&iph1->index.r_ck);
	bl = sizeof(cookie_t);
	memcpy(p, bp2, bl);
	p += bl;

	bp = iph1->sa;
	memcpy(p, bp->v, bp->l);
	p += bp->l;

	bp = (sw == GENERATE ? iph1->id : iph1->id_p);
	memcpy(p, bp->v, bp->l);

	/* compute HASH */
	if ((res = isakmp_prf(iph1->skeyid, buf, iph1)) == 0) goto end;

	error = 0;

	YIPSDEBUG(DEBUG_HASH, plog("isakmp_compute_hash", "compute HASH.\n"));
	YIPSDEBUG(DEBUG_DHASH, pvdump(res));

end:
	if (buf) vfree(buf);
	return(res);
}

/*
 * PRF
 */
static vchar_t *
isakmp_prf(key, buf, iph1)
	vchar_t *key, *buf;
	struct isakmp_ph1 *iph1;
{
	int error = 0;
	vchar_t *res;

	switch (iph1->isa->prf_t) {
	default:
		switch (iph1->isa->hash_t) {
		case OAKLEY_ATTR_HASH_ALG_MD5:
			res = eay_hmacmd5_one(key, buf);
			break;
		case OAKLEY_ATTR_HASH_ALG_SHA:
			res = eay_hmacsha1_one(key, buf);
			break;
		default:
			plog("isakmp_prf",
			    "hash type %d isn't supported.\n", iph1->isa->hash_t);
			return(0);
			break;
		}
	}

	return(res);
}

/*
 * hash
 */
static vchar_t *
isakmp_hash(buf, iph1)
	vchar_t *buf;
	struct isakmp_ph1 *iph1;
{
	int error = 0;
	vchar_t *res;

	switch (iph1->isa->prf_t) {
	default:
		switch (iph1->isa->hash_t) {
		case OAKLEY_ATTR_HASH_ALG_MD5:
			res = eay_md5_one(buf);
			break;
		case OAKLEY_ATTR_HASH_ALG_SHA:
			res = eay_sha1_one(buf);
			break;
		default:
			plog("isakmp_hash",
			    "hash type %d isn't supported.\n", iph1->isa->hash_t);
			return(0);
			break;
		}
	}

	return(res);
}

/*
 * compute sharing secret of DH
 * IN:	grp, *dhp, *dhpub, *dhpriv, *dhpub_p, *dhgxy
 */
static int
isakmp_dh_compute(grp, prime, pub, priv, pub_p, gxy)
	u_int grp;
	vchar_t *prime, *pub, *priv, *pub_p, **gxy;
{
	int error = -1;

	if ((*gxy = vmalloc(prime->l)) == 0) {
		plog("isakmp_dh_compute", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	switch (grp) {
	case OAKLEY_ATTR_GRP_DESC_MODP768:
        if (eay_dh_compute(prime, 2, pub, priv, pub_p, gxy) < 0) {
			goto end;
		}
		break;

	case OAKLEY_ATTR_GRP_DESC_MODP1024:
        if (eay_dh_compute(prime, 2, pub, priv, pub_p, gxy) < 0) {
			goto end;
		}
		break;

	case OAKLEY_ATTR_GRP_DESC_EC2N155:
	case OAKLEY_ATTR_GRP_DESC_EC2N185:
	default:
		plog("isakmp_dh_compute",
		    "dh group %d isn't supported.\n", grp);
		goto end;
	}

	error = 0;

	YIPSDEBUG(DEBUG_DH,
	    plog("isakmp_dh_compute", "compute DH's shared.\n"));
	YIPSDEBUG(DEBUG_DDH, pvdump(*gxy));

end:
	return(error);
}

/*
 * generate values of DH
 * IN:	grp, **dhp, *dhpub, *dhpriv
 */
static int
isakmp_dh_generate(grp, prime, pub, priv)
	u_int grp;
	vchar_t **prime, **pub, **priv;
{
	int error = -1;

	switch (grp) {
	case OAKLEY_ATTR_GRP_DESC_MODP768:
		*prime = &oakley_prime768;
		if (eay_dh_generate(*prime, 2, 0, pub, priv) < 0) {
			goto end;
		}
		break;
	case OAKLEY_ATTR_GRP_DESC_MODP1024:
		*prime = &oakley_prime1024;
		if (eay_dh_generate(*prime, 2, 0, pub, priv) < 0) {
			goto end;
		}
		break;
	case OAKLEY_ATTR_GRP_DESC_EC2N155:
	case OAKLEY_ATTR_GRP_DESC_EC2N185:
	default:
		plog("isakmp_dh_generate",
		    "dhgrp description %d isn't supported.\n", grp);
		goto end;
	}

	error = 0;

	YIPSDEBUG(DEBUG_DH,
	    plog("isakmp_dh_generate", "compute DH's private.\n"));
	YIPSDEBUG(DEBUG_DDH, pvdump(*priv));

	YIPSDEBUG(DEBUG_DH,
	    plog("isakmp_dh_generate", "compute DH's public.\n"));
	YIPSDEBUG(DEBUG_DDH, pvdump(*pub));

end:
	return(error);
}

/*
 * decrypt packet.
 *   save new iv and old iv.
 */
static vchar_t *
isakmp_do_decrypt(iph1, msg, ivdp, ivep)
	struct isakmp_ph1 *iph1;
	vchar_t *msg, *ivdp, *ivep;
{
	vchar_t *buf = 0, *new = 0;
	char *pl;
	caddr_t ks1 = 0, ks2 = 0, ks3 = 0;
	int len;
	u_int8 padlen;
	int error = -1;

	YIPSDEBUG(DEBUG_CRYPT,
	    plog("isakmp_do_decrypt", "begin decryption.\n"));

	/* save new and iv */
	memset(ivep->v, 0, ivep->l);
	memcpy(ivep->v, (caddr_t)&msg->v[msg->l - DESBLOCKLEN], DESBLOCKLEN);

	YIPSDEBUG(DEBUG_CRYPT,
	    plog("isakmp_do_decrypt", "save IV for next.\n"));
	YIPSDEBUG(DEBUG_DCRYPT, pvdump(ivep));

	pl = msg->v + sizeof(struct isakmp);

	len = msg->l - sizeof(struct isakmp);

	/* create buffer */
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_do_decrypt", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy(buf->v, pl, len);

	/* do decrypt */
	switch (iph1->isa->enc_t) {
	case OAKLEY_ATTR_ENC_ALG_DES:
		if ((ks1 = eay_crypto_des_set_key(iph1->key->v)) == 0) goto end;
		if ((new = eay_des_decrypt(buf, ks1, ivdp->v)) == 0) goto end;
		break;
	case OAKLEY_ATTR_ENC_ALG_3DES:
		if ((ks1 = eay_crypto_des_set_key(iph1->key->v)) == 0) goto end;
		if ((ks2 = eay_crypto_des_set_key(iph1->key->v + 8)) == 0) goto end;
		if ((ks3 = eay_crypto_des_set_key(iph1->key->v + 16)) == 0) goto end;
		if ((new = eay_3des_decrypt(buf, ks1, ks2, ks3, ivdp->v)) == 0) goto end;
		break;
	case OAKLEY_ATTR_ENC_ALG_IDEA:
	case OAKLEY_ATTR_ENC_ALG_BL:
	case OAKLEY_ATTR_ENC_ALG_RC5:
	case OAKLEY_ATTR_ENC_ALG_CAST:
	default:
		plog("isakmp_do_decrypt",
		    "encryption algoritym %d isn't supported.\n", iph1->isa->enc_t);
		goto end;
	}

	vfree(buf);
	buf = 0;

	YIPSDEBUG(DEBUG_CRYPT,
	    plog("isakmp_do_decrypt", "decrypted payload, but not trimed.\n"));
	YIPSDEBUG(DEBUG_DCRYPT, pvdump(new));

	YIPSDEBUG(DEBUG_DCRYPT,
	    plog("isakmp_do_decrypt", "using IV,\n");
	    pvdump(ivdp));

	/* trim padding */
	if (isakmp_pad_check) {

		if (isakmp_pad_exclone)
			padlen = new->v[new->l - 1] + 1;
		else
			padlen = new->v[new->l - 1];

		if (padlen > new->l) {
			plog("isakmp_do_decrypt",
			    "invalied padding len=%u, buflen=%u.\n", padlen, new->l);
			YIPSDEBUG(DEBUG_DCRYPT, pvdump(new));
			goto end;
		}
		new->l -= padlen;

		YIPSDEBUG(DEBUG_CRYPT,
		    plog("isakmp_do_decrypt", "trimmed padding len=%u\n", padlen));

	} else {
		YIPSDEBUG(DEBUG_CRYPT,
		    plog("isakmp_do_decrypt", "skip to trim padding.\n"));
		;
	}

	/* create new buffer */
	len = sizeof(struct isakmp) + new->l;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_do_decrypt", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy(buf->v, msg->v, sizeof(struct isakmp));
	memcpy(buf->v + sizeof(struct isakmp), new->v, new->l);
	((struct isakmp *)buf->v)->len = htonl(buf->l);

	YIPSDEBUG(DEBUG_CRYPT, plog("isakmp_do_decrypt", "decrypted.\n"));
	YIPSDEBUG(DEBUG_DCRYPT, pvdump(buf));

	error = 0;

end:
	if (error && buf) {
		vfree(buf);
		buf = 0;
	}
	if (new) vfree(new);
	if (ks1) (void)free(ks1);
	if (ks2) (void)free(ks2);
	if (ks3) (void)free(ks3);

	return(buf);
}

/*
 * encrypt packet.
 *   ph: ISAKMP_PH1 or ISAKMP_PH2
 */
static vchar_t *
isakmp_do_encrypt(iph1, msg, ivep, ivp)
	struct isakmp_ph1 *iph1;
	vchar_t *msg, *ivep, *ivp;
{
	vchar_t *buf = 0, *new = 0;
	char *pl;
	caddr_t ks1 = 0, ks2 = 0, ks3 = 0;
	int len;
	u_int padlen;
	int error = -1;

	YIPSDEBUG(DEBUG_CRYPT,
	    plog("isakmp_do_encrypt", "begin encryption.\n"));
	YIPSDEBUG(DEBUG_DCRYPT, pvdump(msg));

	pl = msg->v + sizeof(struct isakmp);
	len = msg->l - sizeof(struct isakmp);

	/* add padding */
	padlen = isakmp_padlen(len);
	YIPSDEBUG(DEBUG_CRYPT,
	    plog("isakmp_do_encrypt", "pad length = %u\n", padlen));

	/* create buffer */
	if ((buf = vmalloc(len + padlen)) == 0) {
		plog("isakmp_do_encrypt", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy(buf->v, pl, len);

	/* make pad into tail */
	if (isakmp_pad_exclone)
		buf->v[len + padlen - 1] = padlen - 1;
	else
		buf->v[len + padlen - 1] = padlen;

	/* do encrypt */
	switch (iph1->isa->enc_t) {
	case OAKLEY_ATTR_ENC_ALG_DES:
		if ((ks1 = eay_crypto_des_set_key(iph1->key->v)) == 0) goto end;
		if ((new = eay_des_encrypt(buf, ks1, ivep->v)) == 0) goto end;
		break;

	case OAKLEY_ATTR_ENC_ALG_3DES:
		if ((ks1 = eay_crypto_des_set_key(iph1->key->v)) == 0) goto end;
		if ((ks2 = eay_crypto_des_set_key(iph1->key->v + 8)) == 0) goto end;
		if ((ks3 = eay_crypto_des_set_key(iph1->key->v + 16)) == 0) goto end;
		if ((new = eay_3des_encrypt(buf, ks1, ks2, ks3, ivep->v)) == 0) goto end;
		break;

	case OAKLEY_ATTR_ENC_ALG_IDEA:
	case OAKLEY_ATTR_ENC_ALG_BL:
	case OAKLEY_ATTR_ENC_ALG_RC5:
	case OAKLEY_ATTR_ENC_ALG_CAST:
	default:
		plog("isakmp_do_encrypt",
		    "encryption algoritym %d isn't supported.\n", iph1->isa->enc_t);
		goto end;
	}

	YIPSDEBUG(DEBUG_DCRYPT,
	    plog("isakmp_do_encrypt", "encrypted payload, using IV,\n");
	    pvdump(ivep));

	vfree(buf);

	/* save as next iv */
	memset(ivp->v, 0, ivp->l);
	memcpy(ivp->v, (caddr_t)&new->v[new->l - DESBLOCKLEN], DESBLOCKLEN);

	YIPSDEBUG(DEBUG_CRYPT,
	    plog("isakmp_do_encrypt", "save IV for next.\n"));
	YIPSDEBUG(DEBUG_DCRYPT, pvdump(ivp));

	/* create new buffer */
	len = sizeof(struct isakmp) + new->l;
	if ((buf = vmalloc(len)) == 0) {
		plog("isakmp_do_encrypt", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	memcpy(buf->v, msg->v, sizeof(struct isakmp));
	memcpy(buf->v + sizeof(struct isakmp), new->v, new->l);
	((struct isakmp *)buf->v)->len = htonl(buf->l);

	error = 0;

	YIPSDEBUG(DEBUG_CRYPT, plog("isakmp_do_encrypt", "encrypted.\n"));

end:
	if (error && buf) {
		vfree(buf);
		buf = 0;
	}
	if (new) vfree(new);
	if (ks1) (void)free(ks1);
	if (ks2) (void)free(ks2);
	if (ks3) (void)free(ks3);

	return(buf);
}

/* culculate padding length */
static int
isakmp_padlen(len)
	int len;
{
	int padlen;
	int base = DESBLOCKLEN;

	padlen = 8 - len % 8;

	if (isakmp_randpad)
		padlen += ((random() % isakmp_pad_lword + 1 ) * base);

	return(padlen);
}

/*
 * calculate cookie and set.
 */
static int
isakmp_set_cookie(place, to)
	char *place;
	struct sockaddr *to;
{
	vchar_t *buf, *buf2;
	char *p;
	int blen;
	time_t t;
	int error = -1;

	blen = (_ALENBYAF(myaddr[0]->sa_family)
		+ sizeof(u_short)) * 2		/* port */
		+ sizeof(time_t) + local_secret_size;
	if ((buf = vmalloc(blen)) == 0) {
		plog("isakmp_set_cookie", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}
	p = buf->v;

	/* copy my address */
	memcpy(p, _SADDRBYAF(myaddr[0]), _ALENBYAF(myaddr[0]->sa_family));
	p += _ALENBYAF(myaddr[0]->sa_family);
	memcpy(p, &_INPORTBYAF(myaddr[0]), sizeof(u_short));
	p += sizeof(u_short);

	/* copy target address */
	memcpy(p, _SADDRBYAF(to), _ALENBYAF(to->sa_family));
	p += _ALENBYAF(to->sa_family);
	memcpy(p, &_INPORTBYAF(to), sizeof(u_short));
	p += sizeof(u_short);

	/* copy time */
	t = time(0);
	memcpy(p, (caddr_t)&t, sizeof(t));
	p += sizeof(t);

	/* copy random value */
	if ((buf2 = eay_set_random(local_secret_size)) == 0) {
		return(-1);
	}
	memcpy(p, buf2->v, local_secret_size);
	p += local_secret_size;
	vfree(buf2);

	buf2 = eay_sha1_one(buf);
	memcpy(place, buf2->v, sizeof(cookie_t));
	vfree(buf2);

	error = 0;
end:
	return(error);
}

/*
 * save partner's id into isakmp status.
 */
static int
isakmp_id2isa(iph1, id)
	struct isakmp_ph1 *iph1;
	struct isakmp_pl_id *id;
{
	/* save the body of the ID payload */
	if ((iph1->id_p = vmalloc(id->h.len - sizeof(struct isakmp_gen))) == 0) {
		plog("isakmp_id2isa", "vmalloc (%s)\n", strerror(errno));
		goto err;
	}
	memcpy(iph1->id_p->v, (caddr_t)id + sizeof(struct isakmp_gen),
		iph1->id_p->l);

	return(0);
err:
	return(-1);
}

/*
 * save partner's ke and nonce into isakmp status.
 */
static int
isakmp_kn2isa(iph1, ke, nonce)
	struct isakmp_ph1 *iph1;
	struct isakmp_pl_ke *ke;
	struct isakmp_pl_nonce *nonce;
{
	/* save the body of the KE payload */
	if ((iph1->dhpub_p = vmalloc(ke->h.len - sizeof(*ke))) == 0) {
		plog("isakmp_kn2isa", "vmalloc (%s)\n", strerror(errno));
		goto err;
	}
	memcpy(iph1->dhpub_p->v, (caddr_t)ke + sizeof(*ke),
		iph1->dhpub_p->l);

	/* save the body of the NONCE payload */
	if ((iph1->nonce_p = vmalloc(nonce->h.len - sizeof(*nonce))) == 0) {
		plog("isakmp_kn2isa", "vmalloc (%s)\n", strerror(errno));
		goto err;
	}
	memcpy(iph1->nonce_p->v, (caddr_t)nonce + sizeof(*nonce),
		iph1->nonce_p->l);

	return(0);
err:
	return(-1);
}

/* %%%
 * Information Exchange
 */
/*
 * handling to receive Notification payload
 */
static int
isakmp_inf_nrecv(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_n *n;
	char *p;
	u_char *spi;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_inf_nrecv", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_N) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_inf_nrecv",
		    "received invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_N);
		return(-1);
	}

	n = (struct isakmp_pl_n *)((char *)isakmp + sizeof(*isakmp));
	if (n->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_inf_nrecv",
		    "received invalid next payload type %d, expecting %d.\n",
		    n->h.np, ISAKMP_NPTYPE_NONE);
		return(-1);
	}
	n->h.len = ntohs(n->h.len);

	p = (char *)n + sizeof(struct isakmp_pl_n);

	if (ntohl(isakmp->len) - sizeof(struct isakmp)
	     - sizeof(struct isakmp_pl_n) - n->spi_size == 0) {
		plog2(from, "isakmp_inf_nrecv",
		    "invalid spi_size in notification payload.\n");
		return(-1);
	}

	spi = mem2str(p, n->spi_size);

	plog2(from, "isakmp_inf_drecv",
	    "notification message %d, doi=%d prot_id=%d, spi=%s.\n",
	    n->doi, n->prot_id, spi);

	free(spi);

	return(0);
}

/*
 * handling to receive Deletion payload
 */
static int
isakmp_inf_drecv(msg, from, iph1)
	vchar_t *msg;
	struct sockaddr *from;
	struct isakmp_ph1 *iph1;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_d *d;
	char *p;
	int tlen, num_spi;
	u_char *spi;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_inf_drecv", "begin.\n"));

	/* validate the type of next payload */
	if (isakmp->np != ISAKMP_NPTYPE_D) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_inf_drecv",
		    "invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_D);
		return(-1);
	}

	d = (struct isakmp_pl_d *)((char *)isakmp + sizeof(*isakmp));
	if (d->h.np != ISAKMP_NPTYPE_NONE) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_inf_drecv",
		    "invalid next payload type %d, expecting %d.\n",
		    d->h.np, ISAKMP_NPTYPE_NONE);
		return(-1);
	}
	d->h.len = ntohs(d->h.len);

	p = (char *)d + sizeof(struct isakmp_pl_d);
	tlen = ntohl(isakmp->len) - sizeof(struct isakmp)
	     - sizeof(struct isakmp_pl_d);

	num_spi = ntohs(d->num_spi);

	while (tlen > 0) {
		if (num_spi < 0) {
			plog2(from, "isakmp_inf_drecv",
			    "invalid number of spi in deletion payload.\n");
			return(-1);
		}

		if (tlen < d->spi_size) {
			plog2(from, "isakmp_inf_drecv",
			    "invalid spi_size in deletion payload.\n");
			return(-1);
		}

		spi = mem2str(p, d->spi_size);

		plog2(from, "isakmp_inf_drecv",
		    "deletion message received, doi=%d prot_id=%d, spi=%s.\n",
		    d->doi, d->prot_id, spi);

		free(spi);

		tlen -= d->spi_size;
		num_spi--;
	}

	return(0);
}

/*
 * send Information
 */
static int
isakmp_inf_send(v)
	int v;
{
	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_inf_send", "sendto Information (%d).\n", v));

	return(0);
}

typedef unsigned char des_cblock[8];

/* DES Weak Keys */
des_cblock des_weak_keys[] = {
	{ 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
	{ 0x1F, 0x1F, 0x1F, 0x1F, 0xE0, 0xE0, 0xE0, 0xE0 },
	{ 0xE0, 0xE0, 0xE0, 0xE0, 0x1F, 0x1F, 0x1F, 0x1F },
	{ 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE },
/* DES Semi-Weak Keys */
	{ 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE },
	{ 0x1F, 0xE0, 0x1F, 0xE0, 0x0E, 0xF1, 0x0E, 0xF1 },
	{ 0x01, 0xE0, 0x01, 0xE0, 0x01, 0xF1, 0x01, 0xF1 },
	{ 0x1F, 0xFE, 0x1F, 0xFE, 0x0E, 0xFE, 0x0E, 0xFE },
	{ 0x01, 0x1F, 0x01, 0x1F, 0x01, 0x0E, 0x01, 0x0E },
	{ 0xE0, 0xFE, 0xE0, 0xFE, 0xF1, 0xFE, 0xF1, 0xFE },
	{ 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01 },
	{ 0xE0, 0x1F, 0xE0, 0x1F, 0xF1, 0x0E, 0xF1, 0x0E },
	{ 0xE0, 0x01, 0xE0, 0x01, 0xF1, 0x01, 0xF1, 0x01 },
	{ 0xFE, 0x1F, 0xFE, 0x1F, 0xFE, 0x0E, 0xFE, 0x0E },
	{ 0x1F, 0x01, 0x1F, 0x01, 0x0E, 0x01, 0x0E, 0x01 },
	{ 0xFE, 0xE0, 0xFE, 0xE0, 0xFE, 0xF1, 0xFE, 0xF1 },
/* EOD */
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
};

static int
isakmp_check_des_weakkey(key)
	vchar_t *key;
{
	int f, i;
	des_cblock *p = des_weak_keys;
	u_int8 *k = (u_int8 *)key->v;

	while (((caddr_t)p)[0]) {
		f = 1;
		for (i = 0; i < 8; i++) {
			if ((u_int8)((caddr_t)p)[i] != k[i]) {
				f = 0;
				break;
			}
		}
		if (f) return(-1);
		p++;
	}

	return(0);
}

