/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    its contributors.
 * 4. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: pfkey_lib.c,v 1.1.1.1.2.3 1998/07/07 14:04:18 sakane Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <net/pfkeyv2.h>
#include <netinet/in.h>
#include <netkey/keydb.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "var.h"
#include "vmbuf.h"
#include "pfkey.h"
#include "pfkey_lib.h"
#include "debug.h"

static int so_PK;

struct sadb_msg *pfkey_register __P((u_int));
struct pfkey_st *pfkey_acquire __P((struct sadb_msg *));
u_int32 pfkey_getspi __P((u_int32_t, u_int, struct sockaddr *, struct sockaddr *));
int pfkey_update __P((u_int32_t, u_int, struct sockaddr *, struct sockaddr *,
			u_int, u_int, vchar_t *, vchar_t *));
int pfkey_check __P((struct sadb_msg *, caddr_t *));
int pfkey_open __P((void));
void pfkey_close __P((void));
struct sadb_msg *pfkey_recv __P((void));
int pfkey_send __P((struct sadb_msg *, int));
struct pfkey_st *pfkey_new_st __P((void));
void pfkey_free_st __P((struct pfkey_st *));

/* %%% */
/*
 * send SADB_REGISTER message to the kernel and receive a reply from kernel.
 * IN:
 *	satype:	type of SA in order to register.
 * OUT:
 *	0     : error occured, and set errno.
 *	others: a pointer to new allocated buffer in which supported
 *	        algorithms is.
 */
struct sadb_msg *
pfkey_register(satype)
	u_int satype;
{
	struct sadb_msg *newmsg;
	u_int len;

	YIPSDEBUG(DEBUG_PFKEY, plog("pfkey_register", "begin register.\n"));

	/* create new sadb_msg to send. */
	len = sizeof(struct sadb_msg);

	newmsg = CALLOC(len, struct sadb_msg *);
	if (newmsg == 0) {
		plog("pfkey_register", "No more memory.\n");
		errno = ENOBUFS;
		return(0);
	}

	newmsg->sadb_msg_version = PF_KEY_V2;
	newmsg->sadb_msg_type = SADB_REGISTER;
	newmsg->sadb_msg_errno = 0;
	newmsg->sadb_msg_satype = satype;
	newmsg->sadb_msg_len = len;
	newmsg->sadb_msg_seq = 0;
	newmsg->sadb_msg_pid = getpid();

	/* send message */
	if ((len = pfkey_send(newmsg, len)) < 0)
		return(0);

	free(newmsg);

	/* receive message */
	if ((newmsg = pfkey_recv()) == 0)
		return(0);

	return(newmsg);
}

/*
 * allocate a new pfkey status management buffer.
 * IN:
 *	msg: pointer to the SADB_ACQUIRE message.
 * OUT:
 *	0     : error occured, and set errno.
 *	others: a pointer to new allocated pfkey_st buffer.
 */
struct pfkey_st *
pfkey_acquire(msg)
	struct sadb_msg *msg;
{
	struct pfkey_st *pst = 0;
	caddr_t mhp[SADB_EXT_MAX + 1];
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("pfkey_acquire", "begin\n"));

	/* allocate buffer for status management of pfkey message */
	if ((pst = pfkey_new_st()) == 0) goto end;

	pst->satype = msg->sadb_msg_satype;
	pst->seq = msg->sadb_msg_seq;
	pst->pid = msg->sadb_msg_pid;

	memset(mhp, 0, sizeof(caddr_t) * (SADB_EXT_MAX + 1));
	if (pfkey_check(msg, mhp)) {
		pfkey_free_st(pst);
		goto end;
	}

    {
	struct sadb_address *addr;

	addr = (struct sadb_address *)mhp[SADB_EXT_ADDRESS_SRC];
	pst->src0 = CALLOC(addr->sadb_address_len, struct sadb_address *);
	if (pst->src0 == 0) goto err;
	memcpy(pst->src0, (caddr_t)addr, addr->sadb_address_len);
	pst->src = (struct sockaddr *)((caddr_t)addr + sizeof(*addr));

	addr = (struct sadb_address *)mhp[SADB_EXT_ADDRESS_DST];
	pst->dst0 = CALLOC(addr->sadb_address_len, struct sadb_address *);
	if (pst->dst0 == 0) goto err;
	memcpy(pst->dst0, (caddr_t)addr, addr->sadb_address_len);
	pst->dst = (struct sockaddr *)((caddr_t)addr + sizeof(*addr));
    }

	error = 0;
end:
	return(pst);
err:
	if (pst) pfkey_free_st(pst);
	pst = 0;
	goto end;
}

/*
 * send SADB_GETSPI message to the kernel and receive a reply from kernel.
 * IN:
 *	pst:
 * OUT:
 *	0     : error occured, and set errno.
 *	others: a value of SPI.
 */
u_int32
pfkey_getspi(seq, satype, src, dst)
	u_int32_t seq;
	u_int satype;
	struct sockaddr *src, *dst;
{
	struct sadb_msg *newmsg = 0;
	struct sadb_address *addr = 0;
	struct sadb_spirange *spirng = 0;
	caddr_t mhp[SADB_EXT_MAX + 1];
	u_int32_t spi = 0;
	u_int len, salen;
	caddr_t p;

	YIPSDEBUG(DEBUG_PFKEY, plog("pfkey_getspi", "begin getspi.\n"));

	salen = _SALENBYAF(src->sa_family);	/* XXX: assuming same dst. */

	/* create new sadb_msg to reply. */
	len = sizeof(struct sadb_msg)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		/*+ sizeof(struct sadb_spirange)*/ ;

	newmsg = CALLOC(len, struct sadb_msg *);
	if (newmsg == 0) {
		plog("pfkey_getspi", "No more memory.\n");
		errno = ENOBUFS;
		return(0);
	}

	newmsg->sadb_msg_version = PF_KEY_V2;
	newmsg->sadb_msg_type = SADB_GETSPI;
	newmsg->sadb_msg_errno = 0;
	newmsg->sadb_msg_satype = satype;
	newmsg->sadb_msg_len = len;
	newmsg->sadb_msg_seq = seq;
	newmsg->sadb_msg_pid = getpid();
	p = (caddr_t)newmsg + sizeof(struct sadb_msg);

	/* set sadb_address for source */
	addr = (struct sadb_address *)p;
	addr->sadb_address_len = sizeof(struct sadb_address) + _ALIGN8(salen);
	addr->sadb_address_exttype = SADB_EXT_ADDRESS_SRC;
	addr->sadb_address_proto = 0; /* XXX */
	addr->sadb_address_prefixlen = _ALENBYAF(src->sa_family);
	p += sizeof(struct sadb_address);
	bcopy((caddr_t)src, p, salen);
	p += salen;

	/* set sadb_address for destination */
	addr = (struct sadb_address *)p;
	addr->sadb_address_len = sizeof(struct sadb_address) + _ALIGN8(salen);
	addr->sadb_address_exttype = SADB_EXT_ADDRESS_DST;
	addr->sadb_address_proto = 0; /* XXX */
	addr->sadb_address_prefixlen = _ALENBYAF(dst->sa_family);
	p += sizeof(struct sadb_address);
	bcopy((caddr_t)src, p, salen);
	p += salen;

	/* XXX */
	/* proccessing spi range */

	/* send message */
	if ((len = pfkey_send(newmsg, len)) < 0)
		goto end;

	free(newmsg);

	/* receive message */
	if ((newmsg = pfkey_recv()) == 0)
		goto end;

	if (pfkey_check(newmsg, mhp))
		goto end;
	
	spi = ((struct sadb_sa *)mhp[SADB_EXT_SA])->sadb_sa_spi;

end:
	if (newmsg) free(newmsg);
	return(spi);
}

/*
 * send SADB_UPDATE message to the kernel and receive a reply from kernel.
 * IN:
 *	pst:
 * OUT:
 *	0	: success
 *	negative: fail and set errno.
 */
int
pfkey_update(seq, satype, src, dst, auth, enc, k_auth, k_enc)
	u_int32_t seq;
	u_int satype;
	struct sockaddr *src, *dst;
	u_int auth, enc;
	vchar_t *k_auth, *k_enc;
{
	struct sadb_msg *newmsg = 0;
	struct sadb_address *addr;
	struct sadb_key *key;
	u_int len, salen;
	caddr_t p;
	int error = -1;

	YIPSDEBUG(DEBUG_PFKEY, plog("pfkey_getspi", "begin getspi.\n"));

	salen = _SALENBYAF(src->sa_family);	/* XXX: assuming same dst. */

	/* create new sadb_msg to reply. */
	len = sizeof(struct sadb_msg)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		+ sizeof(struct sadb_address) + _ALIGN8(salen);

	if (auth != SADB_AALG_NONE)
		len += (sizeof(struct sadb_key) + k_auth->l);

	if (enc != SADB_EALG_NONE)
		len += (sizeof(struct sadb_key) + k_enc->l);

	newmsg = CALLOC(len, struct sadb_msg *);
	if (newmsg == 0) {
		plog("pfkey_getspi", "No more memory.\n");
		errno = ENOBUFS;
		return(-1);
	}

	newmsg->sadb_msg_version = PF_KEY_V2;
	newmsg->sadb_msg_type = SADB_GETSPI;
	newmsg->sadb_msg_errno = 0;
	newmsg->sadb_msg_satype = satype;
	newmsg->sadb_msg_len = len;
	newmsg->sadb_msg_seq = seq;
	newmsg->sadb_msg_pid = getpid();
	p = (caddr_t)newmsg + sizeof(struct sadb_msg);

	/* set sadb_address for source */
	addr = (struct sadb_address *)p;
	addr->sadb_address_len = sizeof(struct sadb_address) + _ALIGN8(salen);
	addr->sadb_address_exttype = SADB_EXT_ADDRESS_SRC;
	addr->sadb_address_proto = 0; /* XXX */
	addr->sadb_address_prefixlen = _ALENBYAF(src->sa_family);
	p += sizeof(struct sadb_address);
	bcopy((caddr_t)src, p, salen);
	p += salen;

	/* set sadb_address for destination */
	addr = (struct sadb_address *)p;
	addr->sadb_address_len = sizeof(struct sadb_address) + _ALIGN8(salen);
	addr->sadb_address_exttype = SADB_EXT_ADDRESS_DST;
	addr->sadb_address_proto = 0; /* XXX */
	addr->sadb_address_prefixlen = _ALENBYAF(dst->sa_family);
	p += sizeof(struct sadb_address);
	bcopy((caddr_t)src, p, salen);
	p += salen;

	/* set authentication algorithm, if present. */
	if (auth != SADB_AALG_NONE) {
		struct sadb_key m_key;

		m_key.sadb_key_len =
			_ALIGN8(sizeof(m_key) + k_auth->l);
		m_key.sadb_key_exttype = SADB_EXT_KEY_AUTH;
		m_key.sadb_key_bits = k_auth->l * 8;
		m_key.sadb_key_reserved = 0;

		memcpy(p, &m_key, sizeof(m_key));
		p += sizeof(m_key);
		memcpy(p, (caddr_t)&k_auth->v, k_auth->l);
		p += k_auth->l;
	}

	/* set encyption algorithm, if present. */
	if (enc != SADB_EALG_NONE) {
		struct sadb_key m_key;

		m_key.sadb_key_len =
			_ALIGN8(sizeof(m_key) + k_enc->l);
		m_key.sadb_key_exttype = SADB_EXT_KEY_ENCRYPT;
		m_key.sadb_key_bits = k_enc->l * 8;
		m_key.sadb_key_reserved = 0;

		memcpy(p, &m_key, sizeof(m_key));
		p += sizeof(m_key);
		memcpy(p, (caddr_t)&k_enc->v, k_enc->l);
		p += k_enc->l;
	}

	/* send message */
	if ((len = pfkey_send(newmsg, len)) < 0)
		goto end;

	free(newmsg);

	/* receive message */
	if ((newmsg = pfkey_recv()) == 0)
		goto end;

	error = newmsg->sadb_msg_errno;

end:
	if (newmsg) free(newmsg);
	return(error);
}

/*
 * check basic usage for sadb_msg,
 * and set the pointer to each header in this message buffer.
 * This routine is derived from netkey/key.c in KAME.
 * IN:	msg: pointer to message buffer.
 *	mhp: pointer to the buffer allocated of each header.
 * OUT:	0 if success.
 *	other if error, return errno.
 */
int
pfkey_check(msg, mhp)
	struct sadb_msg *msg;
	caddr_t *mhp;
{
	struct sadb_ext *ext;
	int tlen;

	/* check version */
	if (msg->sadb_msg_version != PF_KEY_V2) {
		printf("key_check: PF_KEY version %lu is too old.\n",
		    msg->sadb_msg_version);
		return(EINVAL);
	}

	/* check type */
	if (msg->sadb_msg_type > SADB_MAX) {
		printf("key_check: invalid type %u is passed.\n",
		    msg->sadb_msg_type);
		return(EINVAL);
	}

	/* check SA type */
	switch (msg->sadb_msg_satype) {
	case SADB_SATYPE_UNSPEC:
		if (msg->sadb_msg_type == SADB_FLUSH
		 || msg->sadb_msg_type == SADB_DUMP
		 || msg->sadb_msg_type == SADB_X_SPDADD
		 || msg->sadb_msg_type == SADB_X_SPDDELETE
		 || msg->sadb_msg_type == SADB_X_SPDDUMP
		 || msg->sadb_msg_type == SADB_X_SPDFLUSH)
			break;
		else
			printf("key_check: type UNSPEC is invalid.\n");
		break;
	case SADB_SATYPE_AH:
	case SADB_SATYPE_ESP:
		break;
	case SADB_SATYPE_RSVP:
	case SADB_SATYPE_OSPFV2:
	case SADB_SATYPE_RIPV2:
	case SADB_SATYPE_MIP:
		printf("key_check: type %u isn't supported.\n",
		    msg->sadb_msg_satype);
		return(EOPNOTSUPP);
	default:
		printf("key_check: invalid type %u is passed.\n",
		    msg->sadb_msg_satype);
		return(EINVAL);
	}

	mhp[0] = (caddr_t)msg;

	tlen = msg->sadb_msg_len - sizeof(struct sadb_msg);
	ext = (struct sadb_ext *)((caddr_t)msg + sizeof(struct sadb_msg));

	while (tlen > 0) {
		/* duplicate check */
		/* XXX: Are the KEY_AUTH and KEY_ENCRYPT ? */
		if (mhp[ext->sadb_ext_type] != 0) {
			printf("key_check: duplicate ext_type %u is passed.\n",
				ext->sadb_ext_type);
			return(EINVAL);
		}

		/* set pointer */
		switch (ext->sadb_ext_type) {
		case SADB_EXT_SA:
		case SADB_EXT_LIFETIME_CURRENT:
		case SADB_EXT_LIFETIME_HARD:
		case SADB_EXT_LIFETIME_SOFT:
		case SADB_EXT_ADDRESS_SRC:
		case SADB_EXT_ADDRESS_DST:
		case SADB_EXT_ADDRESS_PROXY:
		case SADB_EXT_KEY_AUTH:
			/* must to be chek weak keys. */
		case SADB_EXT_KEY_ENCRYPT:
			/* must to be chek weak keys. */
		case SADB_EXT_IDENTITY_SRC:
		case SADB_EXT_IDENTITY_DST:
		case SADB_EXT_SENSITIVITY:
		case SADB_EXT_PROPOSAL:
		case SADB_EXT_SUPPORTED_AUTH:
		case SADB_EXT_SUPPORTED_ENCRYPT:
		case SADB_EXT_SPIRANGE:
		case SADB_X_EXT_POLICY:
			mhp[ext->sadb_ext_type] = (caddr_t)ext;
			break;
		default:
			printf("key_check: invalid ext_type %u is passed.\n", ext->sadb_ext_type);
			return(EINVAL);
		}

		tlen -= ext->sadb_ext_len;
		ext = (struct sadb_ext *)((caddr_t)ext + ext->sadb_ext_len);
	}

	/* check field of upper layer protocol and address family */
	if (mhp[SADB_EXT_ADDRESS_SRC] && mhp[SADB_EXT_ADDRESS_DST]) {
		struct sadb_address *src0, *dst0;
		struct sockaddr *src, *dst;

		src0 = (struct sadb_address *)(mhp[SADB_EXT_ADDRESS_SRC]);
		dst0 = (struct sadb_address *)(mhp[SADB_EXT_ADDRESS_DST]);
		src = (struct sockaddr *)((caddr_t)src0 + sizeof(*src0));
		dst = (struct sockaddr *)((caddr_t)dst0 + sizeof(*dst0));

		if (src0->sadb_address_proto != dst0->sadb_address_proto) {
			printf("key_check: upper layer protocol mismatched.\n");
			return(EINVAL);
		}

		if (src->sa_family != dst->sa_family) {
			printf("key_check: address family mismatched.\n");
			return(EINVAL);
		}
		if (src->sa_family != AF_INET
		 && src->sa_family != AF_INET6) {
			printf("key_check: invalid address family.\n");
			return(EINVAL);
		}

		if (src0->sadb_address_prefixlen == 0) {
			src0->sadb_address_prefixlen =
				_ALENBYAF(src->sa_family) * 8;

			YIPSDEBUG(DEBUG_PFKEY,
				printf("key_check: src's prefixlen fixed, "
					"0 -> %u.\n",
					src0->sadb_address_prefixlen));
		}

		if (dst0->sadb_address_prefixlen == 0) {
			dst0->sadb_address_prefixlen =
				_ALENBYAF(dst->sa_family) * 8;

			YIPSDEBUG(DEBUG_PFKEY,
				printf("key_check: dst's prefixlen fixed, "
					"0 -> %u.\n",
					dst0->sadb_address_prefixlen));
		}

	}

	return(0);
}

/* %%%: basic routines */
/*
 * open a socket.
 * OUT:
 *	-1: fail.
 *	others : success and return value of socket.
 */
int
pfkey_open()
{
	if ((so_PK = socket(PF_KEY, SOCK_RAW, PF_KEY_V2)) < 0)
		return(-1);

	return(so_PK);
}

/*
 * close a socket.
 * OUT:
 *	 0: success.
 *	-1: fail.
 */
void
pfkey_close()
{
	(void)close(so_PK);

	return;
}

/*
 * receive sadb_msg data, and return pointer to new buffer allocated.
 * Must free this buffer later.
 * OUT:
 *    0     : error occured.
 *    others: a pointer to sadb_msg structure.
 */
struct sadb_msg *
pfkey_recv()
{
	struct sadb_msg buf, *newmsg;
	int len;

	while ((len = recv(so_PK, (caddr_t)&buf, sizeof(buf), MSG_PEEK)) < 0) {
		if (errno == EINTR) continue;
		return(0);
	}

	if (len < sizeof(buf)) {
		recv(so_PK, (caddr_t)&buf, sizeof(buf), 0);
		errno = EINVAL;
		return(0);
	}

	/* read real message */
	if ((newmsg = CALLOC(buf.sadb_msg_len, struct sadb_msg *)) == 0)
		return(0);

	while ((len = recv(so_PK, (caddr_t)newmsg, buf.sadb_msg_len, 0)) < 0) {
		if (errno == EINTR) continue;
		return(0);
	}

	if (len != buf.sadb_msg_len) {
		errno = EINVAL;
		return(0);
	}

	YIPSDEBUG(DEBUG_NET,
	    plog("pfkey_recv", "message has been received, cmd=%d len=%d\n",
	    newmsg->sadb_msg_type, newmsg->sadb_msg_len));
	YIPSDEBUG(DEBUG_DNET, kdebug_sadb(newmsg));

	return(newmsg);
}

/*
 * send message to a socket.
 * OUT:
 *	 others: success and return length to send.
 *	-1     : fail.
 */
int
pfkey_send(msg, len)
	struct sadb_msg *msg;
	int len;
{
	if ((len = send(so_PK, (caddr_t)msg, len, 0)) < 0)
		return(-1);

	YIPSDEBUG(DEBUG_NET,
		plog("pfkey_send", "message has been sent, len=%d\n", len));
	YIPSDEBUG(DEBUG_DNET, kdebug_sadb(msg));

	return(len);
}

/* %%% pfkey status management routines */
/*
 * create new status record of pfkey message
 */
struct pfkey_st *
pfkey_new_st()
{
	struct pfkey_st *pst;

	/* create new status */
	if ((pst = CALLOC(sizeof(*pst), struct pfkey_st *)) == 0) {
		plog("pfkey_new_st", "malloc (%s)\n", strerror(errno)); 
		return(0);
	}

	return(pst);
}

/*
 * free record from pfkey stauts table
 */
void
pfkey_free_st(st)
	struct pfkey_st *st;
{
	/* XXX: free more !? */
	if (st->sc) sched_kill(st->sc);
	if (st->src) free(st->src);
	if (st->dst) free(st->dst);

	(void)free(st);

	return;
}

