/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* $Id: keydb.h,v 1.1.4.3 1999/01/20 17:26:44 itojun Exp $ */

#ifndef _NETKEY_KEYDB_H_
#define _NETKEY_KEYDB_H_

#ifdef __NetBSD__
# ifdef _KERNEL
#  define KERNEL
# endif
#endif

/* Must include ipsec.h and keyv2.h before in its use. */

/* index structure for SPD and SAD */
/* XXX should be structed both src and dst to sockaddr. */
struct secindex {
	u_int8_t family;	/* AF_INET or AF_INET6 */
	u_int8_t prefs;		/* preference for srouce address */
	u_int8_t prefd;		/* preference for destination address */
	caddr_t src;		/* buffer for source address */
	caddr_t dst;		/* buffer for destination address */
	u_int8_t proto;		/* upper layer Protocol */
	u_int16_t ports;	/* source port */
	u_int16_t portd;	/* destination port */

#define SADB_X_DIR_UNKNOWN	0x00
#define SADB_X_DIR_INBOUND	0x01
#define SADB_X_DIR_OUTBOUND	0x02
	u_int dir;		/* direction */
};

/* security protocol and mode */
#define SADB_X_ESP_TRANS	0
#define SADB_X_AH_TRANS		1
#define SADB_X_ESP_NETWORK	2
#define SADB_X_AH_NETWORK	3
#define SADB_PROTO_MAX		4

/* Security Policy Data Base */
#define SADB_SPD_SIZE	16
struct secpc {
	struct secpc *next;
	struct secpc *prev;
	struct keytree *spt;	/* back pointer to the top of secpc tree */

	int refcnt;		/* reference count */
	u_int8_t state;		/* DEAD or MATURE */
	struct secindex idx;	/* index */

	struct ipsecpolicy *pl;	/* pointer to security policy level */
				/* if 0, then packet matched discard. *
				 * if ~0, then bypass, else do IPsec. */
};

/* Security Association Bundle */
#define SADB_SAD_SIZE	16
struct secasb {
	struct secasb *next;
	struct secasb *prev;
	struct keytree *sabt;	/* back pointer to the top of secasb tree */

	int refcnt;		/* reference count */
	u_int8_t state;		/* DEAD or MATURE */
	struct secindex idx;	/* index */

	/* security association per both protocol and mode */
	struct secas *sa[SADB_PROTO_MAX];

	struct route sab_route;
};

/* Security Association */
struct secas {
	struct secas *next;
	struct secas *prev;
	struct keytree *sat;	/* back pointer to the top of secas tree */

	int refcnt;		/* reference count */
	u_int8_t state;		/* Status of this Association */
	u_int8_t type;		/* Type of this association */
	u_int8_t alg_auth;	/* Authentication Algorithm Identifier */
	u_int8_t alg_enc;	/* Cipher Algorithm Identifier */
	u_int32_t spi;		/* SPI Value, network byte order */
	u_int32_t flags;	/* holder for SADB_KEY_FLAGS */
	struct sadb_key *key_auth;	/* Key for Authentication */
					/* length has been shifted up to 3. */
	struct sadb_key *key_enc;	/* Key for Encryption */
					/* length has been shifted up to 3. */
	struct sockaddr *proxy;	/* Proxy IP address for Destination */
	struct secreplay *replay;	/* replay prevention */
	struct sadb_lifetime *lft_c;	/* CURRENT lifetime, it's constant. */
					/* length has been in 64 bits. */
	struct sadb_lifetime *lft_h;	/* HARD lifetime */
					/* length has been in 64 bits. */
	struct sadb_lifetime *lft_s;	/* SOFT lifetime */
					/* length has been in 64 bits. */
	struct secexpire *exph_h;	/* expiration handler for HARD */
	struct secexpire *exph_s;	/* expiration handler for SOFT */
	caddr_t iv;		/* Initilization Vector */
	u_int ivlen;		/* XXX: quick hack */
	caddr_t misc1;		/* the use for example DES's setkey. */
	caddr_t misc2;
	caddr_t misc3;

	u_int32_t seq;		/* sequence number */
	u_int32_t pid;		/* pid */

	struct secasb *sab;	/* back pointer to the secasb tree */
};

/* replay prevention */
struct secreplay {
	u_int32_t count;
	int wsize;		/* window size, i.g. 4 bytes */
	u_int32_t seq;		/* used by sender */
	u_int32_t lastseq;	/* used by receiver */
	caddr_t bitmap;		/* used by receiver */
};

/* expiration handler */
struct secexpire {
	int type;		/* SOFT or HARD */
	struct secas *sa;	/* pointer to the SA */
};

/* socket table due to send PF_KEY messages. */ 
struct secreg {
	struct secreg *next;
	struct secreg *prev;
	struct keytree *regt;	/* back pointer to the top of secreg tree */

	struct socket *so;
};

/* acquiring list table. */ 
struct secacq {
	struct secacq *next;
	struct secacq *prev;
	struct keytree *acqt;	/* back pointer to the top of secacq tree */

	u_int8_t state;		/* Status of this acquiring */
	u_int32_t seq;
	u_int cnt;		/* counter to handle blocking expiration */

	struct secas *sa;
};

/* Sensitivity Level Specification */
/* nothing */

/* management for the tree and nodes. */
struct keytree {
	struct keynode *head;
	struct keynode *tail;
	int	len;
};

struct keynode {
	struct keynode *next;
	struct keynode *prev;
	struct keytree *back;	/* pointer to the keytree */
};

#if defined(KERNEL)
#define SADB_KILL_INTERVAL	600	/* six seconds */
#else
#if 0	/* shouldn't be here */
#define SADB_KILL_INTERVAL	6000000	/* six seconds */
#endif
#endif

struct key_cb {
	int key_count;
	int any_count;
};

/* sysctl */
#define KEYCTL_DEBUG_LEVEL	1
#define KEYCTL_SPI_TRY		2
#define KEYCTL_SPI_MIN_VALUE	3
#define KEYCTL_SPI_MAX_VALUE	4
#define KEYCTL_RANDOM_INT	5
#define KEYCTL_KILL_INT		6
#define KEYCTL_ACQ_EXP_INT	7
#define KEYCTL_ACQ_MAXTIME	8
#define KEYCTL_MAXID		9

#define KEYCTL_NAMES { \
	{ 0, 0 }, \
	{ "debug", CTLTYPE_INT }, \
	{ "spi_try", CTLTYPE_INT }, \
	{ "spi_min_value", CTLTYPE_INT }, \
	{ "spi_max_value", CTLTYPE_INT }, \
	{ "random_int", CTLTYPE_INT }, \
	{ "kill_int", CTLTYPE_INT }, \
	{ "acq_exp_int", CTLTYPE_INT }, \
	{ "acq_maxtime", CTLTYPE_INT }, \
}

#ifdef IPSEC_DEBUG
#define KEYCTL_VARS { \
	0, \
	&key_debug_level, \
	&key_spi_trycnt, \
	&key_spi_minval, \
	&key_spi_maxval, \
	&key_int_random, \
	&key_int_kill, \
	&key_acq_expint, \
	&key_acq_maxtime, \
}
#else
#define KEYCTL_VARS { \
	0, \
	0, \
	&key_spi_trycnt, \
	&key_spi_minval, \
	&key_spi_maxval, \
	&key_int_random, \
	&key_int_kill, \
	&key_acq_expint, \
	&key_acq_maxtime, \
}
#endif

#define _ALIGN8(a) (1 + (((a) - 1) | (8 - 1)))

#if 1
/* in 64bits */
#define	_UNUNIT64(a)	((a) << 3)
#define	_UNIT64(a)	((a) >> 3)
#else
#define	_UNUNIT64(a)	(a)
#define	_UNIT64(a)	(a)
#endif

#define _KEYLEN(key) \
	((u_int)((key)->sadb_key_bits >> 3))
#define _KEYBITS(key) ((u_int)((key)->sadb_key_bits))
#define _KEYBUF(key) ((caddr_t)((caddr_t)(key) + sizeof(struct sadb_key)))
#define _KEYEXTLEN(ext) \
	((ext) == 0 ? 0 : _UNUNIT64(((struct sadb_ext *)(ext))->sadb_ext_len))

#define _INADDR(in) ((struct sockaddr_in *)(in))

#if defined(INET6)
#define _IN6ADDR(in6) ((struct sockaddr_in6 *)(in6))
#define _SALENBYAF(family) \
	(((family) == AF_INET) ? \
		(u_int)sizeof(struct sockaddr_in) : \
		(u_int)sizeof(struct sockaddr_in6))
#define _ALENBYAF(family) \
	(((family) == AF_INET) ? \
		(u_int)sizeof(struct in_addr) : \
		(u_int)sizeof(struct in6_addr))
#define _INADDRBYAF(saddr) \
	(((saddr)->sa_family == AF_INET) ? \
		(caddr_t)&((struct sockaddr_in *)(saddr))->sin_addr : \
		(caddr_t)&((struct sockaddr_in6 *)(saddr))->sin6_addr)
#define _INPORTBYAF(saddr) \
	(((saddr)->sa_family == AF_INET) ? \
		((struct sockaddr_in *)(saddr))->sin_port : \
		((struct sockaddr_in6 *)(saddr))->sin6_port)
#define _SADDRBYAF(saddr) \
	(((saddr)->sa_family == AF_INET) ? \
		(caddr_t)&((struct sockaddr_in *)(saddr))->sin_addr.s_addr : \
		(caddr_t)&((struct sockaddr_in6 *)(saddr))->sin6_addr.s6_addr)
#else
#define _IN6ADDR(in6) "#error"
#define _SALENBYAF(family) sizeof(struct sockaddr_in)
#define _ALENBYAF(family) sizeof(struct in_addr)
#define _INADDRBYAF(saddr) ((caddr_t)&((struct sockaddr_in *)(saddr))->sin_addr)
#define _INPORTBYAF(saddr) (((struct sockaddr_in *)(saddr))->sin_port)
#define _SADDRBYAF(saddr) \
	((caddr_t)&((struct sockaddr_in *)(saddr))->sin_addr.s_addr)
#endif /* defined(INET6) */

#ifndef KERNEL
#if 0 /* should not be here */
#define KMALLOC(p, t, n) ((p) = (t) malloc((unsigned int)(n)))
#define KFREE(p) free((caddr_t)(p));
#define MALLOC(p, t, n, t1, t2) ((p) = (t) malloc((unsigned int)(n)))
#define panic(param) { printf(param); exit(-1); }
#define kprintf(fmt, v1, v2, ap) vprintf((fmt), (ap));
#define bzero(p, len) memset((p), 0, (len))
#define bcopy(src, dst, len) memcpy((dst), (src), (len))
#endif
#endif

#endif /* _NETKEY_KEYDB_H_ */
