/* wptListView.cpp - Dynamic list view control
 *	Copyright (C) 2000-2004 Timo Schulz
 *	Copyright (C) 2004 Andreas Jobs
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <stdio.h>
#include <windows.h>
#include <commctrl.h>

#include "wptCommonCtl.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptErrors.h"
#include "wptTypes.h"
#include "wptGPG.h"
#include "wptKeylist.h"
#include "../resource.h"


int
listview_new( listview_ctrl_t *ctx )
{
    struct listview_ctrl_s *c;

    c = new struct listview_ctrl_s;
    if( !c )
	BUG( NULL );
    c->cols = 0;
    c->items = 0;
    c->ctrl = NULL;	
    *ctx = c;
    return 0;
} /* listview_new */


void
listview_release( listview_ctrl_t ctx )
{
    if( ctx ) {
	ctx->cols = 0;
	ctx->ctrl = NULL;
	ctx->items = 0;	
	delete ctx; 
	ctx = NULL;
    }
} /* listview_release */


int
listview_add_column( listview_ctrl_t ctx, listview_column_t col )
{	
    int rc = 0;
    LV_COLUMN lvc;

    memset( &lvc, 0, sizeof lvc );
    lvc.mask = LVCF_TEXT | LVCF_WIDTH | LVCF_SUBITEM;
    lvc.pszText = col->fieldname;
    lvc.cx = col->width;
    lvc.fmt = LVCFMT_LEFT;
    lvc.iSubItem = col->pos;
    if( ListView_InsertColumn( ctx->ctrl, col->pos, &lvc ) == -1 )
	rc = 1;
    ctx->cols++;
    return rc;
} /* listview_add_column */


int
listview_add_item( listview_ctrl_t ctx, const char *text )
{
    int rc = 0;
    LV_ITEM lvi;

    memset( &lvi, 0, sizeof lvi );
    lvi.mask = LVIF_TEXT;
    lvi.pszText = (char *)text;
    rc = ListView_InsertItem( ctx->ctrl, &lvi );
    if( rc == -1 )
	rc = 1;	
    ctx->items++;
    return rc;
} /* listview_add_item */


int
listview_add_item2( listview_ctrl_t ctx, const char * text, void * magic )
{
    int rc = 0;
    LV_ITEM lvi;

    memset( &lvi, 0, sizeof lvi );
    lvi.mask = LVIF_TEXT | LVIF_PARAM;
    lvi.pszText = (char *)text;
    lvi.lParam = (LPARAM)magic;
    rc = ListView_InsertItem( ctx->ctrl, &lvi );
    if( rc == -1 )
	rc = 1;	
    ctx->items++;
    return rc;
} /* listview_add_item2 */


void
listview_add_sub_item( listview_ctrl_t ctx, int pos, int col, const char *text )
{	
    ListView_SetItemText( ctx->ctrl, pos, col, (char *)text );
} /* listview_add_sub_item */


int
listview_count_items( listview_ctrl_t ctx, int curr_sel )
{	
    int n;

    n = curr_sel? ListView_GetSelectedCount( ctx->ctrl ) :    
     	          ListView_GetItemCount( ctx->ctrl );
    return n;
} /* listview_count_items */


int
listview_del_item( listview_ctrl_t ctx, int pos )
{
    int rc = 0;

    if( ListView_DeleteItem( ctx->ctrl, pos ) == -1 )
	rc = 1;
    return rc;
} /* listview_del_item */


int
listview_del_items( listview_ctrl_t ctx )
{	
    int n;

    /* delete all selected entries */
    for( n = 0;  n < listview_count_items( ctx, 0 ); n++ ) {
	if ( listview_get_item_state( ctx, n ) )
	    listview_del_item( ctx, n );	
    }
    return 0;
} /* listview_del_items */


int
listview_del_all( listview_ctrl_t ctx )
{	
    int rc = 0;

    if( ListView_DeleteAllItems( ctx->ctrl ) == FALSE )
	rc = 1;
    return rc;
} /* listview_del_all */


int
listview_get_item_state( listview_ctrl_t ctx, int pos )
{	
    return ListView_GetItemState( ctx->ctrl, pos, LVIS_SELECTED );
} /* listview_get_item_state */


int
listview_sort_items( listview_ctrl_t ctx, int sortby, listview_cmp sort_cb )
{	
    HWND hheader;
    HDITEM hdi;
    int idx;

    ListView_SortItems (ctx->ctrl, sort_cb, sortby);
    hheader = ListView_GetHeader (ctx->ctrl);
    if (!hheader)
        return 0;

    /* Remove image from all header fields */
    memset (&hdi, 0, sizeof(hdi));

    for (int i=0; i < 7; i++) {
        hdi.mask = HDI_FORMAT;
        Header_GetItem (hheader, i, &hdi);
        hdi.fmt &= ~HDF_IMAGE;
        Header_SetItem (hheader, i, &hdi);
    }

    switch (sortby & ~KEYLIST_SORT_DESC) {
    case GPGME_ATTR_USERID:    idx = 0; break;
    case GPGME_ATTR_KEYID:     idx = 1; break;
    case GPGME_ATTR_IS_SECRET: idx = 2; break;
    case GPGME_ATTR_LEN:       idx = 3; break;
    case GPGME_ATTR_VALIDITY:  idx = 5; break;
    case GPGME_ATTR_OTRUST:    idx = 6; break;
    case GPGME_ATTR_CREATED:   idx = 7; break;
    case GPGME_ATTR_ALGO:      idx = 8; break;
    default:                   idx = 0; break;
    }

    /* Set image to currently sorted field */
    memset (&hdi, 0, sizeof(hdi));
    hdi.mask = HDI_IMAGE | HDI_FORMAT;
    Header_GetItem (hheader, idx, &hdi);
    hdi.fmt |= HDF_IMAGE | HDF_BITMAP_ON_RIGHT;
    hdi.iImage = imagelist_getindex((sortby & KEYLIST_SORT_DESC) ? 
				    IMI_SORT_DOWNARROW : IMI_SORT_UPARROW);
    Header_SetItem (hheader, idx, &hdi);
    return 0;
} /* listview_sort_items */


int
listview_get_curr_pos( listview_ctrl_t ctx )
{	
    return ListView_GetNextItem( ctx->ctrl, -1, LVNI_SELECTED );
} /* listview_get_curr_pos */


int
listview_get_item_text (listview_ctrl_t ctx, int pos, int col, char *text,
		        int maxbytes)
{
    if (pos == -1) {
	pos = listview_get_curr_pos (ctx);
	if (pos == -1)
	    return -1;
    }
    ListView_GetItemText (ctx->ctrl, pos, col, text, maxbytes);
    return 0;
} /* listview_get_item_text */


void
listview_set_ext_style( listview_ctrl_t ctx )
{	
    ListView_SetExtendedListViewStyle( ctx->ctrl, LVS_EX_FULLROWSELECT );
} /* listview_set_ext_style */


int
listview_set_column_order( listview_ctrl_t ctx, int *array )
{	
    return ListView_SetColumnOrderArray( ctx->ctrl, ctx->cols, array );
} /* listview_set_column_order */


void
listview_select_all (listview_ctrl_t ctx)
{	
    ListView_SetItemState( ctx->ctrl, -1, LVIS_SELECTED, LVIS_SELECTED );
} /* listview_select_all */


void
listview_select_one (listview_ctrl_t ctx, int pos)
{
    ListView_SetItemState (ctx->ctrl, pos, LVIS_SELECTED|LVIS_FOCUSED, LVIS_FOCUSED|LVIS_SELECTED);
}


int
listview_find (listview_ctrl_t ctx, const char * str)
{
    LVFINDINFO inf;
    int pos;

    memset (&inf, 0, sizeof inf);
    inf.flags = LVFI_STRING|LVFI_PARTIAL;
    inf.psz = str;
    pos = ListView_FindItem (ctx->ctrl, -1, &inf);
    return pos;
}

/** Some functions to make the handling with TreeView Controls easier **/
int
treeview_add_item( HWND tree, HTREEITEM parent, const char *text )
{	
    TVINSERTSTRUCT tvis;
	
    memset( &tvis, 0, sizeof tvis );
    tvis.hParent = parent;
    tvis.hInsertAfter = TVI_LAST;
    tvis.item.mask = TVIF_TEXT;
    tvis.item.pszText = (char *)text;
    TreeView_InsertItem( tree, &tvis );

    return 0;
} /* treeview_add_item */