/* wptKeygenDlg.cpp - Key Generation dialog
 *	Copyright (C) 2000-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptNLS.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptErrors.h"
#include "wptUTF8.h"


static void 
clear_dlg_fields( HWND dlg )
{
    SetDlgItemText( dlg, IDC_KEYGEN_SUBKEYBITS, "" );
    SetDlgItemText( dlg, IDC_KEYGEN_NAME, "" );
    SetDlgItemText( dlg, IDC_KEYGEN_EMAIL, "" );
    SetDlgItemText( dlg, IDC_KEYGEN_COMMENT, "" );
    SetDlgItemText( dlg, IDC_KEYGEN_EXPDATE, "" );
    SetDlgItemText( dlg, IDC_KEYGEN_PASSPHRASE, "" );
    SetDlgItemText( dlg, IDC_KEYGEN_PWDCHECK, "" );
} /* clear_dlg_fields */


static void
ask_for_backup( HWND dlg )
{
    int id;
    char * path = NULL, * keyring = NULL;
    const char * name;
    
    path = get_gnupg_path( );
    if( !path )
	BUG( dlg );
    id = msg_box( dlg, 
                 _("It is STRONGLY recommend that you backup your keyrings because they both "
                   "contain VERY important data.\nRemember that your hard disk can crash or the "
                   "files can be deleted by accident; so it is a good\nidea to store them on "
                   "a different mass stoarge like a floppy or CDR!\n\n"
                   "Backup your keyrings now?"),
                 _("WARNING - Important hint" ), MB_YESNO );
    if( id == IDYES ) {
        name = get_filename_dlg( dlg, 1, _("Destination for Public Keyring"), NULL, "pubring.gpg" );
        if( name ) {
	    keyring = make_filename( path, "pubring", "gpg" );
            if( !CopyFile( keyring, name, FALSE ) )
		log_box( _("Key Generation"), MB_ERR, 
		         _("Could not copy %s -> %s"), keyring, name );	    
	    free_if_alloc( keyring );
        }
        name = get_filename_dlg( dlg, 1, _("Destination for Secret Keyring"), NULL, "secring.gpg" );
        if( name ) {
	    keyring = make_filename( path, "secring", "gpg" );
            if( !CopyFile( keyring, name, FALSE ) )
		log_box( _("Key Generation"), MB_ERR,
			 _("Could not copy %s -> %s"), keyring, name );
	    free_if_alloc( keyring );
        }
    }
    free_if_alloc( path );
} /* ask_for_backup */


static void
fill_keytype_box( HWND dlg )
{
    HWND cb = GetDlgItem( dlg, IDC_KEYGEN_KEYTYPE );

#define addstr( cb, str ) \
    SendMessage( (cb), CB_ADDSTRING, 0, (LPARAM)(LPCTSTR)(str) )
    addstr( cb, _("DSA and ELG (default)") );
    addstr( cb, _("DSA and RSA") );
    addstr( cb, _("DSA sign only") );
    addstr( cb, _("RSA sign only") );
    addstr( cb, _("RSA sign and encrypt") );
    addstr( cb, _("RSA and RSA (PGP)") );
    SendMessage( cb, CB_SETCURSEL, 0, 0 );
#undef addstr
} /* fill_keytype_box */


static inline int
get_keytype( HWND dlg )
{
    HWND cb = GetDlgItem( dlg, IDC_KEYGEN_KEYTYPE );
    return SendMessage( cb, CB_GETCURSEL, 0, 0 ) + 1;
} /* get_keytype */


int
keygen_check_date( SYSTEMTIME *st )
{	
    SYSTEMTIME t;
	
    GetSystemTime( &t );
    if( st->wYear > t.wYear || st->wMonth > t.wMonth )
	return 1;
    else if( st->wYear < t.wYear || st->wMonth < t.wMonth || st->wDay < t.wDay)
	return 0;	
    return 1;
} /* keygen_check_date */


BOOL CALLBACK
keygen_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static int hide = 1;
    static int act_expdate = 0;
    SYSTEMTIME st;
    gpgme_error_t err;
    char name[128] = {0}, email[128] = {0}, comment[128] = {0};
    char pwd[128], pwd2[128];
    char t[64], *expire = NULL;
    int bits, use_comment, keytype = 0;    
    char * p;	
    
    switch ( msg ) {
    case WM_INITDIALOG:
	hide = 1;
#ifndef LANG_DE
        SetWindowText( dlg, _("Key Generation") );
        SetDlgItemText( dlg, IDC_KEYGEN_INFO, 
			_("NOTE: Key generation can be a lengthy process! Please wait until "
			  "you get the message that key generation was finished.") );
        SetDlgItemText( dlg, IDC_KEYGEN_SUBKEYINF, _("Subkey size in &bits"));
        SetDlgItemText( dlg, IDC_KEYGEN_NAMEINF, _("&Real name") );
        SetDlgItemText( dlg, IDC_KEYGEN_COMMINF, _("&Comment (optional)") );
        SetDlgItemText( dlg, IDC_KEYGEN_EMAILINF, _("Email &address") );
        SetDlgItemText( dlg, IDC_KEYGEN_EXPINF, _("Key &expiration") );
        SetDlgItemText( dlg, IDC_KEYGEN_PWDINF, _("&Passphrase") );
        SetDlgItemText( dlg, IDC_KEYGEN_REPWDINF, _("&Repeat passphrase") );
        SetDlgItemText( dlg, IDC_KEYGEN_KEYTYPEINF, _("Key &type") );
	SetDlgItemText (dlg, IDC_KEYGEN_CLEAR, _("&Never"));
	SetDlgItemText (dlg, IDC_KEYGEN_HIDEPWD, _("&Hide Typing"));
#endif
        SetDlgItemInt( dlg, IDC_KEYGEN_SUBKEYBITS, 2048, FALSE );
	CheckDlgButton (dlg, IDC_KEYGEN_HIDEPWD, BST_CHECKED);
	CheckDlgButton (dlg, IDC_KEYGEN_EXPNEVER, BST_CHECKED);
	EnableWindow (GetDlgItem (dlg, IDC_KEYGEN_EXPDATE), FALSE);
        fill_keytype_box( dlg );
	center_window( dlg );
        SetForegroundWindow( dlg );
        return TRUE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE ) {
            SetDlgItemText( dlg, IDC_KEYGEN_PASSPHRASE, "" );
            SetDlgItemText( dlg, IDC_KEYGEN_PWDCHECK, "" );
            EndDialog( dlg, TRUE );
        }
        return FALSE;
        
    case WM_COMMAND:
	if (HIWORD (wparam) == BN_CLICKED &&
	    LOWORD (wparam) == IDC_KEYGEN_EXPNEVER) {
	    act_expdate ^= 1;
	    EnableWindow (GetDlgItem (dlg, IDC_KEYGEN_EXPDATE), act_expdate);
	}
	if( HIWORD( wparam ) == BN_CLICKED 
	    && LOWORD( wparam ) == IDC_KEYGEN_HIDEPWD ) {
	    HWND hwnd_a = GetDlgItem( dlg, IDC_KEYGEN_PASSPHRASE );
	    HWND hwnd_b = GetDlgItem( dlg, IDC_KEYGEN_PWDCHECK );
	    hide ^= 1;
	    SendMessage( hwnd_a, EM_SETPASSWORDCHAR, hide? '*' : 0, 0 );
	    SetFocus( hwnd_a );
	    SendMessage( hwnd_b, EM_SETPASSWORDCHAR, hide? '*' : 0, 0 );
	    SetFocus( hwnd_b );

	}

        switch( LOWORD( wparam ) ) {
        case IDOK:
            bits = GetDlgItemInt (dlg, IDC_KEYGEN_SUBKEYBITS, NULL, FALSE);
            if (bits < 1024 || bits > 4096) {
		msg_box (dlg, _("Invalid value. Allowed values 1024-4096 bits."), 
			 _("Key Generation"), MB_ERR);
                return FALSE;
            }
            if (bits > 2048) {
                int id = msg_box (dlg, _("Do you really need such a large key?"), 
				  _("Key Generation"), MB_YESNO);
                if (id == IDNO)
                    bits = 2048;
            }
            if( !GetDlgItemText( dlg, IDC_KEYGEN_NAME, name, sizeof name - 1 ) ) {
                msg_box( dlg, _("Please enter the name."), _("Key Generation"), MB_ERR );
                return FALSE;
            }
	    if (strchr (name, '@')) {
		msg_box (dlg, _("Please do not enter the email address in the name field."),
			 _("Key Generation"), MB_INFO);
		return FALSE;
	    }
            if( !GetDlgItemText(dlg, IDC_KEYGEN_EMAIL, email, sizeof email -1 )
                || !strchr( email, '@') ) {
                msg_box( dlg, _("Please enter a valid email address."), 
			 _("Key Generation"), MB_ERR );
                return FALSE;
            }
            use_comment = GetDlgItemText( dlg, IDC_KEYGEN_COMMENT, comment, sizeof comment -1 );
	    if( use_comment > 0 && strchr( comment, '@' ) ) {
		msg_box( dlg, _("Please do NOT enter the email address in the comment field."),
			 _("Key Generation"), MB_INFO );
		return FALSE;
	    }
            keytype = get_keytype( dlg );
            if (IsDlgButtonChecked (dlg, IDC_KEYGEN_EXPNEVER))
                expire = NULL;
            else {
		DateTime_GetSystemtime (GetDlgItem (dlg, IDC_KEYGEN_EXPDATE), &st);
		_snprintf (t, DIM (t)-1, "%04d-%02d-%02d", st.wYear, st.wMonth, st.wDay);
                expire = t;
	    }
            
            if( !GetDlgItemText( dlg, IDC_KEYGEN_PASSPHRASE, pwd, sizeof pwd -1 ) ) {
                msg_box( dlg, _("Please enter the passphrase."), 
			 _("Key Generation"), MB_ERR );
                return FALSE;
            }
            else if( strlen( pwd ) < 8 ) {
		int id = msg_box( dlg, _("Your passphrase should be at least 8 characters"
					 " long\nand should contain non-alphabetic characters."
					 "\n\nStill proceed?"),
				    _("Key Generation"), MB_ICONWARNING|MB_YESNO );
                if( id == IDNO ) {
                    SetDlgItemText( dlg, IDC_KEYGEN_PASSPHRASE, "" );
                    SetDlgItemText( dlg, IDC_KEYGEN_PWDCHECK, "" );
                    return FALSE;
                }
            }
            if( !GetDlgItemText( dlg, IDC_KEYGEN_PWDCHECK, pwd2, sizeof pwd2 -1 ) ) {
                msg_box( dlg, _("Please repeat the passphrase."), 
			 _("Key Generation"), MB_ERR );
                return FALSE;
            }
            if( strcmp( pwd, pwd2 ) ) {
                msg_box( dlg,  _("Passphrases are NOT identical!" ), 
			 _("Key Generation"), MB_ERR );
		memset( pwd, 0, sizeof pwd );
		memset( pwd2, 0, sizeof pwd2 );
                return FALSE;
            }
	    if( is_8bit_string( pwd ) ) {
		msg_box( dlg, _("The passphrase contains 8-bit characters.\n"
		                "It is not suggested to use charset specific characters."),
			 _("Key Generation"), MB_ERR );
		memset( pwd, 0, sizeof pwd );
		memset( pwd2, 0, sizeof pwd2 );
		SetDlgItemText( dlg, IDC_KEYGEN_PASSPHRASE, "" );
		SetDlgItemText( dlg, IDC_KEYGEN_PWDCHECK, "" );
		return FALSE;
	    }
		                 
            if( !use_comment && !strlen( comment ) ) {
                char *utf8_name;
		utf8_name = wincp_to_utf8 (name, strlen (name));
		if( !utf8_name )
		    BUG( dlg );
                p = gpgme_genkey_params( keytype, bits, utf8_name, NULL, email, expire, pwd );
                free( utf8_name );
            }
            else {
                char *utf8_name, *utf8_comment;
		utf8_name = wincp_to_utf8 (name, strlen (name));
		utf8_comment = wincp_to_utf8 (comment, strlen (comment));               
		if( !utf8_name || !utf8_comment )
		    BUG( dlg );
                p = gpgme_genkey_params( keytype, bits, utf8_name, utf8_comment, email, expire, pwd );
                free( utf8_name );
                free( utf8_comment );
            }
            keygen_cb_dlg_create( );
            err = gpgme_op_genkey_auto( p, keygen_cb, NULL );
            memset( pwd, 0, sizeof pwd );
            memset( pwd2, 0, sizeof pwd2 );
            if( p ) {
                memset( p, 0, strlen( p ) ); /* burn the passphrase! */
                free( p );
            }
            keygen_cb_dlg_destroy( );
            keygen_cb( NULL, NULL, 0, 0, 0 ); /* flush */
            if( err ) {
                msg_box( dlg, gpgme_strerror( err ), _("Key Generation"), MB_ERR );
                return FALSE;
            }
            status_box( dlg, _("Key Generation completed"), _("GnuPG Status") );
            keycache_set_reload( 1 );
            clear_dlg_fields( dlg );
            ask_for_backup( dlg );
            EndDialog( dlg, TRUE );
            return TRUE;
            
        case IDCANCEL:
            SetDlgItemText (dlg, IDC_KEYGEN_PASSPHRASE, "");
            SetDlgItemText (dlg, IDC_KEYGEN_PWDCHECK, "");
            EndDialog (dlg, FALSE);
            return FALSE;
        }
        break;
    }
    
    return FALSE;
} /* keygen_dlg_proc */


BOOL CALLBACK
keygen_wizard_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static key_wizard_s * ctx;
    static int pubkey_algo = GPGME_KEYGEN_DSA_ELG;
    gpgme_error_t err;
    char name[128], email[128];
    char * utf8_name, * p;
    char * pass = NULL;
    int cancel = 0;
    

    switch( msg ) {
    case WM_INITDIALOG:
	ctx = (key_wizard_s *)lparam;
	if (!ctx || (ctx && ctx->interactive == 0))
	    EnableWindow (GetDlgItem (dlg, IDC_KEYWIZARD_EXPERT), FALSE);
	SetDlgItemText (dlg, IDC_KEYWIZARD_USERSA, _("&Prefer RSA keys"));
	SetDlgItemText (dlg, IDC_KEYWIZARD_NAMEINF, _("Real name:"));
	SetDlgItemText (dlg, IDC_KEYWIZARD_EMAILINF, _("Email address:"));
	SetDlgItemText (dlg, IDC_KEYWIZARD_TITLEINF, _("Name and E-Mail Assignment"));
	SetDlgItemText (dlg, IDC_KEYWIZARD_TEXT1INF, _("Every key pair must have a name associated with it. The name and\nemail address let your correspondents that your public key they are\nusing belongs to us."));
	SetDlgItemText (dlg, IDC_KEYWIZARD_TEXT2INF, _("By accosiating an email address with your key pair, you will enable WinPT to assist your correspondents in selecting the correct public\nkey when communicating with you."));
	SetForegroundWindow (dlg);
	center_window (dlg);
	break;

    case WM_SYSCOMMAND:
	if( LOWORD( wparam ) == SC_CLOSE )
	    EndDialog( dlg, FALSE );

    case WM_COMMAND:
	switch( LOWORD( wparam ) ) {
	case IDC_KEYWIZARD_EXPERT:
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_KEYGEN, dlg,
			    keygen_dlg_proc, NULL);
	    EndDialog (dlg, TRUE);
	    break;

	case IDOK:
	    if( !GetDlgItemText( dlg, IDC_KEYWIZARD_NAME, name, sizeof name-1 ) ) {
		msg_box( dlg, _("Please enter the name."), 
			 _("Key Generation Wizard"), MB_ERR );
		return FALSE;
	    }
	    if (strchr (name, '@')) {
		msg_box (dlg, _("Please do not enter the email address in the name field."),
			 _("Key Generation Wizard"), MB_ERR);
		return FALSE;
	    }
	    if( !GetDlgItemText( dlg, IDC_KEYWIZARD_EMAIL, email, sizeof email-1 )
		|| !strchr( email, '@' ) ) { 
		msg_box( dlg, _("Please enter a valid email address."), 
		         _("Key Generation Wizard"), MB_ERR );
		return FALSE;
	    }
	    pass = request_passphrase2 (_("Key Generation"), &cancel);
	    if (cancel)
		return FALSE;
	    utf8_name = wincp_to_utf8 (name, strlen (name));
	    if( !utf8_name )
		BUG( NULL );
	    if (IsDlgButtonChecked (dlg, IDC_KEYWIZARD_USERSA))
		pubkey_algo = GPGME_KEYGEN_DSA_RSA;
	    p = gpgme_genkey_params (pubkey_algo, 2048, utf8_name, 
				     NULL, email, NULL, pass);
	    free( utf8_name );
	    keygen_cb_dlg_create();
	    err = gpgme_op_genkey_auto( p, keygen_cb, NULL );
	    keygen_cb_dlg_destroy();
	    keygen_cb( NULL, NULL, 0, 0, 0 );
	    if( p ) {
		memset( p, 0, strlen( p ) );
		free( p );
	    }
	    sfree_if_alloc (pass);
	    if( err ) {
                msg_box( dlg, gpgme_strerror( err ), _("Key Generation Wizard"), MB_ERR );
                return FALSE;
            }
            status_box( dlg, _("Key Generation completed"), _("GnuPG Status") );
            keycache_set_reload( 1 );
	    EndDialog( dlg, TRUE );
	    break;

	case IDCANCEL:
	    EndDialog( dlg, FALSE );
	    break;
	}
	break;
    }
    return FALSE;
}