/* wptCryptdiskSrv.cpp - Server application for Cryptdisk
 *
 *   Control program for a virtual disk driver for Windows NT/2000/XP.
 *   Copyright (C) 1999, 2000, 2001, 2002 Bo Brantn.
 *
 *   Heavily modified for the use with Cryptdisk by Timo Schulz
 *   (C) 2004 Timo Schulz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <windows.h>
#include <winioctl.h>
#include <stdio.h>
#include <sys/types.h>

#define PGM_NAME "CryptdiskSrv"

/* remember to change the values in wptErrors.h also */
#define CDISK_ERR_LOCK	 201
#define CDISK_ERR_MOUNT  202
#define CDISK_ERR_UMOUNT 203
#define CDISK_ERR_OPEN   204
#define CDISK_ERR_BUSY	 205
#define CDISK_ERR_QUERY  206

#define DEVICE_BASE_NAME    "\\FileDisk"
#define DEVICE_DIR_NAME     "\\Device"		DEVICE_BASE_NAME
#define DEVICE_NAME_PREFIX  DEVICE_DIR_NAME     DEVICE_BASE_NAME

#define FILE_DEVICE_FILE_DISK       0x8000

#define IOCTL_FILE_DISK_OPEN_FILE   CTL_CODE(FILE_DEVICE_FILE_DISK, 0x800, METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS)
#define IOCTL_FILE_DISK_CLOSE_FILE  CTL_CODE(FILE_DEVICE_FILE_DISK, 0x801, METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS)
#define IOCTL_FILE_DISK_QUERY_FILE  CTL_CODE(FILE_DEVICE_FILE_DISK, 0x802, METHOD_BUFFERED, FILE_READ_ACCESS)

enum cmd_t {
    CMD_MOUNT	=0,
    CMD_NEW	=1,
    CMD_UMOUNT	=2,
    CMD_UMOUNT2 =3,
};

struct private_ofi_s {
    INT		    forced;
    INT		    cmd;
    INT		    devnum;
    CHAR	    drvlet;
    LARGE_INTEGER   filesize;
    BOOLEAN         readonly;
    USHORT          flen;
    UCHAR           fname[MAX_PATH+1];
};


typedef struct _OPEN_FILE_INFORMATION {
    LARGE_INTEGER   FileSize;
    BOOLEAN         ReadOnly;
    USHORT          FileNameLength;
    UCHAR           FileName[1];
} OPEN_FILE_INFORMATION, *POPEN_FILE_INFORMATION;


static struct private_ofi_s ofi;
static HANDLE wait_ev = NULL;

static void 
print_lasterr (char * prefix)
{
    LPVOID msg;

    FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER |
        FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        GetLastError(),
        0,
        (LPTSTR) &msg,
        0,
        NULL
        );
    MessageBox (NULL, (LPTSTR)msg, prefix, MB_ICONERROR|MB_OK);
    LocalFree (msg);
}


int
filedisk_mount (int                     DeviceNumber,
		POPEN_FILE_INFORMATION  ofi,
		char                    drivelet,
		BOOLEAN                 CdImage)
{
    char    VolumeName[] = "\\\\.\\ :";
    char    DeviceName[255];
    HANDLE  Device;
    DWORD   BytesReturned;

    VolumeName[4] = drivelet;

    Device = CreateFile (VolumeName,
			 GENERIC_READ | GENERIC_WRITE,
			 FILE_SHARE_READ | FILE_SHARE_WRITE,
			 NULL,
			 OPEN_EXISTING,
			 FILE_FLAG_NO_BUFFERING,
			 NULL);
    if (Device != INVALID_HANDLE_VALUE) {
        SetLastError (ERROR_BUSY);
        print_lasterr (&VolumeName[4]);
        return CDISK_ERR_BUSY;
    }

    if (CdImage)
        sprintf (DeviceName, DEVICE_NAME_PREFIX "Cd" "%u", DeviceNumber);
    else
        sprintf (DeviceName, DEVICE_NAME_PREFIX "%u", DeviceNumber);

    if (!DefineDosDevice (DDD_RAW_TARGET_PATH,
			  &VolumeName[4],
			  DeviceName)) {
        print_lasterr (&VolumeName[4]);
        return CDISK_ERR_OPEN;
    }

    Device = CreateFile (VolumeName,
			 GENERIC_READ | GENERIC_WRITE,
			 FILE_SHARE_READ | FILE_SHARE_WRITE,
			 NULL,
			 OPEN_EXISTING,
			 FILE_FLAG_NO_BUFFERING,
			 NULL);

    if (Device == INVALID_HANDLE_VALUE) {
        print_lasterr (&VolumeName[4]);
        DefineDosDevice (DDD_REMOVE_DEFINITION, &VolumeName[4], NULL);
        return CDISK_ERR_OPEN;
    }

    if (!DeviceIoControl (Device,
			  IOCTL_FILE_DISK_OPEN_FILE,
			  ofi,
			  sizeof(OPEN_FILE_INFORMATION) + ofi->FileNameLength - 1,
			  NULL, 0, &BytesReturned, NULL)) {
        print_lasterr ("Cryptdisk Error");
        DefineDosDevice (DDD_REMOVE_DEFINITION, &VolumeName[4], NULL);
        return CDISK_ERR_MOUNT;
    }

    return 0;
}


static int 
filedisk_unmount2 (char drivelet, int flags, int forced)
{
    char    VolumeName[60] = "\\\\.\\ :";
    HANDLE  Device;
    DWORD   BytesReturned;

    if (!flags)
	VolumeName[4] = drivelet;
    else if (flags & 1)
	sprintf(VolumeName,"\\\\.\\FileDisk%d", drivelet);
    else if (flags & 2)
	sprintf(VolumeName,"\\\\.\\FileDiskCd%d",drivelet);

    Device = CreateFile (VolumeName,
			 GENERIC_READ | GENERIC_WRITE,
			 FILE_SHARE_READ | FILE_SHARE_WRITE,
			 NULL,
			 OPEN_EXISTING,
			 FILE_FLAG_NO_BUFFERING,
			 NULL);
    if (Device == INVALID_HANDLE_VALUE) {
	if (forced)
	    print_lasterr(&VolumeName[4]);
        return CDISK_ERR_OPEN;
    }

    if (!DeviceIoControl (Device, 
			  FSCTL_LOCK_VOLUME,
			  NULL, 0, NULL, 0,
			  &BytesReturned, NULL)) {
	if (forced) {
	    print_lasterr(&VolumeName[4]);
	    return CDISK_ERR_LOCK;
	}
    }

    if (!DeviceIoControl (Device,
			  IOCTL_FILE_DISK_CLOSE_FILE,
			  NULL, 0, NULL, 0, 
			  &BytesReturned, NULL )) {
	if (forced)
	    print_lasterr ("Cryptdisk Error");
        return CDISK_ERR_UMOUNT;
    }

    if (!DeviceIoControl (
        Device, FSCTL_DISMOUNT_VOLUME,
        NULL, 0, NULL, 0,
        &BytesReturned, NULL)) {
	if (forced)
	    print_lasterr (&VolumeName[4]);
        return CDISK_ERR_UMOUNT;
    }

    if (!DeviceIoControl (Device, 
			  FSCTL_UNLOCK_VOLUME,
			  NULL, 0, NULL, 0,
			  &BytesReturned, NULL)) {
	if (forced) {
	    print_lasterr (&VolumeName[4]);
	    return CDISK_ERR_LOCK;
	}
    }

    CloseHandle (Device);

    if (!DefineDosDevice (
        DDD_REMOVE_DEFINITION,
        &VolumeName[4], NULL)) {
	if (forced)
	    print_lasterr (&VolumeName[4]);
        return CDISK_ERR_UMOUNT;
    }

    return 0;
}

int 
filedisk_unmount (char drivelet, int forced)
{	    
    char i;

    if (drivelet) {
	int rc = filedisk_unmount2 (drivelet, 0, forced);
	if (rc) 
	    print_lasterr ("Unmount");
	return rc;
    }
    for (i=0; i<20;i++) {
	if (filedisk_unmount2 (i, 1, 0) == CDISK_ERR_OPEN)
	    break;
    }
    for (i=0; i<20;i++) {
	if (filedisk_unmount2 (i, 2, 0) == CDISK_ERR_OPEN)
	    break;
    }
    return 0;
}


int
filedisk_status (char drivelet, LARGE_INTEGER * size, int * rdonly)
{
    char                    VolumeName[] = "\\\\.\\ :";
    HANDLE                  Device;
    POPEN_FILE_INFORMATION  ofi;
    DWORD                   BytesReturned;

    VolumeName[4] = drivelet;

    Device = CreateFile (VolumeName,
			 GENERIC_READ,
			 FILE_SHARE_READ | FILE_SHARE_WRITE,
			 NULL,
			 OPEN_EXISTING,
			 FILE_FLAG_NO_BUFFERING,
			NULL);

    if (Device == INVALID_HANDLE_VALUE) {
        print_lasterr(&VolumeName[4]);
        return CDISK_ERR_OPEN;
    }

    ofi = (POPEN_FILE_INFORMATION)malloc (sizeof(OPEN_FILE_INFORMATION) + MAX_PATH);

    if (!DeviceIoControl (Device,
			  IOCTL_FILE_DISK_QUERY_FILE,
			  NULL,
			  0,
			  ofi,
			  sizeof(OPEN_FILE_INFORMATION) + MAX_PATH,
			  &BytesReturned, NULL)) {
        print_lasterr (&VolumeName[4]);
        return CDISK_ERR_QUERY;
    }

    if (BytesReturned < sizeof(OPEN_FILE_INFORMATION)) {
        SetLastError (ERROR_INSUFFICIENT_BUFFER);
        print_lasterr (&VolumeName[4]);
        return CDISK_ERR_QUERY;
    }

    if (size)
	*size = ofi->FileSize;
    if (rdonly)
	*rdonly = ofi->ReadOnly;

    return 0;
}


POPEN_FILE_INFORMATION
filedisk_init_info (int devnum, char drivelet, const char * file,
		    int ro, int cd_img, unsigned long size)
{
    POPEN_FILE_INFORMATION pfi;
    char * p;

    pfi = (POPEN_FILE_INFORMATION)
	calloc (1, sizeof(OPEN_FILE_INFORMATION) + strlen (file) + 7);
    p = (char *)pfi->FileName;

    if (file[0] == '\\') {
	if (file[1] == '\\' && file[2] != '.') {
	    /* \\server\share\path\filedisk.img */
	    strcpy(p, "\\??\\UNC");
	    strcat(p, file + 1);
	}
	else /* \Device\Harddisk0\Partition1\path\filedisk.img */
	    strcpy(p, file);
    }	
    else { // c:\path\filedisk.img
	strcpy (p, "\\??\\");
	strcat (p, file);
    }

    pfi->FileNameLength = (USHORT) strlen (p);
    pfi->ReadOnly = ro? TRUE : FALSE;
    if (cd_img)
	pfi->ReadOnly = TRUE;
    if (size)
	pfi->FileSize.QuadPart = size * 1024;
    return pfi;
}


void
filedisk_free_info (POPEN_FILE_INFORMATION pfi)
{
    if (!pfi)
	return;
    free (pfi);
}


LRESULT CALLBACK
def_window_proc (HWND hwnd, UINT msg, WPARAM w, LPARAM l)
{
    COPYDATASTRUCT * cds;

    switch (msg) {
    case WM_DESTROY:
	PostQuitMessage (0);
	break;

    case WM_COPYDATA:
	cds = (COPYDATASTRUCT *)l;
	if (!cds || cds->cbData == 0)
	    break;
	memcpy ((private_ofi_s *)&ofi, cds->lpData, cds->cbData);
	DestroyWindow (hwnd);
	break;
    }
    return DefWindowProc (hwnd, msg, w, l);
}


int WINAPI
WinMain (HINSTANCE h, HINSTANCE hp, LPSTR cl, int show)
{
    POPEN_FILE_INFORMATION pfi;
    WNDCLASS wc = {0, def_window_proc, 0, 0, h, 0, 0, 0, 0, PGM_NAME};
    HWND hwnd;
    MSG msg;
    int rc;

    wait_ev = CreateEvent (NULL, FALSE, FALSE, "cryptdisksrv.wait");
    if (!wait_ev) {
	MessageBox (NULL, "Could not create wait event.", "CryptdiskSrv",
		    MB_ICONERROR|MB_OK);
	return 0;
    }
     
    rc = RegisterClass (&wc);
    if (rc == FALSE) {
	MessageBox (NULL, "Could not register window class", "CryptdiskSrv", 
		    MB_ICONERROR|MB_OK);
	CloseHandle (wait_ev);
	return 0;
    }

    hwnd = CreateWindow (PGM_NAME,
			 PGM_NAME, 
			 0, 0, 0, 0, 0, 
			 NULL, 
			 NULL, 
			 h, 
			 NULL);
    if (hwnd == NULL) {
    	MessageBox (NULL, "Could not create window", "CryptdiskSrv", 
		    MB_ICONERROR|MB_OK);
	CloseHandle (wait_ev);
	return 0;
    }

    UpdateWindow (hwnd);
    while (GetMessage (&msg, hwnd, 0, 0)) {
	TranslateMessage (&msg);	
	DispatchMessage (&msg);
    }

    {
	FILE * logf;

	logf = fopen ("cryptdisk.log", "a+b");
	if (logf) {
	    fprintf (logf, "cmd=%d devnum=%d drive=%c name=%s len=%d size=%d forced=%d\r\n",
			ofi.cmd, ofi.devnum, ofi.drvlet, ofi.fname, ofi.flen, 
			ofi.filesize, ofi.forced);
	    fclose (logf);
	}
    }	    

    SetEvent (wait_ev);
    switch (ofi.cmd) {
    case CMD_NEW:
    case CMD_MOUNT:
	pfi = filedisk_init_info (ofi.devnum, ofi.drvlet, 
			    	  (const char *)ofi.fname,
				    0, 0, ofi.filesize.LowPart);
	rc = filedisk_mount (ofi.devnum, pfi, ofi.drvlet, FALSE);
	filedisk_free_info (pfi);
	break;

    case CMD_UMOUNT:
	rc = filedisk_unmount (ofi.drvlet, ofi.forced);
	break;

    case CMD_UMOUNT2:
	rc = filedisk_unmount (0, 0);
	break;
    }
    ResetEvent (wait_ev);
    CloseHandle (wait_ev);
    return rc;
}
