/* editcard.c - OpenPGP Smartcard Support
 *	Copyright (C) 2003, 2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>

#include "gpgme-config.h"
#ifdef WITH_EDITCARD
#include "ops.h"
#include "context.h"
#include "util.h"
#include "types.h"


#define save_write(val) ((val)? (val) : "")

#define ON_EVENT(_code, _key) (code == (_code) && !strcmp (key, _key))


static void
card_status_handler (gpgme_ctx_t ctx, gpg_status_code_t code, char * args)
{
    if (!ctx || ctx->out_of_core)
	return;
    
    switch( code ) {
    case STATUS_CARDCTRL:
	break;

    case STATUS_KEY_CREATED:	
	safe_free (ctx->keygen_fpr);
	ctx->keygen_fpr = strdup (args+2);
	if (!ctx->keygen_fpr)
	    ctx->out_of_core = 1;
	break;
    }
} /* card_status_handler */


static const char *
statuscard_command_handler (void * opaque, gpg_status_code_t code, 
			    const char * key)
{
    gpgme_ctx_t ctx = opaque;
    gpgme_editcard_t c = ctx->edit_opaque;

    if (!c || ctx->out_of_core)
	return NULL;

    if (ON_EVENT (STATUS_GET_LINE, "cardctrl.insert_card.okay")) 
    {
	if (c->card_cb )
	    return c->card_cb (1, c->cb_value);
    }

    return NULL;
} /* statuscard_command_handler */


static const char *
changepin_command_handler (void * opaque, gpg_status_code_t code, 
			   const char * key)
{
    gpgme_ctx_t ctx = opaque;
    gpgme_editcard_t c = ctx->edit_opaque;
    const char * s = "";

    if( !c || ctx->out_of_core )
	return NULL;

    if (ON_EVENT (STATUS_GET_LINE, "cardctrl.insert_card_okay")) 
    {
	if (c->card_cb)
	    return c->card_cb (1, c->cb_value);
    }
    if (ON_EVENT (STATUS_GET_LINE, "cardutil.change_pin.menu")) 
    {
	if (c->cnt == 0) 
	{
	    c->cnt++;
	    switch (ctx->edit_cmd) {
	    case GPGME_EDITCARD_CHUPIN: return "1";
	    case GPGME_EDITCARD_UNBPIN: return "2";
	    case GPGME_EDITCARD_CHAPIN: return "3";
	    default:
		c->cnt = 0;
		return "Q";
	    }
	}
	else if (c->cnt) {
	    c->cnt = 0;
	    return "Q";
	}
    }
    if (ctx->edit_cmd == GPGME_EDITCARD_CHUPIN) 
    {
	if (ON_EVENT (STATUS_GET_HIDDEN, "passphrase.pin.ask")) 
	{
	    if (c->cnt == 1)
		s = c->u.pin.pinold;
	    else
		s = c->u.pin.pinnew;
	    c->cnt++;
	}
	return s;
    }
    else if (ctx->edit_cmd == GPGME_EDITCARD_CHAPIN) 
    {
	if (ON_EVENT (STATUS_GET_HIDDEN, "passphrase.adminpin.ask")) 
	{
	    if (c->cnt == 1)
		s = c->u.pin.pinold;
	    else
		s = c->u.pin.pinnew;
	    c->cnt++;
	}
	return s;
    }
    else if (ctx->edit_cmd == GPGME_EDITCARD_UNBPIN) 
    {
	/* todo */
    }

    return NULL;
} /* changepin_command_handler */


static const char *
editcard_command_handler (void * opaque, gpg_status_code_t code, const char * key)
{
    gpgme_ctx_t ctx = opaque;
    gpgme_editcard_t c = ctx->edit_opaque;
    
    DEBUG3 ("type=%d code=%d key=%s\n", c->type, code, key);

    if (!c || ctx->out_of_core)
	return NULL;
    
    if (ON_EVENT (STATUS_GET_LINE, "cardctrl.insert_card.okay")) {
	if (c->card_cb)
	    return c->card_cb (1, c->cb_value);
    }

    if (ON_EVENT (STATUS_GET_LINE, "cardedit.prompt")) 
    {
	if (c->cnt == 0) /* first switch in the admin mode */
	{
	    c->cnt++;
	    return "admin";
	}
	if (c->cnt == 1) /* then run the send command */
	{ 
	    c->cnt++;
	    switch (ctx->edit_cmd) 
	    {
	    case GPGME_EDITCARD_NAME:   return "name";
	    case GPGME_EDITCARD_KEYURL: return "url";
	    case GPGME_EDITCARD_LOGIN:  return "login";
	    case GPGME_EDITCARD_SEX:    return "sex";
	    case GPGME_EDITCARD_LANG:   return "lang";
	    case GPGME_EDITCARD_GENKEY: return "generate";
	    default:			return "quit";
	    }
	}
	else if (c->cnt >= 2) /* done: send exit */
	{ 
	    c->cnt = 0;
	    return "quit";
	}
    }
    if (c->cnt > 0 && code == STATUS_GET_HIDDEN) 
    {
	if (!strcmp (key, "passphrase.adminpin.ask"))
	    return c->apin;
	if (!strcmp (key, "passphrase.pin.ask"))
	    return c->upin;
    }
    switch (ctx->edit_cmd) 
    {
    case GPGME_EDITCARD_NAME:
	if (ON_EVENT (STATUS_GET_LINE, "keygen.smartcard.surname"))
	    return save_write (c->u.edit.surname);
	else if (ON_EVENT (STATUS_GET_LINE, "keygen.smartcard.givenname"))
	    return save_write (c->u.edit.givenname);
	break;

    case GPGME_EDITCARD_KEYURL:
	if (ON_EVENT (STATUS_GET_LINE, "cardedit.change_url"))
	    return save_write (c->u.edit.keyurl);
	break;

    case GPGME_EDITCARD_LOGIN:
	if (ON_EVENT (STATUS_GET_LINE, "cardedit.change_login"))
	    return save_write (c->u.edit.login);
	break;

    case GPGME_EDITCARD_SEX:
	if (ON_EVENT (STATUS_GET_LINE, "cardedit.change_sex")) 
	{
	    if (c->u.edit.sex != 'M'
		&& c->u.edit.sex != 'F'
		&& c->u.edit.sex != ' ')
		return " ";
	    else 
	    {
		static char buf[2];
		buf[0] = c->u.edit.sex; buf[1] = 0;
		return buf;
	    }
	}
	break;

    case GPGME_EDITCARD_LANG:
	if (ON_EVENT (STATUS_GET_LINE, "cardedit.change_lang"))
	    return save_write (c->u.edit.lang);
	break;

    case GPGME_EDITCARD_GENKEY:
	if (ON_EVENT (STATUS_GET_HIDDEN, "passphrase.enter"))
	    return c->keygen.passwd;
	if (ON_EVENT (STATUS_GET_BOOL, "cardedit.genkeys.backup_enc") ||
	    ON_EVENT (STATUS_GET_LINE, "cardedit.genkeys.backup_enc"))
	    return c->keygen.flags & GPGME_CARDFLAG_BAKENC? "Y" : "N";
	if (ON_EVENT (STATUS_GET_BOOL, "cardedit.genkeys.replace_keys")) 
	{
	    if (! (c->keygen.flags & GPGME_CARDFLAG_REPLACE))
		c->cancel = 1;
	    return c->keygen.flags & GPGME_CARDFLAG_REPLACE? "Y" : "N";
	}
	else if (ON_EVENT (STATUS_GET_LINE, "keygen.valid"))
	    return save_write (c->keygen.expdate);
	else if (ON_EVENT (STATUS_GET_LINE, "keygen.name"))
	    return save_write (c->keygen.name);
	else if (ON_EVENT (STATUS_GET_LINE, "keygen.email"))
	    return save_write (c->keygen.email);
	else if (ON_EVENT (STATUS_GET_LINE, "keygen.comment"))
	    return save_write (c->keygen.comment);
	break;

    default:
	return NULL;
    }
    return NULL;
} /* editcard_command_handler */


static gpgme_error_t
changepin_start( gpgme_ctx_t ctx )
{
    gpgme_error_t rc;

    ctx->pending=1;

    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
	return rc;

    _gpgme_gpg_set_status_handler( ctx->gpg, card_status_handler, ctx );
    _gpgme_gpg_set_command_handler( ctx->gpg, changepin_command_handler, ctx );

    _gpgme_gpg_add_arg( ctx->gpg, "--change-pin" );

    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    if( rc ) {
	ctx->pending =0;
	_gpgme_gpg_release( &ctx->gpg );
    }

    return rc;
} /* changepin_start */


gpgme_error_t
gpgme_op_changepin( gpgme_ctx_t ctx )
{
    gpgme_error_t rc;

    rc = changepin_start( ctx );
    if( !rc ) {
	gpgme_wait( ctx, 1 );
	ctx->pending = 0;
	if( gpgme_get_process_rc( ctx ) )
	    rc = mk_error( Interal_GPG_Problem );
    }
    return rc;
} /* gpgme_op_changepin */


gpgme_error_t
gpgme_changepin_set( gpgme_editcard_t chpin, int which,
		     const char * pinold, const char * pinnew )
{
    if( !chpin )
	return mk_error( Invalid_Value );
    chpin->type = which;
    chpin->u.pin.pinnew = pinnew;
    chpin->u.pin.pinold = pinold;
    return 0;
} /* gpgme_changepin_set */


static gpgme_error_t
editcard_start (gpgme_ctx_t ctx)
{
    gpgme_error_t rc;

    ctx->pending = 1;

    _gpgme_gpg_release (&ctx->gpg);
    rc = _gpgme_gpg_new (&ctx->gpg);
    if (rc)
	return rc;

    _gpgme_gpg_set_status_handler (ctx->gpg, card_status_handler, ctx);
    if (ctx->edit_cmd)
	_gpgme_gpg_set_command_handler (ctx->gpg, editcard_command_handler, ctx);

    _gpgme_gpg_add_arg (ctx->gpg, "--card-edit");

    rc = _gpgme_gpg_spawn (ctx->gpg, ctx);
    if (rc) {
	ctx->pending = 0;
	_gpgme_gpg_release (&ctx->gpg);
    }

    return rc;
} /* editcard_start */


gpgme_error_t
gpgme_op_editcard (gpgme_ctx_t ctx)
{
    gpgme_error_t rc;

    rc = editcard_start (ctx);
    if (!rc) {
	gpgme_wait (ctx, 1);
	ctx->pending = 0;
	if (((gpgme_editcard_t)ctx->edit_opaque)->cancel)
	    rc = mk_error (Canceled);
	else if (ctx->keygen_fpr)
	    rc = mk_error (No_Error);
	else if (gpgme_get_process_rc (ctx))
	    rc = mk_error (Interal_GPG_Problem);
    }
    return rc;
} /* gpgme_op_editcard */


gpgme_error_t
gpgme_editcard_new (gpgme_editcard_t * r_ctx)
{
    gpgme_editcard_t ctx;

    if (!r_ctx)
	return mk_error (Invalid_Value);
    *r_ctx = NULL;
    ctx = calloc (1, sizeof * ctx);
    if (!ctx)
	return mk_error (Out_Of_Core);
    *r_ctx = ctx;
    return 0;
} /* gpgme_editcard_new */


void
gpgme_editcard_release (gpgme_editcard_t ctx)
{
    if (!ctx)
	return;
    safe_free (ctx->keygen.name);
    safe_free (ctx->keygen.comment);
    safe_free (ctx->keygen.email);
    safe_free (ctx->keygen.expdate);
    safe_free (ctx );
} /* gpgme_editcard_release */


gpgme_error_t
gpgme_editcard_set_passwd (gpgme_editcard_t ctx, const char * passwd)
{
    if (!ctx)
	return mk_error (Invalid_Value);
    ctx->type = GPGME_EDITCARD_GENKEY;
    ctx->keygen.passwd = passwd;
    return 0;
}

gpgme_error_t
gpgme_editcard_set_keygen_params (gpgme_editcard_t ctx, int flags,
				  const char * name, const char * email,
				  const char * comment, const char * expdate)
{
    char * p;

    if (!ctx)
	return mk_error (Invalid_Value);
    if (!name || !email)
	return mk_error (Invalid_Value);

    ctx->type = GPGME_EDITCARD_GENKEY;
    ctx->keygen.flags = flags;

    p = ctx->keygen.name = strdup (name);
    if (!p)
	return mk_error (Out_Of_Core);
    p = ctx->keygen.email = strdup (email);
    if (!p)
	return mk_error (Out_Of_Core);
    p = ctx->keygen.comment = comment? strdup (comment) : strdup ("");
    if (!p)
	return mk_error (Out_Of_Core);
    p = ctx->keygen.expdate = expdate? strdup (expdate) : strdup ("0");
    if (!p)
	return mk_error(Out_Of_Core);
    return 0;
} /* gpgme_editcard_set_keygen_params */


void
gpgme_editcard_set_callback (gpgme_editcard_t ctx, 
			     const char *(*cb) (int code, void * opaque),
			     void * cb_value)
{
    if (!ctx)
	return;
    ctx->card_cb = cb;
    ctx->cb_value = cb_value;
} /* gpgme_editcard_set_callback */


gpgme_error_t
gpgme_editcard_control (gpgme_editcard_t ctx, int cmd, const void * val)
{
    const char * p;

    if (!ctx)
	return mk_error (Invalid_Value);

    switch (cmd) {
    case GPGME_EDITCARD_NAME:  
	ctx->u.edit.givenname = (const char *)val;
	break;

    case GPGME_EDITCARD_NAME2: 
	ctx->u.edit.surname = (const char *)val; 
	break;

    case GPGME_EDITCARD_KEYURL:
	ctx->u.edit.keyurl = (const char *)val; 
	break;

    case GPGME_EDITCARD_LOGIN:
	ctx->u.edit.login = (const char *)val;
	break;

    case GPGME_EDITCARD_SEX:
	p = (const char *)val; 
	ctx->u.edit.sex = *p; 
	break;

    case GPGME_EDITCARD_LANG:  
	ctx->u.edit.lang = (const char *)val;
	break;

    case GPGME_EDITCARD_APIN:
	ctx->apin = (const char *)val; 
	break;

    case GPGME_EDITCARD_UPIN:
	ctx->upin = (const char *)val; 
	break;

    default:
	return mk_error (Invalid_Mode);
    }
    /* in some cases we do not want to change the command ID:
       1. when the given name will be set but the surname command
          already set the command ID.
       2. the pin is stored because the pin is no real command */
    if (cmd == GPGME_EDITCARD_NAME2
	|| cmd == GPGME_EDITCARD_APIN
	|| cmd == GPGME_EDITCARD_UPIN)
	;
    else
	ctx->type = cmd;
    return 0;
} /* gpgme_editcard_control */


static void
statuscard_colon_handler( gpgme_ctx_t ctx, char * line )
{
    enum rectype_t {
	CARD_None = 0,
	CARD_AID,
	CARD_Version,
	CARD_Vendor,
	CARD_Serial,
	CARD_Name,
	CARD_Lang,
	CARD_Sex,
	CARD_Url,
	CARD_Login,
	CARD_MaxPinLen,
	CARD_SigCount,
	CARD_Fpr
    };
    enum rectype_t rectype;
    gpgme_card_t card;
    char * t, * p, * pend;
    int field = 0;

    if( ctx->out_of_core )
	return;

    if( !line )
	return ; /* EOF */
    
    if( !ctx->card && gpgme_card_new( &ctx->card ) ) {
	ctx->out_of_core = 1;
	return;
    }
    card = ctx->card;

    for( p = line; p; p = pend ) {
	field++;
	pend = strchr( p, ':' );
        if( pend ) 
            *pend++ = 0;

	if( field == 1 ) {
	    if( !strcmp( p, "AID" ) )
		rectype = CARD_AID;
	    else if( !strcmp( p, "version" ) )
		rectype = CARD_Version;
	    else if( !strcmp( p, "vendor" ) )
		rectype = CARD_Vendor;
	    else if( !strcmp( p, "serial" ) )
		rectype = CARD_Serial;
	    else if( !strcmp( p, "name" ) )
		rectype = CARD_Name;
	    else if( !strcmp( p, "lang" ) )
		rectype = CARD_Lang;
	    else if( !strcmp( p, "sex" ) )
		rectype = CARD_Sex;
	    else if( !strcmp( p, "url" ) )
		rectype = CARD_Url;
	    else if( !strcmp( p, "login" ) )
		rectype = CARD_Login;
	    else if( !strcmp( p, "maxpinlen" ) )
		rectype = CARD_MaxPinLen;
	    else if( !strcmp( p, "sigcount" ) )
		rectype = CARD_SigCount;
	    else if( !strcmp( p, "fpr" ) )
		rectype = CARD_Fpr;
	    else
		rectype = CARD_None;
	}
	switch( rectype ) {
	case CARD_AID:
	    if( field == 2 ) {
		t = card->aid = strdup( p ); 
		if( !t )
		    ctx->out_of_core = 1;
	    }
	    break;
	case CARD_Version:
	    if( field == 2 ) {
		t = card->version = strdup( p );
		if( !t )
		    ctx->out_of_core = 1;
	    }
	    break;
	case CARD_Vendor:
	    if( field == 3 ) {
		t = card->vendor = strdup( p ); 
		if( !t )
		    ctx->out_of_core = 1;
	    }
	    break;
	case CARD_Serial:
	    if( field == 2 ) {
		t = card->serial = strdup( p );
		if( !t ) 
		    ctx->out_of_core = 1;
	    }
	    break;
	case CARD_Name:
	    if( field == 2 ) {
		t = card->givenname = strdup( p );
		if( !t )
		    ctx->out_of_core = 1;
	    }
	    else if( field == 3 ) {
		t = card->surname = strdup( p );
		if( !t )
		    ctx->out_of_core = 1;
	    }
	    break;
	case CARD_Lang:
	    if( field == 2 )
		card->lang = strdup( p ) ;
	    break;
	case CARD_Sex:
	    if( field == 2 )
		card->sex = *p;
	    break;
	case CARD_Url:
	    if( field == 2 ) {
		safe_free( card->url );
		card->url = calloc( 1, strlen( p ) + 1 );
		if( !card->url ) {
		    ctx->out_of_core = 1;
		    return;
		}
		_gpgme_decode_c_string (p, &card->url, strlen (p) + 1);
	    }
	    break;
	case CARD_Login:
	    if( field == 2 ) {
		t = card->login = strdup( p );
		if( !t )
		    ctx->out_of_core = 1;
	    }
	    break;
	case CARD_MaxPinLen:
	    break;
	case CARD_SigCount:
	    if( field == 2 )
		card->sig_count = atol( p );
	    break;
	case CARD_Fpr:
	    if( field == 2 || field == 3 || field == 4 ) {
		t = card->fpr[ctx->tmp_i++] = strdup( p );
		if( !t )
		    ctx->out_of_core = 1;
	    }
	    break;
	}
    }
} /* statuscard_colon_handler */


static gpgme_error_t
status_card_start( gpgme_ctx_t ctx, gpgme_data_t tmp_out )
{
    gpgme_error_t rc;

    if( !ctx || !tmp_out )
	return mk_error( Invalid_Value );
    _gpgme_data_set_mode( tmp_out, GPGME_DATA_MODE_IN );

    ctx->pending = 1;

    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
	return rc;

    _gpgme_gpg_set_command_handler( ctx->gpg,  statuscard_command_handler, ctx );
    if( ctx->use_logging )
	_gpgme_gpg_set_logging_handler( ctx->gpg, ctx );

    _gpgme_gpg_add_arg( ctx->gpg, "--with-colons" );
    _gpgme_gpg_add_arg( ctx->gpg, "--card-status" );
    _gpgme_gpg_add_arg( ctx->gpg, "--output" );
    _gpgme_gpg_add_arg( ctx->gpg, "-" );
    _gpgme_gpg_add_data( ctx->gpg, tmp_out, 1 );

    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    if( rc ) {
	ctx->pending = 0;
	_gpgme_gpg_release( &ctx->gpg );
    }
    return rc;
} /* status_card_start */


gpgme_error_t
gpgme_op_statuscard( gpgme_ctx_t ctx, gpgme_card_t * ret_card )
{
    gpgme_data_t tmp = NULL;
    gpgme_error_t rc;
    char buf[200];

    if (!ret_card || !ctx)
	return mk_error (Invalid_Value);

    rc = gpgme_data_new (&tmp);
    if (rc)
	return rc;
    
    rc = status_card_start( ctx, tmp );
    if( !rc ) {
	gpgme_wait( ctx, 1 );
	ctx->pending = 0;
	if( gpgme_get_process_rc( ctx ) )
	    rc = mk_error( Interal_GPG_Problem );
    }

    gpgme_data_rewind( tmp );
    while( !rc && gpgme_data_readline( tmp, buf, sizeof buf-2 ) )
	statuscard_colon_handler( ctx, buf );
    gpgme_data_release( tmp );
    ctx->tmp_i = 0;
    *ret_card = ctx->card;
    return rc;
} /* gpgme_op_statuscard */


gpgme_error_t
gpgme_card_new( gpgme_card_t * ret_card )
{
    gpgme_card_t c;

    if (!ret_card)
	return mk_error( Invalid_Value );
    *ret_card = NULL;
    c = calloc( 1, sizeof * c );
    if( !c )
	return mk_error( Out_Of_Core );
    *ret_card = c;
    return 0;
} /* gpgme_card_new */


void
gpgme_card_release( gpgme_card_t card )
{
    int i;

    if( !card )
	return;
    safe_free( card->aid );
    safe_free( card->version );
    safe_free( card->lang );
    safe_free( card->login );
    for( i=0; i < 3; i++ )
	safe_free( card->fpr[i] );
    safe_free( card->surname );
    safe_free( card->givenname );
    safe_free( card->serial );
    safe_free( card->vendor );
    safe_free( card->url );
    safe_free( card );
} /* gpgme_card_release */


const char *
gpgme_card_get_string_attr( gpgme_card_t card, gpgme_attr_t what, 
			    void ** reserved, int idx )
{
    const char * s;

    if( !card )
	return NULL;

    switch( what ) {
    case GPGME_ATTR_CARD_AID:   
	s=card->aid; 
	break;

    case GPGME_ATTR_CARD_VER:   
	s=card->version; 
	break;

    case GPGME_ATTR_CARD_VENDOR:
	s=card->vendor; 
	break;

    case GPGME_ATTR_CARD_NAME:  
	s=card->givenname; 
	break;

    case GPGME_ATTR_CARD_NAME2: 
	s=card->surname; 
	break;

    case GPGME_ATTR_CARD_LANG:  
	s=card->lang; 
	break;

    case GPGME_ATTR_CARD_URL:   
	s=card->url; 
	break;

    case GPGME_ATTR_CARD_LOGIN: 
	s=card->login; 
	break;

    case GPGME_ATTR_CARD_FPR:   
	if( idx >= 0 && idx <= 3 ) 
	    s=card->fpr[idx];	    
	else                       
	    s=NULL;
	break;

    case GPGME_ATTR_CARD_SERIAL:
	s=card->serial; 
	break;

    default:
	s=NULL; 
	break;
    }
    return s;
} /* gpgme_card_get_string_attr */


unsigned long
gpgme_card_get_ulong_attr( gpgme_card_t card, gpgme_attr_t what,
			   void ** reserved, int idx )
{
    unsigned long t;

    if( !card )
	return 0;

    switch( what ) {
    case GPGME_ATTR_CARD_SIGCOUNT:
	t = card->sig_count;
	break;

    case GPGME_ATTR_CARD_SEX:
	t = card->sex; 
	break;

    default:                       
	t = 0; 
	break;
    }
    return t;
} /* gpgme_card_get_ulong_attr */

#endif /* WITH_EDITCARD */