/* keycache-openpgp.c - OpenPGP code for the key cache
 *      Copyright (C) 2001, 2002, 2003, 2005 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <malloc.h>

#include "gpgme-config.h"
#ifdef WITH_OPENPGP
#include "gpgme.h"
#include "key.h"
#include "util.h"
#include "ops.h"
#include "openpgp.h"


static void
buffer_to_string (char * dst, size_t dlen, const byte *buf, size_t nbytes)
{
    char dig[3];
    size_t i;

    memset( dst, 0, dlen );
    for( i = 0; i < nbytes && dlen > 0; i++ ) {
        sprintf( dig, "%02X", buf[i] );
        strcat( dst, dig );
        dlen -= 2;
    }
} /* buffer_to_string */


static void
parse_secring (gpgme_keycache_t cache, const char *kid, const char *secring)
{
    PACKET *pkt = calloc (1, sizeof * pkt);
    PKT_secret_key *sk;
    gpg_iobuf_t inp;
    gpgme_error_t err;
    gpgme_key_t key;
    struct keycache_s *c=NULL;
    char keyid[16+1];

    inp = gpg_iobuf_open (secring);
    if (!inp) {
        safe_free (pkt);
        return;
    }
    gpg_iobuf_ioctl (inp, 3, 1, NULL);
    gpg_init_packet (pkt);
    while (gpg_parse_packet (inp, pkt) != -1) {
        if (pkt->pkttype == PKT_SECRET_KEY) {
            sk = pkt->pkt.secret_key;
            _snprintf (keyid, sizeof keyid-1, "%08lX",
                       gpg_keyid_from_sk (sk, NULL));
            if (kid && strcmp (kid, keyid) != 0)
                goto next;
            err = _gpgme_keycache_find_key2 (cache, keyid, 0, &key, &c);
            /*DEBUG2 ("parse_secring: keyid=%s %p\r\n", keyid, c);*/
            if (!err) {
                key->gloflags.is_protected = sk->is_protected;
                key->gloflags.divert_to_card = sk->protect.s2k.mode==1002? 1 : 0;
                if (c->pubpart != NULL) {
                    c->pubpart->gloflags.is_protected = sk->is_protected;
                    c->pubpart->gloflags.divert_to_card = sk->protect.s2k.mode==1002? 1 : 0;
                    /*DEBUG1 ("parse_secring: sync %s\r\n", keyid);*/
                }
            }
        }
next:
        gpg_free_packet (pkt);
        gpg_init_packet (pkt);
    }
    safe_free (pkt);
    gpg_iobuf_close (inp);
} /* parse_secring */


gpgme_error_t
keycache_prepare2 (gpgme_keycache_t ctx, const char *kid,
                   const char *pubring, const char *secring)
{
    gpgme_error_t err;
    gpgme_key_t key = NULL;
    gpg_iobuf_t inp;
    PACKET *pkt = calloc (1, sizeof * pkt);
    struct revoker_key_s *r;
    const byte *sym_prefs;
    char keyid[16+1], *id = NULL;
    int key_seen = 0;
    size_t nbytes = 0, nsym =0;

    if (secring) {
        parse_secring (ctx, kid, secring);
        if (!pubring) {
            safe_free(pkt);
            return 0;
        }
    }
    inp = gpg_iobuf_open( pubring );
    if( !inp ) {
        safe_free( pkt );
        return mk_error( File_Error );
    }
    gpg_iobuf_ioctl( inp, 3, 1, NULL ); /* disable cache */

    gpg_init_packet( pkt );
    while (gpg_parse_packet (inp, pkt) != -1) {
        if (pkt->pkttype == PKT_PUBLIC_KEY) {
            strcpy (keyid, "");
            key_seen = 1;
        }

        if( pkt->pkttype == PKT_SIGNATURE && pkt->pkt.signature->sig_class == 0x1F ) {
            if (pkt->pkt.signature->numrevkeys == 0)
                goto next;
            nbytes = pkt->pkt.signature->numrevkeys;
            _snprintf (keyid, sizeof keyid -1, "%08X", pkt->pkt.signature->keyid[1]);
            if (kid && strcmp (kid, keyid) != 0)
                goto next;
            err = gpgme_keycache_find_key (ctx, keyid, 0, &key);
            if (err)
                goto next;
            while( nbytes-- ) {
                PKT_signature *sig = pkt->pkt.signature;
                r = calloc( 1, sizeof *r );
                if (!r) {
                    err = mk_error (Out_Of_Core);
                    goto fail;
                }
                r->algo =  sig->revkey[nbytes]->algid;
                buffer_to_string( r->fpr, sizeof r->fpr-1, sig->revkey[nbytes]->fpr, 20 );
                r->next = key->rvks;
                key->rvks = r;
            }
        }
        if( pkt->pkttype == PKT_SIGNATURE && key_seen == 1 ) {
            if( (sym_prefs=gpg_parse_sig_subpkt( pkt->pkt.signature->hashed,
                                       SIGSUBPKT_PREF_SYM, &nsym )) == NULL)
                goto next;
            _snprintf (keyid, sizeof keyid - 1, "%08X", pkt->pkt.signature->keyid[1]);
            if (kid && strcmp (kid, keyid) != 0)
                goto next;
            err = gpgme_keycache_find_key (ctx, keyid, 0, &key);
            if (err)
                goto next;
            else if (nsym) {
                key->sym_prefs = calloc (1, nsym+1);
                if (!key->sym_prefs)
                    return mk_error (Out_Of_Core);
                memcpy (key->sym_prefs, sym_prefs, nsym);
            }
        }
        if (pkt->pkttype == PKT_USER_ID) {
            safe_free (id);
            id = strdup (pkt->pkt.user_id->name);
            if (!id) {
                err = mk_error (Out_Of_Core);
                goto fail;
            }
        }
        if( (pkt->pkttype == PKT_USER_ID || pkt->pkttype == PKT_ATTRIBUTE)
            && pkt->pkt.user_id->attrib_data && key ) {
            PKT_user_id *id = pkt->pkt.user_id;
            key->attrib.used = 1;
            key->attrib.len = id->attrib_len;
            key->attrib.d = malloc (id->attrib_len + 1);
            if( !key->attrib.d ) {
                err = mk_error( Out_Of_Core );
                goto fail;
            }
            memcpy (key->attrib.d, id->attrib_data, id->attrib_len);
            key = NULL;
        }
next:
        gpg_free_packet (pkt);
        gpg_init_packet(pkt);
    }

fail:
    safe_free (id);
    safe_free (pkt);
    gpg_iobuf_close (inp);
    return err;
} /* gpgme_keycache_prepare */


gpgme_error_t
gpgme_keycache_prepare (gpgme_keycache_t ctx, const char *pubr, const char *secr)
{
    return keycache_prepare2 (ctx, NULL, pubr, secr);
}

gpgme_error_t
gpgme_keycache_prepare_single (gpgme_keycache_t ctx, const char *keyid,
                               const char *pubr, const char *secr)
{
    if (!strncmp (keyid, "0x", 2))
        keyid += 2;
    DEBUG1 ("keycache_prepare_single: %s\r\n", keyid);
    return keycache_prepare2 (ctx, keyid, pubr, secr);
}
#endif /*WITH_OPENPGP*/
