/* editkey.c - GPG edit key interface
 *      Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "gpgme-config.h"
#ifdef WITH_EDITKEY
#include "ops.h"
#include "context.h"
#include "util.h"
#include "common-status.h"

#define GET_BOOL STATUS_GET_BOOL
#define GET_LINE STATUS_GET_LINE
#define GET_HIDDEN STATUS_GET_HIDDEN

#define do_check(code, exp, key, val) \
    (((code) == (exp)) && (strcmp ((key), (val) ) == 0))

#define is_sig_cmd(ctx)((ctx)->edit_cmd == GPGME_EDITKEY_SIGN \
                ||      (ctx)->edit_cmd == GPGME_EDITKEY_TSIGN \
                ||      (ctx)->edit_cmd == GPGME_EDITKEY_LSIGN \
                ||      (ctx)->edit_cmd == GPGME_EDITKEY_NRSIGN \
                ||      (ctx)->edit_cmd == GPGME_EDITKEY_NRLSIGN)

struct editkey_result_s {
    int bad_passphrase;
    int already_signed;
    int key_expired;
};


void
_gpgme_release_editkey_result (_editkey_result_t res)
{
    safe_free (res);
}


static void
editkey_status_handler( gpgme_ctx_t ctx, gpg_status_code_t code, char *args )
{
    if (ctx->out_of_core)
        return;

    if (ctx->result_type == RESULT_TYPE_NONE) {
        ctx->result.editk = calloc (1, sizeof * ctx->result.editk);
        if (!ctx->result.editk) {
            ctx->out_of_core = 1;
            return;
        }
        ctx->result_type = RESULT_TYPE_EDITKEY;
    }

    assert (ctx->result_type == RESULT_TYPE_EDITKEY);

    switch (code) {
    case STATUS_BAD_PASSPHRASE:
        ctx->result.editk->bad_passphrase++;
        break;

    case STATUS_ALREADY_SIGNED:
        ctx->result.editk->already_signed++;
        break;

    case STATUS_KEYEXPIRED:
    case STATUS_SIGEXPIRED:
        ctx->result.editk->key_expired = 1;
        break;

    case STATUS_PROGRESS:
        _gpgme_genkey_status_handler (ctx, code, args);
        break;

    case STATUS_KEY_CREATED:
        safe_free( ctx->keygen_fpr );
        ctx->keygen_fpr = strdup( args+2 );
        if( !ctx->keygen_fpr )
            ctx->out_of_core = 1;
        break;
    }

    /*DEBUG2 ("editkey_status: code=%d args=`%s'\n", code, args);*/
} /* editkey_status_handler */


static const char *
cmd_keyserv_handler (gpgme_editkey_t ctx, gpg_status_code_t code, const char * key)
{
    if (do_check (code, GET_LINE, key, "keyedit.add_keyserver"))
        return ctx->u.keyserv.url;
    if (do_check (code, GET_BOOL, key, "keyedit.confirm_keyserver"))
        return "Y";
    if (do_check (code, GET_HIDDEN, key, "passphrase.enter"))
        return ctx->u.keyserv.passwd;
    if (do_check (code, GET_LINE, key, "keyedit.prompt"))
        return "save";

    return NULL;
}

static const char*
cmd_sign_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key )
{
    /*DEBUG2 ("cmd_sign: code=%d args=`%s'\n", code, key);*/

    if( !strcmp( key, "sign_uid.class" ) ) {
        static char id[32];
        sprintf( id, "%d", ctx->u.sign.sig_class );
        return id;
    }
    if (!strcmp (key, "sign_uid.expire"))
        return "Y"; /* the sig expires when the key expires */
    if (!strcmp (key, "siggen.valid"))
        return ctx->u.sign.exp_date;
    if (!strcmp (key, "trustsig_prompt.trust_value"))
        return ctx->u.sign.trust.val;
    if (!strcmp (key, "trustsig_prompt.trust_depth"))
        return ""; /* fixme */
    if (!strcmp (key, "trustsig_prompt.trust_regexp"))
        return ""; /* fixme */
    if( do_check( code, GET_BOOL, key, "sign_uid.local_promote_okay" ) )
        return "Y";
    if( do_check( code, GET_BOOL, key, "sign_uid.okay" ) )
        return "Y";
    if( do_check( code, GET_BOOL, key, "keyedit.sign_all.okay" ) )
        return "Y";
    if( do_check( code, GET_HIDDEN, key, "passphrase.enter" ) )
        return ctx->u.sign.passwd;
    if( do_check( code, GET_LINE, key, "keyedit.prompt" ) )
        return "save";

    return NULL;
} /* cmd_sign_handler */


static const char*
cmd_trust_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key )
{
    if( do_check( code, GET_BOOL, key, "edit_ownertrust.set_ultimate.okay" ) )
        return "Y";
    if( do_check( code, GET_LINE, key, "edit_ownertrust.value" ) )
        return ctx->u.trust.trust_val;
    if( do_check( code, GET_LINE, key, "keyedit.prompt" ) )
        return "save";

    return NULL;
} /* cmd_trust_handler */


static const char*
cmd_adduid_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key )
{
    if( do_check( code, GET_LINE, key, "keygen.name" ) )
        return ctx->u.adduid.name;
    if( do_check( code, GET_LINE, key, "keygen.email" ) )
        return ctx->u.adduid.email;
    if( do_check( code, GET_LINE, key, "keygen.comment" ) ) {
        if( ctx->u.adduid.use_comment )
            return ctx->u.adduid.comment;
        return "";
    }
    if( do_check( code, GET_HIDDEN, key, "passphrase.enter" ) )
        return ctx->u.adduid.passwd;
    if( do_check( code, GET_LINE, key, "keyedit.prompt" ) )
        return "save";

    return NULL;
} /* cmd_adduid_handler */


static const char*
cmd_deluid_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key,
                   int *r_step )
{
    static char buf[64];
    int step = *r_step;

    if( step == 0  && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        sprintf( buf, "uid %d", ctx->u.deluid.id );
        *r_step = step = 1;
        return buf;
    }
    if( step == 1 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 2;
        return "deluid";
    }
    if( step == 2 && do_check( code, GET_BOOL, key, "keyedit.remove.uid.okay" ) ) {
        *r_step = step = 3;
        return "Y";
    }
    if( step == 3 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 0;
        return "save";
    }

    return NULL;
} /* cmd_deluid_handler */


static const char *
cmd_delsig_handler (gpgme_editkey_t ctx, gpg_status_code_t code, const char * key,
                    int * r_step)
{
    static char buf[64];
    int step = *r_step;

    if (step == 0 && do_check (code, GET_LINE, key, "keyedit.prompt"))
    {
        sprintf (buf, "uid %d", ctx->u.delsig.uid);
        *r_step = step = 1;
        return buf;
    }
    if (step == 1 && do_check (code, GET_LINE, key, "keyedit.prompt"))
    {
        *r_step = step = 2;
        return "delsig";
    }
    if (do_check (code, GET_BOOL, key, "keyedit.delsig.unknown") ||
        do_check (code, GET_BOOL, key, "keyedit.delsig.valid"))
    {
        if (++ctx->u.delsig.currno == ctx->u.delsig.signo)
            return "Y";
        else
            return "N";
    }
    if (ctx->u.delsig.signo == 0 &&
        do_check (code, GET_BOOL, key, "keyedit.delsig.selfsig"))
        return "Y";
    if (step == 2 && do_check (code, GET_LINE, key, "keyedit.prompt"))
    {
        *r_step = step = 0;
        return "save";
    }
    return NULL;
}


static const char*
cmd_delkey_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key,
                   int *r_step )
{
    static char buf[64];
    int step = *r_step;

    if( step == 0 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        sprintf( buf, "key %d", ctx->u.delkey.id );
        *r_step = step = 1;
        return buf;
    }
    if( step == 1 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 2;
        return "delkey";
    }
    if( step == 2 && do_check( code, GET_BOOL, key, "keyedit.remove.subkey.okay" ) ) {
        *r_step = step = 3;
        return "Y";
    }
    if( step == 3 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 0;
        return "save";
    }

    return NULL;
} /* cmd_delkey_handler */


static const char*
cmd_addkey_handler (gpgme_editkey_t ctx, gpg_status_code_t code, const char *key)
{
    static char buf[64];

    if( do_check( code, GET_HIDDEN, key, "passphrase.enter" ) )
        return ctx->u.addkey.passwd;
    if( do_check( code, GET_LINE, key, "keygen.algo" ) ) {
        _snprintf( buf, sizeof buf-1, "%d", ctx->u.addkey.algo );
        return buf;
    }
    if( do_check( code, GET_LINE, key, "keygen.size" ) ) {
        _snprintf( buf, sizeof buf-1, "%d", ctx->u.addkey.size );
        return buf;
    }
    if( do_check( code, GET_LINE, key, "keygen.valid" ) ) {
        _snprintf( buf, sizeof buf-1, "%d", ctx->u.addkey.valid );
        return buf;
    }
    if( do_check( code, GET_LINE, key, "keyedit.prompt" ) )
        return "save";
    return NULL;
} /* cmd_addkey_handler */


static const char*
cmd_passwd_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key )
{
    if( do_check( code, GET_HIDDEN, key, "passphrase.enter" )
        && !ctx->u.passwd.send_old ) {
        ctx->u.passwd.send_old = 1;
        return ctx->u.passwd.old_passwd;
    }
    if( do_check( code, GET_HIDDEN, key, "passphrase.enter" ) )
        return ctx->u.passwd.new_passwd;
    if( do_check( code, GET_BOOL, key, "change_passwd.empty.okay" ) )
        return ctx->u.passwd.allow_empty? "Y" : "N";
    if( do_check( code, GET_LINE, key, "keyedit.prompt" ) )
        return "save";
    return NULL;
} /* cmd_passwd_handler */


static const char *
cmd_setpref_handler (gpgme_editkey_t ctx, gpg_statcode_t code, const char * key)
{
    static char buf[128];

    if (do_check (code, GET_LINE, key, "keyedit.prompt") && ctx->u.pref.id == 0) {
        ctx->u.pref.id++;
        _snprintf (buf, sizeof buf-1, "setpref %s", ctx->u.pref.new_prefs);
        return buf;
    }
    if (do_check (code, GET_LINE, key, "keyedit.prompt") && ctx->u.pref.id == 1) {
        ctx->u.pref.id++;
        return "updpref";
    }
    if (do_check (code, GET_BOOL, key, "keyedit.updpref.okay"))
        return "Y";
    if (do_check (code, GET_HIDDEN, key, "passphrase.enter"))
        return ctx->u.pref.passwd;
    if (do_check (code, GET_LINE, key, "keyedit.prompt") && ctx->u.pref.id == 2) {
        ctx->u.pref.id = 0;
        return "save";
    }
    return NULL;
} /* cmd_setpref_handler */


static const char*
cmd_primary_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key,
                     int *r_step )
{
    static char buf[64];
    int step = *r_step;

    if( step == 0 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        sprintf( buf, "uid %d", ctx->u.primary.id );
        *r_step = step = 1;
        return buf;
    }
    if( step == 1 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 2;
        return "primary";
    }
    if( step == 2 && do_check( code, GET_HIDDEN, key, "passphrase.enter" ) ) {
        *r_step = step = 3;
        return ctx->u.primary.passwd;
    }
    if( step == 3 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 0;
        return "save";
    }
    return NULL;
} /* cmd_primary_handler */


static const char*
cmd_expire_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key,
                   int *r_step )
{
    static char buf[64];
    int step = *r_step;

    if( step == 0 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        sprintf(buf, "key %d", ctx->u.expire.id);
        *r_step = step = 1;
        return buf;
    }
    if( step == 1 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 2;
        return "expire";
    }
    if( step == 2 && do_check( code, GET_LINE, key, "keygen.valid" ) ) {
        *r_step = step = 3;
        if( ctx->u.expire.days ) {
            sprintf( buf, "%d", ctx->u.expire.days );
            return buf;
        }
        else
            return ctx->u.expire.date;
    }
    if( step == 3 && do_check( code, GET_HIDDEN, key, "passphrase.enter" ) ) {
        *r_step = step = 4;
        return ctx->u.expire.passwd;
    }
    if( step == 4 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 0;
        return "save";
    }

    return NULL;
} /* cmd_expire_handler */


const char*
cmd_revsig_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key,
                   int *r_step )
{
    static char buf[64];
    int step = *r_step;

    if( step == 0 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        sprintf( buf, "uid %d", ctx->u.revsig.id );
        *r_step = step = 1;
        return buf;
    }
    if( step == 1 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 2;
        return "revsig";
    }
    if( step == 2 && do_check( code, GET_BOOL, key, "ask_revoke_sig.one" ) ) {
        *r_step = step = 3;
        return "Y";
    }
    if( step == 3 && do_check( code, GET_BOOL, key, "ask_revoke_sig.okay" ) ) {
        *r_step = step = 4;
        return "Y";
    }
    if( step == 4 && do_check( code, GET_LINE, key, "ask_revocation_reason.code" ) ) {
        *r_step = step = 5;
        return "0";
    }
    if( step == 5 && do_check( code, GET_LINE, key, "ask_revocation_reason.text" ) ) {
        *r_step = step = 6;
        return "\n";
    }
    if( step == 6 && do_check( code, GET_BOOL, key, "ask_revocation_reason.okay" ) ) {
        *r_step = step = 7;
        return "Y";
    }
    if( step == 7 && do_check( code, GET_HIDDEN, key, "passphrase.enter" ) ) {
        *r_step = step = 8;
        return ctx->u.revsig.passwd;
    }
    if( step == 8 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 0;
        return "save";
    }

    return NULL;
} /* cmd_revsig_handler */


static const char *
cmd_revkey_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char * key,
                    int *r_step )
{
    int step = *r_step;
    static char buf[64];

    if( step == 0 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        sprintf( buf, "key %d", ctx->u.revkey.id );
        *r_step = step = 1;
        return buf;
    }
    if( step == 1 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 2;
        return "revkey";
    }
    if( step == 2 && do_check( code, GET_BOOL, key, "keyedit.revoke.subkey.okay" ) ) {
        *r_step = step = 3;
        return "Y";
    }
    if( step == 3 &&  do_check( code, GET_LINE, key, "ask_revocation_reason.code" ) ) {
        sprintf( buf, "%d", ctx->u.revkey.reason );
        *r_step = step = 4;
        return buf;
    }
    if( step == 4 && do_check( code, GET_LINE, key, "ask_revocation_reason.text" ) ) {
        *r_step = step = 5;
        return "";
    }
    if( step == 5 && do_check( code, GET_BOOL, key, "ask_revocation_reason.okay" ) ) {
        *r_step = step = 6;
        return "Y";
    }
    if( step == 6 && do_check( code, GET_HIDDEN, key, "passphrase.enter" ) ) {
        *r_step = step = 7;
        return ctx->u.revkey.passwd;
    }
    if( step == 7 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 0;
        return "save";
    }
    return NULL;
} /* cmd_revkey_handler */


static const char *
cmd_addrev_handler (gpgme_editkey_t ctx, gpg_status_code_t code, const char * key,
                    gpgme_ctx_t c, int * r_step)
{
    int step = *r_step;


    /*DEBUG3("code=%d key=%s step=%d\n", code, key, step);*/
    if ((step == 0 /*|| c->result.editk->already_signed*/)
        && do_check (code, GET_LINE, key, "keyedit.add_revoker")) {
        *r_step = step = 1;
        if (c->result.editk && c->result.editk->already_signed) {
            *r_step = step = 3;
            return "";
        }
        return ctx->u.addrev.uid;
    }
    if( step == 1 && do_check( code, GET_BOOL, key, "keyedit.add_revoker.okay" ) ) {
        *r_step = step = 2;
        return "Y";
    }
    if( step == 2 && do_check( code, GET_HIDDEN, key, "passphrase.enter" ) ) {
        *r_step = step = 3;
        return ctx->u.addrev.passwd;
    }
    if( step == 3 && do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 0;
        return "save";
    }
    return NULL;
} /* cmd_addrev_handler */


static const char *
cmd_addphoto_handler( gpgme_editkey_t ctx, gpg_status_code_t code, const char *key,
                      int *r_step )
{
    int step = *r_step;

    if( do_check( code, GET_LINE, key, "photoid.jpeg.add" ) )
        return ctx->u.addphoto.jpg;
    if( do_check( code, GET_BOOL, key, "photoid.jpeg.size" ) )
        return "Y";
    if( do_check( code, GET_HIDDEN, key, "passphrase.enter" ) )
        return ctx->u.addphoto.passwd;
    if( do_check( code, GET_LINE, key, "keyedit.prompt" ) ) {
        *r_step = step = 0;
        return "save";
    }
    return NULL;
} /* cmd_addphoto_handler */


static const char *
cmd_enable_disable_handler( gpgme_editkey_t ctx, gpg_status_code_t code,
                            const char * key, int *r_step )
{
    *r_step = 0;
    return "save";
} /* cmd_enable_disable_handler */


static const char *
editkey_command_handler (void * opaque, gpg_status_code_t code, const char * key)
{
    static int step = 0;
    gpgme_ctx_t c = opaque;
    gpgme_editkey_t ctx = c->edit_opaque;

    if (!code)
        return NULL;

    if (!ctx || ctx->type != c->edit_cmd) {
        DEBUG2 ("editkey cmd handler has a wrong type (%d != %d)", ctx->type, c->edit_cmd);
        return NULL;
    }

    switch (c->edit_cmd) {
    case GPGME_EDITKEY_LSIGN:
    case GPGME_EDITKEY_SIGN:
    case GPGME_EDITKEY_NRSIGN:
    case GPGME_EDITKEY_TSIGN:
    case GPGME_EDITKEY_NRLSIGN:
        return cmd_sign_handler (ctx, code, key);

    case GPGME_EDITKEY_TRUST:
        return cmd_trust_handler( ctx, code, key );

    case GPGME_EDITKEY_ADDUID:
        return cmd_adduid_handler( ctx, code, key );

    case GPGME_EDITKEY_DELUID:
        return cmd_deluid_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_DELSIG:
        return cmd_delsig_handler (ctx, code, key, &step);

    case GPGME_EDITKEY_DELKEY:
        return cmd_delkey_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_ADDKEY:
        return cmd_addkey_handler( ctx, code, key );

    case GPGME_EDITKEY_PASSWD:
        return cmd_passwd_handler( ctx, code, key );

    case GPGME_EDITKEY_PRIMARY:
        return cmd_primary_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_EXPIRE:
        return cmd_expire_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_REVSIG:
        return cmd_revsig_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_REVKEY:
        return cmd_revkey_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_ADDREV:
        return cmd_addrev_handler (ctx, code, key, c, &step);

    case GPGME_EDITKEY_ADDPHOTO:
        return cmd_addphoto_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_ENABLE:
        return cmd_enable_disable_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_DISABLE:
        return cmd_enable_disable_handler( ctx, code, key, &step );

    case GPGME_EDITKEY_SETPREF:
        return cmd_setpref_handler (ctx, code, key);

    case GPGME_EDITKEY_KEYSERV:
        return cmd_keyserv_handler (ctx, code, key);
    }

    return NULL;
} /* editkey_command_handler */


static gpgme_error_t
editkey_start (gpgme_ctx_t ctx, const char * keyid)
{
    gpgme_error_t rc;

    if (ctx->result_type == RESULT_TYPE_EDITKEY &&
        ctx->result.editk != NULL)
        memset (ctx->result.editk, 0, sizeof (struct editkey_result_s));

    fail_on_pending_request (ctx);
    ctx->pending = 1;

    /* create a process object */
    _gpgme_gpg_release( &ctx->gpg);
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;

    _gpgme_gpg_set_status_handler ( ctx->gpg, editkey_status_handler, ctx );
    if( ctx->edit_cmd ) {
        rc = _gpgme_gpg_set_command_handler( ctx->gpg, editkey_command_handler, ctx );
        if( rc )
            goto leave;
    }

    if (ctx->locusr) {
        _gpgme_gpg_add_arg (ctx->gpg, "-u ");
        _gpgme_gpg_add_arg (ctx->gpg, ctx->locusr);
    }
    if (is_sig_cmd (ctx) && ((gpgme_editkey_t)ctx->edit_opaque)->u.sign.exp_date)
        _gpgme_gpg_add_arg (ctx->gpg, "--ask-cert-expire");
    if (is_sig_cmd (ctx) && ((gpgme_editkey_t)ctx->edit_opaque)->u.sign.sig_class)
        _gpgme_gpg_add_arg (ctx->gpg, "--ask-cert-level");

    _gpgme_gpg_add_arg (ctx->gpg, "--edit-key");
    _gpgme_gpg_add_arg (ctx->gpg, keyid);
    switch( ctx->edit_cmd ) {
    case GPGME_EDITKEY_SIGN:
        _gpgme_gpg_add_arg( ctx->gpg, "sign" );
        break;

    case GPGME_EDITKEY_KEYSERV:
        _gpgme_gpg_add_arg (ctx->gpg, "keyserver");
        break;

    case GPGME_EDITKEY_LSIGN:
        _gpgme_gpg_add_arg (ctx->gpg, "lsign");
        break;

    case GPGME_EDITKEY_NRSIGN:
        _gpgme_gpg_add_arg (ctx->gpg, "nrsign");
        break;

    case GPGME_EDITKEY_NRLSIGN:
        _gpgme_gpg_add_arg (ctx->gpg, "nrlsign");
        break;

    case GPGME_EDITKEY_TSIGN:
        _gpgme_gpg_add_arg (ctx->gpg, "tsign");
        break;

    case GPGME_EDITKEY_ADDKEY:
        _gpgme_gpg_add_arg (ctx->gpg, "addkey");
        break;

    case GPGME_EDITKEY_DELUID:
        _gpgme_gpg_add_arg( ctx->gpg, "deluid" );
        break;

    case GPGME_EDITKEY_DELKEY:
        _gpgme_gpg_add_arg( ctx->gpg, "delkey" );
        break;

    case GPGME_EDITKEY_DELSIG:
        /* we need to choose the uid before */
        _gpgme_gpg_add_arg (ctx->gpg, "");
        break;

    case GPGME_EDITKEY_ADDUID:
        _gpgme_gpg_add_arg( ctx->gpg, "adduid" );
        break;

    case GPGME_EDITKEY_TRUST:
        _gpgme_gpg_add_arg( ctx->gpg, "trust" );
        break;

    case GPGME_EDITKEY_PASSWD:
        _gpgme_gpg_add_arg( ctx->gpg, "passwd" );
        break;

    case GPGME_EDITKEY_PRIMARY:
        /* we need to choose the uid before */
        _gpgme_gpg_add_arg( ctx->gpg, "" );
        break;

    case GPGME_EDITKEY_EXPIRE:
        /* we need to choose the key before */
        _gpgme_gpg_add_arg( ctx->gpg, "" );
        break;

    case GPGME_EDITKEY_REVSIG:
        /* we need to choose the uid before */
        _gpgme_gpg_add_arg( ctx->gpg, "" );
        break;

    case GPGME_EDITKEY_REVKEY:
        /* we need to choose the key before */
        _gpgme_gpg_add_arg( ctx->gpg, "" );
        break;

    case GPGME_EDITKEY_SETPREF:
        /* we cannot add the command here */
        _gpgme_gpg_add_arg (ctx->gpg, "");

    case GPGME_EDITKEY_ADDREV:
        _gpgme_gpg_add_arg( ctx->gpg, "addrevoker" );
        break;

    case GPGME_EDITKEY_ADDPHOTO:
        _gpgme_gpg_add_arg( ctx->gpg, "addphoto" );
        break;

    case GPGME_EDITKEY_ENABLE:
        _gpgme_gpg_add_arg( ctx->gpg, "enable" );
        break;

    case GPGME_EDITKEY_DISABLE:
        _gpgme_gpg_add_arg( ctx->gpg, "disable" );
        break;

    default:
        rc = mk_error( Invalid_Value );
        goto leave;
    }
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );

leave:
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }

    return rc;
} /* editkey_start */


static int
check_edit_cmd (gpgme_ctx_t ctx)
{
    if (ctx->edit_cmd == GPGME_EDITKEY_DELKEY ||
        ctx->edit_cmd == GPGME_EDITKEY_DELSIG)
        return 0;
    return 1;
}

gpgme_error_t
gpgme_op_editkey( gpgme_ctx_t ctx, const char * keyid )
{
    gpgme_error_t rc;
    int use_key_check = check_edit_cmd (ctx);

    rc = editkey_start( ctx, keyid );
    if( !rc ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
        if( ctx->result.editk->bad_passphrase )
            rc = mk_error( Bad_Passphrase );
        else if( ctx->result.editk->already_signed )
            rc = mk_error( Conflict );
        else if (use_key_check && ctx->result.editk->key_expired)
            rc = mk_error (Invalid_Mode);
        else if( gpgme_get_process_rc( ctx ) )
            rc = mk_error( Internal_GPG_Problem );
    }
    return rc;
} /* gpgme_op_editkey */


gpgme_error_t
gpgme_uid_info_new( gpgme_uidinfo_t *r_inf )
{
    gpgme_uidinfo_t c;

    if( !r_inf )
        return mk_error( Invalid_Value );
    *r_inf = NULL;
    c = calloc( 1, sizeof *c );
    if( !c )
        return mk_error( Out_Of_Core );
    *r_inf = c;
    return 0;
} /* gpgme_uid_info_new */


void
gpgme_uid_info_release (gpgme_uidinfo_t inf)
{
    struct user_id_info_s *i, *list;

    if (!inf)
        return;
    list = inf->list;
    while (list) {
        i = list->next;
        if (list->name) {
            safe_free (list->name);
            list->name = NULL;
        }
        if (list->prefs) {
            safe_free (list->prefs);
            list->prefs = NULL;
        }
        safe_free (list);
        list = i;
    }
} /* gpgme_uid_info_release */


static void
edit_key_colon_handler (gpgme_uidinfo_t inf, char * line)
{
    char *p, *pend;
    int field = 0, len = 0;
    struct user_id_info_s *i, *t;

    if (!line || strlen (line) < 3 || strncmp (line, "uid", 3))
        return;

    i = calloc (1, sizeof *i);
    if (!i)
        return;
    if (!inf->list)
        inf->list = i;
    else {
        for (t=inf->list; t->next; t=t->next)
            ;
        t->next = i;
    }

    p = strdup (line);
    if (!p)
        return;
    while (1) {
        field++;
        pend = strsep (&p, ":");
        if (pend == NULL)
            break;

        switch (field) {
        case 2: /* trust info */
            break;

        case 10: /* user ID */
            i->name = calloc (1, strlen (pend)+1);
            if (!i->name)
                return;
            _gpgme_decode_c_string (pend, &i->name, strlen (pend)+ 1);
            if (strchr (pend, '<') != NULL) {
                int pos = strchr (i->name, '<')- i->name + 1;
                int end = strchr (i->name, '>') - i->name;
                i->email = calloc (1, end-pos+2);
                if (!i->email)
                    return;
                memcpy (i->email, i->name+pos, (end-pos));
            }
            break;

        case 13: /* preferences */
            if (strstr (pend, "mdc")) {
                len = strlen (pend) - 4; /* ,mdc */
                if (strstr (pend, "no-ks-modify")) {
                    i->flags.no_ks_modify = 1;
                    len -= 13; /* ,no-ks-modify */
                }
                i->prefs = calloc (1, len+1);
                if (!i->prefs)
                    return;
                memcpy (i->prefs, pend, len);
                i->prefs[len] = '\0';
                i->flags.mdc = 1;
            }
            else {
                i->prefs = strdup (pend);
                if (!i->prefs)
                    return;
                i->flags.mdc = 0;
            }
            break;

        case 14: /* idx/flags */
            i->idx = atol (pend);
            if (strchr (pend, 'r'))
                i->flags.revoked = 1;
            if (strchr( pend, 'p'))
                i->flags.primary = 1;
            break;
        }
    }
    safe_free (p);
} /* edit_key_colon_handler */


static gpgme_error_t
editkey_get_info_start (gpgme_ctx_t ctx, const char *keyid, gpgme_data_t out)
{
    gpgme_error_t rc;

    fail_on_pending_request( ctx );
    ctx->pending = 1;

    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;

    _gpgme_data_set_mode( out, GPGME_DATA_MODE_IN );

    _gpgme_gpg_add_arg( ctx->gpg, "--with-colons" );
    _gpgme_gpg_add_arg( ctx->gpg, "--edit-key" );
    _gpgme_gpg_add_arg( ctx->gpg, keyid );
    _gpgme_gpg_add_arg( ctx->gpg, "quit" );
    _gpgme_gpg_add_arg( ctx->gpg, "--output" );
    _gpgme_gpg_add_arg( ctx->gpg, "-" );
    _gpgme_gpg_add_data( ctx->gpg, out, 1 );

    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );

leave:
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }
    return rc;
} /* editkey_get_info_start */


gpgme_error_t
gpgme_op_editkey_get_info (gpgme_ctx_t ctx, const char * keyid,
                           gpgme_uidinfo_t * r_inf)
{
    gpgme_error_t err;
    gpgme_data_t out = NULL;
    gpgme_uidinfo_t inf;
    char buf[512];

    if (!ctx)
        return mk_error (Invalid_Value);

    err = gpgme_data_new( &out );
    if (err)
        return err;

    err = editkey_get_info_start( ctx, keyid, out );
    if (!err) {
        gpgme_wait (ctx, 1);
        ctx->pending = 0;
    }
    if (err)
        return err;

    err = gpgme_uid_info_new (&inf);
    if (err)
        return err;

    while (gpgme_data_readline (out, buf, sizeof buf -1))
        edit_key_colon_handler (inf, buf);

    gpgme_data_release (out);
    if (!err && r_inf)
        *r_inf = inf;

    return err;
} /* gpgme_op_editkey_get_info */


int
gpgme_editkey_count_items( gpgme_uidinfo_t inf )
{
    struct user_id_info_s *i;
    int ncount = 0;

    for( i = inf->list; i; i = i->next )
        ncount++;

    return ncount;
} /* gpgme_editkey_count_items */


ulong
gpgme_editkey_get_ulong_attr( gpgme_uidinfo_t inf, int what, int idx )
{
    struct user_id_info_s *i;
    ulong val;

    if( !inf )
        return 0;

    switch( what ) {
    case GPGME_ATTR_UID_REVOKED:
        for( i = inf->list; i && idx; i = i->next, idx-- )
            ;
        if( i )
            val = i->flags.revoked;
        break;

    case GPGME_ATTR_MDC:
        for( i = inf->list; i && idx; i = i->next, idx-- )
            ;
        if( i )
            val = i->flags.mdc;
        break;

    case GPGME_ATTR_UID_PRIMARY:
        for( i = inf->list; i && idx; i = i->next, idx-- )
            ;
        if( i )
            val = i->flags.primary;
        break;

    case GPGME_ATTR_VALIDITY:
        for( i = inf->list; i && idx; i = i->next, idx-- )
            ;
        if( i )
            val = i->validity;
        break;

    case GPGME_ATTR_LEVEL:
        for( i = inf->list; i && idx; i = i->next, idx-- )
            ;
        if( i )
            val = i->idx;
        break;
    }

    return val;
} /* gpgme_editkey_get_ulong_attr */


const char*
gpgme_editkey_get_string_attr (gpgme_uidinfo_t inf, int what, int idx)
{
    const char *val=NULL;
    struct user_id_info_s *i;

    if( !inf )
        return NULL;

    switch( what ) {
    case GPGME_ATTR_NAME:
        for (i = inf->list; i && idx; i = i->next, idx--)
            ;
        if (i)
            val = i->name;
        break;

    case GPGME_ATTR_EMAIL:
        for (i=inf->list; i && idx; i = i->next, idx--)
            ;
        if (i)
            val = i->email;
        break;

    case GPGME_ATTR_UID_PREFS:
        for (i = inf->list; i && idx; i = i->next, idx--)
            ;
        if (i)
            val = i->prefs;
        break;

    default:
        val = NULL;
        break;
    }

    return val;
} /* gpgme_editkey_get_string_attr */

#endif /*WITH_EDITKEY*/
