/*
 *      Copyright (C) 2000,2001 Nikos Mavroyanopoulos
 *
 * This file is part of GNUTLS.
 *
 * GNUTLS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GNUTLS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#define LIBGNUTLS_VERSION "0.1.9"

#define GNUTLS_AES GNUTLS_RIJNDAEL

typedef enum BulkCipherAlgorithm { GNUTLS_NULL_CIPHER=1, GNUTLS_ARCFOUR, GNUTLS_3DES_CBC, GNUTLS_RIJNDAEL_CBC, GNUTLS_TWOFISH_CBC, GNUTLS_RIJNDAEL256_CBC } BulkCipherAlgorithm;
typedef enum KXAlgorithm { GNUTLS_KX_RSA=1, GNUTLS_KX_DHE_DSS, GNUTLS_KX_DHE_RSA, GNUTLS_KX_DH_DSS, GNUTLS_KX_DH_RSA, GNUTLS_KX_DH_ANON, GNUTLS_KX_SRP } KXAlgorithm;
typedef enum CredType { GNUTLS_X509PKI=1, GNUTLS_ANON, GNUTLS_SRP } CredType;
typedef enum MACAlgorithm { GNUTLS_NULL_MAC=1, GNUTLS_MAC_MD5, GNUTLS_MAC_SHA } MACAlgorithm;
typedef enum CompressionMethod { GNUTLS_NULL_COMPRESSION=1, GNUTLS_ZLIB } CompressionMethod;
typedef enum ConnectionEnd { GNUTLS_SERVER=1, GNUTLS_CLIENT } ConnectionEnd;
typedef enum AlertDescription { GNUTLS_CLOSE_NOTIFY, GNUTLS_UNEXPECTED_MESSAGE=10, GNUTLS_BAD_RECORD_MAC=20,
			GNUTLS_DECRYPTION_FAILED, GNUTLS_RECORD_OVERFLOW,  GNUTLS_DECOMPRESSION_FAILURE=30,
			GNUTLS_HANDSHAKE_FAILURE=40, GNUTLS_BAD_CERTIFICATE=42, GNUTLS_UNSUPPORTED_CERTIFICATE,
			GNUTLS_CERTIFICATE_REVOKED, GNUTLS_CERTIFICATE_EXPIRED, GNUTLS_CERTIFICATE_UNKNOWN,
			GNUTLS_ILLEGAL_PARAMETER, GNUTLS_UNKNOWN_CA, GNUTLS_ACCESS_DENIED, GNUTLS_DECODE_ERROR=50,
			GNUTLS_DECRYPT_ERROR, GNUTLS_EXPORT_RESTRICTION=60, GNUTLS_PROTOCOL_VERSION=70,
			GNUTLS_INSUFFICIENT_SECURITY, GNUTLS_INTERNAL_ERROR=80, GNUTLS_USER_CANCELED=90,
			GNUTLS_NO_RENEGOTIATION=100
			} AlertDescription;

typedef enum GNUTLS_Version { GNUTLS_SSL3=1, GNUTLS_TLS1 } GNUTLS_Version;

#define SOCKET int
#define LIST ...

struct GNUTLS_STATE_INT;
typedef struct GNUTLS_STATE_INT* GNUTLS_STATE;

typedef struct {
	unsigned char * data;
	int size;
} gnutls_datum;

typedef enum PKAlgorithm { GNUTLS_PK_RSA = 1, GNUTLS_PK_DSA,	/* sign only */
	GNUTLS_PK_DH
} PKAlgorithm;

/* You cannot use these structures directly
 */
typedef struct gnutls_cert gnutls_cert;
typedef struct gnutls_private_key gnutls_private_key;


/* internal functions */

int gnutls_init(GNUTLS_STATE * state, ConnectionEnd con_end);
int gnutls_deinit(GNUTLS_STATE state);
int gnutls_bye(SOCKET cd, GNUTLS_STATE state, int wait);

int gnutls_handshake(SOCKET cd, GNUTLS_STATE state);
int gnutls_check_pending(GNUTLS_STATE state);
int gnutls_rehandshake(SOCKET cd, GNUTLS_STATE state);

AlertDescription gnutls_get_last_alert( GNUTLS_STATE state);

/* get information on the current state */
BulkCipherAlgorithm gnutls_get_current_cipher( GNUTLS_STATE state);
KXAlgorithm gnutls_get_current_kx( GNUTLS_STATE state);
MACAlgorithm gnutls_get_current_mac_algorithm( GNUTLS_STATE state);
CompressionMethod gnutls_get_current_compression_method( GNUTLS_STATE state);

/* the name of the specified algorithms */
const char *gnutls_cipher_get_name( BulkCipherAlgorithm);
const char *gnutls_mac_get_name( MACAlgorithm);
const char *gnutls_compression_get_name( CompressionMethod);
const char *gnutls_kx_get_name( KXAlgorithm algorithm);


/* error functions */
int gnutls_is_fatal_error( int error);
void gnutls_perror( int error);
const char* gnutls_strerror( int error);

ssize_t gnutls_send(SOCKET cd, GNUTLS_STATE state, void *data, size_t sizeofdata, int flags);
ssize_t gnutls_recv(SOCKET cd, GNUTLS_STATE state, void *data, size_t sizeofdata, int flags);
ssize_t gnutls_write(SOCKET cd, GNUTLS_STATE state, void *data, size_t sizeofdata);
ssize_t gnutls_read(SOCKET cd, GNUTLS_STATE state, void *data, size_t sizeofdata);

/* functions to set priority of cipher suites */
int gnutls_set_cipher_priority( GNUTLS_STATE state, LIST);
int gnutls_set_mac_priority( GNUTLS_STATE state, LIST);
int gnutls_set_compression_priority( GNUTLS_STATE state, LIST);
int gnutls_set_kx_priority( GNUTLS_STATE state, LIST);
int gnutls_set_protocol_priority( GNUTLS_STATE state, LIST);


/* set our version - 0 for TLS 1.0 and 1 for SSL3 */
GNUTLS_Version gnutls_get_current_version(GNUTLS_STATE state);
const char *gnutls_version_get_name(GNUTLS_Version version);


/* get/set session */
int gnutls_set_current_session( GNUTLS_STATE state, void* session, int session_size);
int gnutls_get_current_session( GNUTLS_STATE state, void* session, int *session_size);
/* returns the session ID */
int gnutls_get_current_session_id( GNUTLS_STATE state, void* session, int *session_size);

int gnutls_set_lowat( GNUTLS_STATE state, int num);
int gnutls_set_cache_expiration( GNUTLS_STATE state, int seconds);
int gnutls_set_db_name( GNUTLS_STATE state, char* filename);	
int gnutls_clean_db( GNUTLS_STATE state);

/* returns libgnutls version */
const char* gnutls_check_version();

/* Functions for setting/clearing credentials */
int gnutls_clear_creds( GNUTLS_STATE state);
/* cred is a structure defined by the kx algorithm */
int gnutls_set_cred( GNUTLS_STATE, CredType type, void* cred);

CredType gnutls_get_auth_info_type( GNUTLS_STATE);
const void* gnutls_get_auth_info( GNUTLS_STATE);

/* A null terminated string containing the dnsname.
 * This will only exist if the client supports the dnsname
 * TLS extension. (draft-ietf-tls-extensions)
 */
const char* gnutls_ext_get_dnsname( GNUTLS_STATE);
int gnutls_ext_set_dnsname( GNUTLS_STATE, const char* dnsname);

/* This will set the Common Name field in case of X509PKI
 * authentication. This will be used while verifying the
 * certificate
 */
int gnutls_x509_set_cn( GNUTLS_STATE, const char* cn);


/* Credential structures for SRP - used in gnutls_set_cred(); */

struct DSTRUCT;
typedef struct DSTRUCT* X509PKI_CREDENTIALS;
typedef X509PKI_CREDENTIALS X509PKI_CLIENT_CREDENTIALS;
typedef X509PKI_CREDENTIALS X509PKI_SERVER_CREDENTIALS;

typedef struct DSTRUCT* SRP_SERVER_CREDENTIALS;
typedef struct DSTRUCT* SRP_CLIENT_CREDENTIALS;

typedef struct DSTRUCT* ANON_SERVER_CREDENTIALS;
typedef struct DSTRUCT* ANON_CLIENT_CREDENTIALS;


void gnutls_free_srp_client_sc( SRP_CLIENT_CREDENTIALS sc);
int gnutls_allocate_srp_client_sc( SRP_CLIENT_CREDENTIALS *sc);
int gnutls_set_srp_client_cred( SRP_CLIENT_CREDENTIALS res, char *username, char* password);

void gnutls_free_srp_server_sc( SRP_SERVER_CREDENTIALS sc);
int gnutls_allocate_srp_server_sc( SRP_SERVER_CREDENTIALS *sc);
int gnutls_set_srp_server_cred( SRP_SERVER_CREDENTIALS res, char *password_file, char* password_conf_file);

void gnutls_free_anon_server_sc( ANON_SERVER_CREDENTIALS sc);
int gnutls_allocate_anon_server_sc( ANON_SERVER_CREDENTIALS *sc);
int gnutls_set_anon_server_cred( ANON_SERVER_CREDENTIALS res, int dh_bits);


/* auth structures */

typedef struct {
	int dh_bits;
} ANON_CLIENT_AUTH_INFO;

/* for now it's the same */
typedef ANON_CLIENT_AUTH_INFO ANON_SERVER_AUTH_INFO;

typedef struct {
	char username[256];
} SRP_SERVER_AUTH_INFO;

typedef enum CertificateStatus { GNUTLS_CERT_TRUSTED=1, GNUTLS_CERT_NOT_TRUSTED, GNUTLS_CERT_EXPIRED, GNUTLS_CERT_WRONG_CN, GNUTLS_CERT_INVALID } CertificateStatus;

typedef struct {
	char common_name[256];
	char country[3];
	char organization[256];
	char organizational_unit_name[256];
	char locality_name[256];
	char state_or_province_name[256];
} gnutls_DN;

typedef struct {
	gnutls_DN	  peer_dn;
	gnutls_DN	  issuer_dn;
	CertificateStatus peer_certificate_status;
	int		  peer_certificate_version;
	time_t		  peer_certificate_activation_time;
	time_t		  peer_certificate_expiration_time;
} X509PKI_CLIENT_AUTH_INFO;


/* CERTFILE is an x509 certificate in PEM form.
 * KEYFILE is a pkcs-1 private key in PEM form (for RSA keys).
 */
void gnutls_free_x509_sc( X509PKI_CREDENTIALS sc);
int gnutls_allocate_x509_sc( X509PKI_CREDENTIALS *sc, int ncerts);
int gnutls_set_x509_key( X509PKI_CREDENTIALS res, char *CERTFILE, char* KEYFILE);
int gnutls_set_x509_trust( X509PKI_CREDENTIALS res, char* CAFILE, char* CRLFILE);

#define gnutls_free_x509_server_sc gnutls_free_x509_sc
#define gnutls_allocate_x509_server_sc gnutls_allocate_x509_sc
#define gnutls_set_x509_server_key gnutls_set_x509_key
#define gnutls_set_x509_server_trust gnutls_set_x509_trust

#define gnutls_free_x509_client_sc gnutls_free_x509_sc
#define gnutls_allocate_x509_client_sc gnutls_allocate_x509_sc
#define gnutls_set_x509_client_key gnutls_set_x509_key
#define gnutls_set_x509_client_trust gnutls_set_x509_trust


/* global state functions 
 */
/* In this version global_init accepts two files (pkix.asn, pkcs1.asn).
 * This will not be the case in the final version. These files 
 * are located in the src/ directory of gnutls distribution.
 */
int gnutls_global_init();
void gnutls_global_deinit();

typedef ssize_t (*RECV_FUNC)(SOCKET, void*, size_t,int);
typedef ssize_t (*SEND_FUNC)(SOCKET, const void*, size_t,int);
typedef void (*LOG_FUNC)( const char*);

void gnutls_global_set_send_func( SEND_FUNC send_func);
void gnutls_global_set_recv_func( RECV_FUNC recv_func);


/* error codes appended here */
#define GNUTLS_E_SUCCESS 0
#define	GNUTLS_E_MAC_FAILED  -1
#define	GNUTLS_E_UNKNOWN_CIPHER -2
#define	GNUTLS_E_UNKNOWN_COMPRESSION_ALGORITHM -3
#define	GNUTLS_E_UNKNOWN_MAC_ALGORITHM -4
#define	GNUTLS_E_UNKNOWN_ERROR -5
#define	GNUTLS_E_UNKNOWN_CIPHER_TYPE -6
#define	GNUTLS_E_LARGE_PACKET -7
#define GNUTLS_E_UNSUPPORTED_VERSION_PACKET -8
#define GNUTLS_E_UNEXPECTED_PACKET_LENGTH -9
#define GNUTLS_E_INVALID_SESSION -10
#define GNUTLS_E_UNABLE_SEND_DATA -11
#define GNUTLS_E_FATAL_ALERT_RECEIVED -12
#define GNUTLS_E_RECEIVED_BAD_MESSAGE -13
#define GNUTLS_E_RECEIVED_MORE_DATA -14
#define GNUTLS_E_UNEXPECTED_PACKET -15
#define GNUTLS_E_WARNING_ALERT_RECEIVED -16
#define GNUTLS_E_ERROR_IN_FINISHED_PACKET -18
#define GNUTLS_E_UNEXPECTED_HANDSHAKE_PACKET -19
#define GNUTLS_E_UNKNOWN_KX_ALGORITHM -20
#define	GNUTLS_E_UNKNOWN_CIPHER_SUITE -21
#define	GNUTLS_E_UNWANTED_ALGORITHM -22
#define	GNUTLS_E_MPI_SCAN_FAILED -23
#define GNUTLS_E_DECRYPTION_FAILED -24
#define GNUTLS_E_MEMORY_ERROR -25
#define GNUTLS_E_DECOMPRESSION_FAILED -26
#define GNUTLS_E_COMPRESSION_FAILED -27
#define GNUTLS_E_AGAIN -28
#define GNUTLS_E_EXPIRED -29
#define GNUTLS_E_DB_ERROR -30
#define GNUTLS_E_PWD_ERROR -31
#define GNUTLS_E_INSUFICIENT_CRED -32
#define GNUTLS_E_HASH_FAILED -33
#define GNUTLS_E_PARSING_ERROR -34
#define	GNUTLS_E_MPI_PRINT_FAILED -35
#define GNUTLS_E_AUTH_FAILED -36
#define GNUTLS_E_GOT_HELLO_REQUEST -37
#define GNUTLS_E_GOT_APPLICATION_DATA -38
#define GNUTLS_E_RECORD_LIMIT_REACHED -39
#define GNUTLS_E_ENCRYPTION_FAILED -40
#define GNUTLS_E_ASN1_ERROR -41
#define GNUTLS_E_ASN1_PARSING_ERROR -42
#define GNUTLS_E_X509_CERTIFICATE_ERROR -43
#define GNUTLS_E_PK_ENCRYPTION_FAILED -44
#define GNUTLS_E_PK_DECRYPTION_FAILED -45
#define GNUTLS_E_PK_SIGNATURE_FAILED -46

#define GNUTLS_E_UNIMPLEMENTED_FEATURE -250


/*************************************************/
/* File: cert_asn1.h                             */
/* Description: list of exported object by       */
/*   "cert_asn1.c"                               */
/*************************************************/

#ifndef _GNUTLS_ASN1_H
#define _GNUTLS_ASN1_H

#define PARSE_MODE_CHECK  1
#define PARSE_MODE_CREATE 2

/* List of constants for field type of typedef node_asn  */
#define TYPE_CONSTANT     1
#define TYPE_IDENTIFIER   2
#define TYPE_INTEGER      3
#define TYPE_BOOLEAN      4
#define TYPE_SEQUENCE     5
#define TYPE_BIT_STRING   6
#define TYPE_OCTET_STRING 7
#define TYPE_TAG          8
#define TYPE_DEFAULT      9
#define TYPE_SIZE        10
#define TYPE_SEQUENCE_OF 11
#define TYPE_OBJECT_ID   12
#define TYPE_ANY         13
#define TYPE_SET         14
#define TYPE_SET_OF      15
#define TYPE_DEFINITIONS 16
#define TYPE_TIME        17
#define TYPE_CHOICE      18
#define TYPE_IMPORTS     19
#define TYPE_NULL        20
#define TYPE_ENUMERATED  21


/***********************************************************************/
/* List of constants for specify better the type of typedef node_asn.  */
/***********************************************************************/
/*  Used with TYPE_TAG  */
#define CONST_UNIVERSAL   (1<<8)
#define CONST_PRIVATE     (1<<9)
#define CONST_APPLICATION (1<<10)
#define CONST_EXPLICIT    (1<<11)
#define CONST_IMPLICIT    (1<<12)

#define CONST_TAG         (1<<13)  /*  Used in ASN.1 assignement  */
#define CONST_OPTION      (1<<14)
#define CONST_DEFAULT     (1<<15)
#define CONST_TRUE        (1<<16)
#define CONST_FALSE       (1<<17)

#define CONST_LIST        (1<<18)  /*  Used with TYPE_INTEGER and TYPE_BIT_STRING  */
#define CONST_MIN_MAX     (1<<19)

#define CONST_1_PARAM     (1<<20)

#define CONST_SIZE        (1<<21)

#define CONST_DEFINED_BY  (1<<22)

#define CONST_GENERALIZED (1<<23)
#define CONST_UTC         (1<<24)

#define CONST_IMPORTS     (1<<25)

#define CONST_NOT_USED    (1<<26)
#define CONST_SET         (1<<27)
#define CONST_ASSIGN      (1<<28)

#define CONST_DOWN        (1<<29)
#define CONST_RIGHT       (1<<30)


#define ASN_OK                    0
#define ASN_FILE_NOT_FOUND        1
#define ASN_ELEMENT_NOT_FOUND     2
#define ASN_IDENTIFIER_NOT_FOUND  3
#define ASN_DER_ERROR             4
#define ASN_VALUE_NOT_FOUND       5
#define ASN_GENERIC_ERROR         6
#define ASN_VALUE_NOT_VALID       7
#define ASN_TAG_ERROR             8
#define ASN_TAG_IMPLICIT          9
#define ASN_ERROR_TYPE_ANY       10
#define ASN_SYNTAX_ERROR         11

/******************************************************/
/* Structure definition used for the node of the tree */
/* that rappresent an ASN.1 DEFINITION.               */
/******************************************************/
typedef struct node_asn_struct{
  char *name;                    /* Node name */
  unsigned int type;             /* Node type */
  unsigned char *value;          /* Node value */
  struct node_asn_struct *down;  /* Pointer to the son node */
  struct node_asn_struct *right; /* Pointer to the brother node */
  struct node_asn_struct *left;  /* Pointer to the next list element */ 
} node_asn;



typedef struct static_struct_asn{
  char *name;                    /* Node name */
  unsigned int type;             /* Node type */
  unsigned char *value;          /* Node value */
} static_asn;


/****************************************/
/* Returns the first 8 bits.            */
/* Used with the field type of node_asn */
/****************************************/
#define type_field(x)     (x&0xFF) 


/***************************************/
/*  Functions used by ASN.1 parser     */
/***************************************/
node_asn *
_asn1_add_node(unsigned int type);

node_asn *
_asn1_set_value(node_asn *node,unsigned char *value,unsigned int len);

node_asn *
_asn1_set_name(node_asn *node,char *name);

node_asn *
_asn1_set_right(node_asn *node,node_asn *right);

node_asn *
_asn1_get_right(node_asn *node);

node_asn *
_asn1_get_last_right(node_asn *node);

node_asn *
_asn1_set_down(node_asn *node,node_asn *down);

char *
_asn1_get_name(node_asn *node);

node_asn *
_asn1_get_down(node_asn *node);

node_asn *
_asn1_mod_type(node_asn *node,unsigned int value);

void
_asn1_append_tree(node_asn *node);

node_asn *
_asn1_find_node(node_asn *pointer,char *name);

node_asn *
_asn1_find_up(node_asn *node);


int 
asn1_parser_asn1(char *file_name,node_asn **pointer);

int
asn1_create_structure(node_asn *root,char *source_name,node_asn **pointer,
		 char *dest_name);

int
asn1_delete_structure(node_asn *root);

int 
asn1_write_value(node_asn *root,char *name,unsigned char *value,int len);

int 
asn1_read_value(node_asn *root,char *name,unsigned char *value,int *len);

int
asn1_create_tree(static_asn *root,node_asn **pointer);

#endif


